/**************************************************************************************
* @file         : ActQueue.cpp
* @author       :
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (C) 2019-2020 Robert Bosch GmbH
*                 (C) 2019-2020 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "ActQueue.h"
#include "hmi_trace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#include "trcGenProj/Header/ActQueue.cpp.trc.h"
#endif


/**
 * Navi middleware has a constaint that it cannot handle
 * more than these x simultaneous requests of a kind.
 */
#define ACT_QUEUE_SIZE_MAX      2

/**
 * The maximum time a request should take to be answered.
 * Used to ensure that no unanswered requests will remain in
 * the queue, which would compromise the response order for
 * subsequent requests.
 */
#define ACT_QUEUE_REQUEST_TIMEOUT_MS    10000


ActQueue::ActQueue()
{
}


ActQueue::~ActQueue()
{
}


void ActQueue::push(act_t act)
{
   if (_actQueue.size() < ACT_QUEUE_SIZE_MAX)
   {
      if (_actQueue.empty())
      {
         // only start the timer if this is the first ACT;
         // otherwise a timer is already running
         _timer.start(*this, ACT_QUEUE_REQUEST_TIMEOUT_MS);
      }
   }
   else
   {
      static bool errmemWritten = false;
      if (!errmemWritten)
      {
         // write only once per power-cycle
         ETG_TRACE_ERRMEM(("ACT queue size limit exceeded"));
         errmemWritten = true;
      }
      ETG_TRACE_USR4(("ACT queue size limit exceeded"));
   }
   _actQueue.push_back(act);
}


act_t ActQueue::pop()
{
   if (_actQueue.empty())
   {
      return 0;
   }
   else if (_actQueue.size() > ACT_QUEUE_SIZE_MAX)
   {
      // If more than ACT_QUEUE_SIZE_MAX simultaneous requests are made,
      // the requests exceeding ACT_QUEUE_SIZE_MAX are responded
      // immediately with an empty response.
      // ACTs beyond ACT_QUEUE_SIZE_MAX 'overtake' ACTs before that limit.
      return getAndRemoveAct(ACT_QUEUE_SIZE_MAX);
   }
   else
   {
      // return the oldest ACT from the queue
      // restart timer, if there are more ACTs
      _timer.stop();
      if (_actQueue.size() > 1)
      {
         _timer.start(*this, ACT_QUEUE_REQUEST_TIMEOUT_MS);
      }
      return getAndRemoveAct(0);
   }
}


bool ActQueue::empty()
{
   return _actQueue.empty();
}


void ActQueue::onExpired(
   ::asf::core::Timer& timer,
   ::boost::shared_ptr< ::asf::core::TimerPayload > /*data*/)
{
   if (&timer == &_timer)
   {
      ETG_TRACE_USR4(("ACT queue timer expired - clearing ACT queue"));
      _timer.stop();
      _actQueue.clear();
   }
}


/**
 * get ACT at pos from the queue and delete
 * the element from the queue
 * returns 0, if pos is outside the queue
 * boundaries
 */
act_t ActQueue::getAndRemoveAct(size_t pos)
{
   if (pos >= _actQueue.size())
   {
      return 0;
   }
   act_t act = _actQueue[pos];
   _actQueue.erase(_actQueue.begin() + pos);
   return act;
}
