/**************************************************************************************
* @file         : AddressDetailedInfo.cpp
* @author       : KanagaDurga Balakrishnan
* @addtogroup   :
* @brief        :
* @copyright    : (c) -2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "gui_std_if.h"
#include "util/StrUtf8.h"
#include "AppHmi_NavigationStateMachine.h"
#include "AddressDetailedInfo.h"
#include "CgiExtensions/ImageLoader.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "../Common/Util/StringUtils.h"
#include <iostream>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_DM
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/AddressDetailedInfo.cpp.trc.h"
#endif

static const char* IMG_ICON_FAVORITE_NORMAL  = "AppHmi_NavigationModule#Images#N_Address_DetailedInfo#Icon_AddToFavorite_normalBmp";
static const char* IMG_ICON_FAVORITE_TOUCHED = "AppHmi_NavigationModule#Images#N_Address_DetailedInfo#Icon_AddToFavorite_touchedBmp";
static const char* IMG_ICON_EDIT_NORMAL      = "AppHmi_NavigationModule#Images#N_Address_DetailedInfo#edit_icon_normalBmp";
static const char* IMG_ICON_EDIT_TOUCHED     = "AppHmi_NavigationModule#Images#N_Address_DetailedInfo#edit_icon_touchedBmp";

using namespace navmiddleware;

AddressDetailedInfo* AddressDetailedInfo::poSelf = NULL;

AddressDetailedInfo::AddressDetailedInfo(navmiddleware::NavMiddleware& navMiddleware, InfoStore& infoStore)
   : HMIModelBase(navMiddleware, infoStore)
{
   poSelf = this;
}


AddressDetailedInfo::~AddressDetailedInfo()
{
}


void AddressDetailedInfo::initialize()
{
   _navMiddleware.registerLocationPropertyUpdateCallback(*this);
   _infoStore.registerDataPropertyUpdateCallback(*this);
}


void AddressDetailedInfo::deinitialize()
{
   _navMiddleware.unregisterLocationPropertyUpdateCallback(*this);
   _infoStore.unregisterDataPropertyUpdateCallback(*this);
}


AddressDetailedInfo* AddressDetailedInfo::getInstance()
{
   return poSelf;
}


bool AddressDetailedInfo::onCourierMessage(const ShowDetailedAddressMsg& oMsg)
{
   ETG_TRACE_USR4(("AddressDetailedInfo::onCourierMessage(ShowDetailedAddressMsg())"));

   (*_placeInfoDetailsData).mName        = _infoStore.getAddressInfoDetailsName().c_str();
   (*_placeInfoDetailsData).mAddress     = _infoStore.getAddressInfoDetailsAddress().c_str();
   (*_placeInfoDetailsData).mPhonenumber = _infoStore.getAddressInfoDetailsPhoneNumber().c_str();

   ETG_TRACE_USR4(("AddressDetailedInfo::ShowDetailedAddressMsg Phone number= %s", (*_placeInfoDetailsData).mPhonenumber.GetCString()));
   updateCallButtonState();

   _placeInfoDetailsData.MarkAllItemsModified();
   _placeInfoDetailsData.SendUpdate(true);

   return true;
}


bool AddressDetailedInfo::onCourierMessage(const ResetDetailedAddressMsg& oMsg)
{
   ETG_TRACE_USR4(("AddressDetailedInfo::onCourierMessage(ResetDetailedAddressMsg())"));

   (*_placeInfoDetailsData).mName        = "";
   (*_placeInfoDetailsData).mAddress     = "";
   (*_placeInfoDetailsData).mPhonenumber = "";
   (*_enablePhoneButtonVisibility).mIsPhoneButtonVisible = false;

   _enablePhoneButtonVisibility.MarkAllItemsModified();
   _enablePhoneButtonVisibility.SendUpdate(true);
   _placeInfoDetailsData.MarkAllItemsModified();
   _placeInfoDetailsData.SendUpdate(true);

   POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, false, 0, false)));

   return true;
}


bool AddressDetailedInfo::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   bool bIsMsgConsumed = false;

   const Courier::ViewId ADDRESS_DETAILINFO_SCENENAME = Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#N_Address_DetailedInfo");
   const Courier::Identifier IdCallBtn                = Courier::Identifier("Layer_Call/ButtonWidget");

   const Courier::Identifier senderInfo = oMsg.GetSender();
   const Courier::ViewId sceneName      = oMsg.GetView();
   const enReaction reaction            = oMsg.GetEnReaction();

   if ((ADDRESS_DETAILINFO_SCENENAME == sceneName) && (IdCallBtn == senderInfo) && (enRelease == reaction))
   {
      ETG_TRACE_USR4(("AddressDetailedInfo::onCourierMessage(ButtonReactionMsg), Scene name : %s", sceneName.CStr()));

      const char* phoneNum = _infoStore.getAddressInfoDetailsPhoneNumber().c_str();
      if ((NULL != phoneNum) && (true == _infoStore.getBTActiveDeviceAvailable()))
      {
         ETG_TRACE_USR4(("AddressDetailedInfo Phone number= %s", phoneNum));
         POST_MSG((COURIER_MESSAGE_NEW(BTPhoneDialReqMsg)(phoneNum)));
      }
      bIsMsgConsumed = true;
   }
   return bIsMsgConsumed;
}


//Popup implementation courier messages and handling for two line and two button popup
bool AddressDetailedInfo::onCourierMessage(const UpdateDeletePopupTextReqMsg& oMsg)
{
   bool bIsMsgConsumed = true;
   unsigned int popupType = oMsg.GetPopupTextID();
   ETG_TRACE_USR4(("AddressDetailedInfo::UpdateDeletePopupTextReqMsg: popupType= %d", popupType));
   const Candera::String DATA_CONTEXT_ROUTE_GUIDANCE_CANCEL_FIRSTLINE = LANGUAGE_STRING(TextId_0x1461, "Do you want to cancel guidance??");
   const Candera::String DATA_CONTEXT_EMPTY_STR          = LANGUAGE_STRING(TEXTID_UNAVAILABLE, "");
   const Candera::String DATA_CONTEXT_RESET_ALL_SETTINGS = LANGUAGE_STRING(TextId_0x145E, "Return all settings to their default?");
   const Candera::String DATA_CONTEXT_TEXT_DELETE_ALL    = LANGUAGE_STRING(TextId_0x0686, "all last destinations?");
   const Candera::String DATA_CONTEXT_TEXT_DELETE        = LANGUAGE_STRING(TextId_0x0684, "the selected destination?");
   const Candera::String DATA_CONTEXT__COMMONTEXT_DELETE = LANGUAGE_STRING(TextId_0x067B, "Are you sure you want to delete");

   const Candera::String DATA_CONTEXT__TEXT_REPLACE_HOMEFAV_FIRSTLINE       = LANGUAGE_STRING(TextId_0x0A1F, "Do you want to replace home with");
   const Candera::String DATA_CONTEXT__TEXT_REPLACE_HOMEFAV_SECONDLINE      = LANGUAGE_STRING(TextId_0x0E88, "this address?");

   const Candera::String DATA_CONTEXT__TEXT_WAYPOINT_DELETE_ALL = LANGUAGE_STRING(TextId_0x0F15, "all waypoints and stop guidance?");
   const Candera::String DATA_CONTEXT__TEXT_WAYPOINT_DELETE     = LANGUAGE_STRING(TextId_0x0F13, "this Waypoint?");

   const Candera::String DATA_CONTEXT__TEXT_FAV_DELETE_ALL = LANGUAGE_STRING(TextId_0x067E, "all favourites?");
   const Candera::String DATA_CONTEXT__TEXT_FAV_DELETE     = LANGUAGE_STRING(TextId_0x067C, "this favourite?");

   // TODO: Langaugge Translation as a part of settings.
   const Candera::String DATA_CONTEXT__TEXT_DELETE_TRACKING_DOTS_LINE1 = LANGUAGE_STRING(TextId_0x1462, "Would you like to delete the current");
   const Candera::String DATA_CONTEXT__TEXT_DELETE_TRACKING_DOTS_LINE2 = LANGUAGE_STRING(TextId_0x1463, "tracking dots from the map?");

   const Candera::String ADD_DESTINATION_LINE1 = LANGUAGE_STRING(TextId_0x1466, "Do you want to add as a Destination?");
   const Candera::String NULL_STRING = "";

   const Candera::String DATA_CONTEXT__TEXT_SKIPWAYPOINT_LINE1 = LANGUAGE_STRING(TextId_0x146C, "Do you want to skip the current waypoint?");

   const Candera::String DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE1 = LANGUAGE_STRING(TextId_0x19B0, "Please restart the system so that");
   const Candera::String DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE2 = LANGUAGE_STRING(TextId_0x19B1, "the GNSS setting for satellite source");
   const Candera::String DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE3 = LANGUAGE_STRING(TextId_0x19B2, "can take effect.");

   POST_MSG((COURIER_MESSAGE_NEW(PopupTypeUpdateMsg)(popupType)));   //Send popup type information to controller.

   switch (popupType)
   {
      case  POPUP_PREVIOUS_DESTINATION_DELETE:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT_TEXT_DELETE;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_PREVIOUS_DESTINATION_DELETE_ALL:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT_TEXT_DELETE_ALL;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_REPLACE_HOME:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__TEXT_REPLACE_HOMEFAV_FIRSTLINE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_REPLACE_HOMEFAV_SECONDLINE;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_FAVORITES_DELETE_ALL:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_FAV_DELETE_ALL;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_FAVORITES_DELETE:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_FAV_DELETE;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_ROUTE_GUIDANCE_CANCEL_ROUTE:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT_ROUTE_GUIDANCE_CANCEL_FIRSTLINE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT_EMPTY_STR;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SingleLine_TwoButton"))));
         break;
      }
      case  POPUP_RESET_ALL_SETTINGS:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT_RESET_ALL_SETTINGS;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT_EMPTY_STR;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SingleLine_TwoButton"))));
         break;
      }
      case POPUP_TRACKING_DOTS_DELETE:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__TEXT_DELETE_TRACKING_DOTS_LINE1;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_DELETE_TRACKING_DOTS_LINE2;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_WAYPOINT_DELETE:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_WAYPOINT_DELETE;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_WAYPOINT_DELETE_ALL:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__COMMONTEXT_DELETE;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT__TEXT_WAYPOINT_DELETE_ALL;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_ADD_DESTINATION:
      {
         (*_PopupText).mPopuptextLine1 = ADD_DESTINATION_LINE1;
         (*_PopupText).mPopuptextLine2 = NULL_STRING;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_MultiLine_TwoButton"))));
         break;
      }
      case  POPUP_SKIP_WAYPOINT:
      {
         (*_PopupText).mPopuptextLine1 = DATA_CONTEXT__TEXT_SKIPWAYPOINT_LINE1;
         (*_PopupText).mPopuptextLine2 = DATA_CONTEXT_EMPTY_STR;
         POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_SingleLine_TwoButton"))));
         break;
      }
      case  POPUP_SATELLITE_SOURCE_CHANGE:
      {
         (*_threeLinePopupText).mTextLine1 = DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE1;
         (*_threeLinePopupText).mTextLine2 = DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE2;
         (*_threeLinePopupText).mTextLine3 = DATA_CONTEXT__TEXT_SATELLITE_SOURCE_CHANGE_LINE3;
         _threeLinePopupText.MarkAllItemsModified();
         _threeLinePopupText.SendUpdate(true);
         if (!_infoStore.isNonNaviVariant())
         {
            POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("AppHmi_NavigationModule#NavigationScenes#Pfo_Sys_ThreeLine_Text"))));
         }
         break;
      }
      default:
      {
         bIsMsgConsumed = false;
         break;
      }
   }

   _PopupText.MarkAllItemsModified();
   _PopupText.SendUpdate(true);

   return bIsMsgConsumed;
}


bool AddressDetailedInfo::onCourierMessage(const ShowFavEditIcon_Enable_Delete_Guidance_Buttons& oMsg)
{
   ETG_TRACE_USR4(("AddressDetailedInfo::onCourierMessage(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)"));

   // To update favorite icon in address detailed info scene
   if (oMsg.GetEditIcon() == 0)
   {
      (*_favEditIcon_DeleteBtn).mFav_EditNormalIcon = ImageLoader::getAssetBitmapImage(IMG_ICON_FAVORITE_NORMAL);
      (*_favEditIcon_DeleteBtn).mFav_EditPressedIcon = ImageLoader::getAssetBitmapImage(IMG_ICON_FAVORITE_TOUCHED);
   }
   // To update Edit icon in Home or favorite detailed info scene
   else
   {
      (*_favEditIcon_DeleteBtn).mFav_EditNormalIcon = ImageLoader::getAssetBitmapImage(IMG_ICON_EDIT_NORMAL);
      (*_favEditIcon_DeleteBtn).mFav_EditPressedIcon = ImageLoader::getAssetBitmapImage(IMG_ICON_EDIT_TOUCHED);
   }

   //To hide delete buttons in common address detailed info scene
   if (oMsg.GetDeleteIconButtonVisibility() == 0)
   {
      (*_favEditIcon_DeleteBtn).mDeleteIconButtonVisibility = FALSE;
   }
   //To show delete buttons in favorite, home and previous detailed info scenes
   else
   {
      (*_favEditIcon_DeleteBtn).mDeleteIconButtonVisibility = TRUE;
   }

   if (oMsg.GetGuidanceButtonVisibility() == 0)
   {
      (*_enableGuidanceButtonVisibility).mIsGuidanceButtonVisible = FALSE;
   }
   else
   {
      (*_enableGuidanceButtonVisibility).mIsGuidanceButtonVisible = TRUE;
   }

   if (InfoStoreBase::DETAIL_INFO_TMC == _infoStore.getAddressDetailInfoType())
   {
      (*_enableGuidanceButtonVisibility).mIsFavButtonVisible = false;
   }
   else
   {
      (*_enableGuidanceButtonVisibility).mIsFavButtonVisible = true;
   }

   if (oMsg.GetBackButtonLevelIndex() == 0)
   {
      (*_enableGuidanceButtonVisibility).mBackButtonLevelIndex = LEVEL_ONE;
   }
   else if (oMsg.GetBackButtonLevelIndex() == 1)
   {
      (*_enableGuidanceButtonVisibility).mBackButtonLevelIndex = LEVEL_TWO;
   }
   else
   {
      (*_enableGuidanceButtonVisibility).mBackButtonLevelIndex = LEVEL_THREE;
   }

   if (oMsg.GetCarModeNaviIconVisibility() == 0)
   {
      (*_enableGuidanceButtonVisibility).mIsCarModeNaviIconVisible = FALSE;
   }
   else
   {
      (*_enableGuidanceButtonVisibility).mIsCarModeNaviIconVisible = TRUE;
   }

   _favEditIcon_DeleteBtn.MarkAllItemsModified();
   _favEditIcon_DeleteBtn.SendUpdate(true);
   _enableGuidanceButtonVisibility.MarkAllItemsModified();
   _enableGuidanceButtonVisibility.SendUpdate(true);

   _infoStore.setIsDetailInfoRequested(false);

   return true;
}


void AddressDetailedInfo::FillAddressDetailedinfo(const navmiddleware::LocationAttributeInfos& infos)
{
   ETG_TRACE_USR4(("AddressDetailedInfo::FillAddressDetailedinfo()"));

   std::string locationName, locationAddress, locationPhoneNum;
   const ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>& locationAttributeInfos = infos.getAttributeInfos();
   ::std::vector<const LocationAttributeInfos::LocationAttributeInfo*>::const_iterator infosIterator = locationAttributeInfos.begin();

   for (; infosIterator != locationAttributeInfos.end(); ++infosIterator)
   {
      if (NULL != *infosIterator)
      {
         switch ((*infosIterator)->m_type)
         {
            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__LOCATION_NAME:
            {
               // For POI, location name is valid. For address/coordinates, the address details content is seen in location name
               if (LocationAttributeInfos::LOCATIONTYPE_POI == infos.getLocationType())
               {
                  const LocationAttributeInfos::StringAttributeInfo* locationNameInfo = infos.getLocationNameInfo();
                  if (NULL != locationNameInfo)
                  {
                     locationName = locationNameInfo->m_content;
                     ETG_TRACE_USR4(("AddressDetailedInfo::FillAddressDetailedinfo, Location name = %s", locationName.c_str()));
                  }
               }
            }
            break;

            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__ADDRESS_DETAILS:
            {
               const LocationAttributeInfos::AddressDetailInfo* addressDetailInfo = infos.getAddressDetailInfos();

               if (NULL != addressDetailInfo)
               {
                  locationAddress = addressDetailInfo->getAddressValue(navmiddleware::LocationAttributeInfos::AddressDetailInfo::ADDRESS_INFO_TYPE__FULL_ADDRESS);
                  ETG_TRACE_USR4(("AddressDetailedInfo::FillAddressDetailedinfo, Location address = %s", locationAddress.c_str()));
               }
            }
            break;

            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__PHONE_NUMBER:
            {
               const LocationAttributeInfos::StringAttributeInfo* phoneNumInfo = infos.getPhoneNumberInfo();
               if (NULL != phoneNumInfo)
               {
                  locationPhoneNum = phoneNumInfo->m_content;
                  ETG_TRACE_USR4(("AddressDetailedInfo::FillAddressDetailedinfo, Location phone number = %s", locationPhoneNum.c_str()));
               }
            }
            break;

            case LocationAttributeInfos::LocationAttributeInfo::ATTRIBUTE_TYPE__DESCRIPTION:
            {
               if (true == locationAddress.empty())
               {
                  const LocationAttributeInfos::StringAttributeInfo* descriptionInfo = infos.getDescriptionInfo();
                  if (NULL != descriptionInfo)
                  {
                     locationAddress = descriptionInfo->m_content;
                     ETG_TRACE_USR4(("AddressDetailedInfo::FillAddressDetailedinfo, Location description = %s", locationAddress.c_str()));
                  }
               }
            }
            break;

            default:
            {
               // nothing to do
            }
         }     // end of switch case
      }        // end of if null pointer check
   }           // end of for loop
   /* End of fetch location details */

   _infoStore.setAddressInfoDetailsName(locationName);
   _infoStore.setAddressInfoDetailsAddress(locationAddress);
   _infoStore.setAddressInfoDetailsPhoneNumber(locationPhoneNum);
}


void AddressDetailedInfo::onPropertyUpdateLocationAttributesAvailable()
{
   ETG_TRACE_USR4(("AddressDetailedInfo::onPropertyUpdateLocationAttributesAvailable"));

   (*_enableSetLatLongButton).mIsButtonVisible = true;
   _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsButtonVisibleItem);
   _enableSetLatLongButton.SendUpdate();

   const LocationAttributeInfos& infos = _navMiddleware.getLocationAttributes();
   FillAddressDetailedinfo(infos);
   InfoStoreBase::enAddressDetailInfoType detailInfoType = _infoStore.getAddressDetailInfoType();
   ETG_TRACE_USR4(("AddressDetailedInfo::onPropertyUpdateLocationAttributesAvailable, Is Detail info requested : %d, Detail info type : %d", _infoStore.getIsDetailInfoRequested(), detailInfoType));
   if (true == _infoStore.getIsDetailInfoRequested())
   {
      switch (detailInfoType)
      {
         case InfoStoreBase::DETAIL_INFO_WAYPOINT:
         {
            bool isCarModeNaviIconVisible = _infoStore.getIsGuidanceActive() && _infoStore.getVehicleProfile();
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, true, false, 1, isCarModeNaviIconVisible)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowWaypointDetailedInfoReqMsg)()));
            break;
         }
         case InfoStoreBase::DETAIL_INFO_HIERARCHICAL_ADDRESS:
         {
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 1, false)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            break;
         }
         case InfoStoreBase::DETAIL_INFO_GEOCOORDINATE:
         {
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 1, false)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(ShowPOIDetailsInfoForCoordinatesReqMsg)()));
            break;
         }
         case InfoStoreBase::DETAIL_INFO_MAPPICK:
         {
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, true, 1, false)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(LocationSelectionDoneResMsg)()));
            break;
         }
         case InfoStoreBase::DETAIL_INFO_DESTINATION:
         {
            bool isCarModeNaviIconVisible = _infoStore.getIsGuidanceActive() && _infoStore.getVehicleProfile();
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(0, false, false, 0, isCarModeNaviIconVisible)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(DestoinationInfoBtnSelectionMsg)()));
            break;
         }
         case InfoStoreBase::DETAIL_INFO_FAVORITES:
         {
            POST_MSG((COURIER_MESSAGE_NEW(ShowFavEditIcon_Enable_Delete_Guidance_Buttons)(1, true, true, 0, false)));
            POST_MSG((COURIER_MESSAGE_NEW(ShowDetailedAddressMsg)()));
            POST_MSG((COURIER_MESSAGE_NEW(FavoriteListBtnSelectionMsg)()));
            break;
         }
         default:
            break;
      }
   }
}


void AddressDetailedInfo::onPropertyUpdateLocationAttributesError()
{
   navmiddleware::LocationError locationError = _navMiddleware.getRequestLocationAttributesError().getLocationAttributesError();
   ETG_TRACE_USR4(("AddressDetailedInfo::onPropertyUpdateLocationAttributesError(), Location error : %d", locationError));

   (*_enableSetLatLongButton).mIsButtonVisible = true;
   _enableSetLatLongButton.MarkItemModified(ItemKey::EnableSetLatLongButton::IsButtonVisibleItem);
   _enableSetLatLongButton.SendUpdate();

   if (LOCATION_ERROR__LOCATION_IS_OFF_MAP == locationError)
   {
      POST_MSG((COURIER_MESSAGE_NEW(ShowPopupTextMsg)(POPUP_LOCATION_OFF_MAP)));
   }
}


bool AddressDetailedInfo::updateCallButtonState()
{
   ETG_TRACE_USR4(("AddressDetailedInfo::updateCallButtonState(), isCallPossible : %d", _infoStore.getBTActiveDeviceAvailable()));

   if ((false == (*_placeInfoDetailsData).mPhonenumber.IsEmpty()) && (true == _infoStore.getBTActiveDeviceAvailable()))
   {
      (*_enablePhoneButtonVisibility).mIsPhoneButtonVisible = true;
   }
   else
   {
      (*_enablePhoneButtonVisibility).mIsPhoneButtonVisible = false;
   }
   _enablePhoneButtonVisibility.MarkAllItemsModified();
   _enablePhoneButtonVisibility.SendUpdate(true);
   return true;
}
