/**************************************************************************************
* @file         : ClusterDataClientHandler.cpp
* @author       : Preethi Alagappan (ECH2)
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (c) 2022 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/


#include "hall_std_if.h"
#include "ClusterDataClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::ClusterDataClientHandler::
#include "trcGenProj/Header/ClusterDataClientHandler.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN

using namespace ::asf::core;
using namespace ::VEHICLE_MAIN_FI;
using namespace ::vehicle_main_fi_types;

namespace App {
namespace Core {

static const ::std::string VEHICLE_MAIN_FI_PORT = "vehicleMainFiPort";
ClusterDataClientHandler* ClusterDataClientHandler::_clusterDataClientHandler = NULL;

typedef std::pair< ::navmiddleware::LaneDirection, ClusterLaneDirection > LaneDirectionPair_t; // (Navmid lane direction, cluster lane direction)
static const LaneDirectionPair_t laneDirectionPairs[LANE_DIRECTION_TYPE_COUNT] =
{
   LaneDirectionPair_t(::navmiddleware::LaneDirection__NONE,        ClusterLaneDirection__NONE),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__STRAIGHT,    ClusterLaneDirection__STRAIGHT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__LEFT,        ClusterLaneDirection__LEFT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__RIGHT,       ClusterLaneDirection__RIGHT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__HALF_LEFT,   ClusterLaneDirection__HALF_LEFT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__HALF_RIGHT,  ClusterLaneDirection__HALF_RIGHT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__SHARP_LEFT,  ClusterLaneDirection__SHARP_LEFT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__SHARP_RIGHT, ClusterLaneDirection__SHARP_RIGHT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__UTURN_LEFT,  ClusterLaneDirection__UTURN_LEFT),
   LaneDirectionPair_t(::navmiddleware::LaneDirection__UTURN_RIGHT, ClusterLaneDirection__UTURN_RIGHT)
};


typedef std::pair< VehicleRestrictionsPriorityType, ClusterVehicleRestrictions > VehicleRestrictionsPair_t; // (HMI Vehicle restrictions type, cluster restrictions type)
static const VehicleRestrictionsPair_t restrictionPairs[VEHICLE_RESTRICTIONS_TYPE_COUNT] =
{
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_WEIGHT,          ClusterVehicleRestrictions__WEIGHT),
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_WEIGHT_PER_AXLE, ClusterVehicleRestrictions__WEIGHT_PER_AXLE),
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_WIDTH,           ClusterVehicleRestrictions__WIDTH),
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_HEIGHT,          ClusterVehicleRestrictions__HEIGHT),
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_LENGTH,          ClusterVehicleRestrictions__LENGTH),
   VehicleRestrictionsPair_t(VEHICLE_RESTRICTIONS_GOODS_TYPE,      ClusterVehicleRestrictions__HAZARDOUS_LOAD)
};


ClusterDataClientHandler::ClusterDataClientHandler() :
   _vehicleProxy(VEHICLE_MAIN_FIProxy::createProxy(VEHICLE_MAIN_FI_PORT, *this))
   , m_latitudeValueToCAN(0)
   , m_longitudeValueToCAN(0)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::ClusterDataClientHandler()"));
}


ClusterDataClientHandler* ClusterDataClientHandler::getInstance()
{
   if (NULL == _clusterDataClientHandler)
   {
      _clusterDataClientHandler = new ClusterDataClientHandler();
   }
   return _clusterDataClientHandler;
}


void ClusterDataClientHandler::destroyInstance()
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::destroyInstance()"));
   if (NULL != _clusterDataClientHandler)
   {
      delete _clusterDataClientHandler;
      _clusterDataClientHandler = NULL;
   }
}


void ClusterDataClientHandler::onAvailable(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onAvailable()"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void ClusterDataClientHandler::onUnavailable(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onUnavailable()"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void ClusterDataClientHandler::onVehicleNaviLatitudeResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< VehicleNaviLatitudeResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onVehicleNaviLatitudeResult(%d)", result->getResult()));
}


void ClusterDataClientHandler::onVehicleNaviLatitudeError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< VehicleNaviLatitudeError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onVehicleNaviLatitudeError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onVehicleNaviLongitudeResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< VehicleNaviLongitudeResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onVehicleNaviLongitudeResult(%d)", result->getResult()));
}


void ClusterDataClientHandler::onVehicleNaviLongitudeError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< VehicleNaviLongitudeError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onVehicleNaviLongitudeError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onSendNaviCurrentStreetNameInfoResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviCurrentStreetNameInfoResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviCurrentStreetNameInfoResult(%d)", result->getStatus()));
}


void ClusterDataClientHandler::onSendNaviCurrentStreetNameInfoError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviCurrentStreetNameInfoError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviCurrentStreetNameInfoError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onSendNaviDestinationInfoResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviDestinationInfoResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviDestinationInfoResult(%d)", result->getStatus()));
}


void ClusterDataClientHandler::onSendNaviDestinationInfoError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviDestinationInfoError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviDestinationInfoError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onSendNaviLaneGuidanceInfoResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviLaneGuidanceInfoResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviLaneGuidanceInfoResult(%d)", result->getStatus()));
}


void ClusterDataClientHandler::onSendNaviLaneGuidanceInfoError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviLaneGuidanceInfoError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviLaneGuidanceInfoError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onSendNaviTurnToStreetNameInfoResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviTurnToStreetNameInfoResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviTurnToStreetNameInfoResult(%d)", result->getStatus()));
}


void ClusterDataClientHandler::onSendNaviTurnToStreetNameInfoError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviTurnToStreetNameInfoError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviTurnToStreetNameInfoError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::onSendNaviNavigationWarningInfoResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviNavigationWarningInfoResult >& result)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviNavigationWarningInfoResult(%d)", result->getStatus()));
}


void ClusterDataClientHandler::onSendNaviNavigationWarningInfoError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< SendNaviNavigationWarningInfoError >& error)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::onSendNaviNavigationWarningInfoError(), CCA error code : %d, System error code : %d", error->getCcaErrorCode(), error->getSystemErrorCode()));
}


void ClusterDataClientHandler::updateVehiclePosition(const double& latitude, const double& longitude)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateVehiclePosition(), Latitude : %lf, Longitude : %lf", latitude, longitude));

   unsigned int latitudeValueToCAN = static_cast<unsigned int>((latitude + OFFSET) * SCALING_FACTOR);
   unsigned int longitudeValueToCAN = static_cast<unsigned int>((longitude + OFFSET) * SCALING_FACTOR);

   if (m_latitudeValueToCAN != latitudeValueToCAN)
   {
      m_latitudeValueToCAN = latitudeValueToCAN;
      ETG_TRACE_USR4(("ClusterDataClientHandler::printClusterNaviData(), Latitude : %x", m_latitudeValueToCAN));
      _vehicleProxy->sendVehicleNaviLatitudeStart(*this, m_latitudeValueToCAN);
   }
   if (m_longitudeValueToCAN != longitudeValueToCAN)
   {
      m_longitudeValueToCAN = longitudeValueToCAN;
      ETG_TRACE_USR4(("ClusterDataClientHandler::printClusterNaviData(), Longitude : %x", m_longitudeValueToCAN));
      _vehicleProxy->sendVehicleNaviLongitudeStart(*this, m_longitudeValueToCAN);
   }
}


void ClusterDataClientHandler::updateCurrentStreetName(const ::std::string& streetName)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateCurrentStreetName(%s)", streetName.c_str()));

   ::std::vector<uint8> listData;
   listData.push_back((uint8) ClusterDataType__CURRENT_STREET_NAME);
   for (int index = 0; (index < streetName.length()) && (index < (STREET_NAME_BYTE_COUNT - 1)); index++)
   {
      listData.push_back(streetName[index]);
   }
   listData.push_back('\0');   // String stored in byte array should be terminated with a NULL character

   ::vehicle_main_fi_types::T_CAN_Mesg streetNameCANMsg;
   streetNameCANMsg.setListData(listData);
   if (NULL != _vehicleProxy.get())
   {
      printClusterNaviData(listData);
      _vehicleProxy->sendSendNaviCurrentStreetNameInfoStart(*this, streetNameCANMsg);
   }
}


void ClusterDataClientHandler::updateDestinationInfo(const int& distanceToDest, const int& arrivalTimeHour, const int& arrivalTimeMin, const int& timeToDest)
{
   ::std::vector<uint8> listData;
   listData.push_back((uint8) ClusterDataType__DESTINATION_INFO);

   extractToByteListFromInteger(distanceToDest, listData, DISTANCE_BYTE_COUNT);
   extractToByteListFromInteger(arrivalTimeHour, listData);
   extractToByteListFromInteger(arrivalTimeMin, listData);

   int timeToDestInMinutes = (int)(timeToDest / 60);
   extractToByteListFromInteger(timeToDestInMinutes, listData, TIME_BYTE_COUNT);
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateDestinationInfo() Distance(m) : %d, ETA(h:m) : (%d:%d), Duration(m) : %d", distanceToDest, arrivalTimeHour, arrivalTimeMin, timeToDestInMinutes));

   if (m_destinationInfos != listData)
   {
      m_destinationInfos = listData;
      if (NULL != _vehicleProxy.get())
      {
         printClusterNaviData(listData);
         _vehicleProxy->sendSendNaviDestinationInfoStart(*this, (::vehicle_main_fi_types::T_CAN_Navi_Mesg)m_destinationInfos);
      }
   }
}


void ClusterDataClientHandler::updateLaneGuidanceInfo(const ::navmiddleware::LaneInfos& laneInfos)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateLaneGuidanceInfo()"));

   ::std::vector<ClusterLaneInfo> clusterLaneInfos;
   convertLaneInfosToClusterType(laneInfos, clusterLaneInfos);

   ::std::vector<uint8> listData;
   listData.push_back((uint8) ClusterDataType__LANE_GUIDANCE);

   extractToByteListFromInteger(clusterLaneInfos.size(), listData);
   for (int index = 0; index < clusterLaneInfos.size(); index++)
   {
      int laneDirectionType = clusterLaneInfos[index].laneStatus | clusterLaneInfos[index].laneDirection;
      extractToByteListFromInteger(laneDirectionType, listData);
   }

   if (m_laneInfos != listData)
   {
      m_laneInfos = listData;
      if (NULL != _vehicleProxy.get())
      {
         printClusterNaviData(listData);
         _vehicleProxy->sendSendNaviLaneGuidanceInfoStart(*this, (::vehicle_main_fi_types::T_CAN_Navi_Mesg)m_laneInfos);
      }
   }
}


void ClusterDataClientHandler::updateTurnToStreetName(const ::std::string& streetName)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateTurnToStreetName(%s)", streetName.c_str()));

   ::std::vector<uint8> listData;
   listData.push_back((uint8) ClusterDataType__TURN_TO_STREET_NAME);
   for (int index = 0; (index < streetName.length()) && (index < (STREET_NAME_BYTE_COUNT - 1)); index++)
   {
      listData.push_back(streetName[index]);
   }
   listData.push_back('\0');   // String stored in byte array should be terminated with a NULL character

   if (m_turnToStreetName != listData)
   {
      m_turnToStreetName = listData;
      ::vehicle_main_fi_types::T_CAN_Mesg streetNameCANMsg;
      streetNameCANMsg.setListData(m_turnToStreetName);

      if (NULL != _vehicleProxy.get())
      {
         printClusterNaviData(listData);
         _vehicleProxy->sendSendNaviTurnToStreetNameInfoStart(*this, streetNameCANMsg);
      }
   }
}


void ClusterDataClientHandler::updateVehicleRestrictions(const ::std::vector<VehicleRestrictionsPriorityType>& restrictions)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::updateVehicleRestrictions(), Number of restrictions : %d", restrictions.size()));

   ::std::vector<uint8> listData;
   listData.push_back((uint8) ClusterDataType__NAVIGATION_WARNING);

   extractToByteListFromInteger(restrictions.size(), listData);
   for (int index = 0; (index < restrictions.size()) && (index < VEHICLE_RESTRICTIONS_MAX_COUNT); index++)
   {
      extractToByteListFromInteger(DISTANCE_TO_WARNING, listData, DISTANCE_TO_WARNING_BYTE_COUNT);
      extractToByteListFromInteger(convertVehicleRestrictionToClusterType(restrictions[index]), listData);
   }

   if (NULL != _vehicleProxy.get())
   {
      printClusterNaviData(listData);
      _vehicleProxy->sendSendNaviNavigationWarningInfoStart(*this, (::vehicle_main_fi_types::T_CAN_Navi_Mesg)listData);
   }
}


void ClusterDataClientHandler::convertLaneInfosToClusterType(const ::navmiddleware::LaneInfos& navmidLaneInfos, ::std::vector<ClusterLaneInfo>& clusterLaneInfos)
{
   const std::vector< ::navmiddleware::LaneInfos::LaneInfo >& laneInfos = navmidLaneInfos.getLaneInfos();
   ETG_TRACE_USR4(("ClusterDataClientHandler::convertLaneInfosToClusterType, Number of lanes : %d", laneInfos.size()));

   for (int index = 0; (index < laneInfos.size()) && (index < LANE_MAX_COUNT)
         && (::navmiddleware::LaneStatus__UNKNOWN != laneInfos[index].laneStatus); index++)
   {
      ETG_TRACE_USR4(("ClusterDataClientHandler::convertLaneInfosToClusterType(), Lane index : %d, Lane status : %d, Combined lane direction : 0x%02X, Guided lane direction : 0x%02X",
                      index, laneInfos[index].laneStatus, laneInfos[index].combinedFollowingLaneDirection, laneInfos[index].guidedFollowingLaneDirection));

      ClusterLaneInfo clusterLaneInfo;
      clusterLaneInfo.laneStatus = (::navmiddleware::LaneStatus__RECOMMENDED == laneInfos[index].laneStatus)
                                   ? (ClusterLaneStatus__RECOMMENDED) : (ClusterLaneStatus__NOT_RECOMMENDED);

      if (0 != laneInfos[index].guidedFollowingLaneDirection)
      {
         clusterLaneInfo.laneDirection = convertLaneDirectionToClusterType((::navmiddleware::LaneDirection)laneInfos[index].guidedFollowingLaneDirection);
      }
      else
      {
         for (int laneDirectionMask = ::navmiddleware::LaneDirection__STRAIGHT;
               laneDirectionMask <= ::navmiddleware::LaneDirection__UTURN_RIGHT; laneDirectionMask <<= 1)
         {
            if (laneDirectionMask == (laneInfos[index].combinedFollowingLaneDirection & laneDirectionMask))
            {
               clusterLaneInfo.laneDirection = convertLaneDirectionToClusterType((::navmiddleware::LaneDirection)laneDirectionMask);
               break;
            }
         }
      }
      ETG_TRACE_USR4(("ClusterDataClientHandler::convertLaneInfosToClusterType(), Lane index : %d, Cluster lane status : 0x%02X, Cluster lane direction : 0x%02X",
                      index, clusterLaneInfo.laneStatus, clusterLaneInfo.laneDirection));
      clusterLaneInfos.push_back(clusterLaneInfo);
   }
}


ClusterLaneDirection ClusterDataClientHandler::convertLaneDirectionToClusterType(const ::navmiddleware::LaneDirection& laneDirection)
{
   ClusterLaneDirection clusterLaneDirection = ClusterLaneDirection__NONE;
   for (int index = 0; index < LANE_DIRECTION_TYPE_COUNT; index++)
   {
      if (static_cast<int>(laneDirection) == static_cast<int>(laneDirectionPairs[index].first))
      {
         clusterLaneDirection = laneDirectionPairs[index].second;
         break;
      }
   }
   return clusterLaneDirection;
}


ClusterVehicleRestrictions ClusterDataClientHandler::convertVehicleRestrictionToClusterType(const VehicleRestrictionsPriorityType& restriction)
{
   ClusterVehicleRestrictions clusterVehicleRestriction = ClusterVehicleRestrictions__UNKNOWN;
   for (int index = 0; index < VEHICLE_RESTRICTIONS_TYPE_COUNT; index++)
   {
      if (static_cast<int>(restriction) == static_cast<int>(restrictionPairs[index].first))
      {
         clusterVehicleRestriction = restrictionPairs[index].second;
         break;
      }
   }
   return clusterVehicleRestriction;
}


void ClusterDataClientHandler::extractToByteListFromInteger(int integerData, std::vector<uint8>& byteArray, int numBytesToExtract)
{
   // Extract from LSB to MSB and push to byteArray
   for (int index = 0; index < numBytesToExtract; index++)
   {
      byteArray.push_back((uint8)((integerData >> (8 * index)) & 0xFF));
   }
}


void ClusterDataClientHandler::printClusterNaviData(const ::std::vector<uint8>& listData)
{
   ETG_TRACE_USR4(("ClusterDataClientHandler::printClusterNaviData(), List size : %d", listData.size()));
   if (0 < listData.size())
   {
      ETG_TRACE_USR4(("ClusterDataClientHandler::printClusterNaviData(), List Data : 0x%02X", ETG_LIST(listData.size(), ETG_T8,  &(listData[0]))));
   }
}


}
}
