/**************************************************************************************
* @file         : DiagnosisServiceClientHandler.cpp
* @author       : CMC HMI Team
* @addtogroup   : apphmi_navigation
* @brief        :
* @copyright    : (c) 2018-2018 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#include "DiagnosisServiceClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX                 App::Core::DiagnosisServiceClientHandler::
#include "trcGenProj/Header/DiagnosisServiceClientHandler.cpp.trc.h"
#endif

namespace App {
namespace Core {

DiagnosisServiceClientHandler* DiagnosisServiceClientHandler::_diagnosisServiceClientHandler = NULL;

DiagnosisServiceClientHandler::DiagnosisServiceClientHandler(const std::string& portName = " ")
   : SystemSettings1Stub(portName),
     prepareResponseReceived(0),
     executeResponseReceived(0),
     finalizeResponseReceived(0),
     rObjPrepareSystemSettingRequest(const_cast<PrepareSystemSettingRequest&>(PrepareSystemSettingRequest::getDefaultInstance())),
     rObjExecuteSystemSettingRequest(const_cast<ExecuteSystemSettingRequest&>(ExecuteSystemSettingRequest::getDefaultInstance())),
     rObjFinalizeSystemSettingRequest(const_cast<FinalizeSystemSettingRequest&>(FinalizeSystemSettingRequest::getDefaultInstance()))
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::DiagnosisServiceClientHandler Constructor: %s", portName.c_str()));

   rObjPrepareSystemSettingRequest.clear();
   rObjExecuteSystemSettingRequest.clear();
   rObjFinalizeSystemSettingRequest.clear();

   _diagnosisServiceClientHandler = this; //use the object created from service derived class
}


DiagnosisServiceClientHandler::~DiagnosisServiceClientHandler()
{
   prepareResponseReceived = 0;
   finalizeResponseReceived = 0;

   rObjPrepareSystemSettingRequest.clear();
   rObjExecuteSystemSettingRequest.clear();
   rObjFinalizeSystemSettingRequest.clear();
}


/*
 * GetInstance - To get the defsetservicebase instance
 * @param[in] None
 * @param[out] None
 * @return DiagnosisServiceClientHandler
 */
DiagnosisServiceClientHandler* DiagnosisServiceClientHandler::GetInstance()
{
   if (_diagnosisServiceClientHandler == NULL)
   {
      ETG_TRACE_USR4(("DiagnosisServiceClientHandler::_diagnosisServiceClientHandler is NULL")); //should never reach here
      _diagnosisServiceClientHandler = new DiagnosisServiceClientHandler(); //port should come from respective hmi-service class --need to check if NULL can be returned here
   }
   return _diagnosisServiceClientHandler;
}


/*
 * s_Intialize - To create defsetservicebase instance with service port
 * @param[in] None
 * @param[out] None
 * @return None
 */
void DiagnosisServiceClientHandler::s_Intialize(const std::string& portName)
{
   _diagnosisServiceClientHandler = new DiagnosisServiceClientHandler(portName);
}


/*
 * s_Destrory - To destroy defsetservicebase instance
 * @param[in] None
 * @param[out] None
 * @return None
 */

void DiagnosisServiceClientHandler::s_Destrory()
{
   if (_diagnosisServiceClientHandler != NULL)
   {
      delete _diagnosisServiceClientHandler;
      _diagnosisServiceClientHandler = NULL;
   }
}


/*
 * onPrepareSystemSettingRequest - diag client prepare request for factory restore
 * @param[in] rPrepareSystemSettingRequest
 * @param[out] None
 * @return -
 */
void DiagnosisServiceClientHandler::onPrepareSystemSettingRequest(const ::boost::shared_ptr< PrepareSystemSettingRequest >& rPrepareSystemSettingRequest)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::onPrepareSystemSettingRequest"));

   rObjPrepareSystemSettingRequest = *rPrepareSystemSettingRequest;

   PrepareResDiagMsgData tempPrepareOb;
   tempPrepareOb.sysSetType.assign((*rPrepareSystemSettingRequest).getSysSetType());

   sendPrepareRequest(tempPrepareOb);
}


/*
 * onExecuteSystemSettingRequest - diag client execute request for factory restore
 * @param[in] rExecuteSystemSettingRequest
 * @param[out] None
 * @return -
 */

void DiagnosisServiceClientHandler::onExecuteSystemSettingRequest(const ::boost::shared_ptr< ExecuteSystemSettingRequest >& rExecuteSystemSettingRequest)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::onExecuteSystemSettingRequest"));

   rObjExecuteSystemSettingRequest = *rExecuteSystemSettingRequest;

   ExecuteResDiagMsgData tempExecuteOb;
   tempExecuteOb.sysSetType.assign(rObjExecuteSystemSettingRequest.getSysSetType());

   sendExecuteRequest(tempExecuteOb);
}


/*
 * onFinalizeSystemSettingRequest - diag client finalize request for factory restore
 * @param[in] rFinalizeSystemSettingRequest
 * @param[out] None
 * @return -
 */

void DiagnosisServiceClientHandler::onFinalizeSystemSettingRequest(const ::boost::shared_ptr< FinalizeSystemSettingRequest >& rFinalizeSystemSettingRequest)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::onFinalizeSystemSettingRequest"));

   rObjFinalizeSystemSettingRequest = *rFinalizeSystemSettingRequest;

   FinalizeResDiagMsgData tempFinalizeOb;
   tempFinalizeOb.sysSetType.assign(rObjExecuteSystemSettingRequest.getSysSetType());

   sendFinalizeRequest(tempFinalizeOb);
}


/*
 * sendPrepareRequest - used to send prepare to all registered clients
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendPrepareRequest(const PrepareResDiagMsgData& _rPrepareResDiagMsgData)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::sendPrepareRequest"));

   sendPrepareResponse(0);
}


/*
 * sendPrepareResponse - used to receive prepare to all registered clients, and send PrepareResponse to diag-client
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendPrepareResponse(const int& response)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::sendPrepareResponse"));

   std::vector< PrepareSystemSettingResponseExtendedDataStruct > extendedData;
   if (response == 1)
   {
      prepareResponseReceived = 1;
   }
   //prepareResponseReceived is now hardcoded to 0, till all the hmi servers are validated for its correct usage
   sendPrepareSystemSettingResponse(rObjPrepareSystemSettingRequest.getSysSetID(), rObjPrepareSystemSettingRequest.getSysSetType(), extendedData, 0/*prepareResponseReceived*/, rObjPrepareSystemSettingRequest.getCookie());
   rObjPrepareSystemSettingRequest.clear();
   prepareResponseReceived = 0;
}


/*
 * sendExecuteRequest - used to send execute to all registered clients
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendExecuteRequest(const ExecuteResDiagMsgData& oExecuteResDiagMsgData)
{
   ETG_TRACE_USR4(("DiagServiceHandler reqExecuteResponse sysSetType(%s)", oExecuteResDiagMsgData.sysSetType.c_str()));
   const std::string resetFactoryTypeHMI = "DIA_SYSSET_TYPE_HMI";
   const std::string resetFactoryTypeProduction = "DIA_SYSSET_TYPE_PRODUCTION";
   if (oExecuteResDiagMsgData.sysSetType == resetFactoryTypeHMI)
   {
      // Reset factory from HMI Settings
      sendExecuteResponse(restoreFactorySettingHMI());
   }
   else if (oExecuteResDiagMsgData.sysSetType == resetFactoryTypeProduction)
   {
      // Reset factory from setSystemToDeliveryState
      sendExecuteResponse(restoreFactoryProduction());
   }
   else
   {
      sendExecuteResponse(0);
   }
}


/*
 * sendExecuteResponse - used to receive prepare to all registered clients, and send ExecuteResponse to diag-client
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendExecuteResponse(const int& response)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::sendExecuteResponse"));

   std::vector< ExecuteSystemSettingResponseExtendedDataStruct > extendedData;

   if (response == 1)
   {
      executeResponseReceived = 1;
   }
   //executeResponseReceived is now hardcoded to 0, till all the hmi servers are validated for its correct usage
   sendExecuteSystemSettingResponse(rObjExecuteSystemSettingRequest.getSysSetID(), rObjExecuteSystemSettingRequest.getSysSetType(), extendedData, 0 /*executeResponseReceived*/, rObjExecuteSystemSettingRequest.getCookie());
   rObjExecuteSystemSettingRequest.clear();
   executeResponseReceived = 0;
}


/*
 * sendFinalizeRequest - used to send prepare to all registered clients
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendFinalizeRequest(const FinalizeResDiagMsgData& _rFinalizeResDiagMsgData)

{
   ETG_TRACE_USR4(("DiagServiceHandler reqFinalizeResponse"));
   sendFinalizeResponse(0);
}


/*
 * sendFinalizeResponse - used to receive prepare to all registered clients, and send FinalizeResponse to diag-client
 * @param[in] none
 * @param[out] None
 * @return void
 */
void DiagnosisServiceClientHandler::sendFinalizeResponse(const int& response)
{
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler::sendFinalizeResponse"));

   std::vector< FinalizeSystemSettingResponseExtendedDataStruct > extendedData;

   if (response == 1)
   {
      finalizeResponseReceived = 1;
   }
   //finalizeResponseReceived is now hardcoded to 0, till all the hmi servers are validated for its correct usage
   sendFinalizeSystemSettingResponse(rObjFinalizeSystemSettingRequest.getSysSetID(), rObjFinalizeSystemSettingRequest.getSysSetType(), extendedData, 0/*finalizeResponseReceived*/, rObjFinalizeSystemSettingRequest.getCookie());
   rObjFinalizeSystemSettingRequest.clear();
   finalizeResponseReceived = 0;
}


int DiagnosisServiceClientHandler::restoreFactoryProduction()
{
   int result = 0; // 1 - Restore failed (for Diagnosis), 0 - Restore success (for Diagnosis)
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler restoreFactoryProduction"));
   // send reset factory  update to model
   POST_MSG((COURIER_MESSAGE_NEW(ResetFactoryReqMsg)(RESET_FACTORY_DIA_SYSSET_TYPE_PRODUCTION)));

   return result;
}


int DiagnosisServiceClientHandler::restoreFactorySettingHMI()
{
   int result = 0; // 1 - Restore failed (for Diagnosis), 0 - Restore success (for Diagnosis)
   ETG_TRACE_USR4(("DiagnosisServiceClientHandler restoreFactorySettingHMI"));
   // send reset factory  update to model
   POST_MSG((COURIER_MESSAGE_NEW(ResetFactoryReqMsg)(RESET_FACTORY_DIA_SYSSET_TYPE_HMI)));

   return result;
}


}
}
