/**************************************************************************************
* @file         : ClockClientHandler.cpp
* @author       : ECG5-Atchuta Sashank Kappagantu
* @addtogroup   : AppHmi_Navigation
* @brief        :
* @copyright    : (C) 2018 Robert Bosch GmbH
*                 (C) 2018 Robert Bosch Engineering and Business Solutions Limited
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "ClockClientHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_HALL
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX App::Core::ClockClientHandler::
#include "trcGenProj/Header/ClockClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;

static const int TIME_STR_MAX_LENGTH = 8;   // For example, 12:00pm
static const ::std::string CLOCK_FI_PORT = "clockFiPort";

namespace App {
namespace Core {
using namespace clock_main_fi;
using namespace clock_main_fi_types;
using namespace navmiddleware;

ClockClientHandler* ClockClientHandler::_clockClientHandler = NULL;

ClockClientHandler::ClockClientHandler(const std::string& clockFiPort)
   : _clockProxy(Clock_main_fiProxy::createProxy(clockFiPort, *this))
   , _displayTimeOffset(0)
   , _isAutoModeOn(false)
   , _isTwelveHrFormat(false)
   , _localHourValue(0)
   , _localMinuteValue(0)
{
   if (_clockProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _clockProxy->getPortName());
   }
}


ClockClientHandler::~ClockClientHandler()
{
}


ClockClientHandler* ClockClientHandler::getInstance()
{
   if (NULL == _clockClientHandler)
   {
      _clockClientHandler = new ClockClientHandler(CLOCK_FI_PORT);
   }
   return _clockClientHandler;
}


void ClockClientHandler::destroyInstance()
{
   ETG_TRACE_USR4(("ClockClientHandler::destroyInstance()"));
   if (NULL != _clockClientHandler)
   {
      delete _clockClientHandler;
      _clockClientHandler = NULL;
   }
}


void ClockClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void ClockClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void ClockClientHandler::registerProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockClientHandler::registerProperties()"));

   _clockProxy->sendTimeFormatUpReg(*this);
   _clockProxy->sendNewDateFormatUpReg(*this);
   _clockProxy->sendLocalTimeOffsetUpReg(*this);
   _clockProxy->sendGPS_AutoSyncUpReg(*this);
   _clockProxy->sendLocalTimeDate_MinuteUpdateUpReg(*this);
}


void ClockClientHandler::deregisterProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("ClockClientHandler::deregisterProperties()"));

   _clockProxy->sendTimeFormatRelUpRegAll();
   _clockProxy->sendNewDateFormatRelUpRegAll();
   _clockProxy->sendLocalTimeOffsetRelUpRegAll();
   _clockProxy->sendGPS_AutoSyncRelUpRegAll();
   _clockProxy->sendLocalTimeDate_MinuteUpdateRelUpRegAll();
}


void ClockClientHandler::onTimeFormatError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TimeFormatError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockClientHandler::onTimeFormatError()"));
}


void ClockClientHandler::onTimeFormatStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< TimeFormatStatus >& status)
{
   ETG_TRACE_USR4(("ClockClientHandler::onTimeFormatStatus(%d)", status->getEnTimeFormat()));
   ETG_TRACE_USR4(("ClockClientHandler::Current System Time is %d : %d", _localHourValue, _localMinuteValue));
   char buffer[TIME_STR_MAX_LENGTH] = {'\0'};
   if (status->getEnTimeFormat() == VDCLK_TEN_TimeFormat__VDCLK_EN_TF_Mode12)
   {
      _isTwelveHrFormat = true;
      uint8 hr = _localHourValue;
      if (hr >= 12)
      {
         if (hr != 12)
         {
            hr = hr - 12;
         }
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%2s", hr, _localMinuteValue, "pm");
      }
      else
      {
         if (hr == 0)
         {
            hr = 12;
         }
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%2s", hr, _localMinuteValue, "am");
      }
      (*_headerTime).mText = buffer;
      _headerTime.MarkItemModified(ItemKey::HeaderTime::TextItem);
      _headerTime.SendUpdate();
      POST_MSG((COURIER_MESSAGE_NEW(TimeFormatChangeReqMsg)(TF_12HOUR)));
   }
   else if (status->getEnTimeFormat() == VDCLK_TEN_TimeFormat__VDCLK_EN_TF_Mode24)
   {
      _isTwelveHrFormat = false;
      snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d", _localHourValue, _localMinuteValue);
      (*_headerTime).mText = buffer;
      _headerTime.MarkItemModified(ItemKey::HeaderTime::TextItem);
      _headerTime.SendUpdate();
      POST_MSG((COURIER_MESSAGE_NEW(TimeFormatChangeReqMsg)(TF_24HOUR)));
   }
   else
   {
      POST_MSG((COURIER_MESSAGE_NEW(TimeFormatChangeReqMsg)(TF_UNSET)));
   }
}


void ClockClientHandler::onNewDateFormatError(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< NewDateFormatError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockClientHandler::onNewDateFormatError()"));
}


void ClockClientHandler::onNewDateFormatStatus(const ::boost::shared_ptr< Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< NewDateFormatStatus >& status)
{
   ETG_TRACE_USR4(("ClockClientHandler::onNewDateFormatStatus(%d)", status->getEnDateFormat()));
   POST_MSG((COURIER_MESSAGE_NEW(DateFormatChangeReqMsg)(convertNewDateFormat(status->getEnDateFormat()))));
}


Courier::UInt8 ClockClientHandler::convertNewDateFormat(::clock_main_fi_types::VDCLK_TEN_NewDateFormat vdNewDateFormat)
{
   navmiddleware::DateFormat dateFormat = DATEFORMAT_UNSET;
   switch (vdNewDateFormat)
   {
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dash:
         dateFormat = DATEFORMAT_DD_MM_YYYY_DASH;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_mm_dd_yyyy_Dash:
         dateFormat = DATEFORMAT_MM_DD_YYYY_DASH;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_yyyy_mm_dd_Dash:
         dateFormat = DATEFORMAT_YYYY_MM_DD_DASH;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Slash:
         dateFormat = DATEFORMAT_DD_MM_YYYY_SLASH;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_mm_dd_yyyy_Slash:
         dateFormat = DATEFORMAT_MM_DD_YYYY_SLASH;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_dd_mm_yyyy_Dot:
         dateFormat = DATEFORMAT_DD_MM_YYYY_DOT;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_yyyy_mm_dd_JPN:
         dateFormat = DATEFORMAT_YYYY_MM_DD_JPN;
         break;
      case VDCLK_TEN_NewDateFormat__VDCLK_EN_DF_yyyy_mm_dd_KOR:
         dateFormat = DATEFORMAT_YYYY_MM_DD_KOR;
         break;
      default:
         ETG_TRACE_USR4(("DateFormat unset!"));
         break;
   }

   return static_cast<Courier::UInt8>(dateFormat);
}


void ClockClientHandler::onLocalTimeDate_MinuteUpdateError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::LocalTimeDate_MinuteUpdateError >& error)
{
   ETG_TRACE_ERR(("ClockClientHandler::onLocalTimeDate_MinuteUpdateError()"));
}


void ClockClientHandler::onLocalTimeDate_MinuteUpdateStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::LocalTimeDate_MinuteUpdateStatus >& status)
{
   char buffer[TIME_STR_MAX_LENGTH] = {'\0'};
   _localHourValue   = status->getU8Hours();
   _localMinuteValue = status->getU8Minutes();
   if (_isTwelveHrFormat)
   {
      uint8 hr  = _localHourValue;
      if (hr >= 12)
      {
         if (hr != 12)
         {
            hr = hr - 12;
         }
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%2s", hr, _localMinuteValue, "pm");
      }
      else
      {
         if (hr == 0)
         {
            hr = 12;
         }
         snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d%2s", hr, _localMinuteValue, "am");
      }
   }
   else
   {
      snprintf(buffer, TIME_STR_MAX_LENGTH, "%02d:%02d", _localHourValue, _localMinuteValue);
   }

   (*_headerTime).mText = buffer;
   _headerTime.MarkItemModified(ItemKey::HeaderTime::TextItem);
   _headerTime.SendUpdate();

   //Post a courier message with just the hour.
   POST_MSG((COURIER_MESSAGE_NEW(LocalTimeChangeReqMsg)(status->getU8Hours(), status->getU8Weekday(), status->getU8Minutes())));
}


void ClockClientHandler::onLocalTimeOffsetError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::LocalTimeOffsetError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockClientHandler::onLocalTimeOffsetError"));
}


void ClockClientHandler::onLocalTimeOffsetStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::LocalTimeOffsetStatus >& status)
{
   ETG_TRACE_USR4(("ClockClientHandler::onLocalTimeOffsetStatus(%ld)", status->getS64Offset_ScalarValue()));
   if (proxy == _clockProxy)
   {
      if (static_cast<int>(status->getS64Offset_ScalarValue()) != _displayTimeOffset)
      {
         _displayTimeOffset = static_cast<int>(status->getS64Offset_ScalarValue());
         updateLocalTimeOffsetInfo();
      }
   }
}


void ClockClientHandler::onGPS_AutoSyncError(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< clock_main_fi::GPS_AutoSyncError >& /*error*/)
{
   ETG_TRACE_ERR(("ClockClientHandler::onGPS_AutoSyncError"));
}


void ClockClientHandler::onGPS_AutoSyncStatus(const ::boost::shared_ptr< clock_main_fi::Clock_main_fiProxy >& proxy,
      const ::boost::shared_ptr< clock_main_fi::GPS_AutoSyncStatus >& status)
{
   ETG_TRACE_USR4(("ClockClientHandler::onGPS_AutoSyncStatus(%d)", status->getBPositionBased()));
   if (proxy == _clockProxy)
   {
      if (status->getBPositionBased() != _isAutoModeOn)
      {
         _isAutoModeOn = status->getBPositionBased();
         updateLocalTimeOffsetInfo();
      }
   }
}


void ClockClientHandler::updateLocalTimeOffsetInfo()
{
   POST_MSG((COURIER_MESSAGE_NEW(LocalTimeOffsetInfoUpdMsg)(_displayTimeOffset, _isAutoModeOn)));
}


/**
 * Helper function to get LocalTimeDate  from  MIDW Clock fi.
 * @return : returns local date time
 */
LocalTimeDateType ClockClientHandler::getLocalTimeDate()const
{
   ETG_TRACE_USR4(("ClockClientHandler::getLocalTimeDate"));
   LocalTimeDateType localTimeDate;
   if (_clockProxy && _clockProxy->hasLocalTimeDate_MinuteUpdate())
   {
      localTimeDate.s16Year = _clockProxy->getLocalTimeDate_MinuteUpdate().getS16Year();
      localTimeDate.u8Month = _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Month();
      localTimeDate.u8Day =  _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Day();
      localTimeDate.u8Hours = _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Hours();
      localTimeDate.u8Minutes = _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Minutes();
      localTimeDate.u8Seconds = _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Seconds();
      localTimeDate.u8Weekday = _clockProxy->getLocalTimeDate_MinuteUpdate().getU8Weekday();
   }
   return localTimeDate;
}


} // namespace App
} // namespace Core
