/****************************************************************************
* Copyright (c) 2018-2020 Robert Bosch Car Multimedia GmbH
* duplication and disclosure to third parties is prohibited.
*
* FILE                : NaviApplicationSwitchClientHandler.cpp
* COMPONENT Name      : NaviApplicationSwitchClientHandler
* DESCRIPTION         : DBus client for Application switch intrerface.
* AUTHOR              : Tejas B V
* Date                : 23.08.2017
* Revision History    : 0.1
* Date 23.08.2017     : Initial version
****************************************************************************/

/*****************************************************************
| includes
|----------------------------------------------------------------*/
#include "NaviApplicationSwitchClientHandler.h"
#include "Common/Util/SurfaceSynchronizationHandler.h"
#include "NavDataUpdate/HMIModelNavDataUpdate.h"
#include "hmi_trace_if.h"
#include <string>

/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/
#define PORTNAME "applicationSwitchPort"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_NAVIGATION_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_APPHMI_NAVIGATION
#define ETG_I_TTFIS_CMD_PREFIX "APPHMI_NAVIGATION_"
#define ETG_I_FILE_PREFIX App::Core::NaviApplicationSwitchClientHandler::
#include "trcGenProj/Header/NaviApplicationSwitchClientHandler.cpp.trc.h"
#endif

using namespace ::asf::core;

//NaviApplicationSwitchClientHandler* NaviApplicationSwitchClientHandler::m_poSelfAppSwitchClient = NULL;

NaviApplicationSwitchClientHandler::NaviApplicationSwitchClientHandler(const std::string& portName)
   : m_poApplicationSwitchClient(ApplicationSwitchProxy::createProxy(portName, *this))
   , _activeContextID(CONTEXT_SWITCH_FROM_NONE)
   , _isServiceCallBackReceived(false)
   , _contextToApp(CONTEXT_SWITCH_TO_MAP)
   , _contextFromApp(CONTEXT_SWITCH_FROM_HOMESCREEN)
   , _isDisclaimerConfirmed(false)
   , _lastActiveContextID(CONTEXT_SWITCH_FROM_NONE)
   , _currentActiveNaviContext(enActivityIDs__eActivityID_NONE)
   , _isNavAppForeground(false)
   , _isTrafficDetourContextSwitchWithinApp(false)
   , _isTrafficDetourDisplayValid(false)
{
   if (m_poApplicationSwitchClient.get())
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler m_poApplicationSwitchClient.get is TRUE"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, std::string(portName));
   }
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
}


NaviApplicationSwitchClientHandler::~NaviApplicationSwitchClientHandler()
{
   /*if (NULL != m_poSelfAppSwitchClient)
   {
      delete m_poSelfAppSwitchClient;
      m_poSelfAppSwitchClient = NULL;
   }*/
}


/*NaviApplicationSwitchClientHandler* NaviApplicationSwitchClientHandler::GetClientInstance()
{
   if (NULL == m_poSelfAppSwitchClient)
   {
      m_poSelfAppSwitchClient = new NaviApplicationSwitchClientHandler();
   }
   return m_poSelfAppSwitchClient;
}*/


void NaviApplicationSwitchClientHandler::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onAvailable"));
   StartupSync::getInstance().onAvailable(proxy, stateChange);
}


void NaviApplicationSwitchClientHandler::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onUnavailable"));
   StartupSync::getInstance().onUnavailable(proxy, stateChange);
}


void NaviApplicationSwitchClientHandler::registerProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::registerProperties()"));

   if (m_poApplicationSwitchClient == proxy)
   {
      m_poApplicationSwitchClient->sendMapout_Activate_DeactivateRegister(*this);
      m_poApplicationSwitchClient->sendSig_ActivateContextRegister(*this);
   }
   else
   {
      ETG_TRACE_USR4(("m_poApplicationSwitchClient is not available"));
   }
}


void NaviApplicationSwitchClientHandler::deregisterProperties(const ::boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::deregisterProperties()"));

   m_poApplicationSwitchClient->sendMapout_Activate_DeactivateDeregisterAll();
}


void NaviApplicationSwitchClientHandler::onActivateDeactivateRVCSceneError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActivateDeactivateRVCSceneError >& error)
{
}


void NaviApplicationSwitchClientHandler::onActivateDeactivateRVCSceneResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActivateDeactivateRVCSceneResponse >& response)
{
}


//callback for sendRequestApplcationSwitchRequest
void NaviApplicationSwitchClientHandler::onRequestApplcationSwitchResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestApplcationSwitchResponse >& response)
{
}


void NaviApplicationSwitchClientHandler::onRequestApplcationSwitchError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestApplcationSwitchError >& error)
{
}


void NaviApplicationSwitchClientHandler::onRequestShowHideWaitSceneError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestShowHideWaitSceneError >& error)
{
}


void NaviApplicationSwitchClientHandler::onRequestShowHideWaitSceneResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestShowHideWaitSceneResponse >& response)
{
}


void NaviApplicationSwitchClientHandler::onMapout_activated_deactivatedError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_activated_deactivatedError >& error)
{
}


void NaviApplicationSwitchClientHandler::onMapout_activated_deactivatedResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_activated_deactivatedResponse >& response)
{
}


void NaviApplicationSwitchClientHandler::onMapout_Activate_DeactivateError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_Activate_DeactivateError >& error)
{
}


void NaviApplicationSwitchClientHandler::onMapout_Activate_DeactivateSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Mapout_Activate_DeactivateSignal >& signal)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onMapout_Activate_DeactivateSignal()"));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(StartStopMapStreamReqMsg)(signal->getRegionId(), signal->getMapoutshow(), false)));
}


void NaviApplicationSwitchClientHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onSig_ActivateContextError"));
}


void NaviApplicationSwitchClientHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onSig_ActivateContextSignal %d", signal->getTargetactivityId()));
   _isServiceCallBackReceived = true;
   enActivityIDs targetActivityID = signal->getTargetactivityId();
   if (true == checkIsNaviContext(targetActivityID))
   {
      if ((true == _isNavAppForeground) && (true == HMIModelNavDataUpdate::isMapUpdateInProgress()))
      {
         ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onSig_ActivateContextSignal(), Context switch request is ignored since map update is in progress"));
      }
      else if (_currentActiveNaviContext != targetActivityID)
      {
         SurfaceSynchronizationHandler::setMapLayerVisibility(true);   // Map layer visibility is enabled when navi context is activated
         displayNaviContextScene(targetActivityID);
      }
   }
}


void NaviApplicationSwitchClientHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::onActiveRegionIdError"));
}


void NaviApplicationSwitchClientHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
}


bool NaviApplicationSwitchClientHandler::checkIsNaviContext(const enActivityIDs& targetActivityID)
{
   bool isNaviContext = false;
   if ((targetActivityID == enActivityIDs__eActivityID_NAVI_MAP)
         || (targetActivityID == enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW)
         || (targetActivityID == enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW)
         || (targetActivityID == enActivityIDs__eActivityID_NAVI_DESTINATIONS)
         || (targetActivityID == enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR)
         || (targetActivityID == enActivityIDs__eActivityID_NAVI_GNSS))
   {
      isNaviContext = true;
   }
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::checkIsNaviContext(), Navi context : %d", isNaviContext));
   return isNaviContext;
}


void NaviApplicationSwitchClientHandler::displayNaviContextScene(enActivityIDs& targetActivityID)
{
   if (targetActivityID == enActivityIDs__eActivityID_NAVI_MAP)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      _activeContextID = CONTEXT_SWITCH_FROM_N_MAP;
      _lastActiveContextID = _activeContextID;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
   }
   else if (targetActivityID == enActivityIDs__eActivityID_NAVI_GNSS)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      _activeContextID = CONTEXT_SWITCH_FROM_CABINONLY_GNSS;
      _lastActiveContextID = _activeContextID;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowPosandGPSFromCabinGNSSSettingstReqMsg)(_activeContextID)));
   }
   else if (targetActivityID == enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      if (_isDisclaimerConfirmed)
      {
         _activeContextID = CONTEXT_SWITCH_FROM_SETTINGS;
         _lastActiveContextID = _activeContextID;
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
         POST_MSG((COURIER_MESSAGE_NEW(SettingsContextReqMsg)()));
      }
      else
      {
         _activeContextID = _lastActiveContextID;
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
      }
   }
   else if (targetActivityID == enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      _activeContextID = CONTEXT_SWITCH_FROM_ENGGMENU;
      _lastActiveContextID = _activeContextID;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
   }
   else if (targetActivityID == enActivityIDs__eActivityID_NAVI_DESTINATIONS)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      _activeContextID = CONTEXT_SWITCH_FROM_DESTINATIONS;
      _lastActiveContextID = _activeContextID;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
   }
   else if ((targetActivityID == enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR)
            && (true == _isTrafficDetourDisplayValid))
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Display %d context in Navigation", targetActivityID));
      _activeContextID = CONTEXT_SWITCH_FROM_TRAFFIC_DETOUR;
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
   }
   else
   {
      // do nothing
   }
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ApplicationStateUpdMsg(%d)", oMsg.GetState()));

   if (oMsg.GetState() == hmibase::IN_FOREGROUND)
   {
      _isNavAppForeground = true;
   }
   else
   {
      _isNavAppForeground = false;
   }
   return false;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const SwitchContextFromTrafficDetourReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::SwitchContextFromTrafficDetourReqMsg()"));
   COURIER_UNUSED(oMsg);

   _isTrafficDetourDisplayValid = false;
   if (true == _isTrafficDetourContextSwitchWithinApp)
   {
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(DisplayLastNaviContextReqMsg)(_lastActiveContextID)));
   }
   else
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR, 0, static_cast<enActivityIDs>(0));
   }
   return true;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const SwitchContextFromNavDataUpdateReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::SwitchContextFromNavDataUpdateReqMsg()"));
   COURIER_UNUSED(oMsg);

   _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, _currentActiveNaviContext, 0, static_cast<enActivityIDs>(0));
   return true;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const ValidateNaviEntryConditionReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ValidateNaviEntryConditionReqMsg()"));
   _isDisclaimerConfirmed = oMsg.GetIsDisclaimerConfirmed();
   if (true == _isServiceCallBackReceived)
   {
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Entering into Navigation from AppFlyins"));
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ShowMapOrnewDestFromGadgetReqMsg)(_activeContextID)));
      if (CONTEXT_SWITCH_FROM_SETTINGS == _activeContextID)
      {
         POST_MSG((COURIER_MESSAGE_NEW(SettingsContextReqMsg)()));
      }
      if (EXT_bIsGuidanceReqFromSds)
      {
         POST_MSG((COURIER_MESSAGE_NEW(ShowVehicleInfoReqMsg)()));
      }
   }
   _isServiceCallBackReceived = false;
   return false;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const SendContextReqMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::SendContextReqMsg()"));

   bool isMsgConsumed = false;
   enContextSwitchFromType _contextFromActivityID = oMsg.GetContextFromActivityID();

   if (CONTEXT_SWITCH_FROM_N_MAP == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestSendContext(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_MAP);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_MAP;
      _lastActiveContextID = CONTEXT_SWITCH_FROM_N_MAP;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_MAP));
      isMsgConsumed = true;
   }
   else if (CONTEXT_SWITCH_FROM_CABINONLY_GNSS == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_GNSS, APPID_APPHMI_SYSTEM, enActivityIDs__eActivityID_SETTINGS);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_GNSS;
      _lastActiveContextID = CONTEXT_SWITCH_FROM_CABINONLY_GNSS;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_GNSS));
      isMsgConsumed = true;
   }
   else if (CONTEXT_SWITCH_FROM_DESTINATIONS == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestSendContext(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_DESTINATIONS);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_DESTINATIONS;
      _lastActiveContextID = CONTEXT_SWITCH_FROM_DESTINATIONS;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_DESTINATIONS));
      isMsgConsumed = true;
   }
   else if (CONTEXT_SWITCH_FROM_SETTINGS == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestSendContext(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW;
      _lastActiveContextID = CONTEXT_SWITCH_FROM_SETTINGS;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW));
      isMsgConsumed = true;
   }
   else if (CONTEXT_SWITCH_FROM_ENGGMENU == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestSendContext(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW;
      _lastActiveContextID = CONTEXT_SWITCH_FROM_ENGGMENU;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW));
      isMsgConsumed = true;
   }
   else if (CONTEXT_SWITCH_FROM_TRAFFIC_DETOUR == _contextFromActivityID)
   {
      _applicationSwitchClientComponent->requestSendContext(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR);
      _currentActiveNaviContext = enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR;
      ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context ID %d is updated to Service", enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR));
      isMsgConsumed = true;
   }
   else
   {
      // The _currentActiveNaviContext is reset to NONE when Application is moved to background after receiving the msg from HMIModelComponent
      _currentActiveNaviContext = enActivityIDs__eActivityID_NONE;
      isMsgConsumed = false;
   }
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::SendContextReqMsg(), Current Active Context in Navigation is %d", _currentActiveNaviContext));

   return isMsgConsumed;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const ContextSwitchReqMsg& oMsg)
{
   _contextFromApp = oMsg.GetContextSwitchFromID();
   _contextToApp   = oMsg.GetContextSwitchToID();
   _isServiceCallBackReceived = false;

   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ContextSwitchReqMsg()"));

   if (_contextFromApp == CONTEXT_SWITCH_FROM_DETAILEDINFO)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_PHONE)
      {
         ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::Context switch to PHONE_CALL_LIST is requested from Navigation"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_DESTINATIONS, APPID_APPHMI_PHONE, enActivityIDs__eActivityID_PHONE_CALL_LIST);
      }
   }
   else if (_contextFromApp == CONTEXT_SWITCH_FROM_ENGGMENU)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_TESTMODE)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW, APPID_APPHMI_TESTMODE, enActivityIDs__eActivityID_TESTMODE_SOURCESELECTION_SCREEN);
      }
   }
   else if (_contextFromApp == CONTEXT_SWITCH_FROM_HOMESCREEN)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_NEW_DESTINATION)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_DESTINATIONS);
      }
      else if (_contextToApp == CONTEXT_SWITCH_TO_MAP)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_MAP);
      }
      else
      {
         // do nothing
      }
   }
   else if (_contextFromApp == CONTEXT_SWITCH_FROM_TRAFFIC_DETOUR)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_LAST_ACTIVE_CONTEXT)
      {
         ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ContextSwitchReqMsg() : Context switch from Traffic Detour to last context"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR, 0, static_cast<enActivityIDs>(0));
      }
   }
   else if (_contextFromApp == CONTEXT_SWITCH_FROM_DESTINATIONS)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_HOMESCREEN)
      {
         ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ContextSwitchReqMsg() : Context switch from Destinations to home screen"));
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_DESTINATIONS, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME);
      }
   }
   else if (_contextFromApp == CONTEXT_SWITCH_FROM_NONE)
   {
      if (_contextToApp == CONTEXT_SWITCH_TO_NEW_DESTINATION)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, 0, static_cast<enActivityIDs>(0), APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_DESTINATIONS);
      }
      else if (_contextToApp == CONTEXT_SWITCH_TO_MAP)
      {
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, 0, static_cast<enActivityIDs>(0), APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_MAP);
      }
      else if (_contextToApp == CONTEXT_SWITCH_TO_TRAFFIC_DETOUR)
      {
         ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::ContextSwitchReqMsg() : From any Source context to Traffic Detour context"));
         _isTrafficDetourDisplayValid = true;
         _isTrafficDetourContextSwitchWithinApp = _isNavAppForeground;
         _applicationSwitchClientComponent->requestContextSwitchApplication(2, 0, static_cast<enActivityIDs>(0), APPID_APPHMI_NAVIGATION, enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR);
      }
      else
      {
         // do nothing
      }
   }
   else
   {
      // do nothing
   }

   return true;
}


bool NaviApplicationSwitchClientHandler::onCourierMessage(const MapOutStatusUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("NaviApplicationSwitchClientHandler::MapOutStatusUpdMsg()"));

   uint32_t regionId = oMsg.GetRegionId();
   bool activationStatus = oMsg.GetMapOutState();
   if (NULL != m_poApplicationSwitchClient)
   {
      m_poApplicationSwitchClient->sendMapout_activated_deactivatedRequest(*this, regionId, activationStatus);
   }
   return true;
}
