/* ***************************************************************************************
* FILE:          HMIModelComponent.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  HMIModelComponent.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*21.09.17 Rev. 0.1 Sunil (RBEI/ECG5),Ruchi (RBEI/ECG5),Sonia (RBEI/ECG5)
* 		  		   Initial Revision.
*************************************************************************************** */

#include "gui_std_if.h"

#include "util/StrUtf8.h"
#include "AppHmi_MediaStateMachine.h"
#include "HMIModelComponent.h"
#include "hmi_trace_if.h"
#include <FeatStd/Util/TextId.cpp>
#include "CgiExtensions/ImageLoader.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "AppHmi_MediaTypes.h"
#include "Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#include "Core/UnSupportedPopupHandler/UnSupportedPopupHandler.h"
#include "Common/VariantHandling/VariantHandling.h"

#ifdef WINDOWS_SIMULATION
#include "DemoApp.h"
#endif

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MEDIA_HALL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/HMIModelComponent.cpp.trc.h"
#endif

namespace App {
namespace Core {

FeatStd::TextId id1("0x015B");
FeatStd::TextId id2("0x0E47");
FeatStd::TextId id3("0x0E48");
FeatStd::TextId id4("0x0E49");
FeatStd::TextId id5("0x0157");
FeatStd::TextId id6("0x0E41");
FeatStd::TextId id7("0x0E42");
FeatStd::TextId id8("0x0E43");
FeatStd::TextId id9("0x015A");
FeatStd::TextId id10("0x0E44");
FeatStd::TextId id11("0x0E45");
FeatStd::TextId id12("0x0E46");
FeatStd::TextId id13("0x0906");
FeatStd::TextId id14("0x0E50");
FeatStd::TextId id15("0x0E51");
FeatStd::TextId id16("0x0E52");
FeatStd::TextId id17("0x0208");
FeatStd::TextId id18("0x0E62");
FeatStd::TextId id19("0x0E63");
FeatStd::TextId id20("0x0E64");
FeatStd::TextId id21("0x0209");
FeatStd::TextId id22("0x0E65");
FeatStd::TextId id23("0x0E66");
FeatStd::TextId id24("0x0E67");
FeatStd::TextId id25("0x020A");
FeatStd::TextId id26("0x0E68");
FeatStd::TextId id27("0x0E69");
FeatStd::TextId id28("0x0E6A");
FeatStd::TextId id29("0x020B");
FeatStd::TextId id30("0x0E6B");
FeatStd::TextId id31("0x0E6C");
FeatStd::TextId id32("0x0E6D");
FeatStd::TextId id33("0x0163");
FeatStd::TextId id34("0x0E59");
FeatStd::TextId id35("0x0E5A");
FeatStd::TextId id36("0x0E5B");
FeatStd::TextId id37("0x0E93");
FeatStd::TextId id38("0x0E94");
FeatStd::TextId id39("0x0E95");
FeatStd::TextId id40("0x0E96");
FeatStd::TextId id41("0x0E3E");
FeatStd::TextId id42("0x0E5F");
FeatStd::TextId id43("0x0AB6");
FeatStd::TextId id44("0x0E71"); //Bluetooth Audio Streaming not available
FeatStd::TextId id46("0x0E56"); //No USB device connected.
FeatStd::TextId id47("0x0E4A"); //No iPod connected.
FeatStd::TextId id48("0x0A04"); //No Bluetooth device connected.

FeatStd::TextId id49("0x0156");
FeatStd::TextId id50("0x0E11");
FeatStd::TextId id51("0x0160");
FeatStd::TextId id52("0x015C");
FeatStd::TextId id53("0x0E3F");
FeatStd::TextId id54("0x0E40");
FeatStd::TextId id55("0x0E72");
FeatStd::TextId id56("0x0E73");
FeatStd::TextId id57("0x0E57"); //No USB device connected.
FeatStd::TextId id58("0x0E58"); //No USB device connected.
FeatStd::TextId id59("0x0E4B"); //No iPod connected.
FeatStd::TextId id60("0x0E4C"); //No iPod connected.
FeatStd::TextId id61("0x0A05"); //No Bluetooth device connected.
FeatStd::TextId id62("0x0E3D"); //No Bluetooth device connected.

FeatStd::TextId id70("0x0061"); //All Titles
FeatStd::TextId id71("0x0057"); //All Albums

FeatStd::TextId id72("0x13B2");//Camera port USB
FeatStd::TextId id73("0x13B4");//Camera port 1 HDMI
FeatStd::TextId id74("0x13AF");//Router USB
FeatStd::TextId id75("0x029D");//Bluetooth
FeatStd::TextId id76("0x1A35");//Audio USB
FeatStd::TextId id77("0x1A36");//Phone USB

FeatStd::TextId id78("0x05E3");//Mute

FeatStd::TextId id79("0x13B5");//Camera port 2 HDMI
FeatStd::TextId id80("0x1A14");//Camera port 3 HDMI

HMIModelComponent* HMIModelComponent::_theModelInstance = 0;

/* ***********************************************************
* Function 		: getInstance
* Description 	: function to create the HMIModelComponent singleton instance
* Parameters 	: void
* Return value	: _theModelInstance - HMIModelComponent object instance
* ***********************************************************/
HMIModelComponent& HMIModelComponent::getInstance()
{
   if (_theModelInstance == 0)
   {
      _theModelInstance = new HMIModelComponent();
   }

   // assert(_theModelInstance);
   return *_theModelInstance;
}


/* ***********************************************************
* Function 		: removeInstance
* Description 	: function to delete the singleton instance
* Parameters 	: void
* Return value	: void
* ***********************************************************/
void HMIModelComponent::removeInstance()
{
   if (_theModelInstance)
   {
      delete _theModelInstance;
      _theModelInstance = 0;
   }
   if (_kdsHandler)
   {
      delete _kdsHandler;
      _kdsHandler = NULL;
   }
   if (_cabinSourceRestoration)
   {
      delete _cabinSourceRestoration;
      _cabinSourceRestoration = NULL;
   }
}


/* ***********************************************************
* Function 		: HMIModelComponent
* Description 	: constructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelComponent::HMIModelComponent()
{
   MediaSystemMuteStatus = false;

   m_u8AppleDeviceConnectionType = 0;

   m_u32RepeatMode = MEDIA_C_VALUE_ZERO;

   m_u32PlayBackMode = MEDIA_C_VALUE_ZERO;

   mTACheckboxStatus = MEDIA_C_VALUE_ZERO;
   m_u8selectedDeviceType = 0;
   m_u8LastDevType = 0;
   m_u8ActiveDeviceType = 0;
   shuffleFlag = 0;
   playFlag = 0;
   m_GuiStartUpFlag = 0;
   _kdsHandler = NULL;
   _cabinSourceRestoration = NULL;
   _mediaGadgetIndex = NO_MEDIA_DEVICE_GADGET_INDEX;
   _mediaMetadataGadgetIndexMap.clear();
   initialisemediaMetadataGadgetIndexMap();
   _cabinGadgetTextIndex = NO_MEDIA_DEVICE_GADGET_INDEX;
   _cabinSourceId = SRC_INVALID;
   _restorationEvaluatingStatus = false;
}


/* ***********************************************************
* Function 		: ~HMIModelComponent
* Description 	: destructor
* Parameters 	: void
* Return value	: none
* ***********************************************************/
HMIModelComponent::~HMIModelComponent()
{
   _mediaMetadataGadgetIndexMap.clear();
   removeInstance();
}


bool HMIModelComponent::Init()
{
   return true;
}


void HMIModelComponent::enableorDisableTASettingsBasedOnRegion()
{
   uint8 regionID = 0;
   if (_kdsHandler != NULL)
   {
      regionID = _kdsHandler->getRegionID();
   }
   ETG_TRACE_USR4(("HMIModelComponent::enableorDisableTASettingsBasedOnRegion regionID:%d", regionID));

   if (EUR_REGION == regionID)
   {
      (*oVariantInfo).mTA_Settings = true;
   }
   else
   {
      (*oVariantInfo).mTA_Settings = false;
   }
   oVariantInfo.MarkAllItemsModified();
   oVariantInfo.SendUpdate(true);
}


/* ***********************************************************
* Function 		: onCourierMessage
* Description 	: Incomming startup courier message for HMI data model
* Parameters 	: StartupMsg &oMsg
* Return value	: TRUE if it consumes the message else FALSE
* ***********************************************************/
bool HMIModelComponent::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ActivateApplicationMsg* poActivateMsg = COURIER_MESSAGE_NEW(ActivateApplicationMsg)();

   if (NULL != poActivateMsg)
   {
      g_SM_SourceSelectStatus = 1;
      /* Posting Courier ActivateApplicationMsg message */
      bool bRetVal = poActivateMsg->Post();
      ETG_TRACE_USR4(("HMIModelComponent::onCourierMessage:StartupMsg bRetVal is :%d", bRetVal));
   }
   ETG_TRACE_USR4(("HMIModelComponent::StartupMsg mTACheckboxStatus = %d", mTACheckboxStatus));
   mTACheckboxStatus = tGetTAStatus();
   ETG_TRACE_USR4(("HMIModelComponent::StartupMsg mTACheckboxStatus = %d", mTACheckboxStatus));
   ETG_TRACE_USR4(("HMIModelComponent::StartupMsg (*m_MediaSettingsTAStatus).mi8IsChecked = %d", (*m_MediaSettingsTAStatus).mi8IsChecked));

   vUpdateTACheckbox(mTACheckboxStatus);

   (*m_MediaTAIcon).mTAIconIndex = MEDIA_C_VALUE_ZERO;
   m_MediaTAIcon.MarkAllItemsModified();
   m_MediaTAIcon.SendUpdate();

   //if (g_SM_VARIANT_AUX == true)//smart
   //{
   //   ETG_TRACE_USR4(("AUXVariant True"));
   //   (*m_MediaAuxActivation).mIsAuxActiveVariant = true;
   //   m_MediaAuxActivation.MarkAllItemsModified();
   //   m_MediaAuxActivation.SendUpdate(true);
   //}

   return false;
}


void HMIModelComponent::initialisemediaMetadataGadgetIndexMap()
{
   ETG_TRACE_USR1(("HMIModelComponent::initialisemediaMetadataGadgetIndexMap is entered \n"));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_MEDIA_PLAYER, INTERNAL_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_PHONE_BTAUDIO, INTERNAL_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_MEDIA_AUX, INTERNAL_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CAMERAPORT1_USB_VIDEO, USBCAMPORT_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CAMERAPORT1_USB_AUDIO, USBCAMPORT_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CAMERAPORT1_HDMI, HDMI_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CAMERAPORT2_HDMI, HDMI_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CAMERAPORT3_HDMI, HDMI_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CMR1_USB1_VIDEO, ROUTER_GADGET_INDEX));
   _mediaMetadataGadgetIndexMap.insert(pair<int, MediaGadgetIndex>(SRC_CMR1_USB1_AUDIO, ROUTER_GADGET_INDEX));
}


/* ***********************************************************
* Function 		: bOnDeviceConnectionStatusMsg
* Description 	: updates the device connection status to HMI and will update the MediaMessage popup
* Parameters 	: uint8_t iDeviceTag, int iDeviceType, bool bDeviceConnectionStatus, bool bDeviceActive, bool bPlayablecontent
* Return value	: void
* ***********************************************************/
void HMIModelComponent::vUpdateDeviceConnectionStatus(uint8_t iDeviceTag, int iDeviceType, bool bDeviceConnectionStatus, bool bDeviceActive, bool bPlayablecontent, int u8DeviceDisconnectionReason, int devNum)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :bDeviceConnectionStatus =%d", bDeviceConnectionStatus));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :bDeviceActive =%d", bDeviceActive));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :iDeviceType =%d", iDeviceType));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :Playable content =%d", bPlayablecontent));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :iDeviceTag=%d", iDeviceTag));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :devNum=%d", devNum));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceConnectionStatus :devicedisconnectionreason=%d", u8DeviceDisconnectionReason));
   POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessage)));
   POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessageHUB)));
   bool bMsgConsumed = true;

   //To save device tag of connected device
   uint8_t iCurrentConnDeviceTag = 0;
   //Update the binding source if applicable
   bool bDataUpdated = false;

   (*m_MediaMessageData).mMessageNoDeviceVisible = false;
   (*m_MediaMessageData).mMediaMessageUpdateFlag = getMediaConnectionPopupVisibilityState(iDeviceType); //block the BT pop up tigger;
   if (TRUE == bPlayablecontent)
   {
      (*m_MediaMessageData).mMessage1Visible = true;
      (*m_MediaMessageData).mMessage2Visible = true;
      (*m_MediaMessageData).mMessage3Visible = true;
      (*m_MediaMessageData).mMessage4Visible = true;
      (*m_MediaMessageData).mMessageDeviceConnectedVisible = false;
      (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = false;
      (*m_MediaMessageData).mMediamessageIconVisibleState = false;

      (*m_MediaMessageData).mMediaMessage1 = "";
      (*m_MediaMessageData).mMediaMessage2 = Candera::String(id13);
      (*m_MediaMessageData).mMediaMessage3 = "";
      (*m_MediaMessageData).mMediaMessage4 = "";
   }
   else if (bDeviceConnectionStatus)
   {
      switch (iDeviceType)
      {
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP:
         {
            std::string usb = "USB";
            if (devNum == 2)
            {
               usb = "Phone USB";
               (*m_MediaMessageData).mMediaMessageDeviceConnected = usb.c_str();
            }
            else if (devNum == 1)
            {
               usb = "Audio USB";
               (*m_MediaMessageData).mMediaMessageDeviceConnected = usb.c_str();
            }
            else
            {
               (*m_MediaMessageData).mMediaMessageDeviceConnected = usb.c_str();
            }

            (*m_MediaMessageData).mMessageDeviceConnectedVisible = true;
            (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = false;
            (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_USB_ICON;
            (*m_MediaMessageData).mMediamessageIconVisibleState = true;

            (*m_MediaMessageData).mMessage1Visible = false;
            (*m_MediaMessageData).mMessage2Visible = false;
            (*m_MediaMessageData).mMessage3Visible = false;
            (*m_MediaMessageData).mMessage4Visible = false;

            iCurrentConnDeviceTag = iDeviceTag;
            break;
         }
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE:
            (*m_MediaMessageData).mMediaMessage1 = Candera::String(id49);
            (*m_MediaMessageData).mMediaMessage2 = Candera::String(id41);
            (*m_MediaMessageData).mMediaMessage3 = Candera::String(id53);
            (*m_MediaMessageData).mMediaMessage4 = Candera::String(id54);

            (*m_MediaMessageData).mMessageDeviceConnectedVisible = false;
            (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = false;
            (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_IPOD_ICON;
            (*m_MediaMessageData).mMediamessageIconVisibleState = true;

            (*m_MediaMessageData).mMessage1Visible = true;
            (*m_MediaMessageData).mMessage2Visible = true;
            (*m_MediaMessageData).mMessage3Visible = true;
            (*m_MediaMessageData).mMessage4Visible = true;

            iCurrentConnDeviceTag = iDeviceTag;
            break;
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH:

            iCurrentConnDeviceTag = iDeviceTag;
            break;
         default:
            (*m_MediaMessageData).mMediaMessageUpdateFlag = false;
            break;
      }
   }
   else
   {
      if (g_SM_ActiveMediaSource == iDeviceType)
      {
         //clear the list map and set the listtype to invalid
         tclMediaClientHandler::getInstance().vClearListDetailsMaps();
         HMIModelListImplementation::getInstance().ClearListType();
      }

      switch (iDeviceType)
      {
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE:
            (*m_MediaMessageData).mMediaMessage1 = Candera::String(id17);
            (*m_MediaMessageData).mMediaMessage2 = Candera::String(id18);
            (*m_MediaMessageData).mMediaMessage3 = Candera::String(id19);
            (*m_MediaMessageData).mMediaMessage4 = Candera::String(id20);
            //ipod
            (*m_MediaMessageData).mMessageDeviceConnectedVisible = false;
            (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = false;
            (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_IPOD_ICON;
            (*m_MediaMessageData).mMediamessageIconVisibleState = true;

            (*m_MediaMessageData).mMessage1Visible = true;
            (*m_MediaMessageData).mMessage2Visible = true;
            (*m_MediaMessageData).mMessage3Visible = true;
            (*m_MediaMessageData).mMessage4Visible = true;

            (*m_MediaMessageData).mMediaMessageUpdateFlag = true;

            if (iCurrentConnDeviceTag == iDeviceTag)
            {
               vClearMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD);
               vClearGadgetMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD);
            }
            break;
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP:
         {
            std::string usb = "USB";
            if (devNum == 2)
            {
               usb = "Phone USB";
               (*m_MediaMessageData).mMediaMessageDeviceDisconnected = usb.c_str();
            }
            else if (devNum == 1)
            {
               usb = "Audio USB";
               (*m_MediaMessageData).mMediaMessageDeviceDisconnected = usb.c_str();
            }
            else
            {
               (*m_MediaMessageData).mMediaMessageDeviceDisconnected = usb.c_str();
            }
            (*m_MediaMessageData).mMessageDeviceConnectedVisible = false;
            (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = true;
            (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_USB_ICON;
            (*m_MediaMessageData).mMediamessageIconVisibleState = true;

            (*m_MediaMessageData).mMessage1Visible = false;
            (*m_MediaMessageData).mMessage2Visible = false;
            (*m_MediaMessageData).mMessage3Visible = false;
            (*m_MediaMessageData).mMessage4Visible = false;

            int devNum = tclMediaClientHandler::getInstance().getUSBDevNum();
            if (devNum == 0)
            {
               vClearMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB);
               vClearGadgetMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB);
            }
            break;
         }
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH:
         {
            if (iCurrentConnDeviceTag == iDeviceTag)
            {
               vClearMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH);
               vClearGadgetMetadata(MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH);
            }
            break;
         }
         default:
            ETG_TRACE_USR1(("HMIModelComponent::vUpdateDeviceConnectionStatus :default case"));
            break;
      }
   }
   m_MediaMessageData.MarkAllItemsModified();
   bDataUpdated = m_MediaMessageData.SendUpdate(true);

   if (false == bDataUpdated)
   {
      ETG_TRACE_USR1(("HMIModelComponent::vUpdateDeviceConnectionStatus Data updation failed"));
   }
   else
   {
      ETG_TRACE_USR1(("HMIModelComponent::vUpdateDeviceConnectionStatus NULL Pointer"));
   }
}


bool HMIModelComponent::getMediaConnectionPopupVisibilityState(int deviceType)
{
   bool isPopupVisible = true;
   if (deviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH)
   {
      isPopupVisible = false;
   }
   ETG_TRACE_USR4(("HMIModelComponent::getMediaConnectionPopupVisibilityState :isPopupVisible= %d", isPopupVisible));
   return isPopupVisible;
}


/* ***********************************************************
* Function 		: bOnUpdateDeviceStatusOnSelectionMsg
* Description 	: Function will clear the meta-data information when ever source changed and source disconnected
* Parameters 	: int mediaSourceType
* Return value	: void
* ***********************************************************/
bool HMIModelComponent::bOnUpdateDeviceStatusOnSelectionMsg(uint8 u8DeviceType, uint8 u8UsbNum)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateDeviceStatusOnSelectionMsg usbNum = %d", u8UsbNum));
   ETG_TRACE_USR4(("HMIModelComponent::bOnDeviceConnectionStatusMsg(STATUS MESSAGE) ::No Device Connected %d", u8DeviceType));
   POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessage)));
   POST_MSG((COURIER_MESSAGE_NEW(DeActivatePopUpMsg)(Media_MediaPlayer_Pop_Up_system_Pfo_MediaUnsupportedPopupMessageHUB)));
   bool bMsgConsumed = true;

   bool bDataUpdated = false;

   (*m_MediaMessageData).mMediaMessageUpdateFlag = true;
   (*m_MediaMessageData).mMessageDeviceConnectedVisible = false;
   (*m_MediaMessageData).mMessageDeviceDisconnectedVisible = false;
   (*m_MediaMessageData).mMediamessageIconVisibleState = false;

   (*m_MediaMessageData).mMessage1Visible = false;
   (*m_MediaMessageData).mMessage2Visible = false;
   (*m_MediaMessageData).mMessage3Visible = false;
   (*m_MediaMessageData).mMessage4Visible = false;

   (*m_MediaMessageData).mMessageNoDeviceVisible = false;

   //	ActicvateVideorestrictionpopupmessage *poActicvateVideorestrictionpopupmessage = COURIER_MESSAGE_NEW(ActicvateVideorestrictionpopupmessage)();
   //	if (poActicvateVideorestrictionpopupmessage)
   //		poActicvateVideorestrictionpopupmessage->Post();

   //	return true;
   //}

   switch (u8DeviceType)
   {
      case FI_EN_E8DTY_USB:
      {
         std::string usb = "USB";
         if (u8UsbNum == USB_TWO)
         {
            usb = "Phone USB";
            (*m_MediaMessageData).mMediaMessageNoDevice = usb.c_str();
         }
         else if (u8UsbNum == USB_ONE)
         {
            usb = "Audio USB";
            (*m_MediaMessageData).mMediaMessageNoDevice = usb.c_str();
         }
         else
         {
            (*m_MediaMessageData).mMediaMessageNoDevice = usb.c_str();
         }
         (*m_MediaMessageData).mMediaMessage1 = Candera::String(id51);
         (*m_MediaMessageData).mMediaMessage2 = Candera::String(id46);
         (*m_MediaMessageData).mMediaMessage3 = Candera::String(id57);
         (*m_MediaMessageData).mMediaMessage4 = Candera::String(id58);
         (*m_MediaMessageData).mMessageNoDeviceVisible = true;
         (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_USB_ICON;
         (*m_MediaMessageData).mMediamessageIconVisibleState = true;

         bDataUpdated = true;
         break;
      }
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
         bDataUpdated = true;
         (*m_MediaMessageData).mMediaMessage1 = Candera::String(id52);
         (*m_MediaMessageData).mMediaMessage2 = Candera::String(id47); //szAppleNotDeviceConnectedstatus
         (*m_MediaMessageData).mMediaMessage3 = Candera::String(id59); //szAppleNotDeviceConnectedstatus
         (*m_MediaMessageData).mMediaMessage4 = Candera::String(id60); //szAppleNotDeviceConnectedstatus
         (*m_MediaMessageData).mMediamessageIconindex = POPUP_IDX_IPOD_ICON;
         (*m_MediaMessageData).mMediamessageIconVisibleState = true;

         (*m_MediaMessageData).mMessage1Visible = true;
         (*m_MediaMessageData).mMessage2Visible = true;
         (*m_MediaMessageData).mMessage3Visible = true;
         (*m_MediaMessageData).mMessage4Visible = true;
         break;
      case FI_EN_E8DTY_BLUETOOTH:
      {
         bDataUpdated = false;
         (*m_MediaMessageData).mMediaMessage1 = Candera::String(id48); //"BT Device is not connected";
         (*m_MediaMessageData).mMediaMessage2 = Candera::String(id61);
         (*m_MediaMessageData).mMediaMessage2 = Candera::String(id62);
         (*m_MediaMessageData).mMediaMessageUpdateFlag = false;
      }
      break;

      default:
         (*m_MediaMessageData).mMediaMessageUpdateFlag = false;
         break;
   }
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateDeviceStatusOnSelectionMsg DataUpdated:%d", bDataUpdated));

   if (bDataUpdated)
   {
      m_MediaMessageData.MarkAllItemsModified();
      m_MediaMessageData.SendUpdate(true);
      if (FI_EN_E8DTY_BLUETOOTH != u8DeviceType)
      {
         EnableandShowPopUp* poEnableandShowPopUp = COURIER_MESSAGE_NEW(EnableandShowPopUp)(SM_MEDIA_PID_APP_POPUPMEDIAMESSAGE);
         if (poEnableandShowPopUp)
         {
            poEnableandShowPopUp->Post();
         }
      }
   }

   return bMsgConsumed;
}


/* ***********************************************************
* Function 		: vClearMetadata
* Description 	: Function will clear the meta-data information when ever source changed and source disconnected
* Parameters 	: int mediaSourceType
* Return value	: void
* ***********************************************************/
void HMIModelComponent::vClearMetadata(int mediaSourceType)
{
   ETG_TRACE_USR4(("HMIModelComponent::vClearMetadata mediaSourceType:%d", mediaSourceType));
   //m_pdeferedbuffer = m_poNullImageDataPointer;
   switch (mediaSourceType)
   {
      //USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
         (*m_MediaPlayBackData).mArtistName = "";
         (*m_MediaPlayBackData).mTitleName = "";
         (*m_MediaPlayBackData).mAlbumName = "";
         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::ArtistNameItem);
         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::TitleNameItem);
         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::AlbumNameItem);
         m_MediaPlayBackData.SendUpdate(false);
         break;
      //2 is IPOD
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
         (*m_MediaAppleDeviceData).mArtistName = "";
         (*m_MediaAppleDeviceData).mTitleName = "";
         (*m_MediaAppleDeviceData).mAlbumName = "";
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::ArtistNameItem);
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::TitleNameItem);
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::AlbumNameItem);
         m_MediaAppleDeviceData.SendUpdate(false);
         break;
      //BT

      //BT
      case FI_EN_E8DTY_BLUETOOTH:
         (*m_MediaBTDeviceData).mArtistName = "";
         (*m_MediaBTDeviceData).mTitleName = "";
         (*m_MediaBTDeviceData).mAlbumName = "";
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::ArtistNameItem);
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::TitleNameItem);
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::AlbumNameItem);
         m_MediaBTDeviceData.SendUpdate(false);
         break;
   }
}


/************************************************************************
*FUNCTION: 		vUpdatePlayBackStateOnHomeScreen
*DESCRIPTION : Handles updating of play / pause icon for SD, USB and IPOD device.
*PARAMETER : mPlayState
*
*
*RETURNVALUE :
*
************************************************************************/
void HMIModelComponent::vUpdatePlayBackStateOnHomeScreen(uint8_t mPlayState)
{
   vUpdatePlayStateOnHomeScreen(mPlayState);
}


/************************************************************************
*FUNCTION: 		vUpdatePlayStateOnHomeScreen
*DESCRIPTION : Handles updating of play / pause icon for SD, USB and IPOD device.
*PARAMETER : mPlayState
*
*
*RETURNVALUE :
*
************************************************************************/
void HMIModelComponent::vUpdatePlayStateOnHomeScreen(uint8_t mPlayState)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayStateOnHomeScreen mPlayState   %d  ", mPlayState));
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayStateOnHomeScreen srcid  %d ", srcid));
   if (MediaSourceHandling::getInstance().IsMediaSourceRange(srcid) == true)
   {
      ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayStateOnHomeScreen() mPlayState   %d  ", mPlayState));

      if (mPlayState == MPlay_fi_types::T_e8_MPlayState__e8PBS_PAUSED || mPlayState == MPlay_fi_types::T_e8_MPlayState__e8PBS_STOPPED)
      {
         (*_gadgetplaypauseindex).mPlayPauseIndex = MPlay_fi_types::T_e8_MPlayState__e8PBS_PAUSED;
      }
      else
      {
         (*_gadgetplaypauseindex).mPlayPauseIndex = MPlay_fi_types::T_e8_MPlayState__e8PBS_PLAYING;
      }
      _gadgetplaypauseindex.MarkAllItemsModified();
      _gadgetplaypauseindex.SendUpdate();
   }
}


/************************************************************************
*FUNCTION: 		vUpdatePlayStateOnHomeScreen
*DESCRIPTION : Handles updating of play / pause icon for SD, USB and IPOD device.
*PARAMETER : mPlayState
*
*
*RETURNVALUE :
*
************************************************************************/
void HMIModelComponent::vUpdatePlayBackStateOnCamportUSBHomeScreen(uint8_t mPlayState, int srcid)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackStateOnCamportUSBHomeScreen mPlayState   %d  ", mPlayState));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackStateOnCamportUSBHomeScreen srcid  %d ", srcid));
   if (MediaSourceHandling::getInstance().IsMediaSourceRange(srcid) == true)
   {
      ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayStateOnHomeScreen() mPlayState   %d  ", mPlayState));

      if (mPlayState == EXT_MEDIA_PLAY)
      {
         (*_camportusbgadgetplaypauseindex).mPlayPauseIndex = EXT_MEDIA_PLAY;
      }
      else
      {
         (*_camportusbgadgetplaypauseindex).mPlayPauseIndex = EXT_MEDIA_PAUSE;
      }
      _camportusbgadgetplaypauseindex.MarkAllItemsModified();
      _camportusbgadgetplaypauseindex.SendUpdate();
   }
}


void HMIModelComponent::vUpdatePlayBackStateRouterUSBHomeScreen(uint8_t mPlayState, int srcid)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackStateRouterUSBHomeScreen mPlayState   %d  ", mPlayState));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackStateRouterUSBHomeScreen srcid  %d ", srcid));
   if (MediaSourceHandling::getInstance().IsMediaSourceRange(srcid) == true)
   {
      ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackStateRouterUSBHomeScreen() mPlayState   %d  ", mPlayState));
      if (mPlayState == EXT_MEDIA_PLAY)
      {
         (*_routerusbgadgetplaypauseindex).mPlayPauseIndex = EXT_MEDIA_PLAY;
      }
      else
      {
         (*_routerusbgadgetplaypauseindex).mPlayPauseIndex = EXT_MEDIA_PAUSE;
      }
      _routerusbgadgetplaypauseindex.MarkAllItemsModified();
      _routerusbgadgetplaypauseindex.SendUpdate();
   }
}


/************************************************************************
*FUNCTION: 		vUpdatePlayBackState
*DESCRIPTION: 	Handles updating of play/pause icon for SD, USB and IPOD device.
*PARAMETER:     mPlayState
*
*
*RETURNVALUE:
*
************************************************************************/

void HMIModelComponent::vUpdatePlayBackState(uint8_t mPlayState)
{
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState u8activeRegion  %d ", u8activeRegion));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState activemedia source:%d,mplaystate :%d", g_SM_ActiveMediaSource, mPlayState));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState g_SM_MediaFileType:%d", g_SM_MediaFileType));
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState srcid  %d ", srcid));
   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState iSubSrcId  %d ", iSubSrcId));
   int iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState iDeviceType %d", iDeviceType));
   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::vUpdatePlayBackState iActiveSource  %d ", iActiveSource));

      if ((iActiveSource == INTERNALSOURCE && (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A)))
      {
         if ((g_SM_MediaFileType == SM_MEDIA_VIDEO_FILE) && ((iDeviceType != MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE) && (iDeviceType != MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD)))
         {
            //To display Pause Icon in Play mode
            if (mPlayState == MEDIA_C_VALUE_ZERO)
            {
               (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 0;
               m_MediaUSBPlayPauseData.MarkAllItemsModified();
               m_MediaUSBPlayPauseData.SendUpdate();

               (*m_VideoPlayerFullScreeenPause).mPlayIconName = false;
               m_VideoPlayerFullScreeenPause.MarkAllItemsModified();
               m_VideoPlayerFullScreeenPause.SendUpdate(true);
            }
            // to display Play Icon in pause mode
            else if (mPlayState == MEDIA_C_VALUE_ONE || mPlayState == MEDIA_C_VALUE_TWO)
            {
               (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 1;
               m_MediaUSBPlayPauseData.MarkAllItemsModified();
               m_MediaUSBPlayPauseData.SendUpdate();
               (*m_VideoPlayerFullScreeenPause).mPlayIconName = true;
               m_VideoPlayerFullScreeenPause.MarkAllItemsModified();
               m_VideoPlayerFullScreeenPause.SendUpdate(true);
            }
         }
         //Check for the Audio file and video file of apple devices
         else
         {
            if (mPlayState == MEDIA_C_VALUE_ZERO)
            {
               if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB) || (iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP))
               {
                  (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 0;
                  m_MediaUSBPlayPauseData.MarkAllItemsModified();
                  m_MediaUSBPlayPauseData.SendUpdate();
               }
               else if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD) || (iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE))
               {
                  (*m_MediaApplePlayPauseData).mPlayPauseIndex = 0;
                  m_MediaApplePlayPauseData.MarkAllItemsModified();
                  m_MediaApplePlayPauseData.SendUpdate();
               }
               else if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH))
               {
                  (*m_MediaBTPlayPauseData).mPlayPauseIndex = 0;
                  m_MediaBTPlayPauseData.MarkAllItemsModified();
                  m_MediaBTPlayPauseData.SendUpdate(true);
               }
            }
            else if (mPlayState == MEDIA_C_VALUE_ONE || mPlayState == MEDIA_C_VALUE_TWO)
            {
               if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB) || (iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP))
               {
                  (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 1;
                  m_MediaUSBPlayPauseData.MarkAllItemsModified();
                  m_MediaUSBPlayPauseData.SendUpdate();
               }
               else if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD) || (iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE))
               {
                  (*m_MediaApplePlayPauseData).mPlayPauseIndex = 1;
                  m_MediaApplePlayPauseData.MarkAllItemsModified();
                  m_MediaApplePlayPauseData.SendUpdate();
               }
               else if ((iDeviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH))
               {
                  (*m_MediaBTPlayPauseData).mPlayPauseIndex = 1;
                  m_MediaBTPlayPauseData.MarkAllItemsModified();
                  m_MediaBTPlayPauseData.SendUpdate(true);
               }
            }
         }
      }
      else
      {
         //do nothingg
      }
      MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].playPauseState = mPlayState;
   }
   ETG_TRACE_USR1(("HMIModelComponent::vUpdatePlayBackState():exit"));
}


/************************************************************************
*FUNCTION		: SourceButtonHighlight
*DESCRIPTION	: Handles source highlightion based on the current source selected or being played
*PARAMETER		: uint8_t DeviceType
*RETURNVALUE	: void
************************************************************************/
void HMIModelComponent::vSourceButtonHighlight(uint8_t DeviceType, int SubSrcId)
{
   ETG_TRACE_USR4(("HMIModelComponent::vSourceButtonHighlight DeviceType: SubSrcId  :  %d   %d", DeviceType, SubSrcId));
   int iUSbDevNum = 0;
   (*oIpodButtonData).mi8IsActive = false;
   (*oUSBButtonData).mi8IsActive = false;
   (*oUSB2ButtonData).mi8IsActive = false;
   (*oBTButtonData).mi8IsActive = false;
   (*m_AuxButton).mi8IsActive = false; //smart
   (*m_camport1USBDataBinding).mi8IsActive = false;
   (*m_camport1HDMIDataBinding).mi8IsActive = false;

   switch (DeviceType)
   {
      //USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         iUSbDevNum = tclMediaClientHandler::getInstance().GetActiveUSBSource(SubSrcId);
         ETG_TRACE_USR4(("HMIModelComponent::vSourceButtonHighlight USbDevNum is: %d", iUSbDevNum));

         if (iUSbDevNum == 2)
         {
            (*oUSB2ButtonData).mi8IsActive = true;
            (*oUSBButtonData).mi8IsActive = false;
         }
         else if (iUSbDevNum == 1)
         {
            (*oUSBButtonData).mi8IsActive = true;
            (*oUSB2ButtonData).mi8IsActive = false;
         }
         (*oIpodButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;

         break;
      }
      //2 is IPOD
      case FI_EN_E8DTY_IPOD:
      {
         (*oIpodButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart
         g_SM_IsIAPDevice = false;

         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;

         break;
      }
      case FI_EN_E8DTY_IPHONE:
      {
         (*oIpodButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;

         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         (*oBTButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;

         break;
      }
      case FI_EN_E8DTY_AUX:
      {
         (*m_AuxButton).mi8IsActive = true;//smart
         (*oBTButtonData).mi8IsActive = false;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;

         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;

         break;
      }
      case FI_EN_E8DTY_USB_CAMERAPORT://10
      {
         (*m_camport1USBDataBinding).mi8IsActive = true;
         (*m_camport1HDMIDataBinding).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         break;
      }
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI://11
      {
         (*m_camport1HDMIDataBinding).mi8IsActive = true;
         (*m_camport1USBDataBinding).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         break;
      }
      default:
      {
         ETG_TRACE_USR1(("HMIModelComponent::vSourceButtonHighlight Invalid Device"));
         break;
      }
   }

   oUSBButtonData.MarkAllItemsModified();
   oUSBButtonData.SendUpdate(true);

   oUSB2ButtonData.MarkAllItemsModified();
   oUSB2ButtonData.SendUpdate(true);

   oIpodButtonData.MarkAllItemsModified();
   oIpodButtonData.SendUpdate(true);

   oBTButtonData.MarkAllItemsModified();
   oBTButtonData.SendUpdate(true);

   m_AuxButton.MarkAllItemsModified();//smart
   m_AuxButton.SendUpdate(true);

   m_camport1USBDataBinding.MarkAllItemsModified();
   m_camport1USBDataBinding.SendUpdate(true);

   m_camport1HDMIDataBinding.MarkAllItemsModified();
   m_camport1HDMIDataBinding.SendUpdate(true);
}


void HMIModelComponent::SourceButtonHighlight(int iDeviceType, int  usbdevicenum)
{
   ETG_TRACE_USR4(("HMIModelComponent::SourceButtonHighlight iDeviceType:  %d ,usbdevicenum: %d ", iDeviceType, usbdevicenum));

   (*oIpodButtonData).mi8IsActive = false;
   (*oUSBButtonData).mi8IsActive = false;
   (*oUSB2ButtonData).mi8IsActive = false;
   (*oBTButtonData).mi8IsActive = false;
   (*m_AuxButton).mi8IsActive = false; //smart

   switch (iDeviceType)
   {
      //USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         if (usbdevicenum == USB_TWO)
         {
            (*oUSB2ButtonData).mi8IsActive = true;
            (*oUSBButtonData).mi8IsActive = false;
            (*_MuteInfo).mMuteStatus = Candera::String(id77);
            (*m_MediaPlayBackData).mSourceName = Candera::String(id77);
         }
         else if (usbdevicenum == USB_ONE)
         {
            (*oUSBButtonData).mi8IsActive = true;
            (*oUSB2ButtonData).mi8IsActive = false;
            (*_MuteInfo).mMuteStatus = Candera::String(id76);
            (*m_MediaPlayBackData).mSourceName = Candera::String(id76);
         }
         else
         {
            (*_MuteInfo).mMuteStatus = USBText;
            (*m_MediaPlayBackData).mSourceName = USBText;
         }
         (*oIpodButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::SourceNameItem);
         m_MediaPlayBackData.SendUpdate(false);
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);

         break;
      }
      //2 is IPOD
      case FI_EN_E8DTY_IPOD:
      {
         (*oIpodButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart
         g_SM_IsIAPDevice = false;

         break;
      }
      case FI_EN_E8DTY_IPHONE:
      {
         (*oIpodButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oBTButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         (*oBTButtonData).mi8IsActive = true;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;
         (*m_AuxButton).mi8IsActive = false;//smart

         break;
      }
      case FI_EN_E8DTY_AUX:
      {
         (*m_AuxButton).mi8IsActive = true;//smart
         (*oBTButtonData).mi8IsActive = false;
         (*oUSBButtonData).mi8IsActive = false;
         (*oUSB2ButtonData).mi8IsActive = false;
         (*oIpodButtonData).mi8IsActive = false;

         break;
      }

      default:
      {
         ETG_TRACE_USR1(("HMIModelComponent::SourceButtonHighlight Invalid Device"));
         break;
      }
   }

   oUSBButtonData.MarkAllItemsModified();
   oUSBButtonData.SendUpdate(true);

   oUSB2ButtonData.MarkAllItemsModified();
   oUSB2ButtonData.SendUpdate(true);

   oIpodButtonData.MarkAllItemsModified();
   oIpodButtonData.SendUpdate(true);

   oBTButtonData.MarkAllItemsModified();
   oBTButtonData.SendUpdate(true);

   m_AuxButton.MarkAllItemsModified();//smart
   m_AuxButton.SendUpdate(true);
}


void HMIModelComponent::clearSourceButtonHighlight()
{
   ETG_TRACE_USR1(("HMIModelComponent::clearSourceButtonHighlight"));

   (*oIpodButtonData).mi8IsActive = false;
   (*oUSBButtonData).mi8IsActive = false;
   (*oUSB2ButtonData).mi8IsActive = false;
   (*oBTButtonData).mi8IsActive = false;
   (*m_AuxButton).mi8IsActive = false; //smart
   (*m_camport1USBDataBinding).mi8IsActive = false;
   (*m_camport1HDMIDataBinding).mi8IsActive = false;

   oUSBButtonData.MarkAllItemsModified();
   oUSBButtonData.SendUpdate(true);

   oUSB2ButtonData.MarkAllItemsModified();
   oUSB2ButtonData.SendUpdate(true);

   oIpodButtonData.MarkAllItemsModified();
   oIpodButtonData.SendUpdate(true);

   oBTButtonData.MarkAllItemsModified();
   oBTButtonData.SendUpdate(true);

   m_AuxButton.MarkAllItemsModified(); //smart
   m_AuxButton.SendUpdate(true);

   m_camport1USBDataBinding.MarkAllItemsModified();
   m_camport1USBDataBinding.SendUpdate(true);

   m_camport1HDMIDataBinding.MarkAllItemsModified();
   m_camport1HDMIDataBinding.SendUpdate(true);
}


/************************************************************************
*FUNCTION		: vUpdateLastEntertainmentDeviceMsg
*DESCRIPTION	: Handles switching of the screen w.r.t device or source connected
*PARAMETER		: uint8_t DeviceType
*RETURNVALUE	: bool
************************************************************************/
bool HMIModelComponent::vUpdateLastEntertainmentDeviceMsg(uint8_t DeviceType)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateLastEntertainmentDeviceMsg() DeviceType:  %d ", DeviceType));
   int tu8PrevActiveMediaSource = 0;

   /*(*oIpodButtonData).mi8IsActive = false;
   (*oUSBButtonData).mi8IsActive = false;
   (*oBTButtonData).mi8IsActive = false;*/

   tu8PrevActiveMediaSource = g_SM_ActiveMediaSource;
   g_SM_ActiveMediaSource = DeviceType;
   if (tu8PrevActiveMediaSource != g_SM_ActiveMediaSource)
   {
      g_SM_DevConnTypeChanged = 1;
   }
   else
   {
      g_SM_DevConnTypeChanged = 0;
   }
   ETG_TRACE_USR4(("HMIModelComponent::UpdateLastEntertainmentDeviceMsg g_SM_DevConnTypeChanged is %d", g_SM_DevConnTypeChanged));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateLastEntertainmentDeviceMsg g_SM_ActiveMediaSource: %d", g_SM_ActiveMediaSource));

   PlayAudioVideoSong* poPlayAudioVideoSong = COURIER_MESSAGE_NEW(PlayAudioVideoSong)();
   if (poPlayAudioVideoSong != NULL)
   {
      poPlayAudioVideoSong->Post();
   }

   return TRUE;
}


void HMIModelComponent::vUpdateDeviceTypeToSM(uint32 deviceType)
{
   g_SM_ActiveMediaSource = deviceType;
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateDeviceTypeToSM g_SM_ActiveMediaSource: %d ", g_SM_ActiveMediaSource));
}


void HMIModelComponent::vResetDeviceTypeInSM()
{
   g_SM_ActiveMediaSource = 0;
}


/************************************************************************
*FUNCTION		: bOnUpdateNowPlayingStatusOnHomeScreen
*DESCRIPTION	: Handles updating of now playing status of SD, USB and IPOD device.
*PARAMETER		: std::string ArtistName
*PARAMETER		: std::string TitleName
*RETURNVALUE	: void
************************************************************************/
void HMIModelComponent::bOnUpdateNowPlayingStatusOnHomeScreen()
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateNowPlayingStatusOnHomeScreen"));

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusOnHomeScreen u8activeRegion  %d", u8activeRegion));

#ifdef VARIANT_S_FTR_ENABLE_VIDEO_PLAYBACK

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusOnHomeScreen iActiveSource %d", iActiveSource));
      int m_u8ActiveDeviceType = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].LastDeviceType;

      if ((m_u8ActiveDeviceType == FI_EN_E8DTY_USB || m_u8ActiveDeviceType == FI_EN_E8DTY_MTP || m_u8ActiveDeviceType == FI_EN_E8DTY_IPOD ||
            m_u8ActiveDeviceType == FI_EN_E8DTY_IPHONE || m_u8ActiveDeviceType == FI_EN_E8DTY_BLUETOOTH) &&
            (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A))
      {
         vUpdateStatusHomescreen(INTERNALSOURCE);
      }
   }
#else
   vUpdateStatusHomescreen(INTERNALSOURCE);
#endif
}


///************************************************************************
//*FUNCTION		: vUpdateStatusHomescreen
//*DESCRIPTION	: Handles updating of now playing status of SD, USB and IPOD device.
//*PARAMETER		: std::string ArtistName
//*PARAMETER		: std::string TitleName
//*RETURNVALUE	: void
//************************************************************************/
void HMIModelComponent::vUpdateStatusHomescreen(int iActiveSource)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateStatusHomescreen: activesource :%d", iActiveSource));
   std::string aArtistName = "";
   std::string aTitleName = "";

   uint32 u32MediaCategoryType = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].MediaCategoryType;

   if (iActiveSource == INTERNALSOURCE)
   {
      if (u32MediaCategoryType == MPlay_fi_types::T_e8_MPlayCategoryType__e8CTY_EPISODE)
      {
         aArtistName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].AlbumName;
         aTitleName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TitleName;
      }
      else
      {
         aArtistName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].ArtistName;
         aTitleName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TitleName;
      }
   }

   if (aArtistName == "")
   {
      (*_gadgetplaybackaction).mArtistName = Candera::String(id43);
   }
   else
   {
      (*_gadgetplaybackaction).mArtistName = aArtistName.c_str();
   }
   if (aTitleName == "")
   {
      (*_gadgetplaybackaction).mTitleName = Candera::String(id43);
   }
   else
   {
      (*_gadgetplaybackaction).mTitleName = aTitleName.c_str();
   }
   _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::ArtistNameItem);
   _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::TitleNameItem);
   _gadgetplaybackaction.SendUpdate(false);
}


void HMIModelComponent::vUpdateCamporUSBHomescreenStatus()
{
   ETG_TRACE_USR1(("HMIModelComponent::vUpdateCamporUSBHomescreenStatus"));
   std::string aArtistName = "";
   std::string aTitleName = "";

   aArtistName = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].ArtistName;
   aTitleName = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TitleName;

   if (aArtistName == "")
   {
      (*_cabinAgadgetplaybackaction).mArtistName = Candera::String(id43);
   }
   else
   {
      (*_cabinAgadgetplaybackaction).mArtistName = aArtistName.c_str();
   }
   if (aTitleName == "")
   {
      (*_cabinAgadgetplaybackaction).mTitleName = Candera::String(id43);
   }
   else
   {
      (*_cabinAgadgetplaybackaction).mTitleName = aTitleName.c_str();
   }

   _cabinAgadgetplaybackaction.MarkAllItemsModified();
   _cabinAgadgetplaybackaction.SendUpdate(true);
}


void HMIModelComponent::vUpdateRouterUSBHomescreenStatus()
{
   ETG_TRACE_USR1(("HMIModelComponent::vUpdateRouterUSBHomescreenStatus"));
   std::string aArtistName = "";
   std::string aTitleName = "";
   aArtistName = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].ArtistName;
   aTitleName = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TitleName;
   if (aArtistName == "")
   {
      (*_routergadgetplaybackaction).mArtistName = Candera::String(id43);
   }
   else
   {
      (*_routergadgetplaybackaction).mArtistName = aArtistName.c_str();
   }
   if (aTitleName == "")
   {
      (*_routergadgetplaybackaction).mTitleName = Candera::String(id43);
   }
   else
   {
      (*_routergadgetplaybackaction).mTitleName = aTitleName.c_str();
   }
   _routergadgetplaybackaction.MarkAllItemsModified();
   _routergadgetplaybackaction.SendUpdate(true);
}


/************************************************************************
*FUNCTION		: vOnUpdateNowPlayingMetaDataInfoStatus
*DESCRIPTION	: Handles updating of now playing status of SD, USB and IPOD device.
*PARAMETER		: std::string ArtistName
*PARAMETER		: std::string TitleName
*PARAMETER		: std::string AlbumName
*PARAMETER		: int32 MediaFileType
*PARAMETER		: Candera::String CurrentTrackNumber
*PARAMETER		: Courier::UInt8 CurrentPlayingDeviceType
*PARAMETER		: Candera::String TotalTime
*PARAMETER		: Courier::UInt8 LastDeviceType
*RETURNVALUE	: void
************************************************************************/
void HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus()
{
   ETG_TRACE_USR1(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus"));

   int32 u32MediaFileType = 2;

   std::string aArtistName = "";
   std::string aTitleName = "";
   std::string aAlbumName = "";
   std::string aCurrentTrackNo = "";

   u32MediaFileType = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].MediaFileType;
   aArtistName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].ArtistName;
   aTitleName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TitleName;
   aAlbumName = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].AlbumName;
   aCurrentTrackNo = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].CurrentTrackNumber;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus Artist name : %s", aArtistName.c_str()));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus Title name : %s", aTitleName.c_str()));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus Album name : %s", aAlbumName.c_str()));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus Track Number : %s", aCurrentTrackNo.c_str()));

   m_u8ActiveDeviceType = MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].LastDeviceType;

   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus m_u8ActiveDeviceType %d", m_u8ActiveDeviceType));

   if (g_SM_ActiveMediaSource != m_u8ActiveDeviceType)
   {
      ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus (Clearing Meta-data): g_SM_ActiveMediaSource(%d) != m_u8ActiveDeviceType (%d)"
                      , g_SM_ActiveMediaSource, m_u8ActiveDeviceType));
      vClearMetadata(g_SM_ActiveMediaSource);
   }

   switch (m_u8ActiveDeviceType)
   {
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         int devNum = tclMediaClientHandler::getInstance().getUSBDevNum();

         ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus:devNum %d", devNum));

         if (devNum == 1)
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id76);
            (*m_MediaPlayBackData).mSourceName = Candera::String(id76);
         }
         else if (devNum == 2)
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id77);
            (*m_MediaPlayBackData).mSourceName = Candera::String(id77);
         }
         else
         {
            (*_MuteInfo).mMuteStatus = USBText;
            (*m_MediaPlayBackData).mSourceName = USBText;
         }

         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::SourceNameItem);
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);

         if ((g_SM_MediaFileType == SM_MEDIA_AUDIO_FILE) ||
               ((g_SM_MediaFileType == SM_MEDIA_VIDEO_FILE) && (u8activeRegion == REGION_CABIN_A)))
         {
            if (aArtistName == "")
            {
               (*m_MediaPlayBackData).mArtistName = Candera::String(id43);
            }
            else
            {
               (*m_MediaPlayBackData).mArtistName = aArtistName.c_str();
            }
            if (aTitleName == "")
            {
               (*m_MediaPlayBackData).mTitleName = Candera::String(id43);
            }
            else
            {
               (*m_MediaPlayBackData).mTitleName = aTitleName.c_str();
            }
            if (aAlbumName == "")
            {
               (*m_MediaPlayBackData).mAlbumName = Candera::String(id43);
            }
            else
            {
               (*m_MediaPlayBackData).mAlbumName = aAlbumName.c_str();
            }
            m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::ArtistNameItem);
            m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::TitleNameItem);
            m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::AlbumNameItem);
            (*m_MediaPlayBackData).mCurrentTrackNo = aCurrentTrackNo.c_str();
            m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::CurrentTrackNoItem);
         }
         else if (g_SM_MediaFileType == SM_MEDIA_VIDEO_FILE)
         {
            if (aAlbumName == "")
            {
               (*m_MediaVideoPlayBackData).mVideoTitle = Candera::String(id43);
            }
            else
            {
               (*m_MediaVideoPlayBackData).mVideoTitle = aAlbumName.c_str();
            }
            if (aTitleName == "")
            {
               (*m_MediaVideoPlayBackData).mVideoEpisode = Candera::String(id43);
            }
            else
            {
               (*m_MediaVideoPlayBackData).mVideoEpisode = aTitleName.c_str();
            }
            m_MediaVideoPlayBackData.MarkAllItemsModified();
            m_MediaVideoPlayBackData.SendUpdate(true);
         }
         else
         {
         }
         m_MediaPlayBackData.SendUpdate(false);
         break;
      }

      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         (*m_MediaAppleDeviceData).mSourceName = IPODText;
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::SourceNameItem);
         //}
         (*_MuteInfo).mMuteStatus = IPODText;
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);
         if (g_SM_MediaFileType == SM_MEDIA_AUDIO_FILE)
         {
            if (aArtistName == "")
            {
               (*m_MediaAppleDeviceData).mArtistName = Candera::String(id43);
            }
            else
            {
               (*m_MediaAppleDeviceData).mArtistName = aArtistName.c_str();
            }
            if (aTitleName == "")
            {
               (*m_MediaAppleDeviceData).mTitleName = Candera::String(id43);
            }
            else
            {
               (*m_MediaAppleDeviceData).mTitleName = aTitleName.c_str();
            }
            if (aAlbumName == "")
            {
               (*m_MediaAppleDeviceData).mAlbumName = Candera::String(id43);
            }
            else
            {
               (*m_MediaAppleDeviceData).mAlbumName = aAlbumName.c_str();
            }
            m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::ArtistNameItem);
            m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::TitleNameItem);
            m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::AlbumNameItem);
            (*m_MediaAppleDeviceData).mCurrentTrackNo = aCurrentTrackNo.c_str();
            m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::CurrentTrackNoItem);
         }
         m_MediaAppleDeviceData.SendUpdate(false);
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id75);
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);
         ETG_TRACE_FATAL(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus ShuffleSupported  %d", clMetaData::getInstance().getShuffleSupported()));
         if (clMetaData::getInstance().getShuffleSupported() == false)
         {
            (*m_BTNowplayIcons).mShuffleIconStatus = false;
            (*m_MediaBTShuffleData).mShuffleIndex = 2;
            m_MediaBTShuffleData.MarkAllItemsModified();
            m_MediaBTShuffleData.SendUpdate(true);
         }
         else
         {
            (*m_BTNowplayIcons).mShuffleIconStatus = true;
         }
         ETG_TRACE_FATAL(("HMIModelComponent::vOnUpdateNowPlayingMetaDataInfoStatus RepeatSupported  %d", clMetaData::getInstance().getRepeatSupported()));

         if (clMetaData::getInstance().getRepeatSupported() == false)
         {
            (*m_BTNowplayIcons).mRepeatIconStatus = false;
            (*m_MediaBTRepeatData).mRepeatIndex = 3;
            m_MediaBTRepeatData.MarkAllItemsModified();
            m_MediaBTRepeatData.SendUpdate(true);
         }
         else
         {
            (*m_BTNowplayIcons).mRepeatIconStatus = true;
         }

         if (clMetaData::getInstance().getBrowseSupported() == false)
         {
            (*m_BTNowplayIcons).mBrowseIconStatus = false;
         }
         else
         {
            (*m_BTNowplayIcons).mBrowseIconStatus = true;
         }
         m_BTNowplayIcons.MarkItemModified(ItemKey::BTNowplayIcons::ShuffleIconStatusItem);
         m_BTNowplayIcons.MarkItemModified(ItemKey::BTNowplayIcons::RepeatIconStatusItem);
         m_BTNowplayIcons.MarkItemModified(ItemKey::BTNowplayIcons::BrowseIconStatusItem);
         m_BTNowplayIcons.SendUpdate(true);
         if (aArtistName == "")
         {
            (*m_MediaBTDeviceData).mArtistName = Candera::String(id43);
         }
         else
         {
            (*m_MediaBTDeviceData).mArtistName = aArtistName.c_str();
         }
         if (aTitleName == "")
         {
            (*m_MediaBTDeviceData).mTitleName = Candera::String(id43);
         }
         else
         {
            (*m_MediaBTDeviceData).mTitleName = aTitleName.c_str();
         }
         if (aAlbumName == "")
         {
            (*m_MediaBTDeviceData).mAlbumName = Candera::String(id43);
         }
         else
         {
            (*m_MediaBTDeviceData).mAlbumName = aAlbumName.c_str();
         }

         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::ArtistNameItem);
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::TitleNameItem);
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::AlbumNameItem);
         (*m_MediaBTDeviceData).mCurrentTrackNo = aCurrentTrackNo.c_str();
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::CurrentTrackNoItem);
         m_MediaBTDeviceData.SendUpdate(false);
         break;
      }

      default:
         break;
   }
}


/************************************************************************
*FUNCTION		: bOnUpdateNowPlayingStatus
*DESCRIPTION	: Handles updating of now playing status of SD, USB and IPOD device.
*PARAMETER		: std::string ArtistName
*PARAMETER		: std::string TitleName
*PARAMETER		: std::string AlbumName
*PARAMETER		: int32 MediaFileType
*PARAMETER		: Candera::String CurrentTrackNumber
*PARAMETER		: Courier::UInt8 CurrentPlayingDeviceType
*PARAMETER		: Candera::String TotalTime
*PARAMETER		: Courier::UInt8 LastDeviceType
*RETURNVALUE	: void
************************************************************************/
void HMIModelComponent::bOnUpdateNowPlayingStatus()
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateNowPlayingStatus"));

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatus u8activeRegion  %d ", u8activeRegion));
   // we get activesource after guistart up and nowplaying
#ifdef VARIANT_S_FTR_ENABLE_VIDEO_PLAYBACK //for inf4cv 
   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      if ((iActiveSource == INTERNALSOURCE && (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A)))
      {
         vOnUpdateNowPlayingMetaDataInfoStatus();
      }
   }

#else
   vOnUpdateNowPlayingMetaDataInfoStatus();

#endif
}


/************************************************************************
*FUNCTION		: bOnVideoNowPlayingStatusMsg
*DESCRIPTION	: Handles updating of Video now playing status
*PARAMETER		: std::string VideoTitle
*PARAMETER		: std::string VideoEpisode
*RETURNVALUE	: void
************************************************************************/
void HMIModelComponent::bOnVideoNowPlayingStatusMsg(std::string cVideoTitle, std::string cVideoEpisode)
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnVideoNowPlayingStatusMsg"));
   bool bMsgConsumed = false; // Holds true if Model consumes the message

   (*m_MediaVideoPlayBackData).mVideoTitle = cVideoTitle.c_str();
   (*m_MediaVideoPlayBackData).mVideoEpisode = cVideoEpisode.c_str();
   m_MediaVideoPlayBackData.MarkAllItemsModified();
   m_MediaVideoPlayBackData.SendUpdate(true);

   if (!m_MediaVideoPlayBackData.SendUpdate(true))
   {
      ETG_TRACE_USR1(("HMIModelComponent::bOnVideoNowPlayingStatusMsg(): Data updation failed"));
   }

   bMsgConsumed = true;
}


/************************************************************************
*FUNCTION		: vUpdateGadgetSourceName
*DESCRIPTION	: Handles updating of media source name on homescreen
*PARAMETER		: std::string source
*RETURNVALUE	: void
************************************************************************/
//void HMIModelComponent::vUpdateGadgetSourceName(std::string source, uint8 srcID)
//{
//   ETG_TRACE_USR4(("media_tclMediaModel : vUpdateGadgetSourceName()"))
//
//   (*_gadgetplaybackaction).mMedia_Name = source.c_str();
//   (*_gadgetplaybackaction).mMedia_Type = srcID;
//
//   _gadgetplaybackaction.MarkAllItemsModified();
//   _gadgetplaybackaction.SendUpdate(true);
//
//
//}


void HMIModelComponent::vUpdateNextPerviousNormalState()
{
   ETG_TRACE_USR1(("HMIModelComponent::vUpdateNextPerviousNormalState"));
   bool bMuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNextPerviousNormalState bMuteStatus %d ", bMuteStatus));

   if (bMuteStatus == true)
   {
      vUpdateNextPerviousIconStatus();
   }
   else
   {
      vUpdateMuteStateVisibleGadget(true, false, true);
   }
}


void HMIModelComponent::vUpdateNextPerviousIconStatus()
{
   //To get the current mute status
   bool bMuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   int iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   uint8 iactiveMediaSource = MediaSourceHandling::getInstance().GetActiveMediaSource(srcid, iDeviceType);

   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNextPerviousIconStatus bMuteStatus %d", bMuteStatus));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNextPerviousIconStatus srcid : %d", srcid));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNextPerviousIconStatus iSubSrcId %d", iSubSrcId));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNextPerviousIconStatus iDeviceType %d, iactiveMediaSource = %d, g_SM_ActiveMediaSource = %d", iDeviceType, iactiveMediaSource, g_SM_ActiveMediaSource));

   if (bMuteStatus == true)
   {
      if ((iactiveMediaSource == FI_EN_E8DTY_USB) || (iactiveMediaSource == FI_EN_E8DTY_MTP) || (iactiveMediaSource == FI_EN_E8DTY_BLUETOOTH) || (iactiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT) || (iactiveMediaSource == FI_EN_E8DTY_ROUTER_USB))
      {
         vUpdateMuteStateVisibleGadget(false, true, false);
      }
      else if (iactiveMediaSource == FI_EN_E8DTY_AUX)
      {
         vUpdateMuteStateVisibleGadget(false, false, false);
      }
      else if (g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI || g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2 || g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3)
      {
         vUpdateCabinGadgetVisibility();
      }
      else
      {
         vUpdateMuteStateVisibleGadget(true, false, true);
      }
   }
   else
   {
      if (iactiveMediaSource == FI_EN_E8DTY_AUX)
      {
         vUpdateMuteStateVisibleGadget(false, false, false);
      }
      else if (g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI || g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2 || g_SM_ActiveMediaSource == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3)
      {
         vUpdateCabinGadgetVisibility();
      }
      else
      {
         vUpdateMuteStateVisibleGadget(true, false, true);
      }
   }
}


void HMIModelComponent::vUpdateMuteStatus()
{
   bool bMuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateMuteStatus:mutestatus : %d", bMuteStatus));
   if (bMuteStatus == true)
   {
      (*m_MuteStateVisibleGadget).mMuteImageVisibility = true;
   }
   else
   {
      (*m_MuteStateVisibleGadget).mMuteImageVisibility = false;
   }
   m_MuteStateVisibleGadget.MarkAllItemsModified();
   m_MuteStateVisibleGadget.SendUpdate(true);
}


/**************************************************************************
(SWFL-636,SWFL-637)
**************************************************************************/
bool HMIModelComponent::bOnUpdateRepeatModeStatus(const uint32& u32RepeatMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateRepeatModeStatus repeatmode : %d", u32RepeatMode));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   m_u32RepeatMode = u32RepeatMode;
   switch (m_u32RepeatMode)
   {
      case EN_MEDIA_REPEATMODE_TRACK:
      {
         vOnUpdateRepeatTrackIcon();
         break;
      }
      case EN_MEDIA_REPEATMODE_LIST:
      {
         vOnUpdateRepeatListIcon();
         break;
      }
      case EN_MEDIA_REPEATMODE_OFF:
      {
         vOnUpdateRepeatOffIcon();
         break;
      }
      default:
         break;
   }

   return bMsgConsumed;
}


/**************************************************************************
cameraport repeat status
**************************************************************************/
bool HMIModelComponent::bOnUpdateRepeatModeCamportUSBStatus(const uint32& u32RepeatMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateRepeatModeCamportUSBStatus repeatmode: %d", u32RepeatMode));

   bool bMsgConsumed = false; // Holds true if Model consumes the message
   switch (u32RepeatMode)
   {
      case EN_MEDIA_REPEATMODE_TRACK:
      {
         (*m_MediaUSBCameraPortRepeatData).mRepeatIndex = 1;
         m_MediaUSBCameraPortRepeatData.MarkAllItemsModified();
         m_MediaUSBCameraPortRepeatData.SendUpdate(true);

         break;
      }
      case EN_MEDIA_REPEATMODE_OFF:
      {
         (*m_MediaUSBCameraPortRepeatData).mRepeatIndex = 2;
         m_MediaUSBCameraPortRepeatData.MarkAllItemsModified();
         m_MediaUSBCameraPortRepeatData.SendUpdate(true);

         break;
      }
      case EN_MEDIA_REPEATMODE_LIST:
      {
         (*m_MediaUSBCameraPortRepeatData).mRepeatIndex = 0;
         m_MediaUSBCameraPortRepeatData.MarkAllItemsModified();
         m_MediaUSBCameraPortRepeatData.SendUpdate(true);

         break;
      }
      default:
         break;
   }

   return bMsgConsumed;
}


void HMIModelComponent::vOnUpdateRepeatTrackIcon()
{
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateRepeatTrackIcon g_SM_ActiveMediaSource: %d", g_SM_ActiveMediaSource));

   switch (g_SM_ActiveMediaSource)
   {
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         (*m_MediaUSBRepeatData).mRepeatIndex = 1;
         m_MediaUSBRepeatData.MarkAllItemsModified();
         m_MediaUSBRepeatData.SendUpdate(true);
         break;
      }
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         (*m_MediaAppleRepeatData).mRepeatIndex = 1;
         m_MediaAppleRepeatData.MarkAllItemsModified();
         m_MediaAppleRepeatData.SendUpdate(true);
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         if (clMetaData::getInstance().getRepeatSupported() == false)
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 3;
         }
         else
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 1;
         }
         m_MediaBTRepeatData.MarkAllItemsModified();
         m_MediaBTRepeatData.SendUpdate(true);
         break;
      }
      default:
         break;
   }
}


void HMIModelComponent::vOnUpdateRepeatListIcon()
{
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateRepeatListIcon g_SM_ActiveMediaSource: %d", g_SM_ActiveMediaSource));
   switch (g_SM_ActiveMediaSource)
   {
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         (*m_MediaUSBRepeatData).mRepeatIndex = 0;
         m_MediaUSBRepeatData.MarkAllItemsModified();
         m_MediaUSBRepeatData.SendUpdate(true);
         break;
      }

      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         (*m_MediaAppleRepeatData).mRepeatIndex = 0;
         m_MediaAppleRepeatData.MarkAllItemsModified();
         m_MediaAppleRepeatData.SendUpdate(true);
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         if (clMetaData::getInstance().getRepeatSupported() == false)
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 3;
         }
         else
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 0;
         }

         m_MediaBTRepeatData.MarkAllItemsModified();
         m_MediaBTRepeatData.SendUpdate(true);
         break;
      }
      default:
         break;
   }
}


void HMIModelComponent::vOnUpdateRepeatOffIcon()
{
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateRepeatOffIcon g_SM_ActiveMediaSource: %d", g_SM_ActiveMediaSource));

   switch (g_SM_ActiveMediaSource)
   {
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP://mcode
      {
         (*m_MediaUSBRepeatData).mRepeatIndex = 2;
         m_MediaUSBRepeatData.MarkAllItemsModified();
         m_MediaUSBRepeatData.SendUpdate(true);
         break;
      }
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         (*m_MediaAppleRepeatData).mRepeatIndex = 2;
         m_MediaAppleRepeatData.MarkAllItemsModified();
         m_MediaAppleRepeatData.SendUpdate(true);
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateRepeatOffIcon RepeatSupport: %d", clMetaData::getInstance().getRepeatSupported()));

         if (clMetaData::getInstance().getRepeatSupported() == false)
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 3;
         }
         else
         {
            (*m_MediaBTRepeatData).mRepeatIndex = 2;
         }

         m_MediaBTRepeatData.MarkAllItemsModified();
         m_MediaBTRepeatData.SendUpdate(true);
      }
      default:
         break;
   }
}


/**************************************************************************
router repeat status
**************************************************************************/
bool HMIModelComponent::bOnUpdateRepeatModeRouterStatus(const uint32& u32RepeatMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateRepeatModeRouterStatus repeatmode: %d", u32RepeatMode));

   bool bMsgConsumed = false; // Holds true if Model consumes the message
   switch (u32RepeatMode)
   {
      case EN_MEDIA_REPEATMODE_TRACK:
      {
         (*m_MediaUSBRouterRepeatData).mRepeatIndex = 1;
         m_MediaUSBRouterRepeatData.MarkAllItemsModified();
         m_MediaUSBRouterRepeatData.SendUpdate(true);
         break;
      }
      case EN_MEDIA_REPEATMODE_OFF:
      {
         (*m_MediaUSBRouterRepeatData).mRepeatIndex = 2;
         m_MediaUSBRouterRepeatData.MarkAllItemsModified();
         m_MediaUSBRouterRepeatData.SendUpdate(true);

         break;
      }
      case EN_MEDIA_REPEATMODE_LIST:
      {
         (*m_MediaUSBRouterRepeatData).mRepeatIndex = 0;
         m_MediaUSBRouterRepeatData.MarkAllItemsModified();
         m_MediaUSBRouterRepeatData.SendUpdate(true);

         break;
      }
      default:
         break;
   }

   return bMsgConsumed;
}


/****************************************************************************
(SWFL-638)
*****************************************************************************/
bool HMIModelComponent::bOnUpdatePlayBackModeStatus(const uint32& u32PlayBackMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayBackModeStatus playbackmode:%d ", u32PlayBackMode));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayBackModeStatus Activemediasource: %d", g_SM_ActiveMediaSource));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayBackModeStatus BLUETOOTH shuffle support %d", clMetaData::getInstance().getShuffleSupported()));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   switch (u32PlayBackMode)
   {
      case SHUFFLE_OFF:
      {
         if ((g_SM_ActiveMediaSource == 1) || (g_SM_ActiveMediaSource == 6))
         {
            (*m_MediaUSBShuffleData).mShuffleIndex = 0;
            m_MediaUSBShuffleData.MarkAllItemsModified();
            m_MediaUSBShuffleData.SendUpdate();
         }
         else if ((g_SM_ActiveMediaSource == 2) || (g_SM_ActiveMediaSource == 5))
         {
            (*m_MediaAppleShuffleData).mShuffleIndex = 0;
            m_MediaAppleShuffleData.MarkAllItemsModified();
            m_MediaAppleShuffleData.SendUpdate();
         }
         else if ((g_SM_ActiveMediaSource == 4))
         {
            if (clMetaData::getInstance().getShuffleSupported() == false)
            {
               (*m_MediaBTShuffleData).mShuffleIndex = 2;
            }
            else
            {
               (*m_MediaBTShuffleData).mShuffleIndex = 0;
            }
            m_MediaBTShuffleData.MarkAllItemsModified();
            m_MediaBTShuffleData.SendUpdate(true);
         }
         else
         {
            ETG_TRACE_USR1(("HMIModelComponent::bOnUpdatePlayBackModeStatus : SHUFFLE OFF"));
         }

         break;
      }
      case SHUFFLE_ON:
      {
         if ((g_SM_ActiveMediaSource == 1) || (g_SM_ActiveMediaSource == 6))
         {
            (*m_MediaUSBShuffleData).mShuffleIndex = 1;
            m_MediaUSBShuffleData.MarkAllItemsModified();
            m_MediaUSBShuffleData.SendUpdate();
         }
         else if ((g_SM_ActiveMediaSource == 2) || (g_SM_ActiveMediaSource == 5))
         {
            (*m_MediaAppleShuffleData).mShuffleIndex = 1;
            m_MediaAppleShuffleData.MarkAllItemsModified();
            m_MediaAppleShuffleData.SendUpdate();
         }
         else if ((g_SM_ActiveMediaSource == 4))
         {
            if (clMetaData::getInstance().getShuffleSupported() == false)
            {
               (*m_MediaBTShuffleData).mShuffleIndex = 2;
            }
            else
            {
               (*m_MediaBTShuffleData).mShuffleIndex = 1;
            }
            m_MediaBTShuffleData.MarkAllItemsModified();
            m_MediaBTShuffleData.SendUpdate();
         }
         else
         {
            ETG_TRACE_USR1(("HMIModelComponent::bOnUpdatePlayBackModeStatus: SHUFFLE ON"));
         }

         break;
      }
      default:
         break;
   }

   return true;
}


/****************************************************************************
camera port shuffle status
*****************************************************************************/
bool HMIModelComponent::bOnUpdateShuffleModeCamportUSBStatus(const uint32& u32PlayBackMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateShuffleModeCamportUSBStatus playbckmode %d ", u32PlayBackMode));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   switch (u32PlayBackMode)
   {
      case SHUFFLE_OFF:
      {
         (*m_MediaUSBCameraPortShuffleData).mShuffleIndex = 0;
         m_MediaUSBCameraPortShuffleData.MarkAllItemsModified();
         m_MediaUSBCameraPortShuffleData.SendUpdate();

         break;
      }
      case SHUFFLE_ON:
      {
         (*m_MediaUSBCameraPortShuffleData).mShuffleIndex = 1;
         m_MediaUSBCameraPortShuffleData.MarkAllItemsModified();
         m_MediaUSBCameraPortShuffleData.SendUpdate();

         break;
      }
      default:
         break;
   }

   return true;
}


/****************************************************************************
router shuffle status
*****************************************************************************/
bool HMIModelComponent::bOnUpdateShuffleModeRouterStatus(const uint32& u32PlayBackMode)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateShuffleModeRouterStatus playbckmode %d ", u32PlayBackMode));

   bool bMsgConsumed = false; // Holds true if Model consumes the message

   switch (u32PlayBackMode)
   {
      case SHUFFLE_OFF:
      {
         (*m_MediaUSBRouterShuffleData).mShuffleIndex = 0;
         m_MediaUSBRouterShuffleData.MarkAllItemsModified();
         m_MediaUSBRouterShuffleData.SendUpdate();

         break;
      }
      case SHUFFLE_ON:
      {
         (*m_MediaUSBRouterShuffleData).mShuffleIndex = 1;
         m_MediaUSBRouterShuffleData.MarkAllItemsModified();
         m_MediaUSBRouterShuffleData.SendUpdate();
         break;
      }
      default:
         break;
   }

   return true;
}


/************************************************************************
*FUNCTION: 		bOnUpdateAlbumArt
*DESCRIPTION: 	Handles updating of albumart for SD, USB and IPOD device.
*PARAMETER:
* 				::std::vector< uint8 > u8ImageData,u32ImageDataSize
*
*
*RETURNVALUE: 	void
*
************************************************************************/
void HMIModelComponent::bOnUpdateAlbumArt(const ::std::vector< uint8 > u8ImageData, uint32 u32ImageDataSize)
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateAlbumArt"));
   Candera::Bitmap::SharedPointer bmp = ImageLoader::loadBitmapData(u8ImageData.data(), u32ImageDataSize);

   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateAlbumArt m_u8ActiveDeviceType: %d", m_u8ActiveDeviceType));
   switch (m_u8ActiveDeviceType)
   {
      //For device type 1 -> USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         if (bmp != NULL)
         {
            (*mID3CoverData).mImage = ImageLoader::createImage(bmp);
            mID3CoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
            mID3CoverData.SendUpdate(true);

            (*mID3GadgetCoverData).mGadgetImage = ImageLoader::createImage(bmp);
            mID3GadgetCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
            mID3GadgetCoverData.SendUpdate(true);
         }
         else
         {
            mID3CoverData.SendUpdate(false);
            mID3GadgetCoverData.SendUpdate(false);
         }
         break;
      }

      //For device type 2 -> APPLEDEVICE
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         if (bmp != NULL)
         {
            (*ID3CoverDataForApple).mImage = ImageLoader::createImage(bmp);
            ID3CoverDataForApple.MarkItemModified(ItemKey::ID3CoverDataForApple::ImageItem);
            ID3CoverDataForApple.SendUpdate(true);

            (*mID3GadgetCoverData).mGadgetImage = ImageLoader::createImage(bmp);
            mID3GadgetCoverData.MarkItemModified(ItemKey::ID3CoverDataForApple::ImageItem);
            mID3GadgetCoverData.SendUpdate(true);
         }
         else
         {
            ID3CoverDataForApple.SendUpdate(false);
            mID3GadgetCoverData.SendUpdate(false);
         }
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         if (bmp != NULL)
         {
            (*ID3CoverDataForBT).mImage = ImageLoader::createImage(bmp);
            ID3CoverDataForBT.MarkItemModified(ItemKey::ID3CoverDataForApple::ImageItem);
            ID3CoverDataForBT.SendUpdate(true);

            (*mID3GadgetCoverData).mGadgetImage = ImageLoader::createImage(bmp);
            mID3GadgetCoverData.MarkItemModified(ItemKey::ID3CoverDataForApple::ImageItem);
            mID3GadgetCoverData.SendUpdate(true);
         }
         else
         {
            ID3CoverDataForBT.SendUpdate(false);
            mID3GadgetCoverData.SendUpdate(false);
         }

         break;
      }
      case FI_EN_E8DTY_AUX:
      {
         char buffer[200];
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);

         (*mID3GadgetCoverData).mGadgetImage = bmp;
         mID3GadgetCoverData.MarkAllItemsModified();
         mID3GadgetCoverData.SendUpdate(true);
         break;
      }
      //For device type 3 -> SD
      case FI_EN_E8DTY_SD:
      {
         break;
      }
      default:
         break;
   }
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateAlbumArt : exit"));
}


void HMIModelComponent::vOnUpdateAlbumArtForUsbCamport()
{
   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbCamport() Entered "));
   std::string strAlbumArtPath("/tmp/CMP/AlbumArt/AlbumArt");

   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbCamport() imgfile %s ", strAlbumArtPath.c_str()));
   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbCamport() strAlbumArtPath.size() %d ", strAlbumArtPath.size()));

   if (strAlbumArtPath.size() > 0)
   {
      Candera::Bitmap::SharedPointer bmpImage = ImageLoader::loadBitmapFile(strAlbumArtPath.c_str());
      if (bmpImage != NULL)
      {
         ETG_TRACE_USR1(("HMIModelComponent::vOnUpdateAlbumArtForUsbCamport() Image loaded successfully"));
         (*mID3CamportUSBCoverData).mCamportAlbumImage = ImageLoader::createImage(bmpImage);
         mID3CamportUSBCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3CamportUSBCoverData.SendUpdate(true);

         //newdatabind for gadget camport
         (*mID3CamportUSBGadgetCoverData).mGadgetImage = ImageLoader::createImage(bmpImage);
         mID3CamportUSBGadgetCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3CamportUSBGadgetCoverData.SendUpdate(true);
      }
   }
}


void HMIModelComponent::vOnUpdateAlbumArtForUsbRouter()
{
   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbRouter() Entered "));
   std::string strAlbumArtPath("/tmp/CMR/AlbumArt/AlbumArt");

   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbRouter() imgfile %s ", strAlbumArtPath.c_str()));
   ETG_TRACE_USR4((" HMIModelComponent::vOnUpdateAlbumArtForUsbCamport() strAlbumArtPath.size() %d ", strAlbumArtPath.size()));

   if (strAlbumArtPath.size() > 0)
   {
      Candera::Bitmap::SharedPointer bmpImage = ImageLoader::loadBitmapFile(strAlbumArtPath.c_str());
      if (bmpImage != NULL)
      {
         ETG_TRACE_USR1(("HMIModelComponent::vOnUpdateAlbumArtForUsbRouter() Image loaded successfully"));
         (*mID3RouterUSBCoverData).mRouterAlbumImage = ImageLoader::createImage(bmpImage);
         mID3RouterUSBCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3RouterUSBCoverData.SendUpdate(true);

         //newdatabinding for gadget Router
         (*mID3RouterUSBGadgetCoverData).mGadgetImage = ImageLoader::createImage(bmpImage);
         mID3RouterUSBGadgetCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3RouterUSBGadgetCoverData.SendUpdate(true);
      }
   }
}


/************************************************************************
*FUNCTION: 		bOnUpdateAlbumArtDefaultforInternalMediaSources
*DESCRIPTION: 	Handles updating of default albumart for SD, USB and IPOD device.
*PARAMETER:
* 				void
*
*
*RETURNVALUE: 	void
*
************************************************************************/
void HMIModelComponent::bOnUpdateAlbumArtDefaultforInternalMediaSources()
{
   char buffer[200];
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateAlbumArt ActiveDeviceType: %d", m_u8ActiveDeviceType));
   switch (m_u8ActiveDeviceType)
   {
      //For device type 1 -> USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);
         //nowplaying
         (*mID3CoverData).mImage = bmp;
         mID3CoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3CoverData.SendUpdate(true);

         //for gadget
         bOnUpdateAlbumArtDefaultforGadgetMedia();
         break;
      }

      //For device type 2 -> APPLEDEVICE
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);

         (*ID3CoverDataForApple).mImage = bmp;
         ID3CoverDataForApple.MarkAllItemsModified();
         ID3CoverDataForApple.SendUpdate(true);

         bOnUpdateAlbumArtDefaultforGadgetMedia();

         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);

         (*ID3CoverDataForBT).mImage = bmp;
         ID3CoverDataForBT.MarkAllItemsModified();
         ID3CoverDataForBT.SendUpdate(true);

         bOnUpdateAlbumArtDefaultforGadgetMedia();
         break;
      }
      //For device type 3 -> SD
      case FI_EN_E8DTY_AUX:
      {
         bOnUpdateAlbumArtDefaultforGadgetMedia();
         break;
      }
      default:
         break;
   }
}


void HMIModelComponent::bOnUpdateAlbumArtDefaultforGadgetMedia()
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateAlbumArtDefaultforGadgetMedia"));
   char buffer[200];
   SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
   Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);

   (*mID3GadgetCoverData).mGadgetImage = bmp;
   mID3GadgetCoverData.MarkAllItemsModified();
   mID3GadgetCoverData.SendUpdate(true);
}


void HMIModelComponent::bOnUpdateAlbumArtDefaultforExternalMediaGadget(int srcID)
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdateAlbumArtDefaultforExternalMediaGadget"));
   char buffer[200];
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateAlbumArtDefaultforExternalMediaGadget srcID: %d", srcID));
   switch (srcID)
   {
      //For device type 1 -> USB
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);
         (*mID3CamportUSBGadgetCoverData).mGadgetImage = bmp;
         mID3CamportUSBGadgetCoverData.MarkAllItemsModified();
         mID3CamportUSBGadgetCoverData.SendUpdate(true);
         break;
      }

      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);
         (*mID3RouterUSBGadgetCoverData).mGadgetImage = bmp;
         mID3RouterUSBGadgetCoverData.MarkAllItemsModified();
         mID3RouterUSBGadgetCoverData.SendUpdate(true);

         break;
      }
      default:
         break;
   }
}


#ifdef WINDOWS_SIMULATION

bool HMIModelComponent::onCourierMessage(const MediaOperationMsgReq& oMsg)
{
   ETG_TRACE_USR4(("HMIModelComponent::onCourierMessage:Operationtype:%d", oMsg.GetOperationType()));

   const uint16 optype = EN_MEDIA_PAUSE;
   switch (oMsg.GetOperationType())
   {
      case EN_MEDIA_PLAY:
      case EN_MEDIA_PAUSE:
      {
         if (playFlag == 0)
         {
            playFlag = 1;
            (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 0;
            m_MediaUSBPlayPauseData.MarkAllItemsModified();
            m_MediaUSBPlayPauseData.SendUpdate();
         }
         else
         {
            playFlag = 0;
            (*m_MediaUSBPlayPauseData).mPlayPauseIndex = 1;
            m_MediaUSBPlayPauseData.MarkAllItemsModified();
            m_MediaUSBPlayPauseData.SendUpdate();
         }
      }
      break;
      case EN_MEDIA_SHUFFLE_OFF:
      case EN_MEDIA_SHUFFLE_LIST:
      {
         if (shuffleFlag == 0)
         {
            shuffleFlag = 1;
            (*m_MediaUSBShuffleData).mShuffleIndex = 0;
            m_MediaUSBShuffleData.MarkAllItemsModified();
            m_MediaUSBShuffleData.SendUpdate();
         }
         else
         {
            shuffleFlag = 0;
            (*m_MediaUSBShuffleData).mShuffleIndex = 1;
            m_MediaUSBShuffleData.MarkAllItemsModified();
            m_MediaUSBShuffleData.SendUpdate();
         }
      }

      break;
   }

   return true;
}


#endif

/************************************************************************
*FUNCTION: 		bOnUpdatePlayTimeStatus(SWFL-640,SWFL-653)
*DESCRIPTION: 	Handles updating of playtime for SD, USB and IPOD device.
*PARAMETER:     u32TotalPlaytime,u32Elapsetime,szTotalTime,szPlayTime,NowPlayingDevType
*
*
*RETURNVALUE: 	void
*SWFL		: [SWFL-2454]
*
************************************************************************/
void HMIModelComponent::bOnUpdatePlayTimeStatus(const uint32& u32TotalPlaytime, const uint32& u32Elapsetime
      , const Candera::String& TotalPlaytimeText, const Candera::String ElapsetimeText, const uint8_t NowPlayingDevType)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus devType is %d", NowPlayingDevType));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus totalplaytime is %d", u32TotalPlaytime));

   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus elapsetime is %d", u32Elapsetime));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus Totalplaytime is %s", TotalPlaytimeText.GetCString()));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus Elapsetimetext is %s", ElapsetimeText.GetCString()));

   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus mediafiletype is %d", g_SM_MediaFileType));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayTimeStatus activemediasource is %d", g_SM_ActiveMediaSource));

   if ((g_SM_MediaFileType == SM_MEDIA_VIDEO_FILE) && ((g_SM_ActiveMediaSource != FI_EN_E8DTY_IPHONE) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_IPOD)))
   {
      vUpdatePlayTimeStatus(u32TotalPlaytime, u32Elapsetime, TotalPlaytimeText, ElapsetimeText);
   }
   else
   {
      switch (NowPlayingDevType)
      {
         //For device type 1 -> USB
         case FI_EN_E8DTY_USB:
         case FI_EN_E8DTY_MTP:
         {
            vUpdatePlayTimeStatus(u32TotalPlaytime, u32Elapsetime, TotalPlaytimeText, ElapsetimeText);

            break;
         }
         //For device type 2 -> IPOD
         case FI_EN_E8DTY_IPOD:
         case FI_EN_E8DTY_IPHONE:
         {
            Candera::Float g_TotalTime = 0.0;
            if (g_TotalTime != u32TotalPlaytime)
            {
               g_TotalTime = u32TotalPlaytime;
               (*m_MediaAppleDeviceData).mSliderTotalTime = g_TotalTime;
               m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::SliderTotalTimeItem);

               (*_gadgetplaybackaction).mSliderTotalTime = g_TotalTime;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderTotalTimeItem);
            }

            m_szTotalTimeText = TotalPlaytimeText;
            (*m_MediaAppleDeviceData).mTotalTime = m_szTotalTimeText;
            m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::TotalTimeItem);

            (*_gadgetplaybackaction).mTotalTime = m_szTotalTimeText;
            _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::TotalTimeItem);

            if (u32TotalPlaytime >= u32Elapsetime)
            {
               (*m_MediaAppleDeviceData).mSliderElapsedTime = u32Elapsetime;
               m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::SliderElapsedTimeItem);

               m_szElapsedTimeText = ElapsetimeText;
               (*m_MediaAppleDeviceData).mElapsedTime = m_szElapsedTimeText;
               m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::ElapsedTimeItem);

               (*_gadgetplaybackaction).mSliderElapsedTime = u32Elapsetime;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderElapsedTimeItem);

               (*_gadgetplaybackaction).mElapsedTime = m_szElapsedTimeText;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::ElapsedTimeItem);
            }
            m_MediaAppleDeviceData.SendUpdate(false);
            _gadgetplaybackaction.SendUpdate(false);
            break;
         }
         case FI_EN_E8DTY_BLUETOOTH:
         {
            Candera::Float g_TotalTime = 0.0;
            if (g_TotalTime != u32TotalPlaytime)
            {
               g_TotalTime = u32TotalPlaytime;
               (*m_MediaBTDeviceData).mSliderTotalTime = g_TotalTime;
               m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::SliderTotalTimeItem);

               (*_gadgetplaybackaction).mSliderTotalTime = g_TotalTime;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderTotalTimeItem);
            }
            m_szTotalTimeText = TotalPlaytimeText;
            (*m_MediaBTDeviceData).mTotalTime = m_szTotalTimeText;
            m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::TotalTimeItem);

            (*_gadgetplaybackaction).mTotalTime = m_szTotalTimeText;
            _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::TotalTimeItem);

            if (u32TotalPlaytime >= u32Elapsetime)
            {
               (*m_MediaBTDeviceData).mSliderElapsedTime = u32Elapsetime;
               m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::SliderElapsedTimeItem);

               m_szElapsedTimeText = ElapsetimeText;
               (*m_MediaBTDeviceData).mElapsedTime = m_szElapsedTimeText;
               m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::ElapsedTimeItem);

               (*_gadgetplaybackaction).mSliderElapsedTime = u32Elapsetime;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderElapsedTimeItem);

               (*_gadgetplaybackaction).mElapsedTime = m_szElapsedTimeText;
               _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::ElapsedTimeItem);
            }
            m_MediaBTDeviceData.SendUpdate(false);
            _gadgetplaybackaction.SendUpdate(false);
            break;
         }
         default:
            break;
      }
   }
}


/************************************************************************
*FUNCTION: 		vUpdatePlayTimeStatus
*DESCRIPTION: 	Handles updating of default albumart for SD, USB and IPOD device.
*PARAMETER:
* 				void
*
*
*RETURNVALUE: 	void
*
************************************************************************/
void HMIModelComponent::vUpdatePlayTimeStatus(const uint32& u32TotalPlaytime, const uint32& u32Elapsetime
      , const Candera::String& TotalPlaytimeText, const Candera::String ElapsetimeText)
{
   ETG_TRACE_USR1(("HMIModelComponent::vUpdatePlayTimeStatus Entered"));

   Candera::Float g_TotalTime = 0.0;
   if (g_TotalTime != u32TotalPlaytime)
   {
      g_TotalTime = u32TotalPlaytime;
      (*m_MediaPlayBackData).mSliderTotalTime = g_TotalTime;
      m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::SliderTotalTimeItem);

      (*_gadgetplaybackaction).mSliderTotalTime = g_TotalTime;
      _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderTotalTimeItem);
   }

   m_szTotalTimeText = TotalPlaytimeText;
   (*m_MediaPlayBackData).mTotalTime = m_szTotalTimeText;
   m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::TotalTimeItem);

   (*_gadgetplaybackaction).mTotalTime = m_szTotalTimeText;
   _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::TotalTimeItem);

   if (u32TotalPlaytime >= u32Elapsetime)
   {
      (*m_MediaPlayBackData).mSliderElapsedTime = u32Elapsetime;
      m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::SliderElapsedTimeItem);

      m_szElapsedTimeText = ElapsetimeText;

      (*m_MediaPlayBackData).mElapsedTime = m_szElapsedTimeText;
      m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::ElapsedTimeItem);

      (*_gadgetplaybackaction).mSliderElapsedTime = u32Elapsetime;
      _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::SliderElapsedTimeItem);

      m_szElapsedTimeText = ElapsetimeText;

      (*_gadgetplaybackaction).mElapsedTime = m_szElapsedTimeText;
      _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::ElapsedTimeItem);
   }
   m_MediaPlayBackData.SendUpdate(false);
   _gadgetplaybackaction.SendUpdate(false);
}


bool HMIModelComponent::onCourierMessage(const ButtonReactionMsg& oMsg)
{
   if ((Courier::ViewId("Media#MediaPlayerScenes#M_OPT") == oMsg.GetView())  &&
         (enRelease == oMsg.GetEnReaction()) &&
         (Courier::Identifier("Layer_TrafficAnnouncement/ButtonWidget") == oMsg.GetSender()))
   {
      ETG_TRACE_USR1(("HMIModelComponent::onCourierMessage:ButtonReactionMsg TrafficAnnouncement"));

      mTACheckboxStatus = !((*m_MediaSettingsTAStatus).mi8IsChecked);
      vUpdateTACheckbox(mTACheckboxStatus);
      vSetTAStatus(mTACheckboxStatus);
      return true;
   }

   return false;
}


void HMIModelComponent::vUpdateTAIcon(bool TAIconStatus)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateTAIcon Traffic Announcement  : TAIconStatus: %d", TAIconStatus));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateTAIcon Traffic Announcement  : mTACheckboxStatus: %d", mTACheckboxStatus));
   if (false == TAIconStatus)
   {
      (*m_MediaTAIcon).mTAIconIndex = MEDIA_C_VALUE_ZERO; //Icon_TA_CrossedOut
      m_MediaTAIcon.MarkAllItemsModified();
      m_MediaTAIcon.SendUpdate();
   }
   else if (true == TAIconStatus)
   {
      (*m_MediaTAIcon).mTAIconIndex = MEDIA_C_VALUE_ONE; //Icon_TA
      m_MediaTAIcon.MarkAllItemsModified();
      m_MediaTAIcon.SendUpdate();
   }
}


void HMIModelComponent::ClearPlaytimedata(int devtype)
{
   ETG_TRACE_USR4(("HMIModelComponent::ClearPlaytimedata: devicetype: %d", devtype));

   switch (devtype)
   {
      //For device type 1 -> USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      {
         (*m_MediaPlayBackData).mElapsedTime = "";
         (*m_MediaPlayBackData).mTotalTime = "";
         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::ElapsedTimeItem);
         m_MediaPlayBackData.MarkItemModified(ItemKey::PlaybackAction::TotalTimeItem);
         m_MediaPlayBackData.SendUpdate(false);
         break;
      }
      //For device type 2 -> IPOD
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         (*m_MediaAppleDeviceData).mElapsedTime = "";
         (*m_MediaAppleDeviceData).mTotalTime = "";
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::ElapsedTimeItem);
         m_MediaAppleDeviceData.MarkItemModified(ItemKey::AppleDevice::TotalTimeItem);
         m_MediaAppleDeviceData.SendUpdate(false);
         break;
      }
      case FI_EN_E8DTY_BLUETOOTH:
      {
         (*m_MediaBTDeviceData).mElapsedTime = "";
         (*m_MediaBTDeviceData).mTotalTime = "";
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::ElapsedTimeItem);
         m_MediaBTDeviceData.MarkItemModified(ItemKey::BTPlayInfoUpdate::TotalTimeItem);
         m_MediaBTDeviceData.SendUpdate(false);
         break;
      }
      default:
         break;
   }
}


void HMIModelComponent::vUpdateTACheckbox(uint8 TAStatus)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateTACheckbox Traffic Announcement  : Previous status: %d", (*m_MediaSettingsTAStatus).mi8IsChecked));
   if (VariantHandling::getInstance() != NULL && VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC)
   {
      (*m_MediaSettingsTAStatus).mi8IsChecked = false;
   }
   else
   {
      (*m_MediaSettingsTAStatus).mi8IsChecked = TAStatus;
   }
   m_MediaSettingsTAStatus.MarkAllItemsModified();
   m_MediaSettingsTAStatus.SendUpdate();

   //if (TAStatus == 1) {
   //	(*m_MediaTAIcon).
   //}

   ETG_TRACE_USR4(("HMIModelComponent::vUpdateTACheckbox Traffic Announcement : Is Checked? : %d", (*m_MediaSettingsTAStatus).mi8IsChecked));
}


void HMIModelComponent::vSetTAStatus(uint8 TAStatus)
{
   dp_tclAppHmi_MediaDPMediaTAOptionsStatus objMediaTAOptionStatus;
   tS32 s32ReturnValue = objMediaTAOptionStatus.s32SetData(TAStatus);
   TunerClientHandler::getInstance()->setTAflag(TAStatus);
   ETG_TRACE_USR4(("HMIModelComponent::vSetTAStatus TAStatus = %d", TAStatus));
}


uint8 HMIModelComponent::tGetTAStatus()
{
   uint8 TAStatus = 0;
   dp_tclAppHmi_MediaDPMediaTAOptionsStatus objMediaTAOptionStatus;
   tS32 s32ReturnValue = objMediaTAOptionStatus.s32GetData(TAStatus);

   ETG_TRACE_USR4(("HMIModelComponent::tGetTAStatus TAStatus = %d", TAStatus));
   return TAStatus;
}


void HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort(std::string Elapsedtime, double elapsedslidertime, double totalslidertime, std::string TotalPlayTime)
{
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort totaltime is %d", totalslidertime));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort elapsedslidertime  is %d", elapsedslidertime));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort Elapsedtime is %d", Elapsedtime.c_str()));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort TotalPlayTime is %d", TotalPlayTime.c_str()));

   Candera::String ElapsedTimeText;
   Candera::String TotalPlayTimeText;

   Candera::Float g_TotalTime = 0.0;
   if (g_TotalTime != totalslidertime)
   {
      g_TotalTime = totalslidertime;
      //nowplaying screen
      (*m_USBCameraPortMediaPlayBackData).mSliderTotalTime = g_TotalTime;
      m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::SliderTotalTimeItem);
      m_USBCameraPortMediaPlayBackData.SendUpdate(false);

      //gadget screen
      (*_cabinAgadgetplaybackaction).mSliderTotalTime = g_TotalTime;
      _cabinAgadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionCabinAGadget::SliderTotalTimeItem);
      _cabinAgadgetplaybackaction.SendUpdate(false);
   }

   m_szTotalTimeText = TotalPlayTime.c_str();
//nowplayscreen
   (*m_USBCameraPortMediaPlayBackData).mTotalTime = m_szTotalTimeText;
   m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::TotalTimeItem);
   m_USBCameraPortMediaPlayBackData.SendUpdate(false);
   //gadgetscreen
   (*_cabinAgadgetplaybackaction).mTotalTime = m_szTotalTimeText;
   _cabinAgadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionCabinAGadget::TotalTimeItem);
   _cabinAgadgetplaybackaction.SendUpdate(false);

   if (totalslidertime >= elapsedslidertime)
   {
      (*m_USBCameraPortMediaPlayBackData).mSliderElapsedTime = elapsedslidertime;
      m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::SliderElapsedTimeItem);
      m_USBCameraPortMediaPlayBackData.SendUpdate(false);

      m_szElapsedTimeText = Elapsedtime.c_str();
      (*m_USBCameraPortMediaPlayBackData).mElapsedTime = m_szElapsedTimeText;
      m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::ElapsedTimeItem);
      m_USBCameraPortMediaPlayBackData.SendUpdate(false);

      (*_cabinAgadgetplaybackaction).mSliderElapsedTime = elapsedslidertime;
      _cabinAgadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionCabinAGadget::SliderElapsedTimeItem);
      _cabinAgadgetplaybackaction.SendUpdate(false);

      m_szElapsedTimeText = Elapsedtime.c_str();

      (*_cabinAgadgetplaybackaction).mElapsedTime = m_szElapsedTimeText;
      _cabinAgadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionCabinAGadget::ElapsedTimeItem);
      _cabinAgadgetplaybackaction.SendUpdate(false);
   }
   // }
}


void HMIModelComponent::vOnUpdateElapsedTimeStatusRouter(std::string Elapsedtime, double elapsedslidertime, double totalslidertime, std::string TotalPlayTime)
{
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusRouter totaltime is %d", totalslidertime));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusRouter elapsedslidertime  is %d", elapsedslidertime));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusRouter Elapsedtime is %s", Elapsedtime.c_str()));
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort TotalPlayTime is %s", TotalPlayTime.c_str()));
   Candera::String ElapsedTimeText;
   Candera::String TotalPlayTimeText;
   Candera::Float g_TotalTime = 0.0;
   if (g_TotalTime != totalslidertime)
   {
      g_TotalTime = totalslidertime;
      //nowplaying screen
      (*m_USBRouterMediaPlayBackData).mSliderTotalTime = g_TotalTime;
      m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::SliderTotalTimeItem);
      m_USBRouterMediaPlayBackData.SendUpdate(false);
      //gadget screen
      (*_routergadgetplaybackaction).mSliderTotalTime = g_TotalTime;
      _routergadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionRouterGadget::SliderTotalTimeItem);
      _routergadgetplaybackaction.SendUpdate(false);
   }
   m_szTotalTimeText = TotalPlayTime.c_str();
   //nowplayscreen
   (*m_USBRouterMediaPlayBackData).mTotalTime = m_szTotalTimeText;
   m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::TotalTimeItem);
   m_USBRouterMediaPlayBackData.SendUpdate(false);
   //gadgetscreen
   (*_routergadgetplaybackaction).mTotalTime = m_szTotalTimeText;
   _routergadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionRouterGadget::TotalTimeItem);
   _routergadgetplaybackaction.SendUpdate(false);
   if (totalslidertime >= elapsedslidertime)
   {
      (*m_USBRouterMediaPlayBackData).mSliderElapsedTime = elapsedslidertime;
      m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::SliderElapsedTimeItem);
      m_USBRouterMediaPlayBackData.SendUpdate(false);

      m_szElapsedTimeText = Elapsedtime.c_str();
      (*m_USBRouterMediaPlayBackData).mElapsedTime = m_szElapsedTimeText;
      m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::ElapsedTimeItem);
      m_USBRouterMediaPlayBackData.SendUpdate(false);

      (*_routergadgetplaybackaction).mSliderElapsedTime = elapsedslidertime;
      _routergadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionRouterGadget::SliderElapsedTimeItem);
      _routergadgetplaybackaction.SendUpdate(false);
      m_szElapsedTimeText = Elapsedtime.c_str();
      (*_routergadgetplaybackaction).mElapsedTime = m_szElapsedTimeText;
      _routergadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionRouterGadget::ElapsedTimeItem);
      _routergadgetplaybackaction.SendUpdate(false);
   }
}


void HMIModelComponent::bOnUpdateNowPlayingStatusCamera()
{
   uint8 u8activeRegion = REGION_CABIN_A;
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusCamera is  u8activeRegion  %d ", u8activeRegion));

   bool bmuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusCamera mutestatus %d", bmuteStatus));

   std::string strVideoTitle = "";
   std::string strVideoEpisode = "";

   strVideoTitle = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].ArtistName;
   strVideoEpisode = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TitleName;

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusCamera is  iActiveSource  %d ", iActiveSource));

      if (u8activeRegion == 0 && iActiveSource == CAMPORTUSB)
      {
         if (strVideoTitle == "")
         {
            (*m_USBCameraPortMediaPlayBackData).mArtistName = Candera::String(id43);
         }
         else
         {
            (*m_USBCameraPortMediaPlayBackData).mArtistName = strVideoTitle.c_str();
         }

         if (strVideoEpisode == "")
         {
            (*m_USBCameraPortMediaPlayBackData).mTitleName = Candera::String(id43);
         }
         else
         {
            (*m_USBCameraPortMediaPlayBackData).mTitleName = strVideoEpisode.c_str();
         }

         (*m_USBCameraPortMediaPlayBackData).mAlbumName = "UNKNOWN";
         (*m_USBCameraPortMediaPlayBackData).mSourceName = Candera::String(id72);

         if (bmuteStatus == true)
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id78);
         }
         else
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id72);
         }
         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::SourceNameItem);
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);

         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::SourceNameItem);
         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::ArtistNameItem);
         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::TotalTimeItem);
         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::TitleNameItem);
         m_USBCameraPortMediaPlayBackData.MarkItemModified(ItemKey::USBCameraPortPlaybackAction::AlbumNameItem);
         m_USBCameraPortMediaPlayBackData.SendUpdate(false);
      }
   }
}


void HMIModelComponent::bOnUpdateNowPlayingStatusRouter()
{
   uint8 u8activeRegion = REGION_CABIN_A;
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusRouter is  u8activeRegion  %d ", u8activeRegion));
   bool bmuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusRouter mutestatus %d", bmuteStatus));
   std::string strVideoTitle = "";
   std::string strVideoEpisode = "";
   strVideoTitle = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].ArtistName;
   strVideoEpisode = MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TitleName;
   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateNowPlayingStatusRouter is  iActiveSource  %d ", iActiveSource));
      if (u8activeRegion == REGION_CABIN_A && iActiveSource == ROUTERUSB)
      {
         if (strVideoTitle == "")
         {
            (*m_USBRouterMediaPlayBackData).mArtistName = Candera::String(id43);
         }
         else
         {
            (*m_USBRouterMediaPlayBackData).mArtistName = strVideoTitle.c_str();
         }
         if (strVideoEpisode == "")
         {
            (*m_USBRouterMediaPlayBackData).mTitleName = Candera::String(id43);
         }
         else
         {
            (*m_USBRouterMediaPlayBackData).mTitleName = strVideoEpisode.c_str();
         }
         (*m_USBRouterMediaPlayBackData).mAlbumName = "UNKNOWN";
         (*m_USBRouterMediaPlayBackData).mSourceName = Candera::String(id74);
         if (bmuteStatus == true)
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id78);
         }
         else
         {
            (*_MuteInfo).mMuteStatus = Candera::String(id74);
         }
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::SourceNameItem);
         _MuteInfo.MarkAllItemsModified();
         _MuteInfo.SendUpdate(true);
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::SourceNameItem);
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::ArtistNameItem);
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::TotalTimeItem);
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::TitleNameItem);
         m_USBRouterMediaPlayBackData.MarkItemModified(ItemKey::USBRouterPlaybackAction::AlbumNameItem);
         m_USBRouterMediaPlayBackData.SendUpdate(false);
      }
   }
}


void HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort(uint8_t mPlayState)
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort"));

   uint8 u8activeRegion = REGION_CABIN_A;
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort u8activeRegion  %d ", u8activeRegion));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort mPlayState  %d", mPlayState));

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort iActiveSource  %d ", iActiveSource));

      if (iActiveSource == CAMPORTUSB && u8activeRegion == REGION_CABIN_A)
      {
         uint8_t playState = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].playPauseState;
         ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort playState in cabin   %d", playState));

         if (mPlayState == MEDIA_C_VALUE_ZERO)
         {
            (*m_MediaUSBCameraPortPlayPauseData).mPlayPauseIndex = 0;
            m_MediaUSBCameraPortPlayPauseData.MarkAllItemsModified();
            m_MediaUSBCameraPortPlayPauseData.SendUpdate();
         }
         else
         {
            (*m_MediaUSBCameraPortPlayPauseData).mPlayPauseIndex = 1;
            m_MediaUSBCameraPortPlayPauseData.MarkAllItemsModified();
            m_MediaUSBCameraPortPlayPauseData.SendUpdate();
         }
      }
   }

   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnCameraPort mPlayState  %d)", mPlayState));
}


void HMIModelComponent::bOnUpdatePlayPauseStateOnRouter(uint8_t mPlayState)
{
   ETG_TRACE_USR1(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter"));

   uint8 u8activeRegion = REGION_CABIN_A;
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter u8activeRegion  %d ", u8activeRegion));
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter mPlayState  %d", mPlayState));

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter iActiveSource  %d ", iActiveSource));

      if (iActiveSource == ROUTERUSB && u8activeRegion == REGION_CABIN_A)
      {
         uint8_t playState = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].playPauseState;
         ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter playState in cabin   %d", playState));

         if (mPlayState == MEDIA_C_VALUE_ZERO)
         {
            (*m_MediaRouterPlayPauseData).mPlayPauseIndex = 0;
            m_MediaRouterPlayPauseData.MarkAllItemsModified();
            m_MediaRouterPlayPauseData.SendUpdate();
         }
         else
         {
            (*m_MediaRouterPlayPauseData).mPlayPauseIndex = 1;
            m_MediaRouterPlayPauseData.MarkAllItemsModified();
            m_MediaRouterPlayPauseData.SendUpdate();
         }
      }
   }

   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdatePlayPauseStateOnRouter mPlayState  %d)", mPlayState));
}


void HMIModelComponent::bOnUpdateRepeatStateOnCameraPort(uint8_t mPlayState)
{
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateRepeatStateOnCameraPort PlayState %d", mPlayState));

   //m_u32RepeatMode = mPlayState;

   switch (mPlayState)
   {
      case EN_MEDIA_REPEATMODE_TRACK:
      {
         vOnUpdateRepeatTrackIcon();
         break;
      }
      case EN_MEDIA_REPEATMODE_LIST:
      {
         vOnUpdateRepeatListIcon();
         break;
      }
      case EN_MEDIA_REPEATMODE_OFF:
      {
         vOnUpdateRepeatOffIcon();
         break;
      }
      default:
         break;
   }
}


void HMIModelComponent::bOnUpdateHeaderMuteStatus()
{
   bool bmuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateHeaderMuteStatus mutestatus %d", bmuteStatus));
   int iSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateHeaderMuteStatus iSrcId %d", iSrcId));

   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateHeaderMuteStatus iSubSrcId %d", iSubSrcId));

   int iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateHeaderMuteStatus iDeviceType %d", iDeviceType));
   if (bmuteStatus == true)
   {
      (*_MuteInfo).mMuteStatus = Candera::String(id78);
      (*oMuteInfoforAux).mMuteStatusforAux = Candera::String(id78);
      _MuteInfo.MarkAllItemsModified();
      _MuteInfo.SendUpdate(true);
      oMuteInfoforAux.MarkAllItemsModified();
      oMuteInfoforAux.SendUpdate(true);
   }
   else
   {
      if (iSrcId == SRC_MEDIA_PLAYER)
      {
         if ((iDeviceType == FI_EN_E8DTY_USB) || (iDeviceType == FI_EN_E8DTY_MTP))
         {
            int devNum = tclMediaClientHandler::getInstance().GetActiveUSBnum();//tclMediaClientHandler::getInstance().getUSBDevNum();
            if (devNum == 1)
            {
               (*_MuteInfo).mMuteStatus = Candera::String(id76);
            }
            else if (devNum == 2)
            {
               (*_MuteInfo).mMuteStatus = Candera::String(id77);
            }
            else
            {
               (*_MuteInfo).mMuteStatus = USBText;
            }
         }
         else if ((iDeviceType == FI_EN_E8DTY_IPOD) || (iDeviceType == FI_EN_E8DTY_IPHONE))
         {
            (*_MuteInfo).mMuteStatus = IPODText;
         }
      }
      else if (iSrcId == SRC_PHONE_BTAUDIO)
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id75);
      }
      else if (iSrcId == SRC_MEDIA_AUX)
      {
         (*oMuteInfoforAux).mMuteStatusforAux = "Aux";
         oMuteInfoforAux.MarkAllItemsModified();
         oMuteInfoforAux.SendUpdate(true);
      }
      else if ((iSrcId == SRC_CAMERAPORT1_USB_VIDEO) || (iSrcId == SRC_CAMERAPORT1_USB_AUDIO))
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id72);
      }
      else if (iSrcId == SRC_CAMERAPORT1_HDMI)
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id73);
      }
      else if (iSrcId == SRC_CAMERAPORT2_HDMI)
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id79);
      }
      else if (iSrcId == SRC_CAMERAPORT3_HDMI)
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id80);
      }
      else if (iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
      {
         (*_MuteInfo).mMuteStatus = Candera::String(id74);
      }
      _MuteInfo.MarkAllItemsModified();
      _MuteInfo.SendUpdate(true);
   }
}


void HMIModelComponent::ClearHeaderSourceName()
{
   bool bMuteStatus = MediaSourceHandling::getInstance().getCurrentRegionMuteStatus();
   ETG_TRACE_USR4(("HMIModelComponent::ClearHeaderSourceName mute status %d", bMuteStatus));

   if (bMuteStatus == true)
   {
      (*_MuteInfo).mMuteStatus = Candera::String(id78);
   }
   else
   {
      (*_MuteInfo).mMuteStatus = "";
   }
   _MuteInfo.MarkAllItemsModified();
   _MuteInfo.SendUpdate(true);
}


void HMIModelComponent::vUpdateUsbAlbumIconAndTextVisibility()
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateUsbAlbumIconAndTextVisibility"));
   int iSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent vUpdateUsbAlbumIconAndTextVisibility iSrcId %d", iSrcId));

   if (iSrcId == SRC_CAMERAPORT1_USB_VIDEO || iSrcId == SRC_CAMERAPORT1_USB_AUDIO) //SRC_CAMERAPORT1_USB_AUDIO -488
   {
      (*_AlbumIconTextVisibility).mIconAndTextVisiblity = 0;
   }
   else
   {
      (*_AlbumIconTextVisibility).mIconAndTextVisiblity = 1;
   }
   if (iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
   {
      (*_AlbumIconTextVisibility).mIconAndTextVisiblity = 0;
   }
   else
   {
      (*_AlbumIconTextVisibility).mIconAndTextVisiblity = 1;
   }
   _AlbumIconTextVisibility.MarkItemModified(ItemKey::USBCameraPortAlbumVisibility::IconAndTextVisiblityItem);
   _AlbumIconTextVisibility.SendUpdate(true);
}


void HMIModelComponent::vUpdateNowplayIcons(uint8_t mediaIconIndex)
{
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateNowplayIcons mediaIconIndex:%d", mediaIconIndex));
   if (mediaIconIndex == PODCAST_NOWPLAY_ICON_UPDATE)
   {
      (*m_MediaIconType).mIconArtistVisiblity = 0;
      (*m_MediaIconType).mIconArtistPodcastVisiblity = 1;
      (*m_MediaIconType).mIconTitleVisiblity = 0;
      (*m_MediaIconType).mIconTitlePodcastVisiblity = 1;
      (*m_MediaIconType).mIconAlbumVisiblity = 1;
      (*m_MediaIconType).mIconAlbumAudioBookVisiblity = 0;
   }
   else if (mediaIconIndex == AUDIOBOOK_NOWPLAY_ICON_UPDATE)
   {
      (*m_MediaIconType).mIconArtistVisiblity = 1;
      (*m_MediaIconType).mIconArtistPodcastVisiblity = 0;
      (*m_MediaIconType).mIconTitleVisiblity = 1;
      (*m_MediaIconType).mIconTitlePodcastVisiblity = 0;
      (*m_MediaIconType).mIconAlbumVisiblity = 0;
      (*m_MediaIconType).mIconAlbumAudioBookVisiblity = 1;
   }
   else
   {
      (*m_MediaIconType).mIconArtistPodcastVisiblity = 0;
      (*m_MediaIconType).mIconArtistVisiblity = 1;
      (*m_MediaIconType).mIconTitleVisiblity = 1;
      (*m_MediaIconType).mIconTitlePodcastVisiblity = 0;
      (*m_MediaIconType).mIconAlbumVisiblity = 1;
      (*m_MediaIconType).mIconAlbumAudioBookVisiblity = 0;
   }
   m_MediaIconType.MarkAllItemsModified();
   m_MediaIconType.SendUpdate();
}


void HMIModelComponent::updateMediaSourceInfoInGadget(int deviceType)
{
   int srcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   std::map <int, MediaGadgetIndex>::iterator itr = _mediaMetadataGadgetIndexMap.find(srcId);

   ETG_TRACE_USR1(("HMIModelComponent::updateMediaSourceInfoInGadget: srcId  %d, deviceType  %d (*m_MediaMetaDataGadgetUpdate).mMediaGadgetIndex %d", srcId, deviceType, (*m_MediaMetaDataGadgetUpdate).mMediaGadgetIndex));
   int gadgetIcon = MediaSourceHandling::getInstance().vUpdateGadgetIcons(srcId, deviceType);
   ETG_TRACE_USR1(("HMIModelComponent::updateMediaSourceInfoInGadget: gadgetIcon%d", gadgetIcon));
   if (deviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB || deviceType == MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP)
   {
      if (tclMediaClientHandler::getInstance().getUSBDevNum() == USB_TWO)
      {
         ETG_TRACE_USR1(("HMIModelComponent::updateMediaSourceInfoInGadget USB TWO"));

         (*m_MediaSourceUSBNameInGadget).mUSBName = Candera::String(id77);
         m_MediaSourceUSBNameInGadget.MarkAllItemsModified();
         m_MediaSourceUSBNameInGadget.SendUpdate(true);
      }
      else if (tclMediaClientHandler::getInstance().getUSBDevNum() == USB_ONE)
      {
         ETG_TRACE_USR1(("HMIModelComponent::updateMediaSourceInfoInGadget UBS ONE"));

         (*m_MediaSourceUSBNameInGadget).mUSBName = Candera::String(id76);
         m_MediaSourceUSBNameInGadget.MarkAllItemsModified();
         m_MediaSourceUSBNameInGadget.SendUpdate(true);
      }

      else
      {
         ETG_TRACE_USR1(("HMIModelComponent::updateMediaSourceInfoInGadget USB"));

         (*m_MediaSourceUSBNameInGadget).mUSBName = USBText;
         m_MediaSourceGadgetUpdate.MarkAllItemsModified();
         m_MediaSourceGadgetUpdate.SendUpdate(true);
      }
   }

   if (itr != _mediaMetadataGadgetIndexMap.end())
   {
      (*m_MediaMetaDataGadgetUpdate).mMediaGadgetIndex = _mediaMetadataGadgetIndexMap[srcId];
      if (_restorationEvaluatingStatus == false)
      {
         _mediaGadgetIndex = NO_MEDIA_DEVICE_GADGET_INDEX;
      }
      _cabinGadgetTextIndex = NO_MEDIA_DEVICE_GADGET_INDEX;
      m_MediaMetaDataGadgetUpdate.MarkAllItemsModified();
      m_MediaMetaDataGadgetUpdate.SendUpdate(true);
   }
   if (gadgetIcon != -1)
   {
      (*m_MediaSourceGadgetUpdate).mMediaSourceIndex = gadgetIcon;
      m_MediaSourceGadgetUpdate.MarkAllItemsModified();
      m_MediaSourceGadgetUpdate.SendUpdate(true);
   }
}


/************************************************************************
*Courier message: GuiStartupFinishedUpdMsg
*DESCRIPTION	: Informs once HMI is GUI update is finished
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
bool HMIModelComponent::onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg)
{
   m_GuiStartUpFlag = 1;
   clMetaData::getInstance().setGuiStartupStatus(true);
   ETG_TRACE_USR4(("HMIModelComponent::GuiStartupFinishedUpdMsg m_GuiStartUpFlag %d", m_GuiStartUpFlag));
   UnSupportedPopupHandler::getInstance().callUnsupportedPopUpOnGuiStartUp();
   SpeedLockStatusHandler::getInstance().readVideoRestrictionValue();
   _kdsHandler->readAuxKdsValue();
   int devtype = MediaSourceHandling::getInstance().GetdeviceType();
   int devicetag = tclMediaClientHandler::getInstance().getDeviceTag(devtype);
   //setting BT to On on startup
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent::GuiStartupFinishedUpdMsg devicetag %d", devicetag));
   ETG_TRACE_USR4(("HMIModelComponent::GuiStartupFinishedUpdMsg devtype %d", devtype));
   ETG_TRACE_USR4(("HMIModelComponent::GuiStartupFinishedUpdMsg srcid %d", srcid));
   if (srcid != SRC_SPI_ENTERTAIN)
   {
      updateSpidataVisibilityonMediaGadget(false);
   }
   else
   {
      updateSpidataVisibilityonMediaGadget(true);
   }

   enableorDisableTASettingsBasedOnRegion();
   uint8 iactiveMediaSource = MediaSourceHandling::getInstance().GetActiveMediaSource(srcid, devtype);
   ETG_TRACE_USR4(("HMIModelComponent::GuiStartupFinishedUpdMsg iactiveMediaSource %d", iactiveMediaSource));
   HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(iactiveMediaSource);
   updateMediaSourceInfoInGadget(devtype);
   HMIModelComponent::getInstance().vSourceButtonHighlight(devtype, devicetag);
   SpeedLockStatusHandler::getInstance().handleMediaVideoLayerVisibility(false);

   //To update the gadget on the homescreen.
//  HMIModelComponent::getInstance().bOnUpdateNowPlayingStatusOnHomeScreen();
   HMIModelComponent::getInstance().bOnUpdateNowPlayingStatus();
   //Below code is to hide the camport buttons in cockpit
   //vUpdateExternalSourcesVisibility();

   return false;
}


int HMIModelComponent::getGuiStartupStatus()
{
   ETG_TRACE_USR4(("HMIModelComponent::getGuiStartupStatus guistartupflag %d", m_GuiStartUpFlag));
   return m_GuiStartUpFlag;
}


/************************************************************************
*Courier message: vAuxSourceVisibility
*DESCRIPTION	: sets the visibility of the AUX source button based on the Aux KDS parameter
*PARAMETER		: bVisible
*RETURNVALUE	: none
************************************************************************/
void HMIModelComponent::vAuxSourceVisibility(bool bVisible)
{
   ETG_TRACE_USR4(("HMIModelComponent::vAuxSourceVisibility visible: %d", bVisible));
   (*m_MediaAuxActivation).mIsAuxActiveVariant = bVisible;
   m_MediaAuxActivation.MarkAllItemsModified();
   m_MediaAuxActivation.SendUpdate(true);
}


/************************************************************************
*FUNCTION: 		SPIDataVisibility
*DESCRIPTION: 	Handles SPI visibility of icon and text.
*PARAMETER:     bool val
*
*
*RETURNVALUE:void
*
************************************************************************/
void HMIModelComponent::updateSpidataVisibilityonMediaGadget(bool val)
{
   ETG_TRACE_USR4(("HMIModelComponent::SPIDataVisibility value %d", val));
   if (val == true)
   {
      (*m_SPIVisibilityData).mSPIIsvisible = true;
   }
   else
   {
      (*m_SPIVisibilityData).mSPIIsvisible = false;
   }
   m_SPIVisibilityData.MarkAllItemsModified();
   m_SPIVisibilityData.SendUpdate();
}


Candera::String HMIModelComponent::GetListAllTitleId()
{
   ETG_TRACE_USR4(("HMIModelComponent::GetListAllTitleId titleid %s", Candera::String(id70)));

   return Candera::String(id70);
}


Candera::String HMIModelComponent::GetListAllAlbumId()
{
   ETG_TRACE_USR4(("HMIModelComponent::GetListAllAlbumId allalbum id %s", Candera::String(id71)));

   return Candera::String(id71);
}


void HMIModelComponent::setCabinMediaGadgetIndex(MediaGadgetIndex cabinTextIndex)
{
   ETG_TRACE_USR4(("HMIModelComponent::setCabinMediaGadgetIndex cabinTextIndex %d , _mediaGadgetIndex %d, _restorationEvaluatingStatus %d", cabinTextIndex, _mediaGadgetIndex, _restorationEvaluatingStatus));

   _mediaGadgetIndex = cabinTextIndex;
   if (_mediaGadgetIndex == RESTORATION_TEXT_INDEX)
   {
      _restorationEvaluatingStatus = true;
   }
   else
   {
      _restorationEvaluatingStatus = false;
   }
   ETG_TRACE_USR4(("HMIModelComponent::setCabinMediaGadgetIndex _mediaGadgetIndex %d _cabinSourceId %d", _cabinGadgetTextIndex, _cabinSourceId));
}


void HMIModelComponent::vUpdateCabinGadgetVisibility()
{
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateCabinGadgetVisibility iActiveCabinSrcId %d", iActiveCabinSrcId));
   uint8 u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateCabinGadgetVisibility activeregion %d", u8ActiveRegion));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateCabinGadgetVisibility _mediaGadgetIndex %d", _mediaGadgetIndex));
   ETG_TRACE_USR4(("HMIModelComponent::vUpdateCabinGadgetVisibility _cabinSourceId %d", _cabinSourceId));

   if ((iActiveCabinSrcId == SRC_INVALID && u8ActiveRegion == REGION_CABIN_A) ||
         (iActiveCabinSrcId == SRC_INVALID && u8ActiveRegion == REGION_CABIN_B))
   {
      ETG_TRACE_USR1(("HMIModelComponent::vUpdateCabinGadgetVisibility active cabine source is invalid "));

      onMediaGadgetIndexUpdate(_mediaGadgetIndex);

      (*m_GadgetItemsVisibility).mGadgetButtonsVisibleState = false;
      (*m_GadgetItemsVisibility).mGadgetSourceIconAndTextVisibleState = false;
   }

   else
   {
      if (iActiveCabinSrcId == SRC_CAMERAPORT1_HDMI || iActiveCabinSrcId == SRC_CAMERAPORT2_HDMI || iActiveCabinSrcId == SRC_CAMERAPORT3_HDMI || iActiveCabinSrcId == SRC_MEDIA_AUX)
      {
         vUpdateMuteStateVisibleGadget(false, false, false);
         (*m_GadgetItemsVisibility).mGadgetSourceIconAndTextVisibleState = true;
         (*m_GadgetItemsVisibility).mGadgetButtonsVisibleState = false;
         //  bOnUpdateAlbumArtDefaultforGadgetMedia();
      }
      else if (MediaSourceHandling::getInstance().IsMediaSourceRange(iActiveCabinSrcId) == true)
      {
         (*m_GadgetItemsVisibility).mGadgetButtonsVisibleState = true;
         (*m_GadgetItemsVisibility).mGadgetSourceIconAndTextVisibleState = true;
      }
      else
      {
         (*m_GadgetItemsVisibility).mGadgetButtonsVisibleState = false;
         (*m_GadgetItemsVisibility).mGadgetSourceIconAndTextVisibleState = false;
      }
   }
   m_GadgetItemsVisibility.MarkAllItemsModified();
   m_GadgetItemsVisibility.SendUpdate(true);
}


void HMIModelComponent::vUpdateMuteStateVisibleGadget(bool controlIconStatus, bool playDisableIconVisible, bool playEnableIconVisible)
{
   (*m_MuteStateVisibleGadget).mControlIconStatus = controlIconStatus;
   (*m_MuteStateVisibleGadget).mPlayEnableIconVisible = playEnableIconVisible;
   (*m_MuteStateVisibleGadget).mPlayDisableIconVisible = playDisableIconVisible;
   m_MuteStateVisibleGadget.MarkAllItemsModified();
   m_MuteStateVisibleGadget.SendUpdate(true);
}


void HMIModelComponent::setKdsHandlerInstance(KDSHandler* _kdsHandlerInstance)
{
   if (_kdsHandlerInstance != NULL)
   {
      _kdsHandler = _kdsHandlerInstance;
   }
   else
   {
      ETG_TRACE_USR4(("HMIModelComponent::_kdsHandlerInstance is NULL"));
   }
}


void HMIModelComponent::setCabinSourceRestorationInstance(CabinSourceRestoration* _cabinSourceRestorationInstance)
{
   if (_cabinSourceRestorationInstance != NULL)
   {
      _cabinSourceRestoration = _cabinSourceRestorationInstance;
   }
   else
   {
      ETG_TRACE_ERR(("HMIModelComponent::_cabinSourceRestorationInstance is NULL"));
   }
}


void HMIModelComponent::onMediaGadgetIndexUpdate(MediaGadgetIndex mediaGadgetIndex)
{
   _mediaGadgetIndex = mediaGadgetIndex;
   ETG_TRACE_USR1(("HMIModelComponent::onMediaGadgetIndexUpdate:_mediaGadgetIndex %d _restorationEvaluatingStatus %d", _mediaGadgetIndex, _restorationEvaluatingStatus));
   (*m_MediaMetaDataGadgetUpdate).mMediaGadgetIndex = _mediaGadgetIndex;
   m_MediaMetaDataGadgetUpdate.MarkItemModified(ItemKey::MediaMetaDataUpdateInGadget::MediaGadgetIndexItem);
   m_MediaMetaDataGadgetUpdate.SendUpdate();
}


void HMIModelComponent::bOnUpdateAlbumArtDefaultforExternalMediaSources(int srcID)
{
   char buffer[200];
   ETG_TRACE_USR4(("HMIModelComponent::bOnUpdateAlbumArtDefaultforExternalMediaSources srcID: %d", srcID));
   switch (srcID)
   {
      //For device type 1 -> USB
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);
         //nowplaying
         (*mID3CamportUSBCoverData).mCamportAlbumImage = bmp;
         mID3CamportUSBCoverData.MarkItemModified(ItemKey::ID3CoverData::ImageItem);
         mID3CamportUSBCoverData.SendUpdate(true);

         //for gadget
         bOnUpdateAlbumArtDefaultforExternalMediaGadget(srcID);
         break;
      }

      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      {
         SNPRINTF(buffer, sizeof(buffer), "Media#Images#M_SourceSelection_Media#Icon_CoverBooklet_defaultBmp");
         Candera::MemoryManagement::SharedPointer<Candera::Image2D> bmp = ImageLoader::getAssetBitmapImage(buffer);

         (*mID3RouterUSBCoverData).mRouterAlbumImage = bmp;
         mID3RouterUSBCoverData.MarkAllItemsModified();
         mID3RouterUSBCoverData.SendUpdate(true);

         bOnUpdateAlbumArtDefaultforExternalMediaGadget(srcID);

         break;
      }
      default:
         break;
   }
}


/* ***********************************************************
* Function 		: vClearGadgetMetadata
* Description 	: Function will clear the meta-data information when ever source changed and source disconnected
* Parameters 	: int mediaSourceType
* Return value	: void
* ***********************************************************/
void HMIModelComponent::vClearGadgetMetadata(int mediaSourceType)
{
   ETG_TRACE_USR4(("HMIModelComponent::vClearGadgetMetadata mediaSourceType:%d", mediaSourceType));
   //m_pdeferedbuffer = m_poNullImageDataPointer;
   switch (mediaSourceType)
   {
      //USB
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      case FI_EN_E8DTY_BLUETOOTH:
         (*_gadgetplaybackaction).mArtistName = "";
         (*_gadgetplaybackaction).mTitleName = "";
         _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::ArtistNameItem);
         _gadgetplaybackaction.MarkItemModified(ItemKey::PlaybackActionGadget::TitleNameItem);
         _gadgetplaybackaction.SendUpdate(false);
         break;
   }
}


}//end of Core
}//end of App
