/**************************************************************************************
* @file         : <VehicleSrvClient.cpp>
* @author       : <ECG> <INF4CV>
* @addtogroup   : <AppHmi_System>
* @brief        :
* @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "VehicleSrvClient.h"
#include "AppHmi_MediaStateMachine.h"
#include "Core/MediaClientHandler/MediaClientHandler.h"
#include "ProjectBaseTypes.h"
#include "Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#include "Core/ExternalMediaClientHandler/ExternalMediaClientHandler.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::VehicleSrvClient::
#include "trcGenProj/Header/VehicleSrvClient.cpp.trc.h"
#endif


using namespace ::vehicle_main_fi_types;
using namespace ::VEHICLE_MAIN_FI;


namespace App {
namespace Core {

struct LanguageTable
{
   int LangIndex;
   std::string LangString;
};


#ifndef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
LanguageTable LanguageMappingTable[LANGNO] =
{
   { 12, "en_GB" },
   { 23, "de_DE" },
   { 10, "nl_NL" },
   { 21, "fr_FR" },
   { 29, "it_IT" },
   { 46, "es_ES" },
   { 39, "pt_PT" },
   { 27, "hu_HU" },
   { 42, "ru_RU" },
   { 8, "cs_CZ" },
   { 9, "da_DK" },
   { 35, "nb_NO" },
   { 38, "pl_PL" },
   { 49, "sv_SE" },
   { 47, "es_MX" },
   { 13, "en_US" },
   { 24, "el_GR" },
   { 52, "tr_TR" },
   { 22, "fr_CA" }
};


#endif

#ifdef VARIANT_S_FTR_ENABLE_AUX_SUPPORT
LanguageTable LanguageMappingTable[LANGNO] =
{
   { 12, "en_GB" },
   { 23, "de_DE" },
   { 10, "nl_NL" },
   { 21, "fr_FR" },
   { 29, "it_IT" },
   { 46, "es_ES" },
   { 39, "pt_PT" },
   { 27, "hu_HU" },
   { 8, "cs_CZ" },
   { 9, "da_DK" },
   { 35, "nb_NO" },
   { 38, "pl_PL" },
   { 49, "sv_SE" },
   { 47, "es_LA" },
   { 24, "el_GR" },
   { 52, "tr_TR" },
   { 3, "bg_BG" },
   { 45, "sl_SI" },
   { 44, "sk_SK" },
   { 41, "ro_RO" },
   { 19, "fi_FI" },
   { 7, "hr_HR" }
};


#endif

/**
 * @Constructor
 */
VehicleSrvClient* VehicleSrvClient::_VehicleSrvClient = NULL;
VehicleSrvClient::VehicleSrvClient(): _vehicleProxy(::VEHICLE_MAIN_FIProxy::createProxy("vehicleMainFiPort", *this)),
   _externalMediaClientHandler(NULL)
{
   ETG_TRACE_USR1(("VehicleSrvClient::VehicleSrvClient Constructor is called"));
   if (_vehicleProxy)
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _vehicleProxy->getPortName());
   }
   readLanguageOnStartUp();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
}


/**
* @Destructor
*/

VehicleSrvClient::~VehicleSrvClient()
{
   ETG_TRACE_USR1(("VehicleSrvClient::VehicleSrvClient destructor is called"));
}


/**
 * @onAvailable
 */
void VehicleSrvClient::onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onAvailable is called"));

   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      StartupSync::getInstance().onAvailable(proxy, stateChange);

      _vehicleProxy->sendLanguageUpReg(*this);
      _vehicleProxy->sendSpeedUpReg(*this);
      _vehicleProxy->sendParkingBreakSwitchUpReg(*this);
      _vehicleProxy->sendTrackCommandUpReg(*this);
   }
}


/**
 * @onUnavailable
 */
void VehicleSrvClient::onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onUnavailable is called"));

   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      StartupSync::getInstance().onUnavailable(proxy, stateChange);
      _vehicleProxy->sendLanguageRelUpRegAll();
      _vehicleProxy->sendSpeedRelUpRegAll();
      _vehicleProxy->sendParkingBreakSwitchRelUpRegAll();
      _vehicleProxy->sendTrackCommandRelUpRegAll();
   }
}


void VehicleSrvClient::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("VehicleSrvClient::registerProperties is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _vehicleProxy->sendLanguageUpReg(*this);
      _vehicleProxy->sendSpeedUpReg(*this);
      _vehicleProxy->sendParkingBreakSwitchUpReg(*this);
      _vehicleProxy->sendTrackCommandUpReg(*this);
      _vehicleProxy->sendCabinMediaSourceCommandUpReg(*this);
      _vehicleProxy->sendCabinMediaTrackCommandUpReg(*this);
   }
}


void VehicleSrvClient::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("VehicleSrvClient::deregisterProperties is called"));
   if (_vehicleProxy && (proxy == _vehicleProxy))
   {
      _vehicleProxy->sendLanguageRelUpRegAll();
      _vehicleProxy->sendSpeedRelUpRegAll();
      _vehicleProxy->sendParkingBreakSwitchRelUpRegAll();
      _vehicleProxy->sendTrackCommandRelUpRegAll();
      _vehicleProxy->sendCabinMediaSourceCommandRelUpRegAll();
      _vehicleProxy->sendCabinMediaTrackCommandRelUpRegAll();
   }
}


void VehicleSrvClient::setExternalMediaClientHandlerInstance(ExternalMediaClientHandler* _externalMediaClientHandlerInstance)
{
   ETG_TRACE_COMP(("VehicleSrvClient::setExternalMediaClientHandlerInstance"));
   if (_externalMediaClientHandlerInstance != NULL)
   {
      _externalMediaClientHandler = _externalMediaClientHandlerInstance;
   }
   else
   {
      ETG_TRACE_ERR(("VehicleSrvClient::_externalMediaClientHandler is NULL"));
   }
}


/*
 * onSetLanguageResult - ASF framework callback which is after SetLanguage
 * @param[in] proxy
 * @param[in] result
 * @param[out] None
 * @return void
 * SWFL : [SWFL-2730]
 */
void VehicleSrvClient::onSetLanguageResult(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< SetLanguageResult >& result)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSetLanguageResult result :  %d", result->getResult()));
}


/*
 * onSetLanguageError - ASF framework callback which is after SetLanguage
 * @param[in] proxy
 * @param[in] error
 * @param[out] None
 * @return void
 */
void VehicleSrvClient::onSetLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< SetLanguageError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onSetLanguageError"));
}


/*
 * onLanguageStatus - ASF framework callback which is triggered on every setLanguage()
 * @param[in] proxy
 * @param[in] status
 * @param[out] None
 * @return voidl
 * todo :check As of now, after setLanguage, onLanguageStatus() is received twice from the VD_Vehicle data.
 * First update represents, previously configured Language ID and the second one represents currently configured ID.*/

void VehicleSrvClient::onLanguageStatus(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,  const ::boost::shared_ptr< LanguageStatus >& status)
{
   std::vector<T_Language_SourceTable> langTable = status->getLangTable();
   uint8 langIndex = status->getLanguage();
   ETG_TRACE_USR4(("VehicleSrvClient::onLanguageStatus langIndex=%d", langIndex));
   std::string textid = getLanguageId(langIndex);
   //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(textid.c_str()))->Post();
   //TunerDataPoolConfig::getInstance()->setLanguage(langIndex);
}


/*
* onLanguageStatus - ASF framework callback which is triggered on every speedstatus()
* @param[in] proxy
* @param[in] status
* @param[out] None
* [SYSFL-1047]
* @return voidl*/


void VehicleSrvClient::onSpeedStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onSpeedStatus speedvalue  = %d", status->getSpeedValue()));

   if (proxy == _vehicleProxy)
   {
      SpeedLockStatusHandler::getInstance().updateSpeedValue(status->getSpeedValue());
   }
   else
   {
      ETG_TRACE_USR1(("VehicleSrvClient::onSpeedStatus  service is not available"));
   }
}


void VehicleSrvClient::onSpeedError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::SpeedError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onSpeedError"));
}


void VehicleSrvClient::onParkingBreakSwitchError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::ParkingBreakSwitchError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onParkingBreakSwitchError"));
}


void VehicleSrvClient::onParkingBreakSwitchStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::ParkingBreakSwitchStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onParkingBreakSwitchStatus ParkingBreakSwitch  = %d", status->getParkingBreakSwitchValue()));

   if (proxy == _vehicleProxy)
   {
      SpeedLockStatusHandler::getInstance().updateParkBrakeStatus(status->getParkingBreakSwitchValue());
   }
   else
   {
      ETG_TRACE_USR1(("VehicleSrvClient::onParkingBreakSwitchStatus  service is not available"));
   }
}


std::string VehicleSrvClient::getLanguageId(int index)
{
   std::string langId = "";
   ETG_TRACE_USR4(("VehicleSrvClient::getLanguageId index=%d", index));

   for (int i = 0; i < (sizeof(LanguageMappingTable) / sizeof(LanguageMappingTable[0])); i++)
   {
      if (LanguageMappingTable[i].LangIndex == index)
      {
         ETG_TRACE_USR4(("VehicleSrvClient::getLanguageId Language string=%s", LanguageMappingTable[i].LangString.c_str()));
         langId = LanguageMappingTable[i].LangString.c_str();
      }
   }
   return langId;
}


/**
* Description     : This method will be called on system startup for language translation
*
* @return         : uint8
*/
void VehicleSrvClient::readLanguageOnStartUp()
{
   uint8 languageIndex = 0;
   //uint8 languageIndex = TunerDataPoolConfig::getInstance()->getLanguage();
   std::string textid = getLanguageId(languageIndex);
   ETG_TRACE_USR4(("VehicleSrvClient::readLanguageOnStartUp textid :%s", textid.c_str()));

   //COURIER_MESSAGE_NEW(Courier::SetCultureReqMsg)(Courier::ItemId(textid.c_str()))->Post();
}


/*
 * onLanguageError - ASF framework callback which is triggered if there is any error in setLanguage()
 * @param[in] proxy
 * @param[in] error
 * @param[out] None
 * @return void
 */
void VehicleSrvClient::onLanguageError(const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< LanguageError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onLanguageError"));
}


void VehicleSrvClient::onTrackCommandStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::TrackCommandStatus >& status)
{
   ETG_TRACE_USR4(("VehicleSrvClient::onTrackCommandStatus trackcommand value  = %d", status->getTrackCommandValue()));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(COCKPIT_SINK_ID))
   {
      tclMediaClientHandler::getInstance().onTrackCommandChangeRequest(status->getTrackCommandValue());
   }
}


void VehicleSrvClient::onTrackCommandError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::TrackCommandError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onTrackCommandError"));
}


void VehicleSrvClient::onCabinMediaSourceCommandStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CabinMediaSourceCommandStatus >& status)
{
   uint8 cabinMediaSourceCommandValue = status->getCabinMediaSourceCommandValue();
   ETG_TRACE_COMP(("VehicleSrvClient::onCabinMediaSourceCommandStatus cabinMediaSourceCommandValue = %d", cabinMediaSourceCommandValue));
   int srcId = SRC_INVALID;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

   switch (cabinMediaSourceCommandValue)
   {
      case CAN_CABIN_SOURCE_ID_USB1:
      {
         int devicetagUSB1 = tclMediaClientHandler::getInstance().getUSB1DeviceTag();
         if (devicetagUSB1 != 0)
         {
            srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_USB);
            MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagUSB1, REGION_CABIN_A);
            if (u8activeRegion == REGION_CABIN_A)
            {
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
               if (_applicationSwitchClientComponent != NULL)
               {
                  _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_USB1);
               }
            }
         }
      }
      break;
      case CAN_CABIN_SOURCE_ID_USB2:
      {
         int devicetagUSB2 = tclMediaClientHandler::getInstance().getUSB2DeviceTag();
         if (devicetagUSB2 != 0)
         {
            srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_USB);
            MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagUSB2, REGION_CABIN_A);
            if (u8activeRegion == REGION_CABIN_A)
            {
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
               if (_applicationSwitchClientComponent != NULL)
               {
                  _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_USB2);
               }
            }
         }
      }
      break;
      case CAN_CABIN_SOURCE_ID_BLUETOOTH:
      {
         int devicetagBT = tclMediaClientHandler::getInstance().getDeviceTag(FI_EN_E8DTY_BLUETOOTH);
         if (devicetagBT != 0)
         {
            srcId = tclMediaClientHandler::getInstance().getSourceId(FI_EN_E8DTY_BLUETOOTH);
            MediaSourceHandling::getInstance().requestSourceActivation(srcId, devicetagBT, REGION_CABIN_A);
            if (u8activeRegion == REGION_CABIN_A)
            {
               StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
               if (_applicationSwitchClientComponent != NULL)
               {
                  _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING);
               }
            }
         }
      }
      break;
      /*case CAN_CABIN_SOURCE_ID_ROUTER_USB:
      {
         if (u8activeRegion == REGION_CABIN_A)
         {
            _externalMediaClientHandler->onSourceSelection(EnumConst::entSourceType_CMR);
            if (_applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_ROUTERUSB);
            }
         }
         else
         {
            _externalMediaClientHandler->onExternalSrcActivationWhenInCockpit(EnumConst::entSourceType_CMR, FI_EN_E8DTY_ROUTER_USB);
         }
      }
      break; */
      case CAN_CABIN_SOURCE_ID_CAMPORT_USB:
      {
         if (u8activeRegion == REGION_CABIN_A)
         {
            _externalMediaClientHandler->onSourceSelection(EnumConst::entSourceType_CMP);
            if (_applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_CAMPORTUSB);
            }
         }
         else
         {
            _externalMediaClientHandler->onExternalSrcActivationWhenInCockpit(EnumConst::entSourceType_CMP, FI_EN_E8DTY_USB_CAMERAPORT);
         }
      }
      break;
      case CAN_CABIN_SOURCE_ID_CAMPORT_HDMI_1:
      {
         if (u8activeRegion == REGION_CABIN_A)
         {
            _externalMediaClientHandler->onHdmiSourceSelection(FI_EN_E8DTY_USB_CAMERAPORT_HDMI);
            if (_applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI);
            }
         }
         else
         {
            _externalMediaClientHandler->onExternalSrcActivationWhenInCockpit(FI_EN_E8DTY_USB_CAMERAPORT_HDMI);
         }
      }
      break;
      case CAN_CABIN_SOURCE_ID_CAMPORT_HDMI_2:
      {
         if (u8activeRegion == REGION_CABIN_A)
         {
            _externalMediaClientHandler->onHdmiSourceSelection(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2);
            if (_applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI2);
            }
         }
         else
         {
            _externalMediaClientHandler->onExternalSrcActivationWhenInCockpit(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2);
         }
      }
      break;
      case CAN_CABIN_SOURCE_ID_CAMPORT_HDMI_3:
      {
         if (u8activeRegion == REGION_CABIN_A)
         {
            _externalMediaClientHandler->onHdmiSourceSelection(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3);
            if (_applicationSwitchClientComponent != NULL)
            {
               _applicationSwitchClientComponent->requestContextSwitchApplication(0, APPID_APPHMI_UNKNOWN, enActivityIDs__eActivityID_NONE, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI3);
            }
         }
         else
         {
            _externalMediaClientHandler->onExternalSrcActivationWhenInCockpit(FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3);
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("MediaPlaybackStatusUpdater::onCabinMediaSourceCommandStatus Default case option"));
      }
      break;
   }
}


void VehicleSrvClient::onCabinMediaSourceCommandError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CabinMediaSourceCommandError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onCabinMediaSourceCommandError"));
}


void VehicleSrvClient::onCabinMediaTrackCommandStatus(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& proxy, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CabinMediaTrackCommandStatus >& status)
{
   uint8 cabinMediaTrackCommandValue = status->getCabinMediaTrackCommandValue();
   ETG_TRACE_USR1(("VehicleSrvClient::onCabinMediaTrackCommandStatus cabinMediaTrackCommandValue  = %d", cabinMediaTrackCommandValue));
   if (MediaSourceHandling::getInstance().isInternalMediaSourceActiveInSink(CABIN_A_SINK_ID))
   {
      tclMediaClientHandler::getInstance().onTrackCommandChangeRequest(cabinMediaTrackCommandValue);
   }
   if (MediaSourceHandling::getInstance().isExternalMediaSourceActiveInSink(CABIN_A_SINK_ID))
   {
      _externalMediaClientHandler->onCabinTrackCommandChangeRequest(cabinMediaTrackCommandValue);
   }
}


void VehicleSrvClient::onCabinMediaTrackCommandError(const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::VEHICLE_MAIN_FI::CabinMediaTrackCommandError >& /*error*/)
{
   ETG_TRACE_COMP(("VehicleSrvClient::onCabinMediaTrackCommandError"));
}


}
}
