/**************************************************************************************
 * @file         : SpeedLockStatusHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *18.07.20 Rev. 0.1 Naga Suman(RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/
#include "hall_std_if.h"
#include "Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "Core/MediaClientHandler/MediaClientHandler.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "ProjectBaseTypes.h"
#include "Core/VehicleSrvClient/VehicleSrvClient.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::SpeedLockStatusHandler::
#include "trcGenProj/Header/SpeedLockStatusHandler.cpp.trc.h"
#endif
#else
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#endif

namespace App {
namespace Core {

/************************************************************************
*FUNCTION		: SpeedLockStatusHandler
*DESCRIPTION	: Destructor.
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
SpeedLockStatusHandler::~SpeedLockStatusHandler()
{
}


SpeedLockStatusHandler* SpeedLockStatusHandler::_SpeedLockStatusHandler = NULL;
SpeedLockStatusHandler& SpeedLockStatusHandler::getInstance()
{
   if (_SpeedLockStatusHandler == 0)
   {
      _SpeedLockStatusHandler = new SpeedLockStatusHandler();
   }
   return *_SpeedLockStatusHandler;
}


void SpeedLockStatusHandler::removeInstance()
{
   if (_SpeedLockStatusHandler)
   {
      delete _SpeedLockStatusHandler;
      _SpeedLockStatusHandler = 0;
   }
}


/************************************************************************
*FUNCTION		: SpeedLockStatusHandler
*DESCRIPTION	: Constructor. Initializes the member variables.
*PARAMETER		:
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
SpeedLockStatusHandler::SpeedLockStatusHandler()
{
   ETG_TRACE_USR1(("SpeedLockStatusHandler::SpeedLockStatusHandler Constructor called"));
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   m_iSurfaceState                         = hmibase::SURFACESTATE_INVISIBLE;
   m_currentView                           = "";
   m_previousView                          = "";
   m_videoPlayingRestrictionConfigValue    = 0;
   m_isVideoSpeedLockActive                       = false;
   readVideoRestrictionValue();
#endif
   m_parkBrakeStatus = PARKING_BRAKE_SNA;
   m_speedValue = DEFAULT_SPEED_VALUE;
}


void SpeedLockStatusHandler::updateParkBrakeStatus(uint16 parkBrakeStatus)
{
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateParkingBrakeStatus() ParkingBreakSwitch is :%d", parkBrakeStatus));
   m_parkBrakeStatus = parkBrakeStatus;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   performVideoRestrictionBasedonSpeedAndParkingBrake();
#endif
}


void SpeedLockStatusHandler::updateSpeedValue(uint16 speedValue)
{
   ETG_TRACE_USR4(("SpeedLockStatusHandler::updateSpeedValue() speedValue is :%d", (speedValue / 100)));
   m_speedValue = speedValue;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   performVideoRestrictionBasedonSpeedAndParkingBrake();
#endif
}


bool SpeedLockStatusHandler::getVideoSpeedLockStatus()
{
   bool isVideoSpeedLockActive = true;

   if ((m_speedValue < MAX_SPEED) && (m_parkBrakeStatus == PARKING_BRAKE_ENG))
   {
      isVideoSpeedLockActive = false; //video plays
   }
   ETG_TRACE_USR4(("SpeedLockStatusHandler::getVideoSpeedLockStatus() isVideoSpeedLockActive is :%d", isVideoSpeedLockActive));
   return isVideoSpeedLockActive;
}


bool SpeedLockStatusHandler::isCurrentViewVideoPlayerScene(std::string currentView)
{
   bool isVideoplayerViewID = false;

   if ((0 == currentView.compare("Media#MediaPlayerScenes#M_VideoPlayer")) ||
         0 == currentView.compare("Media#MediaPlayerScenes#M_VideoPlayer_FullScreen"))
   {
      isVideoplayerViewID = true;
   }
   ETG_TRACE_USR1(("SpeedLockStatusHandler::isCurrentViewVideoPlayerScene() isVideoplayerViewID is :%d", isVideoplayerViewID));
   return isVideoplayerViewID;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void SpeedLockStatusHandler::readVideoRestrictionValue()
{
   ETG_TRACE_USR1(("SpeedLockStatusHandler::readVideoRestrictionValue entered"));
   dp_tclKdsLockout _dp_tclKdsLockoutInst;
   _dp_tclKdsLockoutInst.u8GetVideoPlayingRestriction(m_videoPlayingRestrictionConfigValue);
   ETG_TRACE_USR4(("SpeedLockStatusHandler::readVideoRestrictionValue() m_videoPlayingRestrictionConfigValue is :%d", m_videoPlayingRestrictionConfigValue));
}


bool SpeedLockStatusHandler::isSurfaceVisibleOnLayer(const unsigned int layerID, const int surfaceID)
{
   ETG_TRACE_USR4(("SpeedLockStatusHandler::isSurfaceVisibleOnLayer layerID is :%d", layerID));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::isSurfaceVisibleOnLayer surfaceID is :%d", surfaceID));

   bool returnValue = false;
   ilmErrorTypes err = ILM_FAILED;
   t_ilm_int surfaceCount = 0;
   t_ilm_surface* pAllSurfacesurfaces = NULL;

   err = ilm_getSurfaceIDsOnLayer(layerID, &surfaceCount, &pAllSurfacesurfaces);
   if (err == ILM_SUCCESS)
   {
      for (int index = 0; index < surfaceCount; index++)
      {
         int surfaceid = pAllSurfacesurfaces[index];
         if (surfaceid == surfaceID)
         {
            ilmSurfaceProperties surfaceProperties;
            err = ilm_getPropertiesOfSurface(surfaceid, &surfaceProperties);

            if (err == ILM_SUCCESS && surfaceProperties.visibility)
            {
               returnValue = true;
               break;
            }
         }
      }
   }
   ETG_TRACE_USR4(("SpeedLockStatusHandler::isSurfaceVisibleOnLayer returnValue is :%d", returnValue));
   return returnValue;
}


void SpeedLockStatusHandler::handleMediaVideoLayerVisibility(bool visibility)
{
   ilmErrorTypes l_error = ilm_layerSetVisibility(MEDIA_VIDEO_LAYERID, visibility);
   ETG_TRACE_USR4(("SpeedLockStatusHandler::handleMediaVideoLayerVisibility()  Error value %d", l_error));
   if (ILM_SUCCESS == l_error)
   {
      l_error = ilm_commitChanges();
      if (ILM_SUCCESS == l_error)
      {
         ETG_TRACE_USR1(("SpeedLockStatusHandler::handleMediaVideoLayerVisibility()  Layers and surfaces enabled"));
      }

      else
      {
         ETG_TRACE_USR1(("SpeedLockStatusHandler::handleMediaVideoLayerVisibility()  Error in enabling layers and surfaces"));
      }
   }

   else
   {
      ETG_TRACE_USR1(("SpeedLockStatusHandler::handleMediaVideoLayerVisibility() Error in enabling layers and surfaces"));
   }
}


bool SpeedLockStatusHandler::onCourierMessage(const ActiveRenderedView& msg)
{
   ETG_TRACE_USR1(("SpeedLockStatusHandler::onCourierMessage ActiveRenderedView surface id %d ,view name %s", msg.GetSurfaceId(), msg.GetViewName().GetCString()));

   if (msg.GetSurfaceId() == SURFACEID_MAIN_SURFACE_MEDIA)
   {
      m_currentView = msg.GetViewName().GetCString();
      //pass current view as parameter and store ret type in a bool var & use the bool var for comparison
      if ((0 == m_currentView.compare("Media#MediaPlayerScenes#M_VideoPlayer")) &&
            (0 != m_previousView.compare("Media#MediaPlayerScenes#M_VideoPlayer_FullScreen")))
      {
         performVideoSpeedRestrictionOperation();
      }

      m_previousView = m_currentView ;
   }

   ETG_TRACE_USR3(("SpeedLockStatusHandler::onCourierMessage ActiveRenderedView m_currentView %s", m_currentView.c_str()));

   return false;
}


bool SpeedLockStatusHandler::onCourierMessage(const MediaSurfaceStateUpdMsg& msg)
{
   ETG_TRACE_USR1(("SpeedLockStatusHandler::onCourierMessage MediaSurfaceStateUpdMsg surface id %d ,state %d", msg.GetSurfaceId(), msg.GetState()));
   m_iSurfaceState = msg.GetState();
   if (msg.GetState() == hmibase::SURFACESTATE_VISIBLE)
   {
      performVideoSpeedRestrictionOperation();
   }
   else if (msg.GetState() == hmibase::SURFACESTATE_INVISIBLE)
   {
      handleMediaVideoLayerVisibility(false);
      m_previousView = "";
      m_currentView = "";
   }

   return false;
}


void SpeedLockStatusHandler::performVideoRestrictionBasedonSpeedAndParkingBrake()
{
   bool isVideoSpeedLockActive = getVideoSpeedLockStatus();
   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoRestrictionBasedonSpeedAndParkingBrake() isVideoSpeedLockActive is :%d", isVideoSpeedLockActive));

   if (m_isVideoSpeedLockActive != isVideoSpeedLockActive)
   {
      m_isVideoSpeedLockActive = isVideoSpeedLockActive;
      performVideoSpeedRestrictionOperation() ;
   }
}


void SpeedLockStatusHandler::performVideoSpeedRestrictionOperation()
{
   int isGuiUpdated = HMIModelComponent::getInstance().getGuiStartupStatus();

   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoSpeedRestrictionOperation() isGuiUpdated :%d", isGuiUpdated));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoSpeedRestrictionOperation() m_videoPlayingRestrictionConfigValue is :%d", m_videoPlayingRestrictionConfigValue));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoSpeedRestrictionOperation() m_isVideoSpeedLockActive is:%d", m_isVideoSpeedLockActive));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoSpeedRestrictionOperation() m_iSurfaceState is:%d", m_iSurfaceState));
   ETG_TRACE_USR4(("SpeedLockStatusHandler::performVideoSpeedRestrictionOperation() g_SM_ActiveMediaSource is:%d", g_SM_ActiveMediaSource));

   if (isGuiUpdated == 1)
   {
      if (m_videoPlayingRestrictionConfigValue)
      {
         if (isCurrentViewVideoPlayerScene(m_currentView) == true
               && ((g_SM_ActiveMediaSource == FI_EN_E8DTY_USB) || (g_SM_ActiveMediaSource == FI_EN_E8DTY_MTP))
               && (m_iSurfaceState == hmibase::SURFACESTATE_VISIBLE))
         {
            if (m_isVideoSpeedLockActive == false)
            {
               POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_VideoDisableMessage_Close)()));
               handleMediaVideoLayerVisibility(true);
            }
            else if (m_isVideoSpeedLockActive == true)
            {
               POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_VideoDisableMessage_Open)()));
               handleMediaVideoLayerVisibility(false);
            }
         }
      }
      else
      {
         POST_MSG((COURIER_MESSAGE_NEW(Popup_Pfo_VideoDisableMessage_Close)()));
         handleMediaVideoLayerVisibility(true);
      }
   }
}


#endif
}//end of namespace Core
}//end of namespace App
