/**************************************************************************************
* @file         : MediaSourceHandling.cpp
* @author       : RBEI/ECG5-INF4CV_MediaTeam
* @addtogroup   : AppHmi_media
* @brief        :
* @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "hall_std_if.h"
#endif

#include "Core/SourceSwitch/MediaSourceHandling.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "../ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/ExternalMediaClientHandler/ExternalMediaClientHandler.h"
#include "../MediaDefines.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/ExternalMediaSourceHandler/ExternalMediaSourceHandler.h"
#include "App/datapool/MediaDataPoolConfig.h"
#include "Common/VariantHandling/VariantHandling.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::MediaSourceHandling::
#include "trcGenProj/Header/MediaSourceHandling.cpp.trc.h"
#endif

using namespace MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange;
using namespace MASTERAUDIOSERVICE_INTERFACE::SoundProperties;
using namespace org::genivi::audiomanager::CommandInterface;

#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#endif

namespace App {
namespace Core {


MediaSourceHandling* MediaSourceHandling::_mediaSourceHandling = NULL;


MediaSourceHandling& MediaSourceHandling::getInstance()
{
   if (_mediaSourceHandling == 0)
   {
      _mediaSourceHandling = new MediaSourceHandling();
   }
   return *_mediaSourceHandling;
}


/**
 * @Destructor
 */
MediaSourceHandling::~MediaSourceHandling()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

   _audioSourceChangeProxy.reset();
   _soundPropertiesProxy.reset();
   _commandIFProxy.reset();
   m_MuteStatusMap.clear();
   _activeSourceInfo.clear();
   _isMicActive = false;

#endif
   _mediaSourceHandling = NULL;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST


MediaSourceHandling::MediaSourceHandling()

   : _audioSourceChangeProxy(AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this))
   , _soundPropertiesProxy(SoundPropertiesProxy::createProxy("soundPropertiesPort", *this))
   , _commandIFProxy(::CommandInterfaceProxy::createProxy("commandInterfacePort", *this))
   , _externalMediaClientHandler(NULL)
   , _externalMediaSourceHandler(NULL)

{
   ETG_TRACE_USR1(("MediaSourceHandling::MediaSourceHandling constructor \n"));
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "audioSourceChangePort");
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "soundPropertiesPort");
   StartupSync::getInstance().registerPropertyRegistrationIF(this, "commandInterfacePort");
   audioSrcMap.clear();
   mediaSrcMap.clear();
   ETG_I_REGISTER_FILE();
   _mediaSourceHandling = this;
   m_u32RequestedSubSrcId = 255;
   m_requestedRegionId = REGION_INVALID;
   m_u8CurrentDeviceTag = 0;
   initialiseAudioSourceMap();
   subSrcMedia = 0; //
   m_u8sinkId = 0; //
   m_bMuteStatus = false;
   m_Devicetype = -1;
   m_usbdevicenum = 0;
   m_iLastSubsrcid = 0;
   m_iLastSrcid = 0;
   _disconnectedSource = SRC_INVALID;
   InitialiseSrcAudioMap();
   m_MuteStatusMap.clear();
   _activeSourceInfo.clear();
   initialiseSourceAvailabilityMap();
   _isMicActive = false;
   _prevSrcid = 0;


#ifdef VARIANT_S_FTR_ENABLE_VIDEO_PLAYBACK
   m_Audiosrc_Cockpit.clear();
   m_Audiosrc_Cabin.clear();
#endif
}


#else
/**
 * @Constructor
 */
MediaSourceHandling::MediaSourceHandling()
{
   m_SrcId = 0;
   m_iSubSrcValue = 0;
}


#endif

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void MediaSourceHandling::InitialiseSrcAudioMap()
{
   if (g_SM_VARIANT_AUX == 0)
   {
      m_SubsrctoAudiosrc.clear();
      m_Audiosrc.clear();
   }
   else
   {
      m_SubsrctoAudiosrc.insert(pair<int, int>(-1, SRC_MEDIA_AUX));//DEFAULT AUX is inserted for smart code
      m_Audiosrc.push_back(m_SubsrctoAudiosrc);
      m_SubsrctoAudiosrc.clear();
   }
}


void MediaSourceHandling::initialiseAudioSourceMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::initialiseAudioSourceMap is entered \n"));

   for (int sinkId = 0; sinkId < CABIN_B_SINK_ID; sinkId++)
   {
      audioSrcMap[sinkId].srcid  = SRC_INVALID;
      audioSrcMap[sinkId].subsrcid = SUBSRCID_INVALID;
      mediaSrcMap[sinkId].srcid = SRC_INVALID;
      mediaSrcMap[sinkId].subsrcid = SUBSRCID_INVALID;
   }
}


void MediaSourceHandling::initialiseSourceAvailabilityMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::initialiseSourceAvailabilityMap is entered \n"));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CAMERAPORT1_USB_AUDIO, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CAMERAPORT1_USB_VIDEO, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CAMERAPORT1_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CAMERAPORT2_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CAMERAPORT3_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CMR1_USB1_VIDEO, EN_EXTERNAL_MEDIA_DISCONNECTED));
   m_ExternalSourceAvailabilityMap.insert(pair<int, int>(SRC_CMR1_USB1_AUDIO, EN_EXTERNAL_MEDIA_DISCONNECTED));
}


/**
 * registerProperties - Trigger property registration to hmi_master,  called from MediaHall class
 * @param[in] proxy
 * @parm[in] stateChange - state change service for corrosponding  proxy
 * @return void
 */
void MediaSourceHandling::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::registerProperties"));
}


/**
 * deregisterProperties - Trigger property deregistration to hmi_master,  called from MediaHall class
 * @param[in] proxy
 * @parm[in] stateChange - state change service for corrosponding  proxy
 * @return void
 */
void MediaSourceHandling::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::deregisterProperties"));
}


// ServiceAvailableIF implementation
/**
 * onUnavailable -
 * @param[in] proxy
 * @param[in] stateChange
 */
void MediaSourceHandling::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onUnavailable \n"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendDeregisterAll();
   }
   else if (_commandIFProxy && _commandIFProxy == proxy)
   {
      _commandIFProxy->sendMainSinkSoundPropertyChangedDeregisterAll();
   }
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeDeregisterAll();
      _soundPropertiesProxy->sendMuteStateDeregisterAll();
      _soundPropertiesProxy->sendMuteStateMapDeregisterAll();
   }
}


/**
 * onAvailable -
 * @param[in] proxy
 * @param[in] stateChange
 */
void MediaSourceHandling::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onAvailable\n"));
   if (_audioSourceChangeProxy && _audioSourceChangeProxy == proxy)
   {
      _audioSourceChangeProxy->sendSourceListChangedRegister(*this);
      _audioSourceChangeProxy->sendGetSourceListRequest(*this, 2); //2 is group id for media i.e GROUP_MEDIA
      _audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_OTHERS);
      _audioSourceChangeProxy->sendActiveSourceListRegister(*this);
      _audioSourceChangeProxy->sendActiveSourceRegister(*this);
      _audioSourceChangeProxy->sendSourceAvailabilityChangedRegister(*this);

      _audioSourceChangeProxy->sendActiveSourceGet(*this);
   }
   else if (_commandIFProxy && _commandIFProxy == proxy)
   {
      _commandIFProxy->sendMainSinkSoundPropertyChangedRegister(*this);
      _commandIFProxy->sendGetListMainSinkSoundPropertiesRequest(*this, 1); //sink id is 1
   }
   else if (_soundPropertiesProxy && _soundPropertiesProxy == proxy)
   {
      _soundPropertiesProxy->sendVolumeRegister(*this);
      _soundPropertiesProxy->sendVolumeGet(*this);
      _soundPropertiesProxy->sendMuteStateRegister(*this);
      _soundPropertiesProxy->sendMuteStateGet(*this);
      _soundPropertiesProxy->sendMuteStateMapRegister(*this);
   }
}


/**
 * onActivateSourceError - Notification from HMI master when source requested for activation is unsuccessful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onActivateSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActivateSourceError >& error)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onActivateSourceError \n"));
}


/**
 * onActivateSourceResponse - Notification from HMI master when source requested for activation is successful
 * @param[in] proxy
 * @parm[in] response
 * @return void
 */
void MediaSourceHandling::onActivateSourceResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActivateSourceResponse >& /*response*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onActivateSourceResponse \n"));
   //current active device
}


/**
 * onDeactivateSourceError - Notification from HMI master when request for source deactivation is not successful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onDeactivateSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::DeactivateSourceError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onDeactivateSourceError\n"));
}


/**
 * onDeactivateSourceResponse - Notification from HMI master when request for source deactivation is successful
 * @param[in] proxy
 * @parm[in] response
 * @return void
 */
void MediaSourceHandling::onDeactivateSourceResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::DeactivateSourceResponse >& /*response*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onDeactivateSourceResponse \n"));
}


/**
 * onSourceListChangedError - Error notification from HMI master when sourceListChange error happens
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onSourceListChangedError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceListChangedError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSourceListChangedError\n"));
}


/**
 * onSourceListChangedSignal - Notification from HMI master when SourceListChange is obtained
 * @param[in] proxy
 * @parm[in] signal
 * @return void
 */
void MediaSourceHandling::onSourceListChangedSignal(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceListChangedSignal >& signal)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSourceListChangedSignal\n"));
   if (_audioSourceChangeProxy && signal->getGroupId() == GROUP_MEDIA)
   {
      ETG_TRACE_USR1(("MediaSourceHandling::onSourceListChangedSignal sendGetSourceListRequest for media"));
   }
}


/**
 * onGetSourceListError - Notification from HMI master when SourceListChange request is unsuccessful
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onGetSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::GetSourceListError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onGetSourceListError\n"));
}


void MediaSourceHandling::onGetSourceListResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::GetSourceListResponse >& response)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onGetSourceListResponse \n"));

   if (_audioSourceChangeProxy == proxy)
   {
      const ::std::vector< sourceDetails >& SourceList = response->getSources();
      ::std::vector< sourceDetails >::const_iterator it = SourceList.begin();
      while (it != SourceList.end())
      {
         ETG_TRACE_USR4(("MediaSourceHandling::onGetSourceListResponse Sourcelistsize:%d", SourceList.size()));
         ETG_TRACE_USR4(("MediaSourceHandling::onGetSourceListResponse m_u32RequestedSubSrcId = %d ", m_u32RequestedSubSrcId));
         ETG_TRACE_USR4(("MediaSourceHandling::onGetSourceListResponse it->getSrcId() = %d ", it->getSrcId()));
         ETG_TRACE_USR4(("MediaSourceHandling::onGetSourceListResponse it->getSubSrcId() = %d", it->getSubSrcId()));
         ETG_TRACE_USR4(("MediaSourceHandling::onGetSourceListResponse it->getAvailability() = %d", it->getAvailability()));
         if (it->getSrcId() == SRC_MEDIA_PLAYER || it->getSrcId() == SRC_MEDIA_AUX)//smart
         {
            if ((it->getSubSrcId() == m_u32RequestedSubSrcId) && (it->getAvailability() == 1))
            {
               int sinkid;
               sinkid = ((m_requestedRegionId == MAX_REG) ? COCKPIT_SINK_ID : ((m_requestedRegionId == REG_A) ? CABIN_A_SINK_ID : CABIN_B_SINK_ID));
               ETG_TRACE_FATAL(("onGetSourceListResponse sinkid = %d", sinkid));
               sourceData sourceInfo(it->getSrcId(), it->getSubSrcId(), sinkid, 0);
               _audioSourceChangeProxy->sendActivateSourceRequest(*this, sourceInfo, true);
               break;
            }
         }
         else if ((it->getSrcId() == SRC_PHONE_BTAUDIO) && (m_u32RequestedSubSrcId == it->getSubSrcId()))
         {
            int sinkid;
            sinkid = ((m_requestedRegionId == MAX_REG) ? COCKPIT_SINK_ID : ((m_requestedRegionId == REG_A) ? CABIN_A_SINK_ID : CABIN_B_SINK_ID));
            sourceData sourceInfo(it->getSrcId(), it->getSubSrcId(), sinkid, 0);
            _audioSourceChangeProxy->sendActivateSourceRequest(*this, sourceInfo, true);
            break;
         }
         else  if (it->getSrcId() == SRC_CAMERAPORT1_USB_VIDEO || it->getSrcId() == SRC_CAMERAPORT1_USB_AUDIO)
         {
            m_ExternalSourceAvailabilityMap[it->getSrcId()] = it->getAvailability();
            if (_externalMediaSourceHandler != NULL)
            {
               _externalMediaSourceHandler->updateCamportUsbSourceAvailabilityObservors(it->getAvailability());
            }
         }
         else if (it->getSrcId() == SRC_CAMERAPORT1_HDMI || it->getSrcId() == SRC_CAMERAPORT2_HDMI || it->getSrcId() == SRC_CAMERAPORT3_HDMI)
         {
            m_ExternalSourceAvailabilityMap[it->getSrcId()] = it->getAvailability();
            if (_externalMediaSourceHandler != NULL)
            {
               _externalMediaSourceHandler->updateHDMISourceAvailabilityObservors(it->getSrcId(), it->getAvailability());
            }
         }
         else if (it->getSrcId() == SRC_CMR1_USB1_VIDEO || it->getSrcId() == SRC_CMR1_USB1_AUDIO)
         {
            m_ExternalSourceAvailabilityMap[it->getSrcId()] = it->getAvailability();
         }
         ++it;
      } // End of while loop
   }
}


/**
 * vRequestSourceActivation - helper function tused by Media classes to request a source activation
 * @param[in] SrcID, SubSrcID - source id and sub source id
 * @return void
 */
void MediaSourceHandling::requestSourceActivation(int32 SrcID, int32 SubSrcID, int32 regionID)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestSourceActivation SrcID %d ", SrcID));
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestSourceActivation SubSrcID %d", SubSrcID));
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestSourceActivation regionID %d", regionID));

   int sinkid;
   m_u32RequestedSubSrcId = SubSrcID;
   m_requestedRegionId = regionID;
   switch (SrcID)
   {
      case SRC_MEDIA_PLAYER:
      case SRC_PHONE_BTAUDIO:
      {
         _audioSourceChangeProxy->sendGetSourceListRequest(*this, GROUP_MEDIA); //TODO:need to check group media id
         break;
      }
      case SRC_MEDIA_AUX:
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_HDMI:
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      {
         sinkid = ((regionID == MAX_REG) ? COCKPIT_SINK_ID : ((regionID == REG_A) ? CABIN_A_SINK_ID : CABIN_B_SINK_ID));
         sourceData sourceInfo(SrcID, -1, sinkid, 0);
         _audioSourceChangeProxy->sendActivateSourceRequest(*this, sourceInfo, true);
         break;
      }
      default:
      {
         break;
      }
   }
}


void MediaSourceHandling::vRequestSourceDeactivation(int32 SrcID, int32 SubSrcID, int32 regionID)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestSourceDeactivation SrcID %d SubSrcID  %d  regionID  %d", SrcID, SubSrcID, regionID));;
   int sinkid = ((regionID == MAX_REG) ? COCKPIT_SINK_ID : ((regionID == REG_A) ? CABIN_A_SINK_ID : CABIN_B_SINK_ID));
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestSourceDeactivation sinkid %d", sinkid));
   if ((SrcID == SRC_MEDIA_PLAYER) || (SrcID == SRC_PHONE_BTAUDIO))
   {
      sourceData sourceInfo(SrcID, getSubSourceId(sinkid), sinkid, 0);
      _audioSourceChangeProxy->sendDeactivateSourceRequest(*this, sourceInfo);
   }
   else if (SrcID == SRC_CAMERAPORT1_USB_VIDEO || SrcID == SRC_CAMERAPORT1_USB_AUDIO ||
            SrcID == SRC_CAMERAPORT1_HDMI || SrcID == SRC_CAMERAPORT2_HDMI || SrcID == SRC_CAMERAPORT3_HDMI ||
            SrcID == SRC_MEDIA_AUX || SrcID == SRC_CMR1_USB1_VIDEO || SrcID == SRC_CMR1_USB1_AUDIO)
   {
      sourceData sourceInfo(SrcID, -1, sinkid, 0);
      _audioSourceChangeProxy->sendDeactivateSourceRequest(*this, sourceInfo);
   }
   else
   {
      ETG_TRACE_USR1(("MediaSourceHandling::vRequestSourceDeactivation else case "));
   }
}


void MediaSourceHandling::vRequestGlassSourceDeactivation(int32 SrcID, int SinkId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vRequestGlassSourceDeactivation SrcID %d SinkId  %d ", SrcID, SinkId));
   sourceData sourceInfo(SrcID, -1, SinkId, 0);
   _audioSourceChangeProxy->sendDeactivateSourceRequest(*this, sourceInfo);
}


int MediaSourceHandling::getSubSourceId(int sinkid)
{
   int iSubSrcId = SUBSRCID_INVALID;
   ETG_TRACE_USR4(("MediaSourceHandling::getActiveSourceInSink sinkid %d ", sinkid))
   if (!audioSrcMap.empty())
   {
      std::map<int, audioutility>::iterator itr = audioSrcMap.find(sinkid);
      if (itr != audioSrcMap.end())
      {
         iSubSrcId = itr->second.subsrcid;
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getActiveSourceInSink iSubSrcId  %d ", iSubSrcId));
   return iSubSrcId;
}


/**
 * onActiveSourceError - Notification from HMI master when  active source property update is unsuccessful
 * @param[in] proxy, error
 * @return void
 */
void MediaSourceHandling::onActiveSourceError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onActiveSourceError"));
}


#endif


bool MediaSourceHandling::IsMediaSourceRange(int srcid)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::IsMediaSourceRange:sourceid %d ", srcid));
#endif
   if (((srcid >= SRC_MEDIA_AUX) && (srcid <= SRC_SPI_ENTERTAIN)) ||
         ((srcid >= SRC_CAMERAPORT1_CAM1) && (srcid <= SRC_CAMERAPORT4_HDMI)) || (srcid == SRC_CAMERAPORT1_USB_AUDIO) || (srcid == SRC_CMR1_USB1_VIDEO) || (srcid == SRC_CMR1_USB1_AUDIO))
   {
      return true;
   }
   else
   {
      return false;
   }
}


bool MediaSourceHandling::IsTunerSourceRange(int srcid)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::IsTunerSourceRange:sourceid %d ", srcid));
#endif
   if (((srcid >= SRC_TUNER_FM) && (srcid <= SRC_DAB)) || (srcid == SRC_TUNER_XM) || (srcid == SRC_TUNER_DRM) || (srcid == SRC_TUNER_WX))
   {
      return true;
   }
   else
   {
      return false;
   }
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void MediaSourceHandling::onActiveSourceUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceUpdate >& update)
{
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate SrcId:%d  SubSrcId:%d  SinkSrcId:%d", update->getActiveSource().getSrcId(), update->getActiveSource().getSubSrcId(), update->getActiveSource().getSinkId()));
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate SRC_TUNER_FM: %d, SRC_TUNER_AM: %d, SRC_DAB: %d", SRC_TUNER_FM, SRC_TUNER_AM, SRC_DAB));
   int SrcId =  update->getActiveSource().getSrcId();
   int mSrc_id = 0;
   int mSubSrc_id = 0;

   m_u8sinkId = update->getActiveSource().getSinkId();
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate sinkId %d ConnectionState:%d", m_u8sinkId, update->getActiveSource().getConnectionState()));

   int prevSrcId = audioSrcMap[m_u8sinkId].srcid;
   _prevSrcid = prevSrcId;
   int prevSubSrcId = audioSrcMap[m_u8sinkId].subsrcid;
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate prevSrcId %d", prevSrcId));
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate prevSubSrcId %d", prevSubSrcId));

   uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate ui8ActiveRegion :%d", ui8ActiveRegion));

   int iSinkId = ((ui8ActiveRegion == REGION_COCKPIT) ? COCKPIT_SINK_ID : ((ui8ActiveRegion == REGION_CABIN_A) ? CABIN_A_SINK_ID : CABIN_B_SINK_ID));

   if ((IsMediaSourceRange(update->getActiveSource().getSrcId())) && ((prevSrcId != update->getActiveSource().getSrcId()) || (prevSubSrcId != update->getActiveSource().getSubSrcId())) &&
         ((update->getActiveSource().getConnectionState() == STATE_CONNECTED) || (update->getActiveSource().getConnectionState() == STATE_SUSPENDED)))
   {
      bool rangeval = IsMediaSourceRange(update->getActiveSource().getSrcId());
      if (rangeval)
      {
         if (update->getActiveSource().getSrcId() == SRC_SPI_ENTERTAIN)
         {
            HMIModelComponent::getInstance().updateSpidataVisibilityonMediaGadget(true);
         }
         else
         {
            HMIModelComponent::getInstance().updateSpidataVisibilityonMediaGadget(false);
         }
      }
      mSrc_id = update->getActiveSource().getSrcId();
      mSubSrc_id = update->getActiveSource().getSubSrcId();

      ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate mSrc_id %d", mSrc_id));
      ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate mSubSrc_id %d", mSubSrc_id));

      if ((m_u8sinkId == COCKPIT_SINK_ID) || (m_u8sinkId == CABIN_A_SINK_ID) || (m_u8sinkId == CABIN_B_SINK_ID))
      {
         if (update->getActiveSource().getSrcId() != SRC_NAVI_SPEECH)
         {
            audioSrcMap[m_u8sinkId].srcid = update->getActiveSource().getSrcId();
            audioSrcMap[m_u8sinkId].subsrcid = update->getActiveSource().getSubSrcId();

            mediaSrcMap[m_u8sinkId].srcid = update->getActiveSource().getSrcId();
            mediaSrcMap[m_u8sinkId].subsrcid = update->getActiveSource().getSubSrcId();
            ETG_TRACE_USR4(("MediaSourceHandling::onActiveSourceUpdate: audioSrcMap[m_u8sinkId].srcid %d, audioSrcMap[m_u8sinkId].subsrcid %d", audioSrcMap[m_u8sinkId].srcid, audioSrcMap[m_u8sinkId].subsrcid));
         }

         //This map data is used for source highlight
         insertDataToMapForSrcHighlight(update);
      }
      else if (m_u8sinkId == CABIN_A_MEDIAGLASS1_SINK_ID || m_u8sinkId == CABIN_A_MEDIAGLASS2_SINK_ID)
      {
         audioSrcMap[m_u8sinkId].srcid = update->getActiveSource().getSrcId();
         audioSrcMap[m_u8sinkId].subsrcid = update->getActiveSource().getSubSrcId();
      }

      //if source id other than mediaplayed then set the subsrc as 255
      if (update->getActiveSource().getSrcId() == SRC_MEDIA_PLAYER)
      {
         clMetaData::getInstance().setSubSrcID(update->getActiveSource().getSubSrcId());// store the device tag to identify the last played source
         subSrcMedia = update->getActiveSource().getSubSrcId();
         clMetaData::getInstance().setSrcID(update->getActiveSource().getSrcId());
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
         m_Devicetype = tclMediaClientHandler::getInstance().GetMediaDeviceType(subSrcMedia);// getSrcMediaPlayerType();
         m_usbdevicenum = tclMediaClientHandler::getInstance().GetActiveUSBSource(mSubSrc_id);
         HMIModelComponent::getInstance().vUpdateNextPerviousNormalState();
         tclMediaClientHandler::getInstance().getNowplayingStatus();
      }
      else if (update->getActiveSource().getSrcId() == SRC_MEDIA_AUX)//smart
      {
         clMetaData::getInstance().setSubSrcID(update->getActiveSource().getSubSrcId());// store the device tag to identify the last played source
         clMetaData::getInstance().setSrcID(update->getActiveSource().getSrcId());// store the device src id to identify the last played source
         HMIModelComponent::getInstance().bOnUpdateHeaderMuteStatus();
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
         m_Devicetype = 7;
         m_usbdevicenum = 0;
         HMIModelComponent::getInstance().vUpdateNextPerviousNormalState();
         tclMediaClientHandler::getInstance().setActiveAUXSource();
      }
      else if (update->getActiveSource().getSrcId() == SRC_PHONE_BTAUDIO)
      {
         AddMediaSources(mSubSrc_id, mSrc_id, update->getActiveSource().getSinkId());
         clMetaData::getInstance().setSubSrcID(update->getActiveSource().getSubSrcId()); // store the device tag to identify the last played source
         clMetaData::getInstance().setSrcID(update->getActiveSource().getSrcId());
         m_usbdevicenum = 0;
         m_Devicetype = 4;
         tclMediaClientHandler::getInstance().setUSBInactive();
         HMIModelComponent::getInstance().vUpdateNextPerviousNormalState();
         tclMediaClientHandler::getInstance().setActiveBTSource();
         tclMediaClientHandler::getInstance().getNowplayingStatus();
      }
      else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_USB_VIDEO ||
               update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_USB_AUDIO)
      {
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
         m_Devicetype = FI_EN_E8DTY_USB_CAMERAPORT;
         HMIModelComponent::getInstance().vUpdateUsbAlbumIconAndTextVisibility();
         HMIModelComponent::getInstance().vUpdateNextPerviousNormalState();
         if (_externalMediaSourceHandler->isMediaSrcAvailableInPreviousSrcList(CABIN_A_SINK_ID) && ((SrcId != prevSrcId) && (prevSrcId != SRC_CAMERAPORT1_IPCAM && prevSrcId != SRC_CAMERAPORT2_IPCAM && prevSrcId != SRC_CAMERAPORT3_IPCAM && prevSrcId != SRC_CAMERAPORT4_IPCAM)))
         {
            _externalMediaClientHandler->resetMetadataForExternalUsbSources(mSrc_id);
         }
      }
      else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_HDMI)
      {
         m_Devicetype = FI_EN_E8DTY_USB_CAMERAPORT_HDMI;
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
      }
      else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT2_HDMI)
      {
         m_Devicetype = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2;
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
      }
      else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT3_HDMI)
      {
         m_Devicetype = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3;
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
      }
      else if (update->getActiveSource().getSrcId() == SRC_CMR1_USB1_VIDEO ||
               update->getActiveSource().getSrcId() == SRC_CMR1_USB1_AUDIO)
      {
         m_Devicetype = FI_EN_E8DTY_ROUTER_USB;
         //  _externalMediaClientHandler->resetMetadataForExternalUsbSources(mSrc_id);
         AddMediaSources(mSubSrc_id, mSrc_id, m_u8sinkId);
         HMIModelComponent::getInstance().vUpdateNextPerviousNormalState();
         if (_externalMediaSourceHandler->isMediaSrcAvailableInPreviousSrcList(CABIN_A_SINK_ID) && ((SrcId != prevSrcId) && (prevSrcId != SRC_CAMERAPORT1_IPCAM)))
         {
            _externalMediaClientHandler->resetMetadataForExternalUsbSources(mSrc_id);
         }
      }
      else
      {}

      if (iSinkId == m_u8sinkId)
      {
         HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(m_Devicetype);
         StateTransitionHandler::getInstance().stateTransit_onActiveSourceUpdate();
         HMIModelComponent::getInstance().updateMediaSourceInfoInGadget(m_Devicetype);
      }

      tclMediaClientHandler::getInstance().vResetSeekedSliderFlag();
      HMIModelComponent::getInstance().vUpdateCabinGadgetVisibility();
      HMIModelComponent::getInstance().vSourceButtonHighlight(m_Devicetype, mSubSrc_id);
      MediaSourceListHandler::getInstance().updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
   }
   else if (IsTunerSourceRange(update->getActiveSource().getSrcId()))
   {
      ETG_TRACE_USR1(("MediaSourceHandling::onActiveSourceUpdate other than media and temporary sources"));
      mSrc_id = update->getActiveSource().getSrcId();
      mSubSrc_id = update->getActiveSource().getSubSrcId();
      HMIModelComponent::getInstance().SourceButtonHighlight(SRC_INVALID, SRC_INVALID);
      if (iSinkId == m_u8sinkId)
      {
         HMIModelComponent::getInstance().vResetDeviceTypeInSM();
      }
      audioSrcMap[m_u8sinkId].srcid = update->getActiveSource().getSrcId();
      audioSrcMap[m_u8sinkId].subsrcid = update->getActiveSource().getSubSrcId();
      tclMediaClientHandler::getInstance().vResetSeekedSliderFlag();
      //This map data is used for source highlight
      insertDataToMapForSrcHighlight(update);
      MediaSourceListHandler::getInstance().updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
   }
}


bool MediaSourceHandling::getMic1Status()
{
   ETG_TRACE_COMP(("MediaSourceHandling::getMic1Status _isMicActive %d \n", _isMicActive));
   return _isMicActive;
}


#endif


uint8  MediaSourceHandling::GetActiveMediaSource(int iSrcId, int devType)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::GetActiveMediaSource iSrcId %d,devicetype %d \n", iSrcId, devType));
#endif
   uint8 activeMediaSource = FI_EN_E8DTY_UNKNOWN;
   if (iSrcId == SRC_MEDIA_PLAYER)
   {
      if ((devType == FI_EN_E8DTY_USB) || (devType == FI_EN_E8DTY_MTP))
      {
         activeMediaSource = FI_EN_E8DTY_USB;
      }
      else if ((devType == FI_EN_E8DTY_IPOD) || (devType == FI_EN_E8DTY_IPHONE))
      {
         activeMediaSource = FI_EN_E8DTY_IPOD;
      }
      else
      {
         //do nothing
      }
   }
   else if (iSrcId == SRC_PHONE_BTAUDIO)
   {
      activeMediaSource = FI_EN_E8DTY_BLUETOOTH;
   }
   else if (iSrcId == SRC_CAMERAPORT1_USB_VIDEO || iSrcId == SRC_CAMERAPORT1_USB_AUDIO)
   {
      activeMediaSource = FI_EN_E8DTY_USB_CAMERAPORT;
   }
   else if (iSrcId == SRC_CAMERAPORT1_HDMI)
   {
      activeMediaSource = FI_EN_E8DTY_USB_CAMERAPORT_HDMI;
   }
   else if (iSrcId == SRC_CAMERAPORT2_HDMI)
   {
      activeMediaSource = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2;
   }
   else if (iSrcId == SRC_CAMERAPORT3_HDMI)
   {
      activeMediaSource = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3;
   }
   else if (iSrcId == SRC_MEDIA_AUX)
   {
      activeMediaSource = FI_EN_E8DTY_AUX;
   }
   else if (iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
   {
      activeMediaSource = FI_EN_E8DTY_ROUTER_USB;
   }
   else
   {
      //do nothing
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::GetActiveMediaSource activeMediaSource: %d\n", activeMediaSource));
#endif
   return activeMediaSource;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void  MediaSourceHandling::vSetSourceHighlight()
{
   int iCurrentSubSrcId = 0;
   int ideviceType = 0;

   uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::vSetSourceHighlight:ui8ActiveRegion :%d", ui8ActiveRegion));

   if (ui8ActiveRegion == REGION_COCKPIT)
   {
      iCurrentSubSrcId = audioSrcMap[COCKPIT_SINK_ID].subsrcid;
      ideviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iCurrentSubSrcId);
   }
   else if (ui8ActiveRegion == REGION_CABIN_A)
   {
      iCurrentSubSrcId = audioSrcMap[CABIN_A_SINK_ID].subsrcid;
      if (iCurrentSubSrcId == CAMPORTUSB)
      {
         ideviceType = FI_EN_E8DTY_USB_CAMERAPORT;
      }
      else
      {
         ideviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iCurrentSubSrcId);
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::vSetSourceHighlight iCurrentSubSrcId  %d ", iCurrentSubSrcId));
   ETG_TRACE_USR4(("MediaSourceHandling::vSetSourceHighlight ideviceType  %d  ", ideviceType));

   HMIModelComponent::getInstance().vSourceButtonHighlight(ideviceType, iCurrentSubSrcId);
}


#endif


int MediaSourceHandling::vUpdateGadgetIcons(int iSrcId, int currDevType)
{
   bool isCCVariant = false;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::vUpdateGadgetIcons sourceid %d,current device type %d \n", iSrcId, currDevType));
#endif
   int gadgetIcon = -1;
   if (VariantHandling::getInstance() != NULL && VariantHandling::getInstance()->getVariantType() == VARIANTTYPE_CC)
   {
      isCCVariant = true;
   }

   switch (iSrcId)
   {
      case SRC_MEDIA_PLAYER:
      {
         if ((currDevType == FI_EN_E8DTY_USB) || (currDevType == FI_EN_E8DTY_MTP))
         {
            gadgetIcon = isCCVariant ? GADGET_SRC_ICON_IDX_USB_CC : GADGET_SRC_ICON_IDX_USB;
         }
         else if ((currDevType == FI_EN_E8DTY_IPOD) || (currDevType == FI_EN_E8DTY_IPHONE))
         {
            gadgetIcon = isCCVariant ? GADGET_SRC_ICON_IDX_IPOD_CC : GADGET_SRC_ICON_IDX_IPOD;
         }
         else
         {
            //do nothing
         }
         break;
      }
      case SRC_PHONE_BTAUDIO:
      {
         gadgetIcon = isCCVariant ? GADGET_SRC_ICON_IDX_BT_CC : GADGET_SRC_ICON_IDX_BT;

         break;
      }
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      {
         gadgetIcon = isCCVariant ? GADGET_SRC_ICON_IDX_USBCAMPORT_CC : GADGET_SRC_ICON_IDX_USBCAMPORT;

         break;
      }
      case SRC_CAMERAPORT1_HDMI:
      {
         gadgetIcon =  isCCVariant ? GADGET_SRC_ICON_IDX_HDMI_CC : GADGET_SRC_ICON_IDX_HDMI;

         break;
      }
      case SRC_CAMERAPORT2_HDMI:
      {
         gadgetIcon = isCCVariant ? GADGET_SRC_ICON_IDX_HDMI2_CC : GADGET_SRC_ICON_IDX_HDMI2;

         break;
      }
      case SRC_CAMERAPORT3_HDMI:
      {
         gadgetIcon =  isCCVariant ? GADGET_SRC_ICON_IDX_HDMI3_CC : GADGET_SRC_ICON_IDX_HDMI3;

         break;
      }
      case SRC_MEDIA_AUX:
      {
         if (!isCCVariant)
         {
            gadgetIcon = GADGET_SRC_ICON_IDX_AUX;
         }

         break;
      }
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      {
         if (!isCCVariant)
         {
            gadgetIcon = GADGET_SRC_ICON_IDX_ROUTER;
         }

         break;
      }

      default:
      {
         gadgetIcon = INVALID_GADGET_INDEX;
         break;
      }
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("MediaSourceHandling::vUpdateGadgetIcons gadgeticon %d \n", gadgetIcon));
#endif
   return gadgetIcon;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
/**
* CheckSubsrcid - Gets called on change in any mediasource(onactivesourceupdate)
1.On disconnection we are deleting the vector element for that particular key(subsrcid or we can say devicetag)
2.On onactivesourceupdate for media,we are checking if subsrc id is already present in map/not.if it is present we will delete that and add it again ,to maintain like stack
* @param[in] int m_subsrcid, int srcid
* @return bool
*/
bool MediaSourceHandling::UpdateSrcidtoMap(int m_subsrcid, int srcid)
{
   ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap subsrcid :%d", m_subsrcid));
   ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap srcid:%d", srcid));
   std::vector< std::multimap <int, int> > Audiosrc;

   uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap ActiveRegion :%d", ui8ActiveRegion));

   if (m_u8sinkId == COCKPIT_SINK_ID)
   {
      Audiosrc = m_Audiosrc_Cockpit;
   }
   else
   {
      Audiosrc = m_Audiosrc_Cabin;
   }

   for (int index = 0; index < Audiosrc.size(); index++)
   {
      std::multimap <int, int> ::const_iterator mapitr = Audiosrc[index].begin();
      for (mapitr = Audiosrc[index].begin(); mapitr != Audiosrc[index].end(); mapitr++)
      {
         ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap mapitr->first :%d", mapitr->first));
         ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap  mapitr->second:%d", mapitr->second));

         if (mapitr->first == m_subsrcid)
         {
            if (m_subsrcid == -1)
            {
               if (mapitr->second == SRC_CAMERAPORT1_HDMI)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CAMERAPORT2_HDMI)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CAMERAPORT3_HDMI)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CAMERAPORT1_USB_VIDEO)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CAMERAPORT1_USB_AUDIO)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CMR1_USB1_VIDEO)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               else if (mapitr->second == SRC_CMR1_USB1_AUDIO)
               {
                  Audiosrc[index].erase(m_subsrcid);
                  Audiosrc.erase(Audiosrc.begin() + index);
               }
               m_SubsrctoAudiosrc.clear();
               m_SubsrctoAudiosrc.insert(pair<int, int>(m_subsrcid, srcid));
            }
            else
            {
               Audiosrc[index].erase(m_subsrcid);
               Audiosrc.erase(Audiosrc.begin() + index);
               m_SubsrctoAudiosrc.clear();
               m_SubsrctoAudiosrc.insert(pair<int, int>(m_subsrcid, srcid));
            }

            if (m_u8sinkId == COCKPIT_SINK_ID)
            {
               m_Audiosrc_Cockpit.clear();
               m_Audiosrc_Cockpit = Audiosrc;
               m_Audiosrc_Cockpit.push_back(m_SubsrctoAudiosrc);
            }
            else
            {
               m_Audiosrc_Cabin.clear();
               m_Audiosrc_Cabin = Audiosrc;
               m_Audiosrc_Cabin.push_back(m_SubsrctoAudiosrc);
            }
            m_SubsrctoAudiosrc.clear();

            if (ui8ActiveRegion == REGION_COCKPIT)
            {
               m_Audiosrc.clear();
               m_Audiosrc = m_Audiosrc_Cockpit;
            }
            else
            {
               m_Audiosrc.clear();
               m_Audiosrc = m_Audiosrc_Cabin;
            }
            return true;
         }
      }
   }
   return false;
}


void  MediaSourceHandling::UpdateSrcidtoMap(int m_subsrcid)
{
   ETG_TRACE_USR4(("MediaSourceHandling::UpdateSrcidtoMap:Erase subsrc id :%d", m_subsrcid));
   for (int index = 0; index < m_Audiosrc.size(); index++)
   {
      std::map <int, int> ::const_iterator mapitr = m_Audiosrc[index].begin();
      for (mapitr = m_Audiosrc[index].begin(); mapitr != m_Audiosrc[index].end(); mapitr++)
      {
         if (mapitr->first == m_subsrcid)
         {
            m_Audiosrc[index].erase(m_subsrcid);
            m_Audiosrc.erase(m_Audiosrc.begin() + index);
            // m_SubsrctoAudiosrc.clear();
            break;
         }
      }
   }
}


void  MediaSourceHandling::removeExternalSrcIdfromMap(int iSrcId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::removeExternalSrcIdfromMap:: Erase src id :%d", iSrcId));
   for (int index = 0; index < m_Audiosrc_Cabin.size(); index++)
   {
      std::map <int, int> ::const_iterator mapitr = m_Audiosrc_Cabin[index].begin();
      for (mapitr = m_Audiosrc_Cabin[index].begin(); mapitr != m_Audiosrc_Cabin[index].end(); mapitr++)
      {
         if (mapitr->second == iSrcId)
         {
            m_Audiosrc_Cabin[index].erase(iSrcId);
            m_Audiosrc_Cabin.erase(m_Audiosrc_Cabin.begin() + index);
            break;
         }
      }
   }
}


void MediaSourceHandling::removeMediaSourcefromVector(int subsrcid)
{
   ETG_TRACE_USR4(("MediaSourceHandling::removeMediaSourcefromVector sub source id:%d", subsrcid));
   UpdateSrcidtoMap(subsrcid);
   mediaSourcesInVector();
}


void MediaSourceHandling::removeExternalMediaSourcefromVector(int iSrcId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::removeExternalMediaSourcefromVector SrcId :%d", iSrcId));
   removeExternalSrcIdfromMap(iSrcId);
   MediaSourceHandling::getInstance().setActiveSourceStatus();
   mediaSourcesInVector();
}


void MediaSourceHandling::mediaSourcesInVector()
{
   for (int index = 0; index < m_Audiosrc.size(); index++)
   {
      for (map<int, int>::const_iterator iter1 = m_Audiosrc[index].begin(); iter1 != m_Audiosrc[index].end(); iter1++)
      {
         ETG_TRACE_USR4(("MediaSourceHandling::mediaSourcesInVector:key->subsrcid :%d", iter1->first));
         ETG_TRACE_USR4(("MediaSourceHandling::mediaSourcesInVector:value->srcid :%d", iter1->second));
      }
   }
}


/**
* lastActiveMediaSource - Gets called on press of applfly inn media button
This function will help to give the available media source by doing vector.back(which gives the latest media source)
If no media source is available and user presses appflyinn by default we are send update for AUX
* @param[in] empty
* @return void
*/
void MediaSourceHandling::lastActiveMediaSource()
{
   ETG_TRACE_USR4(("MediaSourceHandling::lastActiveMediaSource audiosrc.size :%d", m_Audiosrc.size()));
   if (m_Audiosrc.size() > 0)
   {
      std::multimap <int, int>  Audiosrcmap = m_Audiosrc.back();
      // m_SubsrctoAudiosrc = m_Audiosrc.back();
      m_iLastSubsrcid = (Audiosrcmap.begin())->first;
      m_iLastSrcid = (Audiosrcmap.begin())->second;
      ETG_TRACE_USR4(("MediaSourceHandling::lastActiveMediaSource:first element(key) subsrcid :%d", m_iLastSubsrcid));
      ETG_TRACE_USR4(("MediaSourceHandling::lastActiveMediaSource:Second element(value) srcid  :%d", m_iLastSrcid));
   }
}


void MediaSourceHandling::AddMediaSources(int SubSrc_id, int Src_id, int Sink_id)
{
   ETG_TRACE_USR4(("MediaSourceHandling::AddMediaSources SubSrc_id:%d, Src_id :%d,sink id:%d", SubSrc_id, Src_id, Sink_id));

   bool val = UpdateSrcidtoMap(SubSrc_id, Src_id);
   ETG_TRACE_USR4(("MediaSourceHandling::AddMediaSources value:%d", val));

   if (val == false)
   {
      uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
      ETG_TRACE_USR4(("MediaSourceHandling::AddMediaSources activeRegion   %d ", ui8ActiveRegion));
      m_SubsrctoAudiosrc.insert(pair<int, int>(SubSrc_id, Src_id));
      if (m_u8sinkId == COCKPIT_SINK_ID)
      {
         m_Audiosrc_Cockpit.push_back(m_SubsrctoAudiosrc);
         m_Audiosrc = m_Audiosrc_Cockpit;
      }
      else
      {
         m_Audiosrc_Cabin.push_back(m_SubsrctoAudiosrc);
         m_Audiosrc = m_Audiosrc_Cabin;
      }
      // m_Audiosrc.push_back(m_SubsrctoAudiosrc);

      m_SubsrctoAudiosrc.clear();
   }
   mediaSourcesInVector();
}


int MediaSourceHandling::GetLastSrcId()
{
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastSrcId:last source id:%d ", m_iLastSrcid));

   return m_iLastSrcid;
}


int MediaSourceHandling::GetLastSubSrcId()
{
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastSubSrcId last sub source id :%d ", m_iLastSubsrcid));

   return m_iLastSubsrcid;
}


int MediaSourceHandling::GetdeviceType()
{
   ETG_TRACE_USR4(("MediaSourceHandling::GetdeviceType device type :%d ", m_Devicetype));

   return m_Devicetype;
}


int MediaSourceHandling::GetUSBDeviceNum()
{
   ETG_TRACE_USR4(("MediaSourceHandling::GetUSBDeviceNum:usbdevice number :%d ", m_usbdevicenum));

   return m_usbdevicenum;
}


void MediaSourceHandling::setCurrentDeviceTag(int8 u8CurrentDeviceTag)
{
   ETG_TRACE_USR4(("MediaSourceHandling::setCurrentDeviceTag:current device tag :%d ", u8CurrentDeviceTag));

   m_u8CurrentDeviceTag = u8CurrentDeviceTag;
}


int8 MediaSourceHandling::getCurrentDeviceTag()
{
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentDeviceTag :current device tag:%d ", m_u8CurrentDeviceTag));

   return m_u8CurrentDeviceTag;
}


void MediaSourceHandling::setMuteStatus(bool bMuteStatus)
{
   m_bMuteStatus = bMuteStatus;
}


//bool MediaSourceHandling::getMuteStatus()
//{
//   return m_bMuteStatus;
//}


int MediaSourceHandling::GetLastMediaSrcId()
{
   int iMediaSrcValue = 0;

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastMediaSrcId u8activeRegion %d ", u8activeRegion));

   if (u8activeRegion == REGION_COCKPIT)
   {
      iMediaSrcValue = mediaSrcMap[COCKPIT_SINK_ID].srcid;
   }
   else if (u8activeRegion == REGION_CABIN_A)
   {
      iMediaSrcValue = mediaSrcMap[CABIN_A_SINK_ID].srcid;
   }
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastMediaSrcId MediaSrcValue  %d ", iMediaSrcValue));

   return iMediaSrcValue;
}


int MediaSourceHandling::GetLastMediaSubSrcId()
{
   int iMediaSubSrcValue = 0;

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastMediaSubSrcId u8activeRegion  %d ", u8activeRegion));

   if (u8activeRegion == REGION_COCKPIT)
   {
      iMediaSubSrcValue = mediaSrcMap[COCKPIT_SINK_ID].subsrcid;
   }
   else if (u8activeRegion == REGION_CABIN_A)
   {
      iMediaSubSrcValue = mediaSrcMap[CABIN_A_SINK_ID].subsrcid;
   }
   ETG_TRACE_USR4(("MediaSourceHandling::GetLastMediaSubSrcId iMediaSubSrcValue  %d ", iMediaSubSrcValue));

   return iMediaSubSrcValue;
}


void MediaSourceHandling::setActiveSourceStatus()
{
   uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::setActiveSourceStatus activeRegion   %d ", ui8ActiveRegion));
   m_Audiosrc.clear();

   if (ui8ActiveRegion == REGION_COCKPIT)
   {
      m_Audiosrc = m_Audiosrc_Cockpit;
   }
   else
   {
      m_Audiosrc = m_Audiosrc_Cabin;
   }
}


/**
* onSetMainSinkSoundPropertyError - Error handling for SetMainSinkSoundProperty
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void MediaSourceHandling::onSetMainSinkSoundPropertyError(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetMainSinkSoundPropertyError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetMainSinkSoundPropertyError \n"));
}


/**
* onSetMainSinkSoundPropertyResponse - response for SetMainSinkSoundProperty
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void MediaSourceHandling::onSetMainSinkSoundPropertyResponse(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetMainSinkSoundPropertyResponse >& /*response*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetMainSinkSoundPropertyResponse\n"));
}


/**
* onMainSinkSoundPropertyChangedError - Error handling for sendGetListMainSinkSoundPropertiesRequest
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void MediaSourceHandling::onMainSinkSoundPropertyChangedError(const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::MainSinkSoundPropertyChangedError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onMainSinkSoundPropertyChangedError \n"));
}


/**
 * onMainSinkSoundPropertyChangedSignal - Status/Signal handling for MainSinkSoundPropertyChanged
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void MediaSourceHandling::onMainSinkSoundPropertyChangedSignal(const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::MainSinkSoundPropertyChangedSignal >& signal)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onMainSinkSoundPropertyChangedSignal \n"));
}


/**
 * onGetListMainSinkSoundPropertiesError - Error handling for GetListMainSinkSoundProperties
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void MediaSourceHandling::onGetListMainSinkSoundPropertiesError(const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::GetListMainSinkSoundPropertiesError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onGetListMainSinkSoundPropertiesError \n"));
}


/**
 * onGetListMainSinkSoundPropertiesResponse - Response handling for GetListMainSinkSoundProperties
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void MediaSourceHandling::onGetListMainSinkSoundPropertiesResponse(const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr<AUDIOMANAGER_COMMANDINTERFACE::GetListMainSinkSoundPropertiesResponse >& response)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onGetListMainSinkSoundPropertiesResponse\n"));
}


/**
* onSetVolumeError - Error handling for SetVolume
* @param[in] proxy
* @param[in] error
* @param[out] none
* @return void
*/
void MediaSourceHandling::onSetVolumeError(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetVolumeError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetVolumeError"));
}


/**
 * onSetVolumeResponse - Response handling for SetVolume
 * @param[in] proxy
 * @param[in] error
 * @param[out] none
 * @return void
 */
void MediaSourceHandling::onSetVolumeResponse(const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::CommandInterfaceProxy >& /*proxy*/,
      const ::boost::shared_ptr< AUDIOMANAGER_COMMANDINTERFACE::SetVolumeResponse >& /*response*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetVolumeResponse\n"));
}


/**
 * onVolumeError - This status is received from HMI_Master if there are any errors in volume.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void MediaSourceHandling::onVolumeError(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties:: VolumeError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onVolumeError\n"));
}


/**
 * onVolumeUpdate - This status is received from HMI_Master if there is any change in property volume.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void MediaSourceHandling::onVolumeUpdate(const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr<MASTERAUDIOSERVICE_INTERFACE::SoundProperties:: VolumeUpdate >& update)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onVolumeUpdate\n"));
}


/**
 * onMuteStateError - This status is received from HMI_Master if there are any errors in muteState.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void MediaSourceHandling::onMuteStateError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onMuteStateError\n"));
}


/**
 * onMuteStateUpdate - This status is received from HMI_Master if there is any change in property mutestate.
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void MediaSourceHandling::onMuteStateUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::MuteStateUpdate >& update)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onMuteStateUpdate\n"));
}


/**
 * onSetMuteStateError - called if SetMuteState method is updated with error from media player
 *
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onSetMuteStateError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMuteStateError >& /*error*/)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetMuteStateError\n"));
}


/**
 * onSetMuteStateResponse - Receives info about the set MuteState response and updates the same on video main scene
 *
 * @param[in] proxy
 * @parm[in] onMuteStateUpdate
 * @return void
 */
void MediaSourceHandling::onSetMuteStateResponse(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::SoundProperties::SetMuteStateResponse >& response)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onSetMuteStateResponse \n"));
}


/**
 * onActiveSourceListError - This status is received from HMI_Master if there are any errors in ActiveSourceList request.
 *
 * @param[in] proxy
 * @parm[in] error
 * @return void
 */
void MediaSourceHandling::onActiveSourceListError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListError >& error)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onActiveSourceListError\n"));
}


/**
 * onActiveSourceListUpdate - This status is received from HMI_Master on ActiveSourceList request.
 *
 * @param[in] proxy
 * @parm[in] status
 * @return void
 */
void MediaSourceHandling::onActiveSourceListUpdate(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& proxy,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceListUpdate >& signal)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onActiveSourceListUpdate()"));

   if (signal != NULL)
   {
      std::vector<sourceData>AudioSourceList = signal->getActiveSourceList();
      if (!AudioSourceList.empty())
      {
         std::vector<sourceData>::const_iterator iter = AudioSourceList.begin(); // stack top is list bottom
         _isMicActive = false;
         _activeSourceInfo.clear();
         while (iter != AudioSourceList.end())
         {
            if (IsMediaSourceRange(iter->getSrcId()) == true)
            {
               _activeSourceInfo.insert(pair<int, int>(iter->getSinkId(), iter->getSrcId()));
            }
            if ((iter->getSrcId() == SRC_MIC1_ANNOUNCEMENT) && (_isMicActive == false))
            {
               if (((iter->getSinkId() == CABIN_A_SINK_ID) || (iter->getSinkId() == CABIN_B_SINK_ID)) && (iter->getConnectionState() == STATE_CONNECTED))
               {
                  _isMicActive = true;
               }
            }
            ETG_TRACE_USR3(("MediaSourceHandling::onActiveSourceListUpdate iter SrcID  %d Sink Id   %d   ConnectionState %d, _isMicActive %d", iter->getSrcId(), iter->getSinkId(), iter->getConnectionState(), _isMicActive));
            iter++;
         }

         checkInternalSourceDisconnected();
      }
   }
}


bool MediaSourceHandling::isMediaPlayerAvailableInCabinSink(int sinkId)
{
   bool bSourceAvail = false;
   if (!_activeSourceInfo.empty())
   {
      std::map<int, int>::iterator itr = _activeSourceInfo.find(sinkId);
      if (itr != _activeSourceInfo.end())
      {
         if (itr->second == SRC_MEDIA_PLAYER)
         {
            bSourceAvail = true;
         }
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isMediaPlayerAvailableInCabinSink %d, sinkId %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


bool MediaSourceHandling::isBTAvailableInCabinSink(int sinkId)
{
   bool bSourceAvail = false;
   if (!_activeSourceInfo.empty())
   {
      std::map<int, int>::iterator itr = _activeSourceInfo.find(sinkId);
      if (itr != _activeSourceInfo.end())
      {
         if (itr->second == SRC_PHONE_BTAUDIO)
         {
            bSourceAvail = true;
         }
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isBTAvailableInCabinSink %d, sinkId %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


bool MediaSourceHandling::isMediaPlayerAvailableInPreviousSourceList(int sinkId)
{
   bool bSourceAvail = false;
   std::map<int, audioutility>::iterator itr = audioSrcMap.find(sinkId);
   if (itr != audioSrcMap.end())
   {
      if (itr->second.srcid == SRC_MEDIA_PLAYER)
      {
         _disconnectedSource = itr->second.srcid;
         bSourceAvail = true;
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isMediaPlayerAvailableInPreviousSourceList %d, sinkId  %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


bool MediaSourceHandling::isBTAvailableInPreviousSourceList(int sinkId)
{
   bool bSourceAvail = false;
   std::map<int, audioutility>::iterator itr = audioSrcMap.find(sinkId);
   if (itr != audioSrcMap.end())
   {
      if (itr->second.srcid == SRC_PHONE_BTAUDIO)
      {
         _disconnectedSource = itr->second.srcid;
         bSourceAvail = true;
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isBTAvailableInPreviousSourceList %d, sinkId  %d", bSourceAvail, sinkId));
   return bSourceAvail;
}


void MediaSourceHandling::checkInternalSourceDisconnected()
{
   if (!isMediaPlayerAvailableInCabinSink(CABIN_A_SINK_ID))
   {
      bool isMediaPlayerSrcAvailBefore = isMediaPlayerAvailableInPreviousSourceList(CABIN_A_SINK_ID);
      ETG_TRACE_USR4(("MediaSourceHandling::checkInternalSourceDisconnected CABIN A isMediaPlayerSrcAvailBefore %d  ", isMediaPlayerSrcAvailBefore));
      if (isMediaPlayerSrcAvailBefore)
      {
         vResetAudioSrcMapofCabin(_disconnectedSource);
      }
   }
   if (!isBTAvailableInCabinSink(CABIN_A_SINK_ID))
   {
      bool isBTSrcAvailBefore = isBTAvailableInPreviousSourceList(CABIN_A_SINK_ID);
      ETG_TRACE_USR4(("MediaSourceHandling::checkInternalSourceDisconnected CABIN A isBTSrcAvailBefore %d  ", isBTSrcAvailBefore));
      if (isBTSrcAvailBefore)
      {
         vResetAudioSrcMapofCabin(_disconnectedSource);
      }
   }
   _disconnectedSource = SRC_INVALID;
   ETG_TRACE_USR1(("MediaSourceHandling::checkInternalSourceDisconnected  Exited"));
}


void MediaSourceHandling::vResetCabinGadgetAndAudioSrcMap(int iSrcId, uint32 cabinRegion)
{
   int iCurrentRegionSrcId = getCurrentRegionSrcId();
   ETG_TRACE_USR4(("MediaSourceHandling::vResetCabinGadgetAndAudioSrcMap iSrcId %d   cabinRegion %d", iSrcId, cabinRegion));
   ETG_TRACE_USR4(("MediaSourceHandling::vResetCabinGadgetAndAudioSrcMap iCurrentRegionSrcId %d", iCurrentRegionSrcId));

   if (iCurrentRegionSrcId == iSrcId)
   {
      HMIModelComponent::getInstance().vResetDeviceTypeInSM();
   }
   int iCurrentSinkId = getSinkId(cabinRegion);
   ETG_TRACE_USR4(("MediaSourceHandling::vResetCabinGadgetAndAudioSrcMap iCurrentSinkId %d", iCurrentSinkId));
   audioSrcMap[iCurrentSinkId].srcid = SRC_INVALID;
   audioSrcMap[iCurrentSinkId].subsrcid = SUBSRCID_INVALID;
   removeExternalMediaSourcefromVector(iSrcId);
   HMIModelComponent::getInstance().vUpdateCabinGadgetVisibility();
}


void MediaSourceHandling::vResetAudioSrcMapofCabin(int iSrcId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vResetAudioSrcMapofCabin iSrcId   %d", iSrcId));
   int iCabinASrcId = audioSrcMap[CABIN_A_SINK_ID].srcid;
   int iCurrentRegionSrcId = getCurrentRegionSrcId();
   ETG_TRACE_USR4(("MediaSourceHandling::vResetAudioSrcMapofCabin iCabinASrcId %d", iCabinASrcId));
   ETG_TRACE_USR4(("MediaSourceHandling::vResetAudioSrcMapofCabin iCurrentRegionSrcId %d", iCurrentRegionSrcId));

   if (iCabinASrcId == iSrcId)
   {
      audioSrcMap[CABIN_A_SINK_ID].srcid = SRC_INVALID;
      audioSrcMap[CABIN_A_SINK_ID].subsrcid = SUBSRCID_INVALID;
      _externalMediaClientHandler->setHmiOffWithActiveMediaSrcState(false);
      if (iCurrentRegionSrcId == iSrcId)
      {
         HMIModelComponent::getInstance().vResetDeviceTypeInSM();
         HMIModelComponent::getInstance().vUpdateCabinGadgetVisibility();
         if (iSrcId == SRC_CAMERAPORT1_USB_VIDEO  || iSrcId == SRC_CAMERAPORT1_USB_AUDIO || iSrcId == SRC_CAMERAPORT1_HDMI || iSrcId == SRC_CAMERAPORT2_HDMI || iSrcId == SRC_CAMERAPORT3_HDMI || iSrcId == SRC_CMR1_USB1_VIDEO || iSrcId == SRC_CMR1_USB1_AUDIO)
         {
            removeExternalMediaSourcefromVector(iSrcId);
            vUpdateMediaSrcMapOnDisconnection(EXTERNAL_SOURCE_SUBSRCID);
         }
         StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_DEVICE_REMOVAL);
         StateTransitionHandler::getInstance().stateTransit_onActiveSourceDeviceDisconnection();
         MediaSourceListHandler::getInstance().updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
      }
   }
}


bool MediaSourceHandling::isCamportAvailableInCabinAudioSrcMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::isCamportAvailableInCabinAudioSrcMap Enetered"));

   bool bSourceAvail = false;

   for (int index = 0; index < m_Audiosrc_Cabin.size(); index++)
   {
      for (map<int, int>::const_iterator iter1 = m_Audiosrc_Cabin[index].begin(); iter1 != m_Audiosrc_Cabin[index].end(); ++iter1)
      {
         if (iter1->second == SRC_CAMERAPORT1_USB_VIDEO || iter1->second == SRC_CAMERAPORT1_USB_AUDIO)
         {
            bSourceAvail = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isCamportAvailableInCabinAudioSrcMap SourceAvail %d", bSourceAvail));
   return bSourceAvail;
}


bool MediaSourceHandling::isHdmiAvailableInCabinAudioSrcMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::isHdmiAvailableInCabinAudioSrcMap Enetered"));

   bool bSourceAvail = false;

   for (int index = 0; index < m_Audiosrc_Cabin.size(); index++)
   {
      for (map<int, int>::const_iterator iter1 = m_Audiosrc_Cabin[index].begin(); iter1 != m_Audiosrc_Cabin[index].end(); ++iter1)
      {
         if (iter1->second == SRC_CAMERAPORT1_HDMI || iter1->second == SRC_CAMERAPORT2_HDMI || iter1->second == SRC_CAMERAPORT3_HDMI)
         {
            bSourceAvail = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::isHdmiAvailableInCabinAudioSrcMap SourceAvail %d", bSourceAvail));
   return bSourceAvail;
}


void MediaSourceHandling::onMuteStateMapError(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateMapError >& error)
{
   ETG_TRACE_COMP(("MediaSourceHandling::onMuteStateMapError \n"));
}


void MediaSourceHandling::onMuteStateMapUpdate(const ::boost::shared_ptr< SoundPropertiesProxy >& /*proxy*/, const ::boost::shared_ptr< MuteStateMapUpdate >& update)
{
   m_MuteStatusMap = update->getMuteStateMap();
   std::map<uint16, bool>::iterator itr;

   for (std::map<uint16, bool>::iterator itr = m_MuteStatusMap.begin(); itr != m_MuteStatusMap.end(); ++itr)
   {
      ETG_TRACE_USR4(("MediaSourceHandling::onMuteStateMapUpdate %d", itr->first));
      ETG_TRACE_USR4(("MediaSourceHandling::onMuteStateMapUpdate %d", itr->second));
   }

   HMIModelComponent::getInstance().bOnUpdateHeaderMuteStatus();
   HMIModelComponent::getInstance().vUpdateMuteStatus();
   //mcode
   HMIModelComponent::getInstance().vUpdateNextPerviousIconStatus();
}


bool MediaSourceHandling::getCurrentRegionMuteStatus()
{
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatus eneteredd "));
   bool bMuteStatus = false;
   uint8 iActiveRegion = REGION_INVALID;

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatus u8activeRegion cabin  %d ", u8activeRegion));

   std::map<uint16, bool>::iterator itr;
   for (std::map<uint16, bool>::iterator itr = m_MuteStatusMap.begin(); itr != m_MuteStatusMap.end(); ++itr)
   {
      ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatus %d", itr->second));
      if (itr->second == true)
      {
         ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatusMutestatus %d", itr->first));
         int iSinkId = itr->first;
         iActiveRegion = ((iSinkId == COCKPIT_SINK_ID) ? REGION_COCKPIT : ((iSinkId == CABIN_A_SINK_ID) ? REGION_CABIN_A : REGION_CABIN_B));
         ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatus iActiveRegion  %d ", iActiveRegion));
         if (iActiveRegion == u8activeRegion)
         {
            bMuteStatus = true;
            break;
         }
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionMuteStatusMutestatus bMuteStatus %d", bMuteStatus));
   return bMuteStatus;
}


/************************************************************************
*FUNCTION		: insertMapDataForSrcHighlight
*DESCRIPTION	: This method is called to update map data for source screen highlight.
*PARAMETER		: NA
*RETURNVALUE	: void
************************************************************************/
void MediaSourceHandling::insertDataToMapForSrcHighlight(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::ActiveSourceUpdate > update)
{
   ETG_TRACE_USR4(("MediaSourceHandling::insertDataToMapForSrcHighlight sinkId %d", m_u8sinkId));
   ETG_TRACE_USR4(("MediaSourceHandling::insertDataToMapForSrcHighlight COCKPIT source id, %d", update->getActiveSource().getSrcId()));

   switch (m_u8sinkId)
   {
      case COCKPIT_SINK_ID:
      {
         if ((update->getActiveSource().getSrcId() == SRC_MEDIA_PLAYER) || (update->getActiveSource().getSrcId() == SRC_PHONE_BTAUDIO))
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].activeSource = INTERNALSOURCE;
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].source = E_SOURCE_MEDIA;

            //If media(INTERNALSOURCE source) is activated in cockpit and if media(INTERNALSOURCE source) is playing in cabin,
            //then change the source of cabin also to media(INTERNALSOURCE source).
            //Ex: cabin is iPod & cockpit is changed to USB1, then cabin also should be iPod.
            if (true == isMediaSrcActive(REGION_CABIN_A))
            {
               MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].activeSource = INTERNALSOURCE;
               MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].deviceGroup = update->getActiveSource().getSrcId();
            }
         }
         else if ((SRC_TUNER_FM == update->getActiveSource().getSrcId()) || (SRC_TUNER_AM == update->getActiveSource().getSrcId()) || (SRC_DAB == update->getActiveSource().getSrcId()))
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].source = E_SOURCE_TUNER;
            HMIModelComponent::getInstance().clearSourceButtonHighlight();
         }
         else
         {
         }

         MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].deviceGroup = update->getActiveSource().getSrcId();
      }
      break;

      case CABIN_A_SINK_ID:
      {
         if ((update->getActiveSource().getSrcId() == SRC_MEDIA_PLAYER) || (update->getActiveSource().getSrcId() == SRC_PHONE_BTAUDIO))
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].activeSource = INTERNALSOURCE;
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].source = E_SOURCE_MEDIA;

            //If media(INTERNALSOURCE source) is activated in cabin and if media(INTERNALSOURCE source) is playing in cockpit,
            //then change the source of cockpit also to media(INTERNALSOURCE source).
            //Ex: cockpit is USB1 & cabin is changed to BT then cockpit also should be BT.
            if (true == isMediaSrcActive(REGION_COCKPIT))
            {
               MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].activeSource = INTERNALSOURCE;
               MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].deviceGroup = update->getActiveSource().getSrcId();
            }

            HMIModelComponent::getInstance().vUpdateStatusHomescreen(INTERNALSOURCE);

            uint8_t playState = MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].playPauseState;
            ETG_TRACE_USR4(("MediaSourceHandling::insertDataToMapForSrcHighlight playState in cockpit   %d", playState));
            HMIModelComponent::getInstance().vUpdatePlayStateOnHomeScreen(playState);
            // Sends the play / pause state to HMI Model component
            HMIModelComponent::getInstance().vUpdatePlayBackState(playState);

            HMIModelComponent::getInstance().bOnUpdateNowPlayingStatus();
            tclMediaClientHandler::getInstance().UpdateAlbumArt();
         }
         else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_USB_VIDEO ||
                  update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_USB_AUDIO)
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].activeSource = CAMPORTUSB;
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].source = E_SOURCE_DEFAULT;
         }
         else if (update->getActiveSource().getSrcId() == SRC_CAMERAPORT1_HDMI || update->getActiveSource().getSrcId() == SRC_CAMERAPORT2_HDMI || update->getActiveSource().getSrcId() == SRC_CAMERAPORT3_HDMI)
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].activeSource = CAMPORTHDMI;
         }
         else if (update->getActiveSource().getSrcId() == SRC_CMR1_USB1_VIDEO ||
                  update->getActiveSource().getSrcId() == SRC_CMR1_USB1_AUDIO)
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].activeSource = ROUTERUSB;
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].source = E_SOURCE_DEFAULT;
         }
         else if ((SRC_TUNER_FM == update->getActiveSource().getSrcId()) || (SRC_TUNER_AM == update->getActiveSource().getSrcId()) || (SRC_DAB == update->getActiveSource().getSrcId()))
         {
            MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].source = E_SOURCE_TUNER;
            HMIModelComponent::getInstance().clearSourceButtonHighlight();
         }
         else
         {
            ETG_TRACE_USR4(("MediaSourceHandling::insertDataToMapForSrcHighlight DEFAULT SOURCE: %d", update->getActiveSource().getSrcId()));
         }

         MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].deviceGroup = update->getActiveSource().getSrcId();
      }
      break;

      default:
      {
      }
      break;
   }
}


/************************************************************************
*FUNCTION		: getSrcMediaPlayerType
*DESCRIPTION	: This method is called on region change, to highlight the media in source selection.
*PARAMETER		: NA
*RETURNVALUE	: void
************************************************************************/
void MediaSourceHandling::setSourceBtnHighlight()
{
   uint8 ui8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::setSourceBtnHighlight ActiveRegion = %d", ui8ActiveRegion));

   HMIModelComponent::getInstance().clearSourceButtonHighlight();

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(ui8ActiveRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iDeviceGroup = MediaUtils::getInstance().m_ActiveDeviceDataMap[ui8ActiveRegion].deviceGroup;
      int iMPDeviceType = tclMediaClientHandler::getInstance().getMediaPlayerDeviceType();
      ETG_TRACE_USR4(("MediaSourceHandling::setSourceBtnHighlight iDeviceGroup: %d", iDeviceGroup));
      ETG_TRACE_USR4(("MediaSourceHandling::setSourceBtnHighlight iMPDeviceType: %d", iMPDeviceType));
      uint8 iDevType = GetActiveMediaSource(iDeviceGroup, iMPDeviceType);
      ETG_TRACE_USR4(("MediaSourceHandling::setSourceBtnHighlight iDevType: %d", iDevType));
   }
}


/************************************************************************
*FUNCTION		: isMediaSrcActive
*DESCRIPTION	: This method returns the src type, tuner or media based on region
*PARAMETER		: int iRegion
*RETURNVALUE	: bool
************************************************************************/
bool MediaSourceHandling::isMediaSrcActive(int iRegion)
{
   ETG_TRACE_USR4(("MediaSourceHandling::isMediaSrcActive iRegion: %d", iRegion));
   ETG_TRACE_USR4(("MediaSourceHandling::isMediaSrcActive source: %d", MediaUtils::getInstance().m_ActiveDeviceDataMap[iRegion].source));

   bool bIsMediaSrcActive = false;

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(iRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      if (E_SOURCE_MEDIA == MediaUtils::getInstance().m_ActiveDeviceDataMap[iRegion].source)
      {
         bIsMediaSrcActive = true;
      }
      else
      {
         ETG_TRACE_USR4(("MediaSourceHandling::isMediaSrcActive source: %d", MediaUtils::getInstance().m_ActiveDeviceDataMap[iRegion].source));
      }
   }
   else
   {
      ETG_TRACE_USR4(("MediaSourceHandling::isMediaSrcActive region doesn't present in the map : %d", iRegion));
   }

   return bIsMediaSrcActive;
}


int MediaSourceHandling::getActiveSourceInSink(int sinkid)
{
   int iSrcId = 0;
   ETG_TRACE_USR4(("MediaSourceHandling::getActiveSourceInSink sinkid %d ", sinkid))

   if (!audioSrcMap.empty())
   {
      std::map<int, audioutility>::iterator itr = audioSrcMap.find(sinkid);
      if (itr != audioSrcMap.end())
      {
         iSrcId = itr->second.srcid;
      }
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getActiveSourceInSink iSrcId  %d ", iSrcId));

   return iSrcId;
}


/************************************************************************
*FUNCTION		: onSourceAvailabilityChangedError
*DESCRIPTION	: This method gives the details of the available sources
*PARAMETER		: int iRegion
*RETURNVALUE	: bool
************************************************************************/

void MediaSourceHandling::onSourceAvailabilityChangedError(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceAvailabilityChangedError >& error)
{
   ETG_TRACE_USR4(("MediaSourceHandling::onSourceAvailabilityChangedError() \n"));
}


/************************************************************************
*FUNCTION		: onSourceAvailabilityChangedSignal
*DESCRIPTION	: This method returns the src type, tuner or media based on region
*PARAMETER		: int iRegion
*RETURNVALUE	: bool
************************************************************************/

void MediaSourceHandling::onSourceAvailabilityChangedSignal(const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::AudioSourceChangeProxy >& /*proxy*/,
      const ::boost::shared_ptr< MASTERAUDIOSERVICE_INTERFACE::AudioSourceChange::SourceAvailabilityChangedSignal >& signal)
{
   ETG_TRACE_USR4(("MediaSourceHandling::onSourceAvailabilityChangedSignal SrcId: %d, SubSrcId: %d, Availbility: %d", signal->getSourceAvailability().getSrcId(), signal->getSourceAvailability().getSubSrcId(), signal->getSourceAvailability().getAvailability()));
}


int MediaSourceHandling::getCurrentExternalSourceAvailability(int iExternalSrcId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentExternalSourceAvailability() Entered iExternalSrcId %d", iExternalSrcId));
   int iExternalSrcAvailable = false;
   if (m_ExternalSourceAvailabilityMap.size() != 0)
   {
      iExternalSrcAvailable = m_ExternalSourceAvailabilityMap[iExternalSrcId];
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentExternalSourceAvailability() Exited  iExternalSrcAvailable %d", iExternalSrcAvailable));
   return iExternalSrcAvailable;
}


void  MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection(int iDisconnectedDeviceTag)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection Entered iDisconnectedDeviceTag %d", iDisconnectedDeviceTag));
   std::map< int, audioutility >::iterator itr = mediaSrcMap.begin();

   bool bUsbCamportSourceAvailable = clMetaData::getInstance().getUsbCamportAvailable();
   ETG_TRACE_USR4(("MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection bUsbCamportSourceAvailable %d", bUsbCamportSourceAvailable));

   for (; itr != mediaSrcMap.end(); ++itr)
   {
      ETG_TRACE_USR4(("MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection mediaSrcMap.subsrcid %d", itr->second.subsrcid));
      ETG_TRACE_USR4(("MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection mediaSrcMap.iSinkId %d", itr->first));
      ETG_TRACE_USR4(("MediaSourceHandling::vUpdateMediaSrcMapOnDisconnection mediaSrcMap.srcid %d", itr->second.srcid));

      if (itr->second.subsrcid == iDisconnectedDeviceTag)
      {
         vResetMediaSrcMap(itr->first);
      }
      else if (iDisconnectedDeviceTag == EXTERNAL_SOURCE_SUBSRCID)
      {
         if (itr->second.srcid == SRC_CAMERAPORT1_USB_VIDEO && bUsbCamportSourceAvailable == false)
         {
            vResetMediaSrcMap(itr->first);
         }
         else if (itr->second.srcid == SRC_CAMERAPORT1_HDMI || itr->second.srcid == SRC_CAMERAPORT2_HDMI || itr->second.srcid == SRC_CAMERAPORT3_HDMI)
         {
            vResetMediaSrcMap(itr->first);
         }
         else if (itr->second.srcid == SRC_CMR1_USB1_VIDEO)
         {
            vResetMediaSrcMap(itr->first);
         }
      }
   }
}


void  MediaSourceHandling::vResetMediaSrcMap(int iSinkId)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vResetMediaSrcMap mediaSrcMap.iSinkId %d", iSinkId));
   mediaSrcMap[iSinkId].srcid = SRC_INVALID;
   mediaSrcMap[iSinkId].subsrcid = SUBSRCID_INVALID;
}


void MediaSourceHandling::setExternalMediaClientHandlerInstance(ExternalMediaClientHandler* _externalMediaClientHandlerInstance)
{
   if (_externalMediaClientHandlerInstance != NULL)
   {
      _externalMediaClientHandler = _externalMediaClientHandlerInstance;
   }
   else
   {
      ETG_TRACE_USR1(("MediaSourceListHandler::_externalMediaClientHandler is NULL"));
   }
}


void MediaSourceHandling::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
   }
   else
   {
      ETG_TRACE_USR1(("MediaSourceListHandler::_externalMediaSourceHandler is NULL"));
   }
}


#endif
int MediaSourceHandling::getCurrentRegionSubSrc()
{
   int iSubSrcValue;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   iSubSrcValue = 0;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionSubSrc u8activeRegion  %d ", u8activeRegion));

   if (u8activeRegion == REGION_COCKPIT)
   {
      iSubSrcValue = audioSrcMap[COCKPIT_SINK_ID].subsrcid;
   }
   else if (u8activeRegion == REGION_CABIN_A)
   {
      iSubSrcValue = audioSrcMap[CABIN_A_SINK_ID].subsrcid;
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionSubSrc iSubSrcValue  %d ", iSubSrcValue));
#else
   iSubSrcValue = m_iSubSrcValue;
#endif
   return iSubSrcValue;
}


int MediaSourceHandling::getCurrentRegionSrcId()
{
   int iSrcId ;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   iSrcId = SRC_INVALID;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionSrcId u8activeRegion   %d ", u8activeRegion))

   if (u8activeRegion == REGION_COCKPIT)
   {
      iSrcId = audioSrcMap[COCKPIT_SINK_ID].srcid;
   }
   else if (u8activeRegion == REGION_CABIN_A)
   {
      iSrcId = audioSrcMap[CABIN_A_SINK_ID].srcid;
   }
   ETG_TRACE_USR4(("MediaSourceHandling::getCurrentRegionSubSrc iSrcId  %d ", iSrcId));
#else
   iSrcId = m_SrcId;
#endif
   return iSrcId;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
int MediaSourceHandling::getActiveSrcInRegion(uint8 regionId)
{
   int srcId = SRC_INVALID;
   if (regionId == REGION_COCKPIT)
   {
      srcId = audioSrcMap[COCKPIT_SINK_ID].srcid;
   }
   else if (regionId == REGION_CABIN_A)
   {
      srcId = audioSrcMap[CABIN_A_SINK_ID].srcid;
   }
   ETG_TRACE_USR1(("MediaSourceHandling::getActiveSrcInRegion regionId %d, srcId  %d ", regionId, srcId));
   return srcId;
}


int MediaSourceHandling::getSinkSrcId(int sinkId)
{
   int iSrcId = SRC_INVALID;
   iSrcId = audioSrcMap[sinkId].srcid;
   ETG_TRACE_USR4(("MediaSourceHandling::getSinkSrcId sinkId:%d iSrcId:%d ", sinkId, iSrcId));
   return iSrcId;
}


#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
void MediaSourceHandling::setCurrentRegionSrcId(int srcId)
{
   m_SrcId = srcId;
}


void MediaSourceHandling::setCurrentRegionSubSrc(int iSubSrcValue)
{
   m_iSubSrcValue = iSubSrcValue;
}


#else
bool MediaSourceHandling::isInternalMediaSourceActiveInSink(int sinkId)
{
   int srcId = getActiveSourceInSink(sinkId);
   ETG_TRACE_USR1(("MediaSourceListHandler::isInternalMediaSourceActiveInSink: sinkId %d, srcId %d", sinkId, srcId));

   if ((srcId == SRC_PHONE_BTAUDIO) || (srcId == SRC_MEDIA_PLAYER))
   {
      return true;
   }
   return false;
}


bool MediaSourceHandling::isExternalMediaSourceActiveInSink(int sinkId)
{
   int srcId = getActiveSourceInSink(sinkId);
   bool isExternalMediaSrcActive = false;
   switch (srcId)
   {
      case SRC_CAMERAPORT1_USB_VIDEO:
      case SRC_CAMERAPORT1_USB_AUDIO:
      case SRC_CMR1_USB1_VIDEO:
      case SRC_CMR1_USB1_AUDIO:
      case SRC_CAMERAPORT1_HDMI:
      case SRC_CAMERAPORT2_HDMI:
      case SRC_CAMERAPORT3_HDMI:
      {
         isExternalMediaSrcActive = true;
      }
      break;
      default:
      {
         ETG_TRACE_USR2(("MediaSourceHandling:isExternalMediaSourceActiveInSink Default case option"));
      }
      break;
   }
   ETG_TRACE_USR1(("MediaSourceHandling:isExternalMediaSourceActiveInSink: sinkId %d, srcId %d, isExternalMediaSrcActive %d", sinkId, srcId, isExternalMediaSrcActive))
   return isExternalMediaSrcActive;
}


int MediaSourceHandling::getSinkId(uint32 Cabinregion)
{
   ETG_TRACE_USR4(("MediaSourceHandling::vGetSinkId Cabinregion:%d ", Cabinregion));
   int iSinkId = SINK_INVALID;

   if (Cabinregion == REGION_CABIN_A)
   {
      iSinkId = CABIN_A_SINK_ID;
   }
   else if (Cabinregion == REGION_CABIN_B)
   {
      iSinkId = CABIN_B_SINK_ID;
   }
   else if (Cabinregion == REGION_COCKPIT)
   {
      iSinkId = COCKPIT_SINK_ID;
   }
   return iSinkId;
}


int MediaSourceHandling::getPrevSrcId()
{
   return _prevSrcid;
}


#endif
} // namespace Core
} // namespace App
