/**************************************************************************************
* @file         : RouterPluginClientHandler.cpp
* @author       : ECH - voa4kor
* @addtogroup   : AppHmi_media
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "Core/PluginClientHandlerAppl/RouterPluginClientHandler.h"
#include "../ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/ExternalMediaPlaybackHandler/ExternalMediaPlaybackHandler.h"
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#include "DataModel/HMIModelComponent.h"
#include "Core/Utils/MediaUtils.h"
#include "PluginConstants.h"
#include "MediaCameraPort.h"
#include "MediaRouter.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "Core/Utils/DbDownloadMap.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::RouterPluginClientHandler::

#include "trcGenProj/Header/RouterPluginClientHandler.cpp.trc.h"
#endif

#define CMRSTREAM1_PLAYBACKMODE 43
#define CMRSTREAM3_PLAYBACKMODE 44

namespace App {
namespace Core {


RouterPluginClientHandler::RouterPluginClientHandler()
   : _externalMediaSourceHandler(NULL)
   , _playableContentInterfaceHandler(NULL)
{
   ETG_TRACE_USR1(("RouterPluginClientHandler constructor \n"));
   m_sElapsedplaytimeData = " ";
   m_Elapsedslidervalue = MEDIA_C_VALUE_ZERO;
   m_Totalslidervalue = MEDIA_C_VALUE_ZERO;
   m_totaltime = MEDIA_C_VALUE_ZERO;
   NowPlayingInfoStatus = (struct stNowPlayingInfo)
   {
      "", "", "", MEDIA_C_VALUE_ZERO, "", MEDIA_C_VALUE_ZERO, "", MEDIA_C_VALUE_ZERO
   };
   m_splaytimeData = " ";
   m_strEmpData = " ";
   m_strEmpEpisodeData = " ";
   strEmpData = " ";
   _currentPlayingSongPosition = 0;
   _currentPlayingListType = EnumConst::entListType_Invalid;
   _folderPath = " ";
   _srcType = EnumConst::entSourceType_CMR ;
   _fileType = entFileType__entFileType_unknown;
   initialiseStreamTypeMap();
}


RouterPluginClientHandler::~RouterPluginClientHandler()
{
   ETG_TRACE_USR1(("RouterPluginClientHandler Destructor \n"));
}


void RouterPluginClientHandler::resetCurrentTrackInfo()
{
   _folderPath = " ";
}


void RouterPluginClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("RouterPluginClientHandler::registerProperties"));
}


void RouterPluginClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("RouterPluginClientHandler::deregisterProperties"));
}


void RouterPluginClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("RouterPluginClientHandler::onUnavailable"));
}


void RouterPluginClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("RouterPluginClientHandler::onAvailable"));
}


void RouterPluginClientHandler::initialiseStreamTypeMap()
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::initialiseStreamTypeMap is entered \n"));

   m_StreamTypeMap[REGION_CABIN_A][SRC_CMR1_USB1_VIDEO] = StreamType(STREAM1_PLAY_ACTION, CMRSTREAM1_PLAYBACKMODE, STREAM1_SEEK_TO);
   m_StreamTypeMap[REGION_CABIN_A][SRC_CMR1_USB1_AUDIO] = StreamType(STREAM3_PLAY_ACTION, CMRSTREAM3_PLAYBACKMODE, STREAM3_SEEK_TO);
   m_StreamTypeMap[REGION_CABIN_B][SRC_CMR1_USB1_VIDEO] = StreamType(STREAM2_PLAY_ACTION, CMRSTREAM1_PLAYBACKMODE, STREAM2_SEEK_TO);
   m_StreamTypeMap[REGION_CABIN_B][SRC_CMR1_USB1_AUDIO] = StreamType(STREAM4_PLAY_ACTION, CMRSTREAM3_PLAYBACKMODE, STREAM4_SEEK_TO);
}


void RouterPluginClientHandler::setExternalMediaSrcHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerObject)
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::setExternalMediaSrcHandlerInstance is entered \n"));
   _externalMediaSourceHandler = _externalMediaSourceHandlerObject;
}


void RouterPluginClientHandler::setPlayableContentInterfaceHandlerInstance(PlayableContentInterfaceHandler* _playableContentInterfaceHandlerObject)
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::setPlayableContentInterfaceHandlerInstance is entered \n"));
   _playableContentInterfaceHandler = _playableContentInterfaceHandlerObject;
}


void RouterPluginClientHandler::sendPluginRequest(uint16 ControlValue, int srcID)
{
   ETG_TRACE_USR4(("RouterPluginClientHandler::sendPluginRequest ControlValue  : %d", ControlValue));
   ETG_TRACE_USR4(("RouterPluginClientHandler::sendPluginRequest srcID : %d", srcID));
   ETG_TRACE_USR4(("RouterPluginClientHandler::sendPluginRequest _currentPlayingListType : %d", _currentPlayingListType));

   tU8 l_Operationtype = ExternalMediaPlaybackHandler::getInstance()->getOperationType();
   ETG_TRACE_USR4(("RouterPluginClientHandler::sendPluginRequest l_Operationtype : %d", l_Operationtype));

   uint16 l_ControlId = 0;
   uint16 l_ControlValue = ControlValue;
   uint8 l_ListValue = _currentPlayingListType;
   uint16 l_PositionValue = _currentPlayingSongPosition;

   if ((l_Operationtype == EN_MEDIA_REPEAT_OFF) ||
         (l_Operationtype == EN_MEDIA_REPEAT_TRACK) ||
         (l_Operationtype == EN_MEDIA_REPEAT_LIST))
   {
      l_ControlValue = ExternalMediaPlaybackHandler::getInstance()->getRepeatState();
      l_ControlId = getRepeatModeStreamType(srcID, REGION_CABIN_A);
   }
   else if ((l_Operationtype == EN_MEDIA_SHUFFLE_OFF) || (l_Operationtype == EN_MEDIA_SHUFFLE_LIST))
   {
      l_ControlValue = ExternalMediaPlaybackHandler::getInstance()->getShuffleState();
      l_ControlId = getRepeatModeStreamType(srcID, REGION_CABIN_A);
   }
   else
   {
      l_ControlId = getPlaybackActionStreamType(srcID, REGION_CABIN_A);
   }

   sendpluginUpdate(l_ControlId, l_ControlValue, REGION_CABIN_A, l_ListValue, l_PositionValue);
}


void RouterPluginClientHandler::sendpluginUpdate(uint16 ControlId, uint16 ControlValue, uint8 activeRegion, uint8 listValue, uint16 positionValue)
{
   ETG_TRACE_USR4(("RouterPluginClientHandler::sendpluginUpdate ControlId : %d ControlValue: %d activeRegion: %d listValue :%d positionValue :%d",
                   ControlId, ControlValue, activeRegion, listValue, positionValue));
   uint8 l_ControlId = ControlId;
   uint8 l_ControlValue = ControlValue;
   uint8 l_ListValue = listValue;
   uint16 l_PositionValue = positionValue;
   uint8 srcType = _srcType;
   uint16 fileType = _fileType;
   uint16 filterTag1 = 0;
   uint16 filterTag2 = 0;
   std::string folderPath = _folderPath;
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   if (NULL != pluginData.get())
   {
      pluginData->addEventData(l_ControlId);
      pluginData->addEventData(l_ControlValue);
      pluginData->addEventData(activeRegion);
      pluginData->addEventData(l_ListValue);
      pluginData->addEventData(l_PositionValue);
      pluginData->addEventData(srcType);
      pluginData->addEventData(fileType);
      pluginData->addEventData(filterTag1);
      pluginData->addEventData(filterTag2);
      pluginData->addEventData(folderPath);
      POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_ROUTER, "", pluginData)));
   }
}


void RouterPluginClientHandler::updateCurrentPlayingListType(uint8 Listtype, uint16 Position)
{
   ETG_TRACE_USR3(("RouterPluginClientHandler::updateCurrentPlayingListType Listtype:%d ,Position:%d", Listtype, Position));
   _currentPlayingListType = Listtype;
   _currentPlayingSongPosition  = Position;
}


void RouterPluginClientHandler::updateCurentPlayingFolderPath(std::string folderPath)
{
   _folderPath = folderPath;
   ETG_TRACE_USR1(("RouterPluginClientHandler::updateCurentPlayingFolderPath: _folderPath %s", _folderPath.c_str()));
}


void RouterPluginClientHandler::updateCurrentPlayingSrcType(uint8 srcType)
{
   _srcType = srcType;
   ETG_TRACE_USR1(("RouterPluginClientHandler::updateCurentPlayingFolderPath: _srcType %d", ETG_CENUM(enSubSourceType, _srcType)));
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   if (srcID == SRC_CMR1_USB1_VIDEO)
   {
      _fileType = entFileType__entFileType_Video;
   }
   else if (srcID == SRC_CMR1_USB1_AUDIO)
   {
      _fileType = entFileType__entFileType_Audio;
   }
}


void RouterPluginClientHandler::updateCurrentPlayingFileType(uint16 fileType)
{
   _fileType = fileType;
   ETG_TRACE_USR1(("RouterPluginClientHandler::updateCurentPlayingFolderPath: _fileType %d", ETG_CENUM(entFileType, fileType)));
}


void RouterPluginClientHandler::onPlaySelectedListItem(uint8 listtype, uint16 position, std::string folderpath, int fileType)
{
   uint16 positionValue = position;
   uint8  listtypeValue = listtype;
   _fileType = fileType;
   _folderPath = folderpath;
   boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   uint8 controlValue = MEDIA_BROWSE_LIST_VALUE;
   uint8 controlId = getPlaybackActionStreamType(srcID, u8activeRegion);
   sendpluginUpdate(controlId, controlValue, u8activeRegion, listtypeValue, positionValue);

   ETG_TRACE_USR3(("RouterPluginClientHandler::onPlaySelectedListItem positionValue:%d ,listtypeValue:%d,u8activeRegion:%d ,srcID:%d,controlValue:%d,controlId :%d",
                   positionValue, listtypeValue, u8activeRegion, srcID, controlValue, controlId));
}


int RouterPluginClientHandler::getRepeatModeStreamType(int srcID, int u8activeRegion)
{
   int repeatModeStreamType = CMRSTREAM1_PLAYBACKMODE;
   ::std::map< uint32, ::std::map< int, StreamType > >::iterator itr = m_StreamTypeMap.find(u8activeRegion);
   if (itr != m_StreamTypeMap.end())
   {
      ::std::map< int, StreamType >::iterator itr1 = itr->second.find(srcID);
      if (itr1 != itr->second.end())
      {
         repeatModeStreamType = itr1->second.RepeatMode;
      }
   }

   ETG_TRACE_USR3(("RouterPluginClientHandler::getRepeatModeStreamType repeatModeStreamType:%d ,srcID:%d,u8activeRegion:%d",
                   repeatModeStreamType, srcID, u8activeRegion));
   return repeatModeStreamType;
}


int RouterPluginClientHandler::getPlaybackActionStreamType(int srcID, int u8activeRegion)
{
   int playBackStreamType = STREAM1_PLAY_ACTION;
   ::std::map< uint32, ::std::map< int, StreamType > >::iterator itr = m_StreamTypeMap.find(u8activeRegion);
   if (itr != m_StreamTypeMap.end())
   {
      ::std::map< int, StreamType >::iterator itr1 = itr->second.find(srcID);
      if (itr1 != itr->second.end())
      {
         playBackStreamType = itr1->second.PlayAction;
      }
   }
   ETG_TRACE_USR3(("RouterPluginClientHandler::getPlaybackActionStreamType playBackStreamType:%d ,srcID:%d,u8activeRegion:%d",
                   playBackStreamType, srcID, u8activeRegion));
   return playBackStreamType;
}


void RouterPluginClientHandler::updateSeekToStreamType(uint16 seekPercent)
{
   uint8 l_ListValue = 0;
   uint16 l_PositionValue = 0;
   int seekToStreamType = STREAM1_SEEK_TO;
   int srcID = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ::std::map< uint32, ::std::map< int, StreamType > >::iterator itr = m_StreamTypeMap.find(u8activeRegion);
   if (itr != m_StreamTypeMap.end())
   {
      ::std::map< int, StreamType >::iterator itr1 = itr->second.find(srcID);
      if (itr1 != itr->second.end())
      {
         seekToStreamType = itr1->second.SeekTo;
      }
   }
   ETG_TRACE_USR3(("RouterPluginClientHandler::getPlaybackActionStreamType seekToStreamType:%d srcID:%d u8activeRegion:%d seekpercent:%d",
                   seekToStreamType, srcID, u8activeRegion, seekPercent));
   sendpluginUpdate(seekToStreamType, seekPercent, u8activeRegion, l_ListValue, l_PositionValue);
}


void RouterPluginClientHandler::onMessagePluginUpdateResp(EventDataItem* eventDataItem, EventDataItem* eventDataItemControlVal, EventDataItem* eventDataItemPosition, EventDataItem* trackIndex, EventDataItem* folderPath, EventDataItem* listType, EventDataItem* subSrcType, EventDataItem* region)
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::onCourierMessage:PluginUpdateRespMsg"));
   uint8 u8ControlVal = 0;
   uint32 ElapsedTime = 0;
   uint32 TotalTime = 0;
   uint16 fileType = 0;
   uint16 track = 0;
   std::string path = "";
   uint16 listtype = 0;
   uint8 src = SRC_CMR1_USB1_VIDEO;
   uint16 regionId = 0;
   if (NULL != eventDataItem)
   {
      const EventDataItem::Data& eventData = eventDataItem->getData();
      if (EventDataItem::UINT32 == eventData._type)
      {
         uint32 m_control_id = eventData._value._uint32Value;
         ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:PluginUpdateRespMsg control block id : %d", m_control_id));
         switch (m_control_id)
         {
            case STREAM1_PLAY_ACTION:
            case STREAM3_PLAY_ACTION:
            {
               if (NULL != eventDataItemControlVal)
               {
                  u8ControlVal = getPluginValue(eventDataItemControlVal);
                  ExternalMediaPlaybackHandler::getInstance()->vUpdatePlayItems(u8ControlVal, SRC_CMR1_USB1_VIDEO);
               }
            }
            break;
            case CMRSTREAM1_PLAYBACKMODE:
            case CMRSTREAM3_PLAYBACKMODE:
            {
               if (NULL != eventDataItemControlVal)
               {
                  u8ControlVal = getPluginValue(eventDataItemControlVal);
                  ExternalMediaPlaybackHandler::getInstance()->vUpdateRepeatShuffleItems(u8ControlVal, SRC_CMR1_USB1_VIDEO);
               }
            }
            break;
            case STREAM1_CURRENT_PLAY_TIME:
            case STREAM3_CURRENT_PLAY_TIME:
            {
               if (NULL != eventDataItemControlVal)
               {
                  m_sElapsedplaytimeData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp inside m_sElapsedplaytimeData %s", m_sElapsedplaytimeData.c_str()));
                  if (!m_sElapsedplaytimeData.empty())
                  {
                     m_Elapsedslidervalue = getSlidertime(m_sElapsedplaytimeData);
                     ElapsedTime = m_Elapsedslidervalue;
                     ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp inside m_Elapsedslidervalue %f", ElapsedTime));
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourcePlayTimePluginUpdate(m_sElapsedplaytimeData, m_Elapsedslidervalue, m_Totalslidervalue, m_splaytimeData, SRC_CMR1_USB1_VIDEO);
            }
            break;
            case STREAM1_TOTAL_PLAY_TIME:
            case STREAM3_TOTAL_PLAY_TIME:
            {
               if (NULL != eventDataItemControlVal)
               {
                  m_splaytimeData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp m_splaytimeData %s", m_splaytimeData.c_str()));
                  if (!m_splaytimeData.empty())
                  {
                     NowPlayingInfoStatus.TotalTime = m_splaytimeData;
                     ExternalMediaPlaybackHandler::getInstance()->updateTotalTime(m_splaytimeData);
                     m_Totalslidervalue = getSlidertime(m_splaytimeData);
                     TotalTime = m_Totalslidervalue;
                     ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp inside TotalTime %f", TotalTime));
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourcePlayTimePluginUpdate(m_sElapsedplaytimeData, m_Elapsedslidervalue, m_Totalslidervalue, m_splaytimeData, SRC_CMR1_USB1_VIDEO);
            }
            break;
            case STREAM1_VIDEO_NAME:
            {
               if (NULL != eventDataItemControlVal)
               {
                  m_strEmpData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp m_strEmpData %s", m_strEmpData.c_str()));
                  if (!m_strEmpData.empty())
                  {
                     NowPlayingInfoStatus.ArtistName = m_strEmpData;
                     ExternalMediaPlaybackHandler::getInstance()->updateArtistName(m_strEmpData);
                     ExternalMediaPlaybackHandler::getInstance()->updateTitleName("");
                  }
                  else
                  {
                     ExternalMediaPlaybackHandler::getInstance()->updateArtistName("");
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourceMetaDataPluginUpdate(SRC_CMR1_USB1_VIDEO);
            }
            break;
            case STREAM1_VIDEO_EPISODE:
            {
               if (NULL != eventDataItemControlVal)
               {
                  m_strEmpEpisodeData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp videoepisode %s", m_strEmpEpisodeData.c_str()));

                  if (!m_strEmpEpisodeData.empty())
                  {
                     NowPlayingInfoStatus.TitleName = m_strEmpEpisodeData;
                     ExternalMediaPlaybackHandler::getInstance()->updateTitleName(m_strEmpEpisodeData);
                  }
                  else
                  {
                     ExternalMediaPlaybackHandler::getInstance()->updateTitleName("");
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourceMetaDataPluginUpdate(SRC_CMR1_USB1_VIDEO);
            }
            break;
            case STREAM3_TRACK_NAME:
            {
               if (NULL != eventDataItemControlVal)
               {
                  strEmpData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp TRACK strEmpData %s", strEmpData.c_str()));
                  if (!strEmpData.empty())
                  {
                     NowPlayingInfoStatus.TitleName = strEmpData;
                     //MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_CABIN_A].ArtistName = strEmpData;
                     ExternalMediaPlaybackHandler::getInstance()->updateTitleName(strEmpData);
                  }
                  else
                  {
                     ExternalMediaPlaybackHandler::getInstance()->updateTitleName("");
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourceMetaDataPluginUpdate(SRC_CMR1_USB1_VIDEO);
            }
            break;
            case STREAM3_ARTIST_NAME:
            {
               if (NULL != eventDataItemControlVal)
               {
                  strEmpData = getPluginValueString(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp STREAM3_ARTIST strEmpData %s", strEmpData.c_str()));
                  if (!strEmpData.empty())
                  {
                     NowPlayingInfoStatus.ArtistName = strEmpData;
                     ExternalMediaPlaybackHandler::getInstance()->updateArtistName(strEmpData);
                  }
                  else
                  {
                     ExternalMediaPlaybackHandler::getInstance()->updateArtistName("");
                  }
               }
               ExternalMediaPlaybackHandler::getInstance()->handleExternalSourceMetaDataPluginUpdate(SRC_CMR1_USB1_VIDEO);
            }
            break;
            case USB_DEVICE_STATUS:
            {
               if (NULL != eventDataItemControlVal)
               {
                  int isUsbConnectedToRouter = getPluginValue(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp USB Status isUsbConnectedToRouter %d", isUsbConnectedToRouter));
                  _externalMediaSourceHandler->updateRouterUsbConnectionState(isUsbConnectedToRouter);
               }
            }
            break;
            case MEDIA_DATA_BASE_STATUS:
            {
               if (NULL != eventDataItemControlVal)
               {
                  int dbStatus = getPluginValue(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp dbStatus %d", dbStatus));
                  DbDownloadMap::getInstance().setDbState(EnumConst::entSourceType_CMR, dbStatus);
                  MediaSourceListHandler::getInstance().updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
                  if (dbStatus == EnumConst::entDbState_DBInvalid || dbStatus == EnumConst::entDbState_USBDisconnected)
                  {
                     deallocationRequestToAudio();
                  }
               }
            }
            break;
            case MEDIA_CHECKSUM_STATUS:
            {
               if (NULL != eventDataItemControlVal)
               {
                  int checkSumStatus = getPluginValue(eventDataItemControlVal);
                  ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp checkSumStatus %d", checkSumStatus));
                  _playableContentInterfaceHandler->onCRCStatusReceived(EnumConst::entSourceType_CMR, checkSumStatus);
               }
            }
            break;
            case MEDIA_SIGNAL_SOURCE_CHANGE:
            {
               if (NULL != eventDataItemControlVal)
               {
                  fileType = getPluginValue(eventDataItemControlVal);
               }
               if (NULL != trackIndex)
               {
                  track = getPluginValue(trackIndex);
               }
               if (NULL != folderPath)
               {
                  path = getPluginValueString(folderPath);
               }
               if (NULL != listType)
               {
                  listtype = getPluginValue(listType);
               }
               if (NULL != subSrcType)
               {
                  src = getPluginValue(subSrcType);
               }
               if (NULL != region)
               {
                  regionId = getPluginValue(region);
               }
               ETG_TRACE_USR2(("RouterPluginClientHandler::onCourierMessage:PluginUpdateRespMsg: SIGNAL_SOURCE_CHANGE: fileType %d, track %d, listtype %d, srcType %d, regionId %d, path %s", ETG_CENUM(entFileType, fileType), track, ETG_CENUM(entListType, listtype), ETG_CENUM(enSubSourceType, src), regionId, path.c_str()));
               ExternalMediaPlaybackHandler::getInstance()->onFolderPlaybackSrcChange(fileType, track, path, listtype, src, regionId);
            }
            break;
            default:
               ETG_TRACE_USR1(("RouterPluginClientHandler::onCourierMessage:PluginUpdateRespMsg controlindex default"));
               break;
         }
      }
   }
   ETG_TRACE_USR4(("RouterPluginClientHandler::onCourierMessage:onMessagePluginUpdateResp control value : %d", u8ControlVal));
}


double RouterPluginClientHandler::getSliderValue()
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::getSliderValue m_Totalslidervalue %d", m_Totalslidervalue));
   return m_Totalslidervalue;
}


uint8 RouterPluginClientHandler::getPluginValue(EventDataItem* eventDataItemControlVal)
{
   EventDataItem::Data eventDataControlVal = eventDataItemControlVal->getData();
   ETG_TRACE_USR4(("RouterPluginClientHandler::getPluginValue eventDataItemControlVal  %d", eventDataControlVal._type));
   uint8 l_ControlVal = 0;
   if (EventDataItem::UINT8 == eventDataControlVal._type)
   {
      l_ControlVal = eventDataControlVal._value._uint8Value;
   }
   ETG_TRACE_USR4(("RouterPluginClientHandler::getPluginValue l_ControlVal  %d", l_ControlVal));
   return l_ControlVal;
}


double RouterPluginClientHandler::getSlidertime(std::string totaltimeslidervalue)
{
   ETG_TRACE_USR4(("RouterPluginClientHandler::getSlidertime hh:mm:ss : %s", totaltimeslidervalue.c_str()));

   vector <string> timetokens;
   string hr = "";
   string min = "";
   string sec = "";
   double hr1 = 0;
   double hour = 0;
   double min1 = 0;
   double minute = 0;
   double sec1 = 0;
   double second = 0;
   stringstream check1(totaltimeslidervalue);
   string intermediate = "";
   while (getline(check1, intermediate, ':'))
   {
      timetokens.push_back(intermediate);
   }
   ETG_TRACE_USR4(("RouterPluginClientHandler::getSlidertime timetokens.size() : %d", timetokens.size()));
   if (timetokens.size() >= 1)
   {
      hr = timetokens[0];
      hr1 = atof(hr.c_str());
      hour = hr1 * 3600;
      min = timetokens[1];
      min1 = atof(min.c_str());
      minute = min1 * 60;
      sec = timetokens[2];
      sec1 = atof(sec.c_str());
      second = sec1;
      m_totaltime = hour + minute + second;
      ETG_TRACE_USR4(("RouterPluginClientHandler::getSlidertime m_totaltime : %d", m_totaltime));
      timetokens.clear();//clearing the vector
   }
   return m_totaltime;
}


std::string RouterPluginClientHandler::getPluginValueString(EventDataItem* eventDataItemControlVal)
{
   EventDataItem::Data eventDataControlVal = eventDataItemControlVal->getData();
   std::string l_ControlVal = "";
   if (EventDataItem::STRING == eventDataControlVal._type)
   {
      l_ControlVal = eventDataControlVal._value._stringValue->c_str();
   }
   return l_ControlVal;
}


void RouterPluginClientHandler::deallocationRequestToAudio()
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::deallocationRequestToAudio"));
   int iActiveCabinSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   uint8 iActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("RouterPluginClientHandler::deallocationRequestToAudio iActiveCabinSrcId %d    iActiveRegion   %d", iActiveCabinSrcId, iActiveRegion));
   if (iActiveCabinSrcId == SRC_CMR1_USB1_VIDEO || iActiveCabinSrcId == SRC_CMR1_USB1_AUDIO)
   {
      MediaSourceHandling::getInstance().vRequestSourceDeactivation(iActiveCabinSrcId, -1, iActiveRegion);
   }
}


void RouterPluginClientHandler::resetMetadata()
{
   ETG_TRACE_USR1(("RouterPluginClientHandler::resetMetadata entered"));
   ExternalMediaPlaybackHandler::getInstance()->updateTitleName("");
   ExternalMediaPlaybackHandler::getInstance()->updateArtistName("");
   HMIModelComponent::getInstance().bOnUpdateNowPlayingStatusRouter();
   HMIModelComponent::getInstance().vUpdateRouterUSBHomescreenStatus();
}


}//App
}//Core
