/**************************************************************************************
 * @file         : MediaSourceListHandler.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *01.07.20 Rev. 0.1 Naga Suman(RBEI/ECH2)
 * 		  			Initial Revision.
 **************************************************************************************/
#include "hall_std_if.h"
#include "Core/MediaSourceListHandler/MediaSourceListHandler.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "List/Data/ListDataProviderDistributor.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "ProjectBaseTypes.h"
#include "PluginConstants.h"
#include "App/Core/HmiTranslation_TextIds.h"
#include "Core/BTClientHandler/BTClientHandler.h"
#include "Core/KDSHandler/KDSHandler.h"
#include "CgiExtensions/ImageLoader.h"
#include "Core/Utils/DbDownloadMap.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::MediaSourceListHandler::
#include "trcGenProj/Header/MediaSourceListHandler.cpp.trc.h"
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#include "utest/stubs/AudioDefines.h"
#include "utest/stubs/ProjectBaseTypes.h"
#include "utest/stubs/Core/BTClientHandler.h"
#include "utest/stubs/Core/KDSHandler.h"

#endif
namespace App {
namespace Core {

/************************************************************************
*FUNCTION		: MediaSourceListHandler
*DESCRIPTION	: Destructor.
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
MediaSourceListHandler::~MediaSourceListHandler()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   if (_externalMediaSourceConnectionStateHandler != NULL)
   {
      _externalMediaSourceConnectionStateHandler->removeExternalSourceAvailabilityStatusObserver((*this));
   }
   if (_externalMediaListDBManager != NULL)
   {
      _externalMediaListDBManager->deregisterExternalDBDownloadObservor((*this));
   }
   m_MediaDevices.clear();
   m_MediaCabinDevices.clear();
   ListRegistry::s_getInstance().removeListImplementation(LIST_ID_MEDIA_SOURCE_LIST);
#endif
}


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST

MediaSourceListHandler::MediaSourceListHandler()
{
   m_stateMachineVariantAux = 1;
}


#else
MediaSourceListHandler* MediaSourceListHandler::_mediaSourceListHandler = NULL;
MediaSourceListHandler& MediaSourceListHandler::getInstance()
{
   if (_mediaSourceListHandler == 0)
   {
      _mediaSourceListHandler = new MediaSourceListHandler();
   }
   return *_mediaSourceListHandler;
}


void MediaSourceListHandler::removeInstance()
{
   if (_mediaSourceListHandler)
   {
      delete _mediaSourceListHandler;
      _mediaSourceListHandler = 0;
   }
}


/************************************************************************
*FUNCTION		: MediaSourceListHandler
*DESCRIPTION	: Constructor. Initializes the member variables.
*PARAMETER		:
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
MediaSourceListHandler::MediaSourceListHandler(): _kdsHandler(NULL),
   _systemInfoMap(NULL),
   _externalMediaSourceConnectionStatus(NULL),
   _externalMediaSourceConnectionStateHandler(NULL),
   _externalMediaListDBManager(NULL)
{
   ETG_TRACE_USR1(("MediaSourceListHandler::MediaSourceListHandler Constructor called"));
   _isGlassAvailable = false;
   m_bCamportUsbState = false;
   m_MediaDevices.clear();
   m_MediaCabinDevices.clear();
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_MEDIA_SOURCE_LIST, this);
   initialiseMediaDevicesMap();
   initialiseMediaCabinDevicesMap();
   m_bRouterUsbState = true;
}


void MediaSourceListHandler::setExternalMediaListDBManagerInstance(ExternalMediaListDBManager* _externalMediaListDBManagerInstance)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::setExternalMediaListDBManagerInstance is entered"));

   if (_externalMediaListDBManagerInstance != NULL)
   {
      _externalMediaListDBManager = _externalMediaListDBManagerInstance;
      _externalMediaListDBManager->registerExternalDBDownloadObservor((*this));
   }
   else
   {
      ETG_TRACE_ERR(("MediaSourceListHandler::_externalMediaListDBManager is NULL"));
   }
}


void MediaSourceListHandler::onExternalSourceAvailabilityUpdate()
{
   ETG_TRACE_COMP(("MediaSourceHandling::onExternalSourceAvailabilityUpdate"));
   updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
}


void MediaSourceListHandler::onExternalDBDownloadUpdate(uint8 downloadState, EnumConst::entSourceType sourceType)
{
   ETG_TRACE_USR4(("MediaSourceHandling::onExternalDBDownloadUpdate downloadState:%d sourceType:%d", downloadState, sourceType));
   updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
}


/**
 * Description     : Funtion creates Lists corresponding to the list IDs that it receives
 *
 * @param[in]      : listDataInfo : unique ID for lists
 * @return         : tSharedPtrDataProvider
 */
tSharedPtrDataProvider MediaSourceListHandler::getListDataProvider(const ListDateProviderReqMsg& listDataInfo)
{
   tSharedPtrDataProvider sharedDataProvider = tSharedPtrDataProvider();
   ETG_TRACE_USR3(("MediaSourceListHandler::getListDataProvider listID %d", listDataInfo.GetListId()));
   switch (listDataInfo.GetListId())
   {
      case LIST_ID_MEDIA_SOURCE_LIST:
      {
         sharedDataProvider = getMediaSourceList();
         break;
      }
      default:
         break;
   }
   return sharedDataProvider;
}


/**
 * Description     : Function implemented to populate Media Source list
 *
 * @return         : tSharedPtrDataProvider
 */
tSharedPtrDataProvider MediaSourceListHandler::getMediaSourceList()
{
   uint8 activeRegionID = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("MediaSourceListHandler::getMediaSourceList activeRegionID %d", activeRegionID));
   tSharedPtrDataProvider dataProvider ;

   if (activeRegionID == REGION_COCKPIT)
   {
      dataProvider = updateCockpitMediaSourceToListBuilder(activeRegionID);
   }
   else if (activeRegionID == REGION_CABIN_A || activeRegionID == REGION_CABIN_B)
   {
      dataProvider = updateCabinMediaSourceToListBuilder(activeRegionID);
   }

   return dataProvider;
}


tSharedPtrDataProvider MediaSourceListHandler::updateCockpitMediaSourceToListBuilder(uint8 activeRegion)
{
   const char* const MEDIA_SOURCE_LIST_ITEM = "Layer_MediaDevice";
   ListDataProviderBuilder listBuilder(LIST_ID_MEDIA_SOURCE_LIST, MEDIA_SOURCE_LIST_ITEM);
   ETG_TRACE_USR3(("MediaSourceListHandler::getMediaSourceList m_MediaDevices.size() %d", m_MediaDevices.size()));

   for (int index = 0 ; index < m_MediaDevices.size() ; index++)
   {
      if (isSourceAvailableInCockpitRegion(index))
      {
         listBuilder.AddItem(index, m_MediaDevices[index].deviceType, MEDIA_SOURCE_LIST_ITEM).
         AddData(m_MediaDevices[index].sourceIconIdx).
         AddData(isCockpitSourceInIndexActive(index)).
         AddData(m_MediaDevices[index].deviceText).
         AddData(isSourceInIndexCockpitEnabled(index));
      }
   }
   return listBuilder.CreateDataProvider();
}


tSharedPtrDataProvider MediaSourceListHandler::updateCabinMediaSourceToListBuilder(uint8 activeRegion)
{
   const char* const MEDIA_SOURCE_LIST_ITEM = "Layer_MediaDevice";
   ListDataProviderBuilder listBuilder(LIST_ID_MEDIA_SOURCE_LIST, MEDIA_SOURCE_LIST_ITEM);
   ETG_TRACE_USR3(("MediaSourceListHandler::getMediaSourceList m_MediaCabinDevices.size() %d", m_MediaCabinDevices.size()));

   for (int index = 0 ; index < m_MediaCabinDevices.size() ; index++)
   {
      if (isSourceAvailableInCabinRegion(index))
      {
         DownloadVisibilityDataItemData data = updateCabinDownloadState(index);

         listBuilder.AddItem(index, m_MediaCabinDevices[index].deviceType, MEDIA_SOURCE_LIST_ITEM).
         AddData(m_MediaCabinDevices[index].sourceIconIdx).
         AddData(isCabinSourceInIndexActive(index)).
         AddData(m_MediaCabinDevices[index].deviceText).
         AddData(isSourceInIndexCabinEnabled(index)).
         AddDataBindingUpdater<DownloadVisibilityDataItemDataBindingSource>(data);
         ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinMediaSourceToListBuilder Update to List Builder"));
      }
   }

   return listBuilder.CreateDataProvider();
}


DownloadVisibilityDataItemData MediaSourceListHandler::updateCabinDownloadState(int index)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState index %d ", index));
   DownloadVisibilityDataItemData dataUpdate;
   if (index == EN_CABINSOURCELIST_USBCAMPORT || index == EN_CABINSOURCELIST_ROUTER_USB) //download progress crq
   {
      EnumConst::entSourceType l_sourceType = EnumConst::entSourceType_DEFAULT ;
      if (index == EN_CABINSOURCELIST_ROUTER_USB)
      {
         l_sourceType = EnumConst::entSourceType_CMR;
      }
      else
      {
         l_sourceType = EnumConst::entSourceType_CMP;
      }
      ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState index l_sourceType %d %d ", index, ETG_CENUM(entSourceType, l_sourceType)));
      if ((isSourceInIndexCabinEnabled(index) == false) || (DbDownloadMap::getInstance().getDbDownloadState(l_sourceType) == EnumConst::entDownloadState_Downloadsuccess))
      {
         ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState Download Success"));
         dataUpdate.mDownloadIconVisibilityVal = false;
         dataUpdate.mDownloadAnimationStatus = false;//stop download animation
      }
      else
      {
         dataUpdate.mDownloadIconVisibilityVal = true;
         if (((index == EN_CABINSOURCELIST_USBCAMPORT) && (DbDownloadMap::getInstance().getDbState(EnumConst::entSourceType_CMP) == EnumConst::entDbState_DBInPreparation))
               || ((index == EN_CABINSOURCELIST_ROUTER_USB) && (DbDownloadMap::getInstance().getDbState(EnumConst::entSourceType_CMR) == EXT_CMR_DB_INPREPARATION)))
         {
            ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState Start Download Loading Animation"));
            dataUpdate.mDownloadIconSwitchVal = EXT_DOWNLOAD_LOADING;//switch to animation icon group
            dataUpdate.mDownloadAnimationStatus = true;//start download animation
         }
         else if (DbDownloadMap::getInstance().getDbDownloadState(l_sourceType) == EnumConst::entDownloadState_DownloadError)
         {
            ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState Update Download Failure Icon"));
            dataUpdate.mDownloadIconSwitchVal = EXT_DOWNLOAD_FAILURE;// switch to download failure icon group
            dataUpdate.mDownloadAnimationStatus = false;//stop download animation
         }
         else
         {
            ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState Default Icon"));
            dataUpdate.mDownloadIconVisibilityVal = false;
         }
      }
   }
   else
   {
      ETG_TRACE_USR4(("MediaSourceListHandler::updateCabinDownloadState Not an External Source "));
      dataUpdate.mDownloadIconVisibilityVal = false;
   }

   return dataUpdate;
}


bool MediaSourceListHandler::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   if (oMsg.GetReaction() == enRelease && oMsg.GetListId() == LIST_ID_MEDIA_SOURCE_LIST)
   {
      ETG_TRACE_USR4(("MediaSourceListHandler::onCourierMessage::ButtonReactionMsg LIST_ID_MEDIA_SOURCE_LIST row %d", oMsg.GetHdl()));
      ETG_TRACE_USR4(("MediaSourceListHandler::onCourierMessage::ButtonReactionMsg LIST_ID_MEDIA_SOURCE_LIST column %d", oMsg.GetSubHdl()));
      ETG_TRACE_USR4(("MediaSourceListHandler::onCourierMessage::ButtonReactionMsg LIST_ID_MEDIA_SOURCE_LIST value %d", oMsg.GetListId()));
      onSourceSelectedInSourceList(oMsg.GetSubHdl());
      isMsgConsumed = true;
   }
   return isMsgConsumed;
}


void MediaSourceListHandler::onSourceSelectedInSourceList(unsigned int devicetype)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::onSourceSelectedInSourceList devicetype %d", devicetype));

   bool u8AVPConnectStatus = BTClientHandler::getInstance()->isAVPProfileActiveInConnectedBTDevice();
   bool u8BTConnectStatus = BTClientHandler::getInstance()->isAnyBTDeviceConnected();
   ETG_TRACE_USR4(("MediaSourceListHandler::onSourceSelectedInSourceList u8AVPConnectStatus  %d ", u8AVPConnectStatus));
   ETG_TRACE_USR4(("MediaSourceListHandler::onSourceSelectedInSourceList u8BTConnectStatus  %d ", u8BTConnectStatus));

   if (devicetype == FI_EN_E8DTY_USB2)//for usb2 message calling
   {
      POST_MSG((COURIER_MESSAGE_NEW(USB2SelectDeviceMsg)(FI_EN_E8DTY_USB)));
   }
   else if (devicetype == FI_EN_E8DTY_USB_CAMERAPORT || devicetype == FI_EN_E8DTY_USB_CAMERAPORT_HDMI || devicetype == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2 || devicetype == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SelectDeviceSourceMsg)(static_cast<tenDeviceType>(devicetype))));
   }
   else if (devicetype == FI_EN_E8DTY_BLUETOOTH && u8BTConnectStatus == false)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Media#MediaPlayer_Pop_Up#Pfo_MediaConfirmBTPairing"))));//popuptrigger
   }
   else if (devicetype == FI_EN_E8DTY_BLUETOOTH && u8BTConnectStatus == true && u8AVPConnectStatus == false)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Media#MediaPlayer_Pop_Up#Pfo_MediaBTStreamingUnavailable"))));//popuptrigger
   }
   else if ((devicetype == FI_EN_E8DTY_USB) || (devicetype == FI_EN_E8DTY_AUX)  || (devicetype == FI_EN_E8DTY_IPOD)
            || (devicetype == FI_EN_E8DTY_BLUETOOTH && u8BTConnectStatus == true && u8AVPConnectStatus == true))
   {
      POST_MSG((COURIER_MESSAGE_NEW(SelectDeviceSourceMsg)(static_cast<tenDeviceType>(devicetype))));
   }
   else if (devicetype == FI_EN_E8DTY_ROUTER_USB)
   {
      POST_MSG((COURIER_MESSAGE_NEW(SelectDeviceSourceMsg)(static_cast<tenDeviceType>(devicetype))));
   }
}


void MediaSourceListHandler::setExternalMediaSourceConnectionStateHandlerInstance(ExternalMediaSourceConnectionStateHandler* _externalMediaSourceConnectionStateHandlerInstance)
{
   if (_externalMediaSourceConnectionStateHandlerInstance != NULL)
   {
      _externalMediaSourceConnectionStateHandler = _externalMediaSourceConnectionStateHandlerInstance;
      _externalMediaSourceConnectionStateHandler->addExternalSourceAvailabilityStatusObserver((*this));
   }
   else
   {
      ETG_TRACE_ERR(("MediaSourceListHandler::_externalMediaSourceConnectionStateHandler is NULL"));
   }
}


void MediaSourceListHandler::setExternalMediaSourceConnectionStatusInstance(ExternalMediaSourceConnectionStatus* _externalMediaSourceConnectionStatusInstance)
{
   if (_externalMediaSourceConnectionStatusInstance != NULL)
   {
      _externalMediaSourceConnectionStatus = _externalMediaSourceConnectionStatusInstance;
   }
   else
   {
      ETG_TRACE_ERR(("MediaSourceListHandler::_externalMediaSourceConnectionStatusInstance is NULL"));
   }
}


#endif


void MediaSourceListHandler::initialiseMediaDevicesMap()
{
   ETG_TRACE_USR1(("MediaSourceListHandler::InitialiseMediaDevicesMap"));
   m_MediaDevices[EN_SOURCELIST_IPHONE].deviceType = FI_EN_E8DTY_IPOD;
   m_MediaDevices[EN_SOURCELIST_BLUETOOTH].deviceType = FI_EN_E8DTY_BLUETOOTH;
   m_MediaDevices[EN_SOURCELIST_USB1].deviceType = FI_EN_E8DTY_USB;
   m_MediaDevices[EN_SOURCELIST_USB2].deviceType = FI_EN_E8DTY_USB2;
   m_MediaDevices[EN_SOURCELIST_AUX].deviceType = FI_EN_E8DTY_AUX;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   m_MediaDevices[EN_SOURCELIST_IPHONE].sourceIconIdx = EN_ICON_IPHONE;
   m_MediaDevices[EN_SOURCELIST_BLUETOOTH].sourceIconIdx = EN_ICON_BLUETOOTH;
   m_MediaDevices[EN_SOURCELIST_USB1].sourceIconIdx = EN_ICON_USB1;
   m_MediaDevices[EN_SOURCELIST_USB2].sourceIconIdx = EN_ICON_USB2;
   m_MediaDevices[EN_SOURCELIST_AUX].sourceIconIdx = EN_ICON_AUX;

   m_MediaDevices[EN_SOURCELIST_IPHONE].deviceText = Candera::String(TextId_0x009F);
   m_MediaDevices[EN_SOURCELIST_BLUETOOTH].deviceText = Candera::String(TextId_0x00A1);
   m_MediaDevices[EN_SOURCELIST_USB1].deviceText = Candera::String(TextId_0x1A35);
   m_MediaDevices[EN_SOURCELIST_USB2].deviceText = Candera::String(TextId_0x1A36);
   m_MediaDevices[EN_SOURCELIST_AUX].deviceText = Candera::String(TextId_0x00A0);
#endif
}


void MediaSourceListHandler::initialiseMediaCabinDevicesMap()
{
   ETG_TRACE_USR1(("MediaSourceListHandler::initialiseMediaCabinDevicesMap"));
   m_MediaCabinDevices[EN_CABINSOURCELIST_ROUTER_USB].deviceType = FI_EN_E8DTY_ROUTER_USB;
   m_MediaCabinDevices[EN_CABINSOURCELIST_ROUTER_USB].srcId = SRC_CMR1_USB1_VIDEO;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORT].deviceType = FI_EN_E8DTY_USB_CAMERAPORT;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORT].srcId = SRC_CAMERAPORT1_USB_VIDEO;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI].deviceType = FI_EN_E8DTY_USB_CAMERAPORT_HDMI;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI].srcId = SRC_CAMERAPORT1_HDMI;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_2].deviceType = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_2].srcId = SRC_CAMERAPORT2_HDMI;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_3].deviceType = FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_3].srcId = SRC_CAMERAPORT3_HDMI;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB1].deviceType = FI_EN_E8DTY_USB;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB2].deviceType = FI_EN_E8DTY_USB2;
   m_MediaCabinDevices[EN_CABINSOURCELIST_BLUETOOTH].deviceType = FI_EN_E8DTY_BLUETOOTH;
   m_MediaCabinDevices[EN_CABINSOURCELIST_IPHONE].deviceType = FI_EN_E8DTY_IPOD;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   m_MediaCabinDevices[EN_CABINSOURCELIST_ROUTER_USB].sourceIconIdx = EN_ICON_ROUTER_USB;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORT].sourceIconIdx = EN_ICON_USBCAMERAPORT;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI].sourceIconIdx = EN_ICON_USBCAMERAPORTHDMI;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_2].sourceIconIdx = EN_ICON_USBCAMERAPORTHDMI2;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_3].sourceIconIdx = EN_ICON_USBCAMERAPORTHDMI3;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB1].sourceIconIdx = EN_ICON_USB1;
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB2].sourceIconIdx = EN_ICON_USB2;
   m_MediaCabinDevices[EN_CABINSOURCELIST_BLUETOOTH].sourceIconIdx = EN_ICON_BLUETOOTH;
   m_MediaCabinDevices[EN_CABINSOURCELIST_IPHONE].sourceIconIdx = EN_ICON_IPHONE;

   m_MediaCabinDevices[EN_CABINSOURCELIST_ROUTER_USB].deviceText = Candera::String(TextId_0x13AF);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORT].deviceText = Candera::String(TextId_0x13B2);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI].deviceText = Candera::String(TextId_0x13B4);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_2].deviceText = Candera::String(TextId_0x13B5);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_3].deviceText = Candera::String(TextId_0x1A14);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB1].deviceText = Candera::String(TextId_0x1A35);
   m_MediaCabinDevices[EN_CABINSOURCELIST_USB2].deviceText = Candera::String(TextId_0x1A36);
   m_MediaCabinDevices[EN_CABINSOURCELIST_BLUETOOTH].deviceText = Candera::String(TextId_0x00A1);
   m_MediaCabinDevices[EN_CABINSOURCELIST_IPHONE].deviceText = Candera::String(TextId_0x009F);
#endif
}


bool MediaSourceListHandler::isSourceAvailableInCockpitRegion(int index)
{
   bool isSourceAvailable = false;
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
   int g_SM_VARIANT_AUX = getstateMachineVariantAux();
#endif

   if (m_MediaDevices[index].deviceType == FI_EN_E8DTY_IPOD || m_MediaDevices[index].deviceType == FI_EN_E8DTY_BLUETOOTH ||
         m_MediaDevices[index].deviceType == FI_EN_E8DTY_USB || m_MediaDevices[index].deviceType == FI_EN_E8DTY_USB2)
   {
      isSourceAvailable = true;
   }
   else if (m_MediaDevices[index].deviceType == FI_EN_E8DTY_AUX && g_SM_VARIANT_AUX == 1)
   {
      isSourceAvailable = true;
   }

   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCockpitRegion isSourceAvailable is %d", isSourceAvailable));
   return isSourceAvailable;
}


bool MediaSourceListHandler::isSourceAvailableInCabinRegion(int index)
{
   bool isSourceAvailable = false;
   int iHdmiKdsAvailability = _kdsHandler->getHdmiKDSValue();
   bool iUsbCamportKdsAvailability = _kdsHandler->getUsbCamportKdsValue();

   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCabinRegion iHdmiKdsAvailability is %d", iHdmiKdsAvailability));
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCabinRegion iUsbCamportKdsAvailability is %d", iUsbCamportKdsAvailability));
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCabinRegion m_bRouterUsbState is %d", m_bRouterUsbState));
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCabinRegion index is %d", index));

   if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_IPOD || m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_BLUETOOTH ||
         m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB || m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB2)
   {
      isSourceAvailable = true;
   }
   else if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB_CAMERAPORT && (iUsbCamportKdsAvailability == true) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP1)))
   {
      isSourceAvailable = true;
   }
   else if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI && (iHdmiKdsAvailability == HDMI_KDS_AVAIL) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP1)))
   {
      isSourceAvailable = true;
   }
   else if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2 && (iHdmiKdsAvailability == HDMI_KDS_AVAIL) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP1)) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP2)))
   {
      isSourceAvailable = true;
   }
   else if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3 && (iHdmiKdsAvailability == HDMI_KDS_AVAIL) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP1)) && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMP3)))
   {
      isSourceAvailable = true;
   }
   /*else if (m_MediaCabinDevices[index].deviceType == FI_EN_E8DTY_ROUTER_USB && (_systemInfoMap->getSystemCategoryAvailability(SYS_CATEGORY_CMR)))
   {
      isSourceAvailable = true;
   }*/// Temporarily commented as part of 1606014 - for disabling Router USB
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceAvailableInCabinRegion isSourceAvailable is %d", isSourceAvailable));
   return isSourceAvailable;
}


bool MediaSourceListHandler::isCockpitSourceInIndexActive(int index)
{
   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int  iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   int  iUSbDevNum = tclMediaClientHandler::getInstance().GetActiveUSBSource(iSubSrcId);

   ETG_TRACE_USR4(("MediaSourceListHandler::isCockpitSourceInIndexActive iSubSrcId value is %d", iSubSrcId));
   ETG_TRACE_USR4(("MediaSourceListHandler::isCockpitSourceInIndexActive srcid value is %d", srcid));
   ETG_TRACE_USR4(("MediaSourceListHandler::isCockpitSourceInIndexActive iDeviceType value is %d", iDeviceType));
   ETG_TRACE_USR4(("MediaSourceListHandler::isCockpitSourceInIndexActive iUSbDevNum value is %d", iUSbDevNum));
   bool isSourceIndexActive = false;

   if (index == EN_SOURCELIST_IPHONE && (iDeviceType == FI_EN_E8DTY_IPHONE || iDeviceType == FI_EN_E8DTY_IPOD)
         && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_SOURCELIST_BLUETOOTH && iDeviceType == FI_EN_E8DTY_BLUETOOTH
            && srcid == SRC_PHONE_BTAUDIO)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_SOURCELIST_USB1 &&  iUSbDevNum == USB_ONE
            && (iDeviceType == FI_EN_E8DTY_USB || iDeviceType == FI_EN_E8DTY_MTP)
            && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_SOURCELIST_USB2 &&  iUSbDevNum == USB_TWO
            && (iDeviceType == FI_EN_E8DTY_USB || iDeviceType == FI_EN_E8DTY_MTP)
            && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_SOURCELIST_AUX && srcid == SRC_MEDIA_AUX)
   {
      isSourceIndexActive = true;
   }
   ETG_TRACE_USR4(("MediaSourceListHandler::isCockpitSourceInIndexActive isSourceIndexActive  is %d", isSourceIndexActive));
   return isSourceIndexActive;
}


bool MediaSourceListHandler:: isCabinSourceInIndexActive(int index)
{
   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   int  iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   int  iUSbDevNum = tclMediaClientHandler::getInstance().GetActiveUSBSource(iSubSrcId);

   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive iSubSrcId value is %d", iSubSrcId));
   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive srcid value is %d", srcid));
   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive iDeviceType value is %d", iDeviceType));
   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive iUSbDevNum value is %d", iUSbDevNum));
   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive index value is %d _isGlassAvailable %d", index, _isGlassAvailable));
   bool isSourceIndexActive = false;

   if (index == EN_CABINSOURCELIST_IPHONE && (iDeviceType == FI_EN_E8DTY_IPHONE || iDeviceType == FI_EN_E8DTY_IPOD)
         && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_BLUETOOTH && iDeviceType == FI_EN_E8DTY_BLUETOOTH
            && srcid == SRC_PHONE_BTAUDIO)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_USB1 &&  iUSbDevNum == USB_ONE
            && (iDeviceType == FI_EN_E8DTY_USB || iDeviceType == FI_EN_E8DTY_MTP)
            && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_USB2 &&  iUSbDevNum == USB_TWO
            && (iDeviceType == FI_EN_E8DTY_USB || iDeviceType == FI_EN_E8DTY_MTP)
            && srcid == SRC_MEDIA_PLAYER)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORT)
   {
      if ((srcid == SRC_CAMERAPORT1_USB_VIDEO) || (srcid == SRC_CAMERAPORT1_USB_AUDIO))
      {
         isSourceIndexActive = true;
      }
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI && srcid == SRC_CAMERAPORT1_HDMI)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI_2 && srcid == SRC_CAMERAPORT2_HDMI)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI_3 && srcid == SRC_CAMERAPORT3_HDMI)
   {
      isSourceIndexActive = true;
   }
   else if (index == EN_CABINSOURCELIST_ROUTER_USB && (srcid == SRC_CMR1_USB1_VIDEO || srcid == SRC_CMR1_USB1_AUDIO))
   {
      isSourceIndexActive = true;
   }
   ETG_TRACE_USR4(("MediaSourceListHandler:: isCabinSourceInIndexActive isSourceIndexActive  is %d", isSourceIndexActive));
   return isSourceIndexActive;
}


bool MediaSourceListHandler::isSourceInIndexCockpitEnabled(int index)
{
   bool  isBTEnabled = BTClientHandler::getInstance()->isBluetoothSettingsOn();
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCockpitEnabled index %d", index));
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCockpitEnabled isBTEnabled %d", isBTEnabled));
   bool isSourceIndexEnabled = false;

   if (index == EN_SOURCELIST_BLUETOOTH)
   {
      isSourceIndexEnabled = isBTEnabled;
   }
   else if (index >= EN_SOURCELIST_IPHONE && index <= EN_SOURCELIST_AUX)
   {
      isSourceIndexEnabled = true;
   }
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCockpitEnabled isSourceIndexEnabled %d", isSourceIndexEnabled));
   return isSourceIndexEnabled;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
bool MediaSourceListHandler::isSourceInIndexCabinEnabled(int index)
{
   bool  isBTEnabled = BTClientHandler::getInstance()->isBluetoothSettingsOn();
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCabinEnabled index %d", index));
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCabinEnabled isBTEnabled %d", isBTEnabled));
   bool isSourceIndexEnabled = false;

   if (index >= EN_CABINSOURCELIST_USB1)
   {
      isSourceIndexEnabled = (index == EN_CABINSOURCELIST_BLUETOOTH) ? isBTEnabled : true;
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORT)
   {
      isSourceIndexEnabled = _externalMediaSourceConnectionStatus->getExternalSourceConnectionStatus(m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORT].srcId);
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI)
   {
      isSourceIndexEnabled = _externalMediaSourceConnectionStatus->getExternalSourceConnectionStatus(m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI].srcId);
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI_2)
   {
      isSourceIndexEnabled = _externalMediaSourceConnectionStatus->getExternalSourceConnectionStatus(m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_2].srcId);
   }
   else if (index == EN_CABINSOURCELIST_USBCAMPORTHDMI_3)
   {
      isSourceIndexEnabled = _externalMediaSourceConnectionStatus->getExternalSourceConnectionStatus(m_MediaCabinDevices[EN_CABINSOURCELIST_USBCAMPORTHDMI_3].srcId);
   }
   else if (index == EN_CABINSOURCELIST_ROUTER_USB)
   {
      isSourceIndexEnabled = _externalMediaSourceConnectionStatus->getExternalSourceConnectionStatus(m_MediaCabinDevices[EN_CABINSOURCELIST_ROUTER_USB].srcId);
   }
   ETG_TRACE_USR4(("MediaSourceListHandler::isSourceInIndexCabinEnabled isSourceIndexEnabled %d", isSourceIndexEnabled));
   return isSourceIndexEnabled;
}


#endif


void MediaSourceListHandler::setSystemInfoMapInstance(SystemInfoMap* _systemInfoMapInstance)
{
   if (_systemInfoMapInstance != NULL)
   {
      _systemInfoMap = _systemInfoMapInstance;
   }
   else
   {
      ETG_TRACE_USR1(("MediaSourceListHandler::_systemInfoMapInstance is NULL"));
   }
}


void MediaSourceListHandler::setCamportUsbAvailabilityInSourceList(bool bState)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::setCamportUsbAvailabilityInSourceList bState is %d", bState));
   m_bCamportUsbState = bState;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
#endif
}


void MediaSourceListHandler::setRouterUsbAvailabilityInSourceList(bool bState)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::setRouterUsbAvailabilityInSourceList bState is %d", bState));
   m_bRouterUsbState = bState;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   updateMediaSourceList(LIST_ID_MEDIA_SOURCE_LIST);
#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

void MediaSourceListHandler::updateMediaSourceList(const uint32& listID)
{
   ETG_TRACE_USR4(("MediaSourceListHandler::updateMediaSourceList is %d", listID));
   ListRegistry::s_getInstance().updateList(listID);
}


#endif
void MediaSourceListHandler::setKdsHandlerInstance(KDSHandler* _kdsHandlerInstance)
{
   _kdsHandler = _kdsHandlerInstance;
}


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
void MediaSourceListHandler::setstateMachineVariantAux(int stateMachineVariantAux)
{
   m_stateMachineVariantAux = stateMachineVariantAux ;
}


int MediaSourceListHandler::getstateMachineVariantAux()
{
   return m_stateMachineVariantAux ;
}


#endif
}//end of namespace Core
}//end of namespace App
