/* ***************************************************************************************
* FILE:          MediaHall.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  MediaHall.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "App/Core/MediaHall.h"
#include "unistd.h"
#include "Core/MediaDefines.h"
#include "PluginConstants.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::MediaHall::
#include "trcGenProj/Header/MediaHall.cpp.trc.h"
#endif
#define PLUGIN_NAME_MEDIA "Media"
#define PLUGIN_NAME_ROUTER "Router"
#define CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION  "PlaybackAction"
#define CTRLBLOCK_NAME_MEDIA_HDMISTATUS  "HDMIStatus"
#define CTRLBLOCK_NAME_MEDIA_USBAUDIOSTATUS  "USBAudioStatus"
#define CTRLBLOCK_ROUTER_STREAM3_PLAYBACKACTION "RouterStream3PlayBackAction"
#define CTRLBLOCK_ROUTER_STREAM3_SEEKTO "RouterStream3SeekTo"
#define CTRLBLOCK_STREAM3_ROUTER_TOTALPLAYTIME "RouterSteam3TotalPlayTime"
#define CTRLBLOCK_STREAM3_ROUTER_ELAPSEDTIME "RouterStream3ElapsedTime"
#define CTRLBLOCK_STREAM3_ROUTER_TRACKNAME "RouterStream3TrackName"
#define CTRLBLOCK_STREAM3_ROUTER_ARTISTNAME "RouterStream3ArtistName"
#define CTRLBLOCK_ROUTER_STREAM1_PLAYBACKACTION  "RouterStream1PlayBackAction"
#define CTRLBLOCK_STREAM1_ROUTER_ELAPSEDTIME  "RouterStream1ElapsedTime"
#define CTRLBLOCK_STREAM1_ROUTER_TOTALPLAYTIME  "RouterSteam1TotalPlayTime"
#define CTRLBLOCK_STREAM1_ROUTER_VIDEONAME "RouterStream1VideoName"
#define CTRLBLOCK_STREAM1_ROUTER_VIDEOEPISODE "RouterStream1VideoEpisode"
#define CTRLBLOCK_ROUTER_USBSTATUS "RouterUSBStatus"
#define CTRLBLOCK_ROUTER_STREAM1_SEEKTO "RouterStream1SeekTo"
#define CTRLBLOCK_ROUTER_MEDIA_DB_STATUS "RouterDBStatus"
#define CTRLBLOCK_ROUTER_VIDEO_REPEAT_MODE "RouterUSBVideoRepeatMode"
#define CTRLBLOCK_ROUTER_MEDIA_CHECKSUM_STATUS "RouterUSBChecksumStatus"
#define CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE "SignalSourceChange"
#define CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE "RouterSignalSourceChange"
//#define VARIANT_S_FTR_ENABLE_VIDEO_PLAYBACK

using namespace ::App::Core;
using namespace ::CourierTunnelService::CourierMessageReceiver;

namespace App {
namespace Core {


DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", MediaHall, Info);


/************************************************************************
*FUNCTION		: MediaHall
*DESCRIPTION	: constructor
*PARAMETER		: None
*RETURNVALUE	: void
************************************************************************/
MediaHall::MediaHall()
   : HallComponentBase("", "App.Core.AppHmi_Media", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Media"),
     _VehicleSrvClient(NULL),
     _externalMediaSourceHandler(NULL),
     _externalMediaStateHandler(NULL),
     _externalMediaListDBManager(NULL),
     // _verticalListManager(NULL),
     // _horizontalListManager(NULL),
     //  _listGuiRequestHandler(NULL),
     _externalMediaClientHandler(NULL),
     _externalMediaSourceStateUpdateHandler(NULL),
     _externalMediaSourceConnectionStateHandler(NULL),
     _kdsHandler(NULL),
     _cabinSystemStateHandler(NULL),
     _hmiStateHandler(NULL),
     _playableContentInterfaceHandler(NULL),
     _externalMediaListManager(NULL),
     _cabinSourceRestoration(NULL),
     _playableContentMap(NULL),
     _systemInfoMap(NULL),
     _externalMediaSourceConnectionStatus(NULL),
     _mediaPlaybackStatusUpdater(NULL),
     _mediaPlaybackLongDataUpdater(NULL),
     _navigationModeDBHandler(NULL)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("MediaHall:: Media Hall constructor"));

   _VehicleSrvClient = VehicleSrvClient::poGetInstance();
   _applicationSwitchClientComponent = ApplicationSwitchClientComponent::poGetInstance();
   _applicationSwitchClientHandler = ApplicationSwitchClientHandler::poGetInstance();
   _hmiModelComponent = &(HMIModelComponent::getInstance());
   _hmiModelListImplementation = &(HMIModelListImplementation::getInstance());
   _hmiModelTabImplementation = &(HMIModelTabImplementation::getInstance());
   _mediaClientHandler =  &(tclMediaClientHandler::getInstance());
   _speedStatusHandler =  &(SpeedLockStatusHandler::getInstance());
   _unSupportedPopupHandler =  &(UnSupportedPopupHandler::getInstance());
   _mediaSourceHandling = &(MediaSourceHandling::getInstance());
   _tunerClientHandler = TunerClientHandler::getInstance();
   _externalMediaPlaybackHandler = ExternalMediaPlaybackHandler::getInstance();

   _pluginClientHandler = PluginClientHandler::getInstance();
   _btClientHandler = BTClientHandler::getInstance();
   _stateTransitionHandler = &(StateTransitionHandler::getInstance());
   _mediaSourceListHandler = &(MediaSourceListHandler::getInstance());

   if (NULL == _externalMediaClientHandler)
   {
      _externalMediaClientHandler = new ExternalMediaClientHandler();
   }
   if (NULL == _externalMediaSourceStateUpdateHandler)
   {
      _externalMediaSourceStateUpdateHandler = new ExternalMediaSourceStateUpdateHandler();
   }
   if (NULL == _kdsHandler)
   {
      _kdsHandler = new KDSHandler();
   }
   if (NULL == _systemInfoMap)
   {
      _systemInfoMap = new SystemInfoMap();
   }
   if (NULL == _externalMediaSourceConnectionStatus)
   {
      _externalMediaSourceConnectionStatus = new ExternalMediaSourceConnectionStatus();
   }
   if (NULL == _externalMediaSourceHandler)
   {
      _externalMediaSourceHandler = new ExternalMediaSourceHandler();
   }
   if (NULL == _playableContentMap)
   {
      _playableContentMap = new PlayableContentMap();
   }
   if (NULL == _playableContentInterfaceHandler)
   {
      _playableContentInterfaceHandler = new PlayableContentInterfaceHandler();
   }
   if (NULL == _externalMediaStateHandler)
   {
      _externalMediaStateHandler = new ExternalMediaStateHandler();
   }
   if (NULL == _externalMediaListDBManager)
   {
      _externalMediaListDBManager = new ExternalMediaListDBManager(_externalMediaSourceHandler);
   }
   /* if (NULL == _horizontalListManager)
    {
       _horizontalListManager = new HorizontalListManager(EnumConst::entSourceType_CMP);
    }
    if (NULL == _verticalListManager)
    {
       _verticalListManager = new VerticalListManager(_horizontalListManager, _externalMediaSourceHandler);
    }
    if (NULL == _listGuiRequestHandler)
    {
       _listGuiRequestHandler = new ListGuiRequestHandler(_horizontalListManager, _verticalListManager);
    }*/
   if (NULL == _externalMediaListManager)
   {
      _externalMediaListManager = new ExternalMediaListManager(_externalMediaSourceHandler, _playableContentMap);
   }
   if (NULL == _cabinSystemStateHandler)
   {
      _cabinSystemStateHandler = new CabinSystemStateHandler();
   }
   if (NULL == _hmiStateHandler)
   {
      _hmiStateHandler = new HmiStateHandler();
   }
   if (NULL == _externalMediaSourceConnectionStateHandler)
   {
      _externalMediaSourceConnectionStateHandler = new ExternalMediaSourceConnectionStateHandler();
   }
   if (NULL == _cabinSourceRestoration)
   {
      _cabinSourceRestoration =  new CabinSourceRestoration();
   }
   if (NULL == _mediaPlaybackStatusUpdater)
   {
      _mediaPlaybackStatusUpdater =  new MediaPlaybackStatusUpdater();
   }
   if (NULL == _mediaPlaybackLongDataUpdater)
   {
      _mediaPlaybackLongDataUpdater =  new MediaPlaybackLongDataUpdater();
   }

   if (NULL != _pluginClientHandler)
   {
      std::vector<std::string> camportUSBMediaCtrlBlockNames;
      std::vector<std::string> routerUSBMediaCtrlBlockNames;
      camportUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_NAME_MEDIA_PLAYBACKACTION);
      camportUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_NAME_MEDIA_HDMISTATUS);
      camportUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_NAME_MEDIA_USBAUDIOSTATUS);
      camportUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_NAME_MEDIA_SIGNAL_SOURCE_CHANGE);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_STREAM1_PLAYBACKACTION);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM1_ROUTER_ELAPSEDTIME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM1_ROUTER_TOTALPLAYTIME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM1_ROUTER_VIDEONAME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM1_ROUTER_VIDEOEPISODE);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_USBSTATUS);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_STREAM1_SEEKTO);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_MEDIA_DB_STATUS);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_VIDEO_REPEAT_MODE);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_STREAM3_PLAYBACKACTION);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM3_ROUTER_ELAPSEDTIME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM3_ROUTER_TOTALPLAYTIME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM3_ROUTER_TRACKNAME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_STREAM3_ROUTER_ARTISTNAME);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_STREAM3_SEEKTO);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_MEDIA_CHECKSUM_STATUS);
      routerUSBMediaCtrlBlockNames.push_back(CTRLBLOCK_ROUTER_SIGNAL_SOURCE_CHANGE);
      std::map<std::string, std::vector<std::string> > pluginCtrlBlockMap;
      pluginCtrlBlockMap.insert(std::pair<std::string, std::vector<std::string> >(PLUGIN_NAME_MEDIA, camportUSBMediaCtrlBlockNames));
      pluginCtrlBlockMap.insert(std::pair<std::string, std::vector<std::string> >(PLUGIN_NAME_ROUTER, routerUSBMediaCtrlBlockNames));

      _pluginClientHandler->registerPluginSignals(pluginCtrlBlockMap);
   }

   setExternalMediaClientHandlerDependency();
   setCabinSourceRestorationDependency();
   setKDSHandlerDependency();
   setPlayableContentMapDependency();
   setSystemInfoMapDependency();
   //for factory reset
   DefSetServiceBase::s_Intialize("AppHmiMediaServicePort");//port name needs to be same as port name given in cmc file
   _mdefSetServiceBase = DefSetServiceBase::GetInstance();
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vRegisterforUpdate(this);
   }

   if (_applicationSwitchClientComponent != NULL)
   {
      _applicationSwitchClientComponent->setApplicationID(APPID_APPHMI_MEDIA);
   }

   if (_navigationModeDBHandler == NULL)
   {
      _navigationModeDBHandler = NavigationModeDBHandler::getInstance();
   }
}


void MediaHall::setExternalMediaClientHandlerDependency()
{
   _applicationSwitchClientHandler->setExternalMediaClientHandlerInstance(_externalMediaClientHandler);
   _mediaSourceListHandler->setExternalMediaSourceConnectionStatusInstance(_externalMediaSourceConnectionStatus);
   _mediaSourceListHandler->setExternalMediaSourceConnectionStateHandlerInstance(_externalMediaSourceConnectionStateHandler);
   _mediaSourceListHandler->setExternalMediaListDBManagerInstance(_externalMediaListDBManager);
   _externalMediaSourceConnectionStateHandler->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _externalMediaSourceConnectionStateHandler->setCabinSystemStateHandlerInstance(_cabinSystemStateHandler);
   _externalMediaSourceConnectionStateHandler->setSystemInfoMapInstance(_systemInfoMap);
   _externalMediaSourceConnectionStateHandler->setExternalMediaSourceConnectionStatusInstance(_externalMediaSourceConnectionStatus);
   _externalMediaSourceStateUpdateHandler-> setExternalMediaSourceConnectionStateHandlerInstance(_externalMediaSourceConnectionStateHandler);
   _externalMediaSourceStateUpdateHandler->setExternalMediaSourceConnectionStatusInstance(_externalMediaSourceConnectionStatus);
   _mediaSourceListHandler->setSystemInfoMapInstance(_systemInfoMap);
   _playableContentInterfaceHandler->setExternalMediaClientHandlerInstance(_externalMediaClientHandler);
   // _horizontalListManager -> setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _externalMediaClientHandler->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _mediaClientHandler->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _mediaSourceHandling->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _mediaSourceHandling->setExternalMediaClientHandlerInstance(_externalMediaClientHandler);
   _VehicleSrvClient->setExternalMediaClientHandlerInstance(_externalMediaClientHandler);
   _applicationSwitchClientHandler->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _externalMediaClientHandler->setExternalMediaListDBManagerInstance(_externalMediaListDBManager);
   _cabinSourceRestoration->setExternalMediaSourceHandlerInstance(_externalMediaSourceHandler);
   _cabinSourceRestoration->setExternalMediaListDBManagerInstance(_externalMediaListDBManager);
   _cabinSourceRestoration->setExternalMediaClientHandlerInstance(_externalMediaClientHandler);

   _externalMediaClientHandler->setSystemInfoMapInstance(_systemInfoMap);
   _externalMediaClientHandler->setPlayableContentInterfaceHandler(_playableContentInterfaceHandler);
   _externalMediaClientHandler->setHmiStateHandlerInstance(_hmiStateHandler);
}


void MediaHall::setKDSHandlerDependency()
{
   _mediaSourceListHandler->setKdsHandlerInstance(_kdsHandler);
   _hmiModelComponent->setKdsHandlerInstance(_kdsHandler);
   _mediaClientHandler->setKdsHandlerInstance(_kdsHandler);
}


void MediaHall::setCabinSourceRestorationDependency()
{
   _hmiModelComponent->setCabinSourceRestorationInstance(_cabinSourceRestoration);
}


void MediaHall::setPlayableContentMapDependency()
{
   // _horizontalListManager->setPlayableContentMapInstance(_playableContentMap);
   _playableContentInterfaceHandler->setPlayableContentMapInstance(_playableContentMap);
   _externalMediaClientHandler->setPlayableContentMapInstance(_playableContentMap);
   _cabinSourceRestoration->setPlayableContentMapInstance(_playableContentMap);
   _externalMediaPlaybackHandler->setPlayableContentMapInstance(_playableContentMap);
}


void MediaHall::setSystemInfoMapDependency()
{
   _cabinSystemStateHandler->setSystemInfoMapInstance(_systemInfoMap);
   _cabinSourceRestoration->setSystemInfoMapInstance(_systemInfoMap);
}


/************************************************************************
*FUNCTION		: ~MediaHall
*DESCRIPTION	: destructor
*PARAMETER		: None
*RETURNVALUE	: None
************************************************************************/
MediaHall::~MediaHall()
{
   if (NULL != _mediaClientHandler)
   {
      delete _mediaClientHandler;
      _mediaClientHandler = NULL;
   }
   if (NULL != _externalMediaSourceConnectionStatus)
   {
      delete _externalMediaSourceConnectionStatus;
      _externalMediaSourceConnectionStatus = NULL;
   }
   if (NULL != _externalMediaSourceStateUpdateHandler)
   {
      delete _externalMediaSourceStateUpdateHandler;
      _externalMediaSourceStateUpdateHandler = NULL;
   }
   if (NULL != _mediaSourceListHandler)
   {
      MediaSourceListHandler::getInstance().removeInstance();
   }
   if (NULL != _speedStatusHandler)
   {
      SpeedLockStatusHandler::getInstance().removeInstance();
   }
   if (NULL != _unSupportedPopupHandler)
   {
      UnSupportedPopupHandler::getInstance().removeInstance();
   }

   if (NULL != _tunerClientHandler)
   {
      delete _tunerClientHandler;
      _tunerClientHandler = NULL;
   }
   if (NULL != _externalMediaPlaybackHandler)
   {
      delete _externalMediaPlaybackHandler;
      _externalMediaPlaybackHandler = NULL;
   }
   _hmiModelComponent = NULL;
   if (NULL != _externalMediaClientHandler)
   {
      delete _externalMediaClientHandler;
      _externalMediaClientHandler = NULL;
   }
   if (NULL != _mediaPlaybackLongDataUpdater)
   {
      delete _mediaPlaybackLongDataUpdater;
      _mediaPlaybackLongDataUpdater = NULL;
   }
   if (NULL != _pluginClientHandler)
   {
      delete _pluginClientHandler;
      _pluginClientHandler = NULL;
   }
   if (NULL != _externalMediaSourceHandler)
   {
      delete _externalMediaSourceHandler;
      _externalMediaSourceHandler = NULL;
   }
   if (NULL != _playableContentInterfaceHandler)
   {
      delete _playableContentInterfaceHandler;
      _playableContentInterfaceHandler = NULL;
   }
   if (NULL != _externalMediaListDBManager)
   {
      delete _externalMediaListDBManager;
      _externalMediaListDBManager = NULL;
   }
   /* if (NULL == _listGuiRequestHandler)
    {
       delete _listGuiRequestHandler;
       _listGuiRequestHandler = NULL;
    }
   if (NULL != _horizontalListManager)
    {
       delete _horizontalListManager;
       _horizontalListManager = NULL;
    }
    if (NULL == _verticalListManager)
    {
       delete _verticalListManager;
       _verticalListManager = NULL;
    }*/
   if (_VehicleSrvClient)
   {
      delete _VehicleSrvClient;
   }
   if (NULL != _btClientHandler)
   {
      delete _btClientHandler;
      _btClientHandler = NULL;
   }
   if (NULL != _kdsHandler)
   {
      delete _kdsHandler;
      _kdsHandler = NULL;
   }
   if (NULL != _externalMediaSourceConnectionStateHandler)
   {
      delete _externalMediaSourceConnectionStateHandler;
      _externalMediaSourceConnectionStateHandler = NULL;
   }
   if (NULL != _cabinSourceRestoration)
   {
      delete _cabinSourceRestoration;
      _cabinSourceRestoration = NULL;
   }
   if (NULL != _hmiStateHandler)
   {
      delete _hmiStateHandler;
      _hmiStateHandler = NULL;
   }

   DefSetServiceBase::s_Destrory(); //to destroy DefSetServiceBase instance
   if (_mdefSetServiceBase != NULL)
   {
      _mdefSetServiceBase->vUnRegisterforUpdate(this);
      //donot delete this instance as its not created by us.
      _mdefSetServiceBase = NULL;
   }

   if (NULL != _navigationModeDBHandler)
   {
      delete _navigationModeDBHandler;
      _navigationModeDBHandler = NULL;
   }
}


/************************************************************************
*FUNCTION		: Perform_MediaTabGadget
*SWFL			: [SYSFL-4347]
************************************************************************/
bool MediaHall::onCourierMessage(const SelectHomescreenMediaTab& msg)
{
   ETG_TRACE_USR1(("MediaHall::onCourierMessage:SelectHomescreenMediaTab"));
   if (NULL != _applicationSwitchClientHandler)
   {
      uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
      ETG_TRACE_USR4(("MediaHall::onCourierMessage:SelectHomescreenMediaTab activeRegion:%d", u8activeRegion));

      if (u8activeRegion == MAX_REG) //cockpit
      {
         stNowPlayingInfo NowPlayingInfoStatus;

         HMIModelComponent::getInstance().bOnUpdateNowPlayingStatus();
      }
      else if (u8activeRegion == REG_A) //cabin a
      {
         stNowPlayingInfo NowPlayingInfoStatus;
         HMIModelComponent::getInstance().bOnUpdateNowPlayingStatus();
      }

      ETG_TRACE_USR4(("MediaHall::onCourierMessage:SelectHomescreenMediaTab Active Region::%d", u8activeRegion));
   }

   //func call context switch to source selection screen
   ApplicationSwitchClientHandler::poGetInstance()->reqContextSwitchSrcSelection();
   return true;
}


// This function is called whenver change in the state from appmaster like fg,bg...
bool MediaHall::onCourierMessage(const SwitchApplicationReq& oMsg)
{
   ETG_TRACE_USR4(("MediaHall::onCourierMessage:SwitchApplicationReq application id:%d", oMsg.GetApplicationID()));

   int32_t ApplicationID = oMsg.GetApplicationID();
   switch (ApplicationID)
   {
      case APPID_APPHMI_TUNER:
         ApplicationSwitchClientHandler::poGetInstance()->reqContextSwitchSrcSelectionHome(enActivityIDs__eActivityID_TUNER_SOURCE_SELECTION, APPID_APPHMI_TUNER);
         break;
      case APPID_APPHMI_PHONE:
         ApplicationSwitchClientHandler::poGetInstance()->reqContextSwitchSrcSelectionHome(enActivityIDs__eActivityID_PHONE_DEVICE_SELECTION, APPID_APPHMI_PHONE);
         break;
      default:
         ApplicationSwitchClientHandler::poGetInstance()->reqContextSwitchSrcSelectionHome(enActivityIDs__eActivityID_HOME, APPID_APPHMI_MASTER);
   }

   return true;
}


bool MediaHall::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("MediaHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


//Reset To Factory//////////
/* reqPrepareResponse: To know the response of the respective APP when Restore factory setting is pressed.
* @param[in]: None
* @param[in] :None
*/
void MediaHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR1(("MediaHall::reqPrepareResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendPrepareResponse(0, this);
   }
}


/**
 * reqExecuteResponse: To Execute the response when restore factory setting has been executed..
 * @param[in]: None
 * @param[in] :None
 */
void MediaHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR1(("MediaHall::reqExecuteResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendExecuteResponse(0, this);
      tclMediaClientHandler::getInstance().clearMediaplayerDataForFactoryReset();
   }
}


/**
 * reqExecuteResponse: To Finalize the response when restore factory setting has been executed.
 * @param[in]: None
 * @param[in] :None
 */
void MediaHall:: reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR1(("MediaHall::reqFinalizeResponse"));
   if (_mdefSetServiceBase)
   {
      _mdefSetServiceBase->sendFinalizeResponse(0, this);
   }
}


/**
* Trace command to call onSeek
*
* @param[in]      : Seek type
* @param[in]      : Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_onTestTraceCommand, "TraceCmd_onTestTraceCommand "))
void MediaHall::TraceCmd_onTestTraceCommand()
{
   ETG_TRACE_COMP(("MediaHall::TraceCmd_onTestTraceCommand"));
}


/**
* Trace command to call onSeek
*
* @param[in]      : Seek type
* @param[in]      : Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_Operation, "TraceCmd_Operation %d", uint8))
void MediaHall::TraceCmd_Operation(uint8 operationType)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_Operation: operation type %d", operationType));

   Courier::Message* msg = COURIER_MESSAGE_NEW(MediaOperationMsgReq)(operationType);
   if (msg != 0 && msg->Post())
   {
   }

   //if (m_poMediaHall != NULL)//radar review point
   //{
   //   m_poMediaHall->methodOperation(operationType);
   //}
}


ETG_I_CMD_DEFINE((TraceCmd_onMediaContext, "contextsignalchnage %d %d %d %d %d", uint32, uint32, uint32, uint32, uint32))
void MediaHall::TraceCmd_onMediaContext(uint32 regionid, uint32 currentAppid, uint32 currentactivityid, uint32 targetappid, uint32 targetactivityid)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_onMediaContext: regionid  %d", regionid));
   ETG_TRACE_USR4(("MediaHall::TraceCmd_onMediaContext: currentAppid  %d", currentAppid));
   ETG_TRACE_USR4(("MediaHall::TraceCmd_onMediaContext: currentactivityid  %d", currentactivityid));
   ETG_TRACE_USR4(("MediaHall::TraceCmd_onMediaContext: targetappid  %d", targetappid));
   ETG_TRACE_USR4(("MediaHall::TraceCmd_onMediaContext: targetactivityid  %d", targetactivityid));

   ApplicationSwitchClientHandler::poGetInstance()->MediacontextSignal(regionid, currentAppid, currentactivityid, targetappid, targetactivityid);
}


/**
* Trace command to call wait animation
*
* @param[in]      : enAnimationReqType
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_WaitAnimation, "TraceCmd_WaitAnimation %d", enAnimationReqType))
void MediaHall::TraceCmd_WaitAnimation(enAnimationReqType state)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_WaitAnimation: %d", state));
   Courier::Message* pMsg = COURIER_MESSAGE_NEW(WaitAnimationReqMsg)(state);
   if (pMsg)
   {
      pMsg->Post();
   }
}


void MediaHall::methodOperation(uint8 operationType)
{
   ETG_TRACE_USR4(("MediaHall::methodOperation: operation type %d", operationType));
   POST_MSG((COURIER_MESSAGE_NEW(MediaOperationMsgReq)(operationType)));
}


/**
* Trace command to call unsupported popup usb
*
* @param[in]      : bool
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_UnSupportedPopup, "TraceCmd_UnSupportedPopup %d", bool))
void MediaHall::TraceCmd_UnSupportedPopup(bool state)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_UnSupportedPopup: %d", state));
   if (state == true)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Media#MediaPlayer_Pop_Up_system#Pfo_MediaUnsupportedPopupMessage"));
      if (NULL != msg)
      {
         (void)msg->Post();
      }
   }
   else if (state == false)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("Media#MediaPlayer_Pop_Up_system#Pfo_MediaUnsupportedPopupMessage"));
      if (NULL != msg)
      {
         (void)msg->Post();
      }
   }
}


/**
* Trace command to call unsupported popup HUB
*
* @param[in]      : bool
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_UnSupportedPopupHUB, "TraceCmd_UnSupportedPopupHUB %d", bool))
void MediaHall::TraceCmd_UnSupportedPopupHUB(bool state)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_UnSupportedPopupHUB: %d", state));
   if (state == true)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Media#MediaPlayer_Pop_Up_system#Pfo_MediaUnsupportedPopupMessageHUB"));
      if (NULL != msg)
      {
         (void)msg->Post();
      }
   }
   else if (state == false)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("Media#MediaPlayer_Pop_Up_system#Pfo_MediaUnsupportedPopupMessageHUB"));
      if (NULL != msg)
      {
         (void)msg->Post();
      }
   }
}


/**
* Trace command to initiate USBCamport screen
*
* @param[in]      : void
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_InitiateCamportUSBScreen, "TraceCmd_InitiateCamportUSBScreen"))
void MediaHall::TraceCmd_InitiateCamportUSBScreen()
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_InitiateCamportUSBScreen:"));
   g_SM_ActiveMediaSource = Enum_FI_EN_E8DTY_USB_CAMERAPORT;
   Courier::Message* pMsg = COURIER_MESSAGE_NEW(PlayAudioVideoSong)();
   if (pMsg)
   {
      pMsg->Post();
   }
}


ETG_I_CMD_DEFINE((TraceCmd_setDBDownloadStateVert, "TraceCmd_setDBDownloadStateVert %d %d", int, EnumConst::entSourceType))
void MediaHall::TraceCmd_setDBDownloadStateVert(int dbState, EnumConst::entSourceType sourceType)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_setDBDownloadState: %d", dbState));
   Courier::Message* msg = COURIER_MESSAGE_NEW(DBDownloadStateUpdateMsg)(dbState, sourceType);
   if (msg != 0 && msg->Post())
   {
   }
}


/**
* Trace command to initiate Mock for list interface
*
* @param[in]      : void
* @return         : void
*/
//ETG_I_CMD_DEFINE((TraceCmd_InitiateMockClass, "TraceCmd_InitiateMockClass %d", bool))
//void MediaHall::TraceCmd_InitiateMockClass(bool state)
//{
//	ETG_TRACE_USR4(("MediaHall::TraceCmd_InitiateMockClass: %d", state));
//	_verticalListManager->InitateInterfaceBasedOnCommand(state);
//}

ETG_I_CMD_DEFINE((TraceCmd_enableUSBCamport, "TraceCmd_enableUSBCamport %d", bool))
void MediaHall::TraceCmd_enableUSBCamport(bool bState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_enableUSBCamport: bState %d", bState));
   MediaSourceListHandler::getInstance().setCamportUsbAvailabilityInSourceList(bState);
}


ETG_I_CMD_DEFINE((TraceCmd_connectionStatusExternalUsbCamport, "TraceCmd_connectionStatusExternalUsbCamport %d", int))
void MediaHall::TraceCmd_connectionStatusExternalUsbCamport(int bState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_connectionStatusExternalUsbCamport: bState %d", bState));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ExternalCamportVideoSourceAvailabilityMsg)(bState)));
}


ETG_I_CMD_DEFINE((TraceCmd_connectionStatusExternalUsbCamportAudio, "TraceCmd_connectionStatusExternalUsbCamportAudio %d", int))
void MediaHall::TraceCmd_connectionStatusExternalUsbCamportAudio(int bState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_connectionStatusExternalUsbCamportAudio: bState %d", bState));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ExternalCamportAudioSourceAvailabilityMsg)(bState)));
}


ETG_I_CMD_DEFINE((TraceCmd_connectionStatusExternalHDMI, "TraceCmd_connectionStatusExternalHDMI %d %d", int, int))
void MediaHall::TraceCmd_connectionStatusExternalHDMI(int srcId, int bState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_connectionStatusExternalHDMI: srcId %d, bState %d", srcId, bState));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ExternalHDMISourceAvailabilityMsg)(srcId, bState)));
}


ETG_I_CMD_DEFINE((TraceCmd_SourceAllocation, "TraceCmd_SourceAllocation %d %d %d", int, int, int))
void MediaHall::TraceCmd_SourceAllocation(int srcId, int sinkId, int connectionState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_SourceAllocation: srcId %d   sinkId   %d   connectionState  %d", srcId, sinkId, connectionState));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SourceAllocationMsg)(srcId, sinkId, connectionState);
   if (msg != 0 && msg->Post())
   {
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimHK, "TraceCmd_SimHK %d %d", ETG_I_CENUM(HardKeyCodeEnum), ETG_I_CENUM(hmibase::HardKeyStateEnum)))
void MediaHall::TraceCmd_SimHK(HardKeyCodeEnum keyCode, hmibase::HardKeyStateEnum keyState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_SimHK() %d state %d", ETG_CENUM(HardKeyCodeEnum, keyCode), ETG_CENUM(hmibase::HardKeyStateEnum, keyState)));

   Courier::Message* msg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(keyCode, keyState, 0);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("Hardkey simulated"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_enableRouterUSB, "TraceCmd_enableRouterUSB %d", int))
void MediaHall::TraceCmd_enableRouterUSB(int bState)
{
   ETG_TRACE_USR4(("MediaHall::TraceCmd_enableRouterUSB: bState %d", bState));
   POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(ExternalRouterSourceAvailabilityMsg)(bState)));
}


} // namespace Core
} // namespace App
