/**************************************************************************************
 * @file         : MediaClientHandler.cpp
 * @author       : RBEI/ECG5-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 *21.09.17 Rev. 0.1 Ruchi (RBEI/ECG5)
 * 		  			Initial Revision.
 **************************************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "hall_std_if.h"
#endif
#include "Core/MediaClientHandler/MediaClientHandler.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "Core/ExternalMediaDefines.h"
#include "Core/VehicleSrvClient/VehicleSrvClient.h"
#include "ProjectBaseTypes.h"
#include "Core/SpeedLockStatusHandler/SpeedLockStatusHandler.h"
#include"Core/ExternalMediaSourceActivation/IExternalMediaSourceActivationHandler.h"
#include "Core/ExternalMediaSourceActivation/ExternalMediaSourceActivationFactory.h"

using namespace MPlay_fi_types;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::tclMediaClientHandler::
#include "trcGenProj/Header/MediaClientHandler.cpp.trc.h"
#endif

#define VIDEO_OPERATION_TIMER 5000
#define LASTSONG_TAG_RESET -1

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::SoundProperties;
using namespace ::asf::core;
#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/EnumData.h"
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#endif

namespace App {
namespace Core {


tclMediaClientHandler* tclMediaClientHandler::_mediaClientHandler = NULL;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

tU8 tclMediaClientHandler::m_u8CurrentDevType = 0;
#endif
tclMediaClientHandler& tclMediaClientHandler::getInstance()
{
   if (_mediaClientHandler == 0)
   {
      _mediaClientHandler = new tclMediaClientHandler();
   }
   return *_mediaClientHandler;
}


/************************************************************************
*FUNCTION		: tclMediaClientHandler
*DESCRIPTION	: Destructor.
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
tclMediaClientHandler::~tclMediaClientHandler()
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

   _listMap.clear();
   _sOneItemListText.clear();
   _sTwoItemListText.clear();
   _sFolderListItemText.clear();
   _mediaPlayerProxy.reset();
   _hdmiSrcAvailability.clear();

   //_metadataMap.clear();
   m_DeviceInfoMap.clear();
   m_fileInfoStr = "";
   if (_externalMediaSourceHandler != NULL)
   {
      _externalMediaSourceHandler->removeHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->removeCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->removeRouterSourceAvailabilityInfoObservor((*this));
   }

#endif
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

/************************************************************************
*FUNCTION		: tclMediaClientHandler
*DESCRIPTION	: Constructor. Initializes the member variables.
*PARAMETER		:
*PARAMETER		:
*RETURNVALUE	: NA
************************************************************************/
tclMediaClientHandler::tclMediaClientHandler() : _videoTimer(*this, VIDEO_OPERATION_TIMER),
   _externalMediaSourceHandler(NULL),
   _kdsHandler(NULL)

{
   _mediaPlayerProxy = ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy::createProxy("mediaPlayerFiPort", *this);

   _audioSourceChangeProxy = (::AudioSourceChangeProxy::createProxy("audioSourceChangePort", *this));
   _soundPropertiesProxy = (::SoundPropertiesProxy::createProxy("soundPropertiesPort", *this));

   if (_mediaPlayerProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _mediaPlayerProxy->getPortName());
   }

   if (_audioSourceChangeProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _audioSourceChangeProxy->getPortName());
   }
   if (_soundPropertiesProxy.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _soundPropertiesProxy->getPortName());
   }
   m_routerUsbConnectionState = EN_EXTERNAL_MEDIA_DISCONNECTED ;
   m_playbackstate						   = MPlay_fi_types::T_e8_MPlayState__e8PBS_PLAYING;
   m_TrackPosition                         = 0;
   m_ListHandle                            = 0;
   m_u32LastRequestedListHandle            = 0;
   (Candera::String)m_szCurrentTrackNumber = (Candera::String)"";
   m_u8PrevDevType                         = FI_EN_E8DTY_UNKNOWN;
   m_u8PrevDevTag                          = FI_EN_E8DTY_UNKNOWN;
   m_u8CurrentDevType                      = FI_EN_E8DTY_UNKNOWN;
   strCurrentAlbumArt = "";
   m_u8PrevMediaFileType = 1;
   m_u32PlayBackMode = 0;
   m_u32RepeatMode = 0;
   m_MediaType = 0;
   m_i32requestedSeekTime = 0;
   m_isSeeked = false;
   m_ConnectedDeviceCount = 0;
   m_listReqAct = 0;
   m_u32previousElapsedTime = 0;
   m_fastForwardOnGoing = false;
   m_fastBackwardOnGoing = false;
   _usbCamportAvailability = EN_EXTERNAL_MEDIA_DISCONNECTED;
   m_u8CurrentNowplayingState = ::MPlay_fi_types::T_e8_MPlayNowPlayingState__e8NP_NO_DEVICE;
   _hdmiSrcAvailability.clear();
   initialiseHdmiSrcAvailabilityMap();
   m_StrDevice_USB = (struct stDeviceConnectionInfo)
   {
      "", 0, false, false, 0, false, false, 0
   };
   m_StrDevice_USB2 = (struct stDeviceConnectionInfo)
   {
      "", 0, false, false, 0, false, false, 0
   };
   m_StrDevice_Apple_USB = (struct stDeviceConnectionInfo)
   {
      "", 0, false, false, 0, false, false, 0
   };
   m_StrDevice_BT = (struct stDeviceConnectionInfo)
   {
      "", 0, false, false, 1, false, false, 0
   };
   m_StrDevice_AUX = (struct stDeviceConnectionInfo)//smart
   {
      "", 0, true, false, 1, false, false, 0
   };
   m_deviceusbnum = 0;
   m_u8ChangedDeviceHandle = 0;
   m_currentListState = IDLE;
   m_u32CurrentListType = LISTID_MEDIA_TITLE;
   m_u32ParentListType = 0;
   m_windowStartIndex = 0;
   m_FolderFilewindowStartIndex = 0;
   m_windowSize = 50;
   _remainItem = 0;
   _sOneItemListText.clear();
   _sTwoItemListText.clear();
   _listMap.clear();
   _sFolderListItemText.clear();
   vCreateMapListTypetoMetaDataTag();
   m_fileInfoStr = "";

   m_u8NowPlayingDevType = 0;
   m_u8ActiveDevType = 0;
   m_u32ImageSize = 0;
   m_u32PrevImageSize = 0;
   m_rActivedeviceInfo = (struct stDeviceInfo)
   {
      0, 0, 0
   };
   _mediaSourcesPriority = (struct SourcePrioDetails)
   {
      0, 0
   };
   m_u32NumberOfFiles = 0;
   m_u8CurrentEntGroup = 0;
   m_u8CurrentDeviceTag = 0;
   m_u8PreviousDeviceTag = 0;
   m_u8ConnectedDeviceTag = 0;
   m_e8ConnectedDeviceType = 0;
   m_u8ActiveDeviceType = 0;
   m_u8PrevConnectType = 0;
   m_u8ConnectionType = 0;
   m_selectedDevTag = 0;
   m_selectedDevTag = 0;
   m_u32SelectedIndex = 0;
   m_u8SinkId = 2;
   m_u8DevTagMap.clear();
   m_DeviceInfoMap.clear();
   srcPriorityMap.clear();
   m_enBTDeviceType = FI_EN_E8BT_NOT_CONNECTED;
   enListStatus = EN_LISTSTATUS_INACTIVE;
   enListType = EN_LISTTYPE_FOLDERFILE;
   m_u8CurrentEntGrp = 0; //coverity fix
   m_u8ActiveRegion = MAX_REG; //coverity fix
   _flagTile = 1;//coverity fix
   _folderListInfo._listHandle = 0;
   _folderListInfo._listSize = 0;
   _folderListBrowsingHistory.clear();
   m_u8Filetype = MPlay_fi_types::T_e8_MPlayFileType__e8FT_FOLDER;
   _taActiveValue = 0;
}


#else

tclMediaClientHandler::tclMediaClientHandler()
{
   m_u32CurrentListType = LISTID_MEDIA_TITLE;
   m_deviceusbnum = 0;
   m_deviceType = 0;
}


#endif

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST


void tclMediaClientHandler::initialiseHdmiSrcAvailabilityMap()
{
   ETG_TRACE_USR1(("MediaSourceHandling::initialiseHdmiSrcAvailabilityMap is entered \n"));
   _hdmiSrcAvailability.insert(pair<int, int>(SRC_CAMERAPORT1_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
   _hdmiSrcAvailability.insert(pair<int, int>(SRC_CAMERAPORT2_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
   _hdmiSrcAvailability.insert(pair<int, int>(SRC_CAMERAPORT3_HDMI, EN_EXTERNAL_MEDIA_DISCONNECTED));
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: Startup message
*PARAMETER		: Courier::StartupMsg& oMsg
*PARAMETER		:
*RETURNVALUE	: bool
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage StartupMsg"));
   setUSB2Visibile();
   //Commented as part of bug fix 603452
   // setUSBInactive();
   return true;
}


/************************************************************************
*FUNCTION		: registerProperties
*DESCRIPTION	: Trigger property registration to mediaplayer,  called from MediaHall class
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::registerProperties entered"));
}


/************************************************************************
*FUNCTION		: deregisterProperties
*DESCRIPTION	: Trigger property registration to mediaplayer,  called from MediaHall class
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::deregisterProperties entered"));
}


/************************************************************************
*FUNCTION		: onUnavailable
*DESCRIPTION	: Method called when MediaPlayer service is not availble
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onUnavailable entered"));

   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsRelUpRegAll();
      _mediaPlayerProxy->sendPlaybackStateRelUpRegAll();
      _mediaPlayerProxy->sendNowPlayingRelUpRegAll();
      _mediaPlayerProxy->sendPlaybackModeRelUpRegAll();
      _mediaPlayerProxy->sendRepeatModeRelUpRegAll();
      _mediaPlayerProxy->sendActiveMediaDeviceRelUpRegAll();
      _mediaPlayerProxy->sendPlaytimeRelUpRegAll();
      _mediaPlayerProxy->sendFolderBrowseStructureClearedRelUpRegAll();
      _mediaPlayerProxy->sendMediaPlayerListChangeRelUpRegAll();
   }
}


/************************************************************************
*FUNCTION		: onAvailable
*DESCRIPTION	: Method called when MediaPlayer service is availble
*PARAMETER		: proxy
*PARAMETER		: stateChange - state change service for corrosponding  proxy
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onAvailable entered"));

   //Add your code here
   // Register for all the properties from the service
   if ((proxy == _mediaPlayerProxy) && (_mediaPlayerProxy.get()))
   {
      _mediaPlayerProxy->sendMediaPlayerDeviceConnectionsUpReg(*this);
      _mediaPlayerProxy->sendActiveMediaDeviceUpReg(*this);
      _mediaPlayerProxy->sendNowPlayingUpReg(*this);
      _mediaPlayerProxy->sendPlaybackModeUpReg(*this);
      _mediaPlayerProxy->sendPlaybackStateUpReg(*this);
      _mediaPlayerProxy->sendRepeatModeUpReg(*this);
      _mediaPlayerProxy->sendPlaytimeUpReg(*this);
      _mediaPlayerProxy->sendFolderBrowseStructureClearedUpReg(*this);
      _mediaPlayerProxy->sendNowPlayingGet(*this);
      _mediaPlayerProxy->sendMediaPlayerListChangeUpReg(*this);
   }
}


/************************************************************************
*FUNCTION		: onPlaybackStateStatus (SWFL-629,SWFL-630,SWFL-631,SWFL-632)
*DESCRIPTION	: update info of play/pause state
*PARAMETER		: proxy
*PARAMETER		: PlayBackState
*RETURNVALUE	: void
*SWFL			: [SWFL-2451]
************************************************************************/

void tclMediaClientHandler::onPlaybackStateStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackStateStatus >& status)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onPlaybackStateStatus"));

   m_playbackstate = status->getE8State();
   ETG_TRACE_COMP(("tclMediaClientHandler::onPlaybackStateStatus m_playbackstate :%d ", m_playbackstate));

   //Sends the play/pause state to HMI Model component
   HMIModelComponent::getInstance().vUpdatePlayBackState(status->getE8State());

   //to update the gadget in the homescreen
   HMIModelComponent::getInstance().vUpdatePlayBackStateOnHomeScreen(status->getE8State());
}


/************************************************************************
*FUNCTION		: onPlaybackStateError
*DESCRIPTION	: called if PlayBackState property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::onPlaybackStateError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackStateError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onPlaybackStateError received"));
}


/************************************************************************
*FUNCTION		: onRequestPlaybackActionResult
*DESCRIPTION	: called on getting result for PlaybackAction property from media player
*PARAMETER		: proxy
*PARAMETER		: result
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::onRequestPlaybackActionResult(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::RequestPlaybackActionResult >& result)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onRequestPlaybackActionResult received."));
}


/************************************************************************
*FUNCTION		: onRequestPlaybackActionError
*DESCRIPTION	: called if PlaybackAction property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::onRequestPlaybackActionError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::RequestPlaybackActionError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onRequestPlaybackActionError received."));
}


/************************************************************************
*FUNCTION		: onNowPlayingStatus
*DESCRIPTION	: Receives info about the currently playing track and updates the same on playscreen
*PARAMETER		: proxy
*PARAMETER		: result
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onNowPlayingStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::NowPlayingStatus >& status)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus u32ListHandle 		= %d", status->getU32ListHandle()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus e8NowPlayingState 	= %d", status->getE8NowPlayingState()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus u32Position 		= %d", status->getU32Position()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus u32Tag 				= %d", status->getU32Tag()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::sAlbumArt 	= %s", status->getOMediaObject().getSAlbumArt().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::e8CategoryType= %d", status->getOMediaObject().getE8CategoryType()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::u8DeviceTag	= %d", status->getOMediaObject().getU8DeviceTag()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::e8DeviceType 	= %d", status->getOMediaObject().getE8DeviceType()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::e8MediaType 	= %d", status->getOMediaObject().getE8MediaType()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::fileName 		= %s", status->getOMediaObject().getSFilename().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::fileFormat 	= %d", status->getOMediaObject().getE8FileFormat()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::TotalPlaytime = %d", status->getOMediaObject().getU32TotalPlaytime()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::u32Tag	 	= %d", status->getOMediaObject().getU32Tag()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::u16ExtenedTrackNumber	= %d", status->getOMediaObject().getU16ExtendedTrackNumber()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::u8TrackNumber	= %d", status->getOMediaObject().getU8TrackNumber()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaInfo.sMetaDataField1 		= %s", status->getOMediaObject().getSMetaDataField1().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaInfo.sMetaDataField2 		= %s", status->getOMediaObject().getSMetaDataField2().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::sMetaDataField3 		= %s", status->getOMediaObject().getSMetaDataField3().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus oMediaObject::sMetaDataField4 		= %s", status->getOMediaObject().getSMetaDataField4().c_str()));
   m_u8CurrentDeviceTag = status->getOMediaObject().getU8DeviceTag();
   m_u8CurrentNowplayingState = status->getE8NowPlayingState();
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus CurrentDeviceTag   %d", m_u8CurrentDeviceTag));
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus CurrentNowplayingState   %d", m_u8CurrentNowplayingState));

   int32 iApplicationState = clMetaData::getInstance().getApplicationState();
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus iApplicationState   %d", iApplicationState));

   uint16 iPrevMediaType = _mediaPlayerProxy->hasNowPlaying() ? _mediaPlayerProxy->getNowPlaying().getOMediaObject().getE8MediaType() : 0xffff;
   m_MediaType = status->getOMediaObject().getE8MediaType(); /*store if it is audio or video file type*/
   ETG_TRACE_USR4(("tclMediaClientHandler::onNowPlayingStatus iPrevMediaType %d, m_MediaType  %d ", iPrevMediaType, m_MediaType));
   if (m_MediaType == MTY_VIDEO && iPrevMediaType != m_MediaType && iApplicationState == hmibase::IN_FOREGROUND)
   {
      SpeedLockStatusHandler::getInstance().performVideoSpeedRestrictionOperation();
   }

   if (status->getOMediaObject().getE8DeviceType() == ::MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB && iPrevMediaType != m_MediaType)
   {
      StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_MEDIA_TYPE_CHANGE);
   }

   tU8 iNowPlayingDevType = status->getOMediaObject().getE8DeviceType();

   vUpdateNowPlayingMediaInfo(*status);

   StateTransitionHandler::getInstance().stateTransit_onNowPlayingStatus();
}


uint16 tclMediaClientHandler::currentPlayingMediaType()
{
   ETG_TRACE_USR4(("tclMediaClientHandler::currentPlayingMediaType MediaType %d ", m_MediaType));

   return m_MediaType;
}


void tclMediaClientHandler::vResetSeekedSliderFlag()
{
   if (m_isSeeked == true)
   {
      //resetting the value which used to identify whether the seek is performed, if it's not already resetted
      m_isSeeked = false;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler:: vResetSeekedSliderFlag m_isSeeked %d ", m_isSeeked));
}


/************************************************************************
*FUNCTION		: onNowPlayingError
*DESCRIPTION	: called if NowPlaying property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: result
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::onNowPlayingError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::NowPlayingError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onNowPlayingError received."));
}


/**************************************************************************************
* onRepeatModeStatus - called if onRepeatModeStatus method_result from media player has result.
* (SWFL-636,SWFL-637)
* @param[in] proxy
* @parm[out] status  - Repeat Mode
* @return void
***************************************************************************************/
void tclMediaClientHandler::onRepeatModeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::RepeatModeStatus >& status)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onRepeatModeStatus RepeatMode %d", status->getE8Repeat()));
   vUpdateRepeatModeStatus(status->getE8Repeat());
}


/**************************************************************************************
* onRepeatModeError - called if onRepeatModeError method_result from media player has errors.
*
* @param[in] proxy
* @parm[in] status
* @return void
***************************************************************************************/
void tclMediaClientHandler::onRepeatModeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::RepeatModeError >& /*error*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onRepeatModeError received"));
}


/************************************************************************
*FUNCTION		: vUpdateRepeatModeStatus (SWFL-636,SWFL-637)
*DESCRIPTION	: Sends the Update to the HMIModelComponent
*PARAMETER		: u32RepeatMode
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::vUpdateRepeatModeStatus(const uint32& u32RepeatMode)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateRepeatModeStatus repeatmode :%d", u32RepeatMode));

   m_u32RepeatMode = u32RepeatMode;
   HMIModelComponent::getInstance().bOnUpdateRepeatModeStatus(u32RepeatMode);
}


/********************************************************************
* onPlaybackModeStatus - called if onPlaybackRepeatMode method_result from media player has result.
* (SWFL-638)
* @param[in] proxy
* @parm[int] status
* @return void
*******************************************************************/
void tclMediaClientHandler::onPlaybackModeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackModeStatus >& status)
{
   ETG_TRACE_USR4(("onRandomModeStatus::onPlaybackModeStatus RandomMode %d", status->getE8Mode()));
   vUpdatePlayBackModeStatus(status->getE8Mode());
}


/************************************************************************
* onPlaybackRepeatModeError - called if onPlaybackRepeatMode method_result from media player has errors.
*
* @param[in] proxy
* @parm[in] status
* @return void
*************************************************************************/
void tclMediaClientHandler::onPlaybackModeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaybackModeError >& /*error*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onPlaybackModeError received"));
}


/*********************************************************************
* onGetMediaObjectAlbumArtInfoResult - called if AlbumArtInfo method_result from media player has result.
*(SYSFL-1136 , SYSFL-1140)
* @param[in] proxy
* @parm[in] status
* @return void
*********************************************************************/
void tclMediaClientHandler::onGetMediaObjectAlbumArtInfoResult(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::GetMediaObjectAlbumArtInfoResult >& result)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onGetMediaObjectAlbumArtInfoResult received"));

   uint32 albumArtWidth = ALBUMARTWIDTH;
   uint32 albumArtHeight = ALBUMARTHEIGHT;

   ETG_TRACE_USR4(("tclMediaClientHandler::onGetMediaObjectAlbumArtInfoResult PhotoSize = %d", result->getU32PhotoSize()));

   if ((result->getU32PhotoSize() > 0) /*&& (result->getSMIMEImageSubtype().c_str() != "")*/)
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::onGetMediaObjectAlbumArtInfoResult SMIMEImageSubtype 	= %s", result->getSMIMEImageSubtype().c_str()));
      //strcpy(m_sMIMEImageSubtype, result->getSMIMEImageSubtype().c_str());
      strncpy(m_sMIMEImageSubtype, result->getSMIMEImageSubtype().c_str(), result->getSMIMEImageSubtype().length());

      m_u32ImageSize = result->getU32PhotoSize();

      //Send a method Start for AlbumArt
      _mediaPlayerProxy->sendGetMediaObjectAlbumArtStart(*this, strAlbumArt, albumArtWidth, albumArtHeight);
   }
}


/************************************************************************
*FUNCTION		: onGetMediaObjectAlbumArtInfoError
*DESCRIPTION	: called if AlbumArtInfo property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: result
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onGetMediaObjectAlbumArtInfoError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::GetMediaObjectAlbumArtInfoError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onGetMediaObjectAlbumArtInfoError  received."));
   HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforInternalMediaSources();
}


/*******************************************************************
* onGetMediaObjectAlbumArtResult - called if AlbumArt method_result from media player has result.
*(SYSFL-1136 , SYSFL-1140)
* @param[in] proxy
* @parm[in] status
* @return void
******************************************************************/
void tclMediaClientHandler::onGetMediaObjectAlbumArtResult(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::GetMediaObjectAlbumArtResult >& result)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onGetMediaObjectAlbumArtResult AlbumArt datasize = %d", result->getOImageData().size()));
   m_ImageData = result->getOImageData();
   vUpdateNowPlayingAlbumArt(m_u8ActiveDevType, m_sMIMEImageSubtype, m_u32ImageSize, &(m_ImageData[0]));
}


/************************************************************************
*FUNCTION		: onGetMediaObjectAlbumArtError
*DESCRIPTION	: called if AlbumArt property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: result
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onGetMediaObjectAlbumArtError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::GetMediaObjectAlbumArtError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onGetMediaObjectAlbumArtError received."));
}


/************************************************************************
*FUNCTION:		vUpdateNowPlayingAlbumArt
*DESCRIPTION:	Trigger based on Media service Update and send data to PL
*PARAMETER:
*(SYSFL-1136 , SYSFL-1140)
*
************************************************************************/
void tclMediaClientHandler::vUpdateNowPlayingAlbumArt(tU8 e8MediaDeviceType, const tChar* sMIMEImageSubtype, tU32 u32ImageSize, tU8* vImageData)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingAlbumArt ImageSize =%d", u32ImageSize));
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingAlbumArt PrevImageSize =%d", m_u32PrevImageSize));

   //Image is Valid
   if ((NULL != vImageData) && (u32ImageSize != 0))
   {
      if (m_u32PrevImageSize != u32ImageSize)
      {
         //Update the AlbumArt Image to the HMIModelComponent

         m_u32PrevImageSize = u32ImageSize;
      }
      HMIModelComponent::getInstance().bOnUpdateAlbumArt(m_ImageData, u32ImageSize);
   }
   else
   {
      //Update the default AlbumArt Image to the HMIModelComponent
      HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforInternalMediaSources();
   }
}


/************************************************************************
*FUNCTION		: vUpdatePlayBackModeStatus (SWFL-638)
*DESCRIPTION	: Sends the Update to the HMIModelComponent
*PARAMETER		: u32PlayBackMode
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::vUpdatePlayBackModeStatus(const uint32& u32PlayBackMode)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdatePlayBackModeStatus playbackmode :%d", u32PlayBackMode));

   m_u32PlayBackMode = u32PlayBackMode;
   HMIModelComponent::getInstance().bOnUpdatePlayBackModeStatus(u32PlayBackMode);
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: called on receiving the MediaOperationMsgReq courier message
*PARAMETER		: MediaOperationMsgReq& oMsg
*PARAMETER		:
*RETURNVALUE	: bool
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const MediaOperationMsgReq& oMsg)
{
   tU8 u8OperationType = oMsg.GetOperationType();
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq OperationType: %d", u8OperationType));

   tCU8 u8NextPrevSkipCount = DEFAULT_SKIP_COUNT;

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq activeRegion:  %d ", u8activeRegion));
   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq  activeSource  %d ", iActiveSource));
      ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq  playbackstate  %d ", m_playbackstate));
      ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq  RepeatMode  %d ", m_u32RepeatMode));
      ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq  PlayBackMode  %d ", m_u32PlayBackMode));

      if ((iActiveSource == INTERNALSOURCE && (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A)))
      {
         switch (u8OperationType)
         {
            case EN_MEDIA_PLAY:
            case EN_MEDIA_PAUSE:
            {
               if (m_playbackstate == MPlay_fi_types::T_e8_MPlayState__e8PBS_PLAYING)
               {
                  vSetPlaybackAction(FI_EN_E8PBA_PAUSE, u8NextPrevSkipCount);
               }
               else
               {
                  vSetPlaybackAction(FI_EN_E8PBA_PLAY, u8NextPrevSkipCount);
               }
            }
            break;
            case EN_MEDIA_REPEAT_OFF:
            case EN_MEDIA_REPEAT_TRACK:
            case EN_MEDIA_REPEAT_LIST:
            {
               if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE)
               {
                  _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE);
               }
               else if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE)
               {
                  _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST); //Rep List
               }
               else if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST)
               {
                  _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE); //Rep Off
               }
               else
               {
                  _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE);
               }
            }
            break;

            case EN_MEDIA_SHUFFLE_OFF:
            case EN_MEDIA_SHUFFLE_LIST:
            {
               if (m_u32PlayBackMode == MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL)
               {
                  _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM);//Sh List
               }
               else if (m_u32PlayBackMode == MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM)
               {
                  _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL); //Sh Off
               }
               else
               {
                  ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq OperationType: Invalid shuffle option"));
               }
            }
            break;

            case EN_MEDIA_FASTFORWARD:
            {
               vSetPlaybackAction(FI_EN_E8PBA_FFWD_START, u8NextPrevSkipCount);
            }
            break;

            case EN_MEDIA_FASTREWIND:
            {
               vSetPlaybackAction(FI_EN_E8PBA_FREV_START, u8NextPrevSkipCount);
            }
            break;

            case EN_MEDIA_FASTFORWARD_STOP:
            {
               vSetPlaybackAction(FI_EN_E8PBA_FFWD_STOP, u8NextPrevSkipCount);
            }
            break;

            case EN_MEDIA_FASTREWIND_STOP:
            {
               vSetPlaybackAction(FI_EN_E8PBA_FREV_STOP, u8NextPrevSkipCount);
            }
            break;

            case EN_MEDIA_NEXT:
            {
               vSetPlaybackAction(FI_EN_E8PBA_NEXT, u8NextPrevSkipCount);
            }
            break;
            case EN_MEDIA_PREV:
            {
               vSetPlaybackAction(FI_EN_E8PBA_PREV, u8NextPrevSkipCount);
            }
            break;
            default:
            {
               ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:MediaOperationMsgReq OperationType: Default case option"));
            }
            break;
         }
      }
   }

   return false;
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: called on receiving the MediaOperationMsgReq courier message
*PARAMETER		: MediaOperationMsgReq& oMsg
*PARAMETER		:
*RETURNVALUE	: bool
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const MediaBTOperationMsgReq& oMsg)
{
   tU8 u8OperationType = oMsg.GetOperationType();
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq OperationType: %d", u8OperationType));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq  playbackstate  %d ", m_playbackstate));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq  RepeatMode  %d ", m_u32RepeatMode));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq  PlayBackMode  %d ", m_u32PlayBackMode));

   tCU8 u8NextPrevSkipCount = DEFAULT_SKIP_COUNT;

   switch (u8OperationType)
   {
      case EN_MEDIA_PLAY:
      case EN_MEDIA_PAUSE:
      {
         if (m_playbackstate == MPlay_fi_types::T_e8_MPlayState__e8PBS_PLAYING)
         {
            vSetPlaybackAction(FI_EN_E8PBA_PAUSE, u8NextPrevSkipCount);
         }
         else
         {
            vSetPlaybackAction(FI_EN_E8PBA_PLAY, u8NextPrevSkipCount);
         }
      }
      break;
      case EN_MEDIA_REPEAT_OFF:
      case EN_MEDIA_REPEAT_TRACK:
      case EN_MEDIA_REPEAT_LIST:
      {
         if (m_StrDevice_BT.bRepeatSupported)
         {
            if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE)
            {
               _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE);
            }
            else if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE)
            {
               _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST); //Rep List
            }
            else if (m_u32RepeatMode == MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST)
            {
               _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE); //Rep Off
            }
            else
            {
               _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE);
            }
         }
         else
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq: Repeat Not Supported"));
         }
      }
      break;
      case EN_MEDIA_SHUFFLE_OFF:
      case EN_MEDIA_SHUFFLE_LIST:
      {
         if (m_StrDevice_BT.bShuffleSupported)
         {
            if (m_u32PlayBackMode == MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL)
            {
               _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM);//Sh List
            }
            else if (m_u32PlayBackMode == MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM)
            {
               _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL); //Sh Off
            }
            else
            {
               ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq  Invalid shuffle option"));
            }
         }
         else
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq Shuffle Not supported"));
         }
      }
      break;
      case EN_MEDIA_FASTFORWARD:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FFWD_START, u8NextPrevSkipCount);
      }
      break;
      case EN_MEDIA_FASTREWIND:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FREV_START, u8NextPrevSkipCount);
      }
      break;
      case EN_MEDIA_FASTFORWARD_STOP:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FFWD_STOP, u8NextPrevSkipCount);
      }
      break;
      case EN_MEDIA_FASTREWIND_STOP:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FREV_STOP, u8NextPrevSkipCount);
      }
      break;
      case EN_MEDIA_NEXT:
      {
         vSetPlaybackAction(FI_EN_E8PBA_NEXT, u8NextPrevSkipCount);
      }
      break;
      case EN_MEDIA_PREV:
      {
         vSetPlaybackAction(FI_EN_E8PBA_PREV, u8NextPrevSkipCount);
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:MediaBTOperationMsgReq Default case option"));
      }
      break;
   }
   return true;
}


/************************************************************************
*FUNCTION		: vSetPlaybackAction
*DESCRIPTION	: To send the methodstart to the service
*PARAMETER		: tU32 u32ActionType
*PARAMETER		: tU8 u8NextPrevSkipCount
*RETURNVALUE	: void
************************************************************************/

void tclMediaClientHandler::vSetPlaybackAction(tU32 u32ActionType, tU8 u8NextPrevSkipCount)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vSetPlaybackAction ActionType: %d,NextPrevSkipCount:%d", u32ActionType, u8NextPrevSkipCount));
   switch (u32ActionType)
   {
      case FI_EN_E8PBA_PAUSE:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_PAUSE, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_PLAY:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_PLAY, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_FFWD_START:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_FFWD_START, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_FREV_START:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_FREV_START, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_FFWD_STOP:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_FFWD_STOP, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_FREV_STOP:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_FREV_STOP, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_NEXT:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_NEXT, u8NextPrevSkipCount);
      }
      break;
      case FI_EN_E8PBA_PREV:
      {
         _mediaPlayerProxy->sendRequestPlaybackActionStart(*this, ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_PREV, u8NextPrevSkipCount);
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("tclMediaClientHandler::vSetPlaybackAction: Default case option"));
      }
      break;
   }
}


/************************************************************************
*FUNCTION		: vUpdateMetadtaInfo
*DESCRIPTION	: To update the metadata informations
*PARAMETER		: mplay_MediaPlayer_FI::NowPlayingStatus &oMediaInfo
*PARAMETER		:
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::vUpdateMetadtaInfo(const ::mplay_MediaPlayer_FI::NowPlayingStatus& oMediaInfo)
{
   tU8 iCategoryType = oMediaInfo.getOMediaObject().getE8CategoryType();
   tU32 u32Positions = 0;
   std::string szTotalTime;
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateMetadtaInfoiCategoryTyp %d ", iCategoryType));

   switch (iCategoryType)
   {
      case MPlay_fi_types::T_e8_MPlayCategoryType__e8CTY_EPISODE:
      {
         MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].AlbumName = oMediaInfo.getOMediaObject().getSMetaDataField1();
         MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TitleName = oMediaInfo.getOMediaObject().getSMetaDataField2();
         break;
      }
      case MPlay_fi_types::T_e8_MPlayCategoryType__e8CTY_SONG:
      {
         MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].AlbumName = oMediaInfo.getOMediaObject().getSMetaDataField3();
         MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TitleName = oMediaInfo.getOMediaObject().getSMetaDataField4();
         break;
      }
      default:
      {
         break;
      }
   }

   /*Handle for Current song Hiererchy Info <End>*/
   memset(m_szCurrentTrackNumber, '\0', MEDIA_C_ETA_BUFFER_SIZE);
   ++u32Positions;
   snprintf(m_szCurrentTrackNumber, MEDIA_C_ETA_BUFFER_SIZE, "%d", u32Positions);
   clMetaData::getInstance().szOnConvertPlayTime(szTotalTime, clMetaData::getInstance().getTotalTime()); //temp, later will be changed

   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].ArtistName = oMediaInfo.getOMediaObject().getSMetaDataField2();
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].MediaCategoryType = iCategoryType;
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].MediaFileType = oMediaInfo.getOMediaObject().getE8MediaType();
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].CurrentTrackNumber = m_szCurrentTrackNumber;
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].CurrentPlayingDeviceType = oMediaInfo.getOMediaObject().getE8DeviceType();
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].TotalTime = oMediaInfo.getOMediaObject().getU32TotalPlaytime();
   MediaUtils::getInstance()._metadataMap[REGION_COCKPIT].LastDeviceType = oMediaInfo.getOMediaObject().getE8DeviceType();
}


/************************************************************************
*FUNCTION		: vUpdateNowPlayingMediaInfo(SYSFL-1126)
*DESCRIPTION	: Sends the currently playing  media info to HMI Model component
*PARAMETER		: mplay_MediaPlayer_FI::NowPlayingStatus &oMediaInfo
*PARAMETER		:
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::vUpdateNowPlayingMediaInfo(const ::mplay_MediaPlayer_FI::NowPlayingStatus& oMediaInfo)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo (oMediaInfo.oMediaObject.e8DeviceType.enType) : %d", (oMediaInfo.getOMediaObject().getE8DeviceType())));
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo m_u8CurrentNowplayingState  : %d", m_u8CurrentNowplayingState));
   m_u8NowPlayingDevType = oMediaInfo.getOMediaObject().getE8DeviceType();
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo NowPlayingDevType : %d", m_u8NowPlayingDevType));
   if (((oMediaInfo.getOMediaObject().getE8DeviceType()) != 0))
   {
      m_oMediaInfo = oMediaInfo;	// Deep copy of object from Service
      if (m_oMediaInfo.getOMediaObject().getE8MediaType() != ::MPlay_fi_types::T_e8_MPlayMediaType__e8MTY_UNKNOWN)
      {
         g_SM_MediaFileType = (m_oMediaInfo.getOMediaObject().getE8MediaType() == MTY_VIDEO) ? SM_MEDIA_VIDEO_FILE : SM_MEDIA_AUDIO_FILE;
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo NowPlayingDevType g_SM_MediaFileType : %d", g_SM_MediaFileType));
      }
      tU8 u8CurrentPlayingDeviceType = m_oMediaInfo.getOMediaObject().getE8DeviceType();
      clMetaData::getInstance().setSourceName(szOnGetDeviceName(u8CurrentPlayingDeviceType));
      clMetaData::getInstance().setMediaFileType(m_oMediaInfo.getOMediaObject().getE8MediaType());

      if (oMediaInfo.getOMediaObject().getE8MediaType() == MTY_PODCAST)
      {
         HMIModelComponent::getInstance().vUpdateNowplayIcons(PODCAST_NOWPLAY_ICON_UPDATE);
      }
      else if (oMediaInfo.getOMediaObject().getE8MediaType() == MTY_AUDIOBOOK)
      {
         HMIModelComponent::getInstance().vUpdateNowplayIcons(AUDIOBOOK_NOWPLAY_ICON_UPDATE);
      }
      else
      {
         ETG_TRACE_USR1(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo:MTY_MUSIC"));

         HMIModelComponent::getInstance().vUpdateNowplayIcons(IPOD_NOWPLAY_ICON_UPDATE);
      }

      clMetaData::getInstance().setTotalTime(m_oMediaInfo.getOMediaObject().getU32TotalPlaytime());
      clMetaData::getInstance().setListHandle(m_oMediaInfo.getU32ListHandle());

      ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo Filename= %s", clMetaData::getInstance().getFilename()));
      ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateNowPlayingMediaInfo oMediaInfo.u32Position : %d, m_TrackPosition %d,oMediaInfo.u32ListHandle : %d, m_ListHandle %d"
                      , oMediaInfo.getU32Position(), m_TrackPosition, m_oMediaInfo.getU32ListHandle(), m_ListHandle));

      int32 u32MediaFileType = clMetaData::getInstance().getMediaFileType();

      vUpdateMetadtaInfo(oMediaInfo);

      //To update the gadget on the homescreen.

      HMIModelComponent::getInstance().bOnUpdateNowPlayingStatus();
      HMIModelComponent::getInstance().bOnUpdateNowPlayingStatusOnHomeScreen();

      vAlbumArtUpdate(oMediaInfo);

      m_u8PrevMediaFileType = clMetaData::getInstance().getMediaFileType();

      //Evenif the current status is mute, if user load any file from browselist and check the nowplaying screen, the mute header get replced by active source name.
      //To avoid this, checked the current mute status and updated the header text accordingly
      HMIModelComponent::getInstance().bOnUpdateHeaderMuteStatus();
      HMIModelComponent::getInstance().vUpdatePlayBackState(m_playbackstate);
      HMIModelComponent::getInstance().vUpdatePlayBackStateOnHomeScreen(m_playbackstate);
      HMIModelComponent::getInstance().bOnUpdateRepeatModeStatus(m_u32RepeatMode);
      HMIModelComponent::getInstance().bOnUpdatePlayBackModeStatus(m_u32PlayBackMode);
   }
}


void tclMediaClientHandler::GetMetadataInfo(tU8 region, stNowPlayingInfo& NowPlayingInfo)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::GetMetadataInfo region = %d", region));

   if (region == 0)
   {
      //cabin A
      NowPlayingInfo.ArtistName = MediaUtils::getInstance()._metadataMap[region].ArtistName;
   }
   else if (region == 2)//cockpit
   {
      NowPlayingInfo.ArtistName = MediaUtils::getInstance()._metadataMap[region].ArtistName;
      NowPlayingInfo.TitleName = MediaUtils::getInstance()._metadataMap[region].TitleName;
      NowPlayingInfo.AlbumName = MediaUtils::getInstance()._metadataMap[region].AlbumName;
      NowPlayingInfo.MediaFileType = MediaUtils::getInstance()._metadataMap[region].MediaFileType;
      NowPlayingInfo.CurrentTrackNumber = MediaUtils::getInstance()._metadataMap[region].CurrentTrackNumber;
      NowPlayingInfo.CurrentPlayingDeviceType = MediaUtils::getInstance()._metadataMap[2].CurrentPlayingDeviceType;
      NowPlayingInfo.TotalTime = MediaUtils::getInstance()._metadataMap[region].TotalTime;
      NowPlayingInfo.LastDeviceType = MediaUtils::getInstance()._metadataMap[region].LastDeviceType;
   }
   else
   {
   }
}


/*********************************************************************************************
* Function 	    : vAlbumArtUpdate
* Description 	: updating album art of the song - if album art doesn't exist, update with default album art
* Parameters 	: const ::mplay_MediaPlayer_FI::NowPlayingStatus &oMediaInfo
* Return value	: void
**********************************************************************************************/
void tclMediaClientHandler::vAlbumArtUpdate(const ::mplay_MediaPlayer_FI::NowPlayingStatus& oMediaInfo)
{
   strCurrentAlbumArt.clear();
   ETG_TRACE_USR4(("tclMediaClientHandler::vAlbumArtUpdate AlbumArt Path: %s", m_oMediaInfo.getOMediaObject().getSAlbumArt().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::vAlbumArtUpdate Song being played : %s", m_oMediaInfo.getOMediaObject().getSFilename().c_str()));
   ETG_TRACE_USR4(("tclMediaClientHandler::vAlbumArtUpdate strCurrentAlbumArt !strAlbumArt.empty() %s", strCurrentAlbumArt.c_str()));

   if (m_oMediaInfo.getOMediaObject().getSAlbumArt() != "")
   {
      strAlbumArt = m_oMediaInfo.getOMediaObject().getSAlbumArt();
      MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].strAlbumArt = strAlbumArt;
   }
   else
   {
      strAlbumArt.clear();
      strCurrentAlbumArt.clear();
      MediaUtils::getInstance().m_ActiveDeviceDataMap[REGION_COCKPIT].strAlbumArt = strAlbumArt;
      HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforInternalMediaSources();
   }

   if ((!strAlbumArt.empty()) && ((strCurrentAlbumArt.compare(strAlbumArt) != 0)))
   {
      strCurrentAlbumArt = strAlbumArt;
      _mediaPlayerProxy->sendGetMediaObjectAlbumArtInfoStart(*this, strAlbumArt, ALBUMARTWIDTH, ALBUMARTHEIGHT);
   }
}


/*********************************************************************************************
* Function 	    : UpdateAlbumArt
* Description 	: updating album art of the song - function used to update albumart of cockpit when camport is active in cabin
* Parameters 	: std::string stAlbumArt
* Return value	: void
**********************************************************************************************/
void tclMediaClientHandler::UpdateAlbumArt()
{
   /* uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
    ETG_TRACE_USR4(("tclMediaClientHandler::UpdateAlbumArt u8activeRegion  %d ", u8activeRegion));
    MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
    if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
    {
       int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
       ETG_TRACE_USR4(("tclMediaClientHandler::UpdateAlbumArt iActiveSource  %d ", iActiveSource));

       if ((iActiveSource == INTERNALSOURCE && (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A)))
       {*/
   if (!strAlbumArt.empty())
   {
      HMIModelComponent::getInstance().bOnUpdateAlbumArt(m_ImageData, m_u32ImageSize);
   }
   else
   {
      HMIModelComponent::getInstance().bOnUpdateAlbumArtDefaultforInternalMediaSources();
   }
}


/*********************************************************************************************
* Function 	    : szOnGetDeviceName
* Description 	:
* Parameters 	: tU32 uDeviceType
* Return value	: std::string
**********************************************************************************************/
std::string tclMediaClientHandler::szOnGetDeviceName(tU32 uDeviceType)const
{
   ETG_TRACE_USR4(("tclMediaClientHandler::szOnGetDeviceName devicetype :%d", uDeviceType));
   std::string pszDeviceName = "";
   switch (uDeviceType)
   {
      case FI_EN_E8DTY_USB:
         pszDeviceName = USB_NAME;
         break;
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
         pszDeviceName = IPOD;
         break;
      case FI_EN_E8DTY_BLUETOOTH:
         pszDeviceName = BLUETOOTH;
         break;
      case FI_EN_E8DTY_AUX:
         pszDeviceName = AUX;
         break;
      default:
         break;
   }

   return pszDeviceName;
}


/* ***********************************************************
* Function 		: vMediaPlayerDeviceConnections_Status
(SYSFL-1003, SYSFL-1004 , SYSFL-1005 , SYSFL-1009 , SYSFL-1010 , SYSFL-1011)
* Description 	: update info of all mediaplayer devices connection (e.g. connection, number of file, device type, device tag, indexing state etc.
* Parameters 	: proxy, status
* Return value	: void
* ***********************************************************/
void tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus >& status)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus"));
   int8 i;
   int listsize;

   ::MPlay_fi_types::T_MPlayDeviceInfo oDeviceInfo = status->getODeviceInfo();
   MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = oDeviceInfo.begin();
   std::string sysPath = "";

   ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus device list size= %d", oDeviceInfo.size()));
   for (i = 0; itr != oDeviceInfo.end(); ++itr, ++i)
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus DeviceTag = %d", itr->getU8DeviceTag()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus DeviceName = %s", itr->getSDeviceName().c_str()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus DeviceType  = %d", itr->getE8DeviceType()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus sDeviceSerialNumber = %s", itr->getSDeviceSerialNumber().c_str()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus bDeviceConnected = %d", itr->getBDeviceConnected()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus e8DeviceIndexedState= %d", itr->getE8DeviceIndexedState()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus bDeviceActiveSource = %d", itr->getBDeviceActiveSource()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus No of files = %d", itr->getU32NumberOfFiles()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus DeviceControlSupport = %d", itr->getE8DeviceControlSupport()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus ShuffleSupported  = %d", itr->getBShuffleSupported()));
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus Repeat Supported  = %d", itr->getBRepeatSupported()));
      sysPath = itr->getSDeviceSysPath();
      ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsStatus sysPath= %s", sysPath.c_str()));
   }

   vUpdateConnectedDeviceInfo(*status);
}


/* ***********************************************************
* Function 		: onMediaPlayerDeviceConnectionsError
* Description 	: called if DeviceConnection property is updated with error from media player
* Parameters 	: proxy, error
* Return value	: void
* ***********************************************************/
void tclMediaClientHandler::onMediaPlayerDeviceConnectionsError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
      const boost::shared_ptr< ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsError >& /*error*/)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onMediaPlayerDeviceConnectionsError received."));
}


void tclMediaClientHandler::setActiveBTSource()
{
   m_StrDevice_Apple_USB.bIsActive = FALSE;
   m_StrDevice_USB.bIsActive = FALSE;
   m_StrDevice_USB2.bIsActive = FALSE;
   m_StrDevice_BT.bIsActive = TRUE;
   m_StrDevice_AUX.bIsActive = FALSE;
}


void tclMediaClientHandler::setActiveAUXSource()
{
   m_StrDevice_Apple_USB.bIsActive = FALSE;
   m_StrDevice_USB.bIsActive = FALSE;
   m_StrDevice_USB2.bIsActive = FALSE;
   m_StrDevice_BT.bIsActive = FALSE;
   m_StrDevice_AUX.bIsActive = TRUE;
}


int tclMediaClientHandler::GetActiveUSBnum()
{
   ETG_TRACE_USR4(("tclMediaClientHandler::GetActiveUSBnum usbnum %d ", m_deviceusbnum));
   return m_deviceusbnum;
}


/* ***********************************************************
* Function 		: vUpdateConnectedDeviceInfo
* Description 	: Sends the coonection status to HMI Model component
* Parameters 	: MediaPlayerDeviceConnectionsStatus &oDeviceInfoDataObject
* Return value	: void
* ***********************************************************/
void tclMediaClientHandler::vUpdateConnectedDeviceInfo(const ::mplay_MediaPlayer_FI::MediaPlayerDeviceConnectionsStatus& oDeviceInfoDataObject)
{
   ETG_TRACE_USR1(("tclMediaClientHandler::vUpdateConnectedDeviceInfo"));
   m_oDeviceConnectionStatus = oDeviceInfoDataObject;

   uint8 iCurrListIndex = 0;
   std::string _SysPath = "";

   int iGuiStartupVal = clMetaData::getInstance().getGuiStartupStatus();
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iGuiStartupVal  %d ", iGuiStartupVal));

   int iSubsrcid = MediaSourceHandling::getInstance().GetLastMediaSubSrcId();
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iSubsrcid  %d ", iSubsrcid));

   // New device list update info loop
   for (; iCurrListIndex < m_oDeviceConnectionStatus.getODeviceInfo().size(); ++iCurrListIndex)
   {
      //Identify the device type
      switch (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceType())
      {
         //USB card Device tag connection and disconnection status change
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_USB:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_MTP:
         {
            if (1 == _kdsHandler->getHubKDSValue())
            {
               m_StrDevice_USB.devNum = 0;
               vOnDeviceConnection(&m_StrDevice_USB, iCurrListIndex);
            }
            else
            {
               _SysPath = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getSDeviceSysPath();

               uint8 deviceTag = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag();

               ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateDeviceConnection hub syspath %s", _SysPath.c_str()));
               if (_SysPath != "")
               {
                  if (isDeviceConnectedInEvenPortNumber(_SysPath))
                  {
                     m_StrDevice_USB2.devNum = USB_TWO;
                     m_StrDevice_USB2.u8DeviceTag = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag();
                     vOnDeviceConnection(&m_StrDevice_USB2, iCurrListIndex);
                     m_DeviceInfoMap.insert(pair<int, stDeviceConnectionInfo>(deviceTag, m_StrDevice_USB2));
                  }
                  else
                  {
                     m_StrDevice_USB.devNum = USB_ONE;
                     m_StrDevice_USB.u8DeviceTag = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag();
                     vOnDeviceConnection(&m_StrDevice_USB, iCurrListIndex);
                     m_DeviceInfoMap.insert(pair<int, stDeviceConnectionInfo>(deviceTag, m_StrDevice_USB));
                  }
                  ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo Device Connected In Even Port and device tag is %d ", m_StrDevice_USB2.u8DeviceTag));
               }
               else
               {
                  ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo SysPath is blank search device %d ", deviceTag));
                  itr = m_DeviceInfoMap.find(deviceTag);

                  if (itr == m_DeviceInfoMap.end())
                  {
                     ETG_TRACE_USR1(("tclMediaClientHandler::vUpdateConnectedDeviceInfo Key-value pair not present in map"));
                  }
                  else
                  {
                     ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo SysPath is blank Device Tag %d ", itr->first));
                     ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo SysPath is blank second device number %d ", itr->second.devNum));
                     if (itr->second.devNum == 2)
                     {
                        m_DeviceInfoMap.erase(itr); //delete the entry of usb 2 structure after the device has been disconnected
                        vOnDeviceConnection(&m_StrDevice_USB2, iCurrListIndex); //send disconnection of usb2
                        m_StrDevice_USB2 = (struct stDeviceConnectionInfo)
                        {
                           "", 0, false, false, 0, false, false, 0
                        };
                     }
                     else
                     {
                        m_DeviceInfoMap.erase(itr);
                        vOnDeviceConnection(&m_StrDevice_USB, iCurrListIndex); //send disconnection of usb1
                        m_StrDevice_USB = (struct stDeviceConnectionInfo)
                        {
                           "", 0, false, false, 0, false, false, 0
                        };
                     }
                  }
               }
            }
            break;
         }
         // Apple Device tag connection and disconnection status change
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPOD:
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_IPHONE:
         {
            // Checking Connection Type
            if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8ConnectionType() ==
                  (MPlay_fi_types::T_e8_MPlayConnectionType__e8DCT_USB))//Mixed enum - Coverity issue solution
            {
               vOnDeviceConnection(&m_StrDevice_Apple_USB, iCurrListIndex);
            }
            // Connection Status Update
            // Check BT device connected status prior to the structure update to avoid multiple updates
            break;
         }
         case MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_BLUETOOTH:
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo:: shuffle supported = %d ", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBShuffleSupported()));
            ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo:: repeat supported = %d ", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBRepeatSupported()));
            ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo:: device supported = %d ", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceControlSupport()));

            if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected())
            {
               if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBShuffleSupported() == true)
               {
                  clMetaData::getInstance().setShuffleSupported(true);
               }
               else
               {
                  clMetaData::getInstance().setShuffleSupported(false);
               }
               if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBRepeatSupported() == true)
               {
                  clMetaData::getInstance().setRepeatSupported(true);
               }
               else
               {
                  clMetaData::getInstance().setRepeatSupported(false);
               }
               if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceControlSupport() == 0)
               {
                  clMetaData::getInstance().setBrowseSupported(true);
               }
               else
               {
                  clMetaData::getInstance().setBrowseSupported(false);
               }
               ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo:BLUETOOTH:%d", m_StrDevice_BT.bIsActive));
            }
            vOnDeviceConnection(&m_StrDevice_BT, iCurrListIndex);
            break;
         }
         default:
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::vUpdateConnectedDeviceInfo::InDevice"));
            break;
         }
      }// End of switch

      // Connection Status Update
      if ((m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected() == true) && (iGuiStartupVal == true)
            && (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag() == iSubsrcid))
      {
         clMetaData::getInstance().setGuiStartupStatus(false);
         int iDeviceTag = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag();
         int iDevicetype = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceType();
         int iSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iSrcId %d", iSrcId));
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iDeviceTag  %d", iDeviceTag));
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iDevicetype  %d", iDevicetype));

         uint8 iactiveMediaSource = MediaSourceHandling::getInstance().GetActiveMediaSource(iSrcId, iDevicetype);
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateConnectedDeviceInfo iDeviceType %d", iactiveMediaSource));
         HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(iactiveMediaSource);

         HMIModelComponent::getInstance().updateMediaSourceInfoInGadget(iDevicetype);

         HMIModelComponent::getInstance().vSourceButtonHighlight(iDevicetype, iDeviceTag);
      }
   }// End of For Loop
   ETG_TRACE_USR1(("tclMediaClientHandler::vUpdateConnectedDeviceInfo: exit"));
}


int tclMediaClientHandler::getUSB1DeviceTag()
{
   int deviceTagUSB = 0;
   if (m_StrDevice_USB.bConnectedStatus == true)
   {
      deviceTagUSB = m_StrDevice_USB.u8DeviceTag;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceTag m_StrDevice_USB.bConnectedStatus  %d ,deviceTagUSB %d", m_StrDevice_USB.bConnectedStatus, deviceTagUSB));
   return deviceTagUSB;
}


int tclMediaClientHandler::getUSB2DeviceTag()
{
   int deviceTagUSB2 = 0;
   if (m_StrDevice_USB2.bConnectedStatus == true)
   {
      deviceTagUSB2 = m_StrDevice_USB2.u8DeviceTag;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceTag m_StrDevice_USB2.bConnectedStatus  %d ,deviceTagUSB2 %d", m_StrDevice_USB2.bConnectedStatus, deviceTagUSB2));
   return deviceTagUSB2;
}


int tclMediaClientHandler::getIPODDeviceTag()
{
   int deviceTagIPOD = 0;
   if (m_StrDevice_Apple_USB.bConnectedStatus == true)
   {
      deviceTagIPOD = m_StrDevice_Apple_USB.u8DeviceTag;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceTag m_StrDevice_Apple_USB.bConnectedStatus  %d ,deviceTagIPOD %d", m_StrDevice_Apple_USB.bConnectedStatus, deviceTagIPOD));
   return deviceTagIPOD;
}


int tclMediaClientHandler::getBTDeviceTag()
{
   int deviceTagBT = 0;
   if (m_StrDevice_BT.bConnectedStatus == true)
   {
      deviceTagBT = m_StrDevice_BT.u8DeviceTag;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceTag m_StrDevice_BT.bConnectedStatus  %d ,device Tag BT %d", m_StrDevice_BT.bConnectedStatus, deviceTagBT));
   return deviceTagBT;
}


/**
* SetActiveSource - Gets called on press of applfly inn media button by passing proper subsrcid to identify which device is active
This function will help to give the available media source by matching with the subsrc id,correspoding g_SM_ActiveMediaSource is set.
No media device is connected then we will make AUX as default source.
* @param[in] int
* @return void
*/
void  tclMediaClientHandler::SetActiveSource(int subsrcid, int srcid)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::SetActiveSource subsourceid %d ", subsrcid));
   ETG_TRACE_USR4(("tclMediaClientHandler::SetActiveSource sourceid %d ", srcid));

   int iCurrentDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(subsrcid);
   uint8 activeMediaSource = MediaSourceHandling::getInstance().GetActiveMediaSource(srcid, iCurrentDeviceType);
   ETG_TRACE_USR4(("tclMediaClientHandler::SetActiveSource activeMediaSource %d ", activeMediaSource));

   if (activeMediaSource != FI_EN_E8DTY_UNKNOWN)
   {
      g_SM_ActiveMediaSource = activeMediaSource;
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::SetActiveSource g_SM_ActiveMediaSource %d", g_SM_ActiveMediaSource));
}


/************************************************************************
*FUNCTION: 		 vOnDeviceConnection
*DESCRIPTION:  Check the device connection status and update
*
*PARAMETER:  stDeviceConnectionInfo &StrDevice,tU8 iCurrListIndex
*
*RETURNVALUE: 	tVoid
************************************************************************/
void tclMediaClientHandler::vOnDeviceConnection(stDeviceConnectionInfo* StrDevice, tU8 iCurrListIndex)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection DevNum %d", StrDevice->devNum));
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection currentlistindex %d", iCurrListIndex));

   // On Connection
   bool bDevConnectedStatus = StrDevice->bConnectedStatus;
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection:DeviceConnected %d", bDevConnectedStatus));

   if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected())
   {
//Update the device name and device tag
      StrDevice->sName = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getSDeviceName();
      if (StrDevice->bConnectedStatus == false)
      {
         StrDevice->u8DeviceTag = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag();
         ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection:DeviceConnected for all devices %d", StrDevice->u8DeviceTag));
      }

      //To Eliminate the Lint Warning
      m_u32NumberOfFiles = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU32NumberOfFiles();
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection:numberoffiles %d", m_u32NumberOfFiles));

      if (m_u32NumberOfFiles == -1)
      {
         StrDevice->u32PlayableFiles = 0;
      }
      else
      {
         StrDevice->u32PlayableFiles = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU32NumberOfFiles();
      }
      StrDevice->bConnectedStatus = TRUE;
      // StrDevice->bIsActive = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceActiveSource();
      StrDevice->bRepeatSupported = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBRepeatSupported();
      StrDevice->bShuffleSupported = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBShuffleSupported();

      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection  StrDevice->sName %s", StrDevice->sName));
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection  StrDevice->u8DeviceTag %d", StrDevice->u8DeviceTag));
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection  StrDevice->u32PlayableFiles %d", StrDevice->u32PlayableFiles));
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection  StrDevice->bIsActive %d", StrDevice->bIsActive));
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection  StrDevice->devNum %d", StrDevice->devNum));

      //Post the Device connection pop up only if change in status
      if (!bDevConnectedStatus)
      {
         HMIModelComponent::getInstance().vUpdateDeviceConnectionStatus(StrDevice->u8DeviceTag,
               m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceType(),
               m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected(), StrDevice->bIsActive, 0, StrDevice->disconnectReason, StrDevice->devNum);
      }
   }
   //Device Disconnected
   else if (!(m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected()))
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection::m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag() %d", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag()));

      //Check for the connection status flag and the device tag
      if ((StrDevice->bConnectedStatus) && (StrDevice->u8DeviceTag == m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag()))
      {
         /*
         On disconnection we are erasing metadata,clearing the sourceheader name,albumart is made default.
         */
         MediaSourceHandling::getInstance().removeMediaSourcefromVector(m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag());
         MediaSourceHandling::getInstance().vUpdateMediaSrcMapOnDisconnection(m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag());
         //Clear the Device Name and connection status flag
         //StrDevice->sName.empty();//to fix coverity
         StrDevice->bConnectedStatus = FALSE;
         StrDevice->u32PlayableFiles = 0;
         StrDevice->bRepeatSupported = FALSE;
         StrDevice->bShuffleSupported = FALSE;

         //Resetted the subsource ID which used to reconize which is the last active source
         //Reset only if the last played source device tag and disconnected device's tag are same
         int iPrevActiveTag = clMetaData::getInstance().getSubSrcID();
         ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceConnection:: iPrevActiveTag  %d", iPrevActiveTag));
         if (iPrevActiveTag == m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag())
         {
            clMetaData::getInstance().setSubSrcID(LASTSONG_TAG_RESET);
         }
         //Post the Device disconnection pop up

         HMIModelComponent::getInstance().vUpdateDeviceConnectionStatus(m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag(), m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getE8DeviceType(),
               m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getBDeviceConnected(), StrDevice->bIsActive, 0, StrDevice->disconnectReason, StrDevice->devNum);

         if (StrDevice->bIsActive)
         {
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_DEVICE_REMOVAL);
            StateTransitionHandler::getInstance().stateTransit_onActiveSourceDeviceDisconnection();
         }

         StrDevice->bIsActive = FALSE;
      }
      /*else
      //{
      //   ETG_TRACE_USR1(("tclMediaClientHandler::vOnDeviceDisConnection  Nothing"));
      //   //   StrDevice->bIsActive = FALSE;
      }*/
   }
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: Check the device connection status of USB1 and USB2 and update
*PARAMETER		: USB2SelectDeviceMsg
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const USB2SelectDeviceMsg& oMsg)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:USB2SelectDeviceMsg: devicesource %d", oMsg.GetDeviceSrc()));

   _flagTile = 0; //reset the _flagTile
   // Local Device connection status variable
   stDeviceConnectionInfo structDevice_Name = (struct stDeviceConnectionInfo)
   {
      "", 0, false, false, 0
   };

   //// Requested Source Information
   tU8 u8RequestedSrc = oMsg.GetDeviceSrc();

   int iCurrentSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   int iCurrentDeviceType = GetMediaDeviceType(iCurrentSubSrcId);
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:USB2SelectDeviceMsg:iCurrentDeviceType : %d,iCurrentSubSrcId:  %d  ", iCurrentDeviceType, iCurrentSubSrcId));

   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:USB2SelectDeviceMsg:USB u8RequestedSrc? : %d ", u8RequestedSrc));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:USB2SelectDeviceMsg:USB2 isActive? : %d ", m_StrDevice_USB2.bIsActive));

   //To show the nowplaying screen when user select the USB2/MTP2 second time
   if (((FI_EN_E8DTY_MTP == iCurrentDeviceType) || (FI_EN_E8DTY_USB == iCurrentDeviceType)) && (FI_EN_E8DTY_USB == u8RequestedSrc)
         && (m_StrDevice_USB2.bIsActive == 1) && (m_StrDevice_USB2.u32PlayableFiles > 0))
   {
      HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(iCurrentDeviceType);
      StateTransitionHandler::getInstance().activateMediaState(MEDIA_STATE_NOWPLAYING);

      m_StrDevice_USB.bIsActive = FALSE;
   }
   else
   {
      // Check the requested device type if current source is not the requested source
      switch (u8RequestedSrc)
      {
         case FI_EN_E8DTY_USB:
         case FI_EN_E8DTY_MTP:
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:USB2SelectDeviceMsg USB2 bConnectedStatus? : %d ", m_StrDevice_USB2.bConnectedStatus));

            if (m_StrDevice_USB2.bConnectedStatus)
            {
               m_StrDevice_USB.bIsActive = FALSE;
               m_StrDevice_USB2.bIsActive = TRUE;
               structDevice_Name = m_StrDevice_USB2;
            }

            //HK: The below line is to know which USB device is activated, usb1 / usb2. 1 is for usb1 & 2 is for usb2.
            if ((m_StrDevice_USB2.bConnectedStatus))
            {
               clMetaData::getInstance().setUSBDeviceNumber(2);
            }

            vOnDeviceSelection(structDevice_Name, u8RequestedSrc, USB_TWO);
         }
      }
   }

   return false;
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: Check the device connection status and update
*PARAMETER		: SelectDeviceSourceMsg
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const SelectDeviceSourceMsg& oMsg)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: %d", oMsg.GetDeviceSrc()));

   _flagTile = 0; //reset the _flagTile

   m_u32PrevImageSize = 0;
   // Default for source selection so set to true
   bool bselectionFlag = TRUE;

   // Local Device connection status variable
   stDeviceConnectionInfo structDevice_Name = (struct stDeviceConnectionInfo)
   { "", 0, false, false, 0
   };

   //// Requested Source Information
   tU8 u8RequestedSrc = oMsg.GetDeviceSrc();

   m_u8ActiveRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: activeregion %d, SMactiveRegion %d", m_u8ActiveRegion, g_SM_ActiveRegion));

   if (m_u8ActiveRegion == REG_A)
   {
      m_u8SinkId = CABIN_A_SINK_ID;
   }
   else if (m_u8ActiveRegion == REG_B)
   {
      m_u8SinkId = CABIN_B_SINK_ID;
   }
   else
   {
      m_u8SinkId = COCKPIT_SINK_ID;
   }

   // Check for IPOD/BT selection for a connected IPHONE
   // Set Requested source to IPHONE

   int iCurrentSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   int iCurrentDeviceType = GetMediaDeviceType(iCurrentSubSrcId);
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: CurrentSubSrcId %d", iCurrentSubSrcId));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: CurrentDeviceType %d", iCurrentDeviceType));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg devicetagUSB1  %d", m_StrDevice_USB.u32PlayableFiles));

   if ((u8RequestedSrc == FI_EN_E8DTY_AUX) && (MediaSourceHandling::getInstance().getCurrentRegionSrcId() == SRC_MEDIA_AUX))
   {
      iCurrentDeviceType = FI_EN_E8DTY_AUX;
   }
   // Check for current source same as requested source
   //Condition satisfy when media source launch after the audio allocation(on 2nd time)
   //In case of BT, the number of playable files is zero by default .So no checking the number of playable files
   if (((FI_EN_E8DTY_USB == u8RequestedSrc) && ((FI_EN_E8DTY_USB == iCurrentDeviceType) || (FI_EN_E8DTY_MTP == iCurrentDeviceType)) &&
         (m_StrDevice_USB.bIsActive == TRUE) && (m_StrDevice_USB.u32PlayableFiles > 0)) ||
         ((iCurrentDeviceType == FI_EN_E8DTY_IPHONE || iCurrentDeviceType == FI_EN_E8DTY_IPOD) && (u8RequestedSrc == FI_EN_E8DTY_IPOD)) ||
         ((iCurrentDeviceType == FI_EN_E8DTY_BLUETOOTH) && (u8RequestedSrc == FI_EN_E8DTY_BLUETOOTH)) ||
         ((iCurrentDeviceType == FI_EN_E8DTY_AUX) && (u8RequestedSrc == FI_EN_E8DTY_AUX)))
   {
      HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(iCurrentDeviceType);
      StateTransitionHandler::getInstance().activateMediaState(MEDIA_STATE_NOWPLAYING);

      m_StrDevice_USB2.bIsActive = FALSE;
   }
   else
   {
      // Check the requested device type if current source is not the requested source
      switch (u8RequestedSrc)
      {
         //BT
         case FI_EN_E8DTY_BLUETOOTH:
         {
            // Check for iAP2 Device via BT
            structDevice_Name = m_StrDevice_BT;
            setUSBInactive();

            break;
         }
         // Apple Via USB
         case FI_EN_E8DTY_IPOD:
         case FI_EN_E8DTY_IPHONE:
         {
            structDevice_Name = m_StrDevice_Apple_USB;

            setUSBInactive();
            break;
         }

         //AUX
         //smart
         case FI_EN_E8DTY_AUX:
         {
            structDevice_Name = m_StrDevice_AUX;
            g_SM_ActiveMediaSource = 7;
            m_StrDevice_USB.bIsActive = FALSE;
            m_StrDevice_USB2.bIsActive = FALSE;
            break;
         }
         // USB
         case FI_EN_E8DTY_USB:
         case FI_EN_E8DTY_MTP:
         {
            if (m_StrDevice_USB.bConnectedStatus)
            {
               m_StrDevice_USB2.bIsActive = FALSE;
               m_StrDevice_USB.bIsActive = TRUE;

               structDevice_Name = m_StrDevice_USB;
            }
            break;
         }
         default:
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:SelectDeviceSourceMsg: INVALID DEVICE"));
            bselectionFlag = false;
            break;
         }
      }// End of Switch

      // Source selection
      if (bselectionFlag)
      {
         //HK: The below line is to know which USB device is activated, usb1 / usb2. 1 is for usb1 & 2 is for usb2.
         if ((m_StrDevice_USB.bConnectedStatus))
         {
            clMetaData::getInstance().setUSBDeviceNumber(1);
         }

         vOnDeviceSelection(structDevice_Name, u8RequestedSrc, USB_ONE);
      }
   }

   return false;
}


/************************************************************************
*FUNCTION		: vOnDeviceSelection
*DESCRIPTION	: Updates the HMIModelComponent
*PARAMETER		: StrDevice
*PARAMETER		: u8RequestedSrc
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
void tclMediaClientHandler::vOnDeviceSelection(stDeviceConnectionInfo StrDevice, tU8 u8RequestedSrc, tU8 u8UsbNum)
{
   bool bSourceFlag = FALSE;
   bool bNoPlayableContent = FALSE;
   bool bDeviceConnected = FALSE;

   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection: StrDevice.bConnectedStatus %d StrDevice.u32PlayableFilesm %d  "
                   , StrDevice.bConnectedStatus, StrDevice.u32PlayableFiles));
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection: Device Tag %d ", StrDevice.u8DeviceTag));
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection: Requested source %d ", u8RequestedSrc));
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection: UsbNum %d ", u8UsbNum));

   // If Apple Device Connected
   if (StrDevice.bConnectedStatus)
   {
      // Device Connected
      bDeviceConnected = TRUE;

      // Playble Files: Set Source Flag and Device Tag Info
      if ((StrDevice.u32PlayableFiles > 0))
      {
         bSourceFlag = TRUE;
      }
      // No Playble Content
      else
      {
         if (u8RequestedSrc == FI_EN_E8DTY_BLUETOOTH)
         {
            bSourceFlag = BTClientHandler::getInstance()->getConnectedDeviceStatus(StrDevice.sName);
            //bSourceFlag = TRUE;
         }
         else if (m_u32NumberOfFiles != -1)
         {
            bNoPlayableContent = TRUE;
         }
         if ((u8RequestedSrc == FI_EN_E8DTY_IPHONE) || (u8RequestedSrc == FI_EN_E8DTY_IPOD))
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::vOnDeviceSelection NO OF FILES IS ZERO and do audioallocation"));//in iphone songs can be played from spotify apps
            bNoPlayableContent = FALSE;
            bSourceFlag = TRUE;
         }
      }
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection:SourceFlag:%d", bSourceFlag));
   ETG_TRACE_USR4(("tclMediaClientHandler::vOnDeviceSelection:NoPlayableContent:%d", bNoPlayableContent));

   // Request Source Allocation
   if (bSourceFlag)
   {
      StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
      AudioAllocationRequest(u8RequestedSrc, StrDevice); // pass the entire structure (didn't do)
   }

   // Request Pop-up
   if (bNoPlayableContent)
   {
      //Post trigger to show no playable content
      HMIModelComponent::getInstance().vUpdateDeviceConnectionStatus(StrDevice.u8DeviceTag, u8RequestedSrc, bDeviceConnected, 0, bNoPlayableContent, StrDevice.disconnectReason, StrDevice.devNum);
   }
   else if (!bDeviceConnected)
   {
      HMIModelComponent::getInstance().bOnUpdateDeviceStatusOnSelectionMsg(u8RequestedSrc, u8UsbNum);
   }
}


/************************************************************************
*FUNCTION		: vClearListDetailsMaps
*DESCRIPTION	: Clear the listmap
*PARAMETER		: void
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::vClearListDetailsMaps()
{
   ETG_TRACE_USR1(("tclMediaClientHandler::vClearListDetailsMaps"));
   std::map<uint32, ListUtilityDatabase* >::const_iterator Iter_DB = _listMap.begin();
   for (; Iter_DB != _listMap.end(); ++Iter_DB)
   {
      ListUtilityDatabase* tempReference = Iter_DB->second;
      delete tempReference;
   }
   _listMap.clear();
}


/************************************************************************
*FUNCTION		: getMediaPlayerDeviceType
*DESCRIPTION	: This method returns the media player source device type, USB or iPod.
*PARAMETER		: NA
*RETURNVALUE	: int
************************************************************************/
int tclMediaClientHandler::getMediaPlayerDeviceType()
{
   ETG_TRACE_USR4(("tclMediaClientHandler::getMediaPlayerDeviceType:m_rActivedeviceInfo.u8DeviceType:%d", m_rActivedeviceInfo.u8DeviceType));

   return m_rActivedeviceInfo.u8DeviceType;
}


/************************************************************************
*FUNCTION		: AudioAllocationRequest
*DESCRIPTION	: sends audio allocation request
*PARAMETER		: tU8 u8SelectedDevType, tU8 u8SelectedDevTag
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::AudioAllocationRequest(tU8 u8SelectedDevType, stDeviceConnectionInfo StrDevice)
{
   m_selectedDevTag = StrDevice.u8DeviceTag;
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::AudioAllocationRequest activeRegion  %d ", u8activeRegion));
   ETG_TRACE_USR4(("tclMediaClientHandler::AudioAllocationRequest u8SelectedDevType %d", u8SelectedDevType));
   //Check the device type
   int srcId = getSourceId(u8SelectedDevType);
   ETG_TRACE_USR4(("tclMediaClientHandler::AudioAllocationRequest: srcId %d", srcId));
   if (srcId != SRC_INVALID)
   {
      MediaSourceHandling::getInstance().requestSourceActivation(srcId, m_selectedDevTag, u8activeRegion);
   }
}


void tclMediaClientHandler::onActiveMediaDeviceError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::ActiveMediaDeviceError >& /*error*/)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onActiveMediaDeviceError received "));
}


void tclMediaClientHandler::onActiveMediaDeviceStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::ActiveMediaDeviceStatus >& status)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onActiveMediaDeviceStatus active device tag = %d isActive = %d", status->getU8DeviceTag(), status->getBDeviceActiveSource()));

   vUpdateActiveMediaDeviceStatus(*status);
}


void tclMediaClientHandler::vUpdateActiveMediaDeviceStatus(const ::mplay_MediaPlayer_FI::ActiveMediaDeviceStatus& oActiveDeviceInfoDataObject)
{
   tCU32 u32listsize = m_oDeviceConnectionStatus.getODeviceInfo().size();
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateActiveMediaDeviceStatus :ListSize: %d", u32listsize));

   if (m_rActivedeviceInfo.u8DeviceTag != oActiveDeviceInfoDataObject.getU8DeviceTag())
   {
      vClearListDetailsMaps();
      HMIModelListImplementation::getInstance().ClearListType();
   }
   m_rActivedeviceInfo.u8DeviceTag = oActiveDeviceInfoDataObject.getU8DeviceTag();
   m_selectedDevTag = m_rActivedeviceInfo.u8DeviceTag;
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateActiveMediaDeviceStatus:DeviceTag: %d", m_rActivedeviceInfo.u8DeviceTag));
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateActiveMediaDeviceStatus:u8CurrentEntGrp: %d", m_u8CurrentEntGrp));

   m_rActivedeviceInfo.u8DeviceType = 0;
   m_rActivedeviceInfo.bDeviceActiveSource = FALSE;
   // Find the device type
   for (tU8 u8Index = 0; u8Index < u32listsize; u8Index++)
   {
      if (m_rActivedeviceInfo.u8DeviceTag == m_oDeviceConnectionStatus.getODeviceInfo().at(u8Index).getU8DeviceTag())
      {
         ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateActiveMediaDeviceStatus:DeviceType: %d", m_rActivedeviceInfo.u8DeviceType));
         m_rActivedeviceInfo.u8DeviceType =
            m_oDeviceConnectionStatus.getODeviceInfo().at(u8Index).getE8DeviceType();
         m_rActivedeviceInfo.bDeviceActiveSource =
            m_oDeviceConnectionStatus.getODeviceInfo().at(u8Index).getBDeviceConnected();
         break;
      }
   }

   int devNum = 0;
   if (m_rActivedeviceInfo.u8DeviceType == 1)
   {
      devNum = getUSBDevNum();
   }
}


/************************************************************************
*FUNCTION		: onPlaytimeStatus(SWFL-640,SWFL-653)
*DESCRIPTION	: called on getting result for Playtime property from media player
*PARAMETER		: proxy
*PARAMETER		: status
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onPlaytimeStatus(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaytimeStatus >& status)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onPlaytimeStatus received: with ElapsedTime :%d TotalTime:%d", status->getU32ElapsedPlaytime(), status->getU32TotalPlaytime()));
   uint32 iCurrentElapsedTime = status->getU32ElapsedPlaytime();
   ETG_TRACE_USR4(("tclMediaClientHandler onPlaytimeStatus iCurrentElapsedTime %d", iCurrentElapsedTime));
   ETG_TRACE_USR4(("tclMediaClientHandler::onPlaytimeStatus m_u32previousElapsedTime :%d ", m_u32previousElapsedTime));
   ETG_TRACE_USR4(("tclMediaClientHandler::onPlaytimeStatus m_i32requestedSeekTime :%d ", m_i32requestedSeekTime));
   ETG_TRACE_USR4(("tclMediaClientHandler::onPlaytimeStatus m_isSeeked :%d ", m_isSeeked));

   if (m_u32previousElapsedTime != iCurrentElapsedTime)
   {
      if (m_isSeeked == true)
      {
         vUpdateMediaPlayTime(status->getU32TotalPlaytime(), status->getU32ElapsedPlaytime());
         m_isSeeked = false;
      }
      else if (m_isSeeked == false)
      {
         vUpdateMediaPlayTime(status->getU32TotalPlaytime(), status->getU32ElapsedPlaytime());
      }

      m_u32previousElapsedTime = status->getU32ElapsedPlaytime();
      ETG_TRACE_USR4(("tclMediaClientHandler onPlaytimeStatus m_isSeeked %d", m_isSeeked));
      ETG_TRACE_USR4(("tclMediaClientHandler::onPlaytimeStatus received: with m_u32previousElapsedTime :%d ", m_u32previousElapsedTime));
   }
}


/************************************************************************
*FUNCTION		: onPlaytimeError
*DESCRIPTION	: called if Playtime property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onPlaytimeError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::PlaytimeError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onPlaytimeError received"));
}


/************************************************************************
*FUNCTION		: vUpdateMediaPlayTime(SWFL-640,SWFL-653)
*DESCRIPTION	: Sends the play time info to HMI Model component
*PARAMETER		: u32TotalPlaytime
*PARAMETER		: u32Elapsetime
*RETURNVALUE	: void
*SWFL			: [SWFL-2454]
************************************************************************/
void tclMediaClientHandler::vUpdateMediaPlayTime(const tU32& u32TotalPlaytime, const tU32& u32Elapsetime)
{
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateMediaPlayTime Active region : %d", u8activeRegion));

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(u8activeRegion);
   if (MediaUtils::getInstance().m_ActiveDeviceDataMapItr != MediaUtils::getInstance().m_ActiveDeviceDataMap.end())
   {
      int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
      ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateMediaPlayTime   iActiveSource %d", iActiveSource))
      ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateMediaPlayTime u32TotalPlaytime :%d, u32Elapsetime:%d", u32TotalPlaytime, u32Elapsetime));
      ETG_TRACE_USR4(("tclMediaClientHandler::vUpdateMediaPlayTime m_u8NowPlayingDevType :  %d", m_u8NowPlayingDevType));

      if (iActiveSource == INTERNALSOURCE && (u8activeRegion == REGION_COCKPIT || u8activeRegion == REGION_CABIN_A))
      {
         Candera::String szPlayTime;
         Candera::String szTotalTime;

         clMetaData::getInstance().setTotalTime(u32TotalPlaytime);
         clMetaData::getInstance().setPlayTime(u32Elapsetime);

         szOnConvertPlayTime(szTotalTime, u32TotalPlaytime);
         szOnConvertPlayTime(szPlayTime, u32Elapsetime);

         HMIModelComponent::getInstance().bOnUpdatePlayTimeStatus(u32TotalPlaytime, u32Elapsetime, szTotalTime, szPlayTime, m_u8NowPlayingDevType);
      }
      else
      {
         //do nothing
      }
   }
}


/*********************************************************************************************
* Function 	    : szOnConvertPlayTime
* Description 	:
* Parameters 	: Candera::String& refTime
* Parameters 	: tU32 u32PlayTime
* Return value	: None
**********************************************************************************************/
tVoid tclMediaClientHandler::szOnConvertPlayTime(Candera::String& refTime, tU32 u32PlayTime)
{
   //ETG_TRACE_USR4(("tclMediaAppLogic::szOnConvertPlayTime(%d)", u32PlayTime));
   ETG_TRACE_USR4(("tclMediaClientHandler::szOnConvertPlayTime u32PlayTime :  %d", u32PlayTime));

   tU32 u32Hours = ZERO;
   tU32 u32Minutes = ZERO;
   tU32 u32Seconds = ZERO;
   tU32 u32RemainingSecs = ZERO;
   char szTime[TIME_BUFFER_SIZE];
   /*Time Conversion based on Universal Time Conversion Formula*/
   u32Hours = u32PlayTime / (60 * 60);
   u32RemainingSecs = u32PlayTime % (60 * 60);
   u32Minutes = u32RemainingSecs / 60;
   u32Seconds = u32RemainingSecs % 60;
   memset(szTime, '\0', TIME_BUFFER_SIZE);
   if (u32Hours > ZERO)
   {
      snprintf(szTime, (TIME_BUFFER_SIZE), "%01d:%02d:%02d", u32Hours, u32Minutes, u32Seconds);
   }
   else
   {
      snprintf(szTime, (TIME_BUFFER_SIZE), "%02d:%02d", u32Minutes, u32Seconds);
   }
   refTime = szTime;
}


/************************************************************************
*FUNCTION		: onSeekToResult
*DESCRIPTION	: called on getting result for SeekTo property from media player
*PARAMETER		: proxy
*PARAMETER		: status
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onSeekToResult(const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< mplay_MediaPlayer_FI::SeekToResult >& result)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onSeekToResult called"));
}


/************************************************************************
*FUNCTION		: onSeekToError
*DESCRIPTION	: called if SeekTo property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/
void tclMediaClientHandler::onSeekToError(const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< mplay_MediaPlayer_FI::SeekToError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onSeekToError called"));
}


/************************************************************************
*FUNCTION		: onCourierBindingItemChange_PlaybackActionItem
*DESCRIPTION	: called whenever there is tap or drag on slider
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/
bool tclMediaClientHandler::onCourierBindingItemChange_PlaybackActionItem(const Courier::Request& request)
{
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("HMIModelComponent::vOnUpdateElapsedTimeStatusCameraPort u8activeRegion  %d ", u8activeRegion));

   int iActiveSource = MediaUtils::getInstance().m_ActiveDeviceDataMap[u8activeRegion].activeSource;
   ETG_TRACE_USR4(("HMIModelComponent : vOnUpdateElapsedTimeStatusCameraPort u8activeRegion is CABIN A   %d", iActiveSource));

   if (iActiveSource == INTERNALSOURCE)
   {
      m_MediaPlayBackData.SetValue(request.ItemKey(), request.GetItemValue());
      //ETG_TRACE_USR4(("tclMediaClientHandler::onCourierBindingItemChange_MediaMetaDataItem() called   %d", request.GetItemValue()));
      unsigned int seektime = static_cast<unsigned int>((*m_MediaPlayBackData).mSliderElapsedTime);
      m_i32requestedSeekTime = seektime;
      m_isSeeked = true;
      ETG_TRACE_USR4(("tclMediaClientHandler::onCourierBindingItemChange_PlaybackActionItem seek time %d", seektime));
      _mediaPlayerProxy->sendSeekToStart(*this, seektime * 1000, ::MPlay_fi_types::T_e8_MPlayPlaypointFormat__e8PPF_ABSOLUTE);
   }

   return false;
}


/************************************************************************
*FUNCTION		: onCourierBindingItemChange_BTPlayInfoUpdateItem
*DESCRIPTION	: called whenever there is tap or drag on slider
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/
bool tclMediaClientHandler::onCourierBindingItemChange_BTPlayInfoUpdateItem(const Courier::Request& request)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierBindingItemChange_MediaMetaDataItem called"));
   m_MediaBTDeviceData.SetValue(request.ItemKey(), request.GetItemValue());
   unsigned int seektime = static_cast<unsigned int>((*m_MediaBTDeviceData).mSliderElapsedTime);
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierBindingItemChange_BTPlayInfoUpdateItem seek time %d", seektime));
   _mediaPlayerProxy->sendSeekToStart(*this, seektime * 1000, ::MPlay_fi_types::T_e8_MPlayPlaypointFormat__e8PPF_ABSOLUTE);
   return true;
}


/************************************************************************
*FUNCTION		: onCourierBindingItemChange_AppleDeviceItem
*DESCRIPTION	: called whenever there is tap or drag on slider
*PARAMETER		: proxy
*PARAMETER		: error
*RETURNVALUE	: void
************************************************************************/
bool tclMediaClientHandler::onCourierBindingItemChange_AppleDeviceItem(const Courier::Request& request)
{
   m_MediaAppleDeviceData.SetValue(request.ItemKey(), request.GetItemValue());
   unsigned int seektime = static_cast<unsigned int>((*m_MediaAppleDeviceData).mSliderElapsedTime);
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierBindingItemChange_AppleDeviceItem seek time %d", seektime));
   m_i32requestedSeekTime = seektime;
   m_isSeeked = true;
   _mediaPlayerProxy->sendSeekToStart(*this, seektime * 1000, ::MPlay_fi_types::T_e8_MPlayPlaypointFormat__e8PPF_ABSOLUTE);
   return true;
}


/**
* Trace command to call onSeek
*
* @param[in]      : Seek type
* @param[in]      : Seek direction
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_TrackChange, "TraceCmd_TrackChange %d", uint8))
void tclMediaClientHandler::TraceCmd_TrackChange(uint8 operationType)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::TraceCmd_TrackChange: operation type %d", operationType));
   //_mediaClientHandler->
}


/*
*  Toggle_Context : Toggle Button Functionality for Media homescreen toggle button
* @param[out] None
[SYSFL - 5463]
* @return         : void
*/
bool tclMediaClientHandler::PerformToggleContext()
{
   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::Toggle_Context u8activeRegion cabin  %d ", u8activeRegion));

   int iLastActiveMediasrcid = MediaSourceHandling::getInstance().GetLastMediaSrcId();
   int iLastActiveMediasubsrcid = MediaSourceHandling::getInstance().GetLastMediaSubSrcId();
   ETG_TRACE_USR4(("tclMediaClientHandler::Toggle_Context iLastActiveMediasrcid: %d", iLastActiveMediasrcid));
   ETG_TRACE_USR4(("tclMediaClientHandler::Toggle_Context iLastActiveMediasubsrcid: %d", iLastActiveMediasubsrcid));

   if (GetMediaDeviceConnectedStatus(iLastActiveMediasubsrcid))
   {
      MediaSourceHandling::getInstance().requestSourceActivation(iLastActiveMediasrcid, iLastActiveMediasubsrcid, u8activeRegion);
   }
   else if (g_SM_VARIANT_AUX == 1 && iLastActiveMediasrcid == SRC_MEDIA_AUX  && u8activeRegion == REGION_COCKPIT)
   {
      MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_AUX, -1, u8activeRegion);
   }
   else if (iLastActiveMediasrcid == SRC_CAMERAPORT1_USB_VIDEO)
   {
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_USB_CAMERAPORT);
      iExternalMediaSourceActivationHandler->requestSourceActivation(u8activeRegion);
   }
   else if (iLastActiveMediasrcid == SRC_CAMERAPORT1_USB_AUDIO)
   {
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_USB_CAMERAPORT);
      iExternalMediaSourceActivationHandler->requestSourceActivation(u8activeRegion);
   }
   else if (iLastActiveMediasrcid == SRC_CAMERAPORT1_HDMI || iLastActiveMediasrcid == SRC_CAMERAPORT2_HDMI || iLastActiveMediasrcid == SRC_CAMERAPORT3_HDMI)
   {
      _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
      _mediaSourcesPriority.sourceId = iLastActiveMediasrcid;

      MediaSourceHandling::getInstance().requestSourceActivation(_mediaSourcesPriority.sourceId, _mediaSourcesPriority.sourceTag, u8activeRegion);
   }
   else if (iLastActiveMediasrcid == SRC_CMR1_USB1_VIDEO || iLastActiveMediasrcid == SRC_CMR1_USB1_AUDIO)
   {
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_ROUTER_USB);
      iExternalMediaSourceActivationHandler->requestSourceActivation(u8activeRegion);
   }
   else
   {
      bool isPrioSourceConnected = getPrioSourceConnectedInfo();

      ETG_TRACE_USR4(("tclMediaClientHandler::Toggle_Context, isPrioSourceConnected: %d", isPrioSourceConnected));

      if (isPrioSourceConnected)
      {
         if (_mediaSourcesPriority.sourceId == SRC_CAMERAPORT1_USB_VIDEO)
         {
            IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_USB_CAMERAPORT);
            iExternalMediaSourceActivationHandler->requestSourceActivation(u8activeRegion);
         }
         else if (_mediaSourcesPriority.sourceId == SRC_CMR1_USB1_VIDEO)
         {
            IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(FI_EN_E8DTY_ROUTER_USB);
            iExternalMediaSourceActivationHandler->requestSourceActivation(u8activeRegion);
         }
         else
         {
            MediaSourceHandling::getInstance().requestSourceActivation(_mediaSourcesPriority.sourceId, _mediaSourcesPriority.sourceTag, u8activeRegion);
         }
      }
      else
      {
         if (g_SM_VARIANT_AUX == 1 && u8activeRegion == REGION_COCKPIT)
         {
            MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_AUX, -1, u8activeRegion);
         }
         else
         {
            ApplicationSwitchClientHandler::poGetInstance()->reqHomeScreenToggle();
         }
      }
   }

   StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_OTHERS);
   return true;
}


#endif


int tclMediaClientHandler::getSourceId(int iDeviceType)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("tclMediaClientHandler::getSourceId devicetype: %d", iDeviceType));
#endif
   int srcId = SRC_INVALID;
   switch (iDeviceType)
   {
      case FI_EN_E8DTY_BLUETOOTH:
      {
         srcId = SRC_PHONE_BTAUDIO;
         break;
      }
      case FI_EN_E8DTY_USB:
      case FI_EN_E8DTY_MTP:
      case FI_EN_E8DTY_IPOD:
      case FI_EN_E8DTY_IPHONE:
      {
         srcId = SRC_MEDIA_PLAYER;
         break;
      }
      //smart
      case FI_EN_E8DTY_AUX:
      {
         srcId = SRC_MEDIA_AUX;
         break;
      }
      default:
         break;
   }
   return srcId;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
std::string tclMediaClientHandler::getBTDeviceName(uint8_t deviceTag)
{
   std::string devName;
   ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceName  deviceTag =  %d", deviceTag));
   for (int iCurrListIndex = 0; iCurrListIndex < m_oDeviceConnectionStatus.getODeviceInfo().size(); ++iCurrListIndex)
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceName device name =  %s", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getSDeviceName().c_str()));
      if (m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getU8DeviceTag() == deviceTag)
      {
         devName = m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getSDeviceName();
         ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceName device name =  %s", devName.c_str()));
         return devName;
         //break;
      }
   }

   //To fix coverity issue
   return "";
   //ETG_TRACE_USR4(("tclMediaClientHandler::getBTDeviceName device name =  %s", m_oDeviceConnectionStatus.getODeviceInfo().at(iCurrListIndex).getSDeviceName().c_str()));
}


/**onExpired - This function is called when Timer Expires

* @param[in] timer
* @parm[in] data
* @return void
*/
void tclMediaClientHandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onExpired Received Timer expired\n %d", timer.getAct()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onExpired Received _videoTimer Timer expired\n %d", _videoTimer.getAct()));
   if ((&timer == &_videoTimer) && (data->getReason() == asf::core::TimerPayload_Reason__Completed))
   {
      _videoTimer.stop();
      //post courier message to switch to full screen

      POST_MSG((COURIER_MESSAGE_NEW(SwitchFullScreenVideo)()));//case when no source is connected, switch back to Tuner context
   }
}


/**
* onCourierMessage StartVideoTimerMsg - helper function to Start the Timer for Video FullScreen when it enters the video screen
* @param[in]:StartVideoTimerMsg
* @return bool
*/
bool tclMediaClientHandler::onCourierMessage(const StartVideoTimerMsg& oMsg)
{
   _videoTimer.start();

   return false;
}


/**
* onCourierMessage StopVideoTimerMsg - helper function to stop the Timer
* @param[in]:StopVideoTimerMsg
* @return bool
*/
bool tclMediaClientHandler::onCourierMessage(const StopVideoTimerMsg& oMsg)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage:StopVideoTimerMsg  Received Stop \n"));
   _videoTimer.stop();

   return true;
}


/**
* isDeviceConnectedInEvenPortNumber - Check the Mount Point of the Connected Device
* @param[in] mountPoint
* @param[out] none
* @return bool
*/
bool tclMediaClientHandler::isDeviceConnectedInEvenPortNumber(std::string& sysPath)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::isDeviceConnectedInEvenPortNumber Mount Point is %s", sysPath.c_str()));
   bool status = false;
   ::std::size_t found = ::std::string::npos;
   if (!sysPath.empty())
   {
      found = sysPath.find_last_of(".");
   }
   if (found != ::std::string::npos)
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::isDeviceConnectedInEvenPortNumber PortNumber Substring is %s", sysPath.substr(found + 1).c_str()));
      int portNumber = atoi(sysPath.substr(found + 1).c_str());
      ETG_TRACE_USR4(("tclMediaClientHandler::isDeviceConnectedInEvenPortNumber PortNumber is %d", portNumber));
      if (portNumber <= 0) //If device connected Internally(Direct Connection), No External Hub Connection
      {
         status = false;
      }
      else
      {
         if (portNumber % 2)
         {
            status = true; //For Reno only- Device Connected in EVEN Port Number (2nd USB Port number is returned as 3)
         }
         else
         {
            status = false; //For Reno only- Device Connected in ODD Port Number (1st USB Port number is returned as 2)
         }
      }
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::isDeviceConnectedInEvenPortNumber status is %d", status));

   return status;
}


void tclMediaClientHandler::setUSB2Visibile()
{
   if (_kdsHandler->getHubKDSValue() == 1) // invisible
   {
      (*oUSB2Visibility).mUSB2Isvisible = 0;

      (*USB1Button).misConnectedWithHub = "USB";
   }
   else
   {
      (*oUSB2Visibility).mUSB2Isvisible = 1;

      (*USB1Button).misConnectedWithHub = "USB 1";
   }

   USB1Button.MarkAllItemsModified();
   USB1Button.SendUpdate(true);

   oUSB2Visibility.MarkAllItemsModified();
   oUSB2Visibility.SendUpdate(true);
}


int tclMediaClientHandler::getUSBDevNum()
{
   int devNum = 0;

   ETG_TRACE_USR4(("tclMediaClientHandler::getUSBDevNum USB1 :%d", m_StrDevice_USB.bIsActive));
   ETG_TRACE_USR4(("tclMediaClientHandler::getUSBDevNum USB2 :%d", m_StrDevice_USB2.bIsActive));

   if (_kdsHandler->getHubKDSValue() == 2)
   {
      if (m_StrDevice_USB2.bIsActive == TRUE)
      {
         devNum = 2;
      }
      else
      {
         devNum = 1;
      }
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getUSBDevNum devNum = %d ", devNum));

   return devNum;
}


void tclMediaClientHandler::setUSBInactive()
{
   ETG_TRACE_USR1(("tclMediaClientHandler::setUSBInactive "));
   m_StrDevice_USB2.bIsActive = FALSE;
   m_StrDevice_USB.bIsActive = FALSE;
}


/************************************************************************
*FUNCTION		: onCourierMessage
*DESCRIPTION	: on medile tile press, set flag to not switch to nowplaying automatically
*PARAMETER		: Open_SourceSelection_Screen
*RETURNVALUE	: TRUE if it consumes the courier message else FALSE
************************************************************************/
bool tclMediaClientHandler::onCourierMessage(const Open_SourceSelection_Screen& oMsg)
{
   ETG_TRACE_USR1(("tclMediaClientHandler::onCourierMessage:Open_SourceSelection_Screen "));
   _flagTile = 1;

   //HMIModelComponent::getInstance().vUpdateExternalSourcesVisibility();
   return false;
}


/************************************************************************
*FUNCTION		: getActiveDeviceType
*DESCRIPTION	: Function to get the active device type as USB1 or USB2
*PARAMETER		:
*RETURNVALUE	:m_rActivedeviceInfo.u8DeviceType
************************************************************************/
int tclMediaClientHandler::getActiveDeviceType()
{
   ETG_TRACE_USR4(("tclMediaClientHandler::getActiveDeviceType:m_rActivedeviceInfo.u8DeviceType %d ", m_rActivedeviceInfo.u8DeviceType));

   return m_rActivedeviceInfo.u8DeviceType;
}


/************************************************************************
*FUNCTION		: clearMediaplayerDataForFactoryReset
*DESCRIPTION	: called when factory reset is performed, it clears the data for media player.
*PARAMETER		: Nil
*RETURN VALUE	: void
************************************************************************/
void tclMediaClientHandler::clearMediaplayerDataForFactoryReset()
{
   _mediaPlayerProxy->sendClearMediaPlayerDataStart(*this);
}


/************************************************************************
*FUNCTION		: onClearMediaPlayerDataError
*DESCRIPTION	: called if ClearMediaPlayerData property is updated with error from media player
*PARAMETER		: proxy
*PARAMETER		: error
*RETURN VALUE	: void
************************************************************************/
void tclMediaClientHandler::onClearMediaPlayerDataError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::ClearMediaPlayerDataError >& error)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onClearMediaPlayerDataError is received."));
}


/************************************************************************
*FUNCTION		: onClearMediaPlayerDataResult
*DESCRIPTION	: called on getting result for PlaybackAction property from media player
*PARAMETER		: proxy
*PARAMETER		: result
*RETURN VALUE	: void
************************************************************************/
void tclMediaClientHandler::onClearMediaPlayerDataResult(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy  >& proxy,
      const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::ClearMediaPlayerDataResult >& result)
{
   ETG_TRACE_COMP(("tclMediaClientHandler::onClearMediaPlayerDataResult is received."));
}


int tclMediaClientHandler::getDeviceTag(int devicetype)
{
   uint32 devicetag = 0;
   if (_mediaPlayerProxy->hasMediaPlayerDeviceConnections())
   {
      ::MPlay_fi_types::T_MPlayDeviceInfo deviceInfo = _mediaPlayerProxy->getMediaPlayerDeviceConnections().getODeviceInfo();
      ::MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = deviceInfo.begin();
      for (; itr != deviceInfo.end(); ++itr)
      {
         if ((itr->getE8DeviceType() == devicetype) && (itr->getBDeviceConnected()))
         {
            devicetag = itr->getU8DeviceTag();
            break;
         }
      }
   }
   ETG_TRACE_USR4(("tclMediaClientHandler::getDeviceTag devicetag = %d ", devicetag));

   return devicetag;
}


/**
* GetMediaDeviceConnectedStatus - to get currently playing device type
*
* @param[in] deviceTag
* @parm[out]
* @return uint32 media player device type
*/
bool tclMediaClientHandler::GetMediaDeviceConnectedStatus(uint8 deviceTag)
{
   ETG_TRACE_USR2(("tclMediaClientHandler::GetMediaDeviceConnectedStatus devicetag:%d ", deviceTag));

   ::MPlay_fi_types::T_MPlayDeviceInfo deviceInfo = _mediaPlayerProxy->getMediaPlayerDeviceConnections().getODeviceInfo();
   ::MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = deviceInfo.begin();
   for (; itr != deviceInfo.end(); ++itr)
   {
      if (itr->getU8DeviceTag() == deviceTag && (itr->getBDeviceConnected()))
      {
         return true;
      }
   }
   return false;
}


/**
* AppleHIDCommandMsg Courier message - called when user performs play,pause,scan next track,scan previous track, repeat and random operations.
*
* @param[in] msg
* @return bool
*/
bool tclMediaClientHandler::onCourierMessage(const AppleHIDCommandMsg& msg)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage AppleHIDCommandMsg - HID Command : %d, KeyEvent: %d : ", msg.GetHIDCommand(), msg.GetKeyEvent()));
   ETG_TRACE_USR4(("tclMediaClientHandler::onCourierMessage AppleHIDCommandMsg - playbackstate : %d ", m_playbackstate));

   ::MPlay_fi_types::T_e_MPlayHIDCommand HIDCommand = static_cast < ::MPlay_fi_types::T_e_MPlayHIDCommand >(msg.GetHIDCommand());
   ::MPlay_fi_types::T_e_MPlayKeyEvent keyEvent = static_cast < ::MPlay_fi_types::T_e_MPlayKeyEvent >(msg.GetKeyEvent());
   if ((HIDCommand == ::MPlay_fi_types::T_e_MPlayHIDCommand__HID_PLAY) || (HIDCommand == ::MPlay_fi_types::T_e_MPlayHIDCommand__HID_PAUSE))
   {
      if (m_playbackstate == ::MPlay_fi_types::T_e8_MPlayAction__e8PBA_PLAY)
      {
         ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send pause command: %d ", msg.GetHIDCommand()));
         ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send pause key event: %d ", msg.GetKeyEvent()));
         _mediaPlayerProxy->sendSendAppleHIDCommandStart(*this, ::MPlay_fi_types::T_e_MPlayHIDCommand__HID_PAUSE, keyEvent);
      }
      else
      {
         ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send play command: %d ", msg.GetHIDCommand()));
         ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send play key event: %d ", msg.GetKeyEvent()));
         _mediaPlayerProxy->sendSendAppleHIDCommandStart(*this, ::MPlay_fi_types::T_e_MPlayHIDCommand__HID_PLAY, keyEvent);
      }
   }
   else
   {
      ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send next and pervious command: %d ", msg.GetHIDCommand()));
      ETG_TRACE_USR4(("AppleHIDCommandMsg - HID Command send next and pervious key event: %d ", msg.GetKeyEvent()));
      _mediaPlayerProxy->sendSendAppleHIDCommandStart(*this, HIDCommand, keyEvent);
   }
   return true;
}


/**
* onRequestPlaybackActionResult - called if SendAppleHIDCommand method_result from media player was successful.
*
* @param[in] proxy
* @parm[in] result
* @return void
*/
void tclMediaClientHandler::onSendAppleHIDCommandResult(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::SendAppleHIDCommandResult >& result)
{
   ETG_TRACE_COMP(("onSendAppleHIDCommandResult received"));
}


/**
* onRequestPlaybackActionError - called if SendAppleHIDCommand method_result from media player has errors.
*
* @param[in] proxy
* @parm[in] error
* @return void
*/
void tclMediaClientHandler::onSendAppleHIDCommandError(const ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/, const boost::shared_ptr< ::mplay_MediaPlayer_FI::SendAppleHIDCommandError >& /*error*/)
{
   ETG_TRACE_COMP(("onSendAppleHIDCommandError received"));
}


void tclMediaClientHandler::vResetMediaSourcesPriorityStruct()
{
   _mediaSourcesPriority = SourcePrioDetails();
}


void tclMediaClientHandler::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
      _externalMediaSourceHandler->addHdmiSourceAvailabilityInfoObservor((*this));
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
      _externalMediaSourceHandler->addRouterSourceAvailabilityInfoObservor((*this));
   }
   else
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::_externalMediaSourceHandler is NULL"));
   }
}


void tclMediaClientHandler::setKdsHandlerInstance(KDSHandler* _kdsHandlerInstance)
{
   _kdsHandler = _kdsHandlerInstance;
}


void tclMediaClientHandler::onHdmiSourceAvailabilityUpdate(int srcId, int hdmiavailable)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onHdmiSourceAvailabilityUpdate:srcId %d, hdmiavailable %d", srcId, hdmiavailable));
   std::map <int, int>::iterator itr = _hdmiSrcAvailability.find(srcId);
   if (itr != _hdmiSrcAvailability.end())
   {
      itr -> second = hdmiavailable;
   }
}


void tclMediaClientHandler::onCamportUsbSourceAvailabilityUpdate(int usbCamportAvailable)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onCamportUsbSourceAvailabilityUpdate iUSBCamportAvailability  %d", usbCamportAvailable));
   _usbCamportAvailability = usbCamportAvailable;
}


void tclMediaClientHandler::onRouterSourceAvailabilityUpdate(int iRouterAvailability)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onRouterSourceAvailabilityUpdate iRouterAvailability  %d", iRouterAvailability));
   m_routerUsbConnectionState = iRouterAvailability;
}


/************************************************************************
*FUNCTION		: getPrioSourceConnectedInfo
*DESCRIPTION	: Function to handle the request source activation for sources like USB,MTP,IPOD
*PARAMETER		: SelectedDeviceTag
*RETURNVALUE	:
************************************************************************/

bool tclMediaClientHandler::getPrioSourceConnectedInfo()
{
   ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo devicetag "));
   int iSourceConnected = 0;

   vResetMediaSourcesPriorityStruct();

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("tclMediaClientHandler::getPrioSourceConnectedInfo u8activeRegion cabin  %d ", u8activeRegion));

   if (m_StrDevice_USB.bConnectedStatus)
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo in uSB1"));
      _mediaSourcesPriority.sourceTag = m_StrDevice_USB.u8DeviceTag;
      _mediaSourcesPriority.sourceId = SRC_MEDIA_PLAYER;
      iSourceConnected = 1;
   }
   else if (m_StrDevice_USB2.bConnectedStatus)
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo in uSB2"));
      _mediaSourcesPriority.sourceTag = m_StrDevice_USB2.u8DeviceTag;
      _mediaSourcesPriority.sourceId = SRC_MEDIA_PLAYER;
      iSourceConnected = 1;
   }
   else if (m_StrDevice_Apple_USB.bConnectedStatus)
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo in Ipod"));
      _mediaSourcesPriority.sourceTag = m_StrDevice_Apple_USB.u8DeviceTag;
      _mediaSourcesPriority.sourceId = SRC_MEDIA_PLAYER;
      iSourceConnected = 1;
   }
   else if (m_StrDevice_BT.bConnectedStatus)
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo in Bluetooth"));
      _mediaSourcesPriority.sourceTag = m_StrDevice_BT.u8DeviceTag;
      _mediaSourcesPriority.sourceId = SRC_PHONE_BTAUDIO;
      iSourceConnected = 1;
   }
   else
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::getPrioSourceConnectedInfo is InDevice"));
   }

   ETG_TRACE_USR4(("tclMediaClientHandler::in getPrioSourceConnectedInfo iSourceConnected   %d ", iSourceConnected));

   if ((u8activeRegion == REGION_CABIN_A || u8activeRegion == REGION_CABIN_B) && iSourceConnected == 0)
   {
      ETG_TRACE_USR4(("tclMediaClientHandler::getPrioSourceConnectedInfo, _usbCamportAvailability: %d, m_routerUsbConnectionState %d", _usbCamportAvailability, m_routerUsbConnectionState));

      if (_usbCamportAvailability == EN_EXTERNAL_MEDIA_CONNECTED)
      {
         _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
         _mediaSourcesPriority.sourceId = SRC_CAMERAPORT1_USB_VIDEO;
         iSourceConnected = 1;
      }
      else if (_hdmiSrcAvailability[SRC_CAMERAPORT1_HDMI] == EN_EXTERNAL_MEDIA_CONNECTED)
      {
         _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
         _mediaSourcesPriority.sourceId = SRC_CAMERAPORT1_HDMI;
         iSourceConnected = 1;
      }
      else if (_hdmiSrcAvailability[SRC_CAMERAPORT2_HDMI] == EN_EXTERNAL_MEDIA_CONNECTED)
      {
         _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
         _mediaSourcesPriority.sourceId = SRC_CAMERAPORT2_HDMI;
         iSourceConnected = 1;
      }
      else if (_hdmiSrcAvailability[SRC_CAMERAPORT3_HDMI] == EN_EXTERNAL_MEDIA_CONNECTED)
      {
         _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
         _mediaSourcesPriority.sourceId = SRC_CAMERAPORT3_HDMI;
         iSourceConnected = 1;
      }
      else if (m_routerUsbConnectionState == EN_EXTERNAL_MEDIA_CONNECTED)
      {
         _mediaSourcesPriority.sourceTag = EXTERNAL_SOURCE_SUBSRCID;
         _mediaSourcesPriority.sourceId = SRC_CMR1_USB1_VIDEO;
         iSourceConnected = 1;
      }
   }

   if (iSourceConnected == 1)
   {
      return true;
   }
   else
   {
      return false;
   }
}


int tclMediaClientHandler::getMediaType()
{
   return m_MediaType;
}


void tclMediaClientHandler::getNowplayingStatus()
{
   ETG_TRACE_USR4(("tclMediaClientHandler::getNowplayingStatus "));
   _mediaPlayerProxy->sendNowPlayingGet(*this);
}


#endif

/**
* GetDeviceType - to get currently playing device type
*
* @param[in] deviceTag
* @parm[out]
* @return uint32 media player device type
*/
uint32 tclMediaClientHandler::GetMediaDeviceType(uint8 deviceTag)
{
   ETG_TRACE_USR2(("tclMediaClientHandler::GetMediaDeviceType:devicetag:%d ", deviceTag));
   uint32 deviceType;
#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
   deviceType = m_deviceType;
#else
   deviceType = MPlay_fi_types::T_e8_MPlayDeviceType__e8DTY_UNKNOWN;
   if (_mediaPlayerProxy->hasMediaPlayerDeviceConnections())
   {
      ::MPlay_fi_types::T_MPlayDeviceInfo deviceInfo = _mediaPlayerProxy->getMediaPlayerDeviceConnections().getODeviceInfo();
      ::MPlay_fi_types::T_MPlayDeviceInfo::iterator itr = deviceInfo.begin();
      for (; itr != deviceInfo.end(); ++itr)
      {
         if (itr->getU8DeviceTag() == deviceTag)
         {
            deviceType = itr->getE8DeviceType();
            break;
         }
      }
   }
   ETG_TRACE_USR2(("tclMediaClientHandler::GetMediaDeviceType:getdevicetype:%d ", deviceType));
#endif
   return deviceType;
}


/* ***********************************************************
* Function 		: GetActiveUSBSource
* Description 	: called if get onactivesourceupdate for SRC_MEDIA_PLAYER to remember which USB to highlight
* Parameters 	: SubSrc_id
* Return value	: int
* ***********************************************************/
int tclMediaClientHandler::GetActiveUSBSource(int SubSrc_id)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::GetActiveSourceHighlight mSubSrc_id  is %d", SubSrc_id));
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

   if ((SubSrc_id == m_StrDevice_USB2.u8DeviceTag))
   {
      m_deviceusbnum = USB_TWO;
      m_StrDevice_USB2.bIsActive = TRUE;
      m_StrDevice_USB.bIsActive = FALSE;
      m_StrDevice_Apple_USB.bIsActive = FALSE;
      m_StrDevice_BT.bIsActive = FALSE;
      m_StrDevice_AUX.bIsActive = FALSE;
   }
   else if ((SubSrc_id == m_StrDevice_USB.u8DeviceTag))
   {
      m_deviceusbnum = USB_ONE;
      m_StrDevice_USB.bIsActive = TRUE;
      m_StrDevice_USB2.bIsActive = FALSE;
      m_StrDevice_Apple_USB.bIsActive = FALSE;
      m_StrDevice_BT.bIsActive = FALSE;
      m_StrDevice_AUX.bIsActive = FALSE;
   }
   else if (SubSrc_id == m_StrDevice_Apple_USB.u8DeviceTag)
   {
      m_StrDevice_Apple_USB.bIsActive = TRUE;
      m_StrDevice_USB.bIsActive = FALSE;
      m_StrDevice_USB2.bIsActive = FALSE;
      m_StrDevice_BT.bIsActive = FALSE;
      m_StrDevice_AUX.bIsActive = FALSE;
   }
   else
   {
      ETG_TRACE_USR1(("tclMediaClientHandler::GetActiveUSBSource invalid subscrcid"));
   }
#endif
   return m_deviceusbnum;
}


#ifdef VARIANT_S_FTR_ENABLE_UNITTEST

void tclMediaClientHandler::setMediaDeviceType(int iDeviceType)
{
   m_deviceType = iDeviceType;
}


void tclMediaClientHandler::setActiveUSBSource(int iUSbDevNum)
{
   m_deviceusbnum = iUSbDevNum;
}


#else
void tclMediaClientHandler::onTrackCommandChangeRequest(uint32 u8trackCommandValue)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onTrackCommandChangeRequest trackCommandValue: %d", u8trackCommandValue));

   tCU8 u8NextPrevSkipCount = DEFAULT_SKIP_COUNT;

   switch (u8trackCommandValue)
   {
      case TrackCommand_Pause:
      {
         vSetPlaybackAction(FI_EN_E8PBA_PAUSE, u8NextPrevSkipCount);
      }
      break;
      case TrackCommand_Play:
      {
         vSetPlaybackAction(FI_EN_E8PBA_PLAY, u8NextPrevSkipCount);
      }
      break;
      case TrackCommand_RepeatTrack:
      {
         _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_ONE);
      }
      break;
      case TrackCommand_RepeatAll:
      {
         _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_LIST);
      }
      break;
      case TrackCommand_RandomAll:
      {
         _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_RANDOM);//Sh List
      }
      break;
      case TrackCommand_RandomRepeatOFF:
      {
         _mediaPlayerProxy->sendRepeatModeSet(*this, ::MPlay_fi_types::T_e8_MPlayRepeat__e8RPT_NONE); //Rep Off
         _mediaPlayerProxy->sendPlaybackModeSet(*this, ::MPlay_fi_types::T_e8_MPlayMode__e8PBM_NORMAL); //Sh Off
      }
      break;
      case TrackCommand_FastForward:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FFWD_START, u8NextPrevSkipCount);
         m_fastForwardOnGoing = true;
      }
      break;
      case TrackCommand_FastBackward:
      {
         vSetPlaybackAction(FI_EN_E8PBA_FREV_START, u8NextPrevSkipCount);
         m_fastBackwardOnGoing = true;
      }
      break;
      case TrackCommand_Next:
      {
         vSetPlaybackAction(FI_EN_E8PBA_NEXT, u8NextPrevSkipCount);
      }
      break;
      case TrackCommand_Previous:
      {
         vSetPlaybackAction(FI_EN_E8PBA_PREV, u8NextPrevSkipCount);
      }
      break;
      case TrackCommand_NotAvailable:
      {
         if (m_fastForwardOnGoing == true)
         {
            vSetPlaybackAction(FI_EN_E8PBA_FFWD_STOP, u8NextPrevSkipCount);
            m_fastForwardOnGoing = false;
         }
         else if (m_fastBackwardOnGoing == true)
         {
            vSetPlaybackAction(FI_EN_E8PBA_FREV_STOP, u8NextPrevSkipCount);
            m_fastBackwardOnGoing = false;
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("tclMediaClientHandler:onTrackCommandChangeRequest Default case option"));
      }
      break;
   }
}


#endif
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
bool tclMediaClientHandler::onCourierMessage(const onHardKeyPress& msg)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onHardKeyPress HKCode: %d %d", msg.GetHKCode(), msg.GetHKState()));
   int hardKeyCode = msg.GetHKCode();
   int hardkeyState = msg.GetHKState();
   bool retVal = vRequestRemoteCommands(hardkeyState, hardKeyCode);
   return retVal;
}


bool tclMediaClientHandler::vRequestRemoteCommands(uint32 hardkeyPress, uint32 hardKeyValue)
{
   ETG_TRACE_USR4(("tclMediaClientHandler::onHardKeyPress HKValue: %d %d", hardkeyPress, hardKeyValue));
   tCU8 u8NextPrevSkipCount = DEFAULT_SKIP_COUNT;
   bool retValue = true;

   switch (hardKeyValue)
   {
      case HARDKEYCODE_REMOTECONTROL_NEXT:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)   //button released value(1)
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::onHardKeyPress of next,this function will call next song"));
            vSetPlaybackAction(FI_EN_E8PBA_NEXT, u8NextPrevSkipCount);
         }
         else
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::onHardKeyPress invalid key state = %d", hardkeyPress));
         }
      }
      break;
      case HARDKEYCODE_REMOTECONTROL_PREVIOUS:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)   //button released value(1)
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::onHardKeyPress of previous,this function will call previous song"));
            vSetPlaybackAction(FI_EN_E8PBA_PREV, u8NextPrevSkipCount);
         }
         else
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::onHardKeyPress invalid key state = %d", hardkeyPress));
         }
      }
      break;
      case HARDKEYCODE_REMOTECONTROL_MEDIA:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)
         {
            ETG_TRACE_USR1(("tclMediaClientHandler::onHardKeyPress of media,this function will call sourceselection or nowplaying based on activesource"));
            POST_MSG((COURIER_MESSAGE_NEW(ButtonMainSrcActivate)()));
         }
         else
         {
            ETG_TRACE_USR4(("tclMediaClientHandler::onHardKeyPress invalid key state = %d", hardkeyPress));
         }
      }
      break;

      case HARDKEYCODE_REMOTECONTROL_TRAFFIC:
      {
         if (Enum_hmibase_HARDKEYSTATE_UP == hardkeyPress)
         {
            ETG_TRACE_USR3(("tclMediaClientHandler::onHardKeyPress on Traffic button pressed in setting screen traffic button get checked"));
            _taActiveValue = _taActiveValue ? 0 : 1;

            HMIModelComponent::getInstance().vUpdateTACheckbox(_taActiveValue);
            ETG_TRACE_USR4(("HMIModelComponent::onHardKeyPress Traffic Announcement : Is Checked? : %d", _taActiveValue));
         }
         else
         {
            ETG_TRACE_USR3(("tclMediaClientHandler::onHardKeyPress HARDKEYCODE_REMOTECONTROL_TRAFFIC invalid key state = %d", hardkeyPress));
         }
      }
      break;
      default:
      {
         retValue = false;
         break;
      }
   }
   return retValue;
}


void tclMediaClientHandler::setTAActiveStatus(int taActiveStatus)
{
   _taActiveValue = taActiveStatus ;
}


#endif
}//end of namespace Core
}//end of namespace App
