/**************************************************************************************
 * @file         : VerticalListManager.cpp
 * @author       : RBEI/ECH2-INF4CV_MediaTeam
 * @addtogroup   : AppHmi_media
 * @brief        :
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 * 		  			Initial Revision.
 **************************************************************************************/

#include "hall_std_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "Core/ListHandling/ListContainer/ListContainer.h"
#include "Core/ListHandling/VerticalListManager/VerticalListManager.h"
#include "Core/ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "DataModel/HMIModelListImplementation.h"
#include "Core/ListHandling/EMPListInterfaceHandler/EMPListInterfaceHandler.h"
#include "Core/ListHandling/EMPListPlaybackHandler/EMPListPlaybackHandler.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "Core/ListHandling/EMPListInterfaceHandler/IEMPListHandler.h"
#include "Core/ListHandling/EMPListInterfaceHandler/EMPListFactory.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "Core/Utils/DbDownloadMap.h"
#include <vector>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX                 App::Core::VerticalListManager::
#include "trcGenProj/Header/VerticalListManager.cpp.trc.h"
#endif


namespace App {
namespace Core {

//Instance which created to call TTfs command
VerticalListManager* VerticalListManager::_verticalListManagerinstance = NULL;


VerticalListManager::VerticalListManager(HorizontalListManager* horizontalListManagerObj
      , ExternalMediaSourceHandler* mediaSourceHandler, EnumConst::entSourceType sourceType)
   : _listInfoDBHandler(NULL)
   , _empListFactory(NULL)
   , _container(NULL)
   , _listConfiguration(NULL)
   , _listSession(NULL)
   , _queueHandler(NULL)
   , _empListInterfaceHandler(NULL)
   , _mockEMPListInterfaceHandler(NULL)
   , _empListPlaybackHandler(NULL)
   , _verticalListGuiResponseHandler(NULL)
{
   ETG_TRACE_USR1(("VerticalListManager parameterised constructor \n"));
   _currentSourceType = sourceType;

   _horizontalListManager = horizontalListManagerObj;
   _externalMediaSourceHandler = mediaSourceHandler;

   _verticalListManagerinstance = this;
   initializeInstances();
   initializeMemberVariables();
   registerObservors();
   _absoluteIndex1 = 0;
   _absoluteIndex2 = 0;
   _currentFolderPath = "/";
   _selectedRow = 0;
   _fileType = entFileType__entFileType_unknown;
}


VerticalListManager::~VerticalListManager()
{
   ETG_TRACE_USR1(("VerticalListManager Destructor \n"));
   deregisterObservors();
   clearInstance();
}


void VerticalListManager::registerObservors()
{
   ETG_TRACE_USR4(("VerticalListManager:: registerObservors "));

   if (NULL != _horizontalListManager)
   {
      _horizontalListManager->registerObserver((*this));
   }
}


void VerticalListManager::deregisterObservors()
{
   ETG_TRACE_USR4(("VerticalListManager:: deregisterObservors "));

   if (NULL != _horizontalListManager)
   {
      _horizontalListManager->deregisterObserver((*this));
   }
}


void VerticalListManager::initializeMemberVariables()
{
   ETG_TRACE_USR1(("VerticalListManager initializeMemberVariables \n"));
   _currentListType = EnumConst::entListType_Invalid;
   m_PrevListType = EnumConst::entListType_Invalid;
   m_windowSize = WINDOW_SIZE;
   m_bMockActive = false;
   m_categorySelectionUpdateReceived = false;
   m_verticalStartIndex = 0;
   m_CurrentWindowSize = 20;
   m_mediaState = MEDIA_STATE_INVALID;
   if (_listSession != NULL)
   {
      _listSession->resetSession();
   }
}


void VerticalListManager::initializeInstances()
{
   ETG_TRACE_USR1(("VerticalListManager initializeInstances \n"));

   if (NULL == _listInfoDBHandler)
   {
      _listInfoDBHandler = new ListInfoDatabaseHandler();
   }

   if (NULL == _empListFactory)
   {
      _empListFactory = new EMPListFactory();
   }

   if (NULL == _container)
   {
      _container = new ListContainer();
   }

   if (NULL == _listConfiguration)
   {
      _listConfiguration = new ListConfiguration();
   }

   if (NULL == _listSession)
   {
      _listSession = new ListSession();
   }

   if (NULL == _queueHandler)
   {
      _queueHandler = new ListQueueHandler(_listInfoDBHandler);
   }

   if (NULL == _empListInterfaceHandler)
   {
      _empListInterfaceHandler = new EMPListInterfaceHandler(this, _container, _listInfoDBHandler, _currentSourceType);
      _empListFactory->setInstanceForEMP(_empListInterfaceHandler);
   }

   if (NULL == _mockEMPListInterfaceHandler)
   {
      _mockEMPListInterfaceHandler = new MockEMPListInterfaceHandler(this, _container, _listInfoDBHandler);
      _empListFactory->setInstanceForMockEMP(_mockEMPListInterfaceHandler);
   }

   if (NULL == _verticalListGuiResponseHandler)
   {
      _verticalListGuiResponseHandler = new VerticalListGuiResponseHandler(_container, _listConfiguration);
   }

   if (NULL == _empListPlaybackHandler)
   {
      _empListPlaybackHandler = new EMPListPlaybackHandler(this, _currentSourceType, _externalMediaSourceHandler);
   }
}


void VerticalListManager::clearInstance()
{
   ETG_TRACE_USR1(("VerticalListManager clearInstance \n"));

   if (NULL != _listInfoDBHandler)
   {
      delete _listInfoDBHandler;
      _listInfoDBHandler = NULL;
   }

   if (NULL != _listConfiguration)
   {
      delete _listConfiguration;
      _listConfiguration = NULL;
   }

   if (NULL != _container)
   {
      delete _container;
      _container = NULL;
   }

   if (NULL != _listSession)
   {
      delete _listSession;
      _listSession = NULL;
   }

   if (NULL != _queueHandler)
   {
      delete _queueHandler;
      _queueHandler = NULL;
   }

   if (NULL != _empListInterfaceHandler)
   {
      delete _empListInterfaceHandler;
      _empListInterfaceHandler = NULL;
   }

   if (NULL != _mockEMPListInterfaceHandler)
   {
      delete _mockEMPListInterfaceHandler;
      _mockEMPListInterfaceHandler = NULL;
   }

   if (NULL != _verticalListGuiResponseHandler)
   {
      delete _verticalListGuiResponseHandler;
      _verticalListGuiResponseHandler = NULL;
   }

   if (NULL != _empListPlaybackHandler)
   {
      delete _empListPlaybackHandler;
      _empListPlaybackHandler = NULL;
   }

   if (NULL != _verticalListManagerinstance)
   {
      _verticalListManagerinstance = NULL;
   }
}


/* ***********************************************************
* Function 		: onCategorySelection
* Description 	: While selecting the tab from horizontal list tab
* Parameters 	:listType
* Return value	: void
* ***********************************************************/
void VerticalListManager::onCategorySelection(EnumConst::entListType listType)
{
   ETG_TRACE_USR4(("VerticalListManager::onCategorySelection:listType %d", ETG_CENUM(entListType, listType)));
   m_categorySelectionUpdateReceived = true;
   m_PrevListType = _currentListType;

   ETG_TRACE_USR4(("VerticalListManager::onCategorySelection:m_PrevListType %d", ETG_CENUM(entListType, m_PrevListType)));
   if (m_PrevListType == EnumConst::entListType_Video_Languages)
   {
      IEMPListHandler* iEMPListHandler = _empListFactory->getEMPListInterfaceObject(m_bMockActive);
      iEMPListHandler->updateSelectedLanguageList();
   }
   if (listType == EnumConst::entListType_Folder)
   {
      _currentFolderPath = "/";
      _container->updateFolderpath(_currentFolderPath);
   }
   if (listType != EnumConst::entListType_Invalid)
   {
      enListState currentListState = _listInfoDBHandler->getListStateOnListDataRequest(listType);
      ETG_TRACE_USR4(("VerticalListManager::onCategorySelection currentListState %d, ", ETG_CENUM(enListState, currentListState)));
      if (currentListState != EN_LIST_STATE_PROCESSING)
      {
         _currentListType = listType;
         m_verticalStartIndex = 0;
         onListDataRequest(LIST_ID_EXTERNAL_MEDIA_VER, m_verticalStartIndex, m_windowSize);
      }
      else
      {
         updateToQueue(0, m_CurrentWindowSize);
      }
   }
}


void VerticalListManager::processQueuedListRequest()
{
   ETG_TRACE_USR1(("VerticalListManager::processQueuedListRequest Entered"));
   struct queueListInfo queuedListInfo = _queueHandler->getQueuedListInfo();
   _queueHandler->clearListQueue();
   _currentListType = queuedListInfo.listType;
   onListDataRequest(LIST_ID_EXTERNAL_MEDIA_VER, queuedListInfo.startIndex, queuedListInfo.windowSize);
}


void VerticalListManager::resetListState()
{
   ETG_TRACE_USR1(("VerticalListManager::resetListState Called"));
   _currentListType = EnumConst::entListType_Invalid;
   _currentSourceType = EnumConst::entSourceType_DEFAULT;
   m_categorySelectionUpdateReceived = false;
   m_verticalStartIndex = 0;
   m_CurrentWindowSize = 0;
//  m_CurrentActiveSrcId = SRC_INVALID;
   _listSession->resetSession();
   _queueHandler->clearListQueue();
   _listInfoDBHandler->updateListStateOnDeviceDisconnection();
   _container->resetMembers();
   _horizontalListManager->resetMembers();
}


bool VerticalListManager::onCourierMessage(const CurrentMediaStateUpdMsg& msg)
{
   int iActiveSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("VerticalListManager::onCourierMessage CurrentMediaStateUpdMsg  msg.GetCurrentState():%d   iActiveSrcId  %d", msg.GetCurrentState(), iActiveSrcId));

   m_mediaState = msg.GetCurrentState();
   if (iActiveSrcId == SRC_CAMERAPORT1_USB_VIDEO || iActiveSrcId == SRC_CAMERAPORT1_USB_AUDIO ||  iActiveSrcId == SRC_CMR1_USB1_VIDEO)
   {
      if (msg.GetCurrentState() != MEDIA_STATE_LIST)
      {
         _queueHandler->clearListQueue();
         _listInfoDBHandler->updateListStateOnListExit();
      }
   }

   return false;
}


bool VerticalListManager::onCourierMessage(const MockCreateListandSliceListResMsg& msg)
{
   ETG_TRACE_USR1(("VerticalListManager::MockCreateListandSliceListResMsg"));
   _mockEMPListInterfaceHandler->onCourierMessage(msg);
   return false;
}


/* ***********************************************************
* Function 		: onListDataRequest
* Description 	: Updates the List data Based on the List Type
* Parameters 	: listId,startIndex,windowSize
* Return value	: void
* ***********************************************************/
void VerticalListManager::onListDataRequest(uint16 listId, uint16 startIndex, uint16 windowSize)
{
   ETG_TRACE_USR4(("VerticalListManager::onListDataRequest _currentListType %d, ", ETG_CENUM(entListType, _currentListType)));
   ETG_TRACE_USR4(("VerticalListManager::onListDataRequest listId %u, startindex %d, windowSize %u", listId, startIndex, windowSize));

   _listSession->setNextListSession(LIST_SESSION_GUI_REQUEST_RECEIVED);

   if (_currentListType != EnumConst::entListType_Invalid)
   {
      enListState currentListState = _listInfoDBHandler->getListStateOnListDataRequest(_currentListType);
      ETG_TRACE_USR4(("VerticalListManager::onListDataRequest currentListState %d, ", currentListState));

      m_verticalStartIndex = startIndex;
      m_CurrentWindowSize = windowSize;
      _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_PROCESSING);
      getListData(currentListState);
   }
   else
   {
      ETG_TRACE_USR1(("VerticalListManager::onListDataRequest listtype invalid case"));
      _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_SENT);
      _verticalListGuiResponseHandler->updateEmptyListData(_currentListType);
      _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_ACTIVE);
   }
}


void VerticalListManager::updateToQueue(int startIndex, int windowSize)
{
   _queueHandler->clearListQueue();
   struct queueListInfo queuedListInfo;
   queuedListInfo.listType = _currentListType;
   queuedListInfo.startIndex = startIndex;
   queuedListInfo.windowSize = windowSize;
   _queueHandler->queueListRequest(queuedListInfo);
}


void VerticalListManager::getListData(enListState currentListState)
{
   ETG_TRACE_USR4(("VerticalListManager::getListData listState %d, ", currentListState));
   if (currentListState == EN_LIST_STATE_UNKNOWN || currentListState == EN_LIST_STATE_INACTIVE)
   {
      sendListRequest();
   }
   else if (currentListState == EN_LIST_STATE_ACTIVE)
   {
      int totalListSize = _listInfoDBHandler->getCurrentListSize(_currentListType);
      ETG_TRACE_USR4(("VerticalListManager::getListData totalListSize %d, ", totalListSize));
      if (totalListSize > 0)
      {
         _listSession->setNextListSession(LIST_SESSION_SLICE_REQUEST);
         IEMPListHandler* iEMPListHandler = _empListFactory->getEMPListInterfaceObject(m_bMockActive);

         if (_currentListType == EnumConst::entListType_Folder)
         {
            ETG_TRACE_USR4(("VerticalListManager getListData _currentFolderPath  %s  ", _currentFolderPath.c_str()));
            iEMPListHandler->sliceListRequest(m_verticalStartIndex, getListTypeWindowSize(), _currentFolderPath);
         }
         else
         {
            iEMPListHandler->sliceListRequest(_currentListType, m_verticalStartIndex, getListTypeWindowSize(), _absoluteIndex1, _absoluteIndex2);
         }
         //_empListInterfaceHandler->sliceListRequest(_currentListType, m_verticalStartIndex, m_CurrentWindowSize);
      }
      else
      {
         _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_SENT);
         _verticalListGuiResponseHandler->updateEmptyListData(_currentListType);
         _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_ACTIVE);
      }
   }
}


void VerticalListManager::onListItemPressed(int selectedRow, int selectedColumn)
{
   _selectedRow = selectedRow;
   uint16 absoluteIndex = _container->getAbsoluteIndex(selectedRow);
   EnumConst::entListType nextListType = EnumConst::entListType_Folder;
   if (_currentListType != EnumConst::entListType_Folder)
   {
      nextListType = _listConfiguration->getNextListType(_currentListType);
   }
   ETG_TRACE_USR4(("VerticalListManager::onListItemPressed Entered activeListType %d  selectedRow %d  selectedColumn %d absoluteIndex %d", ETG_CENUM(entListType, _currentListType), selectedRow, ETG_CENUM(entListType, selectedColumn), absoluteIndex));
   ETG_TRACE_USR4(("VerticalListManager::onListItemPressed Entered subListType %d ", ETG_CENUM(entListType, nextListType)));

   _fileType = _container->getFileType(selectedRow);

   if (selectedColumn == EnumConst::entListType_Video_Languages)
   {
      onVideoLanguageListItemPressed(selectedRow, selectedColumn);
   }
   else
   {
      enListState currentListState = _listInfoDBHandler->getListStateOnListDataRequest(_currentListType);
      if (currentListState == EN_LIST_STATE_PROCESSING)
      {
         updateToQueue(0, m_CurrentWindowSize);
      }
      else
      {
         if (nextListType == EnumConst::entListType_Invalid)
         {
            onPlayableItemSelection(_currentListType, absoluteIndex);
         }
         else if (_currentListType == EnumConst::entListType_Folder)
         {
            if (_fileType == entFileType__entFileType_Folder)
            {
               _currentFolderPath = getNextFolderPath(_currentFolderPath);
               requestToLoadSubList(_currentListType);
            }
            else
            {
               onPlayableItemSelection(_currentListType, absoluteIndex);
            }
         }
         else
         {
            _currentListType = nextListType;
            requestToLoadSubList(nextListType);
         }
      }
   }
}


void VerticalListManager::onSourceAllocationListUpdate()
{
   ETG_TRACE_USR4(("VerticalListManager:: onSourceAllocationListUpdate "));
   _empListPlaybackHandler->onSourceAllocationUpdate();
}


void VerticalListManager::onPlayableItemSelection(EnumConst::entListType subListType, uint16 absoluteIndex)
{
   ETG_TRACE_USR4(("VerticalListManager::onPlayableItemSelection subListType:  %d    absoluteIndex  %d", ETG_CENUM(entListType, subListType), absoluteIndex));
   uint8 activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   int requestedSrcId = _listConfiguration->requestedSourceId(_currentListType, _currentSourceType, _fileType);
   int currentActiveSrcId = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("VerticalListManager::onPlayableItemSelection requestedSrcId:  %d    currentActiveSrcId  %d", requestedSrcId, currentActiveSrcId));

   if (requestedSrcId != currentActiveSrcId)
   {
      _empListPlaybackHandler->playItemRequest(subListType, absoluteIndex, false, _currentFolderPath, _fileType);
      int activeDeviceType = getDeviceType();
      IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(activeDeviceType);
      if (iExternalMediaSourceActivationHandler)
      {
         StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_SONG_SELECT_IN_LIST);
         iExternalMediaSourceActivationHandler->toggleSource();
         iExternalMediaSourceActivationHandler->requestSourceActivation(activeRegion);
      }
   }
   else
   {
      _empListPlaybackHandler->playItemRequest(subListType, absoluteIndex, true, _currentFolderPath, _fileType);
   }
}


void VerticalListManager::requestToLoadSubList(EnumConst::entListType subListType)
{
   enListState currentListState = _listInfoDBHandler->getListStateOnListDataRequest(subListType);
   ETG_TRACE_USR4(("VerticalListManager::requestToLoadSubList  nextListType %d   currentListState  %d   ", ETG_CENUM(entListType, subListType), currentListState));
   m_verticalStartIndex = 0;
   if (currentListState == EN_LIST_STATE_INACTIVE)
   {
      //For slice request
      onListCreateResponse();
   }
   else if (currentListState == EN_LIST_STATE_ACTIVE)
   {
      //For creating the list
      updateAbsoluteIndex(_currentListType);
      sendListRequest();
   }
}


void VerticalListManager::sendListRequest()
{
   ETG_TRACE_USR4(("VerticalListManager::sendListRequest entered:_currentListType %d", _currentListType));
   ETG_TRACE_USR4(("VerticalListManager sendListRequest _currentFolderPath  %s  ", _currentFolderPath.c_str()));
   // uint16 absoluteIndex = 0;
   _listSession->setNextListSession(LIST_SESSION_CREATE_REQUEST);
   IEMPListHandler* iEMPListHandler = _empListFactory->getEMPListInterfaceObject(m_bMockActive);

   if (_currentListType == EnumConst::entListType_Folder)
   {
      iEMPListHandler->createListRequest(_currentListType, _currentFolderPath);
      setFileInfoStr(_currentFolderPath);
   }
   else
   {
      iEMPListHandler->createListRequest(_currentListType, _absoluteIndex1, _absoluteIndex2);
      setFileInfoStr("Browse");
   }

   //_empListInterfaceHandler->createListRequest(_currentListType, absoluteIndex);
}


void VerticalListManager::onListCreateResponse()
{
   uint32 currentListSize = _listInfoDBHandler->getCurrentListSize(_currentListType);
   ETG_TRACE_USR4(("VerticalListManager::onListCreateResponse currentListSize:%d ", currentListSize));
   if (currentListSize > 0)
   {
      _listSession->setNextListSession(LIST_SESSION_SLICE_REQUEST);
      IEMPListHandler* iEMPListHandler = _empListFactory->getEMPListInterfaceObject(m_bMockActive);
      HMIModelListImplementation::getInstance().setWaitAnimationState(ANIMATION_START);

      if (_currentListType == EnumConst::entListType_Folder)
      {
         ETG_TRACE_USR4(("VerticalListManager onListCreateResponse _currentFolderPath  %s  ", _currentFolderPath.c_str()));
         iEMPListHandler->sliceListRequest(m_verticalStartIndex, getListTypeWindowSize(), _currentFolderPath);
      }
      else
      {
         iEMPListHandler->sliceListRequest(_currentListType, m_verticalStartIndex, getListTypeWindowSize(), _absoluteIndex1, _absoluteIndex2);
         //_empListInterfaceHandler->sliceListRequest(_currentListType, m_verticalStartIndex, m_CurrentWindowSize);
      }
   }
   else
   {
      _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_SENT);
      _verticalListGuiResponseHandler->updateEmptyListData(_currentListType);
      _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_ACTIVE);
   }
}


void VerticalListManager::onListLoadingComplete()
{
   _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_ACK);
   _verticalListGuiResponseHandler->updateListInfoText(false);
   _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_ACTIVE);

   if (_queueHandler->isListRequestQueued() && _queueHandler->isRequestValid(_currentListType))
   {
      processQueuedListRequest();
   }
}


void VerticalListManager::onListSliceResponse()
{
   uint32 currentListSize = _listInfoDBHandler->getCurrentListSize(_currentListType);
   ETG_TRACE_USR4(("VerticalListManager::onListSliceResponse currentListSize:%d ", currentListSize));
   if (_currentListType == EnumConst::entListType_Video_Languages)
   {
      calculateAllLanguageButtonStatus();
   }
   _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_SENT);
   _verticalListGuiResponseHandler->updateListContent(_currentListType, m_verticalStartIndex, currentListSize);

   onListLoadingComplete();
}


void VerticalListManager::calculateAllLanguageButtonStatus()
{
   std::vector<bool> languageSelStatus = _container->getIsLanguageSelected();
   bool min = *min_element(languageSelStatus.begin(), languageSelStatus.end());
   bool max = *max_element(languageSelStatus.begin(), languageSelStatus.end());
   bool allLanguageButtonStatus = (min == max) ? min : (min & max);
   ETG_TRACE_USR1(("VerticalListManager::calculateAllLanguageButtonStatus: allLanguageButtonStatus: %d", allLanguageButtonStatus));
   _container->updateIsAllLanguageSelected(allLanguageButtonStatus);
}


EnumConst::entListType VerticalListManager::getActiveListType()
{
   ETG_TRACE_USR4(("VerticalListManager::getActiveListType _currentListType:%d ", ETG_CENUM(entListType, _currentListType)));
   return _currentListType;
}


void VerticalListManager::onPreviousButtonPressed()
{
   if (_currentListType != EnumConst::entListType_Folder)
   {
      EnumConst::entListType  PrevListType  = _listConfiguration->getPreviousListType(_currentListType);
      if (PrevListType != EnumConst::entListType_Invalid)
      {
         updateList(PrevListType);
      }
      ETG_TRACE_USR4(("VerticalListManager::onPreviousButtonPressed PrevListType  %d", ETG_CENUM(entListType, PrevListType)));
   }
   else
   {
      _currentFolderPath = getPreviousFolderPath(_currentFolderPath);
      updateList(_currentListType);
   }
   ETG_TRACE_USR4(("VerticalListManager::onPreviousButtonPressed _currentListType:%d ", ETG_CENUM(entListType, _currentListType)));
}


void VerticalListManager::onAllButtonPressed()
{
   EnumConst::entListType allButtonListType = _listConfiguration->getAllListType(_currentListType);
   ETG_TRACE_USR4(("VerticalListManager::onAllButtonPressed _currentListType:%d   allButtonListType  %d", ETG_CENUM(entListType, _currentListType), ETG_CENUM(entListType, allButtonListType)));

   if (allButtonListType != EnumConst::entListType_Invalid)
   {
      updateList(allButtonListType);
   }
}


void VerticalListManager::updateList(EnumConst::entListType requestedListType)
{
   enListState currentListState = _listInfoDBHandler->getListStateOnListDataRequest(_currentListType);
   ETG_TRACE_USR4(("VerticalListManager::updateList _currentListType:%d   requestedListType  %d", ETG_CENUM(entListType, _currentListType), ETG_CENUM(entListType, requestedListType)));
   ETG_TRACE_USR4(("VerticalListManager::updateList currentListState:%d ", ETG_CENUM(enListState, currentListState)));

   if (currentListState == EN_LIST_STATE_PROCESSING)
   {
      updateToQueue(0, m_CurrentWindowSize);
   }
   else
   {
      _currentListType = requestedListType;
      requestToLoadSubList(_currentListType);
   }
}


/**
* Trace command to initiate Mock list interface
*
* @param[in]      : void
* @return         : void
*/
ETG_I_CMD_DEFINE((TraceCmd_InitiateMockClass, "TraceCmd_InitiateMockClass %d", bool))
void VerticalListManager::TraceCmd_InitiateMockClass(bool state)
{
   ETG_TRACE_USR4(("VerticalListManager::TraceCmd_InitiateMockClass: %d", state));
   if (_verticalListManagerinstance)
   {
      _verticalListManagerinstance->setMockInterfaceState(state);
   }
}


void VerticalListManager::setMockInterfaceState(bool state)
{
   ETG_TRACE_USR4(("setMockInterfaceState:: state %d", state));
   m_bMockActive = state;
}


int VerticalListManager::getDeviceType()
{
   int deviceType = FI_EN_E8DTY_UNKNOWN;

   if (_currentSourceType == EnumConst::entSourceType_CMP)
   {
      deviceType = FI_EN_E8DTY_USB_CAMERAPORT;
   }
   else if (_currentSourceType == EnumConst::entSourceType_CMR)
   {
      deviceType = FI_EN_E8DTY_ROUTER_USB;
   }
   ETG_TRACE_USR4(("VerticalListManager::getDeviceType: deviceType: %d", deviceType));
   return deviceType;
}


void VerticalListManager::onListErrorResponse()
{
   ETG_TRACE_USR4(("VerticalListManager::onListErrorResponse: %d", _currentListType));

   _listSession->setNextListSession(LIST_SESSION_GUI_RESPONSE_SENT);
   _verticalListGuiResponseHandler->updateEmptyListData(_currentListType);
   _listInfoDBHandler->updateListState(_currentListType, EN_LIST_STATE_UNKNOWN);
}


void VerticalListManager::updateAbsoluteIndex(EnumConst::entListType nextListType)
{
   switch (nextListType)
   {
      case EnumConst::entListType_Genre_Album:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Genre, _selectedRow);
         break;
      }
      case EnumConst::entListType_Genre_Album_Song:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Genre, _selectedRow);
         _absoluteIndex2 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Genre_Album, _selectedRow);
         break;
      }
      case EnumConst::entListType_Genre_Song:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Genre, _selectedRow);
         break;
      }
      case EnumConst::entListType_Artist_Album:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Artist, _selectedRow);
         break;
      }
      case EnumConst::entListType_Artist_Album_Song:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Artist, _selectedRow);
         _absoluteIndex2 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Artist_Album, _selectedRow);
         break;
      }
      case EnumConst::entListType_Artist_Song:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Artist, _selectedRow);
         break;
      }
      case EnumConst::entListType_Album_Song :
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Album, _selectedRow);
         break;
      }
      case EnumConst::entListType_Video_episode:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Video, _selectedRow);
         break;
      }
      case EnumConst::entListType_Playlist_song:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Playlist, _selectedRow);
         break;
      }
      case EnumConst::entListType_Folder_file:
      {
         _absoluteIndex1 = _listInfoDBHandler->getAbsoluteIndex(EnumConst::entListType_Folder, _selectedRow);
         break;
      }
   }
}


int VerticalListManager::getListTypeWindowSize()
{
   int windowSize = ((_listConfiguration->getListSliceReqType(_currentListType)) == EN_WINDOW_SLICE_REQ_TYPE) ? m_CurrentWindowSize : (_listInfoDBHandler->getCurrentListSize(_currentListType));
   ETG_TRACE_USR1(("VerticalListManager::getListTypeWindowSize: windowSize %d", windowSize));
   return windowSize;
}


void VerticalListManager::onVideoLanguageListItemPressed(int32 Hdl, int32 subHdl)
{
   ETG_TRACE_COMP(("VerticalListManager::onVideoLanguageListItemPressed: Hdl %d, subHdl %d", Hdl, subHdl));
   uint32 currentListSize = _listInfoDBHandler->getCurrentListSize(_currentListType);
   std::vector<bool> languageSelStatus = _container->getIsLanguageSelected();
   std::vector<bool>::iterator itr = languageSelStatus.begin();
   if (Hdl == 0)
   {
      _container->updateIsAllLanguageSelected(!(_container->isAllLanguageSelected()));
      for (int index = 0; itr != languageSelStatus.end(); itr++, index++)
      {
         languageSelStatus[index] = _container->isAllLanguageSelected();
      }
      _container->setIsLanguageSelected(languageSelStatus);
   }
   else
   {
      for (int index = 0; itr != languageSelStatus.end(); itr++, index++)
      {
         if (index == Hdl - 1)
         {
            languageSelStatus[index] = !(languageSelStatus[index]);
            break;
         }
      }
      _container->setIsLanguageSelected(languageSelStatus);
      calculateAllLanguageButtonStatus();
   }
   _verticalListGuiResponseHandler->updateListContent(_currentListType, m_verticalStartIndex, currentListSize);
}


void VerticalListManager::onExternalBrowseScreenExit()
{
   ETG_TRACE_COMP(("VerticalListManager::onExternalBrowseScreenExit: _currentListType = %d", ETG_CENUM(entListType, _currentListType)));
   IEMPListHandler* iEMPListHandler = _empListFactory->getEMPListInterfaceObject(m_bMockActive);
   if (_currentListType == EnumConst::entListType_Video_Languages)
   {
      iEMPListHandler->updateSelectedLanguageList();
   }
}


std::string VerticalListManager::getNextFolderPath(std::string strPath)
{
   ETG_TRACE_COMP(("VerticalListManager::getNextFolderPath"));
   std::string fileName = _container->getFolderName(_selectedRow);
   std::string slash = "/";

   if (strPath != " ")
   {
      strPath = strPath + fileName.append(slash);
   }
   _container->updateFolderpath(strPath);
   ETG_TRACE_USR4(("VerticalListManager getNextFolderPath strPath  %s  ", strPath.c_str()));
   return strPath;
}


void VerticalListManager::setFileInfoStr(std::string _fileInfo)
{
   ETG_TRACE_USR4(("VerticalListManager setFileInfoStr _fileInfo  %s  ", _fileInfo.c_str()));

   if (_fileInfo.compare("Browse") == 0)//check if the string is browse, map it to ID
   {
      (*_browseText).mBrowseText = "Browse";//Candera::String(id45);
   }
   else
   {
      (*_browseText).mBrowseText = _fileInfo.c_str();
   }

   _browseText.MarkAllItemsModified();
   _browseText.SendUpdate();
}


std::string VerticalListManager::getPreviousFolderPath(std::string strPath)
{
   ETG_TRACE_USR4(("VerticalListManager::getPreviousFolderPath  strPath %s", strPath.c_str()));
   int count = 0;

   ::std::size_t length = strPath.find_last_of("/");
   ETG_TRACE_USR4(("VerticalListManager::getPreviousFolderPath  length %d", length));

   if (length  != ::std::string::npos)
   {
      if (strPath[length] == '/')
      {
         length--;
         count++;
      }

      for (int i = length; i >= 0; i--)
      {
         if (!strPath.empty())
         {
            if (strPath[i] == '/')
            {
               strPath = strPath.erase(strPath.size() - count);

               if (strPath.empty())
               {
                  strPath = "/";
               }
               break;
            }
            count++;
         }
         else
         {
            strPath = "/";
            break;
         }
      }
   }
   _container->updateFolderpath(strPath);
   ETG_TRACE_USR4(("VerticalListManager getPreviousFolderPath strPath  %s  ", strPath.c_str()));
   return strPath;
}


}// namespace Core
}// namespace App
