/**************************************************************************************
* @file         : ExternalMediaPlaybackHandler.cpp
* @author       : ECH - voa4kor
* @addtogroup   : AppHmi_media
* @brief        :
* @copyright    : (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "hall_std_if.h"
#include "Core/ExternalMediaPlaybackHandler/ExternalMediaPlaybackHandler.h"
#include "Core/PluginClientHandlerAppl/IPluginClientAppl.h"
#include "Core/PluginClientHandlerAppl/PluginClientFactory.h"
#include "../ApplicationSwitchClientHandler/ApplicationSwitchClientHandler.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include"Core/ExternalMediaSourceActivation/IExternalMediaSourceActivationHandler.h"
#include "Core/ExternalMediaSourceActivation/ExternalMediaSourceActivationFactory.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include "PluginConstants.h"
#include "MediaCameraPort.h"
#include "AppHmi_MasterBase/AudioInterface/AudioDefines.h"
#include "DataModel/HMIModelComponent.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::ExternalMediaPlaybackHandler::

#include "trcGenProj/Header/ExternalMediaPlaybackHandler.cpp.trc.h"
#endif


namespace App {
namespace Core {

ExternalMediaPlaybackHandler* ExternalMediaPlaybackHandler::_externalMediaPlaybackHandler = NULL;

ExternalMediaPlaybackHandler::ExternalMediaPlaybackHandler(): _playableContentMap(NULL)
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler constructor \n"));
   _externalMediaPlaybackHandler = NULL;
   PluginClientFactory::getPluginClient(SRC_CAMERAPORT1_USB_VIDEO);
   PluginClientFactory::getPluginClient(SRC_CMR1_USB1_VIDEO);

   initializeInstances();
   initializePlaybackActionMap();
   initializeExternalSourceTypeMap();
   initializeExternalSourceRepeatPlaybackActionMap();
   m_extPlaybackstate = EXT_MEDIA_PLAY;
   lastPlaybackState = EXT_MEDIA_PLAY;
   m_extRepeatShufflestate = EXT_NORMAL;

   u8OperationType = EN_MEDIA_PLAY;
   m_u8ActiveRegion = MAX_REG;
}


ExternalMediaPlaybackHandler::~ExternalMediaPlaybackHandler()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler Destructor \n"));

   clearInstance();
}


void ExternalMediaPlaybackHandler::setPlayableContentMapInstance(PlayableContentMap* _playableContentMapObject)
{
   if (_playableContentMapObject != NULL)
   {
      _playableContentMap = _playableContentMapObject;
      ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::setPlayableContentMapInstance _playableContentMapObject is Created"));
   }
   else
   {
      ETG_TRACE_ERR(("ExternalMediaPlaybackHandler::setPlayableContentMapInstance _playableContentMapObject is NULL"));
   }
}


void ExternalMediaPlaybackHandler::initializeInstances()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler initializeInstances \n"));
}


void ExternalMediaPlaybackHandler::clearInstance()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler clearInstance \n"));
}


ExternalMediaPlaybackHandler* ExternalMediaPlaybackHandler::getInstance()
{
   if (NULL == _externalMediaPlaybackHandler)
   {
      _externalMediaPlaybackHandler = new ExternalMediaPlaybackHandler();
   }
   return _externalMediaPlaybackHandler;
}


void ExternalMediaPlaybackHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::registerProperties"));
}


void ExternalMediaPlaybackHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::deregisterProperties"));
}


void ExternalMediaPlaybackHandler::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::onUnavailable"));
}


void ExternalMediaPlaybackHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ExternalMediaListDBManager::onAvailable"));
}


void ExternalMediaPlaybackHandler::initializePlaybackActionMap()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::initializePlaybackActionMap is entered \n"));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_PLAY, EXT_MEDIA_PAUSE));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_PAUSE, EXT_MEDIA_PLAY));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_NEXT, EXT_MEDIA_NEXT));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_PREV, EXT_MEDIA_PREVIOUS));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_FASTREWIND, MEDIA_VIDEOSOURCERESTORE));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_FASTREWIND_STOP, EXT_MEDIA_PLAY));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_FASTFORWARD, EXT_MEDIA_SOURCEACTIVE));
   _playbackActionMap.insert(pair<int, int>(EN_MEDIA_FASTFORWARD_STOP, EXT_MEDIA_PLAY));
}


void ExternalMediaPlaybackHandler::initializeExternalSourceTypeMap()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::initializeExternalSourceTypeMap is entered \n"));
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_AUDIO].srcType = EnumConst::entSourceType_CMP;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_AUDIO].deviceType = FI_EN_E8DTY_USB_CAMERAPORT;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_AUDIO].srcId = SRC_CAMERAPORT1_USB_AUDIO;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_VIDEO].srcType = EnumConst::entSourceType_CMP;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_VIDEO].deviceType = FI_EN_E8DTY_USB_CAMERAPORT;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMP_VIDEO].srcId = SRC_CAMERAPORT1_USB_VIDEO;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_AUDIO].srcType = EnumConst::entSourceType_CMR;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_AUDIO].deviceType = FI_EN_E8DTY_ROUTER_USB;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_AUDIO].srcId = SRC_CMR1_USB1_AUDIO;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_VIDEO].srcType = EnumConst::entSourceType_CMR;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_VIDEO].deviceType = FI_EN_E8DTY_ROUTER_USB;
   _externalSourceTypeMap[EnumConst::enSubSourceType_CMR_VIDEO].srcId = SRC_CMR1_USB1_VIDEO;
}


void ExternalMediaPlaybackHandler::initializeExternalSourceRepeatPlaybackActionMap()
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::initializeExternalSourceRepeatPlaybackActionMap is entered \n"));
   _repeatshufflePlaybackActionMap.insert(pair<int, int>(EN_MEDIA_REPEAT_OFF, EXT_REPEAT_TRACK));
   _repeatshufflePlaybackActionMap.insert(pair<int, int>(EN_MEDIA_REPEAT_TRACK, EXT_REPEAT_LIST));
   _repeatshufflePlaybackActionMap.insert(pair<int, int>(EN_MEDIA_REPEAT_LIST, EXT_REPEAT_SHUFFLE_OFF));
   _repeatshufflePlaybackActionMap.insert(pair<int, int>(EN_MEDIA_SHUFFLE_OFF, EXT_SHUFFLE_LIST));
   _repeatshufflePlaybackActionMap.insert(pair<int, int>(EN_MEDIA_SHUFFLE_LIST, EXT_REPEAT_SHUFFLE_OFF));
}


bool ExternalMediaPlaybackHandler::onCourierMessage(const MediaOperationMsgReq& oMsg)
{
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq OperationType: %d, m_extPlaybackstate: %d", oMsg.GetOperationType(), m_extPlaybackstate));

   uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq u8activeRegion cabin  %d ", u8activeRegion));

   int iActiveSource = MediaUtils::getInstance().getActiveSource(u8activeRegion);
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq calling iActiveSource:  %d ", iActiveSource));
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq calling  extPlaybackstate: %d", m_extPlaybackstate));
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq calling  extRepeatstate: %d", m_extRepeatShufflestate));

   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq calling  u8OperationType: %d", u8OperationType));

   if (((iActiveSource == CAMPORTUSB) || (iActiveSource == ROUTERUSB)) &&
         ((u8activeRegion == REGION_CABIN_A) || (u8activeRegion == REGION_CABIN_B)))
   {
      onPlaybackActionRequest(oMsg.GetOperationType(), iActiveSource);
   }
   return true;
}


void ExternalMediaPlaybackHandler::onPlaybackActionRequest(uint8 operationType, int activeSource)
{
   u8OperationType = operationType;
   int srcID = MediaSourceHandling::getInstance().getActiveSrcInRegion(REGION_CABIN_A);
   ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::onPlaybackActionRequest: srcID %d, operationType %d, activeSource %d", srcID, operationType, activeSource));
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(srcID);
   if (l_iPluginClientAppl != NULL)
   {
      l_iPluginClientAppl->sendPluginRequest(getControlValue(operationType, activeSource), srcID);
   }
   if ((operationType == EN_MEDIA_NEXT) || (operationType == EN_MEDIA_PREV) ||
         (operationType == EN_MEDIA_FASTREWIND_STOP) || (operationType == EN_MEDIA_FASTFORWARD_STOP))
   {
      updatePlaybackState();
   }
}


int ExternalMediaPlaybackHandler::getControlValue(tU8 u8OperationType, int activeSource)
{
   int controlValue = 0 ;
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getControlValue:u8OperationType activeSource m_extPlaybackstate m_extRepeatstate  %d %d %d %d",
                   u8OperationType, activeSource, m_extPlaybackstate, m_extRepeatShufflestate));

   switch (u8OperationType)
   {
      case EN_MEDIA_PLAY:
      case EN_MEDIA_PAUSE:
      {
         if (m_extPlaybackstate == EXT_MEDIA_PLAY)
         {
            _playbackActionMap[u8OperationType] = EXT_MEDIA_PAUSE;
            m_extPlaybackstate = EXT_MEDIA_PAUSE;
         }
         else
         {
            _playbackActionMap[u8OperationType] = EXT_MEDIA_PLAY;
            m_extPlaybackstate = EXT_MEDIA_PLAY;
         }
         lastPlaybackState = m_extPlaybackstate;
      }
      break;
      case EN_MEDIA_REPEAT_OFF:
      case EN_MEDIA_REPEAT_TRACK:
      case EN_MEDIA_REPEAT_LIST:
      {
         switch (m_extRepeatShufflestate)
         {
            case EXT_REPEAT_SHUFFLE_OFF:
            case EXT_SHUFFLE_LIST:
            {
               m_extRepeatShufflestate = EXT_REPEAT_TRACK;
            }
            break;
            case EXT_REPEAT_TRACK:
            {
               m_extRepeatShufflestate = EXT_REPEAT_LIST;
            }
            break;
            case EXT_REPEAT_LIST :
            {
               m_extRepeatShufflestate = EXT_REPEAT_SHUFFLE_OFF;
            }
            break;

            default:
            {
               ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::onCourierMessage:repeat operation  option is not called"));
            }
            ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getControlValue next m_extRepeatstate %d ", m_extRepeatShufflestate));
         }
      }
      break;
      case EN_MEDIA_SHUFFLE_OFF:
      case EN_MEDIA_SHUFFLE_LIST:
      {
         switch (m_extRepeatShufflestate)
         {
            case EXT_REPEAT_SHUFFLE_OFF:
            case EXT_REPEAT_LIST:
            case EXT_REPEAT_TRACK:
            {
               m_extRepeatShufflestate = EXT_SHUFFLE_LIST;
            }
            break;
            case EXT_SHUFFLE_LIST:
            {
               m_extRepeatShufflestate = EXT_REPEAT_SHUFFLE_OFF;
            }
            break;
            default:
            {
               ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::onCourierMessage:shuffle operation  option is not called"));
            }
            ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getControlValue next m_extshufflestate %d ", m_extRepeatShufflestate));
         }
      }
      break;
      default:
      {
         ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::onCourierMessage:MediaOperationMsgReq calling Default  option"));
      }
      break;
   }
   controlValue = updateControlValue(u8OperationType, activeSource);
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getControlValue controlValue %d ", controlValue));

   return controlValue;
}


bool ExternalMediaPlaybackHandler::isRepeatOrShuffleOperation(tU8 operationType)
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::isRepeatOrShuffleOperation:operationType %d ", operationType));
   bool isOperationRepeatOrShuffle = false;
   if ((operationType == EN_MEDIA_REPEAT_OFF) || (operationType == EN_MEDIA_REPEAT_TRACK) ||
         (operationType == EN_MEDIA_REPEAT_LIST) || (operationType == EN_MEDIA_SHUFFLE_LIST) ||
         (operationType == EN_MEDIA_SHUFFLE_OFF))
   {
      isOperationRepeatOrShuffle = true;
   }
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::isRepeatOrShuffleOperation:isOperationRepeatOrShuffle %d ", isOperationRepeatOrShuffle));

   return isOperationRepeatOrShuffle;
}


int ExternalMediaPlaybackHandler::updateControlValue(tU8 operationType, int activeSource)
{
   int controlValue = EXT_REPEAT_SHUFFLE_OFF;
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::updateControlValue:operationType %d , activeSource  %d", operationType, activeSource));

   if ((isRepeatOrShuffleOperation(operationType)))
   {
      if (_repeatshufflePlaybackActionMap.size() != 0)
      {
         controlValue = _repeatshufflePlaybackActionMap[u8OperationType];
      }
   }
   else
   {
      if (_playbackActionMap.size() != 0)
      {
         controlValue = _playbackActionMap[u8OperationType];
      }
   }
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::updateControlValue controlValue %d", controlValue));

   return controlValue;
}


uint8 ExternalMediaPlaybackHandler::getPlaybackState()
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getPlaybackState Calling PlaybackState function m_extPlaybackstate %d", m_extPlaybackstate));
   return m_extPlaybackstate;
}


uint8 ExternalMediaPlaybackHandler::getlastPlaybackState()
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getlastPlaybackState Calling PlaybackState function lastPlaybackState %d", lastPlaybackState));
   return lastPlaybackState;
}


tU8 ExternalMediaPlaybackHandler::getOperationType()
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getOperationType u8OperationType %d ", u8OperationType));
   return u8OperationType;
}


uint8 ExternalMediaPlaybackHandler::getShuffleState()
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getShuffleState m_extRepeatShufflestate %d", m_extRepeatShufflestate));
   return m_extRepeatShufflestate;
}


uint8 ExternalMediaPlaybackHandler::getRepeatState()
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::getRepeatState m_extRepeatShufflestate %d ", m_extRepeatShufflestate));
   return m_extRepeatShufflestate;
}


void ExternalMediaPlaybackHandler::updatePlaybackState()
{
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::updatePlaybackState m_extPlaybackstate:%d", m_extPlaybackstate));
   if (m_extPlaybackstate == EXT_MEDIA_PAUSE)
   {
      m_extPlaybackstate = EXT_MEDIA_PLAY;
   }
}


void ExternalMediaPlaybackHandler::setPlaybackState(uint8 playbackState)
{
   m_extPlaybackstate = playbackState;
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::setPlaybackState: m_extPlaybackstate = %d", m_extPlaybackstate));
}


void ExternalMediaPlaybackHandler::updateOperationType()
{
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::updateOperationType u8OperationType:%d", u8OperationType));
   if (u8OperationType != EN_MEDIA_PLAY)
   {
      u8OperationType = EN_MEDIA_PLAY;
   }
}


void ExternalMediaPlaybackHandler::setOperationType(tU8 operationType)
{
   u8OperationType = operationType;
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::setOperationType u8OperationType:%d", u8OperationType));
}


void ExternalMediaPlaybackHandler::handleExternalSourcePlayTimePluginUpdate(std::string ElapsedplaytimeData, double Elapsedslidervalue, double Totalslidervalue, std::string PlaytimeData, int srcID)
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::handleExternalSourcePlayTimePluginUpdate srcID ,Elapsedslidervalue , Totalslidervalue :%d ,%d,%d ", srcID,  Elapsedslidervalue, Totalslidervalue));
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::handleExternalSourcePlayTimePluginUpdate ElapsedplaytimeData  : %s ", ElapsedplaytimeData.c_str()));
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::handleExternalSourcePlayTimePluginUpdate PlaytimeData %s ", PlaytimeData.c_str()));

   if ((srcID == SRC_CAMERAPORT1_USB_VIDEO) || (srcID == SRC_CAMERAPORT1_USB_AUDIO))
   {
      HMIModelComponent::getInstance().vOnUpdateElapsedTimeStatusCameraPort(ElapsedplaytimeData, Elapsedslidervalue, Totalslidervalue, PlaytimeData); //done only for camport add different bindings for time
   }
   else if (srcID == SRC_CMR1_USB1_VIDEO || (srcID == SRC_CMR1_USB1_AUDIO))
   {
      HMIModelComponent::getInstance().vOnUpdateElapsedTimeStatusRouter(ElapsedplaytimeData, Elapsedslidervalue, Totalslidervalue, PlaytimeData);
   }
}


void ExternalMediaPlaybackHandler::handleExternalSourceMetaDataPluginUpdate(int srcID)
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::handleExternalSourceMetaDataPluginUpdate srcID  :%d ", srcID));

   if ((srcID == SRC_CAMERAPORT1_USB_VIDEO) || (srcID == SRC_CAMERAPORT1_USB_AUDIO))
   {
      HMIModelComponent::getInstance().bOnUpdateNowPlayingStatusCamera();
      HMIModelComponent::getInstance().vUpdateCamporUSBHomescreenStatus();
   }
   else if (srcID == SRC_CMR1_USB1_VIDEO || (srcID == SRC_CMR1_USB1_AUDIO))
   {
      ETG_TRACE_USR1(("ExternalMediaPlaybackHandler::handlePluginUpdate Router GUI Update"));
      HMIModelComponent::getInstance().bOnUpdateNowPlayingStatusRouter();
      HMIModelComponent::getInstance().vUpdateRouterUSBHomescreenStatus();
   }
}


void  ExternalMediaPlaybackHandler::vUpdatePlayItems(uint8 u8ControlVal, int srcId)
{
   uint8 l_extPlaybackstate = getPlaybackState();
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdatePlayItems controlvalue:%d", u8ControlVal));
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdatePlayItems ActiveRegion:%d", m_u8ActiveRegion));
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdatePlayItems l_extPlaybackstate:%d", l_extPlaybackstate));

   if (u8ControlVal == EXT_MEDIA_PAUSE)
   {
      m_extPlaybackstate = EXT_MEDIA_PAUSE;
      if ((srcId == SRC_CAMERAPORT1_USB_VIDEO) || (srcId == SRC_CAMERAPORT1_USB_AUDIO))
      {
         HMIModelComponent::getInstance().bOnUpdatePlayPauseStateOnCameraPort(EXT_MEDIA_PAUSE);//play
         // to update the gadget in the homescreen
         HMIModelComponent::getInstance().vUpdatePlayBackStateOnCamportUSBHomeScreen(EXT_MEDIA_PAUSE, srcId);
      }
      else if (srcId == SRC_CMR1_USB1_VIDEO || (srcId == SRC_CMR1_USB1_AUDIO))
      {
         HMIModelComponent::getInstance().bOnUpdatePlayPauseStateOnRouter(EXT_MEDIA_PAUSE);//play
         // to update the gadget in the homescreen
         HMIModelComponent::getInstance().vUpdatePlayBackStateRouterUSBHomeScreen(EXT_MEDIA_PAUSE, srcId);
      }
   }
   else if (u8ControlVal == EXT_MEDIA_PLAY)
   {
      m_extPlaybackstate = EXT_MEDIA_PLAY;
      if ((srcId == SRC_CAMERAPORT1_USB_VIDEO) || (srcId == SRC_CAMERAPORT1_USB_AUDIO))
      {
         HMIModelComponent::getInstance().bOnUpdatePlayPauseStateOnCameraPort(EXT_MEDIA_PLAY);//pause
         // to update the gadget in the homescreen
         HMIModelComponent::getInstance().vUpdatePlayBackStateOnCamportUSBHomeScreen(EXT_MEDIA_PLAY, srcId);
      }
      else if (srcId == SRC_CMR1_USB1_VIDEO || (srcId == SRC_CMR1_USB1_AUDIO))
      {
         HMIModelComponent::getInstance().bOnUpdatePlayPauseStateOnRouter(EXT_MEDIA_PLAY);//pause
         // to update the gadget in the homescreen
         HMIModelComponent::getInstance().vUpdatePlayBackStateRouterUSBHomeScreen(EXT_MEDIA_PLAY, srcId);
      }
   }
}


int ExternalMediaPlaybackHandler::getExternalSourceShuffleStatus(int srcId)
{
   uint8 l_extShufflestate = getShuffleState();
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::getExternalSourceShuffleStatus l_extShufflestate , srcId:%d %d", l_extShufflestate, srcId));

   int shuffleStatus = SHUFFLE_OFF;
   if (srcId == SRC_CMR1_USB1_VIDEO || srcId == SRC_CMR1_USB1_AUDIO || srcId == SRC_CAMERAPORT1_USB_VIDEO || srcId == SRC_CAMERAPORT1_USB_AUDIO)
   {
      shuffleStatus = (EXT_SHUFFLE_LIST == l_extShufflestate) ? SHUFFLE_ON : SHUFFLE_OFF;
   }
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::getExternalSourceShuffleStatus shuffleStatus:%d", shuffleStatus));

   return shuffleStatus;
}


int ExternalMediaPlaybackHandler::getExternalSourceRepeatStatus(int srcId)
{
   uint8 l_extRepeatstate = getRepeatState();
   int repeatStatus = EN_MEDIA_REPEATMODE_OFF;
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdateExternalSourceRepeatStatus srcId : %d ,l_extRepeatstate:%d", srcId, l_extRepeatstate));

   if ((srcId == SRC_CMR1_USB1_VIDEO || srcId == SRC_CMR1_USB1_AUDIO || srcId == SRC_CAMERAPORT1_USB_VIDEO) || (srcId == SRC_CAMERAPORT1_USB_AUDIO))
   {
      if ((EXT_REPEAT_TRACK == l_extRepeatstate))
      {
         repeatStatus = EN_MEDIA_REPEATMODE_TRACK ;
      }
      else if (EXT_REPEAT_LIST == l_extRepeatstate)
      {
         repeatStatus = EN_MEDIA_REPEATMODE_LIST;
      }
      else
      {
         repeatStatus = EN_MEDIA_REPEATMODE_OFF;
      }
   }
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdateExternalSourceRepeatStatus repeatStatus:%d", repeatStatus));

   return 	repeatStatus;
}


void ExternalMediaPlaybackHandler::vUpdateRepeatShuffleItems(uint8 u8ControlVal, int srcId)
{
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdateRepeatShuffleItems controlvalue:%d", u8ControlVal));

   int shuffleStatus = getExternalSourceShuffleStatus(srcId);
   int repeatStatus  = getExternalSourceRepeatStatus(srcId);

   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdateRepeatShuffleItems srcId:%d ,shuffleStatus:%d , repeatStatus:%d , m_extRepeatShufflestate %d  ", srcId, shuffleStatus, repeatStatus, m_extRepeatShufflestate));

   if ((srcId == SRC_CAMERAPORT1_USB_VIDEO) || (srcId == SRC_CAMERAPORT1_USB_AUDIO))
   {
      switch (u8ControlVal)
      {
         case EXT_REPEAT_SHUFFLE_OFF:
         {
            m_extRepeatShufflestate = EXT_REPEAT_SHUFFLE_OFF;
            HMIModelComponent::getInstance().bOnUpdateShuffleModeCamportUSBStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeCamportUSBStatus(EN_MEDIA_REPEATMODE_OFF);//repeat on/off track
         }
         break;
         case EXT_REPEAT_TRACK:
         {
            m_extRepeatShufflestate = EXT_REPEAT_TRACK;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeCamportUSBStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeCamportUSBStatus(EN_MEDIA_REPEATMODE_TRACK);//repeat on/off track
         }
         break;
         case EXT_REPEAT_LIST:
         {
            m_extRepeatShufflestate = EXT_REPEAT_LIST;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeCamportUSBStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeCamportUSBStatus(EN_MEDIA_REPEATMODE_LIST);//repeat on/off track
         }
         break;
         case EXT_SHUFFLE_LIST:
         {
            m_extRepeatShufflestate = EXT_SHUFFLE_LIST;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeCamportUSBStatus(SHUFFLE_ON);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeCamportUSBStatus(EN_MEDIA_REPEATMODE_OFF);//repeat on/off track
         }
         break;
         default:
            ETG_TRACE_USR1(("RouterPluginClientHandler::vUpdateRepeatShuffleItems camport control value is default"));
            break;
      }
   }
   else if (srcId == SRC_CMR1_USB1_VIDEO || srcId == SRC_CMR1_USB1_AUDIO)
   {
      switch (u8ControlVal)
      {
         case EXT_REPEAT_SHUFFLE_OFF:
         {
            m_extRepeatShufflestate = EXT_REPEAT_SHUFFLE_OFF;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeRouterStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeRouterStatus(EN_MEDIA_REPEATMODE_OFF);//repeat on/off track
         }
         break;
         case EXT_REPEAT_TRACK:
         {
            m_extRepeatShufflestate = EXT_REPEAT_TRACK;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeRouterStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeRouterStatus(EN_MEDIA_REPEATMODE_TRACK);//repeat on/off track
         }
         break;
         case EXT_REPEAT_LIST:
         {
            m_extRepeatShufflestate = EXT_REPEAT_LIST;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeRouterStatus(SHUFFLE_OFF);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeRouterStatus(EN_MEDIA_REPEATMODE_LIST);//repeat on/off track
         }
         break;
         case EXT_SHUFFLE_LIST:
         {
            m_extRepeatShufflestate = EXT_SHUFFLE_LIST;

            HMIModelComponent::getInstance().bOnUpdateShuffleModeRouterStatus(SHUFFLE_ON);//shuffle on/off
            HMIModelComponent::getInstance().bOnUpdateRepeatModeRouterStatus(EN_MEDIA_REPEATMODE_OFF);//repeat on/off track
         }
         break;
         default:
            ETG_TRACE_USR1(("RouterPluginClientHandler::vUpdateRepeatShuffleItems router control value is default"));
            break;
      }
   }
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::vUpdateRepeatShuffleItems m_extRepeatShufflestate:%d ", m_extRepeatShufflestate));
}


void ExternalMediaPlaybackHandler::updateTotalTime(std::string totaltime)
{
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::UpdateTotalTime totaltime %s ", totaltime.c_str()));
   MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TotalTime = totaltime;
}


void ExternalMediaPlaybackHandler::updateArtistName(std::string artistName)
{
   ETG_TRACE_USR3(("ExternalMediaPlaybackHandler::updateArtistName artistName %s ", artistName.c_str()));
   MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].ArtistName = artistName;;
}


void ExternalMediaPlaybackHandler::updateTitleName(std::string titleName)
{
   ETG_TRACE_USR4(("ExternalMediaPlaybackHandler::updateTitleName  titleName  %s", titleName.c_str()));
   MediaUtils::getInstance()._metadataMap[REGION_CABIN_A].TitleName = titleName;
}


void ExternalMediaPlaybackHandler::onFolderPlaybackSrcChange(uint16 fileType, uint16 track, std::string path, uint16 listtype, uint8 src, uint16 regionId)
{
   ETG_TRACE_COMP(("ExternalMediaPlaybackHandler::onFolderPlaybackSrcChange: fileType %d, listtype %d, src %d, regionId %d, path %s", ETG_CENUM(entFileType, fileType), ETG_CENUM(entListType, listtype), ETG_CENUM(enSubSourceType, src), regionId, path.c_str()));
   EnumConst::enSubSourceType subSrcType = static_cast<EnumConst::enSubSourceType>(src);
   if ((src == EnumConst::enSubSourceType_CMP_AUDIO) || (src == EnumConst::enSubSourceType_CMR_AUDIO))
   {
      _playableContentMap->setPlayableAudioContent(_externalSourceTypeMap[subSrcType].srcType, true);
   }
   else if ((src == EnumConst::enSubSourceType_CMP_VIDEO) || (src == EnumConst::enSubSourceType_CMR_VIDEO))
   {
      _playableContentMap->setPlayableVideoContent(_externalSourceTypeMap[subSrcType].srcType, true);
   }
   IPluginClientAppl* l_iPluginClientAppl = PluginClientFactory::getPluginClient(_externalSourceTypeMap[subSrcType].srcId);
   if (l_iPluginClientAppl != NULL)
   {
      l_iPluginClientAppl->updateCurrentPlayingListType(listtype, track);
      l_iPluginClientAppl->updateCurentPlayingFolderPath(path);
      l_iPluginClientAppl->updateCurrentPlayingSrcType(src);
      l_iPluginClientAppl->updateCurrentPlayingFileType(fileType);
   }
   IExternalMediaSourceActivationHandler* iExternalMediaSourceActivationHandler = ExternalMediaSourceActivationFactory::getExternalActivationSourceHandlerObject(_externalSourceTypeMap[subSrcType].deviceType);
   iExternalMediaSourceActivationHandler->reset();
   iExternalMediaSourceActivationHandler->updateActivePlayableContentType();
   iExternalMediaSourceActivationHandler->requestSourceActivation(regionId);
}


}//App
}//Core
