/***************************************************************************
* Copyright(c) 2018-2020 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch GmbH.
***************************************************************************/
#include "ApplicationSwitchClientHandler.h"
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "ProjectBaseExchangingTextures.h"
#include "hmi_trace_if.h"
#include <string>
#include "hall_std_if.h"
#include "../../DataModel/HMIModelComponent.h"
#include "Core/SourceSwitch/MediaSourceHandling.h"
#include "Core/StateTransitionHandler/StateTransitionHandler.h"
#include "App/Core/MediaHall.h"
#include "Core/Utils/MediaUtils.h"
#include "Core/MediaDefines.h"
#include"Core/ExternalMediaSourceActivation/IExternalMediaSourceActivationHandler.h"
#include "Core/ExternalMediaSourceActivation/ExternalMediaSourceActivationFactory.h"
#include "Core/Utils/DbDownloadMap.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS         TR_CLASS_APPHMI_MEDIA_HALL
#define ETG_I_TRACE_CHANNEL             TR_TTFIS_APPHMI_MEDIA
#define ETG_I_TTFIS_CMD_PREFIX          "APPHMI_MEDIA_"
#define ETG_I_FILE_PREFIX               App::Core::ApplicationSwitchClientHandler::
#include "trcGenProj/Header/ApplicationSwitchClientHandler.cpp.trc.h"
#endif


/*****************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------*/
#define PORTNAME "applicationSwitchPort"

#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "utest/stubs/Apphmi_Media_Types_Common.h"
#include "utest/stubs/ApplicationSwitchConst.h"
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;
#endif
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;
#endif
namespace App {
namespace Core {

ApplicationSwitchClientHandler* ApplicationSwitchClientHandler::m_poSelfAppSwitchClient = NULL;

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST

ApplicationSwitchClientHandler::ApplicationSwitchClientHandler()
   : m_poApplicationSwitchProxy(ApplicationSwitchProxy::createProxy(PORTNAME, *this))
   , m_u8activeRegion(REGION_INVALID)
   , _externalMediaClientHandler(NULL)
   , _externalMediaSourceHandler(NULL)
{
   SignalId = 0;
   m_sCurrentViewName = "";
   _hdmiActivityIdToDevTypMap.clear();
   initialiseHdmiActivityIdToDevTypeMap();
   _hmiInfoServiceProxyClient = (::HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this));
   m_iUsbCamportAvailable = EN_EXTERNAL_MEDIA_DISCONNECTED;
   _sourceType = EnumConst::entSourceType_DEFAULT;
   if (_hmiInfoServiceProxyClient.get())
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::ApplicationSwitchClientHandler StartupSync _hmiInfoProxyClient"));
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxyClient->getPortName());
   }
}


#else

ApplicationSwitchClientHandler::ApplicationSwitchClientHandler()
{
   m_u8activeRegion = REGION_INVALID;
}


#endif
ApplicationSwitchClientHandler::~ApplicationSwitchClientHandler()
{
   if (NULL != m_poSelfAppSwitchClient)
   {
      delete m_poSelfAppSwitchClient;
      m_poSelfAppSwitchClient = NULL;
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   _hdmiActivityIdToDevTypMap.clear();
   clearInstance();
#endif
}


ApplicationSwitchClientHandler* ApplicationSwitchClientHandler::poGetInstance()
{
   /* No more than one instance of ApplicationSwitchClientHandler should be created */
   if (NULL == m_poSelfAppSwitchClient)
   {
      m_poSelfAppSwitchClient = new ApplicationSwitchClientHandler();
   }
   return m_poSelfAppSwitchClient;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
void ApplicationSwitchClientHandler::clearInstance()
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler clearInstance \n"));
}


void ApplicationSwitchClientHandler::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onAvailable"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdRegister(*this);
      m_poApplicationSwitchProxy->sendActiveRegionIdGet(*this);
      m_poApplicationSwitchProxy->sendSig_ActivateContextRegister(*this);
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestRegister(*this);
   }
}


void ApplicationSwitchClientHandler::onUnavailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onUnavailable"));
   if (m_poApplicationSwitchProxy && m_poApplicationSwitchProxy == proxy)
   {
      m_poApplicationSwitchProxy->sendActiveRegionIdDeregisterAll();
      m_poApplicationSwitchProxy->sendSig_ActivateContextDeregisterAll();
   }
   if (_hmiInfoServiceProxyClient && _hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequestDeregisterAll();
   }
}


void ApplicationSwitchClientHandler::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::registerProperties"));
}


void ApplicationSwitchClientHandler::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::deregisterProperties"));
}


void ApplicationSwitchClientHandler::onToggleGadgetAudioSourceError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< ToggleGadgetAudioSourceError >& /*error*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onToggleGadgetAudioSourceError"));
}


void ApplicationSwitchClientHandler::onToggleGadgetAudioSourceResponse(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< ToggleGadgetAudioSourceResponse >& /* response*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onToggleGadgetAudioSourceResponse"));
}


/**
* Description     : Virtual function implemented to get update of onSig_ActivateContextError Error
*
* @param[in]      : proxy: the client side representation of the ContextSwitch IF
* @param[in]      : error : The error message of 'onSig_ActivateContextError'
* @return         : void
*/

void ApplicationSwitchClientHandler::onSig_ActivateContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextError >& error)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onSig_ActivateContextError"));
}


void ApplicationSwitchClientHandler::onRequestContextSwitchError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestContextSwitchError >& error)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onRequestContextSwitchError"));
}


void ApplicationSwitchClientHandler::onRequestContextSwitchResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< RequestContextSwitchResponse >& response)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onRequestContextSwitchResponse"));
}


void ApplicationSwitchClientHandler::onSendContextError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextError >& error)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onSendContextError"));
}


void ApplicationSwitchClientHandler::onSendContextResponse(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< SendContextResponse >& response)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onSendContextResponse"));
}


void ApplicationSwitchClientHandler::initialiseHdmiActivityIdToDevTypeMap()
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::initialiseHdmiActivityIdToDevTypeMap is entered \n"));
   _hdmiActivityIdToDevTypMap.insert(pair<uint32, int>(enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI1, FI_EN_E8DTY_USB_CAMERAPORT_HDMI));
   _hdmiActivityIdToDevTypMap.insert(pair<uint32, int>(enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI2, FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2));
   _hdmiActivityIdToDevTypMap.insert(pair<uint32, int>(enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI3, FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3));
}


/************************************************************************
*FUNCTION		: onSig_ActivateContextSignal
*DESCRIPTION	: on press of settings button in common popup(volume popup) will navigate to settings screen of media when in media application
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: void
*SWFL			: [SWFL-2450] [SWFL-2449] [SWFL-2448]
************************************************************************/
void ApplicationSwitchClientHandler::onSig_ActivateContextSignal(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy, const ::boost::shared_ptr< Sig_ActivateContextSignal >& signal)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal Target Activity ID %d", signal->getTargetactivityId()));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal region id %d", signal->getRegionId()));
   uint32 activityID = signal->getTargetactivityId();
   uint8 regionID = signal->getRegionId();
   SignalId = signal->getTargetactivityId();

   if (activityID == enActivityIDs__eActivityID_HOME || activityID == enActivityIDs__eActivityID_CABINA_HOME || activityID == enActivityIDs__eActivityID_CABINB_HOME)
   {
      StateTransitionHandler::getInstance().initializeMediaStateTransitions();
   }
   switch (activityID)
   {
      case enActivityIDs__eActivityID_MEDIA_SETTINGS_VIEW://128 settings  screen
         StateTransitionHandler::getInstance().stateTransit_onSettingContextRequest();

         break;

      case enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION:
         //120 source selection  screen
         StateTransitionHandler::getInstance().stateTransit_onSourceSelectionContextUpdate();
         break;
      case enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTUSB1:
      {
         if (_externalMediaClientHandler != NULL)
         {
            _externalMediaClientHandler->onSourceSelection(EnumConst::entSourceType_CMP);
         }
         break;
      }

      case enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI1:
      case enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI2:
      case enActivityIDs__eActivityID_MEDIA_MCP_CAMPORTHDMI3:
      {
         int deviceType = FI_EN_E8DTY_UNKNOWN;
         std::map <uint32, int>::iterator itr = _hdmiActivityIdToDevTypMap.find(activityID);
         if (itr != _hdmiActivityIdToDevTypMap.end())
         {
            deviceType = itr -> second;
         }
         bool bIsHDMISourceActive = false;
         int iSrcId = SRC_INVALID;
         //404 MCP camporthdmi1 activity id
         if (_externalMediaSourceHandler != NULL)
         {
            iSrcId = _externalMediaClientHandler->getHdmiSrcId(deviceType);
            bIsHDMISourceActive = _externalMediaClientHandler->getHdmiSourceActiveState(iSrcId);
         }
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal:bIsHDMISourceActive: %d,  iSrcId    %d", bIsHDMISourceActive, iSrcId));

         if (_externalMediaClientHandler != NULL)
         {
            _externalMediaClientHandler->onHdmiSourceSelection(deviceType);
         }
         break;
      }
      case enActivityIDs__eActivityID_MEDIA_MCP_ROUTERUSB1:
      {
         if (_externalMediaClientHandler != NULL)
         {
            //_externalMediaClientHandler->onSourceSelection(EnumConst::entSourceType_CMR); // Temporarily commented as part of 1606014 - for disabling Router USB
         }
         break;
      }
      case enActivityIDs__eActivityID_MEDIA_MCP_USB1:
      {
         //134 MCP USB1 activity id
         int devicetagUSB1 = tclMediaClientHandler::getInstance().getUSB1DeviceTag();
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal devicetagUSB1: %d", devicetagUSB1));
         if (devicetagUSB1 != 0)
         {
            g_SM_ActiveMediaSource = FI_EN_E8DTY_USB;
            MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_PLAYER, devicetagUSB1, regionID);
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
         }
         break;
      }

      case enActivityIDs__eActivityID_MEDIA_MCP_USB2:
      {
         //135 MCP USB2 activity id
         int devicetagUSB2 = tclMediaClientHandler::getInstance().getUSB2DeviceTag();
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal devicetagUSB2: %d", devicetagUSB2));
         if (devicetagUSB2 != 0)
         {
            g_SM_ActiveMediaSource = FI_EN_E8DTY_USB;
            MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_PLAYER, devicetagUSB2, regionID);
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
         }
         break;
      }

      case enActivityIDs__eActivityID_MEDIA_MCP_BTSTREAMING:
      {
         //133 MCP BT activity id
         int devicetagBT = tclMediaClientHandler::getInstance().getBTDeviceTag();
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal devicetagBT: %d", devicetagBT));
         if (devicetagBT != 0)
         {
            g_SM_ActiveMediaSource = FI_EN_E8DTY_BLUETOOTH;
            MediaSourceHandling::getInstance().requestSourceActivation(SRC_PHONE_BTAUDIO, devicetagBT, regionID);
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
         }
         break;
      }

      case enActivityIDs__eActivityID_MEDIA_MCP_IPOD:
      {
         //136 MCP ipod activity id
         int devicetagIPOD = tclMediaClientHandler::getInstance().getIPODDeviceTag();
         ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onSig_ActivateContextSignal devicetagIPOD: %d", devicetagIPOD));
         if (devicetagIPOD != 0)
         {
            g_SM_ActiveMediaSource = FI_EN_E8DTY_IPOD;
            MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_PLAYER, devicetagIPOD, regionID);
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
         }
         break;
      }
      case enActivityIDs__eActivityID_MEDIA_MCP_AUX:
      {
         //137 MCP AUX activity id
         g_SM_ActiveMediaSource = FI_EN_E8DTY_AUX;
         MediaSourceHandling::getInstance().requestSourceActivation(SRC_MEDIA_AUX, -1, regionID);
         StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
         break;
      }
      case enActivityIDs__eActivityID_MEDIA_AUX: //122
      case enActivityIDs__eActivityID_MEDIA_USB: //124
      case enActivityIDs__eActivityID_MEDIA_BTSTREAMING://bluetooth main screen //125
      case enActivityIDs__eActivityID_MEDIA_IPOD://127
      case enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI://408
      case enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI2:
      case enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI3:
      case enActivityIDs__eActivityID_MEDIA_CAMPORTUSB://409
      case enActivityIDs__eActivityID_MEDIA_ROUTERUSB:
      case enActivityIDs__eActivityID_MEDIA_PLAY://121
      {
         /*
         on press of media appfly inn we get media_play from master,here we will check for active media sources (lastActiveMediaSource)and will request
         for audio allocation for the same by extarcting srcid and subsrcid from lastActiveMediaSource,GetActiveSubSrcId(),GetActiveSrcId().
         */
         if (MediaSourceHandling::getInstance().getMic1Status() == false)
         {
            StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_APPFLY_INN);
            activeMediaAudioAllocationRequest();
            StateTransitionHandler::getInstance().stateTransit_onAppFlyInn();
         }
      }
      break;
      default:
         break;
   }
   g_SM_ActiveMediaActivityID = activityID;// Added to hold the active activity ID as part of bugfix like default screen flickering while context switch within applications
}


void ApplicationSwitchClientHandler::onToggleGadgetAudioSourceRequestError(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& /* proxy*/, const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestError >& /*error*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onToggleGadgetAudioSourceRequestError(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< ToggleGadgetAudioSourceRequestError >& error) not yet implemented"));
}


void  ApplicationSwitchClientHandler::onToggleGadgetAudioSourceRequestSignal(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr<ToggleGadgetAudioSourceRequestSignal >& signal)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onToggleGadgetAudioSourceRequestSignal(const ::boost::shared_ptr< HmiInfoServiceProxy >& proxy, const ::boost::shared_ptr< ToggleGadgetAudioSourceRequestSignal >& signal) not yet implemented"));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onToggleGadgetAudioSourceRequestSignal destapplication %d", signal->getDestApplicationId()));
   if (signal->getDestApplicationId() == APPID_APPHMI_MEDIA)
   {
      tclMediaClientHandler::getInstance().PerformToggleContext();
   }
}


void ApplicationSwitchClientHandler::activeMediaAudioAllocationRequest()
{
   MediaSourceHandling::getInstance().lastActiveMediaSource();
   int srcid = MediaSourceHandling::getInstance().GetLastMediaSrcId();
   int subsrcid = MediaSourceHandling::getInstance().GetLastMediaSubSrcId();
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::activeMediaAudioAllocationRequest srcid: %d", srcid));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::activeMediaAudioAllocationRequest subsrcid: %d", subsrcid));
   tclMediaClientHandler::getInstance().SetActiveSource(subsrcid, srcid);
   if (srcid == SRC_INVALID && subsrcid == SUBSRCID_INVALID)//srcid and subsrcid is 0,when no device is connected  or device is disconnected
   {
      ETG_TRACE_USR4(("ApplicationSwitchClientHandler::naviagte to sourceselection screen when no device is active"));
      StateTransitionHandler::getInstance().activateMediaState(MEDIA_STATE_SOURCESELECTION);
   }
   else if ((MediaSourceHandling::getInstance().getCurrentRegionSrcId() != srcid) || (MediaSourceHandling::getInstance().getCurrentRegionSubSrc() != subsrcid))

   {
      uint8 u8activeRegion = ApplicationSwitchClientHandler::poGetInstance()->getActiveRegion();

      MediaSourceHandling::getInstance().requestSourceActivation(srcid, subsrcid, u8activeRegion);

      StateTransitionHandler::getInstance().setStateTransitionReason(ST_TRANSITION_REASON_USER_SRC_REQUEST);
   }
   HMIModelComponent::getInstance().bOnUpdateHeaderMuteStatus();
}


#endif

const uint8 ApplicationSwitchClientHandler::getActiveRegion()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActiveRegion active region %d", m_u8activeRegion));
   return m_u8activeRegion;
}


void ApplicationSwitchClientHandler::setActiveRegion(int activeRegion)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::setActiveRegion active region %d", activeRegion));
   m_u8activeRegion = activeRegion;
}


bool ApplicationSwitchClientHandler::IsMediaSourceRangeActivity(uint32 activityid)
{
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::IsMediaSourceRangeActivity activity id %d ", activityid));
#endif
   if (((activityid >= enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION) && (activityid <= enActivityIDs__eActivityID_MEDIA_IPOD)) || ((activityid >= enActivityIDs__eActivityID_MEDIA_SOURCE_SWITCH) && (activityid <= enActivityIDs__eActivityID_MEDIA_LIST_VIEW)))
   {
      return true;
   }
   else
   {
      return false;
   }
}


uint16 ApplicationSwitchClientHandler::getActivityIdFromDeviceType(uint16 deviceType)
{
   uint16 activityID = 0;
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getActivityIdFromDeviceType deviceType  %d ", deviceType));
#endif
   switch (deviceType)
   {
      case FI_EN_E8DTY_USB:
         activityID = enActivityIDs__eActivityID_MEDIA_USB;  //USB
         break;

      case FI_EN_E8DTY_USB_CAMERAPORT:
         activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTUSB;  //USBCAMPORT
         break;

      case FI_EN_E8DTY_IPOD:
         activityID = enActivityIDs__eActivityID_MEDIA_IPOD; //IPOD
         break;
      case FI_EN_E8DTY_BLUETOOTH:
         activityID = enActivityIDs__eActivityID_MEDIA_BTSTREAMING;  //BT
         break;
      case FI_EN_E8DTY_IPHONE:
         activityID = enActivityIDs__eActivityID_MEDIA_IPOD;//IPHONE
         break;
      case FI_EN_E8DTY_MTP:
         activityID = enActivityIDs__eActivityID_MEDIA_USB;//MTP
         break;
      case FI_EN_E8DTY_AUX:
         activityID = enActivityIDs__eActivityID_MEDIA_AUX;  //USB
         break;
      case FI_EN_E8DTY_SOURCESELECTIONSCREEN:
         activityID = enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION;  //Source selection screen
         break;
      case FI_EN_E8DTY_MEDIASETTINGSSCREEN:
         activityID = enActivityIDs__eActivityID_MEDIA_SETTINGS_VIEW;//settings screen(M_OPT)
         break;
      case FI_EN_E8DTY_MEDIALISTSCREEN:
         activityID = enActivityIDs__eActivityID_MEDIA_LIST_VIEW;//To inform master that current screen is media List, so do not display the Appflyins.Context switch call will not trigger.
         break;
      case FI_EN_E8DTY_MEDIAVIDEOFULLSCREEN:
         activityID = enActivityIDs__eActivityID_MEDIA_PLAY_FULLSCREEN;//To inform master that current screen is media video fullscreen, so do not display the Appflyins.Context switch call will not trigger.
         break;
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI:
         activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI;
         break;
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2:
         activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI2;
         break;
      case FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3:
         activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI3;
         break;
      case FI_EN_E8DTY_ROUTER_USB:
         activityID = enActivityIDs__eActivityID_MEDIA_ROUTERUSB;
         break;
      default:
         break;
   }
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
   ETG_TRACE_USR2(("ApplicationSwitchClientHandler::getActivityIdFromDeviceType activityID %d", activityID));
#endif
   return activityID;
}


#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
uint8 ApplicationSwitchClientHandler::getactivityID()
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::getactivityID SignalId  %d ", SignalId));

   return SignalId;
}


bool ApplicationSwitchClientHandler::onCourierMessage(const Courier::StartupMsg& oMsg)
{
   return true;
}


void ApplicationSwitchClientHandler::onActiveRegionIdError(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< ActiveRegionIdError >& error)
{
   ETG_TRACE_COMP(("ApplicationSwitchClientHandler::onActiveRegionIdError "));
}


void ApplicationSwitchClientHandler::onActiveRegionIdUpdate(const ::boost::shared_ptr< ApplicationSwitchProxy >& proxy,
      const ::boost::shared_ptr< ActiveRegionIdUpdate >& update)
{
   m_u8activeRegion = update->getActiveRegionId();
   g_SM_ActiveRegion = m_u8activeRegion;

   MediaSourceHandling::getInstance().setActiveSourceStatus();
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate activeRegion: %d", m_u8activeRegion));
   int iSubSrcId = MediaSourceHandling::getInstance().getCurrentRegionSubSrc();
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate iSubSrcId %d", iSubSrcId));

   int iDeviceType = tclMediaClientHandler::getInstance().GetMediaDeviceType(iSubSrcId);
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate iDeviceType %d", iDeviceType));

   int srcid = MediaSourceHandling::getInstance().getCurrentRegionSrcId();
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate srcid %d", srcid));

   uint8 iactiveMediaSource = MediaSourceHandling::getInstance().GetActiveMediaSource(srcid, iDeviceType);
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onActiveRegionIdUpdate iactiveMediaSource %d", iactiveMediaSource));
   HMIModelComponent::getInstance().vUpdateDeviceTypeToSM(iactiveMediaSource);

   HMIModelComponent::getInstance().vAuxSourceVisibility(false);

   HMIModelComponent::getInstance().vUpdateCabinGadgetVisibility();

   MediaUtils::getInstance().m_ActiveDeviceDataMapItr = MediaUtils::getInstance().m_ActiveDeviceDataMap.find(m_u8activeRegion);

   HMIModelComponent::getInstance().updateMediaSourceInfoInGadget(iDeviceType);

   HMIModelComponent::getInstance().vSourceButtonHighlight(iDeviceType, iSubSrcId);

   //To update the current mute status in nowplaying screen
   HMIModelComponent::getInstance().bOnUpdateHeaderMuteStatus();
   //To update the current mute icon in gadget screen
   HMIModelComponent::getInstance().vUpdateMuteStatus();

   HMIModelComponent::getInstance().vUpdateNextPerviousIconStatus();
}


/************************************************************************
*FUNCTION		: LayerSoundSettingsMsg
*DESCRIPTION	: PRESSING SOUND SETTINGS BUTTON IN SETTINGS SCREEN OF MEDIA WHERE IT TAKES TO SYSTEN SETTINGS
*PARAMETER		: NA
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/
bool ApplicationSwitchClientHandler::onCourierMessage(const LayerSoundSettingsMsg& oMsg)
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::onCourierMessage:LayerSoundSettingsMsg "));

   uint16 activityID = 0;
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCourierMessage:LayerSoundSettingsMsg activeRegion %d ", m_u8activeRegion));

   if (m_u8activeRegion == REGION_COCKPIT)
   {
      activityID = enActivityIDs__eActivityID_SETTINGS_VOLUME;
   }
   else if (m_u8activeRegion == REGION_CABIN_A)
   {
      activityID = 186;
   }
   /*else
   {
   //do nothing
   }*/

   m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, m_u8activeRegion, APPID_APPHMI_MEDIA, enActivityIDs__eActivityID_MEDIA_SETTINGS_VIEW, APPID_APPHMI_SYSTEM, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCourierMessage:LayerSoundSettingsMsg activityID %d ", activityID));

   return true;
}


/************************************************************************
*FUNCTION		: ButtonMainSrcActivate
*DESCRIPTION	: from master home screen media tile is pressed in the centre part
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/
bool ApplicationSwitchClientHandler::onCourierMessage(const ButtonMainSrcActivate& oMsg)
{
   ETG_TRACE_USR3(("ApplicationSwitchClientHandler::onCourierMessage:ButtonMainSrcActivate  ActiveMediaSource %d", g_SM_ActiveMediaSource));

   uint16 activityID = 0;
   if ((g_SM_ActiveMediaSource != FI_EN_E8DTY_USB) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_IPOD) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_BLUETOOTH) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_IPHONE) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_MTP) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_AUX) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_USB_CAMERAPORT) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_USB_CAMERAPORT_HDMI) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_USB_CAMERAPORT_HDMI_2) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_USB_CAMERAPORT_HDMI_3) && (g_SM_ActiveMediaSource != FI_EN_E8DTY_ROUTER_USB))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION;
   }
   else
   {
      activityID = getActivityIdFromDeviceType(g_SM_ActiveMediaSource);
   }
   ETG_TRACE_USR3(("ApplicationSwitchClientHandler::onCourierMessage:ButtonMainSrcActivate  activityID %d", activityID));

   //switch context to playscreen
   m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, m_u8activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID/*enActivityIDs__eActivityID_TUNER_FM1*/);

   return true;
}


/************************************************************************
*FUNCTION		: reqContextSwitchSrcSelection
*DESCRIPTION	: func call context switch to source selection screen
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/

void ApplicationSwitchClientHandler::reqContextSwitchSrcSelection()
{
   //switch context to media
   uint8 activityID = enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION;
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::reqContextSwitchSrcSelection activityID %d", activityID));

   m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, m_u8activeRegion, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID/*enActivityIDs__eActivityID_TUNER_FM1*/);
}


/************************************************************************
*FUNCTION		: reqContextSwitchSrcSelectionRadio
*DESCRIPTION	: func call context switch to source selection screen
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/

void ApplicationSwitchClientHandler::reqContextSwitchSrcSelectionHome(uint8 activityID, uint8 appId)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::reqContextSwitchSrcSelection activityID %d", activityID));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::reqContextSwitchSrcSelection appId %d", appId));
   m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, m_u8activeRegion, APPID_APPHMI_MEDIA, enActivityIDs__eActivityID_MEDIA_SOURCE_SELECTION, appId, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID/*enActivityIDs__eActivityID_TUNER_FM1*/);
}


/************************************************************************
*FUNCTION		: HomescreenToggle
*DESCRIPTION	: func call when toggle button from homescreen is pressed
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/
bool ApplicationSwitchClientHandler::onCourierMessage(const HomescreenToggle& oMsg)
{
   reqHomeScreenToggle();
   return true;
}


/************************************************************************
*FUNCTION		: MediaSceneMsg
*DESCRIPTION	: func call when entered into USB/IPOD/IPHONE Scene
*PARAMETER		: uint16
*RETURNVALUE	: bool
************************************************************************/
bool ApplicationSwitchClientHandler::onCourierMessage(const MediaSceneMsg& oMsg)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCourierMessage:MediaSceneMsg media type %d", oMsg.GetMediaType()));
   uint16 requestedmediatype = oMsg.GetMediaType();
   uint16 activityID = 0;
   activityID = getActivityIdFromDeviceType(requestedmediatype);
   sendStoreContextRequest(activityID);
   return true;
}


uint16 ApplicationSwitchClientHandler::getMediaActivityID(std::string viewid)
{
   uint16 activityID = 0;
   if (0 == viewid.compare("Media#MediaPlayerScenes#M_VideoPlayer"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_USB;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_USB"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_USB;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_AUX"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_AUX;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_BT"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_BTSTREAMING;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_AppleDevice"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_IPOD;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_CAMPORTHDMI_1"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_CAMPORTHDMI_2"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI2;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_CAMPORTHDMI_3"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTHDMI3;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_USBCAMERAPORT"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_CAMPORTUSB;
   }
   else if (0 == viewid.compare("Media#MediaPlayerScenes#M_USBROUTER"))
   {
      activityID = enActivityIDs__eActivityID_MEDIA_ROUTERUSB;
   }
   return activityID;
}


bool ApplicationSwitchClientHandler::onCourierMessage(const ActiveRenderedView& msg)
{
   uint16 activityID = 0;
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::onCourierMessage ActiveRenderedView surface id %d ,view name %s", msg.GetSurfaceId(), msg.GetViewName().GetCString()));
   if (msg.GetSurfaceId() == SURFACEID_MAIN_SURFACE_MEDIA)
   {
      m_sCurrentViewName = msg.GetViewName().GetCString();
      activityID = getMediaActivityID(m_sCurrentViewName);
      sendStoreContextRequest(activityID);
   }
   return false;
}


bool ApplicationSwitchClientHandler::onCourierMessage(const ApplicationStateUpdMsg& msg)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCourierMessage:ApplicationStateUpdMsg msg.GetState = %d", msg.GetState()));
   int32 iApplicationState = msg.GetState();
   clMetaData::getInstance().setApplicationState(iApplicationState);
   if (msg.GetState() == hmibase::IN_BACKGROUND &&
         (0 == m_sCurrentViewName.compare("Media#MediaPlayerScenes#M_VideoPlayer_FullScreen")))
   {
      sendStoreContextRequest(enActivityIDs__eActivityID_MEDIA_USB);
   }
   StateTransitionHandler::getInstance().setApplicationState(msg.GetState());
   return false;
}


void ApplicationSwitchClientHandler::sendStoreContextRequest(uint16 activityID)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::sendStoreContextRequest: %d", activityID));
   if (activityID != 0)
   {
      m_poApplicationSwitchProxy->sendSendContextRequest(*this, m_u8activeRegion, APPID_APPHMI_MEDIA, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)activityID);
   }
}


void ApplicationSwitchClientHandler::MediacontextSignal(uint32 regionid, uint32 currentAppid, uint32 currentactivityid, uint32 targetappid, uint32 targetactivityid)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::MediacontextSignal regionid %d", regionid));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::MediacontextSignal currentAppid %d", currentAppid));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::MediacontextSignal currentactivityid %d", currentactivityid));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::MediacontextSignal targetappid %d", targetappid));
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::MediacontextSignal targetactivityid %d", targetactivityid));

   m_poApplicationSwitchProxy->sendRequestContextSwitchRequest(*this, regionid, currentAppid, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)currentactivityid, targetappid, (bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch::enActivityIDs)targetactivityid);
}


/************************************************************************
*FUNCTION		: reqHomeScreenToggle
*DESCRIPTION	: func call context switch to source selection screen
*PARAMETER		: courier msg
*PARAMETER		: NA
*RETURNVALUE	: bool
************************************************************************/

void ApplicationSwitchClientHandler::reqHomeScreenToggle()
{
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::reqHomeScreenToggle"));
   ETG_TRACE_USR1(("ApplicationSwitchClientHandler::reqHomeScreenToggle"));
   _hmiInfoServiceProxyClient->sendToggleGadgetAudioSourceRequest(*this, m_u8activeRegion, APPID_APPHMI_MEDIA, APPID_APPHMI_TUNER);
}


void ApplicationSwitchClientHandler::setExternalMediaClientHandlerInstance(ExternalMediaClientHandler* _externalMediaClientHandlerInstance)
{
   if (_externalMediaClientHandlerInstance != NULL)
   {
      _externalMediaClientHandler = _externalMediaClientHandlerInstance;
   }
   else
   {
      ETG_TRACE_USR1(("ApplicationSwitchClientHandler::_externalMediaClientHandler is NULL"));
   }
}


void ApplicationSwitchClientHandler::setExternalMediaSourceHandlerInstance(ExternalMediaSourceHandler* _externalMediaSourceHandlerInstance)
{
   if (_externalMediaSourceHandlerInstance != NULL)
   {
      _externalMediaSourceHandler = _externalMediaSourceHandlerInstance;
      _externalMediaSourceHandler->addCamportUsbSourceAvailabilityObservor((*this));
   }
   else
   {
      ETG_TRACE_USR1(("MediaSourceListHandler::_externalMediaSourceHandler is NULL"));
   }
}


void ApplicationSwitchClientHandler::onCamportUsbSourceAvailabilityUpdate(int iUSBCamportAvailability)
{
   ETG_TRACE_USR4(("ApplicationSwitchClientHandler::onCamportUsbSourceAvailabilityUpdate iUSBCamportAvailability  %d", iUSBCamportAvailability));
   m_iUsbCamportAvailable = iUSBCamportAvailability;
}


#endif
}


}
