/* ***********************************;****************************************************
* FILE:          CGIAppController.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CGIAppController.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include "AppHmi_MasterStateMachine.h"
#include "App/Core/HmiAppCtrl/ProxyHandler_AppExt.h"
#include "ProjectBaseTypes.h"
#include "CGIAppController.h"
#include "Project.h"
#include "ApplicationSwitchConst.h"
#include "../RegionHandling/RegionHandling.h"
#include "App/Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponent.h"
#include "AppBase/ScreenBrokerClient/ScreenBrokerClient.h"
#include "ScreenBrokerSettings/Settings.h"
#include <Core/Utility/MasterUtility.h>

#define Enum_COCKPIT 2


#define STEERING_WHEEL_LONGRESS_TIMER 100 //TIME IN ms
using namespace ::bosch::cm::ai::hmi::appctrl::HMIAppCtrl;

#include "hmi_trace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/CGIAppController.cpp.trc.h"
#endif


//int CGIAppController::gadget_instanceid = 0;


/************************************************
Function 	: HKStatusChangedUpdMsg
Parameters 	: GetHKState, GetHKCode
Description : Implementation for consuming hard key press from framework

SYSFL/SWFL : Implemented SWFL 2562, SWFL 2633/SYSFL 4696, SYSFL 5519,SYSFL 5520,SYSFL 5633,SYSFL 5634
************************************************/
bool CGIAppController::onCourierMessage(const HKStatusChangedUpdMsg& msg)
{
   ETG_TRACE_USR4(("CGIAppController: HKStatusChangedUpdMsg HK_Code :%u HK_Status : %d", msg.GetHKCode(), msg.GetHKState()));

   bool msgConsumed = false;
   HardKeyCodeEnum hkcode = static_cast<HardKeyCodeEnum>(msg.GetHKCode());

   POST_MSG((COURIER_MESSAGE_NEW(ClonedHKStatusChangedUpdMsg)(hkcode, msg.GetHKState(), msg.GetDisplayID(), msg.GetUserData())));

#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
   if (IN_tu8AntitheftStatus != HMI_SM_C_HU_UNLOCKED)
   {
      ETG_TRACE_USR4(("DataModelSound: AntiTheft Enabled, HU is not Unlocked "));
      return true;
   }
#endif

#ifdef VARIANT_S_FTR_ENABLE_CABIN_A_SUPPORT

   if (msg.GetHKState() == hmibase::HARDKEYSTATE_UP)
   {
      AppTaskArray* p = AppTaskArray::getInstance();
      if (p)
      {
         const AppTask* task = p->getFollowerByHkCode(_lastAppId, msg.GetHKCode());

         if (task)
         {
            nextApp(task->getAppId());
            msgConsumed = true;
         }
      }
   }
#endif
   if (!msgConsumed)
   {
      switch (hkcode)
      {
         case HARDKEYCODE_HK_HOME:
         {
            if ((fnIsInf4cvSmartVariant()) && (msg.GetHKState() == hmibase::HARDKEYSTATE_LONG1))
            {
               POST_MSG((COURIER_MESSAGE_NEW(HandleSettingsMsg)(REGION_COCKPIT)))
            }
            msgConsumed = true;
         }
         break;
         case HARDKEYCODE_HK_SETTINGS:
         {
            if ((isValidKey(hkcode)) && (msg.GetHKState() == hmibase::HARDKEYSTATE_UP))
            {
               POST_MSG((COURIER_MESSAGE_NEW(HandleSettingsMsg)(REGION_COCKPIT)))
               msgConsumed = true;
            }
         }
         break;
         case HARDKEYCODE_HK_CABIN_SETTINGS:
         {
            if ((isValidKey(hkcode)) && (msg.GetHKState() == hmibase::HARDKEYSTATE_UP))
            {
               POST_MSG((COURIER_MESSAGE_NEW(HandleSettingsMsg)(fnGetLastActiveCabinRegionId())))
               msgConsumed = true;
            }
         }
         break;
         case HARDKEYCODE_REMOTECONTROL_SETUP:
         {
            if (msg.GetHKState() == hmibase::HARDKEYSTATE_UP)
            {
               POST_MSG((COURIER_MESSAGE_NEW(HandleSettingsMsg)(fnGetRegionId())))
               msgConsumed = true;
            }
         }
         break;
         case HARDKEYCODE_HK_MAP:
         case HARDKEYCODE_REMOTECONTROL_MAP:
         {
            if (msg.GetHKState() == hmibase::HARDKEYSTATE_UP)
            {
               hk_MapOutHandling();
               msgConsumed = true;
            }
         }
         break;
         case HARDKEYCODE_HK_PHONE_ACCEPT:
         case HARDKEYCODE_HK_PHONE_REJECT:
         case HARDKEYCODE_SWC_PHONE_PICKUP:
         case HARDKEYCODE_SWC_PHONE_REJECT:
         case HARDKEYCODE_HK_NAV:
         case HARDKEYCODE_HK_CAMERA:
         case HARDKEYCODE_HK_MCP_MON_CABIN:
         case HARDKEYCODE_REMOTECONTROL_TRAFFIC:
         case HARDKEYCODE_REMOTECONTROL_NAVI:
         case HARDKEYCODE_REMOTECONTROL_PHONE:
         case HARDKEYCODE_REMOTECONTROL_MEDIA:
         case HARDKEYCODE_REMOTECONTROL_TUNER:
         case HARDKEYCODE_REMOTECONTROL_VOICE:
         {
            ETG_TRACE_USR1(("HKStatusChangedUpdMsg :-> Keyforwading when msg is not consumed"));
            msgConsumed = CGIAppControllerProject::onCourierMessage(msg);
         }
         break;
         default:
            break;
      }
   }
   return msgConsumed;
}


void CGIAppController::hk_MapOutHandling()
{
   uint32 regionId = fnGetLastActiveCabinRegionId();

   ETG_TRACE_USR4(("CGIAppController: Mapout message fired ->: RegionId: %d", regionId));
   POST_MSG((COURIER_MESSAGE_NEW(ToggleMapOutStatus)(regionId)));
}


// ------------------------------------------------------------------------
/* bool CGIAppController::onCourierMessage(const JoystickStatusChangedUpdMsg& msg)
{
return false;
} */
// ------------------------------------------------------------------------
bool CGIAppController::onCourierMessage(const ApplicationRenderedUpdMsg& msg)
{
   //if (msg.GetSurfaceID() == static_cast<Courier::UInt32>(hmibase::SURFACEID_STATUSBAR))
   //{
   //   unsigned int surf[2] = {static_cast<unsigned int>(SURFACEID_MAIN_SURFACE_PLAYGROUND), static_cast<unsigned int>(hmibase::SURFACEID_STATUSBAR)};
   //   (void)removeSynchronizedSurfaces(surf, 2);
   //}
   //else
   //{
   //   (void)removeSynchronizedSurfaces(surface, sizeof(surface) / sizeof(unsigned int));
   //}

   // simply consume the message here, so it will not be processed by the base class
   return true;
}


// ------------------------------------------------------------------------

bool CGIAppController::onCourierMessage(const SubSurfaceVisibleMsg& msg)
{
   ETG_TRACE_USR1(("SubSurfaceVisibleMsg status %d", msg.GetStatus()));
   if (msg.GetDisplayRegion() == hmibase::STATUS_LINE)
   {
      hmibase::services::hmiappctrl::DisplayCmdPayload payload(displayData(0, hmibase::app::base::appHmi_Master, hmibase::app::base::appHmi_Master, hmibase::STATUS_LINE, 0, msg.GetStatus()));
      _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
   }
   else if (msg.GetDisplayRegion() == hmibase::FOOTER_LINE)
   {
      hmibase::services::hmiappctrl::DisplayCmdPayload payload(displayData(0, hmibase::app::base::appHmi_Master, hmibase::app::base::appHmi_Master, hmibase::FOOTER_LINE, 0, msg.GetStatus()));
      _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
   }
   return true;
}


// ------------------------------------------------------------------------
bool CGIAppController::onCourierMessage(const ActivateApplicationReqMsg& msg)
{
   enApplicationId appId = static_cast<enApplicationId>(msg.GetAppId());
   uint32 surface = msg.GetSurfaceId();
   std::string name;

   ETG_TRACE_USR1(("ActivateApplication %u", ETG_CENUM(enApplicationId, appId)));
   switch (appId)
   {
      case APPID_APPHMI_MASTER:
      {
         name = hmibase::app::base::appHmi_Master;
      }
      break;
      case APPID_APPHMI_TUNER:
      {
         name = hmi::apps::reference::appHmi_Tuner;
      }
      break;
      case APPID_APPHMI_MEDIA:
      {
         name = hmi::apps::reference::appHmi_Media;
      }
      break;
      case APPID_APPHMI_PHONE:
      {
         name = hmi::apps::reference::appHmi_Phone;
      }
      break;
      case APPID_APPHMI_NAVIGATION:
      {
         name = hmi::apps::reference::appHmi_Navigation;
      }
      break;
      case APPID_APPHMI_SYSTEM:
      {
         name = hmi::apps::reference::appHmi_System;
      }
      break;
      case APPID_APPHMI_TESTMODE:
      {
         name = hmi::apps::reference::appHmi_Testmode; //for testmode launch
      }
      break;
      case APPID_APPHMI_SDS:
      {
         name = hmi::apps::reference::appHmi_Sds;
      }
      break;
      case APPID_APPHMI_SPI:
      {
         name = hmi::apps::reference::appHmi_SPI;
      }
      break;
      case APPID_APPHMI_CAMERA:
      {
         name = hmi::apps::reference::appHmi_Camera;
      }
      break;
      case APPID_APPHMI_SXM:
      {
         name = hmi::apps::reference::appHmi_Sxm;
      }
      break;
      default:
         return false;
   }

   hmibase::services::hmiappctrl::DisplayCmdPayload payload(displayData(0, _hmiAppCtrlProxyHandler.getName(), name, static_cast<int32>(::bosch::cm::ai::hmi::appctrl::HMIAppCtrl::displayRegion__MAIN_VIEW), 0, msg.GetDisplayStatus()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
   if (msg.GetDisplayStatus() == 1)
   {
      POST_MSG((COURIER_MESSAGE_NEW(EnableConnectionReqMsg)(AudioConnectionRequest::VISIBLE_APPLICATION_REQUEST, appId, 0, 0, 0)))//master to undertand who next appl active
      POST_MSG((COURIER_MESSAGE_NEW(FirstApplicationActivatedUpdMsg)()))
      _lastAppId = appId;
   }
   return true;
}


bool CGIAppController::onCourierMessage(const ActivateMasterApplicationReqMsg& msg)
{
   hmibase::services::hmiappctrl::DisplayCmdPayload payload(displayData(0, _hmiAppCtrlProxyHandler.getName(), hmibase::app::base::appHmi_Master, static_cast<int32>(::bosch::cm::ai::hmi::appctrl::HMIAppCtrl::displayRegion__MAIN_VIEW), 0, msg.GetDisplayStatus()));
   _hmiAppCtrlProxyHandler.sendLocalMessageHMIAppCtrl(payload);
   return true;
}


bool CGIAppController::onCourierMessage(const EvalSubSurfaceReqMsg& msg)
{
   return true;
}


bool CGIAppController::isAppRunning(const AppTask* app)
{
   // check if app is running
   if (app->isRunning() == false && app->getExternalStart() == true)
   {
      std::string strCmd("systemctl start ");
      strCmd += app->getServiceName();
      system(strCmd.c_str());
   }
   //while (app.isRunning()==false)
   //   sleep(1000);
   return true;
}


/************************************************
Function 	: activateApp
Parameters 	: status
Description : This function is called to send request for MCP Application switch

SYSFL/SWFL : Implemented SWFL 2562
************************************************/
bool CGIAppController::activateApp(const AppTask* app)
{
   POST_MSG((COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("Popups#Scenes#HomeScreen_1_POPUPEXP_DEV"))));
   if (isAppRunning(app) == false)
   {
      return false;
   }
   if (app->getNeedsAudio())
   {
      const Courier::Int32 udataFM = 0; //acRequestSourceFM
      const Courier::Int32 udataAM = 1; //acRequestSourceAM()
      const Courier::Int32 udataMedia = 2; //acRequestSourceMedia()
      POST_MSG((COURIER_MESSAGE_NEW(EnableConnectionReqMsg)(AudioConnectionRequest::VISIBLE_APPLICATION_REQUEST, app->getAppId(), 0, 0, 0)))
   }

   ETG_TRACE_USR4(("CGIAppController: activateApp other applications HK_Code :%u", app->getAppId()));
   int region = (APPLICATIONSWITCH_TYPES::Region)(fnGetRegionId());

   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchFromHomeMsg)(region, sm_activeapplicationID, 0, app->getAppId(), 0)));
   _lastAppId = app->getAppId();
   return 	false;
}


/************************************************
Function 	: nextApp
Parameters 	: status
Description : This function is called to check the application in task array

SYSFL/SWFL : Implemented SWFL 2562
// SM access function for next application
************************************************/
void CGIAppController::nextApp(hmibase::appid_t appId)
{
   ETG_TRACE_USR4(("CGIAppController: nextApp %d", appId));
   AppTaskArray* p = AppTaskArray::getInstance();
   if (p)
   {
      const AppTask* app = 0;
      hmibase::appid_t currAppId = _lastAppId;

      if (appId > 0)
      {
         // switch to app ...
         app = p->getTaskByAppId(appId);
         if (app)
         {
            activateApp(app);
            return;
         }
         currAppId = 0;
      }
   }
}


/************************************************
Function 	: SurfaceStateChangedUpdMsg
Parameters 	: status
Description : This function is called to check the application in task array

SYSFL/SWFL : Implemented SWFL 2562
// SM access function for next application
************************************************/
bool CGIAppController::onCourierMessage(const SurfaceStateChangedUpdMsg& msg)
{
   ETG_TRACE_USR4(("CGIAppController::SurfaceStateChangedUpdMsg SurfaceId:%d, State%d", msg.GetSurfaceId(), msg.GetState()));
   POST_MSG((COURIER_MESSAGE_NEW(ClonedSurfaceStateChangedUpdMsg)(msg.GetSurfaceId(), msg.GetState())));
   if (msg.GetState() == (::hmibase::SURFACESTATE_VISIBLE))
   {
      uint32 u32surfaceID = msg.GetSurfaceId();
      if ((u32surfaceID == static_cast<uint32>(SUB_SURFACEID_APPFLYIN_SURFACE)) || (u32surfaceID == static_cast<uint32>(SUB_SURFACEID_COMMONBUTTON_SURFACE)))
      {
         ScreenBrokerClient::GetInstance().SetInputFocusPublic(u32surfaceID, true, true, 255, ScreenBrokerProtocol::InputFocus::Default);
      }
   }
   return CGIAppControllerProject::onCourierMessage(msg);
}


bool CGIAppController::isValidKey(uint32 hkcode)
{
   bool retvalue = false;
   uint32 regionID = fnGetRegionId();
   switch (hkcode)
   {
      case HARDKEYCODE_HK_SETTINGS:
      {
         if (regionID == REGION_COCKPIT)
         {
            retvalue = true;
         }
         break;
      }
      case HARDKEYCODE_HK_CABIN_SETTINGS:
      {
         if ((regionID == REGION_CABIN_A) || (regionID == REGION_CABIN_B))
         {
            retvalue = true;
         }
         break;
      }
      default:
         break;
   }
   return retvalue;
}


bool fnIsAVRoutingAdaperXMLAvailable()
{
   bool isValid = false;
   ::std::string tFileName = "/usr/lib/avdecc/plugins/AVRoutingAdapter.xml";
   if (!tFileName.empty())
   {
      struct stat buffer;
      if ((stat(tFileName.c_str(), &buffer) == 0) && (S_ISREG(buffer.st_mode)) && (buffer.st_size > 0))
      {
         isValid = true;
      }
   }
   return isValid;
}
