/**
 * @file         : DataModelSource.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelSource is to handle the data model implementation of
 *                 feature source list
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "DataModelSource.h"
#include "DataModelSourceDefines.h"
#include "ApplicationSwitchConst.h"
#include <View/CGI/CgiExtensions/ImageLoader.h>
#include <Core/VariantHandling/VariantHandling.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/AudioInterface/AudioInterfaceHelper.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>
#include <App/Core/MediaClientHandler/MediaClientHandlerTypes.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerInterface.h>
#include <Core/CabinConnectionHandler/CabinSrcInfo.h>


using namespace ::App::Core;
using namespace ::hmibase::apphmi_master::audio;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#define ETG_I_FILE_PREFIX App::DataModel::DataModelSource::
#include "trcGenProj/Header/DataModelSource.cpp.trc.h"
#endif


namespace App {
namespace datamodel {


DataModelSource::DataModelSource() : _userSelectedKey(SOURCE_KEY_NONE)
   , _variantHandling(NULL)
   , _mediaClientHandler(NULL)
   , _contextSwitchHandler(NULL)
   , _regionHandling(NULL)
   , _hmiInfoServiceServerComponent(NULL)
{
   ETG_TRACE_USR4(("DataModelSource: CTOR"));
}


DataModelSource::~DataModelSource()
{
   ETG_TRACE_USR4(("DataModelSource: DTOR"));
   _hmiInfoServiceServerComponent = NULL;
}


bool DataModelSource::onCourierMessage(const HKStatusUpdateMsg& oMsg)
{
   bool isMsgConsumed = false;
   switch (oMsg.GetHkCode())
   {
      case HARDKEYCODE_HK_SRC_COCKPIT:
      case HARDKEYCODE_HK_SRC_CABIN:
      {
         isMsgConsumed = true;
         const RegionSourcesInfo* imp = getRegionSourcesInfoForActiveSceneId();
         if ((NULL != imp) && (imp->getHKCode() == oMsg.GetHkCode()))
         {
            switch (oMsg.GetHkState())
            {
               case ::hmibase::HARDKEYSTATE_UP:
               {
                  toggleUserSelectedKey((*imp));
                  break;
               }
               case ::hmibase::HARDKEYSTATE_LONG1:
               {
                  if (_userSelectedKey != SOURCE_KEY_NONE)
                  {
                     activateSource((*imp), _userSelectedKey);
                  }
                  break;
               }
               default:
                  break;
            }
         }
      }
      default:
         break;
   }
   return isMsgConsumed;
}


bool DataModelSource::onCourierMessage(const SceneStatusInfoMsg& oMsg)
{
   bool isMsgConsumed = false;
   for (RegionsSourceInfo::iterator itr = _regionsSourceInfo.begin(); (itr != _regionsSourceInfo.end()); ++itr)
   {
      if ((*itr).getSceneId() == oMsg.GetViewId())
      {
         isMsgConsumed         = true;
         enSceneStatus tStatus = static_cast< enSceneStatus >(oMsg.GetActiveStatus());
         if ((*itr).getSceneStatus() != tStatus)
         {
            (*itr).setSceneStatus(tStatus);
            if (tStatus == SCENE_STATUS_ACTIVE)
            {
               _userSelectedKey = SOURCE_KEY_NONE;
               ListRegistry::s_getInstance().updateList((*itr).getListId());
            }
            else if (tStatus == SCENE_STATUS_INACTIVE)
            {
               (*itr).resetListDataProvider();
            }
            else
            {}
         }
         break;
      }
   }
   return isMsgConsumed;
}


bool DataModelSource::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool isMsgConsumed = false;
   const RegionSourcesInfo* imp = getRegionSourcesInfoForActiveSceneId();
   if ((NULL != imp) && (imp->getListId() == oMsg.GetListId()))
   {
      isMsgConsumed = true;
      activateSource((*imp), oMsg.GetHdl());
   }
   return isMsgConsumed;
}


void DataModelSource::onUpdateSourceListChange(const uint32 srcIndex)
{
   const RegionSourcesInfo* imp = getRegionSourcesInfoForActiveSceneId();
   if (NULL != imp)
   {
      for (SourcesInfo::const_iterator itr = _sourcesInfo.begin(); (itr != _sourcesInfo.end()); ++itr)
      {
         if ((*itr).getSourceIndex() == srcIndex)
         {
            const ::std::vector< uint32 >& tSrcKeysInfo = imp->getSourceKeysInfo();
            if (::std::find(tSrcKeysInfo.begin(), tSrcKeysInfo.end(), (*itr).getSourceKey()) != tSrcKeysInfo.end())
            {
               ListRegistry::s_getInstance().updateList(imp->getListId());
            }
            break;
         }
      }
   }
}


void DataModelSource::onUpdateConnectionInfo(std::map< int, Connection* > /* info */)
{
   const RegionSourcesInfo* imp = getRegionSourcesInfoForActiveSceneId();
   if (NULL != imp)
   {
      ListRegistry::s_getInstance().updateList(imp->getListId());
   }
}


void DataModelSource::onDeviceConnections(const ::std::vector< DeviceConnectionInfo >& info)
{
   bool isModified = (updateSourceDeviceId(info, "USB1", SOURCE_KEY_MEDIA_USB1)) ? true : false;
   isModified      = ((updateSourceDeviceId(info, "USB2", SOURCE_KEY_MEDIA_USB2)) || (isModified)) ? true : false;
   isModified      = ((updateSourceDeviceId(info, "IPOD", SOURCE_KEY_MEDIA_IPOD)) || (isModified)) ? true : false;
   if (isModified)
   {
      const RegionSourcesInfo* imp = getRegionSourcesInfoForActiveSceneId();
      if (NULL != imp)
      {
         ListRegistry::s_getInstance().updateList(imp->getListId());
      }
   }
}


void DataModelSource::toggleUserSelectedKey(const RegionSourcesInfo& info)
{
   tSharedPtrDataProvider tProvider = info.getListDataProvider();
   if ((NULL != _resourceController.get()) && (!tProvider.PointsToNull()) && (tProvider->listSize() > 0))
   {
      uint32 tNextIndex  = 0;
      uint32 tNextSrcKey = SOURCE_KEY_NONE;
      if (_userSelectedKey != SOURCE_KEY_NONE)
      {
         for (uint32 tIndex = 0; (tIndex < tProvider->listSize()); ++tIndex)
         {
            DataItem* tItem = (*tProvider)[tIndex].GetPointerToSharedInstance();
            if ((NULL != tItem) && (tItem->getHdlRow() == _userSelectedKey))
            {
               tNextIndex = ((tIndex + 1) < tProvider->listSize()) ? (tIndex + 1) : 0;
               break;
            }
         }
      }
      for (uint32 tIndex = tNextIndex; (tIndex < tProvider->listSize()); ++tIndex)
      {
         DataItem* tItem = (*tProvider)[tIndex].GetPointerToSharedInstance();
         if (NULL != tItem)
         {
            tNextIndex  = tIndex;
            tNextSrcKey = tItem->getHdlRow();
            break;
         }
      }
      if (tNextSrcKey != _userSelectedKey)
      {
         _userSelectedKey = tNextSrcKey;
         ListRegistry::s_getInstance().updateList(info.getListId());
         ::Courier::Message* msg = COURIER_MESSAGE_NEW(ListChangeMsg)(info.getListId(), ListChangeSet, ((tNextIndex / 0x12) * (0x12)));
         if (NULL != msg)
         {
            msg->Post();
         }
      }
   }
}


bool DataModelSource::updateSourceDeviceId(const ::std::vector< DeviceConnectionInfo >& info, const ::std::string& type, const uint32 srcKey)
{
   bool isUpdated  = false;
   SourceInfo* imp = getMutableSourceInfo(srcKey);
   if (NULL != imp)
   {
      int32 tDeviceId = -1;
      for (::std::vector< DeviceConnectionInfo >::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         if ((*itr).getDeviceType() == type)
         {
            tDeviceId = (*itr).getDeviceId();
            break;
         }
      }
      if (imp->getSourceDeviceId() != tDeviceId)
      {
         isUpdated = true;
         imp->setSourceDeviceId(tDeviceId);
      }
   }
   return isUpdated;
}


void DataModelSource::activateSource(const RegionSourcesInfo& info, const uint32 srcKey)
{
   if (NULL != _contextSwitchHandler)
   {
      for (SourcesInfo::const_iterator itr = _sourcesInfo.begin(); (itr != _sourcesInfo.end()); ++itr)
      {
         if ((*itr).getSourceKey() == srcKey)
         {
            ContextSwitchRequestInfo tInfo;
            tInfo.setRegionId(info.getRegionId());
            tInfo.setTargetContextId((*itr).getContextId());
            tInfo.setTargetApplicationId(getApplicationIdForSrc((*itr).getSourceIndex()));
            _contextSwitchHandler->onContextSwitchRequest(tInfo);
            ::Courier::ViewId tViewId(AppHmi_MasterStateMachineImpl::GetNameFromId(info.getSceneId()));
            ::Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, tViewId);
            if (NULL != msg)
            {
               msg->Post();
            }
            break;
         }
      }
   }
}


tSharedPtrDataProvider DataModelSource::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   if (NULL != _resourceController.get())
   {
      RegionSourcesInfo* imp = getMutableRegionSourcesInfoForActiveSceneId();
      if ((NULL != imp) && (imp->getListId() == oMsg.GetListId()))
      {
         SystemEnquiryInfo tSystemEnquiryInfo(imp->getRegionId(), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM);
         bool isValidToLoadNoList = (imp->getSceneId() == Popups_Scenes_SRC_Cabin) ? true : false;
         isValidToLoadNoList      = ((isValidToLoadNoList) && (NULL != _regionHandling)) ? true : false;
         isValidToLoadNoList      = ((isValidToLoadNoList) && (SYSTEM_STATUS_CONNECTED != _regionHandling->getStatus(tSystemEnquiryInfo))) ? true : false;
         if (isValidToLoadNoList)
         {
            ListDataProviderBuilder listBuilder(imp->getListId(), "FlexText_NoSource");
            listBuilder.AddItem(0, 0, "FlexText_NoSource");

            dataProvider = listBuilder.CreateDataProvider();
            imp->setListDataProvider(dataProvider);
            if (!dataProvider.PointsToNull())
            {
               dataProvider->setCacheOnOff(false);
            }
         }
         else
         {
            ETG_TRACE_USR4(("DataModelSource: getListDataProvider: ListId: %d", imp->getListId()));
            ListDataProviderBuilder listBuilder(imp->getListId(), "SRCButton");
            int32 tActiveSrcKey = getActiveSourceKey(imp->getRegionId());
            for (::std::vector< uint32 >::const_iterator itr = imp->getSourceKeysInfo().begin(); (itr != imp->getSourceKeysInfo().end()); ++itr)
            {
               const SourceInfo* imp1 = getSourceInfo((*itr));
               if (NULL != imp1)
               {
                  bool isSrcAvailable = false;
                  if (CabinSrcInfo::isCockpitSource(imp1->getSourceIndex()))
                  {
                     isSrcAvailable = (!imp1->getSourceDeviceIdStatus()) ? _resourceController->isSourceAvailable(imp1->getSourceIndex()) : \
                                      _resourceController->isSourceAvailable(imp1->getSourceIndex(), imp1->getSourceDeviceId());
                  }
                  else
                  {
                     if (_hmiInfoServiceServerComponent != NULL)
                     {
                        isSrcAvailable = _hmiInfoServiceServerComponent->getCabinSourceStatus(static_cast<int16>(imp1->getSourceIndex()));
                     }
                  }
                  ETG_TRACE_USR4(("DataModelSource: getListDataProvider: SourceIndex: %d, DeviceId: %d, IsSrcAvailable: %d", imp1->getSourceIndex(), imp1->getSourceDeviceId(), isSrcAvailable));
                  if (isSrcAvailable)
                  {
                     ListDataProviderBuilder::ListItem* listItem = &(listBuilder.AddItem((*itr), 0));
                     if (NULL != listItem)
                     {
                        ::Courier::DataBindingUpdater<SourceListItemInfoDataBindingSource>* listItemDB = listItem->template AddDataBindingUpdater<SourceListItemInfoDataBindingSource>();
                        if (NULL != listItemDB)
                        {
                           ::Courier::DataItemContainer<SourceListItemInfoDataBindingSource>* listItemData = &(listItemDB->GetValueContainer());
                           if (NULL != listItemData)
                           {
                              (*(*listItemData)).mItemText          = imp1->getSourceName();
                              (*(*listItemData)).mItemIsActive      = ((_userSelectedKey == (*itr)) || (tActiveSrcKey == (*itr))) ? true : false;
                              (*(*listItemData)).mItemNormalBitmap  = ImageLoader::getAssetBitmapImage(imp1->getSourceIconNormalBitmap().GetCString());
                              (*(*listItemData)).mItemActiveBitmap  = ImageLoader::getAssetBitmapImage(imp1->getSourceIconActiveBitmap().GetCString());
                              (*(*listItemData)).mItemPressedBitmap = ImageLoader::getAssetBitmapImage(imp1->getSourceIconPressedBitmap().GetCString());
                              (*listItemData).MarkAllItemsModified();
                           }
                        }
                     }
                  }
               }
            }
            dataProvider = listBuilder.CreateDataProvider();
            imp->setListDataProvider(dataProvider);
            if (!dataProvider.PointsToNull())
            {
               dataProvider->setCacheOnOff(false);
            }
         }
      }
   }
   return dataProvider;
}


void DataModelSource::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelSource: onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      RegionSourcesInfo* imp = getMutableRegionSourcesInfoForActiveSceneId();
      if ((NULL != imp) && (imp->getRegionId() == (*itr)))
      {
         ListRegistry::s_getInstance().updateList(imp->getListId());
         break;
      }
   }
}


SourcesInfo DataModelSource::getSourcesInfo() const
{
#define CONFIGURE_SOURCES_INFO
#include "SourcesInfoConfiguration.dat"
#undef CONFIGURE_SOURCES_INFO
}


RegionsSourceInfo DataModelSource::getRegionsSourceInfo() const
{
#define CONFIGURE_REGIONS_SOURCE_INFO
#include "SourcesInfoConfiguration.dat"
#undef CONFIGURE_REGIONS_SOURCE_INFO
}


const SourceInfo* DataModelSource::getSourceInfo(const uint32 key) const
{
   const SourceInfo* imp = NULL;
   for (SourcesInfo::const_iterator itr = _sourcesInfo.begin(); (itr != _sourcesInfo.end()); ++itr)
   {
      if ((*itr).getSourceKey() == key)
      {
         imp = (&(*itr));
         break;
      }
   }
   return imp;
}


const RegionSourcesInfo* DataModelSource::getRegionSourcesInfoForActiveSceneId() const
{
   const RegionSourcesInfo* imp = NULL;
   for (RegionsSourceInfo::const_iterator itr = _regionsSourceInfo.begin(); (itr != _regionsSourceInfo.end()); ++itr)
   {
      if ((*itr).getSceneStatus() == SCENE_STATUS_ACTIVE)
      {
         imp = (&(*itr));
         break;
      }
   }
   return imp;
}


uint32 DataModelSource::getActiveSourceKey(const uint32 regionId) const
{
   int32 tSrcIndex = SRC_INVALID;
   int32 tDeviceId = SRC_INVALID;
   uint32 tSrcKey  = SOURCE_KEY_NONE;
   if ((NULL != _connectionController.get()) && (_connectionController->fetchCurrentActiveSource(fnGetAudioSinkIdForRegion(regionId), tSrcIndex, tDeviceId)))
   {
      for (SourcesInfo::const_iterator itr = _sourcesInfo.begin(); (itr != _sourcesInfo.end()); ++itr)
      {
         if (((*itr).getSourceIndex() == tSrcIndex) && ((!(*itr).getSourceDeviceIdStatus()) || ((*itr).getSourceDeviceId() == tDeviceId)))
         {
            tSrcKey = (*itr).getSourceKey();
            break;
         }
      }
   }
   return tSrcKey;
}


bool DataModelSource::isSourceIdAvailable(const int16 sourceId) const
{
   bool status = false;
   if (NULL != _regionHandling)
   {
      status = _regionHandling->isSourceIdSupported(sourceId, SYSTEM_CATEGORY_NONE, REGION_INVALID);
   }
   ETG_TRACE_USR4(("DataModelSource: isSourceIdAvailable: SourceId: %d, Status: %d", sourceId, status));
   return status;
}


void DataModelSource::onCabinSourceStatusUpdate()
{
   ETG_TRACE_USR4(("DataModelSource: onCabinSourceStatusUpdate"));
   RegionSourcesInfo* imp = getMutableRegionSourcesInfoForActiveSceneId();
   if (imp != NULL)
   {
      ListRegistry::s_getInstance().updateList(imp->getListId());
   }
}


} //namespace DataModel
} //namespace App
