/**
 * @file         : DataModelHome.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelHome is to handle model implementation of HomeScreen
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "DataModelHome.h"
#include "ApplicationSwitchConst.h"
#include <CgiExtensions/ImageLoader.h>
#include <List/Data/ListDataProviderDistributor.h>
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/AudioInterface/AudioInterfaceHelper.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>
#include <Core/CabinConnectionHandler/CabinConnectionUtility.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>
#include <Core/CabinConnectionHandler/CabinConnectionHandlerDefines.h>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_01_if.h"
#include "dp_generic_if.h"
#endif


using namespace ::App::Core;
using namespace bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#define ETG_I_FILE_PREFIX App::DataModel::DataModelHome::
#include "trcGenProj/Header/DataModelHome.cpp.trc.h"
#endif

namespace App {
namespace datamodel {

#define INVALID_REGION -1

DataModelHome* DataModelHome::_theModelInstance = 0;


DataModelHome::DataModelHome(): _variantHandler(NULL)
   , _regionHandling(NULL)
   , _contextSwitchHandler(NULL)
   , _cabinConnectionUtility(NULL)
   , _cabinConnectionHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
   , m_u32CockpitLastActiveAudioGadgetInfo(0)
   , m_u32CabinALastActiveAudioGadgetInfo(0)
   , m_u32CabinBLastActiveAudioGadgetInfo(0)
{
   getGadgetInfoFromDP();

   _CockpitCabinContextData.MarkAllItemsModified();
   _CockpitCabinContextData.SendUpdate(true);
   _variantType = fnGetVariantType();
   _regionValue = INVALID_REGION;
   if (VehicleConfigurationReader::getInstance() != NULL)
   {
      VehicleConfigurationReader::getInstance()->registerForRegionUpdate(this);
   }
}


DataModelHome::~DataModelHome()
{
   _variantHandler = NULL;
   _regionHandling = NULL;
   _contextSwitchHandler = NULL;
   _cabinConnectionUtility = NULL;
   _cabinConnectionHandler = NULL;
   _hmiInfoServiceServerComponent = NULL;
   if (VehicleConfigurationReader::getInstance() != NULL)
   {
      VehicleConfigurationReader::getInstance()->deRegisterForRegionUpdate(this);
   }
}


void DataModelHome::deleteInstance()
{
   if (NULL != _theModelInstance)
   {
      delete _theModelInstance;
      _theModelInstance = NULL;
   }
}


DataModelHome& DataModelHome::getInstance()
{
   if (NULL == _theModelInstance)
   {
      _theModelInstance = new DataModelHome();
   }
   return *_theModelInstance;
}


void DataModelHome::updateRegionStatusValue(const uint8 regionValue)
{
   ETG_TRACE_USR4(("DataModelHome: updateRegionStatusValue %d ", regionValue));
   _regionValue = regionValue;
}


bool DataModelHome::isMapoutSupported()
{
   bool retVal = false;
   switch (_regionValue)
   {
      case EUR:
      case NAM:
      case ISR:
      case RES1:
      case RES2:
      case RES3:
      case RES4:
      case RES5:
      case RES6:
      case RES7:
      case RES8:
      case RES9:
      case RES10:
      case SNA:
         retVal = true;
         break;
      case TUR:
      case GCC:
         retVal = false;
         break;
      default:
         break;
   }
   return true;
}


void DataModelHome::intialize()
{
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->addMixSrcStatusListener(this);
   }
#endif
   if (NULL != _regionHandling)
   {
      _regionHandling->addCMAStatusListener((*this));
      if ((_variantType ==  VARIANT_TYPE_NAVI) || (_variantType == VARIANT_TYPE_CABIN_ONLY))
      {
         _regionHandling->addCMGStatusListener((*this));
      }
   }
   if (NULL != _hmiInfoServiceServerComponent)
   {
      _hmiInfoServiceServerComponent->addSpiDeviceInfoListener((*this));
      _hmiInfoServiceServerComponent->addVideoMixIndexListener((*this));
   }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->addMapOutStateListener((*this));
   }
#endif
   initListInfo();
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   updateMapoutTileonCMGUnavailable();
#endif
}


void DataModelHome::finalize()
{
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->removeMixSourceStatusListener(this);
   }
#endif
   if (NULL != _regionHandling)
   {
      _regionHandling->removeCMAStatusListener((*this));
      if ((_variantType ==  VARIANT_TYPE_NAVI) || (_variantType == VARIANT_TYPE_CABIN_ONLY))
      {
         _regionHandling->removeCMGStatusListener((*this));//Block Non Navi
      }
   }
   if (NULL != _hmiInfoServiceServerComponent)
   {
      _hmiInfoServiceServerComponent->removeSpiDeviceInfoListener((*this));
      _hmiInfoServiceServerComponent->removeVideoMixIndexListener((*this));
   }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->removeMapOutStateListener((*this));
   }
#endif
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
void DataModelHome::updateMapoutTileonCMGUnavailable()
{
   if ((NULL != _regionHandling) && ((_variantType ==  VARIANT_TYPE_NAVI) || (_variantType == VARIANT_TYPE_CABIN_ONLY)))
   {
      SupportedRegionsInfo tSupportedRegionsInfo = _regionHandling->getSupportedRegionsInfo();
      for (SupportedRegionsInfo::const_iterator itr = tSupportedRegionsInfo.begin(); (itr != tSupportedRegionsInfo.end()); ++itr)
      {
         if (_regionHandling->isCabinRegion((*itr).first))
         {
            SystemsIdInfo tInfo;
            _regionHandling->fetchSystemsIdInfo(tInfo, SYSTEM_CATEGORY_CMG, (*itr).first);
            if (tInfo.empty())
            {
               for (std::map< uint32, HomeListInfo >::iterator tItr = _homeListInfo.begin(); tItr != _homeListInfo.end(); ++tItr)
               {
                  HomeListInfo& info = tItr->second;
                  if (info._regionId == (*itr).first)
                  {
                     for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
                     {
                        if ((*vitr).getItemIndex() == GADGET_ITEM_INDEX_MAPOUT)
                        {
                           vitr->setGadgetContentAvailable(false);
                           vitr->setAnimationStatus(ANIMATION_COMPLETED);
                           break;
                        }
                     }
                     break;
                  }
               }
            }
         }
      }
   }
}


#endif


bool DataModelHome::onCourierMessage(const Courier::StartupMsg& /*oMsg*/)
{
   if ((App::Core::MasterUtility::getKDSVehicleType() == DP_INF4CV_VEHICLETYPE_COACH))
   {
      if (fnGetVariantType() ==  VARIANT_TYPE_CABIN_ONLY)
      {
         (*m_WidgetControllersData).mPagination_visibility = 0;
         (*m_WidgetControllersData).mSwipe_enable_dis      = 0;
         (*m_WidgetControllersData).mScroll_enable_dis     = 1;
         (*m_WidgetControllersData).mBG_scro_enable_dis    = 0;
      }
      else
      {
         (*m_WidgetControllersData).mPagination_visibility = 1;
         (*m_WidgetControllersData).mSwipe_enable_dis      = 1;
         (*m_WidgetControllersData).mScroll_enable_dis     = 0;
         (*m_WidgetControllersData).mBG_scro_enable_dis    = 1;
      }
   }
   else
   {
      (*m_WidgetControllersData).mPagination_visibility = 0;
      (*m_WidgetControllersData).mSwipe_enable_dis      = 0;
      (*m_WidgetControllersData).mScroll_enable_dis     = 1;
      (*m_WidgetControllersData).mBG_scro_enable_dis    = 0;
   }
   m_WidgetControllersData.MarkAllItemsModified();
   m_WidgetControllersData.SendUpdate(true);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HOMESCREEN_LEFT_VERTICAL_SWIPE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HOMESCREEN_RIGHT_VERTICAL_SWIPE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HS_CABINA_RIGHT_VERTICAL_SWIPE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE, this);
   ListRegistry::s_getInstance().addListImplementation(LIST_ID_HS_CABINB_RIGHT_VERTICAL_SWIPE, this);
   return false;
}


bool DataModelHome::onCourierMessage(const SceneStatusInfoMsg& oMsg)
{
   bool msgProcessed = false;
   switch (oMsg.GetViewId())
   {
      case M1_Scenes_MasterHome:
      {
         msgProcessed = true;
         updateSceneStatus(oMsg.GetViewId(), static_cast<enSceneStatus>(oMsg.GetActiveStatus()));
         activateDeactivateGadgets(oMsg.GetViewId(), static_cast<enSceneStatus>(oMsg.GetActiveStatus()));
         if (oMsg.GetActiveStatus() == SCENE_STATUS_INACTIVE)
         {
            resetDataProviders(oMsg.GetViewId());
         }
         break;
      }
      case M1_Scenes_HomeScreen_CabinA:
      case M1_Scenes_HomeScreen_CabinB:
      {
         msgProcessed = true;
         updateSceneStatus(oMsg.GetViewId(), static_cast<enSceneStatus>(oMsg.GetActiveStatus()));
         if (oMsg.GetActiveStatus() == SCENE_STATUS_ACTIVE)
         {
            performSubSystemUpdate();
         }
         else if (oMsg.GetActiveStatus() == SCENE_STATUS_INACTIVE)
         {
            activateDeactivateGadgets(oMsg.GetViewId(), static_cast<enSceneStatus>(oMsg.GetActiveStatus()));
            resetDataProviders(oMsg.GetViewId());
         }
         else
         {}
         break;
      }
      default:
         break;
   }
   return msgProcessed;
}


bool DataModelHome::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   bool msgProcessed = false;
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if ((_variantType ==  VARIANT_TYPE_NAVI) || (_variantType == VARIANT_TYPE_CABIN_ONLY))
   {
      switch (oMsg.GetListId())
      {
         case LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE:
         case LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE:
         {
            if ((NULL != _cabinConnectionHandler) && (NULL != _regionHandling))
            {
               uint32 regionId = _regionHandling->getRegionId();
               if ((oMsg.GetHdl() == GADGET_ITEM_INDEX_MAPOUT) && (oMsg.GetReaction() == enRelease))
               {
                  switch (oMsg.GetSubHdl())
                  {
                     case 0:
                     {
                        _cabinConnectionHandler->onToggleMapOutStatus(regionId);
                        break;
                     }
                     case 1:
                     {
                        bool isValid = false;
                        uint8 lastVideoMixIndex = getVideoMixIndex(regionId);
                        uint8 newVideoMixIndex = (lastVideoMixIndex == CABIN_VIDEO_MIX_INDEX_MAP_OUT) ? CABIN_VIDEO_MIX_INDEX_NONE : CABIN_VIDEO_MIX_INDEX_MAP_OUT;
                        if (newVideoMixIndex != lastVideoMixIndex)
                        {
                           if (newVideoMixIndex == CABIN_VIDEO_MIX_INDEX_NONE)
                           {
                              isValid = true;
                           }
                           else if (newVideoMixIndex == CABIN_VIDEO_MIX_INDEX_MAP_OUT)
                           {
                              if ((isGlassA2Configured(regionId)) && (isGlassSystemsValid(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX)))
                              {
                                 isValid = true;
                              }
                           }
                           else
                           {}
                        }
                        if (isValid)
                        {
                           _cabinConnectionHandler->onSetCabinVideoMixInfo(regionId, newVideoMixIndex);
                        }
                        break;
                     }
                     default:
                        break;
                  }
               }
            }
            msgProcessed = true;
            break;
         }
         default:
            break;
      }
   }
#endif
   return msgProcessed;
}


bool DataModelHome::onCourierMessage(const AnimationWidgetIndMsg& oMsg)
{
   bool msgProcessed = false;
   ListProviderEventInfo eventinfo;
   if (ListProviderEventInfo::GetItemIdentifierInfo(oMsg.GetSender(), eventinfo))
   {
      uint32 listId = eventinfo.getListId();
      uint32 hdlRow = eventinfo.getHdlRow();
      ETG_TRACE_USR4(("DataModelHome: AnimationWidgetIndMsg: ListId: %d : HdlRow : %d : IterationCount: %d", listId, hdlRow, oMsg.GetIterationCount()));
      std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.find(listId);
      if (itr != _homeListInfo.end())
      {
         HomeListInfo& info = itr->second;
         for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
         {
            uint32 tempHdl = (vitr->getInstanceId() == 0) ? vitr->getItemIndex() : vitr->getInstanceId();
            if (hdlRow == tempHdl)
            {
               if (40 == oMsg.GetIterationCount())
               {
                  vitr->setAnimationStatus(ANIMATION_COMPLETED);
                  updateListContent(listId, ::std::vector< uint32 >(1, tempHdl));
               }
               msgProcessed = true;
               break;
            }
         }
      }
   }
   return msgProcessed;
}


bool DataModelHome::onCourierMessage(const hmibase::gadget::ProviderUpdateIndicationMsg& oMsg)
{
   const std::vector< int32 >& instids  = oMsg.GetInstIds();
   ::std::map< uint32, ::std::vector< uint32 > > modifiedlist;
   for (std::vector< int32 >::const_iterator itr_instance = instids.begin(); (itr_instance != instids.end()); ++itr_instance)
   {
      ETG_TRACE_USR4(("DataModelHome: ProviderUpdateIndicationMsg: Id: %d", (*itr_instance)));
      bool terminate = false;
      int32 gadgetInstanceId = (*(itr_instance));
      for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); ((itr != _homeListInfo.end()) && (terminate == false)) ; ++itr)
      {
         HomeListInfo& info = itr->second;
         if ((isViewVisible(info._viewId)) && (!info._dataProvider.PointsToNull()))
         {
            for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
            {
               if (vitr->getInstanceId() == gadgetInstanceId)
               {
                  if ((vitr->getGadgetActiveStatus()) && (false == vitr->getGadgetContentAvailable()))
                  {
                     ETG_TRACE_USR4(("DataModelHome: ProviderUpdateIndicationMsg: GadgetContentAvailable: Id: %d", (*itr_instance)));
                     vitr->setGadgetContentAvailable(true);
                     vitr->setAnimationStatus(ANIMATION_COMPLETED);
                     uint32 tempHdl                                               = ((*vitr).getInstanceId() == 0) ? (*vitr).getItemIndex() : (*vitr).getInstanceId();
                     ::std::map< uint32, ::std::vector< uint32 > >::iterator mItr = modifiedlist.find(itr->first);
                     if (mItr == modifiedlist.end())
                     {
                        modifiedlist.insert(::std::make_pair(itr->first, ::std::vector< uint32 >()));
                        mItr = modifiedlist.find(itr->first);
                     }
                     if ((mItr != modifiedlist.end()) && (::std::find(mItr->second.begin(), mItr->second.end(), tempHdl) == mItr->second.end()))
                     {
                        mItr->second.push_back(tempHdl);
                     }
                  }
                  terminate = true;
                  break;
               }
            }
         }
      }
   }
   for (::std::map< uint32, ::std::vector< uint32 > >::iterator mItr = modifiedlist.begin(); (mItr != modifiedlist.end()); ++mItr)
   {
      updateListContent(mItr->first, mItr->second);
   }
   return true;
}


bool DataModelHome::onCourierMessage(const DataModelHomeListUpdateReqMsg& oMsg)
{
   ETG_TRACE_USR4(("DataModelHome: DataModelHomeListUpdateReqMsg::ListId:: %d", oMsg.GetListId()));
   updateMapOutTile(oMsg.GetListId(), true);
   return true;
}


void DataModelHome::updateListContent(const uint32 listId, const ::std::vector< uint32 >& hdlInfo)
{
   switch (listId)
   {
      case LIST_ID_HOMESCREEN_LEFT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE:
         updateLeftListContent(listId, hdlInfo);
         break;
      case LIST_ID_HOMESCREEN_RIGHT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINA_RIGHT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINB_RIGHT_VERTICAL_SWIPE:
         updateRightListContent(listId, hdlInfo);
         break;
      default:
         break;
   }
}


void DataModelHome::updateLeftListContent(const uint32 listId, const ::std::vector< uint32 >& hdlInfo)
{
   ETG_TRACE_USR4(("DataModelHome: updateLeftListContent: ListId: %d", listId));
   HomeListInfo* info = getHomeListInfo(listId);
   if ((info != NULL) && (!hdlInfo.empty()) && (!info->_dataProvider.PointsToNull()) && (isViewVisible(info->_viewId)))
   {
      ListDataProviderBuilder listBuilder(listId);
      for (::std::vector< uint32 >::const_iterator itr = hdlInfo.begin(); (itr != hdlInfo.end()); ++itr)
      {
         int32 currentPosition = -1;
         for (uint32 index = 0; (index < info->_dataProvider->listSize()); ++index)
         {
            tSharedPtrListDataItem dataItem = info->_dataProvider->operator[](index);
            if ((!dataItem.PointsToNull()) && (dataItem->getHdlRow() == (*itr)))
            {
               currentPosition = index;
               break;
            }
         }
         if (currentPosition > -1)
         {
            for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
            {
               uint32 tempHdl = ((*vitr).getInstanceId() == 0) ? (*vitr).getItemIndex() : (*vitr).getInstanceId();
               if (((*vitr).getGadgetVisibleStatus()) && (tempHdl == (*itr)))
               {
                  ETG_TRACE_USR4(("DataModelHome: updateLeftListContent: Id: %d, ContentStatus: %d, AnimationStatus: %d", tempHdl, vitr->getGadgetContentAvailable(), vitr->getAnimationStatus()));
                  ListDataProviderBuilder::ListItem* listItem = &((listBuilder.AddItem(tempHdl, 0UL, (*vitr).getGadgetDataItemContext())).SetIndexToUpdate(currentPosition));
                  updateLeftListItem(listItem, (*vitr), info->_regionId);
                  break;
               }
            }
         }
      }
      tSharedPtrDataProviderUpdater dataProvider = listBuilder.CreateDataProviderUpdater();
      if (!dataProvider.PointsToNull())
      {
         ::Courier::Message* msg = COURIER_MESSAGE_NEW(ListDataProviderUpdMsg)(dataProvider);
         if (NULL != msg)
         {
            msg->Post();
         }
      }
   }
}


void DataModelHome::updateRightListContent(const uint32 listId, const ::std::vector< uint32 >& hdlInfo)
{
   ETG_TRACE_USR4(("DataModelHome: updateRightListContent: ListId: %d", listId));
   HomeListInfo* info = getHomeListInfo(listId);
   if ((info != NULL) && (!hdlInfo.empty()) && (!info->_dataProvider.PointsToNull()) && (isViewVisible(info->_viewId)))
   {
      ListDataProviderBuilder listBuilder(listId);
      for (::std::vector< uint32 >::const_iterator itr = hdlInfo.begin(); (itr != hdlInfo.end()); ++itr)
      {
         int32 currentPosition = -1;
         for (uint32 index = 0; (index < info->_dataProvider->listSize()); ++index)
         {
            tSharedPtrListDataItem dataItem = info->_dataProvider->operator[](index);
            if ((!dataItem.PointsToNull()) && (dataItem->getHdlRow() == (*itr)))
            {
               currentPosition = index;
               break;
            }
         }
         if (currentPosition > -1)
         {
            for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
            {
               uint32 tempHdl = ((*vitr).getInstanceId() == 0) ? (*vitr).getItemIndex() : (*vitr).getInstanceId();
               if (((*vitr).getGadgetVisibleStatus()) && (tempHdl == (*itr)))
               {
                  ETG_TRACE_USR4(("DataModelHome: updateRightListContent: Id: %d, ContentStatus: %d, AnimationStatus: %d", vitr->getInstanceId(), vitr->getGadgetContentAvailable(), vitr->getAnimationStatus()));
                  ListDataProviderBuilder::ListItem* listItem = &((listBuilder.AddItem(tempHdl, 0UL, (*vitr).getGadgetDataItemContext())).SetIndexToUpdate(currentPosition));
                  updateRightListItem(listItem, (*vitr), info->_regionId);
                  break;
               }
            }
         }
      }
      tSharedPtrDataProviderUpdater dataProvider = listBuilder.CreateDataProviderUpdater();
      if (!dataProvider.PointsToNull())
      {
         ::Courier::Message* msg = COURIER_MESSAGE_NEW(ListDataProviderUpdMsg)(dataProvider);
         if (NULL != msg)
         {
            msg->Post();
         }
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
void DataModelHome::updateCabinMapOut(::Courier::DataItemContainer<HomescreenGadgetInfoDataBindingSource>* listItemData, const HomeListItemInfo& homeListItemInfo, const uint32 regionId)
{
   if (NULL != listItemData)
   {
      uint8 mapOutStatus = getMapOutStatus(regionId);
      (*(*listItemData)).mGadgetMapOutText = Candera::String(FeatStd::TextId(getMapOutStatusString(mapOutStatus)));
      if (mapOutStatus == STATE_CONNECTED)
      {
         (*(*listItemData)).mGadgetIcon    = ImageLoader::getAssetBitmapImage((homeListItemInfo).getBitmapIcon().c_str());
      }
      else
      {
         (*(*listItemData)).mGadgetIcon    = ImageLoader::getAssetBitmapImage((homeListItemInfo).getDisabledBitmapIcon().c_str());
      }
   }
}


void DataModelHome::updateCabinMixMapOut(ListDataProviderBuilder::ListItem* listItem,
      ::Courier::DataItemContainer<HomescreenGadgetInfoDataBindingSource>* listItemData, const HomeListItemInfo& homeListItemInfo, const uint32 regionId)
{
   if (NULL != listItem)
   {
      uint32 tempHdl = ((homeListItemInfo).getInstanceId() == 0) ? (homeListItemInfo).getItemIndex() : (homeListItemInfo).getInstanceId();
      listItem->AddId(tempHdl, 1UL);
      if (NULL != listItemData)
      {
         (*(*listItemData)).mGadgetVideoMixIconVisibility = false;
         (*(*listItemData)).mGadgetVideoMixText = Candera::String();
         if (isGlassA2Configured(regionId))
         {
            (*(*listItemData)).mGadgetVideoMixIconVisibility = true;
            (*(*listItemData)).mGadgetVideoMixIconEnableStatus = false;
            (*(*listItemData)).mGadgetVideoMixText = Candera::String(FeatStd::TextId("0x1A3A"));
            if ((NULL != _regionHandling) && (_regionHandling->isAllStatusSame(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED)))
            {
               if (isGlassSystemsValid(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX))
               {
                  (*(*listItemData)).mGadgetVideoMixIconEnableStatus = true;
                  if (getVideoMixIndex(regionId) == CABIN_VIDEO_MIX_INDEX_MAP_OUT)
                  {
                     (*(*listItemData)).mGadgetVideoMixText = Candera::String(FeatStd::TextId("0x1A3B"));
                  }
               }
            }
         }
      }
   }
}


#endif


bool DataModelHome::isRegionViewVisible(const uint32 regionId)
{
   bool isValid = false;
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); itr != _homeListInfo.end(); ++itr)
   {
      HomeListInfo& info = itr->second;
      if ((info._regionId == regionId) && (isViewVisible(info._viewId)))
      {
         isValid = true;
         break;
      }
   }
   ETG_TRACE_USR4(("DataModelHome: isRegionViewVisible: regionId: %d, isValid: %d", regionId, isValid));
   return isValid;
}


void DataModelHome::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelHome: onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      if (isRegionViewVisible((*itr)))
      {
         performSubSystemUpdate(true);
         break;
      }
   }
   ApplicationContextInfo activeContextInfo;
   if ((info.getStatus() != SYSTEM_STATUS_CONNECTED) && (NULL != _contextSwitchHandler) && (_contextSwitchHandler->fetchActiveContextInfo(activeContextInfo)))
   {
      if (::std::find(tRegionsIdInfo.begin(), tRegionsIdInfo.end(), activeContextInfo.getRegionId()) != tRegionsIdInfo.end())
      {
         ApplicationContextInfo defaultContextInfo;
         if (_contextSwitchHandler->fetchDefaultContextInfoForRegion(activeContextInfo.getRegionId(), defaultContextInfo))
         {
            if (activeContextInfo.getContextId() != defaultContextInfo.getContextId())
            {
               ContextSwitchRequestInfo reqInfo;
               reqInfo.setRegionId(defaultContextInfo.getRegionId());
               reqInfo.setSourceContextId(0);
               reqInfo.setSourceApplicationId(0);
               reqInfo.setTargetApplicationId(APPID_APPHMI_MASTER);
               reqInfo.setTargetContextId(defaultContextInfo.getContextId());
               _contextSwitchHandler->onContextSwitchRequest(reqInfo);
            }
         }
      }
   }
}


void DataModelHome::onCMAConnectedUpdate(const uint32 regionId, const bool update)
{
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); itr != _homeListInfo.end(); ++itr)
   {
      HomeListInfo& info = itr->second;
      if ((info._regionId == regionId) && (isViewVisible(info._viewId)))
      {
         updateMapOutTile(itr->first, false);
         activateDeactivateGadgetsForList(itr->first, SCENE_STATUS_ACTIVE);
         if (update)
         {
            ListRegistry::s_getInstance().updateList(itr->first);
         }
      }
   }
}


void DataModelHome::onCMADisConnectedUpdate(const uint32 regionId, const bool update)
{
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); itr != _homeListInfo.end(); ++itr)
   {
      HomeListInfo& info = itr->second;
      if ((info._regionId == regionId) && (isViewVisible(info._viewId)))
      {
         for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
         {
            vitr->setGadgetContentAvailable(false);
            vitr->setAnimationStatus(ANIMATION_COMPLETED);
            performGadgetActiveStatus(vitr->getInstanceId(), regionId, false);
         }
         if (update)
         {
            ListRegistry::s_getInstance().updateList(itr->first);
         }
      }
   }
}


void DataModelHome::onCMANoneUpdate(const uint32 regionId, const bool update)
{
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); (itr != _homeListInfo.end()); ++itr)
   {
      HomeListInfo& info = itr->second;
      if ((info._regionId == regionId) && (isViewVisible(info._viewId)))
      {
         for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
         {
            vitr->setGadgetContentAvailable(false);
         }
         if (update)
         {
            ListRegistry::s_getInstance().updateList(itr->first);
         }
      }
   }
}


void DataModelHome::onCMGStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelHome: onCMGStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      if (isRegionViewVisible((*itr)))
      {
         bool terminate = false;
         for (std::map< uint32, HomeListInfo >::iterator sItr = _homeListInfo.begin(); ((sItr != _homeListInfo.end()) && (terminate == false)); ++sItr)
         {
            HomeListInfo& info = sItr->second;
            if ((info._regionId == (*itr)) && (isViewVisible(info._viewId)))
            {
               switch (sItr->first)
               {
                  case LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE:
                  case LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE:
                     updateMapOutTile(sItr->first, true);
                     terminate = true;
                     break;
                  default:
                     break;
               }
            }
         }
         break;
      }
   }
}


void DataModelHome::performSubSystemUpdate(const bool update)
{
   if (NULL != _regionHandling)
   {
      uint32 regionId = _regionHandling->getRegionId();
      uint8 status    = _regionHandling->getStatus(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      ETG_TRACE_USR4(("DataModelHome: performSubSystemUpdate: regionId: %d, status: %d", regionId, status));
      if (status == SYSTEM_STATUS_NONE)
      {
         onCMANoneUpdate(regionId, update);
      }
      else if (status == SYSTEM_STATUS_CONNECTED)
      {
         onCMAConnectedUpdate(regionId, update);
      }
      else
      {
         onCMADisConnectedUpdate(regionId, update);
      }
   }
}


void DataModelHome::updateSceneStatus(const uint32 viewId, const enSceneStatus status)
{
   ::std::map <uint32, enSceneStatus>::iterator itr = _sceneStatus.find(viewId);
   if (itr == _sceneStatus.end())
   {
      _sceneStatus.insert(::std::make_pair(viewId, static_cast<enSceneStatus>(status)));
   }
   else if (itr->second != status)
   {
      itr->second = status;
   }
}


void DataModelHome::resetDataProviders(const uint32 viewId)
{
   for (::std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); (itr != _homeListInfo.end()); ++itr)
   {
      if (itr->second._viewId == viewId)
      {
         itr->second._dataProvider = tSharedPtrDataProvider();
      }
   }
}


void DataModelHome::updateMapOutTile(const uint32 listID, const bool update)
{
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   if ((_variantType ==  VARIANT_TYPE_NAVI) || (_variantType == VARIANT_TYPE_CABIN_ONLY))
   {
      ETG_TRACE_USR4(("DataModelHome: updateMapOutTile: listID: %d update:%d", listID, update));
      HomeListInfo* info = getHomeListInfo(listID);
      if ((NULL != info) && (NULL != _regionHandling) && (NULL != _cabinConnectionUtility))
      {
         for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
         {
            if ((vitr->getInstanceId() == 0) && (vitr->getItemIndex() == GADGET_ITEM_INDEX_MAPOUT) && (vitr->getGadgetVisibleStatus() == true))
            {
               bool isCMAConnected = _regionHandling->isAllStatusSame(SystemEnquiryInfo(info->_regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED);
               bool tIsAllSystemNone = true;
               bool tIsAnySystemConnected = false;
               uint8 tCategory = (getVideoMixIndex(info->_regionId) == CABIN_VIDEO_MIX_INDEX_NONE) ? SYSTEM_GROUP_CATEGORY_NONE : SYSTEM_GROUP_CATEGORY_CMG_CABIN;
               SinkIdsInfo tInfo = _cabinConnectionUtility->getGlassSinksInfo(info->_regionId, tCategory);
               for (SinkIdsInfo::iterator itr = tInfo.begin(); (itr != tInfo.end()); ++itr)
               {
                  bool isAllSystemNone = (_regionHandling->isAllStatusSame(SystemEnquiryInfo(info->_regionId, SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_NONE, (*itr)))               ? true : false;
                  if (!isAllSystemNone)
                  {
                     tIsAllSystemNone = false;
                  }
                  bool isAnySystemConnected = (_regionHandling->isAnyStatusSame(SystemEnquiryInfo(info->_regionId, SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED, (*itr))) ? true : false;
                  tIsAnySystemConnected = ((isAnySystemConnected) || (tIsAnySystemConnected)) ? true : false;
               }
               if (tIsAllSystemNone)
               {
                  if ((_cabinConnectionHandler != NULL) && (_cabinConnectionHandler->getCabinMapOutStatus(info->_regionId) == STATE_RESTORE))
                  {
                     vitr->setGadgetContentAvailable(true);
                  }
                  else
                  {
                     vitr->setGadgetContentAvailable(false);
                  }
               }
               else if ((tIsAnySystemConnected) && (isCMAConnected))
               {
                  vitr->setGadgetContentAvailable(true);
               }
               else
               {
                  vitr->setGadgetContentAvailable(false);
                  vitr->setAnimationStatus(ANIMATION_COMPLETED);
               }
               if (update)
               {
                  updateListContent(listID, ::std::vector< uint32 >(1, vitr->getItemIndex()));
               }
               break;
            }
         }
      }
   }
#endif
}


void DataModelHome::activateDeactivateGadgets(const uint32 viewId, const uint32 status)
{
   if (viewId == M1_Scenes_MasterHome)
   {
      POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ExternalImageProviderReqMsg)(CLOCK_GADGET, status)));
   }
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); itr != _homeListInfo.end(); ++itr)
   {
      HomeListInfo& info = itr->second;
      if (viewId == info._viewId)
      {
         activateDeactivateGadgetsForList(itr->first, status);
      }
   }
}


void DataModelHome::activateDeactivateGadgetsForList(const uint32 listId, const uint32 status)
{
   ETG_TRACE_USR4(("DataModelHome: activateDeactivateGadgetsFor List: %d, status: %d", listId, status));
   HomeListInfo* info = getHomeListInfo(listId);
   if (NULL != info)
   {
      for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
      {
         if ((vitr->getInstanceId() > 0) && (vitr->getGadgetVisibleStatus() == true))
         {
            uint8 textureStatus = (status == SCENE_STATUS_ACTIVE) ? TEXTURE_ACTIVE : TEXTURE_INACTIVE;
            bool activeStatus   = (status == SCENE_STATUS_ACTIVE) ? true : false;
            if (vitr->getGadgetActiveStatus() != activeStatus)
            {
               vitr->setGadgetActiveStatus(activeStatus);
               if (activeStatus)
               {
                  vitr->setGadgetContentAvailable((vitr->getAnimationStatus() == ANIMATION_COMPLETED) ? true : false);
               }
               else
               {
                  vitr->setGadgetContentAvailable((vitr->getAnimationStatus() == ANIMATION_COMPLETED) ? vitr->getGadgetContentAvailable() : false);
               }
               POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ExternalImageProviderReqMsg)(vitr->getInstanceId(), textureStatus)));
            }
         }
      }
   }
}


bool DataModelHome::isViewVisible(const uint32 viewId)
{
   bool isValid = false;
   ::std::map<uint32, enSceneStatus>::const_iterator itr = _sceneStatus.find(viewId);
   if ((itr != _sceneStatus.end()) && (itr->second == SCENE_STATUS_ACTIVE))
   {
      isValid = true;
   }
   return isValid;
}


void DataModelHome::updateDirectTextureRender(int sinkId, int srcId)
{
   ETG_TRACE_COMP(("DataModelHome: updateDirectTextureRender SrcId %d", srcId));
   uint32 regionId = fnGetRegionIdForAudioSink(sinkId);
   uint32 appid = getApplicationIdForSrc(srcId);
   switch (appid)
   {
      case APPID_APPHMI_MEDIA :
      {
         performGadgetVisibilityStatus(TUNER_GADGET, regionId, false);
         performGadgetVisibilityStatus(SXM_GADGET, regionId, false);
         performGadgetVisibilityStatus(MEDIA_GADGET, regionId, true, true);

         setGadgetInfoToDP(MEDIA_GADGET, regionId);
         break;
      }
      case APPID_APPHMI_TUNER:
      {
         performGadgetVisibilityStatus(MEDIA_GADGET, regionId, false);
         performGadgetVisibilityStatus(SXM_GADGET, regionId, false);
         performGadgetVisibilityStatus(TUNER_GADGET, regionId, true, true);

         setGadgetInfoToDP(TUNER_GADGET, regionId);
         break;
      }
      case APPID_APPHMI_SXM:
      {
         performGadgetVisibilityStatus(TUNER_GADGET, regionId, false);
         performGadgetVisibilityStatus(MEDIA_GADGET, regionId, false);
         performGadgetVisibilityStatus(SXM_GADGET, regionId, true, true);

         setGadgetInfoToDP(SXM_GADGET, regionId);
         break;
      }
      default:
         break;
   }
}


void DataModelHome::performGadgetActiveStatus(const uint32 gadgetInstanceId, const uint32 regionId, const bool status)
{
   bool isUpdated = false;
   ETG_TRACE_COMP(("DataModelHome: performGadgetActiveStatus::gadgetInstanceId: %d : activeStatus: %d ", gadgetInstanceId, status));
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); ((itr != _homeListInfo.end()) && (!isUpdated)); ++itr)
   {
      HomeListInfo& info = itr->second;
      for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
      {
         if ((vitr->getInstanceId() == gadgetInstanceId) && (info._regionId == regionId))
         {
            isUpdated = true;
            if (vitr->getGadgetActiveStatus() != status)
            {
               vitr->setGadgetContentAvailable((vitr->getAnimationStatus() == ANIMATION_COMPLETED) ? vitr->getGadgetContentAvailable() : false);
               if (!status)
               {
                  vitr->setGadgetActiveStatus(status);
                  POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ExternalImageProviderReqMsg)(vitr->getInstanceId(), TEXTURE_INACTIVE)));
               }
               else
               {
                  if (vitr->getGadgetVisibleStatus() == true)
                  {
                     vitr->setGadgetActiveStatus(status);
                     POST_MSG((COURIER_MESSAGE_NEW(::hmibase::gadget::ExternalImageProviderReqMsg)(vitr->getInstanceId(), TEXTURE_ACTIVE)));
                  }
               }
            }
            break;
         }
      }
   }
}


void DataModelHome::performGadgetVisibilityStatus(const uint32 gadgetInstanceId, const uint32 regionId, const bool status, const bool update)
{
   bool isUpdated = false;
   ETG_TRACE_COMP(("DataModelHome: performGadgetVisibilityStatus::gadgetInstanceId:: %d :: visibilityStatus:: %d ", gadgetInstanceId, status));
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); ((itr != _homeListInfo.end()) && (!isUpdated)); ++itr)
   {
      HomeListInfo& info = itr->second;
      for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
      {
         if ((vitr->getInstanceId() == gadgetInstanceId) && (info._regionId == regionId))
         {
            isUpdated = true;
            if ((vitr->getGadgetVisibleStatus() != status))
            {
               vitr->setGadgetVisibleStatus(status);
               if (isViewVisible(info._viewId))
               {
                  performGadgetActiveStatus(vitr->getInstanceId(), regionId, status);
                  if (update)
                  {
                     ListRegistry::s_getInstance().updateList(itr->first);
                  }
               }
            }
            break;
         }
      }
   }
}


void DataModelHome::onDeviceInfoUpdate(const std::string deviceName, const uint8 deviceType, const uint8 deviceStatus)
{
   ETG_TRACE_USR4(("DataModelHome: onDeviceInfoUpdate"));

   bool terminate = false;
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); ((itr != _homeListInfo.end()) && (terminate == false)); ++itr)
   {
      HomeListInfo& info = itr->second;
      for (std::vector< HomeListItemInfo >::iterator vitr = info._listItemInfoVector.begin(); (vitr != info._listItemInfoVector.end()); ++vitr)
      {
         if (vitr->getInstanceId() == SPI_GADGET)
         {
            bool update = (strcmp(deviceName.c_str(), vitr->getTouchAreaDynamicText().c_str()) != 0) ? true : false;
            if (update)
            {
               vitr->setTouchAreaDynamicText(deviceName.c_str());
               if (isViewVisible(info._viewId) && (vitr->getGadgetVisibleStatus() == true))
               {
                  updateListContent(itr->first, ::std::vector< uint32 >(1, vitr->getInstanceId()));
               }
            }
            terminate = true;
            break;
         }
      }
   }
}


void DataModelHome::onVideoMixIndexUpdate(const uint32 regionId, const uint8 videoMixIndex, const uint8 reason)
{
   bool terminate = false;
   for (std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.begin(); ((itr != _homeListInfo.end()) && (terminate == false)); ++itr)
   {
      HomeListInfo& info = itr->second;
      if (info._regionId == regionId)
      {
         switch (itr->first)
         {
            case LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE:
            case LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE:
            {
               //NOTE: CabinConnectionHandler updates Mapout status once the respective CB execution is done. So to
               //      avoid the wrong mapout status read, triggered the DataModelHomeListUpdateReqMsg instead of
               //      update list interface to terminate this execution and to update the list
               //      in next execution with correct mapout status.
               ::Courier::Message* msg = COURIER_MESSAGE_NEW(DataModelHomeListUpdateReqMsg)(itr->first, 0, 0, false, false);
               if (msg != NULL)
               {
                  msg->Post();
               }
               terminate = true;
               break;
            }
            default:
               break;
         }
      }
   }
}


void DataModelHome::initListInfo()
{
   switch (App::Core::MasterUtility::getKDSVehicleType())
   {
      case DP_INF4CV_VEHICLETYPE_COACH:
      {
         _homeListInfo = getCoachListInfo();
      }
      break;
      case DP_INF4CV_VEHICLETYPE_TRUCK:
      {
         _homeListInfo = getTruckListInfo();
      }
      break;
      case DP_INF4CV_VEHICLETYPE_SMART:
      {
         _homeListInfo = getSmartListInfo();
      }
      break;
      default:
         break;
   }
}


HomeListInfo* DataModelHome::getHomeListInfo(const uint32 listId)
{
   HomeListInfo* info = NULL;
   std::map< uint32, HomeListInfo >::iterator itr = _homeListInfo.find(listId);
   if (itr != _homeListInfo.end())
   {
      info = &itr->second;
   }
   return info;
}


::std::map<Courier::UInt32, HomeListInfo> const DataModelHome::getCoachListInfo() const
{
   if (_variantType ==  VARIANT_TYPE_NAVI)
   {
#include "HomesScreenListInfoCoach.dat"
   }
   else if (_variantType ==  VARIANT_TYPE_CABIN_ONLY)
   {
#include "HomesScreenListInfoCabinOnly.dat"
   }
   else
   {
#include "HomesScreenListInfoCoachNonNavi.dat"
   }
}


::std::map<Courier::UInt32, HomeListInfo> const DataModelHome::getSmartListInfo()const
{
#include "HomeScreenListInfoSmart.dat"
}


::std::map<Courier::UInt32, HomeListInfo> const DataModelHome::getTruckListInfo() const
{
   if (App::Core::MasterUtility::getVariantSelection_KDSValue() ==  DP_INF4CV_VARIANTSELECTION_NAVI)
   {
#include "HomeScreenListInfoTruck.dat"
   }
   else
   {
#include "HomeScreenListInfoSmart.dat"
   }
}


tSharedPtrDataProvider DataModelHome::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   ETG_TRACE_COMP(("DataModelHome: getListDataProvider::ListId:: %d::vehicleType:: %d", oMsg.GetListId(),  App::Core::MasterUtility::getKDSVehicleType()));
   switch (oMsg.GetListId())
   {
      case LIST_ID_HOMESCREEN_LEFT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE:
         dataProvider = getListDataProviderLeftList(oMsg.GetListId());
         break;
      case LIST_ID_HOMESCREEN_RIGHT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINA_RIGHT_VERTICAL_SWIPE:
      case LIST_ID_HS_CABINB_RIGHT_VERTICAL_SWIPE:
         dataProvider = getListDataProviderRightList(oMsg.GetListId());
         break;
      default:
         break;
   }
   return dataProvider;
}


tSharedPtrDataProvider DataModelHome::getListDataProviderLeftList(::Courier::UInt32 currentListID)
{
   ETG_TRACE_USR4(("DataModelHome: getListDataProviderLeftList::ListId:: %d", currentListID));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   HomeListInfo* info = getHomeListInfo(currentListID);
   if (info != NULL)
   {
      ListDataProviderBuilder listBuilder(currentListID);
      for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
      {
         if ((*vitr).getGadgetVisibleStatus())
         {
            uint32 tempHdl = ((*vitr).getInstanceId() == 0) ? (*vitr).getItemIndex() : (*vitr).getInstanceId();
            ETG_TRACE_USR4(("DataModelHome: getListDataProviderLeftList: Id: %d, ContentStatus: %d, AnimationStatus: %d", tempHdl, vitr->getGadgetContentAvailable(), vitr->getAnimationStatus()));
            ListDataProviderBuilder::ListItem* listItem = &(listBuilder.AddItem(tempHdl, 0UL, (*vitr).getGadgetDataItemContext()));
            updateLeftListItem(listItem, (*vitr), info->_regionId);
         }
      }
      dataProvider        = listBuilder.CreateDataProvider();
      info->_dataProvider = dataProvider;
      if (!dataProvider.PointsToNull())
      {
         dataProvider->setCacheOnOff(false);
      }
   }
   return dataProvider;
}


tSharedPtrDataProvider DataModelHome::getListDataProviderRightList(::Courier::UInt32 currentListID)
{
   ETG_TRACE_USR4(("DataModelHome: getListDataProviderRightList::ListId:: %d", currentListID));
   tSharedPtrDataProvider dataProvider = tSharedPtrDataProvider();
   HomeListInfo* info = getHomeListInfo(currentListID);
   if (info != NULL)
   {
      ListDataProviderBuilder listBuilder(currentListID);
      for (std::vector< HomeListItemInfo >::iterator vitr = info->_listItemInfoVector.begin(); (vitr != info->_listItemInfoVector.end()); ++vitr)
      {
         if ((*vitr).getGadgetVisibleStatus())
         {
            ETG_TRACE_USR4(("DataModelHome: getListDataProviderRightList: Id: %d, ContentStatus: %d, AnimationStatus: %d", vitr->getInstanceId(), vitr->getGadgetContentAvailable(), vitr->getAnimationStatus()));
            ListDataProviderBuilder::ListItem* listItem = &(listBuilder.AddItem((*vitr).getInstanceId(), 0UL, (*vitr).getGadgetDataItemContext()));
            updateRightListItem(listItem, (*vitr), info->_regionId);
         }
      }
      dataProvider        = listBuilder.CreateDataProvider();
      info->_dataProvider = dataProvider;
      if (!dataProvider.PointsToNull())
      {
         dataProvider->setCacheOnOff(false);
      }
   }
   return dataProvider;
}


void DataModelHome::updateLeftListItem(ListDataProviderBuilder::ListItem* listItem, HomeListItemInfo& homeListItemInfo, const uint32 regionId)
{
   bool _isMapoutSupported = isMapoutSupported();

   if (NULL != listItem)
   {
      ::Courier::DataBindingUpdater<HomescreenGadgetInfoDataBindingSource>* listItemDB  = listItem->template AddDataBindingUpdater<HomescreenGadgetInfoDataBindingSource>();
      if (NULL != listItemDB)
      {
         ::Courier::DataItemContainer<HomescreenGadgetInfoDataBindingSource>* listItemData = &(listItemDB->GetValueContainer());
         if (NULL != listItemData)
         {
            uint8 instID = homeListItemInfo.getInstanceId();
            (*(*listItemData)).mGadgetInstanceId          = homeListItemInfo.getInstanceId();
            (*(*listItemData)).mGadgetMargin              = homeListItemInfo.getGadgetMargin();
            (*(*listItemData)).mGadgetSize                = homeListItemInfo.getGadgetSize();
            (*(*listItemData)).mGadgetTouchAreaText       = homeListItemInfo.getTouchAreaDynamicText().empty() ?
                  Candera::String(::Courier::HashId(homeListItemInfo.getTouchAreaText().c_str()).HashValue()) :
                  Candera::String(homeListItemInfo.getTouchAreaDynamicText().c_str(), homeListItemInfo.getTouchAreaDynamicText().length());

            bool gadgetContentAvailable = homeListItemInfo.getGadgetContentAvailable();

            bool flag = false;

            if (_variantType == VARIANT_TYPE_CABIN_ONLY)
            {
               ETG_TRACE_USR4(("DataModelHome:updateLeftListItem getInstanceId==>%d gadgetContentAvailable==>%d mapoutEnabled==>%d", instID, gadgetContentAvailable, _isMapoutSupported));
               if (instID != MAPOUT_ID)
               {
                  flag = gadgetContentAvailable;
                  (*(*listItemData)).mGadgetContentAvailable = gadgetContentAvailable;
               }
               else
               {
                  if (_isMapoutSupported)
                  {
                     flag = gadgetContentAvailable;
                     (*(*listItemData)).mGadgetContentAvailable = gadgetContentAvailable;
                  }
                  else
                  {
                     flag = false;
                     (*(*listItemData)).mGadgetContentAvailable    = false;
                     (*(*listItemData)).mGadgetMapOutText = Candera::String("Mapout is not supported");
                  }
               }
            }
            else
            {
               flag = gadgetContentAvailable;
               (*(*listItemData)).mGadgetContentAvailable = gadgetContentAvailable;
            }

            if (flag == true)
            {
               (*(*listItemData)).mGadgetAnimationStatus     = false;
               (*(*listItemData)).mGadgetBg                  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapBg().c_str());
               (*(*listItemData)).mGadgetIconAreaBgNormal    = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapIconNormalBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaBgNormal   = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmaptouchAreaNormalBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaBgPressed  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmaptouchAreaPressedBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaTextColour = homeListItemInfo.getTextColor();
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
               if (homeListItemInfo.getItemIndex() == GADGET_ITEM_INDEX_MAPOUT)
               {
                  updateCabinMapOut(listItemData, homeListItemInfo, regionId);
                  updateCabinMixMapOut(listItem, listItemData, homeListItemInfo, regionId);
               }
               else
#endif
               {
                  (*(*listItemData)).mGadgetIcon = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapIcon().c_str());
               }
            }
            else
            {
               (*(*listItemData)).mGadgetBg                  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapBg().c_str());
               (*(*listItemData)).mGadgetIcon                = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapIcon().c_str());
               (*(*listItemData)).mGadgetNATransparentBg     = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapTransparentBg().c_str());
               (*(*listItemData)).mGadgetNATextPosition      = homeListItemInfo.getGadgetFunctionNAMargin();
               (*(*listItemData)).mGadgetNAAnimationPosition = homeListItemInfo.getGadgetLoadingIconMargin();

               enAnimationStatus animationStatus = homeListItemInfo.getAnimationStatus();
               ETG_TRACE_USR4(("DataModelHome:updateLeftListItem DISABLED BITMAPS animationStatus==>%d", animationStatus));

               if (animationStatus != ANIMATION_COMPLETED)
               {
                  homeListItemInfo.setAnimationStatus(ANIMATION_STARTED);
                  (*(*listItemData)).mGadgetAnimationStatus = true;
               }
               else
               {
                  (*(*listItemData)).mGadgetAnimationStatus   = false;

                  if (_variantType == VARIANT_TYPE_CABIN_ONLY)
                  {
                     ETG_TRACE_USR4(("DataModelHome:updateLeftListItem getInstanceId==>%d gadgetContentAvailable==>%d mapoutEnabled==>%d", instID, gadgetContentAvailable, _isMapoutSupported));
                     if (instID != MAPOUT_ID)
                     {
                        (*(*listItemData)).mGadgetFunctionAvailable = FUN_NOT_AVAIL;
                     }
                     else
                     {
                        if (_isMapoutSupported)
                        {
                           (*(*listItemData)).mGadgetFunctionAvailable = FUN_NOT_AVAIL;
                        }
                        else
                        {
                           (*(*listItemData)).mGadgetFunctionAvailable = MAP_NOT_SUP;
                        }
                     }
                  }
                  else
                  {
                     (*(*listItemData)).mGadgetFunctionAvailable = FUN_NOT_AVAIL;
                  }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
                  if (homeListItemInfo.getItemIndex() == GADGET_ITEM_INDEX_MAPOUT)
                  {
                     updateCabinMixMapOut(listItem, listItemData, homeListItemInfo, regionId);
                  }
#endif
               }
               (*listItemData).MarkAllItemsModified();
            }
         }
      }
   }
}


void DataModelHome::updateRightListItem(ListDataProviderBuilder::ListItem* listItem, HomeListItemInfo& homeListItemInfo, const uint32 regionId)
{
   if (NULL != listItem)
   {
      ::Courier::DataBindingUpdater<HomescreenGadgetInfoDataBindingSource>* listItemDB  = listItem->template AddDataBindingUpdater<HomescreenGadgetInfoDataBindingSource>();
      if (NULL != listItemDB)
      {
         ::Courier::DataItemContainer<HomescreenGadgetInfoDataBindingSource>* listItemData = &(listItemDB->GetValueContainer());
         if (NULL != listItemData)
         {
            (*(*listItemData)).mGadgetInstanceId          = homeListItemInfo.getInstanceId();
            (*(*listItemData)).mGadgetMargin              = homeListItemInfo.getGadgetMargin();
            (*(*listItemData)).mGadgetSize                = homeListItemInfo.getGadgetSize();
            (*(*listItemData)).mGadgetTouchAreaText       = homeListItemInfo.getTouchAreaDynamicText().empty() ?
                  Candera::String(::Courier::HashId(homeListItemInfo.getTouchAreaText().c_str()).HashValue()) :
                  Candera::String(homeListItemInfo.getTouchAreaDynamicText().c_str(), homeListItemInfo.getTouchAreaDynamicText().length());
            (*(*listItemData)).mGadgetContentAvailable    = homeListItemInfo.getGadgetContentAvailable();
            if (true  == homeListItemInfo.getGadgetContentAvailable())
            {
               (*(*listItemData)).mGadgetAnimationStatus     = false;
               (*(*listItemData)).mGadgetBg                  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapBg().c_str());
               (*(*listItemData)).mGadgetIconAreaBgNormal    = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapIconNormalBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaBgNormal   = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmaptouchAreaNormalBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaBgPressed  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmaptouchAreaPressedBg().c_str());
               (*(*listItemData)).mGadgetTouchAreaTextColour = homeListItemInfo.getTextColor();
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
               if (homeListItemInfo.getItemIndex() == GADGET_ITEM_INDEX_MAPOUT)
               {
                  updateCabinMapOut(listItemData, homeListItemInfo, regionId);
                  updateCabinMixMapOut(listItem, listItemData, homeListItemInfo, regionId);
               }
               else
#endif
               {
                  (*(*listItemData)).mGadgetIcon = ImageLoader::getAssetBitmapImage(homeListItemInfo.getBitmapIcon().c_str());
               }
            }
            else
            {
               (*(*listItemData)).mGadgetBg                  = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapBg().c_str());
               (*(*listItemData)).mGadgetIcon                = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapIcon().c_str());
               (*(*listItemData)).mGadgetNATransparentBg     = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapTransparentBg().c_str());
               (*(*listItemData)).mGadgetNATextPosition      = homeListItemInfo.getGadgetFunctionNAMargin();
               (*(*listItemData)).mGadgetNAAnimationPosition = homeListItemInfo.getGadgetLoadingIconMargin();
               (*(*listItemData)).mGadgetIcon = ImageLoader::getAssetBitmapImage(homeListItemInfo.getDisabledBitmapIcon().c_str());
               if (homeListItemInfo.getAnimationStatus() != ANIMATION_COMPLETED)
               {
                  homeListItemInfo.setAnimationStatus(ANIMATION_STARTED);
                  (*(*listItemData)).mGadgetAnimationStatus = true;
               }
               else
               {
                  (*(*listItemData)).mGadgetAnimationStatus   = false;
                  (*(*listItemData)).mGadgetFunctionAvailable = FUN_NOT_AVAIL;
               }
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
               if (homeListItemInfo.getItemIndex() == GADGET_ITEM_INDEX_MAPOUT)
               {
                  updateCabinMixMapOut(listItem, listItemData, homeListItemInfo, regionId);
               }
#endif
            }
            (*listItemData).MarkAllItemsModified();
         }
      }
   }
}


void DataModelHome::currentMixSourceConnectionStatus(const int16 sourceId, const bool status, const int16 sinkId, int mixSourceIndex)
{
   int micSource = MixSource_UNKNOWN;
   ETG_TRACE_USR4(("DataModelHome: currentMixSourceConnectionStatus sourceId: %d, status: %d, mixSourceIndex: %d", sourceId, status, mixSourceIndex));
   switch (mixSourceIndex)
   {
      case MixSource_MIC2:
      {
         micSource = (status == true) ? MixSource_MIC2 : MixSource_UNKNOWN;
      }
      break;
      case MixSource_MIC3:
      {
         micSource = (status == true) ? MixSource_MIC3 : MixSource_UNKNOWN;
      }
      break;
      case MixSource_NO_CARMODE:
      {
         micSource = (status == true) ? MixSource_NO_CARMODE : MixSource_UNKNOWN;
      }
      break;
      case MixSource_MIC2_CARMODE:
      {
         micSource = (status == true) ? MixSource_MIC2_CARMODE : (((NULL != _regionHandling) && (_regionHandling->getCarModeStatusUpdate())) ? MixSource_NO_CARMODE : MixSource_UNKNOWN);
      }
      break;
      case MixSource_MIC3_CARMODE:
      {
         micSource = (status == true) ? MixSource_MIC3_CARMODE : (((NULL != _regionHandling) && (_regionHandling->getCarModeStatusUpdate())) ? MixSource_NO_CARMODE : MixSource_UNKNOWN);
      }
      break;
      default:
         micSource = MixSource_UNKNOWN;
         break;
   }
   (*m_MICIconCabinData).mCabinMicSourceIndex = micSource;
   m_MICIconCabinData.MarkItemModified(::ItemKey::MicCabinStatus::CabinMicSourceIndexItem);
   m_MICIconCabinData.SendUpdate(false);
}


void DataModelHome::getGadgetInfoFromDP()
{
   dp_tclAppHmi_MasterLastActiveCockpitAudioGadget lastActiveCockpitAudioGadget;
   dp_tclAppHmi_MasterLastActiveCabinAAudioGadget  lastActiveCabinAAudioGadget;
   dp_tclAppHmi_MasterLastActiveCabinBAudioGadget  lastActiveCabinBAudioGadget;

   m_u32CockpitLastActiveAudioGadgetInfo = lastActiveCockpitAudioGadget.tGetData();
   m_u32CabinALastActiveAudioGadgetInfo  = lastActiveCabinAAudioGadget.tGetData();
   m_u32CabinBLastActiveAudioGadgetInfo  = lastActiveCabinBAudioGadget.tGetData();

   ETG_TRACE_USR4(("DataModelHome: getGadgetInfoFromDP, CockpitGadget: %d, CabinAGadget: %d, CabinBGadget: %d",
                   m_u32CockpitLastActiveAudioGadgetInfo, m_u32CabinALastActiveAudioGadgetInfo, m_u32CabinBLastActiveAudioGadgetInfo));
}


void DataModelHome::setGadgetInfoToDP(uint32 gadgetId, uint32 regionId)
{
   ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, gadgetId: %d, regionId: %d", gadgetId, regionId));
   ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, CockpitGadget: %d, CabinAGadget: %d, CabinBGadget: %d",
                   m_u32CockpitLastActiveAudioGadgetInfo, m_u32CabinALastActiveAudioGadgetInfo, m_u32CabinBLastActiveAudioGadgetInfo));

   switch (regionId)
   {
      case REGION_CABIN_A:
      {
         if (m_u32CabinALastActiveAudioGadgetInfo != gadgetId)
         {
            m_u32CabinALastActiveAudioGadgetInfo = gadgetId;
            dp_tclAppHmi_MasterLastActiveCabinAAudioGadget lastActiveCabinAAudioGadget;
            lastActiveCabinAAudioGadget.s32SetData(m_u32CabinALastActiveAudioGadgetInfo);
            ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, Data written to DP: CabinAGadget: %d", m_u32CabinALastActiveAudioGadgetInfo));
         }
      }
      break;

      case REGION_CABIN_B:
      {
         if (m_u32CabinBLastActiveAudioGadgetInfo != gadgetId)
         {
            m_u32CabinBLastActiveAudioGadgetInfo = gadgetId;
            dp_tclAppHmi_MasterLastActiveCabinBAudioGadget lastActiveCabinBAudioGadget;
            lastActiveCabinBAudioGadget.s32SetData(m_u32CabinBLastActiveAudioGadgetInfo);
            ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, Data written to DP: CabinBGadget: %d", m_u32CabinBLastActiveAudioGadgetInfo));
         }
      }
      break;

      case REGION_COCKPIT:
      {
         if (m_u32CockpitLastActiveAudioGadgetInfo != gadgetId)
         {
            m_u32CockpitLastActiveAudioGadgetInfo = gadgetId;
            dp_tclAppHmi_MasterLastActiveCockpitAudioGadget lastActiveCockpitAudioGadget;
            lastActiveCockpitAudioGadget.s32SetData(m_u32CockpitLastActiveAudioGadgetInfo);
            ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, Data written to DP: CockpitGadget: %d", m_u32CockpitLastActiveAudioGadgetInfo));
         }
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("DataModelHome: setGadgetInfoToDP, invalid regionId: %d", regionId));
      }
      break;
   }
}


bool DataModelHome::isLastActiveGadget(uint32 gadgetId, uint32 regionId) const
{
   ETG_TRACE_USR4(("DataModelHome: isLastActiveGadget, gadgetId: %d, regionId: %d", gadgetId, regionId));
   ETG_TRACE_USR4(("DataModelHome: isLastActiveGadget, CockpitGadget: %d, CabinAGadget: %d, CabinBGadget: %d",
                   m_u32CockpitLastActiveAudioGadgetInfo, m_u32CabinALastActiveAudioGadgetInfo, m_u32CabinBLastActiveAudioGadgetInfo));

   bool isLastActiveGadget = false;

   switch (regionId)
   {
      case REGION_CABIN_A:
      {
         if (m_u32CabinALastActiveAudioGadgetInfo == gadgetId)
         {
            isLastActiveGadget = true;
         }
      }
      break;

      case REGION_CABIN_B:
      {
         if (m_u32CabinBLastActiveAudioGadgetInfo == gadgetId)
         {
            isLastActiveGadget = true;
         }
      }
      break;

      case REGION_COCKPIT:
      {
         if (m_u32CockpitLastActiveAudioGadgetInfo == gadgetId)
         {
            isLastActiveGadget = true;
         }
      }
      break;

      default:
      {
         ETG_TRACE_USR4(("DataModelHome: isLastActiveGadget, invalid regionId: %d", regionId));
      }
      break;
   }

   return isLastActiveGadget;
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
void DataModelHome::onMapOutStatusUpdate(const uint32 regionId, const SinksMapOutInfo& /* info */)
{
   uint32 listId = (regionId == REGION_CABIN_A) ? LIST_ID_HS_CABINA_LEFT_VERTICAL_SWIPE : \
                   ((regionId == REGION_CABIN_B) ? LIST_ID_HS_CABINB_LEFT_VERTICAL_SWIPE : LIST_ID_HOMESCREEN_LEFT_VERTICAL_SWIPE);
   updateListContent(listId, ::std::vector< uint32 >(1, GADGET_ITEM_INDEX_MAPOUT));
}


const char* DataModelHome::getMapOutStatusString(const uint8 mapOutStatus)
{
   const char* mapOutText = "";

   switch (mapOutStatus)
   {
      case STATE_CONNECTED:
      {
         mapOutText = "0x13AB";   //"Map out enabled"
         break;
      }
      case STATE_DISCONNECTED:
      {
         mapOutText = "0x13AC";   //"Map out disabled"
         break;
      }
      case STATE_CONNECTING:
      {
         mapOutText = "0x1725";   //"Enabling Map out..."
         break;
      }
      case STATE_RESTORE:
      {
         mapOutText = "0x1A1D";   //Restoration is in progress
      }
      default:
         break;
   }

   return mapOutText;
}


uint8 DataModelHome::getMapOutStatus(const uint32 regionId) const
{
   uint8 mapOutStatus = STATE_UNKNOWN;
   if (NULL != _cabinConnectionHandler)
   {
      mapOutStatus = _cabinConnectionHandler->getCabinMapOutStatus(regionId);
   }
   return mapOutStatus;
}


uint8 DataModelHome::getVideoMixIndex(const uint32 regionId) const
{
   uint8 videoMixIndex = CABIN_VIDEO_MIX_INDEX_NONE;
   if (NULL != _hmiInfoServiceServerComponent)
   {
      videoMixIndex = _hmiInfoServiceServerComponent->getCabinVideoMixIndex(regionId);
   }
   return videoMixIndex;
}


bool DataModelHome::isGlassA2Configured(const uint32 regionId)
{
   bool isGlassA2Configured = false;
   if (NULL != _cabinConnectionUtility)
   {
      SinkIdsInfo tSinkIdsInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, SYSTEM_GROUP_CATEGORY_CMG_CABIN_MIX);
      {
         if (!tSinkIdsInfo.empty())
         {
            isGlassA2Configured = true;
         }
      }
   }
   return isGlassA2Configured;
}


bool DataModelHome::isGlassSystemsValid(const uint32 regionId, const uint8 systemGroupCategory)
{
   bool isValid = false;
   if (NULL != _cabinConnectionUtility)
   {
      SinkIdsInfo tSinkIdsInfo = _cabinConnectionUtility->getGlassSinksInfo(regionId, systemGroupCategory);
      for (SinkIdsInfo::iterator itr = tSinkIdsInfo.begin(); itr != tSinkIdsInfo.end(); ++itr)
      {
         if (_cabinConnectionUtility->isGlassSystemsValid((*itr)))
         {
            isValid = true;
            break;
         }
      }
   }
   return isValid;
}


#endif
} //namespace DataModel
} //namespace App
