/**
 * @file         : DataModelSound.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelSound is to handle model implementation of Sound features
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef _DataModelSound_H
#define _DataModelSound_H


#include "asf/core/Types.h"
#include "ProjectBaseTypes.h"
#include "AppHmi_MasterStateMachine.h"
#include "CgiExtensions/DataBindingItem.hpp"
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <App/Core/AudioInterface/SoundControllerPrjInterface.h>
#include <App/Core/VehicleSrvClient/IVehicleSrvClientInterface.h>
#include <App/Core/AudioInterface/ResourceControllerPrjInterface.h>


class SoundControllerPrj;


namespace App {
namespace datamodel {


class DataModelSound : public IMuteStateListenerCB
   , public IVolumeChangeListenerCB
   , public ISoundPropertyListenerCB
   , public ::App::Core::IVolumeCommandValueUpdateListenerCB
   , public ::App::Core::IVolumeCommandValueUpdateListenerCCVariant
   , public ::App::Core::ICMAStatusListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      virtual ~DataModelSound();
      static void deleteInstance();
      static DataModelSound& getInstance();
      void onMuteStateUpdate(const uint32 sinkId, const int state);
      void onVolumeChangeUpdate(const uint32 sinkId, const int volume);
      void performMuteRequest(uint32 regionID);
      void performMuteDemuteRequest(uint32 regionID, bool bMuteState);
      void onSoundPropertyUpdate(const int16 nSinkId);
      void onVolumeCommandValueUpdate(uint32 regionId, const uint8 volumeCmndValue);
      void onVolumeCommandValueUpdateCC(const uint32 regionID, const uint8 volumeLevel);
      void onCMAStatusUpdate(const ::App::Core::SystemStatusInfo& info);

      inline void initialize()
      {
         if (NULL != _regionHandling)
         {
            _regionHandling->addCMAStatusListener((*this));
         }
         if (NULL != _soundController.get())
         {
            _soundController->addMuteStateListener((*this));
            _soundController->addVolumeChangeListener((*this));
            _soundController->addSoundPropertyListener((*this));
         }
         if (NULL != _VehicleSrvClient)
         {
            _VehicleSrvClient->addVolumeCommandListener((*this));
            _VehicleSrvClient->addCCVariantVolumeListener((*this));
         }
      }

      inline void finalize()
      {
         if (NULL != _regionHandling)
         {
            _regionHandling->removeCMAStatusListener((*this));
         }
         if (NULL != _soundController.get())
         {
            _soundController->removeMuteStateListener((*this));
            _soundController->removeVolumeChangeListener((*this));
            _soundController->removeSoundPropertyListener((*this));
         }
         if (NULL != _VehicleSrvClient)
         {
            _VehicleSrvClient->removeVolumeCommandListener((*this));
            _VehicleSrvClient->removeCCVariantVolumeListener((*this));
         }
      }

      inline void setConnectionControllerPrj(const ::boost::shared_ptr< IConnectionControllerPrj >& imp)
      {
         _connectionControllerPrj = imp;
      }

      inline void setSoundController(const ::boost::shared_ptr< ISoundControllerPrj >& imp)
      {
         _soundController = imp;
      }

      inline void setVehicleSrvClientInstance(::App::Core::IVehicleSrvClient& imp)
      {
         _VehicleSrvClient = (&imp);
      }

      inline void setRegionHandling(::App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      COURIER_MSG_MAP_BEGIN(TR_CLASS_APPHMI_MASTER_COURIER_PAYLOAD_MODEL_COMP)
      ON_COURIER_MESSAGE(SetVolumeReqMsg)
      ON_COURIER_MESSAGE(SetMicVolumeReqMsg)
      ON_COURIER_MESSAGE(HKStatusUpdateMsg)
      ON_COURIER_MESSAGE(Courier::StartupMsg)
      ON_COURIER_MESSAGE(SetSystemMuteReqMsg)
      ON_COURIER_MESSAGE(ToggleMuteStateReqMsg)
      ON_COURIER_MESSAGE(EncoderStatusUpdateMsg)
      ON_COURIER_MESSAGE(Courier::UpdateModelMsg)
      ON_COURIER_MESSAGE(SetHmiStartupMuteRequest)
      COURIER_MSG_MAP_DELEGATE_DEFAULT_BEGIN()
      COURIER_MSG_MAP_DELEGATE_DEFAULT_ENDS()
      COURIER_MSG_MAP_ENDS();

      COURIER_BINDING_MAP_BEGIN()
      COURIER_BINDING_ITEM_CHANGE(VolumeInfoItem)
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      COURIER_BINDING_ITEM_CHANGE(MIC1VolumeInfoItem)
#endif
      COURIER_BINDING_MAP_END()

   private:
      /**
       *  Member Functions Declaration
       */
      DataModelSound();
      DataModelSound(const DataModelSound& obj);
      DataModelSound& operator= (const DataModelSound& obj);
      SoundControllerPrj* getSoundController() const;
      uint8 getRegionIdForKey(const uint8 code) const;
      bool isValidToProcessCabinRequest(const uint32 regionId) const;
      bool onCourierMessage(const SetVolumeReqMsg& oMsg);
      bool onCourierMessage(const SetMicVolumeReqMsg& oMsg);
      bool onCourierMessage(const HKStatusUpdateMsg& oMsg);
      bool onCourierMessage(const Courier::StartupMsg& oMsg);
      bool onCourierMessage(const SetSystemMuteReqMsg& oMsg);
      bool onCourierMessage(const ToggleMuteStateReqMsg& oMsg);
      bool onCourierMessage(const EncoderStatusUpdateMsg& oMsg);
      bool onCourierMessage(const SetHmiStartupMuteRequest& oMsg);
      bool onCourierBindingItemChange_VolumeInfoItem(const Courier::Request& request);
      void updateVolumeInfo(int val);
      void updateCabinAVolumeInfo(int val);
      void updateVolumeRequest(int VolumeChange, uint32 regionId);
      void PerformSWCVolumeChange(const uint32 u32HkCode);
      void PerformVolumeChange(const uint32 u32HkCode);
      bool PerformEncoderStatusChange(const uint32 u32HkCode);
      void updateSystemMuteInfo_Cockpit(int state);
      void updateSystemMuteInfo_CabinA(int state);
      void updateSystemMuteInfo_CabinB(int state);
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      void updateMicMuteInfo(const uint32 sinkId, int status);
      void updateMicVolumeInfo(const uint32 sinkId, int val);
      bool onCourierBindingItemChange_MIC1VolumeInfoItem(const Courier::Request& request);
#endif
      void updateCockpitMuteStateAtStartup(int status);
      void updateCabinAMuteStateAtStartup(int status);
      void updateCabinBMuteStateAtStartup(int status);
      void updateMuteStatusAtStartup();
      void updateDataBindningVariablesAtStartup();
      void updateSystemMuteForCMAStatus(const uint32 regionId, const uint8 status, const uint8 systemId);
      void updateVolumeAndMuteForCMAStatus(const uint32 regionId, const uint8 status, const uint8 systemId);
      bool isCabinVolumeAndMuteApplicable(const uint8 status);
      /**
       *  Member Variables Declaration
       */
      uint8 _currentVolume;
      std::map<uint8, uint8> _volumeReqCount;
      ::boost::shared_ptr< ISoundControllerPrj > _soundController;
      ::boost::shared_ptr< IConnectionControllerPrj > _connectionControllerPrj;
      DataBindingItem<VolumeInfoDataBindingSource> _volumeInfoData;
      DataBindingItem<SystemMuteInfoDataBindingSource> _SystemMuteInfoData;
      ::App::Core::IVehicleSrvClient* _VehicleSrvClient;
      ::App::Core::IRegionHandling* _regionHandling;
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      DataBindingItem<MIC1VolumeInfoDataBindingSource> m_MICVolData;
#endif
      static DataModelSound* _theModelInstance;
};


} //namespace DataModel
} //namespace App


#endif /* _DataModelSound_H */
