/**
 * @file         : DataModelControlPanel.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : DataModelControlPanel is to handle the data model implementation of
 *                 control panel feature.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "DataModelControlPanel.h"
#include "ApplicationSwitchConst.h"
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>


using namespace ::App::Core;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM
#include "trcGenProj/Header/DataModelControlPanel.cpp.trc.h"
#endif


namespace App {
namespace datamodel {


DataModelControlPanel::DataModelControlPanel()
{
   clear();
   initActivationLockedContextsInfo();
}


DataModelControlPanel::~DataModelControlPanel()
{
   clear();
}


DataModelControlPanel::DataModelControlPanel(const DataModelControlPanel& obj)
{
   clear();
   *this = obj;
}


DataModelControlPanel& DataModelControlPanel::operator= (const DataModelControlPanel& obj)
{
   if (this != (&obj))
   {
      clear();
      _displayStatus                = obj._displayStatus;
      _queuedReqStatus              = obj._queuedReqStatus;
      _displayReqStatus             = obj._displayReqStatus;
      _contextSwitchHandler         = obj._contextSwitchHandler;
      _activationLockedContextsInfo = obj._activationLockedContextsInfo;
   }
   return (*this);
}


void DataModelControlPanel::finalize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->removeRegionListener((*this));
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _regionHandling->removeCMAStatusListener((*this));
#endif
   }
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->removeActiveContextListener((*this));
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->removeMic1StatusListener(this);
   }
#endif
}


void DataModelControlPanel::initialize()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->addRegionListener((*this));
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _regionHandling->addCMAStatusListener((*this));
#endif
   }
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->addActiveContextListener((*this));
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->addMic1StatusListener(this);
   }
#endif
}


void DataModelControlPanel::initActivationLockedContextsInfo()
{
   if (_activationLockedContextsInfo.empty())
   {
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_RVC);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_RVC_EARLY);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_DIAGNOSIS);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_RVC_SETTINGS);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_SECURITY_APP);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_SPI_FULL_SCREEN);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_LEGAL_DISCLAIMER);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_SPI_APPLICATION_LIST);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_TESTMODE_WITHOUTNOTCH_SCREEN);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_TESTMODE_SOURCESELECTION_SCREEN);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_NAVI_TESTMODE_VIEW);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_SETTINGS_SWDL_MIS_STAGE1_PROGRESS);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_SETTINGS_SWDL_MIS_STAGE1_RESULT);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_KDS_INPROGRESS);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_NAVI_TRAFFIC_DETOUR);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_MIC1);
      _activationLockedContextsInfo.push_back(enActivityIDs__eActivityID_MIC_SETTINGS);
   }
}


bool DataModelControlPanel::onCourierMessage(const SubSurfaceUpdMsg& oMsg)
{
   bool isConsumed = false;
   if (oMsg.GetDisplayRegion() == ::hmibase::STATUS_LINE)
   {
      isConsumed = true;
      switch (oMsg.GetState())
      {
         case ::hmibase::IN_BACKGROUND:
         case ::hmibase::TO_BACKGROUND:
         {
            if (isDisplayStatusSameAsRequest(false))
            {
               _displayStatus    = false;
               _displayReqStatus = -1;
            }
            break;
         }
         case ::hmibase::IN_FOREGROUND:
         case ::hmibase::FORCE_IN_FOREGROUND:
         {
            if (isDisplayStatusSameAsRequest(true))
            {
               _displayStatus    = true;
               _displayReqStatus = -1;
            }
            break;
         }
         default:
            break;
      }
      if ((!isSwitchOngoing()) && (_queuedReqStatus != -1))
      {
         bool temp        = static_cast< bool >(_queuedReqStatus);
         _queuedReqStatus = -1;
         executeCommonButtonActivationRequest(temp);
      }
   }
   return isConsumed;
}


bool DataModelControlPanel::onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg)
{
   _mic1KeyInfoDB.SendUpdate(true);
   _controlPanelInfoDB.SendUpdate(true);
   return false;
}


bool DataModelControlPanel::onCourierMessage(const CommonButtonVisibleStatusUpdMsg& oMsg)
{
   if (!oMsg.GetVisibleStatus())
   {
      evalCommonButtonActivationRequest(false);
   }
   else
   {
      ApplicationContextInfo tInfo;
      if ((NULL != _contextSwitchHandler) && (_contextSwitchHandler->fetchActiveContextInfo(tInfo)))
      {
         ::std::vector< uint32 >::const_iterator itr = ::std::find(_activationLockedContextsInfo.begin(), _activationLockedContextsInfo.end(), tInfo.getContextId());
         evalCommonButtonActivationRequest((itr != _activationLockedContextsInfo.end()) ? false : true);
      }
   }
   return true;
}


bool DataModelControlPanel::onCourierMessage(const ExecuteCommonButtonActivationRequestMsg& oMsg)
{
   executeCommonButtonActivationRequest(oMsg.GetActiveStatus());
   return true;
}


bool DataModelControlPanel::onCourierMessage(const ControlPanelButtonPressUpdate& msg)
{
   if (NULL != _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent->sendControlPanelButtonPressUpdateSignal(msg.GetControlPanelButton(), msg.GetButtonstate(), msg.GetReceiverAppID());
   }
   return true;
}


void DataModelControlPanel::onRegionUpdate(const uint32 regionId)
{
   uint8 variantType = fnGetVariantType();

   ETG_TRACE_USR4(("DataModelControlPanel:onRegionUpdate mContext==>%d regionId ==>%d fnGetVariantType==>%d ", (*_controlPanelInfoDB).mContext, regionId, variantType));
   if ((*_controlPanelInfoDB).mContext != regionId)
   {
      if (variantType ==  VARIANT_TYPE_CABIN_ONLY)
      {
         (*_controlPanelInfoDB).mContext = CabinOnly_Notchcnt;
         (*_controlPanelInfoDB).mCPGrpLineContect = CP_GrpLine_CabinOnly;
      }
      else
      {
         (*_controlPanelInfoDB).mContext = regionId;
         (*_controlPanelInfoDB).mCPGrpLineContect = CP_GrpLine_Default;
      }
      _controlPanelInfoDB.MarkItemModified(::ItemKey::ControlPanelInfo::ContextItem);
      _controlPanelInfoDB.SendUpdate(false);
   }
}


void DataModelControlPanel::onActiveContextUpdate(const ApplicationContextInfo& info)
{
   bool isValid = getCommonButtonVisibility(info.getContextId());
   if ((!isValid) || ((isValid) && ((NULL != _contextSwitchHandler) && (info.getApplicationId() == _contextSwitchHandler->getActiveApplicationId()))))
   {
      evalCommonButtonActivationRequest(isValid);
   }
}


void DataModelControlPanel::onApplicationSwitchComplete(const uint32 appId)
{
   ApplicationContextInfo contextInfo;
   if ((NULL != _contextSwitchHandler) && (_contextSwitchHandler->fetchActiveContextInfo(contextInfo)))
   {
      bool isValid = getCommonButtonVisibility(contextInfo.getContextId());
      if ((appId == contextInfo.getApplicationId()) && (isValid))
      {
         evalCommonButtonActivationRequest(isValid);
      }
   }
}


bool DataModelControlPanel::getCommonButtonVisibility(const uint32 contextId)
{
   ::std::vector< uint32 >::const_iterator itr = ::std::find(_activationLockedContextsInfo.begin(), _activationLockedContextsInfo.end(), contextId);
   bool isValid = (itr != _activationLockedContextsInfo.end()) ? false : true;
   return isValid;
}


void DataModelControlPanel::evalCommonButtonActivationRequest(const bool status)
{
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(EvalCommonButtonActivationRequestMsg)(status);
   if (NULL != msg)
   {
      msg->Post();
   }
}


void DataModelControlPanel::executeCommonButtonActivationRequest(const bool status)
{
   if (!isSwitchOngoing())
   {
      if ((!isDisplayStatusSameAsCurrent(status)) && (!isDisplayStatusSameAsRequest(status)))
      {
         ::Courier::Message* msg = COURIER_MESSAGE_NEW(SubSurfaceVisibleMsg)("appHmi_Master", hmibase::STATUS_LINE, 0x01, status);
         if (NULL != msg)
         {
            msg->Post();
            _displayReqStatus = status;
         }
      }
   }
   else
   {
      _queuedReqStatus = status;
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void DataModelControlPanel::currentMIC1ConnectionStatus(const bool status)
{
   ETG_TRACE_USR4(("DataModelControlPanel: currentMIC1ConnectionStatus status:%d", status));
   //Courier Msg to activate Mic ControlPanelPopUp
   Courier::Message* msg = COURIER_MESSAGE_NEW(MicStatus)(status);
   if (msg != 0)
   {
      msg->Post();
   }
}


void DataModelControlPanel::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   ETG_TRACE_USR4(("DataModelControlPanel onCMAStatusUpdate: Status: %d, SystemId: %d", info.getStatus(), info.getSystemId()));
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      updateMicIcon((*itr), info.getStatus());
   }
}


void DataModelControlPanel::updateMicIcon(const uint32 regionId, const uint8 status)
{
   if (regionId == Enum_REGION_CABIN_A)
   {
      bool micIconEnableStatus = isMicIconEnable(status);
      if (micIconEnableStatus != (*_controlPanelInfoDB).mMicIconEnableStatus)
      {
         (*_controlPanelInfoDB).mMicIconEnableStatus = micIconEnableStatus;
         _controlPanelInfoDB.MarkItemModified(::ItemKey::ControlPanelInfo::MicIconEnableStatusItem);
         _controlPanelInfoDB.SendUpdate(false);
      }
   }
}


void DataModelControlPanel::onUpdateMic1Status(const bool micStatus, const uint8 keyType)
{
   ETG_TRACE_USR4(("MasterHall: onUpdateMic1Status micStatus:%d, keyType:%d", micStatus, keyType));
   bool mic1TouchabilityStatus = ((keyType == MIC_HARDKEY) && (micStatus)) ? false : true;
   if (mic1TouchabilityStatus != (*_mic1KeyInfoDB).mMic1TouchabilityStatus)
   {
      (*_mic1KeyInfoDB).mMic1TouchabilityStatus = mic1TouchabilityStatus;
      _mic1KeyInfoDB.MarkItemModified(::ItemKey::Mic1KeyInfo::Mic1TouchabilityStatusItem);
      _mic1KeyInfoDB.SendUpdate(false);
   }
}


bool DataModelControlPanel::isMicIconEnable(const uint8 status)
{
   bool isEnabled = false;
   switch (status)
   {
      case SYSTEM_STATUS_CONNECTED:
      case SYSTEM_STATUS_HOURLOGIC_DISCONNECTED:
      case SYSTEM_STATUS_PNM_DISCONNECTED:
      {
         isEnabled = true;
         break;
      }
      default:
      {
         break;
      }
   }
   return isEnabled;
}


#endif
} //namespace datamodel
} //namespace App
