/* ***************************************************************************************
* FILE:          DataModel.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  DataModel.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2017-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "gui_std_if.h"
#include <string>
#include <stdio.h>
#include <sstream>
#include "DataModel.h"
#include "ApplicationSwitch.h"
#include "ApplicationSwitchConst.h"
#include "AppHmi_MasterStateMachine.h"
#include <App/Core/RegionHandling/RegionHandling.h>
#include <App/Core/SyncDisplay/SyncDisplayHandler.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerInterface.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_DM

#define ETG_I_FILE_PREFIX                 App::DataModel::DataModel::
#include "trcGenProj/Header/DataModel.cpp.trc.h"
#endif

using namespace std;
using namespace ::App::Core;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;

namespace App {
namespace datamodel {
//const uint32 HMI_C_PRECODE  = 0x11111111;
//const uint32 HMI_C_PRECODE  = 0xFFFFFFFF;
const uint32 HMI_C_DEFAULTVIN = 0xFFFFFFFF;
const uint32 HMI_C_CANVIN   = 0x88888888;
App::datamodel::DataModel* App::datamodel::DataModel::_theInstance = 0;


void DataModel::Init()
{
   if (_theInstance == 0)
   {
      _theInstance = new DataModel();                                                     //lint !e1713
   }
   ETG_TRACE_USR4(("DataModel Init"));
}


DataModel::DataModel() : _regionHandling(NULL)
   , _contextSwitchHandler(NULL)
   , _appMasterForeground(false)
{
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
   _u8AntiTheftStatus = HMI_SM_C_HU_UNKNOWN;
#endif
   ETG_TRACE_USR4(("DataModel constructor"));
}


DataModel::~DataModel()
{
   _appMasterForeground  = false;
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
   _u8AntiTheftStatus    = HMI_SM_C_HU_UNKNOWN;
#endif
   ETG_TRACE_USR4(("DataModel destructor"));
}


DataModelHome& getDM()
{
   return DataModelHome::getInstance();
}


/*************************************************************************
*FUNCTION:      StartDisclaimerTimerReqMsg
*DESCRIPTION:   This is the courier message function handler for extracting
*               Timer start request from StartDisclaimerTimerReqMsg
*PARAMETER:     oMsg - Courier message received.
*RETURNVALUE:   bool
*                   true, if the message is consumed
*                   false, if the message should be forwarded
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    17.09.2019
*************************************************************************/
bool DataModel::onCourierMessage(const StartDisclaimerTimerReqMsg& oMsg)
{
   ETG_TRACE_USR3(("DataModel::StartDisclaimerTimerReqMsg"));
   // App::Core::SyncDisplayHandler::poGetInstance()->HideAnimationLayerOnDisclaimer(); //commented as per review comments to check requirements
   if (!_timer.running())
   {
      _timer.setTimeout(0, 5000, COURIER_MESSAGE_NEW(ExpiredDisclaimerTimerMsg)());
      _timer.start();
   }
   return true;
}


/*************************************************************************
*FUNCTION:      StopDisclaimerTimerReqMsg
*DESCRIPTION:   This is the courier message function handler for extracting
*               Timer stop request from StopDisclaimerTimerReqMsg
*PARAMETER:     oMsg - Courier message received.
*RETURNVALUE:   bool
*                   true, if the message is consumed
*                   false, if the message should be forwarded
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    17.09.2019
*************************************************************************/
bool DataModel::onCourierMessage(const StopDisclaimerTimerReqMsg& oMsg)
{
   if (_timer.running())
   {
      _timer.stop();
   }
   return true;
}


/*************************************************************************
*FUNCTION:      HandleSettingsMsg
*DESCRIPTION:   This is the courier message function handler for extracting
*               Hardkey Home info and Settings softkey info
*PARAMETER:     oMsg - Courier message received.
*RETURNVALUE:   bool
*                   true, if the message is consumed
*                   false, if the message should be forwarded
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    27.09.2019
*************************************************************************/
bool DataModel::onCourierMessage(const HandleSettingsMsg& oMsg)
{
   ETG_TRACE_USR3(("DataModel::HandleSettingsMsg"));
   bool isPerformed = false;
   bool contextSwitchUpdateRequired = true;
   ApplicationContextInfo contextInfo;
   ContextSwitchRequestInfo reqInfo;

   if ((NULL == _contextSwitchHandler) || (NULL == _regionHandling))
   {
      return true;
   }

   uint32 tRegionId = _regionHandling->getRegionId();
   if (_regionHandling->isCabinRegion(tRegionId))
   {
      uint8 tSystemStatus = _regionHandling->getStatus(SystemEnquiryInfo(tRegionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM));
      if ((tSystemStatus != SYSTEM_STATUS_CONNECTED) && (tSystemStatus != SYSTEM_STATUS_HOURLOGIC_DISCONNECTED) && (fnGetVariantType() !=  VARIANT_TYPE_CABIN_ONLY))
      {
         Courier::Message* msg = COURIER_MESSAGE_NEW(SystemFailureUpdate)(static_cast<enRegionID>(tRegionId), SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_AVDECC, SYSTEM_STATUS_DISCONNECTED, SYSTEM_GROUP_CATEGORY_NONE);
         if (msg != 0)
         {
            msg->Post();
         }
         isPerformed = true;
         contextSwitchUpdateRequired = false;
      }
   }

   if ((!isPerformed) && (_contextSwitchHandler->fetchActiveContextInfo(contextInfo)))
   {
      reqInfo.setRegionId(_regionHandling->getRegionId());
      reqInfo.setSourceContextId(0);
      switch (contextInfo.getApplicationId())
      {
         case APPID_APPHMI_MASTER:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_MASTER);
            reqInfo.setTargetApplicationId(APPID_APPHMI_SYSTEM);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_SETTINGS);
         }
         break;
         case APPID_APPHMI_MEDIA:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_MEDIA);
            reqInfo.setTargetApplicationId(APPID_APPHMI_MEDIA);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_MEDIA_SETTINGS_VIEW);
         }
         break;
         case APPID_APPHMI_PHONE:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_PHONE);
            reqInfo.setTargetApplicationId(Enum_APPID_APPHMI_PHONE);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_PHONE_SETTINGS_VIEW);
         }
         break;
         case APPID_APPHMI_TUNER:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_TUNER);
            reqInfo.setTargetApplicationId(Enum_APPID_APPHMI_TUNER);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_TUNER_SETTINGS_VIEW);
         }
         break;
         case APPID_APPHMI_SPI:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_SPI);
            reqInfo.setTargetApplicationId(APPID_APPHMI_SYSTEM);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_SETTINGS);
         }
         break;
         case APPID_APPHMI_NAVIGATION:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_NAVIGATION);
            reqInfo.setTargetApplicationId(Enum_APPID_APPHMI_NAVIGATION);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_NAVI_SETTINGS_VIEW);
         }
         break;
         case APPID_APPHMI_CAMERA:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_CAMERA);
            reqInfo.setTargetApplicationId(APPID_APPHMI_SYSTEM);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_SETTINGS);
         }
         break;
         case APPID_APPHMI_SXM:
         {
            reqInfo.setSourceApplicationId(APPID_APPHMI_SXM);
            reqInfo.setTargetApplicationId(Enum_APPID_APPHMI_SXM);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_SXM_SETTINGS_VIEW);
         }
         break;
         default:
         {
            contextSwitchUpdateRequired = false;
         }
         break;
      }
   }

   if (contextSwitchUpdateRequired)
   {
      _contextSwitchHandler->onContextSwitchRequest(reqInfo);
   }
   return true;
}


bool DataModel::onCourierMessage(const HKStatusUpdateMsg& oMsg)
{
   ETG_TRACE_USR4(("DataModel::HKStatusUpdateMsg: HkCode: %d", oMsg.GetHkCode()));
   bool retvalue = false;
   if (oMsg.GetHkCode() == HARDKEYCODE_REMOTECONTROL_HOME)
   {
      if (oMsg.GetHkState() == hmibase::HARDKEYSTATE_UP)
      {
         if (!isMasterApplicationForeground())
         {
            performHomeScreenContextSwitch(fnGetRegionId());
            bool retvalue = true;
         }
         else
         {
            performHomeScreenRegionSwitch();
            bool retvalue = true;
         }
      }
      else if (oMsg.GetHkState() == hmibase::HARDKEYSTATE_LONGUP)
      {
         performHomeScreenContextSwitch(REGION_COCKPIT);
      }
   }
   return retvalue;
}


void DataModel::updateKDSValueToModel(uint8 kdsStatus)
{
   ETG_TRACE_USR4(("DataModel::updateKDSValueToModel:%d", kdsStatus));
   if ((kdsStatus == 1) && ((*m_KDSStatusData).mKDSStatus != kdsStatus))
   {
      (*m_KDSStatusData).mKDSStatus = 0; // KDS Success
      m_KDSStatusData.MarkItemModified(::ItemKey::KDSStatusInfo::KDSStatusItem);
      m_KDSStatusData.SendUpdate(false);
   }
   else if ((kdsStatus == 2) && ((*m_KDSStatusData).mKDSStatus != kdsStatus))
   {
      (*m_KDSStatusData).mKDSStatus = 1; //KDS Failure
      m_KDSStatusData.MarkItemModified(::ItemKey::KDSStatusInfo::KDSStatusItem);
      m_KDSStatusData.SendUpdate(false);
   }
   else
   {}
   POST_MSG((COURIER_MESSAGE_NEW(KDSStatusUpdMsg)(kdsStatus)));
}


void DataModel::performHomeScreenContextSwitch(uint32 regionID) const
{
   ETG_TRACE_USR4(("DataModel::performHomeScreenContextSwitch regionID: %d", regionID));
   POST_MSG((COURIER_MESSAGE_NEW(ContextSwitchFromHomeMsg)(regionID, 0, 0, Enum_APPID_APPHMI_MASTER, APPLICATIONSWITCH_TYPES::enActivityIDs__eActivityID_HOME)));
}


void DataModel::performHomeScreenRegionSwitch() const
{
   uint32 regionID = fnGetRegionId();
   ETG_TRACE_USR4(("DataModel::performHomeScreenRegionSwitch regionID: %d", regionID));

   switch (regionID)
   {
      case REGION_COCKPIT:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ChangeHomeStateMsg)(REGION_CABIN_A)));
         break;
      }
      case REGION_CABIN_A:
      {
         if (fnIsRegionSupported(REGION_CABIN_B))
         {
            POST_MSG((COURIER_MESSAGE_NEW(ChangeHomeStateMsg)(REGION_CABIN_B)));
         }
         else
         {
            POST_MSG((COURIER_MESSAGE_NEW(ChangeHomeStateMsg)(REGION_COCKPIT)));
         }
         break;
      }
      case REGION_CABIN_B:
      {
         POST_MSG((COURIER_MESSAGE_NEW(ChangeHomeStateMsg)(REGION_COCKPIT)));
         break;
      }
      default:
      {
         break;
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
/*************************************************************************
*FUNCTION:      ReqHMIReadVIN
*DESCRIPTION:   This is the courier message function handler for extracting
*               VIN value from ReqHMIReadVIN
*PARAMETER:     oMsg - Courier message received.
*RETURNVALUE:   bool
*                   true, if the message is consumed
*                   false, if the message should be forwarded
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    27.09.2019
*************************************************************************/
bool DataModel::onCourierMessage(const ReqHMIReadVIN& oMsg)
{
   ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN"));
   // if (_bVINStatus)
   {
      uint32 _u32CANVIN = 0;
      long int _u32DIAGVIN = App::Core::diagnosisClientHandler::poGetInstance()->u32GetVINValue();
      ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN, _u32DIAGVIN = %x", _u32DIAGVIN));
      if (HMI_C_DEFAULTVIN == _u32DIAGVIN)
      {
         ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN,Default VIN present."));
         _u8AntiTheftStatus = HMI_SM_C_VID_MISSING;
      }
      else
      {
         if (App::Core::VehicleSrvClient::poGetInstance()->fetchGetVINValue(_u32CANVIN))
         {
            if (_u32DIAGVIN == _u32CANVIN)
            {
               _u8AntiTheftStatus = HMI_SM_C_HU_UNLOCKED;
               ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN,CANVIN matches with DIAGVIN"));
            }
            else
            {
               _u8AntiTheftStatus = HMI_SM_C_HU_LOCKED;
               ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN,none matches"));
            }
         }
         else
         {
            _u8AntiTheftStatus = HMI_SM_C_HU_UNKNOWN;
         }
      }
   }
   ETG_TRACE_USR3(("DataModel::ReqHMIReadVIN:%d", _u8AntiTheftStatus));
   POST_MSG((COURIER_MESSAGE_NEW(UpdateAntiTheftStatus)(_u8AntiTheftStatus)));
   return true;
}


/*************************************************************************
*FUNCTION:      UpdateVINStatus
*DESCRIPTION:   This is the courier message function handler for extracting
*               VINStatus from UpdateVINStatus
*PARAMETER:     oMsg - Courier message received.
*RETURNVALUE:   bool
*                   true, if the message is consumed
*                   false, if the message should be forwarded
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    27.09.2019
*************************************************************************/
bool DataModel::onCourierMessage(const UpdateVINStatus& oMsg)
{
   ETG_TRACE_USR3(("DataModel::UpdateVINStatus"));
   POST_MSG((COURIER_MESSAGE_NEW(UpdateVINStatusToHMI)()));
   return true;
}


/*************************************************************************
*FUNCTION:      u8GetAntiTheftStatus
*DESCRIPTION:   This getter function is foor getting AntiTheftStatusg
*PARAMETER:     NA
*RETURNVALUE:   uint8
*                   0 - UnlockHMI
                    1 - Configuration Required
                    2 - Device Locked
*AUTHOR:        Mohana Venkatesh (RBEI/ECG2 mve8cob)
*CREATED ON:    08.11.2019
*************************************************************************/
uint8 DataModel::u8GetAntiTheftStatus()
{
   return _u8AntiTheftStatus;
}


#endif


bool DataModel::onCourierMessage(const ApplicationStateUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("DataModel::onCourierMessage(ApplicationStateUpdMsg) state: %d ", oMsg.GetState()));

   if (oMsg.GetState() == hmibase::IN_FOREGROUND)
   {
      _appMasterForeground = true;
   }
   else
   {
      _appMasterForeground = false;
   }
   return false;
}


bool DataModel::isMasterApplicationForeground() const
{
   return _appMasterForeground;
}


} //datamodel
} //App
