/**
 * @file         : VariantHandling.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : VariantHandling
 * @brief        : VariantHandling is to handle the variant information
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */
#include "hall_std_if.h"
#include "VariantHandling.h"
#include "hmi_trace_if.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "dp_generic_if_Mock.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/VariantHandling.cpp.trc.h"
#endif


/**
 *  Static Variables Initialization
 */

namespace App {
namespace Core {

static VariantHandling* _variantHandlingInstance = NULL;


#ifndef WIN32 /* Linux / Other OS env except Windows */
pthread_mutex_t VariantHandling::_safeLock = PTHREAD_MUTEX_INITIALIZER;
#endif
} //namespace Core
} //namespace App


/**
 *  SM "C" Functions Definition
 */
uint16 fnGetVariantId()
{
   uint16 variantId = 0;
   if (NULL != ::App::Core::_variantHandlingInstance)
   {
      variantId = App::Core::_variantHandlingInstance->getVariantId();
   }
   return variantId;
}


uint16 fnGetVariantType()
{
   uint16 variantType = 0;
   if (NULL != ::App::Core::_variantHandlingInstance)
   {
      variantType = App::Core::_variantHandlingInstance->getVariantType();
   }
   return variantType;
}


bool fnIsInf4cvSmartVariant()
{
   bool status = (fnGetVariantId() == VEHICLE_TYPE_SMART) ? true : false;
   return status;
}


/**
 *  Member Functions Definition
 */

namespace App {
namespace Core {

VariantHandling::VariantHandling() : _variantId(0xFFFF)
   , _variantType(0xFF)
   , _sxmAvailable(0xFF)
   , _btAvailable(0xFF)
   , _cpAvailable(0xFF)
   , _aaAvailable(0xFF)
   , _mySpinAvailable(0xFF)
   , _isHDMIAvailable(0xFF)
   , _isCamportUSBAvailable(0xFF)
   , _regionType(REGION_TYPE_UNKNOWN)
   , _glassAVSourcePresTime(0xFFFF)
   , _glassAnalogueCameraPresTime(0xFFFF)
   , _glassIPCameraPresTime(0XFFFF)
   , _amplifierUSBVideoSourcePresTime(0xFFFF)
   , _amplifierHDMIPresTime(0xFFFF)
   , _amplifierAudioPresTime(0xFFFF)
   , _ampDRMPresTime(0xFFFF)
   , _glassDRMPresTime(0xFFFF)
{
   ETG_TRACE_USR1(("VariantHandling: CTOR"));
   _variantHandlingInstance = this;
   readKDSVariantType();
   readKDSVariantId();
   readKDSSXMAvailability();
   readKDSBTAvailability();
   readKDSCPAvailability();
   readKDSAndroidAutoAvailability();
   readKDSMySpinAvailability();
   readKDSHDMIAvailability();
   readKDSCMPUSBAvailability();
   readKDSRegionType();
   readKDSPresentationTimeAVSource();
   readKDSPresentationTimeAnalogueCamera();
   readKDSPresentationTimeIPCamera();
   readKDSPresentationTimeAmplifier_USBVideo();
   readKDSPresentationTimeAmplifier_HDMI();
   readKDSPresentationTimeAmplifier_Audio();
   readKDSPresentationTimeGlass_DRM();
   readKDSPresentationTimeAmp_DRM();
}


VariantHandling::~VariantHandling()
{
   ETG_TRACE_USR1(("VariantHandling: DTOR"));
   _variantHandlingInstance = NULL;
}


bool VariantHandling::isSPIAvailable()
{
   bool isSPIvalid = false;
   if (isCPAvailable() || isAndroidAutoAvailable() || isMySpinAvailable())
   {
      isSPIvalid = true;
   }
   return isSPIvalid;
}


uint8 VariantHandling::getVariantType()
{
   uint8 variantType = 0;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_lock(&_safeLock);
#endif
   variantType = _variantType;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_unlock(&_safeLock);
#endif
   return variantType;
}


uint16 VariantHandling::getVariantId()
{
   uint16 variantId = 0;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_lock(&_safeLock);
#endif
   variantId = _variantId;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_unlock(&_safeLock);
#endif
   return variantId;
}


bool VariantHandling::isSXMAvailable()
{
   bool sxmAvailable = ((_sxmAvailable == 0xFF) || (_sxmAvailable == 0)) ? false : true;
   return sxmAvailable;
}


bool VariantHandling::isBTAvailable()
{
   bool btAvailable = ((_btAvailable == 0xFF) || (_btAvailable == 0)) ? false : true;
   return btAvailable;
}


bool VariantHandling::isCPAvailable()
{
   bool cpAvailable = ((_cpAvailable == 0xFF) || (_cpAvailable == 0)) ? false : true;
   return cpAvailable;
}


bool VariantHandling::isAndroidAutoAvailable()
{
   bool aaAvailable = ((_aaAvailable == 0xFF) || (_aaAvailable == 0)) ? false : true;
   return aaAvailable;
}


bool VariantHandling::isMySpinAvailable()
{
   bool mySpinAvailable = ((_mySpinAvailable == 0xFF) || (_mySpinAvailable == 0)) ? false : true;
   return mySpinAvailable;
}


bool VariantHandling::isHDMIAvailable()
{
   bool hdmiAvailable = ((_isHDMIAvailable == 0XFF) || (_isHDMIAvailable == 0)) ? false : true;
   return hdmiAvailable;
}


bool VariantHandling::isCamportUSBAvailable()
{
   bool camportUSBAvailable = ((_isCamportUSBAvailable == 0XFF) || (_isCamportUSBAvailable == 0)) ? false : true;
   return camportUSBAvailable;
}


bool VariantHandling::isEURRegion()
{
   bool eurRegion = (_regionType == REGION_TYPE_OTH_EUR) ? true : false;
   return eurRegion;
}


uint16 VariantHandling::getPresentationTimeAVSource()
{
   uint16 _avSourcePresTime = ((_glassAVSourcePresTime == 0xFFFF) ? 0 : _glassAVSourcePresTime);
   return _avSourcePresTime;
}


uint16 VariantHandling::getPresentationTimeAnalogueCamera()
{
   uint16 _analogueCameraPresTime = ((_glassAnalogueCameraPresTime == 0xFFFF) ? 0 : _glassAnalogueCameraPresTime);
   return _analogueCameraPresTime;
}


uint16 VariantHandling::getPresentationTimeIPCamera()
{
   uint16 _ipCameraPresTime = ((_glassIPCameraPresTime == 0xFFFF) ? 0 : _glassIPCameraPresTime);
   return _ipCameraPresTime;
}


uint16 VariantHandling::getPresentationTimeAmplifier_USBVideo()
{
   uint16 _usbVideoSourcePresTime = ((_amplifierUSBVideoSourcePresTime == 0xFFFF) ? 0 : _amplifierUSBVideoSourcePresTime);
   return _usbVideoSourcePresTime;
}


uint16 VariantHandling::getPresentationTimeAmplifier_HDMI()
{
   uint16 _hdmiSourcePresTime = ((_amplifierHDMIPresTime == 0xFFFF) ? 0 : _amplifierHDMIPresTime);
   return _hdmiSourcePresTime;
}


uint16 VariantHandling::getPresentationTimeAmplifier_Audio()
{
   uint16 _audioSourcePresTime = ((_amplifierAudioPresTime == 0xFFFF) ? 0 : _amplifierAudioPresTime);
   return _audioSourcePresTime;
}


uint16 VariantHandling::getPresentationTimeGlass_DRM()
{
   uint16 _DRMSourcePresTime = ((_glassDRMPresTime == 0xFFFF) ? 0 : _glassDRMPresTime);
   return _DRMSourcePresTime;
}


uint16 VariantHandling::getPresentationTimeAmp_DRM()
{
   uint16 _DRMSourcePresTime = ((_ampDRMPresTime == 0xFFFF) ? 0 : _ampDRMPresTime);
   return _DRMSourcePresTime;
}


void VariantHandling::readKDSVariantType()
{
   uint8 variantType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "VariantSelection", &variantType, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::getKDSVariantType::%d", variantType));
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_lock(&_safeLock);
#endif
      _variantType = variantType;
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_unlock(&_safeLock);
#endif
   }
}


void VariantHandling::readKDSVariantId()
{
   uint8 u8VehicleType[2] = {0};
   uint16 u16VehicleType = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "VehicleType", &u8VehicleType[0], 2))
   {
      u16VehicleType = static_cast<uint16>((u8VehicleType[0] << 8) | u8VehicleType[1]);
      ETG_TRACE_USR1(("VariantHandling::getKDSVehicleType::%d", u16VehicleType));
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_lock(&_safeLock);
#endif
      _variantId = u16VehicleType;
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_unlock(&_safeLock);
#endif
   }
}


void VariantHandling::readKDSSXMAvailability()
{
   uint8 sxmAvailable = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "XMTuner", &sxmAvailable, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSSXMAvailability::%d", sxmAvailable));
      _sxmAvailable = sxmAvailable;
   }
}


void VariantHandling::readKDSBTAvailability()
{
   uint8 btAvaialble = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "Bluetooth", &btAvaialble, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSBTAvailability::%d", btAvaialble));
      _btAvailable = btAvaialble;
   }
}


void VariantHandling::readKDSCPAvailability()
{
   uint8 cpAvaialble = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "CarPlay", &cpAvaialble, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSCPAvailability::%d", cpAvaialble));
      _cpAvailable = cpAvaialble;
   }
}


void VariantHandling::readKDSAndroidAutoAvailability()
{
   uint8 aaAvailable = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "AndroidAuto", &aaAvailable, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSAndroidAutoAvailability::%d", aaAvailable));
      _aaAvailable = aaAvailable;
   }
}


void VariantHandling::readKDSMySpinAvailability()
{
   uint8 mySpinAvailable = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("SystemConfiguration1", "MySPIN", &mySpinAvailable, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSMySpinAvailability::%d", mySpinAvailable));
      _mySpinAvailable = mySpinAvailable;
   }
}


void VariantHandling::readKDSHDMIAvailability()
{
   uint8 isHDMIAvailable = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AIVIVariantCoding", "HDMIApp", &isHDMIAvailable, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSHDMIAvailability::%d", isHDMIAvailable));
      _isHDMIAvailable = isHDMIAvailable;
   }
}


void VariantHandling::readKDSCMPUSBAvailability()
{
   uint8 isCamportUSBAvailable = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("CIS_Configuration", "CameraportUSB", &isCamportUSBAvailable, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSCMPUSBAvailability::%d", isCamportUSBAvailable));
      _isCamportUSBAvailable = isCamportUSBAvailable;
   }
}


void VariantHandling::readKDSRegionType()
{
   uint8 u8RegionType = REGION_TYPE_UNKNOWN;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("VehicleInformation", "DestinationRegion1", &u8RegionType, 1))
   {
      ETG_TRACE_USR1(("VariantHandling::readKDSRegionType::%d", u8RegionType));
      _regionType = (enRegionType)u8RegionType;
   }
}


void VariantHandling::readKDSPresentationTimeAVSource()
{
   uint8 u8AVSource[2] = {0};
   uint16 u16AVSource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeGlassUSBVideo", &u8AVSource[0], 2))
   {
      u16AVSource = static_cast<uint16>((u8AVSource[0] << 8) | u8AVSource[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAVSource::%d", u16AVSource));
      _glassAVSourcePresTime = u16AVSource;
   }
}


void VariantHandling::readKDSPresentationTimeAnalogueCamera()
{
   uint8 u8AnalogueCamera[2] = {0};
   uint16 u16AnalogueCamera = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeGlassAnalogueCamera", &u8AnalogueCamera[0], 2))
   {
      u16AnalogueCamera = static_cast<uint16>((u8AnalogueCamera[0] << 8) | u8AnalogueCamera[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAnalogueCamera::%d", u16AnalogueCamera));
      _glassAnalogueCameraPresTime = u16AnalogueCamera;
   }
}


void VariantHandling::readKDSPresentationTimeIPCamera()
{
   uint8 u8IPCamera[2] = {0};
   uint16 u16IPCamera = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeGlassIPCamera", &u8IPCamera[0], 2))
   {
      u16IPCamera = static_cast<uint16>((u8IPCamera[0] << 8) | u8IPCamera[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeIPCamera::%d", u16IPCamera));
      _glassIPCameraPresTime = u16IPCamera;
   }
}


void VariantHandling::readKDSPresentationTimeAmplifier_USBVideo()
{
   uint8 u8USBVideo[2] = {0};
   uint16 u16USBVideo = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeAmplifierUSBVideo", &u8USBVideo[0], 2))
   {
      u16USBVideo = static_cast<uint16>((u8USBVideo[0] << 8) | u8USBVideo[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAmplifier_USBVideo::%d", u16USBVideo));
      _amplifierUSBVideoSourcePresTime = u16USBVideo;
   }
}


void VariantHandling::readKDSPresentationTimeAmplifier_HDMI()
{
   uint8 u8HDMISource[2] = {0};
   uint16 u16HDMISource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeAmplifierHDMI", &u8HDMISource[0], 2))
   {
      u16HDMISource = static_cast<uint16>((u8HDMISource[0] << 8) | u8HDMISource[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAmplifier_HDMI::%d", u16HDMISource));
      _amplifierHDMIPresTime = u16HDMISource;
   }
}


void VariantHandling::readKDSPresentationTimeAmplifier_Audio()
{
   uint8 u8AudioSource[2] = {0};
   uint16 u16AudioSource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeAmplifierAudio", &u8AudioSource[0], 2))
   {
      u16AudioSource = static_cast<uint16>((u8AudioSource[0] << 8) | u8AudioSource[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAmplifier_Audio::%d", u16AudioSource));
      _amplifierAudioPresTime = u16AudioSource;
   }
}


void VariantHandling::readKDSPresentationTimeGlass_DRM()
{
   uint8 u8DRMSource[2] = {0};
   uint16 u16DRMSource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeGlassDRM", &u8DRMSource[0], 2))
   {
      u16DRMSource = static_cast<uint16>((u8DRMSource[0] << 8) | u8DRMSource[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeGlass_DRM::%d", u16DRMSource));
      _glassDRMPresTime = u16DRMSource;
   }
}


void VariantHandling::readKDSPresentationTimeAmp_DRM()
{
   uint8 u8DRMSource[2] = {0};
   uint16 u16DRMSource = 0;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("AVDECC_Config", "PresentationTimeAmplifierDRM", &u8DRMSource[0], 2))
   {
      u16DRMSource = static_cast<uint16>((u8DRMSource[0] << 8) | u8DRMSource[1]);
      ETG_TRACE_USR1(("VariantHandling::readKDSPresentationTimeAmp_DRM::%d", u16DRMSource));
      _ampDRMPresTime = u16DRMSource;
   }
}


} //namespace Core
} //namespace App
