/**
 * @file         : SystemMonitorHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : SystemMonitorHandler
 * @brief        : SystemMonitorHandler is to monitor the the CAN signal information
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "SystemMonitorHandler.h"
#include "hmi_trace_if.h"
#include "ProjectBaseTypes.h"
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <AppHmi_MasterBase/AudioInterface/Connection.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>
#include <AppHmi_MasterBase/AudioInterface/ResourceController.h>
#include <App/Core/AudioInterface/SoundControllerPrjInterface.h>
#include <App/Core/MediaClientHandler/MediaClientHandlerTypes.h>
#include <App/Core/VehicleSrvClient/IVehicleSrvClientInterface.h>
#include <App/Core/AudioInterface/ResourceControllerPrjInterface.h>
#include <App/Core/MediaClientHandler/MediaClientHandlerInterface.h>
#include <AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h>


using namespace ::hmibase::apphmi_master::audio;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/SystemMonitorHandler.cpp.trc.h"
#endif


/**
 *  Defines Declaration
 */

#define SRC_OFF 0
#define MEDIASTATUS_CMD_MUTE 1
#define MEDIASTATUS_CMD_ERROR 2
#define MEDIASTATUS_CMD_UNMUTE 0


/**
 *  Member Functions Definition
 */

namespace App {
namespace Core {


SystemMonitorHandler::SystemMonitorHandler(): _currentCMDSourceIndex(SRC_OFF)
   , _regionHandling(NULL)
   , _vehicleServiceClient(NULL)
   , _mediaClientHandler(NULL)
{
   ETG_TRACE_USR1(("SystemMonitorHandler: CTOR"));
   initializeAudioSourceToCMDIndexInfo();
}


SystemMonitorHandler::~SystemMonitorHandler()
{
   ETG_TRACE_USR1(("SystemMonitorHandler: DTOR"));
   _audioSourceCMDIndexInfo.clear();
   _currentCMDSourceIndexInfo.clear();
   _currentCMDSourceIndex = SRC_OFF;
   _regionHandling        = NULL;
   _vehicleServiceClient  = NULL;
   _mediaClientHandler    = NULL;
}


void SystemMonitorHandler::initializeAudioSourceToCMDIndexInfo()
{
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_DAB,              ""),     16));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_TUNER_FM,         ""),      1));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_TUNER_AM,         ""),      1));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_TUNER_XM,         ""),     14));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_PHONE_BTAUDIO,    ""),      5));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_MEDIA_PLAYER,  "USB1"),     3));
   _audioSourceCMDIndexInfo.insert(std::make_pair(std::make_pair(SRC_MEDIA_PLAYER,  "USB2"),     4));
}


void SystemMonitorHandler::onMuteStateUpdate(const uint32 sinkId, const int state)
{
   ETG_TRACE_USR4(("SystemMonitorHandler: onMuteStateUpdate: <-: SinkId = %d : Mutestate = %d", sinkId, state));
   if ((NULL != _regionHandling) && (NULL != _vehicleServiceClient))
   {
      uint32 tCMDMuteState = getCMDMuteState(state);
      switch (_regionHandling->getRegionIdForAudioSink(sinkId))
      {
         case REGION_COCKPIT:
         {
            ETG_TRACE_USR4(("SystemMonitorHandler: onMuteStateUpdate: ->: RegionId = %d : Mutestate = %d", REGION_COCKPIT, tCMDMuteState));
            _vehicleServiceClient->sendCockpitMuteUpdateToCluster(tCMDMuteState);
            break;
         }
         case REGION_CABIN_A:
         {
            ETG_TRACE_USR4(("SystemMonitorHandler: onMuteStateUpdate: ->: RegionId = %d : Mutestate = %d", REGION_CABIN_A, tCMDMuteState));
            _vehicleServiceClient->sendCabinAMuteUpdateToCluster(tCMDMuteState);
            break;
         }
         default :
            break;
      }
   }
}


void SystemMonitorHandler::onVolumeChangeUpdate(const uint32 sinkId, const int volume)
{
   ETG_TRACE_USR4(("SystemMonitorHandler: onVolumeChangeUpdate: <-: SinkId = %d : Volume = %d", sinkId, volume));
   if ((NULL != _regionHandling) && (NULL != _vehicleServiceClient))
   {
      uint32 regionId = _regionHandling->getRegionIdForAudioSink(sinkId);
      switch (regionId)
      {
         case REGION_COCKPIT:
         {
            ETG_TRACE_USR4(("SystemMonitorHandler: onVolumeChangeUpdate: ->: RegionId = %d : Volume = %d", REGION_COCKPIT, volume));
            _vehicleServiceClient->sendCockpitVolumeUpdateToCluster(static_cast<uint8>(volume));
            break;
         }
         case REGION_CABIN_A:
         {
            ETG_TRACE_USR4(("SystemMonitorHandler: onVolumeChangeUpdate: ->: RegionId = %d : Volume = %d", REGION_CABIN_A, volume));
            _vehicleServiceClient->sendCabinAVolumeUpdateToCluster(static_cast<uint8>(volume));
            break;
         }
         default :
            break;
      }
   }
}


void SystemMonitorHandler::onUpdateConnectionInfo(std::map< int, Connection* >  connectionStatusInfo)
{
   if (NULL != _vehicleServiceClient)
   {
      uint32 tCMDSourceIndex = SRC_OFF;
      if ((fetchActiveSourceCMDIndexForRegion(connectionStatusInfo, REGION_COCKPIT, tCMDSourceIndex)) && (tCMDSourceIndex != getCurrentCMDSourceIndexForRegion(REGION_COCKPIT)))
      {
         _currentCMDSourceIndexInfo[REGION_COCKPIT] = tCMDSourceIndex;
         ETG_TRACE_USR4(("SystemMonitorHandler: onUpdateConnectionInfo: ->: RegionId = %d : SourceIndex = %d", REGION_COCKPIT, tCMDSourceIndex));
         _vehicleServiceClient->sendCockpitActiveAudioSourceToCluster(tCMDSourceIndex);
      }
      if ((fetchActiveSourceCMDIndexForRegion(connectionStatusInfo, REGION_CABIN_A, tCMDSourceIndex)) && (tCMDSourceIndex != getCurrentCMDSourceIndexForRegion(REGION_CABIN_A)))
      {
         _currentCMDSourceIndexInfo[REGION_CABIN_A] = tCMDSourceIndex;
         ETG_TRACE_USR4(("SystemMonitorHandler: onUpdateConnectionInfo: ->: RegionId = %d : SourceIndex = %d", REGION_CABIN_A, tCMDSourceIndex));
         _vehicleServiceClient->sendCabinAActiveAudioSourceToCluster(tCMDSourceIndex);
      }
   }
}


uint8 SystemMonitorHandler::getCMDMuteState(const int state) const
{
   uint8 muteState = MEDIASTATUS_CMD_ERROR;
   switch (state)
   {
      case STATE_MUTED:
      {
         muteState = MEDIASTATUS_CMD_MUTE;
         break;
      }
      case STATE_UNMUTED:
      {
         muteState = MEDIASTATUS_CMD_UNMUTE;
         break;
      }
      default:
         break;
   }
   return muteState;
}


uint8 SystemMonitorHandler::getCMDAudioSourceIndex(const uint32 sourceId, const uint32 deviceId) const
{
   uint8 tSrcIndex = SRC_OFF;
   ::std::string deviceType = getDeviceTypeforDeviceId(sourceId, deviceId);
   ::std::map< ::std::pair< uint32, ::std::string >, uint32 >::const_iterator itr = _audioSourceCMDIndexInfo.find(::std::make_pair(sourceId, deviceType));
   if (itr != _audioSourceCMDIndexInfo.end())
   {
      tSrcIndex = itr->second;
   }
   return tSrcIndex;
}


::std::string SystemMonitorHandler::getDeviceTypeforDeviceId(const uint32 sourceId, const uint32 deviceId) const
{
   ::std::string deviceType = "";
   ::std::vector< DeviceConnectionInfo > info;
   if ((sourceId == SRC_MEDIA_PLAYER) && (NULL != _mediaClientHandler) && (_mediaClientHandler->fetchDeviceConnectionsInfo(info)))
   {
      for (::std::vector< DeviceConnectionInfo >::const_iterator itr = info.begin(); itr != info.end(); ++itr)
      {
         if (itr->getDeviceId() == deviceId)
         {
            deviceType = itr->getDeviceType();
            break;
         }
      }
   }
   return deviceType;
}


bool SystemMonitorHandler::isValidSource(const uint32 sourceId, const uint32 deviceId)
{
   bool isValid = false;
   if (getCMDAudioSourceIndex(sourceId, deviceId) > SRC_OFF)
   {
      if ((NULL != _resourceController) && ((_resourceController->isMainSource(sourceId)) || (_resourceController->isAnnouncementSource(sourceId))))
      {
         isValid = true;
      }
   }
   return isValid;
}


bool SystemMonitorHandler::fetchActiveSourceCMDIndexForRegion(const std::map< int, Connection* >& info, const uint32 regionId, uint32& tCMDSourceIndex)
{
   bool isFetched          = true;
   ResourceController* imp = (NULL != _resourceController.get()) ? dynamic_cast<ResourceController*>(_resourceController.get()) : NULL;
   tCMDSourceIndex                   = SRC_OFF;
   if ((NULL != imp) && (NULL != _regionHandling))
   {
      int16 tSinkId = static_cast<int16>(_regionHandling->getAudioSinkIdForRegion(regionId));
      for (std::map<int, Connection*>::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         Connection* connection = itr->second;
         if (NULL != connection)
         {
            Sink* poSink     = connection->getSink();
            Source* poSource = connection->getSource();
            if ((NULL != poSink) && (NULL != poSource) && (tSinkId == static_cast<int16>(imp->GetIndexOnSinkID(poSink->GetID()))))
            {
               if (isValidSource(poSource->GetIndex(), poSource->GetDeviceId()))
               {
                  if (connection->getState() == STATE_CONNECTED)
                  {
                     tCMDSourceIndex = getCMDAudioSourceIndex(poSource->GetIndex(), poSource->GetDeviceId());
                     break;
                  }
                  else if ((connection->getState() == STATE_CONNECTING) || (connection->getState() == STATE_DISCONNECTING))
                  {
                     isFetched       = false;
                     tCMDSourceIndex = getCurrentCMDSourceIndexForRegion(regionId);
                     break;
                  }
                  else
                  {}
               }
            }
         }
      }
   }
   return isFetched;
}


} //namespace Core
} //namespace App


/**
 *  undef
 */

#undef SRC_OFF
#undef MEDIASTATUS_CMD_MUTE
#undef MEDIASTATUS_CMD_ERROR
#undef MEDIASTATUS_CMD_UNMUTE
