/**
 * @file         : RegionHandling.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : RegionHandling is to handle the region information
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef REGIONHANDLING_HEADER
#define REGIONHANDLING_HEADER


#ifndef WIN32 /* Linux / Other OS env except Windows */
#include "pthread.h"
#endif
#include "RegionHandlingTypes.h"
#include "RegionHandlingInterface.h"
#include "AppHmi_MasterStateMachine.h"
#include <string>
#include <asf/core/Timer.h>
#include <Core/HmiState/HmiStateHandlerInterface.h>
#include "AppBase/ServiceAvailableIF.h"
#include "bosch/cm/ai/hmi/hmiinfoservice/HmiInfoService.h"
#include "bosch/cm/ai/hmi/hmiinfoservice/HmiInfoServiceProxy.h"
#include <CgiExtensions/DataBindingItem.hpp>
#include <App/Core/NanoMsgClientHandler/NanoMsgClientHandler.h>


/**
 *  Defines Declaration
 */

#define REGION_HANDLING_LISTENER_DATA_DECL_BEGIN()

#define ADD_LISTENER_DATA_DECL(_dataName_, _dataType_)                      \
   private:                                                                 \
      ::std::vector< I##_dataName_##ListenerCB* > _##_dataName_##Listeners; \
      void clear##_dataName_##ListenersInfo();                              \
      void send##_dataName_##Update(const _dataType_& info);                \
   public:                                                                  \
      void add##_dataName_##Listener(I##_dataName_##ListenerCB& imp);       \
      void remove##_dataName_##Listener(I##_dataName_##ListenerCB& imp);    \

#define REGION_HANDLING_LISTENER_DATA_DECL_END()


/**
 *  Classes Forward Declaration
 */

class TiXmlElement;
class StartupSync;

/**
 *  Classes Declaration
 */

namespace App {
namespace Core {

using namespace ::App::Core;
using namespace ::asf::core;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;


class RegionHandling : public IRegionHandling
   , public IPNMGrp10StatusCB
   , public IHmiStateListenerCB
   , public ISubSytemsErrorStatusCB
   , public ::asf::core::TimerCallbackIF
   , public StartupSync::PropertyRegistrationIF
   , public hmibase::ServiceAvailableIF
   , public NavigationModeChangeTriggerCallbackIF
{
   public:
      /**
       *  Member Functions Declaration
       */
      RegionHandling();
      virtual ~RegionHandling();
      void onAvailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/);
      void onUnavailable(const ::boost::shared_ptr< ::asf::core::Proxy >& proxy, const ::asf::core::ServiceStateChange& /*stateChange*/);
      virtual void registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);
      virtual void deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/);

      SupportedRegionsInfo getSupportedRegionsInfo() const;
      int16 getAudioSinkIdForRegion(const uint32 regionId) const;
      uint8 getSystemGroupForSinkId(const int16 sinkId) const;
      uint32 getRegionId() const;
      uint32 getLastActiveCabinRegionId() const;
      uint32 getRegionIdForAudioSink(const uint32 sinkId) const;
      uint8 getSystemGroupCategoryForSinkId(const int16 sinkId) const;
      bool isRegionSupported(const uint32 regionId) const;
      bool isSinkGroupCategorySame(const int16 sinkId, const uint8 category) const;
      void setRegionId(const uint32 regionId);
      void onHmiStateUpdate(const uint8 hmiState);
      void onPNMGrp10StatusUpdate(const bool status);
      void onSubSystemsErrorStatus(const SubSystemsErrorStatusInfo& info);
      void onPluginEntityDetailsUpdate(const ::boost::shared_ptr< EventListDataUtility >& info);
      void onExpired(::asf::core::Timer& timer, ::boost::shared_ptr< ::asf::core::TimerPayload > data);
      void dumpSystemsInfo();
      bool isCMGConfiguredForGroupCategory(const uint32 regionId, const uint8 groupCategory);
      bool getCarModeStatusUpdate() const;
      bool getCoachModeStatusUpdate() const;

      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status of first system in
       *       particular category of the region
       */
      inline ::std::string getSystemIPAddress(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         return (_systemsInfo.getSystemIPAddressForSystemId(info, systemId));
      }

      inline int16 getAudioSinkId() const
      {
         return (getAudioSinkIdForRegion(getRegionId()));
      }

      inline uint8 getSystemCategoryForSinkId(const int16 sinkId) const
      {
         return (_systemsInfo.getSystemCategoryForSinkId(sinkId));
      }

      inline uint8 getSystemCategoryForSourceId(const int16 sourceId) const
      {
         return (_systemsInfo.getSystemCategoryForSourceId(sourceId));
      }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      inline void setNanoMsgClientHandler(INanoMsgClientHandler& imp)
      {
         _nanoMsgClientHandler = (&imp);
      }
#endif
      /*
       * INFO:
       *    -> The value "0xFF" of variable "systemId" denotes the status of first
       *       system in particular category of the region
       */
      inline uint8 getStatus(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         return (_systemsInfo.getStatusForSystemId(info, systemId));
      }

      /*
       * INFO:
       *    -> The value "0xFF" of variable "systemId" denotes the status of first
       *       system in particular category of the region
       */
      inline uint8 getLastStatus(const SystemEnquiryInfo& info, const uint8 systemId = 0xFF) const
      {
         return (_systemsInfo.getLastStatusForSystemId(info, systemId));
      }

      inline bool isCabinRegion(const uint32 regionId) const
      {
         bool isValid = true;
         switch (regionId)
         {
            case REGION_COCKPIT:
            case REGION_INVALID:
            {
               isValid = false;
            }
            default:
               break;
         }
         return isValid;
      }

      bool fetchRegionIdForSinkId(const int16 sinkId, uint32& regionId) const
      {
         return (_systemsInfo.fetchRegionIdForSinkId(sinkId, regionId));
      }

      /*
       * INFO:
       *    -> REGION_INVALID denotes analyse all the regions of the system
       *    -> Sink id "-1" denotes all the system
       */
      bool isAllStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId = -1) const
      {
         return (_systemsInfo.isAllStatusSame(info, status, sinkId));
      }

      /*
       * INFO:
       *    -> REGION_INVALID denotes analyse all the regions of the system
       *    -> Sink id "-1" denotes all the system
       */
      bool isAnyStatusSame(const SystemEnquiryInfo& info, const uint8 status, const int16 sinkId = -1) const
      {
         return (_systemsInfo.isAnyStatusSame(info, status, sinkId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSinkIdsInfo(SinkIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSinkIdsInfo(info, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool isSinkIdSupported(const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.isSinkIdSupported(sinkId, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSourceIdsInfo(SourceIdsInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSourceIdsInfo(info, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSystemsIdInfo(SystemsIdInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSystemsIdInfo(info, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool isSourceIdSupported(const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.isSourceIdSupported(sourceId, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool isSystemGroupSupported(const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const
      {
         SystemsIdInfo info;
         bool status = (_systemsInfo.fetchSystemsIdInfoForGroup(info, systemGroup, systemCategory, regionId)) ? true : false;
         return status;
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */

      bool fetchSystemsGroupInfo(SystemsGroupInfo& info, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSystemsGroupInfo(info, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSinkIdsInfoForGroup(SinkIdsInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSinkIdsInfoForGroup(info, systemGroup, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSystemsIdInfoForGroup(SystemsIdInfo& info, const uint8 systemGroup, const uint8 systemCategory, const uint32 regionId) const
      {
         return _systemsInfo.fetchSystemsIdInfoForGroup(info, systemGroup, systemCategory, regionId);
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSystemsIdInfoForSinkId(SystemsIdInfo& info, const int16 sinkId, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSystemsIdInfoForSinkId(info, sinkId, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSystemsIdInfoForSourceId(SystemsIdInfo& info, const int16 sourceId, const uint8 systemCategory, const uint32 regionId) const
      {
         return (_systemsInfo.fetchSystemsIdInfoForSourceId(info, sourceId, systemCategory, regionId));
      }

      /*
       * INFO:
       *    -> SYSTEM_GROUP_CATEGORY_NONE denotes analyse all groupCategory
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSinkIdsInfoForGroupCategory(SinkIdsInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const;

      /*
       * INFO:
       *    -> SYSTEM_GROUP_CATEGORY_NONE denotes analyse all groupCategory
       *    -> SYSTEM_CATEGORY_NONE denotes analyse all the systems
       *    -> REGION_INVALID denotes analyse all the regions of the system
       */
      bool fetchSystemsIdInfoForGroupCategory(SystemsIdInfo& info, const uint8 groupCategory, const uint8 systemCategory, const uint32 regionId) const;
      bool fetchSinkIdsInfoForSystemId(SinkIdsInfo& info, const uint8 systemId) const
      {
         return (_systemsInfo.fetchSinkIdsInfoForSystemId(info, systemId));
      }

      inline void finalize()
      {
         if (NULL != _hmiStateHandler)
         {
            _hmiStateHandler->removeHmiStateListener(this);
            _hmiStateHandler->removePNMGrp10StatusListener(this);
            _hmiStateHandler->removeSubSytemsErrorStatusListener(this);
         }
      }

      inline void initialize()
      {
         if (NULL != _hmiStateHandler)
         {
            _hmiStateHandler->addHmiStateListener(this);
            _hmiStateHandler->addPNMGrp10StatusListener(this);
            _hmiStateHandler->addSubSytemsErrorStatusListener(this);
         }
         sendCabinsSystemsInfoUpdate(_systemsInfo.getCabinsSystemsInfo());
         sendRegionInfoUpdate();
      }

      inline void setHMIStateHandler(IHmiStateHandler& imp)
      {
         _hmiStateHandler = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

      inline void setApplicationSwitchServerComponent(IApplicationSwitchServerComponent& imp)
      {
         _applicationSwitchServerComponent = (&imp);
      }

      virtual void onNavigationModeChangeTriggerError(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< NavigationModeChangeTriggerError >& error);
      virtual void onNavigationModeChangeTriggerSignal(const ::boost::shared_ptr< HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr< NavigationModeChangeTriggerSignal >& signal);

      REGION_HANDLING_LISTENER_DATA_DECL_BEGIN()
      ADD_LISTENER_DATA_DECL(Region, uint32)
      ADD_LISTENER_DATA_DECL(CMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(CMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(CMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(CMRStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(AVDECCCMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(AVDECCCMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(AVDECCCMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(AVDECCCMRStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(NANOMsgCMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(NANOMsgCMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(NANOMsgCMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(NANOMsgCMRStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(VoltageCMCStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(VoltageCMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(VoltageCMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(VoltageCMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(VoltageCMRStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(ThermalCMCStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(ThermalCMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(ThermalCMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(ThermalCMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(ThermalCMRStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(PowerModeCMAStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(PowerModeCMPStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(PowerModeCMGStatus, SystemStatusInfo)
      ADD_LISTENER_DATA_DECL(PowerModeCMRStatus, SystemStatusInfo)
      REGION_HANDLING_LISTENER_DATA_DECL_END()

   private:
      /**
       *  Member Typedefs Declaration
       */
      typedef ::std::map< ::std::pair< uint8, uint8 >, void(RegionHandling::*)(const SystemStatusInfo&) > ListenerUpdateFunctionsInfo;
      /**
       *  Member Functions Declaration
       */
      RegionHandling(const RegionHandling& obj);
      RegionHandling& operator= (const RegionHandling& obj);
      uint8 getSystemId(const std::string ipAddress) const;
      uint8 getSystemPosition(TiXmlElement* entityElement, const ::std::string category);
      uint64 getEntityId(TiXmlElement* entityElement, const ::std::string category);
      ::std::string getGroupCFGSinkName(TiXmlElement* element, ::std::string linkName, ::std::string category);
      ::std::string getAttribute(TiXmlElement* element, const char* atrribute, ::std::string category, const bool errmem = true);
      bool fetchEntitySourceIdsInfo(SourceIdsInfo& info, TiXmlElement* entityElement, const ::std::string category);
      void initializeSystemsInfo();
      void initializeCabinSystemsInfo();
      void initializeCockpitSystemInfo();
      void initializeSupportedRegionsInfo();
      void initializeCabinSinksStaticInfo();
      void initializeListenerUpdateFunctionsInfo();
      void initializeCabinCMASystemsInfo(TiXmlElement* element);
      void initializeCabinCMPSystemsInfo(TiXmlElement* element);
      void initializeCabinCMGSystemsInfo(TiXmlElement* element);
      void initializeCabinCMRSystemsInfo(TiXmlElement* element);
      void startSystemIdentificationTimer();
      void stopSystemIdentificationTimer(const bool flush);
      void storeLastActiveCabinRegionId(const uint32 regionId);
      void sendCabinsSystemsInfoUpdate(const SystemsInfoType& info);
      void sendRegionInfoUpdate();
      void sendStatusUpdate(const SystemInfo& info, const uint8 statusCategory);
      void sendStatusUpdate(const SystemsInfoType& info, const uint8 statusCategory);
      void cabinCarmodeIconUpdate();
      DataBindingItem<MicCabinStatusDataBindingSource> m_MICIconCabinData;
      /*
       * NOTE:
       *    -> The system id "0xFF" denotes the status is applicable for all systems
       */
      void evalAndUpdateCabinSystemStatus(const uint8 hmiState, const uint8 systemId = 0xFF);

      inline uint8 getHmiState() const
      {
         uint8 state = 0;
         if (NULL != _hmiStateHandler)
         {
            state = _hmiStateHandler->getHmiState();
         }
         return state;
      }

      inline uint8 getLastHmiState() const
      {
         uint8 state = 0;
         if (NULL != _hmiStateHandler)
         {
            state = _hmiStateHandler->getLastHmiState();
         }
         return state;
      }

      inline bool getPNMGrp10Status() const
      {
         bool status = false;
         if (NULL != _hmiStateHandler)
         {
            status = _hmiStateHandler->getPNMGrp10Status();
         }
         return status;
      }

      template< typename CBType >
      inline void addListener(::std::vector< CBType* >& info, CBType& imp)
      {
         if (::std::find(info.begin(), info.end(), (&imp)) == info.end())
         {
            info.push_back((&imp));
         }
      }

      template< typename CBType >
      inline void removeListener(::std::vector< CBType* >& info, CBType& imp)
      {
         typename ::std::vector< CBType* >::iterator itr = ::std::find(info.begin(), info.end(), (&imp));
         if (itr != info.end())
         {
            info.erase(itr);
         }
      }
      /**
       *  Member Variables Declaration
       */
      uint32 _regionId;
      uint32 _lastActiveCabinRegionId;
      SystemsInfo _systemsInfo;
      SinksStaticInfo _sinksStaticInfo;
      IHmiStateHandler* _hmiStateHandler;
      SupportedRegionsInfo _supportedRegionsInfo;

      bool isMIC2SwitchActive;
      bool isMIC3SwitchActive;
      /*
       * INFO:
       *   -> Param1 : System category
       *   -> Param2 : System status category
       *   -> Param3 : update send function
       */
      ListenerUpdateFunctionsInfo _listenerUpdateFunctionsInfo;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
      IApplicationSwitchServerComponent* _applicationSwitchServerComponent;
      ::asf::core::Timer* _systemIdentificationTimer;
      bool _carModeStatus;
      bool _coachModeStatus;
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      INanoMsgClientHandler* _nanoMsgClientHandler;
#endif
      ::boost::shared_ptr< HmiInfoServiceProxy> _hmiInfoServiceProxyClient;
      act_t _navigationModeChgUpd_RegId;
#ifndef WIN32 /* Linux / Other OS env except Windows */
      static pthread_mutex_t _safeLock;
#endif
      int _variantType;
};


} //namespace Core
} //namespace App


/**
 *  undef
 */

#undef REGION_HANDLING_LISTENER_DATA_DECL_BEGIN
#undef ADD_LISTENER_DATA_DECL
#undef REGION_HANDLING_LISTENER_DATA_DECL_END


#endif /* REGIONHANDLING_HEADER */
