/**
 * @file         : MasterHall.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : MasterHall is to handle HALL / Model component features
 *                 instantiation and courier message delegation
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "hall_std_if.h"
#include "MasterHall.h"
#include "Project.h"
#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_01_if.h"
#include "dp_generic_if.h"
#endif
#include "spm_core_fi_typesConst.h"
#include <Core/LanguageHandler.h>
#include <DataModel/DataModel.h>
#include <Core/LanguageHandler.h>
#include <DataModel/DataModelPower.h>
#include <DataModel/DataModelSound.h>
#include "AppHmi_MasterStateMachine.h"
#include <Core/Utility/MasterUtility.h>
#include <DataModel/Home/DataModelHome.h>
#include <DataModel/DataModelControlPanel.h>
#include <Core/RVCTestMode/RearViewCamera.h>
#include <Core/HeatCtrlClientHandler/HeatCtrlClientHandler.h>
#include <Core/AppFlyInListHandler/AppFlyIn.h>
#include <Core/RegionHandling/RegionHandling.h>
#include <Core/SyncDisplay/SyncDisplayHandler.h>
#include <Core/BTClientHandler/BTClientHandler.h>
#include <Core/HmiState/HmiStateHandlerFactory.h>
#include <Core/VariantHandling/VariantHandling.h>
#include <Core/SystemMonitorHandler/SystemMonitorHandler.h>
#include <Core/Diagnosis/diagnosisClientHandler.h>
#include <Core/VehicleSrvClient/VehicleSrvClient.h>
#include <Core/AudioInterface/SoundControllerPrj.h>
#include <Core/NanoMsgClientHandler/NanoMsgClientHandler.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandler.h>
#include <Core/HeatCtrlClientHandler/HeatCtrlClientHandler.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponent.h>
#include <Core/ContextPriorityHandler/ContextPriorityHandler.h>
#include <Core/ApplicationSurfaceInfoHandler/ApplicationSurfaceInfoHandler.h>
#include <Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponent.h>
#include <Core/AudioInterface/ResourceControllerPrj.h>
#include <Core/MediaClientHandler/MediaClientHandler.h>
#include <Core/RootContextHandler/RootContextHandler.h>
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
#include <Core/MasterGPIOHandler/MasterGPIOHandler.h>
#endif
#ifdef VARIANT_S_FTR_ENABLE_PLUGINS
#include <Common/PluginClientHandler/PluginClientHandler.h>
#endif
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
#include <Core/CabinConnectionHandler/CabinsActivityInfoDpHandler.h>
#include <Core/CabinConnectionHandler/CabinVideoMixInfoDpHandler.h>
#include <Core/CabinConnectionHandler/CabinConnectionHandler.h>
#include <Core/CabinConnectionHandler/CabinConnectionUtility.h>
#include <Core/CabinConnectionHandler/CabinRestorationHandler.h>
#include <DataModel/Source/DataModelSource.h>
#include <DataModel/DataModelErrorPopup.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/MasterHall.cpp.trc.h"
#endif

#define HMI_INFOSERVICE_INTERFACE ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService

using namespace ::hmibase;
using namespace ::App::datamodel;
using namespace ::languageHandler;
using namespace ::hmibase::apphmi_master::audio;
using namespace ::CourierTunnelService::CourierMessageReceiver;
using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;
using namespace ::bosch::cm::ai::hmi::masteraudioservice::AudioSourceChange;


/**
 *  Memeber Functions Definition
 */

namespace App {
namespace Core {


MasterHall::MasterHall() : LanguageHandlerAdapter(true)
   , _regionHandling(NULL)
   , _hmiStateHandler(NULL)
   , _variantHandling(NULL)
   , _systemMonitorHandler(NULL)
   , _navigationModeDBHandler(NULL)
   , _syncDisplayHandler(NULL)
   , _mediaClientHandler(NULL)
   , _rootContextHandler(NULL)
   , _contextSwitchHandler(NULL)
   , _contextPriorityHandler(NULL)
   , _applicationSurfaceInfoHandler(NULL)
   , _dataModelControlPanel(NULL)
   , _applicationSwitchServerComponent(NULL)
   , _heatClientHandler(NULL)
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   , _nanoMsgClientHandler(NULL)
   , _masterGPIOHandler(NULL)
#endif
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   , _dataModelSource(NULL)
   , _dataModelErrorPopup(NULL)
   , _cabinsActivityInfoDpHandler(NULL)
   , _cabinRestorationHandler(NULL)
   , _cabinVideoMixInfoDpHandler(NULL)
   , _cabinConnectionHandler(NULL)
   , _cabinConnectionUtility(NULL)
#endif

{
   ETG_TRACE_USR1(("MasterHall: CTOR"));
   DP_vCreateDatapool();
   LanguageHandlerAdapter::initLanguageHandler();
   createInstance();
   initialize();
}


MasterHall::~MasterHall()
{
   ETG_TRACE_USR1(("MasterHall: DTOR"));
   finalize();
   deleteInstance();
}


void MasterHall::finalize()
{
   finalizeAppFlyIn();
   finalizeDataModelHome();
   finalizeRegionHandling();
   finalizeDataModelSound();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   finalizeDataModelSource();
#endif
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   finalizeMasterGPIOHandler();
#endif
   finalizeDefSetServiceBase();
   finalizeDataModelControlPanel();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   finalizeCabinConnectionHandler();
   finalizeDataModelErrorPopop();
#endif

   finalizeSyncDisplayHandler();
   finalizeSoundConrollerPrj();
   finalizeSystemMonitorHandler();
   finalizeDataModelPower();
}


void MasterHall::initialize()
{
   initializeAppFlyIn();
   initializeDataModel();
   initializeDataModelHome();
   initializeDataModelSound();
   initializeDataModelPower();
   initializeRegionHandling();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   initializeDataModelSource();
#endif
   initializeDefSetServiceBase();
   initializeSMExternalVariables();
   initializeContextSwitchHandler();
   initializeDataModelControlPanel();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   initializeCabinRestorationHandler();
   initializeCabinConnectionHandler();
   initializeHmiInfoServiceServerComponent();
   initializeCabinConnectionUtility();
   initializeDataModelErrorPopop();
#endif
   initializeApplicationSwitchServerComponent();
   initializeSyncDisplayHandler();
   initializeSystemMonitorHandler();
   initializeDiagnosisClientHandler();
   initializeSoundControllerPrj();
   initializeConnectionControllerPrj();
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   initializeNanoMsgClientHandler();
#endif
}


void MasterHall::createInstance()
{
   if (NULL == _regionHandling)
   {
      _regionHandling = new RegionHandling();
   }
   if (NULL == _hmiStateHandler)
   {
      _hmiStateHandler = createHmiStateHandler();
   }
   if (NULL == _variantHandling)
   {
      _variantHandling = new VariantHandling();
   }
   if (NULL == _systemMonitorHandler)
   {
      _systemMonitorHandler = new SystemMonitorHandler();
   }
   if (NULL == _navigationModeDBHandler)
   {
      _navigationModeDBHandler = NavigationModeDBHandler::getInstance();
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL == _masterGPIOHandler)
   {
      _masterGPIOHandler = new MasterGPIOHandler();
   }
   if (NULL == _nanoMsgClientHandler)
   {
      _nanoMsgClientHandler = new NanoMsgClientHandler();
   }
#endif
   if (NULL == _syncDisplayHandler)
   {
      _syncDisplayHandler = new SyncDisplayHandler();
   }
   if (NULL == _mediaClientHandler)
   {
      _mediaClientHandler = new MediaClientHandler();
   }
   if (NULL == _rootContextHandler)
   {
      _rootContextHandler = new RootContextHandler();
   }
   if (NULL == _contextSwitchHandler)
   {
      _contextSwitchHandler = new ContextSwitchHandler();
   }
   if (NULL == _contextPriorityHandler)
   {
      _contextPriorityHandler = new ContextPriorityHandler();
   }
   if (NULL == _applicationSurfaceInfoHandler)
   {
      _applicationSurfaceInfoHandler = new ApplicationSurfaceInfoHandler();
   }
   if (NULL == _dataModelControlPanel)
   {
      _dataModelControlPanel = new DataModelControlPanel();
   }
   if (NULL == _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent = new ApplicationSwitchServerComponent();
   }
   if (NULL == _heatClientHandler)
   {
      _heatClientHandler = new HeatCtrlClientHandler();
   }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   if (NULL == _dataModelErrorPopup)
   {
      _dataModelErrorPopup = new DataModelErrorPopup();
   }
   if (NULL == _dataModelSource)
   {
      _dataModelSource = new DataModelSource();
   }
   if (NULL == _cabinsActivityInfoDpHandler)
   {
      _cabinsActivityInfoDpHandler = new CabinsActivityInfoDpHandler();
   }
   if (NULL == _cabinRestorationHandler)
   {
      _cabinRestorationHandler = new CabinRestorationHandler();
   }
   if (NULL == _cabinVideoMixInfoDpHandler)
   {
      _cabinVideoMixInfoDpHandler = new CabinVideoMixInfoDpHandler();
   }
   if (NULL == _cabinConnectionHandler)
   {
      _cabinConnectionHandler = new CabinConnectionHandler();
   }
   if (NULL == _cabinConnectionUtility)
   {
      _cabinConnectionUtility = new CabinConnectionUtility();
   }
#endif
   DataModel::Init();
   (void)AppFlyIn::getInstance();
   (void)DataModelHome::getInstance();
   (void)DataModelPower::getInstance();
   (void)DataModelSound::getInstance();
   (void)diagnosisClientHandler::poGetInstance();
   (void)RearViewCamera::poGetInstance();
   (void)BTClientHandler::poGetInstance();
   (void)VehicleSrvClient::poGetInstance();
#ifdef VARIANT_S_FTR_ENABLE_PLUGINS
   (void)PluginClientHandler::getInstance();
#endif
   (void)HmiInfoServiceServerComponent::poGetInstance();
   DefSetServiceBase::s_Intialize("AppHmi_MasterServicePort");
   APPHMI_MASTER_INIT_AUDIO_COMPONENTS(ConnectionControllerPrj, SoundControllerPrj, ResourceControllerPrj);
}


void MasterHall::deleteInstance()
{
   if (NULL != _regionHandling)
   {
      delete _regionHandling;
      _regionHandling = NULL;
   }
   if (NULL != _hmiStateHandler)
   {
      delete _hmiStateHandler;
      _hmiStateHandler = NULL;
   }
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   if (NULL != _masterGPIOHandler)
   {
      delete _masterGPIOHandler;
      _masterGPIOHandler = NULL;
   }
   if (NULL != _nanoMsgClientHandler)
   {
      delete _nanoMsgClientHandler;
      _nanoMsgClientHandler = NULL;
   }
#endif
   if (NULL != _syncDisplayHandler)
   {
      delete _syncDisplayHandler;
      _syncDisplayHandler = NULL;
   }
   if (NULL != _mediaClientHandler)
   {
      delete _mediaClientHandler;
      _mediaClientHandler = NULL;
   }
   if (NULL != _rootContextHandler)
   {
      delete _rootContextHandler;
      _rootContextHandler = NULL;
   }
   if (NULL != _variantHandling)
   {
      delete _variantHandling;
      _variantHandling = NULL;
   }
   if (NULL != _systemMonitorHandler)
   {
      delete _systemMonitorHandler;
      _systemMonitorHandler = NULL;
   }
   if (NULL != _navigationModeDBHandler)
   {
      delete _navigationModeDBHandler;
      _navigationModeDBHandler = NULL;
   }
   if (NULL != _contextSwitchHandler)
   {
      delete _contextSwitchHandler;
      _contextSwitchHandler = NULL;
   }
   if (NULL != _contextPriorityHandler)
   {
      delete _contextPriorityHandler;
      _contextPriorityHandler = NULL;
   }
   if (NULL != _applicationSurfaceInfoHandler)
   {
      delete _applicationSurfaceInfoHandler;
      _applicationSurfaceInfoHandler = NULL;
   }
   if (NULL != _dataModelControlPanel)
   {
      delete _dataModelControlPanel;
      _dataModelControlPanel = NULL;
   }
   if (NULL != _applicationSwitchServerComponent)
   {
      delete _applicationSwitchServerComponent;
      _applicationSwitchServerComponent = NULL;
   }
   if (NULL != _heatClientHandler)
   {
      delete _heatClientHandler;
      _heatClientHandler = NULL;
   }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   if (NULL != _dataModelErrorPopup)
   {
      delete _dataModelErrorPopup;
      _dataModelErrorPopup = NULL;
   }
   if (NULL != _dataModelSource)
   {
      delete _dataModelSource;
      _dataModelSource = NULL;
   }
   if (NULL != _cabinsActivityInfoDpHandler)
   {
      delete _cabinsActivityInfoDpHandler;
      _cabinsActivityInfoDpHandler = NULL;
   }
   if (NULL != _cabinRestorationHandler)
   {
      delete _cabinRestorationHandler;
      _cabinRestorationHandler = NULL;
   }
   if (NULL != _cabinVideoMixInfoDpHandler)
   {
      delete _cabinVideoMixInfoDpHandler;
      _cabinVideoMixInfoDpHandler = NULL;
   }
   if (NULL != _cabinConnectionHandler)
   {
      delete _cabinConnectionHandler;
      _cabinConnectionHandler = NULL;
   }
   if (NULL != _cabinConnectionUtility)
   {
      delete _cabinConnectionUtility;
      _cabinConnectionUtility = NULL;
   }
#endif
   DefSetServiceBase::s_Destrory();
}


void MasterHall::initLanguageDpHandler()
{
   if (NULL != LanguageHandler::getInstance())
   {
      LanguageHandler::getInstance()->initLanguageDpHandler<dp_tclAppHmi_MasterLanguageCode>();
   }
}


void MasterHall::finalizeAppFlyIn()
{
   AppFlyIn* imp = AppFlyIn::getInstance();
   if (NULL != imp)
   {
      imp->finalize();
   }
}


void MasterHall::initializeAppFlyIn()
{
   AppFlyIn* imp = AppFlyIn::getInstance();
   if (NULL != imp)
   {
      AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
      imp->setCabinConnectionHandler((*_cabinConnectionHandler));
#endif
      imp->setContextSwitchHandler((*_contextSwitchHandler));
      imp->setVariantHandling((*_variantHandling));
      imp->setRegionHandling((*_regionHandling));
      imp->setConnectionControllerPrj(tIns.getConnectionController<ConnectionControllerPrj>());
      imp->initialize();
   }
}


void MasterHall::finalizeDataModelHome()
{
   DataModelHome::getInstance().finalize();
}


void MasterHall::finalizeRegionHandling()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->finalize();
   }
}


void MasterHall::finalizeDataModelSound()
{
   DataModelSound::getInstance().finalize();
}


void MasterHall::finalizeDataModelPower()
{
   DataModelPower::getInstance().finalize();
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
void MasterHall::finalizeDataModelSource()
{
   if (NULL != _dataModelSource)
   {
      _dataModelSource->finalize();
   }
}


#endif


void MasterHall::initializeDataModelHome()
{
   DataModelHome::getInstance().setVariantHandling(_variantHandling);
   DataModelHome::getInstance().setRegionHandling((*_regionHandling));
   DataModelHome::getInstance().setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   DataModelHome::getInstance().setCabinConnectionHandler((*_cabinConnectionHandler));
   DataModelHome::getInstance().setCabinConnectionUtility((*_cabinConnectionUtility));
#endif
   DataModelHome::getInstance().setContextSwitchHandler((*_contextSwitchHandler));
   DataModelHome::getInstance().intialize();
}


void MasterHall::initializeDataModelSound()
{
   AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
   DataModelSound::getInstance().setSoundController(tIns.getSoundController<SoundControllerPrj>());
   DataModelSound::getInstance().setConnectionControllerPrj(tIns.getConnectionController<ConnectionControllerPrj>());
   DataModelSound::getInstance().setRegionHandling((*_regionHandling));
   DataModelSound::getInstance().setVehicleSrvClientInstance(*(VehicleSrvClient::poGetInstance()));
   DataModelSound::getInstance().initialize();
}


void MasterHall::initializeSoundControllerPrj()
{
   AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
   if (NULL != tIns.getSoundController<SoundControllerPrj>())
   {
      tIns.getSoundController<SoundControllerPrj>()->setResourceController(tIns.getResourceController<ResourceControllerPrj>());
      tIns.getSoundController<SoundControllerPrj>()->initialize();
   }
}


void MasterHall::initializeConnectionControllerPrj()
{
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   AudioControllerObjectManager& tIns                  = AudioControllerObjectManager::getInstance();
   ::boost::shared_ptr< IConnectionControllerPrj > imp = tIns.getConnectionController<ConnectionControllerPrj>();
   if (NULL != imp.get())
   {
      imp->setCabinConnectionHandler((*_cabinConnectionHandler));
   }
#endif
}


void MasterHall::initializeDataModelPower()
{
   DataModelPower& imp = DataModelPower::getInstance();
   imp.setVariantHandling((*_variantHandling));
   imp.setRegionHandling((*_regionHandling));
   imp.setDataModelSound((DataModelSound::getInstance()));
   imp.setHMIStateHandler((*_hmiStateHandler));
   imp.setRearViewCamera((*(RearViewCamera::poGetInstance())));
   imp.setDataModel((DataModel::getInstance()));
   imp.setVehicleSrvClientInstance(*(VehicleSrvClient::poGetInstance()));
   imp.initialize();
}


void MasterHall::initializeDataModel()
{
   DataModel& imp = DataModel::getInstance();
   imp.setContextSwitchHandler((*_contextSwitchHandler));
   imp.setRegionHandling((*_regionHandling));
}


void MasterHall::initializeDiagnosisClientHandler()
{
   if (NULL != diagnosisClientHandler::poGetInstance())
   {
      diagnosisClientHandler::poGetInstance()->setDataModel((::App::datamodel::DataModel::getInstance()));
   }
}


void MasterHall::initializeRegionHandling()
{
   if (NULL != _regionHandling)
   {
      _regionHandling->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _regionHandling->setApplicationSwitchServerComponent((*_applicationSwitchServerComponent));
      _regionHandling->setHMIStateHandler((*_hmiStateHandler));
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _regionHandling->setNanoMsgClientHandler((*_nanoMsgClientHandler));
#endif
      _regionHandling->initialize();
   }
}


void MasterHall::initializeSyncDisplayHandler()
{
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->setVariantHandling((*_variantHandling));
      _syncDisplayHandler->setHMIStateHandler((*_hmiStateHandler));
      _syncDisplayHandler->initialize();
   }
}


void MasterHall::initializeSystemMonitorHandler()
{
   if (NULL != _systemMonitorHandler)
   {
      AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
      _systemMonitorHandler->setSoundController(tIns.getSoundController<SoundControllerPrj>());
      _systemMonitorHandler->setConnectionController(tIns.getConnectionController<ConnectionControllerPrj>());
      _systemMonitorHandler->setResourceController(tIns.getResourceController<ResourceControllerPrj>());
      _systemMonitorHandler->setVehicleSrvClient(*(VehicleSrvClient::poGetInstance()));
      _systemMonitorHandler->setMediaClientHandler(*(_mediaClientHandler));
      _systemMonitorHandler->setRegionHandling((*_regionHandling));
      _systemMonitorHandler->initialize();
   }
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
void MasterHall::initializeDataModelSource()
{
   if (NULL != _dataModelSource)
   {
      AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
      _dataModelSource->setVariantHandling((*_variantHandling));
      _dataModelSource->setMediaClientHandler((*_mediaClientHandler));
      _dataModelSource->setContextSwitchHandler((*_contextSwitchHandler));
      _dataModelSource->setResourceController(tIns.getResourceController<ResourceControllerPrj>());
      _dataModelSource->setConnectionController(tIns.getConnectionController<ConnectionControllerPrj>());
      _dataModelSource->setRegionHandling((*_regionHandling));
      _dataModelSource->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _dataModelSource->initialize();
   }
}


#endif


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void MasterHall::finalizeMasterGPIOHandler()
{
   if (NULL != _masterGPIOHandler)
   {
      _masterGPIOHandler->_vCloseGpio_MICSwitch();
   }
}


#endif


void MasterHall::finalizeSyncDisplayHandler()
{
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->finalize();
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void MasterHall::initializeNanoMsgClientHandler()
{
   if (NULL != _nanoMsgClientHandler)
   {
      _nanoMsgClientHandler->setRegionHandling((*_regionHandling));
      _nanoMsgClientHandler->setHMIStateHandler((*_hmiStateHandler));
   }
}


#endif


void MasterHall::finalizeDefSetServiceBase()
{
   if (NULL != DefSetServiceBase::GetInstance())
   {
      DefSetServiceBase::GetInstance()->vUnRegisterforUpdate(this);
   }
}


void MasterHall::initializeDefSetServiceBase()
{
   if (NULL != DefSetServiceBase::GetInstance())
   {
      DefSetServiceBase::GetInstance()->vRegisterforUpdate(this);
   }
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
void MasterHall::initializeMasterGPIOHandler()
{
   if (NULL != _masterGPIOHandler)
   {
      _masterGPIOHandler->_vConfigureGpio_MICSwitch();
   }
}


#endif


void MasterHall::finalizeDataModelControlPanel()
{
   if (NULL != _dataModelControlPanel)
   {
      _dataModelControlPanel->finalize();
   }
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
void MasterHall::finalizeCabinConnectionHandler()
{
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->finalize();
   }
}


void MasterHall::finalizeDataModelErrorPopop()
{
   if (NULL != _dataModelErrorPopup)
   {
      _dataModelErrorPopup->finalize();
   }
}


#endif


void MasterHall::finalizeSoundConrollerPrj()
{
   AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
   if (NULL != tIns.getSoundController<SoundControllerPrj>())
   {
      tIns.getSoundController<SoundControllerPrj>()->finalize();
   }
}


void MasterHall::finalizeSystemMonitorHandler()
{
   if (NULL != _systemMonitorHandler)
   {
      _systemMonitorHandler->finalize();
   }
}


void MasterHall::initializeSMExternalVariables()
{
   int dp_VehicleType = App::Core::MasterUtility::getKDSVehicleType();
   if (dp_VehicleType == VEHICLE_TYPE_SMART)
   {
      ETG_TRACE_USR4(("HMI_Master getKDSVehicleType = %d", dp_VehicleType));
      sm_IN_bIsDisclaimerActive = 1;
      sm_IN_bIsAntiTheftActive = 1;
#ifdef VARIANT_S_FTR_ENABLE_ANTI_THEFT
      int dp_AntiTheft = App::Core::MasterUtility::getDpAntiTheft();
      ETG_TRACE_USR4(("HMI_Master getKDSVehicleType = %d", dp_AntiTheft));
      if (dp_AntiTheft == 1) //if antitheft enabled
      {
         sm_IN_bIsAntiTheftActive = 1;
      }
      else
      {
         sm_IN_bIsAntiTheftActive = 0;
      }
#endif
   }
   else
   {
      sm_IN_bIsDisclaimerActive = 0;
      sm_IN_bIsAntiTheftActive = 0;
   }
}


void MasterHall::initializeContextSwitchHandler()
{
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->setRegionHandling((*_regionHandling));
      _contextSwitchHandler->setHMIStateHandler((*_hmiStateHandler));
      _contextSwitchHandler->setRootContextHandler((*_rootContextHandler));
      _contextSwitchHandler->setContextPriorityHandler((*_contextPriorityHandler));
      _contextSwitchHandler->setApplicationSurfaceInfoHandler((*_applicationSurfaceInfoHandler));
      _contextSwitchHandler->setApplicationSwitchServerComponent((*_applicationSwitchServerComponent));
      ApplicationsContextInfo tInfo;
      ApplicationContextInfo tApplicationContextInfo;
      tApplicationContextInfo.setApplicationId(APPID_APPHMI_MASTER);
      if (fnGetVariantType() !=  VARIANT_TYPE_CABIN_ONLY)
      {
         tApplicationContextInfo.setRegionId(Region__REG_COCKPIT);

         tApplicationContextInfo.setContextId(enActivityIDs__eActivityID_HOME);
         tInfo.push_back(tApplicationContextInfo);
      }
      tApplicationContextInfo.setRegionId(Region__REG_A);
      tApplicationContextInfo.setContextId(enActivityIDs__eActivityID_CABINA_HOME);
      tInfo.push_back(tApplicationContextInfo);
      tApplicationContextInfo.setRegionId(Region__REG_B);
      tApplicationContextInfo.setContextId(enActivityIDs__eActivityID_CABINB_HOME);
      tInfo.push_back(tApplicationContextInfo);
      _contextSwitchHandler->setDefaultApplicationsContextInfo(tInfo);
   }
}


void MasterHall::initializeDataModelControlPanel()
{
   if (NULL != _dataModelControlPanel)
   {
      _dataModelControlPanel->setRegionHandling((*_regionHandling));
      _dataModelControlPanel->setContextSwitchHandler((*_contextSwitchHandler));
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
      _dataModelControlPanel->setCabinConnectionHandler((*_cabinConnectionHandler));
#endif
      _dataModelControlPanel->setApplicationSwitchServerComponent((*_applicationSwitchServerComponent));
      _dataModelControlPanel->initialize();
   }
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
void MasterHall::initializeDataModelErrorPopop()
{
   if (NULL != _dataModelErrorPopup)
   {
      _dataModelErrorPopup->setRegionHandling((*_regionHandling));
      _dataModelErrorPopup->setHMIStateHandler((*_hmiStateHandler));
      _dataModelErrorPopup->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _dataModelErrorPopup->setCabinConnectionUtility((*_cabinConnectionUtility));
      _dataModelErrorPopup->initialize();
   }
}


#endif


void MasterHall::initializeApplicationSwitchServerComponent()
{
   if (NULL != _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent->setRegionHandling((*_regionHandling));
      _applicationSwitchServerComponent->setContextSwitchHandler((*_contextSwitchHandler));
      _applicationSwitchServerComponent->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _applicationSwitchServerComponent->setAppFlyIn((*(AppFlyIn::getInstance())));
   }
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
void MasterHall::initializeCabinRestorationHandler()
{
   if (NULL != _cabinRestorationHandler)
   {
      _cabinRestorationHandler->setRegionHandling((*_regionHandling));
      _cabinRestorationHandler->setHMIStateHandler((*_hmiStateHandler));
      _cabinRestorationHandler->setSyncDisplayHandler((*_syncDisplayHandler));
      _cabinRestorationHandler->setCabinVideoMixInfoDpHandler((*_cabinVideoMixInfoDpHandler));
      _cabinRestorationHandler->setCabinsActivityInfoDpHandler((*_cabinsActivityInfoDpHandler));
      _cabinRestorationHandler->setCabinConnectionHandler((*_cabinConnectionHandler));
      _cabinRestorationHandler->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _cabinRestorationHandler->initialize();
   }
}


void MasterHall::finalizeCabinRestorationHandler()
{
   if (NULL != _cabinRestorationHandler)
   {
      _cabinRestorationHandler->finalize();
   }
}


void MasterHall::initializeCabinConnectionHandler()
{
   if (NULL != _cabinConnectionHandler)
   {
      AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
      _cabinConnectionHandler->setRegionHandling((*_regionHandling));
      _cabinConnectionHandler->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
      _cabinConnectionHandler->setResourceController(tIns.getResourceController<ResourceControllerPrj>());
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      _cabinConnectionHandler->setNanoMsgClientHandler((*_nanoMsgClientHandler));
#endif
      _cabinConnectionHandler->setHMIStateHandler((*_hmiStateHandler));
      _cabinConnectionHandler->setCabinConnectionUtility((*_cabinConnectionUtility));
      _cabinConnectionHandler->setCabinsActivityInfoDpHandler((*_cabinsActivityInfoDpHandler));
      _cabinConnectionHandler->setCabinVideoMixInfoDpHandler((*_cabinVideoMixInfoDpHandler));
      _cabinConnectionHandler->setCabinRestorationHandler((*_cabinRestorationHandler));
      _cabinConnectionHandler->initialize();
   }
}


void MasterHall::initializeHmiInfoServiceServerComponent()
{
   if (NULL != HmiInfoServiceServerComponent::poGetInstance())
   {
      HmiInfoServiceServerComponent::poGetInstance()->setCabinVideoMixInfoImpl((*_cabinConnectionHandler));
   }
}


void MasterHall::initializeCabinConnectionUtility()
{
   if (NULL != _cabinConnectionUtility)
   {
      AudioControllerObjectManager& tIns = AudioControllerObjectManager::getInstance();
      _cabinConnectionUtility->setRegionHandling(*_regionHandling);
      _cabinConnectionUtility->setVariantHandling(*_variantHandling);
      _cabinConnectionUtility->setCabinRestorationHandler(*_cabinRestorationHandler);
      _cabinConnectionUtility->setApplicationSwitchServerComponent(*_applicationSwitchServerComponent);
      _cabinConnectionUtility->setConnectionControllerPrj(tIns.getConnectionController<ConnectionControllerPrj>());
      _cabinConnectionUtility->setHmiInfoServiceServerComponent((*(HmiInfoServiceServerComponent::poGetInstance())));
   }
}


#endif

bool MasterHall::delegateMessage(const ::Courier::Message& Msg)
{
   COURIER_MSG_DELEGATE_TO_REF(DataModel::getInstance())
   COURIER_MSG_DELEGATE_TO_REF(DataModelHome::getInstance())
   COURIER_MSG_DELEGATE_TO_REF(DataModelPower::getInstance())
   COURIER_MSG_DELEGATE_TO_REF(DataModelSound::getInstance())
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   COURIER_MSG_DELEGATE_TO_OBJ(_dataModelSource)
#endif
   COURIER_MSG_DELEGATE_TO_OBJ(_dataModelControlPanel)
   COURIER_MSG_DELEGATE_TO_OBJ(diagnosisClientHandler::poGetInstance())
   COURIER_MSG_DELEGATE_TO_OBJ(AppFlyIn::getInstance())
#ifdef VARIANT_S_FTR_ENABLE_PLUGINS
   COURIER_MSG_DELEGATE_TO_OBJ(PluginClientHandler::getInstance())
#endif
   COURIER_MSG_DELEGATE_TO_CLASS(LanguageHandlerAdapter)
   COURIER_MSG_DELEGATE_TO_CLASS(hmibase::app::master::AppHmi_MasterHallBase)
   COURIER_MSG_DELEGATE_TO_REF(_mPopupTimer)
   return false;
}


#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION

bool MasterHall::onCourierMessage(const ActivateMIC& oMsg)
{
   ETG_TRACE_USR4(("MasterHall: ActivateMicMsg Status:%d, Value:%d, KeyType:%d", oMsg.GetStatus(), oMsg.GetValue(), oMsg.GetKeyType()));
   if (oMsg.GetValue() == REGION_COCKPIT)
   {
      if (_cabinConnectionHandler != NULL)
      {
         _cabinConnectionHandler->onUpdateMic1Status(oMsg.GetStatus());
      }
      if (_dataModelControlPanel != NULL)
      {
         _dataModelControlPanel->onUpdateMic1Status(oMsg.GetStatus(), oMsg.GetKeyType());
      }
      if (NULL != _syncDisplayHandler)
      {
         _syncDisplayHandler->onMIC1StatusUpdate(oMsg.GetStatus());
      }
   }
   return true;
}


#endif


bool MasterHall::onChangeHmiSubStateReq()
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->updateHMIStateChangeToSM(_hmiStateHandler->getHmiState());
   }
   return true;
}


bool MasterHall::onCourierMessage(const ChangeRegionMsg& oMsg)
{
   ETG_TRACE_USR4(("MasterHall: ChangeRegionMsg: RegionId: %d", oMsg.GetRegionId()));
   if (NULL != _regionHandling)
   {
      _regionHandling->setRegionId(oMsg.GetRegionId());
   }
   return true;
}


bool MasterHall::onCourierMessage(const StartBeepReqMsg& oMsg)
{
   AudioControllerObjectManager::getInstance().getSoundController<SoundController>()->requestBeep(oMsg.GetBeepType());
   return true;
}


bool MasterHall::onCourierMessage(const DumpSystemsInfo& oMsg)
{
   if (NULL != _regionHandling)
   {
      _regionHandling->dumpSystemsInfo();
   }
   return true;
}


bool MasterHall::onCourierMessage(const UpdateSPM_HMI_ON& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->sendFirstPicDrawnUpdate();
   }
   return true;
}


bool MasterHall::onCourierMessage(const ShowHideRVCSceneMsg& oMsg)
{
   ETG_TRACE_USR4(("MasterHall: ShowHideRVCSceneMsg: Status: %d", oMsg.GetShowhide()));
   if (NULL != RearViewCamera::poGetInstance())
   {
      RearViewCamera::poGetInstance()->onActivateDeactivateRVC(oMsg.GetShowhide());
   }
   return true;
}


bool MasterHall::onCourierMessage(const ReqSpeechSessionMsg& /* oMsg */)
{
   if (NULL != _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent->sendSpeechSessionSignal(true);
   }
   return true;
}


bool MasterHall::onCourierMessage(const Activate_Speech_Msg& oMsg)
{
   if (NULL != _applicationSwitchServerComponent)
   {
      _applicationSwitchServerComponent->sendSPIVoiceRecognitionSignal(oMsg.GetValue());
   }
   return true;
}


bool MasterHall::onCourierMessage(const ClearLastApplication& oMsg)
{
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->onClearLastApplication();
   }
   return true;
}


bool MasterHall::onCourierMessage(const HourLogicModeStateMsg& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->sendStandByRequest(!(oMsg.GetStatus()));
   }
   return true;
}


bool MasterHall::onCourierMessage(const EnableConnectionReqMsg& oMsg)
{
   bool res = true;
   ETG_TRACE_USR4(("MasterHall: ConnectionRequest: Request: %d", oMsg.GetRequestId()));
   switch (oMsg.GetRequestId())
   {
      /* case AudioConnectionRequest::GUI_STARTUP_FINISHED:
         AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>().setStartUpAvailable();
         break; */
      case AudioConnectionRequest::GROUP_REQUEST:
         AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>()->groupRequest(oMsg.GetUserData1());
         break;
      case AudioConnectionRequest::LAST_APPLICATION_REQUEST:
      {
         //AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>()->sendLastApplicationRequest();
         if (NULL != _contextSwitchHandler)
         {
            bool lastApplicationDebounceTimerStatus = false;
            if (NULL != _hmiStateHandler)
            {
               lastApplicationDebounceTimerStatus = (_hmiStateHandler->getHmiState() == ::spm_core_fi_types::T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_PHONE) ? true : false;
            }
            _contextSwitchHandler->onSwitchLastApplication(lastApplicationDebounceTimerStatus);
         }
         break;
      }
      case AudioConnectionRequest::VISIBLE_APPLICATION_REQUEST:
         AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>()->setVisibleApplication(static_cast<enApplicationId>(oMsg.GetUserData1()));
         break;
      default:
         res = false;
         break;
   }
   return res;
}


bool MasterHall::onCourierMessage(const SimSubSystemErrorStatus& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      SubSystemsErrorStatusInfo tInfo;
      ::std::pair< uint8, uint16 > temp(oMsg.GetCategory(), oMsg.GetErrorInfo());
      tInfo.insert(::std::pair< uint8, ::std::pair< uint8, uint16 > >(oMsg.GetSystemId(), temp));
      _hmiStateHandler->handleSubSystemsErrorStatus(tInfo);
   }
   return true;
}


bool MasterHall::onCourierMessage(const SimHmiStateChange& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->onHmiStateStatus(oMsg.GetHmiState());
   }
   return true;
}


bool MasterHall::onCourierMessage(const SimVideoMixIndexInfo& oMsg)
{
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->onSetCabinVideoMixInfo(oMsg.GetRegionId(), oMsg.GetVideoMixIndex(), oMsg.GetReason());
   }
#endif
   return true;
}


bool MasterHall::onCourierMessage(const GuiStartupFinishedUpdMsg& oMsg)
{
   ETG_TRACE_USR4(("MasterHall: HMI_Master Calling GuiStartupFinishedUpdMsg"));
   AudioControllerObjectManager::getInstance().getConnectionController<ConnectionControllerPrj>()->setStartUpAvailable();
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   initializeMasterGPIOHandler();
#endif
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->onGUIInitializationStatus(true);
   }
   return hmibase::app::master::AppHmi_MasterHallBase::onCourierMessage(oMsg);
}


bool MasterHall::onCourierMessage(const ContextSwitchFromHomeMsg& oMsg)
{
   if (NULL != _applicationSwitchServerComponent) //TODO: Re-Visit -sve2cob
   {
      //INFO: This is responsible to deactivate the speech session on press of home button
      _applicationSwitchServerComponent->sendSpeechSessionSignal(false);
   }
   if (NULL != _contextSwitchHandler)
   {
      ContextSwitchRequestInfo tInfo;
      tInfo.setRegionId(oMsg.GetRegionId());
      tInfo.setSourceContextId(oMsg.GetSourceActivityId());
      tInfo.setTargetContextId(oMsg.GetTargetActivityId());
      tInfo.setSourceApplicationId(oMsg.GetSourceAppId());
      tInfo.setTargetApplicationId(oMsg.GetTargetAppId());
      _contextSwitchHandler->onContextSwitchRequest(tInfo);
   }
   return true;
}


bool MasterHall::onCourierMessage(const DumpContextSwitchStackInfo& /* oMsg */)
{
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->dumpSwitchStackInfo();
   }
   return true;
}


bool MasterHall::onCourierMessage(const FaciaON_OFF_Btn_ResetReqMsg& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->sendRestartRequest();
   }
   return true;
}


bool MasterHall::onCourierMessage(const ApplicationSwitchCompleteMsg& oMsg)
{
   /* if (NULL != HmiInfoServiceServerComponent::poGetInstance())
   {
      HmiInfoServiceServerComponent::poGetInstance()->sendForegroundApplicationUpdate(sm_activeapplicationID);
   } */
   return true;
}


bool MasterHall::onCourierMessage(const PluginEntityDetailsUpdateMsg& oMsg)
{
   if (NULL != _regionHandling)
   {
      _regionHandling->onPluginEntityDetailsUpdate(oMsg.GetPluginEntityListData());
   }
   return true;
}


bool MasterHall::onCourierMessage(const StoreContextSwitchFromHomeMsg& oMsg)
{
   if (NULL != _contextSwitchHandler)
   {
      ApplicationContextInfo tInfo;
      tInfo.setRegionId(oMsg.GetRegionId());
      tInfo.setApplicationId(oMsg.GetTargetAppId());
      tInfo.setContextId(oMsg.GetTargetActivityId());
      _contextSwitchHandler->onStoreContext(tInfo);
   }
   return true;
}


bool MasterHall::onCourierMessage(const FaciaON_OFF_Btn_ShutDownReqMsg& oMsg)
{
   if (NULL != _hmiStateHandler)
   {
      _hmiStateHandler->sendHKeyPressedRequest();
   }
   return true;
}


bool MasterHall::onCourierMessage(const ExecuteContextSwitchRequestMsg& oMsg)
{
   if (NULL != _contextSwitchHandler)
   {
      ContextSwitchRequestInfo tInfo;
      tInfo.setRegionId(oMsg.GetRegionId());
      tInfo.setSourceApplicationId(oMsg.GetSourceAppId());
      tInfo.setTargetApplicationId(oMsg.GetTargetAppId());
      tInfo.setSourceContextId(oMsg.GetSourceActivityId());
      tInfo.setTargetContextId(oMsg.GetTargetActivityId());
      _contextSwitchHandler->onExecuteContextSwitchRequest(tInfo, oMsg.GetExecuteCode());
   }
   return true;
}


bool MasterHall::onCourierMessage(const ApplicationSwitchCompleteUpdMsg& oMsg)
{
   uint32 appId = 0;
   if ((NULL != _applicationSurfaceInfoHandler) && (oMsg.GetSurfaceState() == ::hmibase::SURFACESTATE_VISIBLE))
   {
      appId = _applicationSurfaceInfoHandler->getApplicationId(oMsg.GetSurfaceID());
   }

   if (appId != 0)
   {
      if (NULL != _contextSwitchHandler)
      {
         _contextSwitchHandler->onApplicationSwitchComplete(appId);
      }

      AppFlyIn* imp = AppFlyIn::getInstance();
      if (NULL != imp)
      {
         imp->onApplicationSwitchComplete(appId);
      }

      if (NULL != _dataModelControlPanel)
      {
         _dataModelControlPanel->onApplicationSwitchComplete(appId);
      }

      if (NULL != HmiInfoServiceServerComponent::poGetInstance())
      {
         HmiInfoServiceServerComponent::poGetInstance()->sendForegroundApplicationUpdate(appId);
      }
   }
   return true;
}


bool MasterHall::onCourierMessage(const FirstApplicationActivatedUpdMsg& /*oMsg*/)
{
   if (NULL != _syncDisplayHandler)
   {
      _syncDisplayHandler->sendFirstApplicationActiveUpdate();
   }
   return true;
}


bool MasterHall::onCourierMessage(const NavigationModeActiveMsg& oMsg)
{
   ETG_TRACE_USR4(("MasterHall:NavigationModeActiveMsg: carMode= %d coachMode= %d", oMsg.GetCarModeStatus(), oMsg.GetCoachModeStatus()));
   if (NULL != HmiInfoServiceServerComponent::poGetInstance())
   {
      HmiInfoServiceServerComponent::poGetInstance()->sendNavigationModeChangeSignal(oMsg.GetCarModeStatus(), oMsg.GetCoachModeStatus());
   }
   return true;
}


bool MasterHall::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("MasterHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
bool MasterHall::onCourierMessage(const MapOutStatusUpdMsg& oMsg)
{
   if (_cabinConnectionHandler != NULL)
   {
      _cabinConnectionHandler->onMapOutStatusUpdMsg(oMsg.GetRegionId(), oMsg.GetMapOutState());
   }
   return true;
}


bool MasterHall::onCourierMessage(const ToggleMapOutStatus& oMsg)
{
   if (_cabinConnectionHandler != NULL)
   {
      _cabinConnectionHandler->onToggleMapOutStatus(oMsg.GetRegionId());
   }
   return true;
}


#endif

void MasterHall::reqPrepareResponse(const PrepareResDiagMsgData& /*oPrepareResDiagMsgData*/)
{
   ETG_TRACE_USR4(("MasterHall: reqPrepareResponse"));
   if (NULL != DefSetServiceBase::GetInstance())
   {
      DefSetServiceBase::GetInstance()->sendPrepareResponse(0, this);
   }
}


void MasterHall::reqExecuteResponse(const ExecuteResDiagMsgData& /*oExecuteResDiagMsgData*/)
{
   ETG_TRACE_USR4(("MasterHall: reqExecuteResponse"));
   if (NULL != DefSetServiceBase::GetInstance())
   {
      DefSetServiceBase::GetInstance()->sendExecuteResponse(0, this);
   }
}


void MasterHall:: reqFinalizeResponse(const FinalizeResDiagMsgData& /*oFinalizeResDiagMsgData*/)
{
   ETG_TRACE_USR4(("MasterHall: reqFinalizeResponse"));
   if (NULL != DefSetServiceBase::GetInstance())
   {
      DefSetServiceBase::GetInstance()->sendFinalizeResponse(0, this);
   }
}


} //namespace Core
} //namespace App
