/**
 * @file         : MasterGui.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : INF4CV is to handle GUI / Model component features
 *                 instantiation and courier message delegation
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "gui_std_if.h"
#include "MasterGui.h"
#include <stdlib.h>
#include <HMI/CGIComponents/CGIApp.h>
#include <Core/IlmClient/IlmClient.h>
#include <AppBase/IApplicationSettings.h>
#include <Widgets/2D/WidgetGestureConfig.h>
#include <Core/KeyRouter/KeyRouterFactory.h>
#include <HMI/CGIComponents/AppViewSettings.h>
#include <Common/AppBaseSettings/KeyMapping.h>
#include <Core/KeyRouter/IKeyRouterUserInterface.h>
#include <Core/UserInterfaceHandler/UserInterfaceHandler.h>


using namespace ::hmi;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_MASTER_MAIN
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_MASTER
#define ETG_I_TTFIS_CMD_PREFIX            "HMI_MASTER_"
#define ETG_I_FILE_PREFIX                 App::Core::MasterGui::
#include "trcGenProj/Header/MasterGui.cpp.trc.h"
#endif // VARIANT_S_FTR_ENABLE_TRC_GEN


//                                            main surface,                                             video surface,                             popup surfaces
//DEFAULT_APPSETTINGS(static_cast<unsigned int>(SURFACEID_MAIN_SURFACE_MASTER), static_cast<unsigned int>(SURFACEID_NONE), static_cast<unsigned int>(SURFACEID_TOP_POPUP_SURFACE_MASTER), static_cast<unsigned int>(SURFACEID_CENTER_POPUP_SURFACE_MASTER))
APPSETTINGS_BEGIN()
APPSETTINGS_ADD_MAINSURFACE(SURFACEID_MAIN_SURFACE_MASTER),

                            APPSETTINGS_ADD_POPUPSURFACE(SURFACEID_TOP_POPUP_SURFACE_MASTER),
                            APPSETTINGS_ADD_POPUPSURFACE(SURFACEID_CENTER_POPUP_SURFACE_MASTER),

                            APPSETTINGS_END();


namespace App {
namespace Core {


MasterGui::MasterGui() : GuiComponentBase(hmibase::app::base::appHmi_Master, appSettings)
   , _ilmClient(new IlmClient())
{
   ETG_I_REGISTER_FILE();
   ETG_I_REGISTER_CHN(TraceCmd_NotProcessedMsg);
   _hmiAppCtrlProxyHandler.setTraceId(TR_CLASS_APPHMI_MASTER_APPCTRL_PROXY);
   // this process makes use of sync block mechanism to consume/receive pre rendered imaged from other processes direct texture
   SYNC_BLOCK_CONSUMER_ID(765)
   SYNC_BLOCK_PROVIDER_INSTANCES(TEXTURE_ID_COUNT)
   ScreenBrokerClient::SurfaceDisplayMap::registerItem(SURFACEID_MAIN_SURFACE_MASTER, 0x01);
   ScreenBrokerClient::SurfaceDisplayMap::registerItem(SURFACEID_TOP_POPUP_SURFACE_MASTER, 0x01);
   ScreenBrokerClient::SurfaceDisplayMap::registerItem(SURFACEID_CENTER_POPUP_SURFACE_MASTER, 0x01);
   _userInterfaceHandler = new UserInterfaceHandler();
   if (NULL != _userInterfaceHandler)
   {
      _userInterfaceHandler->setHmiAppCtrlAppInfoImpl(&hmiAppCtrlStubHandler);
   }
   ::std::vector<uint32> tDisplayInfo;
   tDisplayInfo.push_back(0); //INFO: As of now display id is used as 0 in HKStatusChangedUpdMsg
   ::keyRouterFactory::KeyRouterFactory::createInstance();
   ::keyRouterFactory::KeyRouterFactory::getInstance()->createKeyRouters(tDisplayInfo);
   ::keyRouterFactory::KeyRouterFactory::getInstance()->setKeyRouterStubImpl(&hmiAppCtrlStubHandler);
   ::keyRouterFactory::KeyRouterFactory::getInstance()->setHmiAppCtrlAppInfoImpl(&hmiAppCtrlStubHandler);
   ::keyRouterFactory::KeyRouterFactory::getInstance()->setKeyRouterUserInterfaceImpl(_userInterfaceHandler);
}


MasterGui::~MasterGui()
{
   if (_ilmClient != NULL)
   {
      delete _ilmClient;
      _ilmClient = NULL;
   }
   if (NULL != _userInterfaceHandler)
   {
      delete _userInterfaceHandler;
      _userInterfaceHandler = NULL;
   }
}


unsigned int MasterGui::getDefaultTraceClass()
{
   return TR_CLASS_APPHMI_MASTER_MAIN;
}


void MasterGui::preRun()
{
//	DP_vCreateDatapool();
//	PersistentValuesRead();
   configureHomeScreenGesture();
}


void MasterGui::postRun()
{
//	PersistentValuesWrite();
}


void MasterGui::setupCgiInstance()
{
   std::string assetFileName = hmibase::app::base::SystemConfiguration::getAssetFileName();
   setCgiApp(new CGIApp(assetFileName.c_str(), _hmiAppCtrlProxyHandler));
}


bool MasterGui::onCourierMessage(const GuiStartupFinishedUpdMsg& msg)
{
   bool isGuiComponentBaseMsg = false;
   //hmiAppCtrlStubHandler.setGuiInitialized();
   isGuiComponentBaseMsg = GuiComponentBase::OnMessage(msg);
   if (isGuiComponentBaseMsg)
   {
      ETG_TRACE_USR4(("MasterGui::onCourierMessage : GuiComponentBase::OnMessage :: True"));
      isGuiComponentBaseMsg = false;
   }
   return isGuiComponentBaseMsg;
}


void MasterGui::configureHomeScreenGesture()
{
   ETG_TRACE_USR4(("MasterGui::configureHomeScreenGesture()"));
   const WidgetGestureConfig::Swipe& defaultSwipe = WidgetGestureConfig::getDefaultSwipe();
   WidgetGestureConfigRegistry::registerItem(HS_HORIZONTAL_SWIPE_GESTURE_CONFIG_ID, WidgetGestureConfig::Swipe(defaultSwipe.Enabled, defaultSwipe.MinSpeed, 100));
   const WidgetGestureConfig::Tap& defaultTap = WidgetGestureConfig::getDefaultTap();
   WidgetGestureConfigRegistry::registerItem(HS_GADGET_TAB_GESTURE_CONFIG_ID, WidgetGestureConfig::Tap(defaultTap.Enabled, defaultTap.HoldTime, defaultTap.RepeatTime, defaultTap.DoubleTapTime, 30));
}


ETG_I_CMD_DEFINE((TraceCmd_DumpSystemsInfo, "TraceCmd_DumpSystemsInfo"))
void MasterGui::TraceCmd_DumpSystemsInfo()
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_DumpSystemsInfo"));
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(DumpSystemsInfo)();
   if (msg != NULL)
   {
      (void)msg->Post();
   }
}


ETG_I_CMD_DEFINE((TraceCmd_VOLUME_DECREASE_CHANGE, "TraceCmd_VOLUME_DECREASE_CHANGE "))
void MasterGui::TraceCmd_VOLUME_DECREASE_CHANGE()
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_VOLUME_DECREASE_CHANGE"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SetVolumeReqMsg)(-1, 2);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("Volume decrease value simulated"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_VOLUME_INCREASE_CHANGE, "TraceCmd_VOLUME_INCREASE_CHANGE "))
void MasterGui::TraceCmd_VOLUME_INCREASE_CHANGE()
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_VOLUME_INCREASE_CHANGE"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SetVolumeReqMsg)(+1, 2);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("Volume increase value simulated"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_DumpContextSwitchStackInfo, "TraceCmd_DumpContextSwitchStackInfo"))
void MasterGui::TraceCmd_DumpContextSwitchStackInfo()
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_DumpContextSwitchStackInfo"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(DumpContextSwitchStackInfo)();
   if (msg != NULL)
   {
      msg->Post();
   }
}


ETG_I_CMD_DEFINE((TraceCmd_MIC_ANNOUNCEMENT, "TraceCmd_MIC_ANNOUNCEMENT %d %d", uint8, bool))
void MasterGui::TraceCmd_MIC_ANNOUNCEMENT(const uint8 mic, const bool micStatus)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_MIC_ANNOUNCEMENT"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(ActivateMIC)(mic, micStatus, MIC_HARDKEY);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_FATAL(("Mic Announcement update"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_KDS_UPDATE, "TraceCmd_KDS_UPDATE %d", uint8))
void MasterGui::TraceCmd_KDS_UPDATE(uint8 kdsStatus)
{
   ETG_TRACE_USR4(("TraceCmd_KDS_UPDATE()"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(KDSStatusUpdMsg)(kdsStatus);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("kdsStatus update"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_MIC_STATUS, "TraceCmd_MIC_STATUS %d", bool))
void MasterGui::TraceCmd_MIC_STATUS(const bool micStatus)
{
   ETG_TRACE_USR4(("TraceCmd_MIC_STATUS()"));
   Courier::Message* msg = COURIER_MESSAGE_NEW(MicStatus)(micStatus);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("MicStatus update"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimulateRVC, "TraceCmd_SimulateRVC %d", bool))
void MasterGui::TraceCmd_SimulateRVC(const bool status)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimulateRVC: Status: %d", status));
   if (sm_RVCStatus != status)
   {
      sm_RVCStatus = status;
      Courier::Message* msg = COURIER_MESSAGE_NEW(ActivateRVCSceneMsg)(status);
      if (msg != NULL)
      {
         msg->Post();
      }
   }
}


ETG_I_CMD_DEFINE((TraceCmd_ANTITHEFT_CHANGE, "TraceCmd_ANTITHEFT_CHANGE %d", uint8))
void MasterGui::TraceCmd_ANTITHEFT_CHANGE(const uint8 status)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_HMISTATE_CHANGE: Status: %d", status));
   Courier::Message* msg = COURIER_MESSAGE_NEW(UpdateAntiTheftStatus)(status);
   if (msg != 0 && msg->Post())
   {}
}


ETG_I_CMD_DEFINE((TraceCmd_SimHmiStateChange, "TraceCmd_SimHmiStateChange %d", uint8))
void MasterGui::TraceCmd_SimHmiStateChange(const uint8 state)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimHmiStateChange: State: %d", state));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SimHmiStateChange)(state);
   if (msg != 0 && msg->Post())
   {}
}


ETG_I_CMD_DEFINE((TraceCmd_SYSTEMHEAT_POPUPSTATE, "TraceCmd_SHOW_SYSTEMHEAT_POPUP %d", int))
void MasterGui::TraceCmd_SYSTEMHEAT_POPUPSTATE(const int popupState)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SYSTEMHEAT_POPUPSTATE: PopupState: %d", popupState));
   Courier::Message* msg = COURIER_MESSAGE_NEW(EmergencyShutDownReqMsg)(popupState);
   if (msg != 0 && msg->Post())
   {}
}


void MasterGui::TraceCmd_NotProcessedMsg(const unsigned char* pcu8Data)
{
   if (pcu8Data)
   {
      ETG_TRACE_USR4(("MasterGui: TraceCmd_NotProcessedMsg: 0x%*x", ETG_LIST_LEN(pcu8Data[0]), ETG_LIST_PTR_T8(pcu8Data)));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_onHourlogicpopupContext, "TraceCmd_onHourlogicpopupContext %d ", bool))
void MasterGui::TraceCmd_onHourlogicpopupContext(const bool popupstatus)
{
   if (popupstatus == true)
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Show, Courier::ViewId("Popups#Scenes#Pfo_HourLogicMode"));
      if (msg != 0 && msg->Post())
      {}
   }
   else
   {
      Courier::Message* msg = COURIER_MESSAGE_NEW(::PopupReqMsg)(hmibase::popups::Hide, Courier::ViewId("Popups#Scenes#Pfo_HourLogicMode"));
      if (msg != 0 && msg->Post())
      {}
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimPluginEntityDetailsUpdate, "TraceCmd_SimPluginEntityDetailsUpdate %d %s", bool, ETG_I_STRING))
void MasterGui::TraceCmd_SimPluginEntityDetailsUpdate(const bool status, const ::std::string entityId)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimPluginEntityDetailsUpdate: Status: %d, EntityId: %s", status, entityId.c_str()));
   ::boost::shared_ptr< EventListDataUtility > tEntityDetails(EventListDataUtility::newEventListDataUtility());
   if (NULL != tEntityDetails.get())
   {
      EventDataUtility* tDataUtility = EventDataUtility::newEventDataUtility();
      if (NULL != tDataUtility)
      {
         uint64 tEntityId = 0;
         if (!entityId.empty())
         {
            char* tEnd;
            tEntityId = ::std::strtoull(entityId.c_str(), &tEnd, 16);
         }
         tDataUtility->addEventData(tEntityId);
         tDataUtility->addEventData("");
         tDataUtility->addEventData(status);
         tEntityDetails->addEventListData(tDataUtility);
         Courier::Message* msg = COURIER_MESSAGE_NEW(PluginEntityDetailsUpdateMsg)(tEntityDetails);
         if (msg != NULL)
         {
            msg->Post();
         }
      }
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimHK, "TraceCmd_SimHK %d %d", ETG_I_CENUM(HardKeyCodeEnum), ETG_I_CENUM(hmibase::HardKeyStateEnum)))
void MasterGui::TraceCmd_SimHK(const HardKeyCodeEnum keyCode, const hmibase::HardKeyStateEnum keyState)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimHK: Key: %d, State: %d", ETG_CENUM(HardKeyCodeEnum, keyCode), ETG_CENUM(hmibase::HardKeyStateEnum, keyState)));
   Courier::Message* msg = COURIER_MESSAGE_NEW(HKStatusChangedUpdMsg)(keyCode, keyState, 0);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("Hardkey simulated"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimSubSystemErrorStatus, "TraceCmd_SimSubSystemErrorStatus %d %d %d", ETG_I_CENUM(enSystemCategory), uint8, uint16))
void MasterGui::TraceCmd_SimSubSystemErrorStatus(const enSystemCategory category, const uint8 systemId, const uint16 errorInfo)
{
   /* INFO:
    * PNM Error (CMC) => 0010000000000000 (‭8192‬)
    * HourLogic Error (CMC) => 1001000000000000 (36864)
    * No Error (Subsystems)  => 0001001001000101 (‭4677‬)
    * NANOMsg Error (Subsystems) => 0001001001000110 (‭4678‬)
    * UserOFF Error (Subsystems) => 0010001001000110 (‭‭‭8774‬‬‬)
    * Safemode Error (Subsystems) => 0011001001000110 (‭‭12870‬‬)
    * Voltage Over Warning Error (Subsystems) => 0001000010000101 (‭4229‬)
    * Voltage Under WarningError (Subsystems) => 0001000100000101 (‭4357‬)
    * Critical High Thermal Error (Subsystems ) => 0001011001000101 (‭5701‬)
    */
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimSubSystemErrorStatus: Category: %d, SystemId: %d, ErrorInfo: %d", ETG_CENUM(enSystemCategory, category), systemId, errorInfo));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SimSubSystemErrorStatus)(category, systemId, errorInfo);
   if (msg != NULL)
   {
      msg->Post();
   }
}


ETG_I_CMD_DEFINE((TraceCmd_SimVideoMixInfo, "TraceCmd_SimVideoMixInfo %d %d %d", ETG_I_CENUM(enRegionID), ETG_I_CENUM(enCabinVideoMixIndex), ETG_I_CENUM(enCabinVideoMixChangeReason)))
void MasterGui::TraceCmd_SimVideoMixInfo(const enRegionID regionId, const enCabinVideoMixIndex videoMixIndex, const enCabinVideoMixChangeReason reason)
{
   ETG_TRACE_USR4(("MasterGui: TraceCmd_SimVideoMixInfo: regionId: %d, VideoMixIndex: %d reason: %d", ETG_CENUM(enRegionID, regionId), ETG_CENUM(enCabinVideoMixIndex, videoMixIndex), ETG_CENUM(enCabinVideoMixChangeReason, reason)));
   Courier::Message* msg = COURIER_MESSAGE_NEW(SimVideoMixIndexInfo)(regionId, videoMixIndex, reason);
   if (msg != 0 && msg->Post())
   {
      ETG_TRACE_USR4(("VideoMix Info simulated"));
   }
}


ETG_I_CMD_DEFINE((TraceCmd_setNavigationModeActive, "TraceCmd_setNavigationModeActive %d %d", bool, bool))
void MasterGui::TraceCmd_setNavigationModeActive(const bool carmode, const bool coachmode)
{
   ETG_TRACE_USR4(("TraceCmd_setNavigationModeActive: %d %d", carmode, coachmode));
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(NavigationModeActiveMsg)(carmode, coachmode);
   if (msg != 0)
   {
      msg->Post();
   }
}


} // namespace Core
} // namespace App
