/**
 * @file         : MasterGPIOHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : MasterGPIOHandler is to handle ISR of MIC switch
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "MasterGPIOHandler.h"
#include "hmi_trace_if.h"
#include "AppHmi_MasterMessages.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/MasterGPIOHandler.cpp.trc.h"
#endif

#ifndef WIN32
#define TRIGGER_MUTEX_LOCK(_var_)    pthread_mutex_lock(&_var_);
#define TRIGGER_MUTEX_UNLOCK(_var_)  pthread_mutex_unlock(&_var_);
#endif

/**
 *  Static Variables Initialization
 */

namespace App {
namespace Core {


tBool MasterGPIOHandler::m_bMICState                         = false;
::asf::core::Timer* MasterGPIOHandler::_debouncingTimer      = NULL ;
tBool MasterGPIOHandler::m_lastSentMicState                  = false;
pthread_mutex_t MasterGPIOHandler::_safeLock                 = PTHREAD_MUTEX_INITIALIZER;
OSAL_tGPIODevID MasterGPIOHandler::m_GpioId_MIC              = 0;
OSAL_tIODescriptor MasterGPIOHandler::m_hGpioHandle_MIC      = 0;
} // namespace Core
} // namespace App


/**
 *  Memeber Functions Definition
 */

namespace App {
namespace Core {


MasterGPIOHandler::MasterGPIOHandler()
{
   _debouncingTimer = new ::asf::core::Timer(*this, 750);
}


MasterGPIOHandler::~MasterGPIOHandler()
{
   if (_debouncingTimer != NULL)
   {
      _debouncingTimer->stop();
      delete _debouncingTimer;
      _debouncingTimer = NULL;
   }
}


OSAL_tpfGPIOCallback MasterGPIOHandler::_pfCallbackGpioStateChange_MICSwitch(tVoid* pArg)
{
   tU32 u32OSALError;
   OSAL_trGPIOData gpioData;
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(pArg);
   ETG_TRACE_USR4(("_pfCallbackGpioStateChange_MICSwitch() entered"));
   if ((m_GpioId_MIC != 0) && (m_hGpioHandle_MIC != OSAL_ERROR))
   {
      gpioData.tId = m_GpioId_MIC;
      if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))
      {
         u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_USR4(("_bGetGPIOState_MICSwitch() Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         //apply lock
         TRIGGER_MUTEX_LOCK(_safeLock)
         m_bMICState = gpioData.unData.bState;
         ETG_TRACE_USR4(("MIC Switch State = %u", m_bMICState));
         validateMICState(m_bMICState);
         TRIGGER_MUTEX_UNLOCK(_safeLock)
         //remove lock
      }
   }
   return (0);
}


void MasterGPIOHandler::postCourierMessage(bool state)
{
   ETG_TRACE_USR4(("postCourierMessage = %d", state));
   ::Courier::Message* msg = COURIER_MESSAGE_NEW(ActivateMIC)(REGION_COCKPIT, state, MIC_HARDKEY);
   if (NULL != msg)
   {
      msg->Post();
   }
}


void MasterGPIOHandler::onExpired(asf::core::Timer& timer, boost::shared_ptr<asf::core::TimerPayload> data)
{
   ETG_TRACE_USR4(("MasterGPIOHandler::onExpired, m_lastSentMicState:%u, m_bMICState:%u", m_lastSentMicState, m_bMICState));
   if ((NULL != _debouncingTimer) && ((&timer) == (_debouncingTimer)))
   {
      TRIGGER_MUTEX_LOCK(_safeLock)
      _debouncingTimer->stop();
      if (m_lastSentMicState != m_bMICState)
      {
         postMessageAndStartTimer(m_bMICState);
      }
      TRIGGER_MUTEX_UNLOCK(_safeLock)
   }
}


void MasterGPIOHandler::validateMICState(const bool micState)
{
   ETG_TRACE_USR4(("validateMICState before m_lastSentMicState = %d, mic state = %d ", m_lastSentMicState, micState));
   if (m_lastSentMicState != micState)
   {
      if (_debouncingTimer != NULL && !_debouncingTimer->isActive())
      {
         postMessageAndStartTimer(micState);
      }
   }
   ETG_TRACE_USR4(("validateMICState after m_lastSentMicState = %d, mic state = %d ", m_lastSentMicState, micState));
}


void MasterGPIOHandler::postMessageAndStartTimer(const bool micState)
{
   postCourierMessage(micState);
   m_lastSentMicState = micState;

   ETG_TRACE_USR4(("postMessageAndStartTimer : state = %d", micState));
   _debouncingTimer->start();
}


tBool MasterGPIOHandler::bGetMICSwitchStatus()
{
   return m_bMICState;
}


tVoid MasterGPIOHandler::_vCloseGpio_MICSwitch(tVoid)
{
   ETG_TRACE_USR4(("_vCloseGpio_MICSwitch() entered."));
   OSAL_trGPIOData  pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;
   if ((m_GpioId_MIC != 0) && (m_hGpioHandle_MIC != OSAL_ERROR))
   {
      pinData.tId = m_GpioId_MIC;
      pinData.unData.bState = FALSE;
      (tVoid)OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData));
      gpioCallbackData.rData.unData.pfCallback = NULL;
      gpioCallbackData.pvArg = (tVoid*) this;
      gpioCallbackData.rData.tId = m_GpioId_MIC;
      (tVoid)OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
      (tVoid)OSAL_s32IOClose(m_hGpioHandle_MIC);
      m_hGpioHandle_MIC = OSAL_ERROR;
   }
}


tVoid MasterGPIOHandler::_vConfigureGpio_MICSwitch(tVoid)
{
   ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() entered."));
#ifndef GEN3X86
   tU32 u32OSALError;
   OSAL_trGPIOData pinData;
   OSAL_trGPIOCallbackData  gpioCallbackData;
   m_GpioId_MIC      = (OSAL_tGPIODevID)OSAL_EN_MIC_PRIVATE_SW;
   m_hGpioHandle_MIC = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
   if (m_hGpioHandle_MIC == OSAL_ERROR)
   {
      u32OSALError = OSAL_u32ErrorCode();
      ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() OSAL_IOOpen for dev_GPIO failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
   }
   else
   {
      if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_INPUT, (tS32)m_GpioId_MIC))
      {
         u32OSALError = OSAL_u32ErrorCode();
         ETG_TRACE_USR4(("m_hGpioHandle_MIC() Set GPIO as input failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
      }
      else
      {
         pinData.tId                              = m_GpioId_MIC;
         pinData.unData.u16Edge                   = OSAL_GPIO_EDGE_BOTH;
         gpioCallbackData.pvArg                   = (tVoid*) this;
         gpioCallbackData.rData.tId               = m_GpioId_MIC;
         gpioCallbackData.rData.unData.pfCallback = (OSAL_tpfGPIOCallback)&_pfCallbackGpioStateChange_MICSwitch;
         if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)(&gpioCallbackData)))
         {
            u32OSALError = OSAL_u32ErrorCode();
            ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() callback registration failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
         }
         else if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)(&pinData)))
         {
            u32OSALError = OSAL_u32ErrorCode();
            ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() Set Trigger Edge (Both Edges) failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
            gpioCallbackData.rData.unData.pfCallback = NULL;
            OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
         }
         else
         {
            pinData.unData.bState = TRUE;
            if (OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&pinData)) == OSAL_ERROR)
            {
               u32OSALError = OSAL_u32ErrorCode();
               ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() Enable Interrupt failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               gpioCallbackData.rData.unData.pfCallback = NULL;
               OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&gpioCallbackData);
            }
            else
            {
               ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() : Success"));
               OSAL_trGPIOData gpioData;
               gpioData.tId = m_GpioId_MIC;
               if (OSAL_ERROR == OSAL_s32IOControl(m_hGpioHandle_MIC, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&gpioData)))
               {
                  u32OSALError = OSAL_u32ErrorCode();
                  ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() : Get GPIO State failed. 'Osal Error = %u', %s", u32OSALError, OSAL_coszErrorText(u32OSALError)));
               }
               else
               {
                  TRIGGER_MUTEX_LOCK(_safeLock)
                  if (m_bMICState != gpioData.unData.bState)
                  {
                     m_bMICState = gpioData.unData.bState;
                     validateMICState(m_bMICState);
                     ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() : MIC Switch State = %u", m_bMICState));
                  }
                  TRIGGER_MUTEX_UNLOCK(_safeLock)
               }
            }
         }
      }
   }
#else
   ETG_TRACE_USR4(("_vConfigureGpio_MICSwitch() OSAL_IOOpen is temporarily disabled for dev_GPIO in LSIM because of getting access violations"));
   m_GpioId_MIC      = (OSAL_tGPIODevID)OSAL_EN_MIC_PRIVATE_SW;
   m_hGpioHandle_MIC = OSAL_ERROR;

#endif
}


} // namespace Core
} // namespace App
