/**************************************************************************************
* @file         : ilm_client.cpp
* @author       :
* @addtogroup   : AppHmi_Master
* @brief        :
* @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/

#include "IlmClient.h"
#include <ilm/ilm_client.h>
#include <algorithm>
#include "hmi_trace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/IlmClient.cpp.trc.h"
#endif


IlmClient* _pIlmClient = NULL;


IlmClient::IlmClient()
{
   if (_pIlmClient != NULL)
   {
      ETG_TRACE_FATAL(("IlmClient created more than once in hmi master!!!"));
   }
   init();
   _pIlmClient = this;
}


IlmClient::~IlmClient()
{
   _pIlmClient = NULL;
}


ilmErrorTypes IlmClient::init()
{
   ilmErrorTypes ilmErr = ilm_init();
   ETG_TRACE_USR4(("IlmClient::init, result: %s", ILM_ERROR_STRING(ilmErr)));
   return ilmErr;
}


IlmClient* IlmClient::getInstance()
{
   return _pIlmClient;
}


void IlmClient::getVisibleSurfaceList(std::vector<int>& surfaceVector, const unsigned int displayId, int surfaceState, int surfaceIdent)
{
   ilmScreenProperties screenProperties;

   ilmErrorTypes err = ilm_getPropertiesOfScreen(displayId, &screenProperties);
   if (err == ILM_SUCCESS)
   {
      for (unsigned int lIndex = 0; lIndex < screenProperties.layerCount; lIndex++)
      {
         t_ilm_int surfaceCount = 0;
         t_ilm_surface* pAllSurfacesurfaces;
         unsigned int layerid = screenProperties.layerIds[lIndex];
         // get all surfaces on the layer
         err = ilm_getSurfaceIDsOnLayer(layerid, &surfaceCount, &pAllSurfacesurfaces);

         if (err == ILM_SUCCESS)
         {
            for (int sIndex = 0; sIndex < surfaceCount; sIndex++)
            {
               ilmSurfaceProperties surfaceProperties;
               int surfaceid = pAllSurfacesurfaces[sIndex];
               err = ilm_getPropertiesOfSurface(surfaceid, &surfaceProperties);
               if (err == ILM_SUCCESS)
               {
                  if (surfaceProperties.visibility
                        && ((surfaceState == VISIBLE) ||
                            (surfaceState == VISIBLE_INPUT_KEY && surfaceAcceptsInput(surfaceid, ILM_INPUT_DEVICE_KEYBOARD))))
                  {
                     int surfaceAttr = surfaceid;
                     if (surfaceIdent == CREATOR_PID)
                     {
                        surfaceAttr = surfaceProperties.creatorPid;
                     }
                     std::vector<int>::iterator itr = ::std::find(surfaceVector.begin(), surfaceVector.end(), surfaceAttr);
                     if (itr != surfaceVector.end())
                     {
                        surfaceVector.erase(itr);
                     }
                     surfaceVector.push_back(surfaceAttr);
                  }
               }
            }
         }
         if (pAllSurfacesurfaces != 0)
         {
            free(pAllSurfacesurfaces);
            pAllSurfacesurfaces = 0;
         }
      }
   }
}


void IlmClient::getVisibleSurfaceList(std::vector<int>& surfaceVector, int surfaceState, int surfaceIdent)
{
   getVisibleSurfaceList(surfaceVector, DISPLAY_ID, surfaceState, surfaceIdent);
}


bool IlmClient::surfaceAcceptsInput(t_ilm_surface surfaceId, ilmInputDevice inputDevice)
{
   bool accepts = false;
   ilmSurfaceProperties surfaceProperties;
   if (ilm_getPropertiesOfSurface(surfaceId, &surfaceProperties) == ILM_SUCCESS)
   {
      accepts = (surfaceProperties.inputDevicesAcceptance & inputDevice);
   }
   else
   {
      ETG_TRACE_USR4(("IlmClient::surfaceAcceptsInput for surface %u and inputDevice %.10s, unable to retrieve surface properties", surfaceId, inputDeviceToString(inputDevice).c_str()));
   }
   ETG_TRACE_USR4(("IlmClient::surfaceAcceptsInput for surface %u and inputDevice %.10s, accepts: %s", surfaceId, inputDeviceToString(inputDevice).c_str(), accepts ? "true" : "false"));
   return accepts;
}


std::string IlmClient::inputDeviceToString(ilmInputDevice inputDevice)
{
   std::string ret;
   if (inputDevice & ILM_INPUT_DEVICE_KEYBOARD)
   {
      ret += "keyboard,";
   }
   if (inputDevice & ILM_INPUT_DEVICE_POINTER)
   {
      ret += "pointer,";
   }
   if (inputDevice & ILM_INPUT_DEVICE_TOUCH)
   {
      ret += "touch,";
   }

   // remove trailing comma
   if (!ret.empty())
   {
      ret.resize(ret.size() - 1);
   }

   return ret;
}
