/**
 * @file         : HmiStateHandlerInterface.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : HmiStateHandlerInterface is to handle the interface implementation of
 *                 feature HmiStateHandler
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef HMI_STATE_HANDLER_INTERFACE_H
#define HMI_STATE_HANDLER_INTERFACE_H


#include <map>
#include <asf/core/Types.h>


/**
 *  Typedefs Declaration
 */

/*
 * INFO:
 *   -> Key : System id
 *   -> Param1 : System category
 *   -> Param2 : Error info
 */
typedef ::std::map< uint8, ::std::pair< uint8, uint16 > > SubSystemsErrorStatusInfo;


/**
 *  Classes Forward Declaration
 */

namespace App {
namespace Core {


class MasterHall;
class IPNMGrp10StatusCB;
class IHourLogicExpireStatusCB;
class IHmiStateListenerCB;
class ISubSytemsErrorStatusCB;
}// namespace Core
}// namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class IHmiStateHandler
{
   public:
      /**
       *  Member Function Declaration
       */
      IHmiStateHandler() {}
      virtual ~IHmiStateHandler() {}
      virtual bool isPowerStateON() const = 0;
      virtual uint8 getHmiState() const = 0;
      virtual uint8 getLastHmiState() const = 0;
      virtual bool getPNMGrp10Status() const = 0;
      virtual bool getHourLogicExpireStatus() const = 0;
      virtual void addHmiStateListener(IHmiStateListenerCB* imp) = 0;
      virtual void addPNMGrp10StatusListener(IPNMGrp10StatusCB* imp) = 0;
      virtual void addHourLogicExpireStatusListener(IHourLogicExpireStatusCB* imp) = 0;
      virtual void addSubSytemsErrorStatusListener(ISubSytemsErrorStatusCB* imp) = 0;
      virtual void removeHmiStateListener(IHmiStateListenerCB* imp) = 0;
      virtual void removePNMGrp10StatusListener(IPNMGrp10StatusCB* imp) = 0;
      virtual void removeHourLogicExpireStatusListener(IHourLogicExpireStatusCB* imp) = 0;
      virtual void removeSubSytemsErrorStatusListener(ISubSytemsErrorStatusCB* imp) = 0;
      virtual void sendRestartRequest() = 0;
      virtual void sendHKeyPressedRequest() = 0;
      virtual void sendFirstPicDrawnUpdate() = 0;
      virtual void sendUserOnRequest(const bool status) = 0;
      virtual void sendSubsystemerrorStatusGetRequest() = 0;
      virtual void sendDiagStatus(const bool diagstatus) = 0;
      virtual void sendStandByRequest(const bool status) = 0;
      virtual void onHmiStateStatus(const uint8 hmiState) = 0;
      virtual void updateHMIStateChangeToSM(const uint8 hmiState) = 0;

   private:
      /**
       *  Friend classes Declaration
       */
      friend class MasterHall;
      /**
       *  Member Functions Declaration
       */
      virtual void handleSubSystemsErrorStatus(const SubSystemsErrorStatusInfo& info) = 0;
};


class IPNMGrp10StatusCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IPNMGrp10StatusCB() {}
      virtual ~IPNMGrp10StatusCB() {}
      virtual void onPNMGrp10StatusUpdate(const bool status) = 0;
};


class IHourLogicExpireStatusCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IHourLogicExpireStatusCB() {}
      virtual ~IHourLogicExpireStatusCB() {}
      virtual void onHourLogicExpireStatusUpdate(const bool status) = 0;
};


class IHmiStateListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IHmiStateListenerCB() {}
      virtual ~IHmiStateListenerCB() {}
      virtual void onHmiStateUpdate(const uint8 hmiState) = 0;
};


class ISubSytemsErrorStatusCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      ISubSytemsErrorStatusCB() {}
      virtual ~ISubSytemsErrorStatusCB() {}
      virtual void onSubSystemsErrorStatus(const SubSystemsErrorStatusInfo& info) = 0;
};


}// namespace Core
}// namespace App


#endif /* HMI_STATE_HANDLER_INTERFACE_H */
