/**
 * @file         : HmiStateHandlerHelper.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : HmiStateHandlerHelper is to handle the HmiStateHandler feature
 *                 helper implementation
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "HmiStateHandlerHelper.h"
#include "AppHmi_MasterTypes.h"
#include "spm_core_fi_typesConst.h"
#include "NanoMsgLCMInterfaceConst.h"


using namespace ::spm_core_fi_types;
using namespace ::bosch::cm::di::NanoMsgLCMInterface;


/**
 *  Member Functions (like "C")
 */

namespace App {
namespace Core {


uint8 convertSystemCategoryFromLCM(const uint8 category)
{
   uint8 tCategory = SYSTEM_CATEGORY_NONE;
   switch (category)
   {
      case enDevID__enDevID_Media_Center:
      {
         tCategory = SYSTEM_CATEGORY_CMC;
         break;
      }
      case enDevID__enDevID_Media_Apm:
      {
         tCategory = SYSTEM_CATEGORY_CMA;
         break;
      }
      case enDevID__enDevID_Media_CamPort:
      {
         tCategory = SYSTEM_CATEGORY_CMP;
         break;
      }
      case enDevID__enDevID_Media_Glass:
      {
         tCategory = SYSTEM_CATEGORY_CMG;
         break;
      }
      case enDevID__enDevID_Media_Router:
      {
         tCategory = SYSTEM_CATEGORY_CMR;
         break;
      }
      default:
         break;
   }
   return tCategory;
}


uint8 convert3BitSystemVoltageErrorStatusFromLCM(const uint8 status)
{
   uint8 tStatus = SYSTEM_VOLTAGE_STATUS_NONE;
   switch (status)
   {
      case 0x01:
      {
         tStatus = SYSTEM_VOLTAGE_STATUS_NOERROR;
         break;
      }
      case 0x02:
      {
         tStatus = SYSTEM_VOLTAGE_STATUS_OVER_WARNING_ERROR;
         break;
      }
      case 0x03:
      {
         tStatus = SYSTEM_VOLTAGE_STATUS_OVER_CRITICAL_ERROR;
         break;
      }
      case 0x04:
      {
         tStatus = SYSTEM_VOLTAGE_STATUS_UNDER_WARNING_ERROR;
         break;
      }
      case 0x05:
      {
         tStatus = SYSTEM_VOLTAGE_STATUS_UNDER_CRITICAL_ERROR;
         break;
      }
      default:
         break;
   }
   return tStatus;
}


uint8 convert3BitSystemThermalErrorStatusFromLCM(const uint8 status)
{
   uint8 tStatus = SYSTEM_THERMAL_STATUS_NONE;
   switch (status)
   {
      case 0x01:
      {
         tStatus = SYSTEM_THERMAL_STATUS_NOERROR;
         break;
      }
      case 0x02:
      {
         tStatus = SYSTEM_THERMAL_STATUS_HIGH_ERROR;
         break;
      }
      case 0x03:
      {
         tStatus = SYSTEM_THERMAL_STATUS_CRITICAL_HIGH_ERROR;
         break;
      }
      case 0x04:
      {
         tStatus = SYSTEM_THERMAL_STATUS_OVER_CRITICAL_ERROR;
         break;
      }
      case 0x05:
      {
         tStatus = SYSTEM_THERMAL_STATUS_LOW_ERROR;
         break;
      }
      case 0x06:
      {
         tStatus = SYSTEM_THERMAL_STATUS_CRITICAL_LOW_ERROR;
         break;
      }
      case 0x07:
      {
         tStatus = SYSTEM_THERMAL_STATUS_UNDER_CRITICAL_LOW_ERROR;
         break;
      }
      default:
         break;
   }
   return tStatus;
}


uint8 convert3BitSystemPowerModeStatusFromLCM(const uint8 status)
{
   uint8 tStatus = SYSTEM_STATUS_NONE;
   switch (status)
   {
      case 0x01:
      {
         tStatus = SYSTEM_STATUS_CONNECTED;
         break;
      }
      case 0x02:
      {
         tStatus = SYSTEM_STATUS_USER_OFF_DISCONNECTED;
         break;
      }
      case 0x03:
      {
         tStatus = SYSTEM_STATUS_INACTIVE_DISCONNECTED;
         break;
      }
      case 0x04:
      {
         tStatus = SYSTEM_STATUS_SWUPDATE_DISCONNECTED;
         break;
      }
      case 0x05:
      {
         tStatus = SYSTEM_STATUS_DIAGNOSIS_DISCONNECTED;
         break;
      }
      default:
         break;
   }
   return tStatus;
}


uint8 convert2BitSystemConnectionStatusFromLCM(const uint8 status)
{
   uint8 tStatus = SYSTEM_STATUS_NONE;
   switch (status)
   {
      case 0x01:
      {
         tStatus = SYSTEM_STATUS_CONNECTED;
         break;
      }
      case 0x02:
      {
         tStatus = SYSTEM_STATUS_DISCONNECTED;
         break;
      }
      default:
         break;
   }
   return tStatus;
}


uint8 convertSystemCategoryFromString(const ::std::string& category)
{
   uint8 tCategory = SYSTEM_CATEGORY_NONE;
   if (category == "CMC")
   {
      tCategory = SYSTEM_CATEGORY_CMC;
   }
   else if (category == "CMA")
   {
      tCategory = SYSTEM_CATEGORY_CMA;
   }
   else if (category == "CMP")
   {
      tCategory = SYSTEM_CATEGORY_CMP;
   }
   else if (category == "CMG")
   {
      tCategory = SYSTEM_CATEGORY_CMG;
   }
   else if (category == "CMR")
   {
      tCategory = SYSTEM_CATEGORY_CMR;
   }
   else
   {}
   return tCategory;
}


uint8 getBitValue(const uint16 errorInfo, const uint8 index, const uint8 bits)
{
   return ((errorInfo >> index)  & ((0x01 << bits) - 0x01));
}


uint8 getSystemStatusForSPMState(const uint8 state, const bool pnmGrp10Status)
{
   uint8 tStatus = SYSTEM_STATUS_NONE;
   if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF)
   {
      tStatus = SYSTEM_STATUS_HMI_OFF_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE)
   {
      tStatus = SYSTEM_STATUS_INACTIVE_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY)
   {
      tStatus = SYSTEM_STATUS_USER_OFF_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF)
   {
      tStatus = SYSTEM_STATUS_INFOTAINMENTOFF_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE)
   {
      tStatus = SYSTEM_STATUS_GOODBYE_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER)
   {
      tStatus = SYSTEM_STATUS_COCKPIT_DISCONNECTED;
      if (pnmGrp10Status)
      {
         tStatus = SYSTEM_STATUS_PNM_DISCONNECTED;
      }
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE)
   {
      tStatus = SYSTEM_STATUS_HOURLOGIC_DISCONNECTED;
   }
   else if (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_REMOTE_PROGRAMMING)
   {
      tStatus = SYSTEM_STATUS_SWUPDATE_DISCONNECTED;
   }
   else
   {}
   return tStatus;
}


uint8 getSPMStateForSystemStatus(const uint8 systemState)
{
   uint8 tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI;
   if (systemState == SYSTEM_STATUS_HMI_OFF_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF;
   }
   else if (systemState == SYSTEM_STATUS_INACTIVE_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE;
   }
   else if (systemState == SYSTEM_STATUS_USER_OFF_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY;
   }
   else if (systemState == SYSTEM_STATUS_INFOTAINMENTOFF_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF;
   }
   else if (systemState == SYSTEM_STATUS_GOODBYE_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE;
   }
   else if ((systemState == SYSTEM_STATUS_COCKPIT_DISCONNECTED) || (systemState == SYSTEM_STATUS_PNM_DISCONNECTED))
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER;
   }
   else if (systemState == SYSTEM_STATUS_HOURLOGIC_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE;
   }
   else if (systemState == SYSTEM_STATUS_SWUPDATE_DISCONNECTED)
   {
      tStatus = T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_REMOTE_PROGRAMMING;
   }
   else
   {}
   return tStatus;
}


bool isSPMHMIStateON(const uint8 state)
{
   bool isValid = true;
   switch (state)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_STANDBY:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_GOODBYE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF:
      {
         isValid = false;
         break;
      }
      default:
         break;
   }
   return isValid;
}


bool isSPMHMIStateShowHMI(const uint8 state)
{
   bool isValid = false;
   switch (state)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_OFF_DIAG:
      {
         isValid = true;
         break;
      }
      default:
         break;
   }
   return isValid;
}


bool isSPMHMIStateAnimation(const uint8 state)
{
   bool isValid = (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_ANIMATION) ? true : false;
   return isValid;
}


bool isSPMHMIStateInfotainmentOFF(const uint8 state)
{
   bool isValid = (state == T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF) ? true : false;
   return isValid;
}


bool isLastHmiStateValidForMic1Restoration(uint8 lastHmiState, uint8 pnmGrpStatus)
{
   bool isValid = true;

   switch (lastHmiState)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF:
      {
         isValid = false;
         break;
      }
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER:
      {
         if (!pnmGrpStatus)
         {
            isValid = false;
            break;
         }
      }
      default:
         break;
   }
   return isValid;
}


bool isLastHmiStateValidForMixedSourceRestoration(uint8 lastHmiState)
{
   bool isValid = true;
   switch (lastHmiState)
   {
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_INACTIVE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_INFOTAINMENT_OFF:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_HMI_HOUR_MODE:
      case T_SPM_e32_HMI_STATE__SPM_U32_HMISTATE_SHOW_ON_PARTIAL_USER:
      {
         isValid = false;
         break;
      }
      default:
         break;
   }
   return isValid;
}


}// namespace Core
}// namespace App
