/**
 * @file         : ContextSwitchHandlerInterface.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ContextSwitchHandlerInterface is to handle the ContextSwitchHandler
 *                 feature interface implementation.
 * @copyright    : (c) 2020-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CONTEXT_SWITCH_HANDLER_INTERFACE_H
#define CONTEXT_SWITCH_HANDLER_INTERFACE_H


#include <asf/core/Types.h>
#include "ContextSwitchHandlerConstants.h"


/**
 *  Classes Forward Declaration
 */

namespace App {
namespace Core {


class ApplicationContextInfo;
class ContextSwitchRequestInfo;
class IRegionHandling;
class IHmiStateHandler;
class IRootContextHandler;
class IContextPriorityHandler;
class IActiveContextListenerCB;
class IApplicationSurfaceInfoHandler;
class IApplicationSwitchServerComponent;
} //namespace Core
} //namespace App


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


/**
 * IContextSwitchHandler is used to
 *  - Provide interface impl between Users and ContextSwitchHandler.
 */
class IContextSwitchHandler
{
   public:
      /**
       *  Member Functions Declaration
       */
      IContextSwitchHandler() {}
      virtual ~IContextSwitchHandler() {}
      virtual uint32 getActiveApplicationId() const = 0;
      virtual bool fetchActiveContextInfo(ApplicationContextInfo& info) const = 0;
      virtual bool fetchActiveContextInfo(const uint32 appId, ApplicationContextInfo& info) const = 0;
      virtual bool fetchDefaultContextInfoForRegion(const uint32 regionId, ApplicationContextInfo& info) const = 0;
      virtual void addActiveContextListener(IActiveContextListenerCB& imp) = 0;
      virtual void removeActiveContextListener(IActiveContextListenerCB& imp) = 0;
      virtual void setRegionHandling(IRegionHandling& imp) = 0;
      virtual void setHMIStateHandler(IHmiStateHandler& imp) = 0;
      virtual void setRootContextHandler(IRootContextHandler& imp) = 0;
      virtual void setContextPriorityHandler(IContextPriorityHandler& imp) = 0;
      virtual void setDefaultApplicationsContextInfo(const ApplicationsContextInfo& info) = 0;
      virtual void setApplicationSurfaceInfoHandler(IApplicationSurfaceInfoHandler& imp) = 0;
      virtual void setApplicationSwitchServerComponent(IApplicationSwitchServerComponent& imp) = 0;
      virtual void dumpSwitchStackInfo() = 0;
      virtual void onApplicationSwitchComplete(const uint32 appId) = 0;
      virtual void onStoreContext(const ApplicationContextInfo& info) = 0;
      virtual void onContextSwitchRequest(const ContextSwitchRequestInfo& info) = 0;
      virtual void onSwitchLastApplication(const bool lastApplicationDebounceTimerStatus = false) = 0;
      virtual void onExecuteContextSwitchRequest(const ContextSwitchRequestInfo& info, const uint32 code) = 0;
      virtual void onClearLastApplication() = 0;
};


/**
 * IActiveContextListenerCB is used to
 *  - Provide interface to communicate the active context update to listeners.
 */
class IActiveContextListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      IActiveContextListenerCB() {}
      virtual ~IActiveContextListenerCB() {}
      virtual void onActiveContextUpdate(const ApplicationContextInfo& info) = 0;
};


} //namespace Core
} //namespace App


#endif /* CONTEXT_SWITCH_HANDLER_INTERFACE_H */
