/**
 * @file         : MonoGlassController.h
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : Controls connection/disconnection of cabin sources to glass sink
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#ifndef MONOGLASSCONTROLLER_H
#define MONOGLASSCONTROLLER_H


#include <asf/core/Types.h>
#include "MonoGlassControllerInterface.h"


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class MonoGlassController: public IMonoGlassController
{
   public:
      /**
       *  Member Functions Declaration
       */
      MonoGlassController();
      virtual ~MonoGlassController();
      bool onSourceConnect(CabinConnectionHandlerInfo& info);
      bool onSourceDisconnect(CabinConnectionHandlerInfo& info);
      bool fetchAVRestoreInfo(const int16 sinkId, HB_AUDIO::stSourceInfo& info);
      void onCMAStatusUpdate(const SystemStatusInfo& info);
      void onConnectionRemoved(CabinConnectionHandlerInfo& info);
      void onNewConnectionUpdate(CabinConnectionHandlerInfo& info);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      void onMapOutActivation(CabinConnectionHandlerInfo& info);
      void onMapOutDeactivation(CabinConnectionHandlerInfo& info);
#endif

      inline void setCabinConnectionUtility(CabinConnectionUtility& imp)
      {
         _cabinConnectionUtility = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

      inline void updateMonoModeStatus(const uint32 regionId, const bool status)
      {
         _monoModeInfo[regionId] = status;
      }

   private:
      /**
       *  Member Functions Declaration
       */
      SinkIdsInfo getSupportedGlassSinksInfo(const uint32 regionId, const bool checkSystems = false) const;
      int16 getAmpSinkIdForRegionId(const uint32 regionId) const;
      uint32 getRegionIdForSinkId(const int16 sinkId) const;
      bool isGlassSinkSupported(const int16 sinkId) const;
      bool onAVSourceRequest(CabinConnectionHandlerInfo& info);
      bool onCameraSourceRequest(CabinConnectionHandlerInfo& info);
      bool onAvPausedConnectGlass(CabinConnectionHandlerInfo& info);
      bool onVideoSourceDisconnect(CabinConnectionHandlerInfo& info);
      bool onAmplifierSourceRequest(CabinConnectionHandlerInfo& info);
      void removeSinkFromAVRestoreInfo(const int16 sinkId);
      void onNewConnectionUpdateForAmp(CabinConnectionHandlerInfo& info);
      void onNewConnectionUpdateForGlass(CabinConnectionHandlerInfo& info);
      void restoreAVSource(const uint32 regionId);

      inline bool isMonoModeActive(const uint32 regionId) const
      {
         bool isValid = false;
         MonoModeInfo::const_iterator itr = _monoModeInfo.find(regionId);
         if ((itr == _monoModeInfo.end()) || (itr->second))
         {
            isValid = true;
         }
         return isValid;
      }

      /**
       *  Member Variables Declaration
       */
      MonoModeInfo _monoModeInfo;
      SinkConnectionMap _AVRestoreInfo;
      CabinConnectionUtility* _cabinConnectionUtility;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
};


} //namespace Core
} //namespace App


#endif //MONOGLASSCONTROLLER_H
