/**
 * @file         : CabinRestorationHandlerTypes.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinRestorationHandlerTypes is to handle the CabinRestorationHandler
 *                 feature types implementation
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CABIN_RESTORATIONHANDLER_TYPES_H
#define CABIN_RESTORATIONHANDLER_TYPES_H


#include "AppHmi_MasterTypes.h"
#include "CabinRestorationHandlerConstants.h"


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class RestoreSinkInfo
{
   public:
      /**
       *  Member Function Declaration
       */
      RestoreSinkInfo()
      {
         clear();
      }

      virtual ~RestoreSinkInfo()
      {
         clear();
      }

      RestoreSinkInfo(const RestoreSinkInfo& obj)
      {
         clear();
         *this = obj;
      }

      RestoreSinkInfo& operator = (const RestoreSinkInfo& obj)
      {
         if (this != &obj)
         {
            clear();
            _sinkId          = obj._sinkId;
            _sourceId        = obj._sourceId;
            _lastSourceId    = obj._lastSourceId;
            _state           = obj._state;
            _lastState       = obj._lastState;
            _subSourceId     = obj._subSourceId;
            _lastSubSourceId = obj._lastSubSourceId;
            _status          = obj._status;
            _systemsStatus   = obj._systemsStatus;
         }
         return *this;
      }

      void clear()
      {
         _sinkId          = -1;
         _sourceId        = -1;
         _lastSourceId    = -1;
         _state           = 0;
         _lastState       = 0;
         _subSourceId     = 0;
         _lastSubSourceId = 0;
         _status          = CABIN_RESTORE_STATUS_NONE;
         _systemsStatus.clear();
      }

      inline void setSinkId(const int16 sinkId)
      {
         _sinkId = sinkId;
      }

      inline int16 getSinkId() const
      {
         return _sinkId;
      }

      inline void setSourceId(const int16 sourceId)
      {
         _sourceId = sourceId;
      }

      inline int16 getSourceId() const
      {
         return _sourceId;
      }

      inline void setLastSourceId(const int16 lastSourceId)
      {
         _lastSourceId = lastSourceId;
      }

      inline int16 getLastSourceId() const
      {
         return _lastSourceId;
      }

      inline void setState(const uint8 state)
      {
         _state = state;
      }

      inline uint8 getState() const
      {
         return _state;
      }

      inline void setLastState(const uint8 lastState)
      {
         _lastState = lastState;
      }

      inline uint8 getLastState() const
      {
         return _lastState;
      }

      inline void setSubSourceId(const uint32 subSourceId)
      {
         _subSourceId = subSourceId;
      }

      inline uint32 getSubSourceId() const
      {
         return _subSourceId;
      }

      inline void setLastSubSourceId(const uint32 subSourceId)
      {
         _lastSubSourceId = subSourceId;
      }

      inline uint32 getLastSubSourceId() const
      {
         return _lastSubSourceId;
      }

      inline void setRestoreStatus(const uint8 status)
      {
         _status = status;
      }

      inline uint8 getRestoreStatus() const
      {
         return _status;
      }

      inline void setSystemsStatusInfo(const SystemsStatusInfo& info)
      {
         _systemsStatus = info;
      }

      inline const SystemsStatusInfo& getSystemsStatusInfo() const
      {
         return _systemsStatus;
      }

      inline SystemsStatusInfo& getMutableSystemsStatusInfo()
      {
         return _systemsStatus;
      }

      bool isAllSystemsReady() const
      {
         bool isValid = true;
         for (SystemsStatusInfo::const_iterator itr = _systemsStatus.begin(); (itr != _systemsStatus.end()); ++itr)
         {
            if (!(*itr).second)
            {
               isValid = false;
               break;
            }
         }
         return isValid;
      }

      bool setSystemStatus(const uint8 systemId, const bool status)
      {
         bool isModified = false;
         for (SystemsStatusInfo::iterator itr = _systemsStatus.begin(); (itr != _systemsStatus.end()); ++itr)
         {
            if ((*itr).first == systemId)
            {
               isModified    = true;
               (*itr).second = status;
               break;
            }
         }
         return isModified;
      }

      void addToSystemsStatusInfo(const uint8 systemId)
      {
         SystemsStatusInfo::const_iterator itr = _systemsStatus.begin();
         for (; (itr != _systemsStatus.end()); ++itr)
         {
            if ((*itr).first == systemId)
            {
               break;
            }
         }
         if (itr == _systemsStatus.end())
         {
            _systemsStatus.push_back(::std::pair < uint8, bool >(systemId, false));
         }
      }

      void removeFromSystemsStatusInfo(const uint8 systemId)
      {
         for (SystemsStatusInfo::iterator itr = _systemsStatus.begin(); (itr != _systemsStatus.end()); ++itr)
         {
            if ((*itr).first == systemId)
            {
               _systemsStatus.erase(itr);
               break;
            }
         }
      }

   private:
      /**
       *  Member Variable Declaration
       */
      int16 _sinkId;
      int16 _sourceId;
      int16 _lastSourceId;
      uint8 _state;
      uint8 _lastState;
      uint32 _subSourceId;
      uint32 _lastSubSourceId;
      uint8 _status;
      SystemsStatusInfo _systemsStatus;
};


class ActivityRestoreSinksInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      ActivityRestoreSinksInfo()
      {
         clear();
      }

      virtual ~ActivityRestoreSinksInfo()
      {
         clear();
      }

      ActivityRestoreSinksInfo(const ActivityRestoreSinksInfo& obj)
      {
         clear();
         *this = obj;
      }

      ActivityRestoreSinksInfo& operator = (const ActivityRestoreSinksInfo& obj)
      {
         if (this != (&obj))
         {
            clear();
            _activityType          = obj._activityType;
            _restoreSinksInfoGroup = obj._restoreSinksInfoGroup;
         }
         return (*this);
      }

      void clear()
      {
         _activityType = CABIN_ACTIVITY_NONE;
         _restoreSinksInfoGroup.clear();
      }

      inline void setActivityType(const uint32 activityType)
      {
         _activityType = activityType;
      }

      inline uint32 getActivityType() const
      {
         return _activityType;
      }

      void addToRestoreSinksInfoGroup(const RestoreSinkInfo& info, const uint8 group)
      {
         RestoreSinksInfoGroup::iterator itr = _restoreSinksInfoGroup.find(group);
         if (itr != _restoreSinksInfoGroup.end())
         {
            if (NULL == getMutableRestoreSinkInfo(info.getSinkId(), group))
            {
               (*itr).second.push_back(info);
            }
         }
         else
         {
            RestoreSinksInfo tInfo;
            tInfo.push_back(info);
            _restoreSinksInfoGroup.insert(::std::make_pair(group, tInfo));
         }
      }

      void removeFromRestoreSinksInfoGroup(const uint8 group)
      {
         RestoreSinksInfoGroup::iterator itr = _restoreSinksInfoGroup.find(group);
         if (itr != _restoreSinksInfoGroup.end())
         {
            _restoreSinksInfoGroup.erase(itr);
         }
      }

      /*
       * INFO:
       *    -> SINKGROUP_NONE denotes iterate through all groups until the sink id is found
       *    -> A sink can belong to only one SinkGroup
       */
      void removeRestoreSinkInfo(const uint32 sinkId, const uint8 group = SINKGROUP_NONE)
      {
         bool isRemoved = false;
         for (RestoreSinksInfoGroup::iterator itr = _restoreSinksInfoGroup.begin(); (itr != _restoreSinksInfoGroup.end()) ; ++itr)
         {
            if ((group == SINKGROUP_NONE) || ((*itr).first == group))
            {
               for (RestoreSinksInfo::iterator titr = (*itr).second.begin(); (titr != (*itr).second.end()); ++titr)
               {
                  if ((*titr).getSinkId() == sinkId)
                  {
                     isRemoved = true;
                     (void)((*itr).second.erase(titr));
                     break;
                  }
               }
               if ((group != SINKGROUP_NONE) || (isRemoved))
               {
                  break;
               }
            }
         }
         cleanRestoreSinksInfoGroup();
      }

      void cleanRestoreSinksInfoGroup()
      {
         const RestoreSinksInfoGroup dupRestoreSinksInfoGroup = _restoreSinksInfoGroup;
         for (RestoreSinksInfoGroup::const_iterator itr = dupRestoreSinksInfoGroup.begin(); (itr != dupRestoreSinksInfoGroup.end()); ++itr)
         {
            if ((*itr).second.empty())
            {
               _restoreSinksInfoGroup.erase((*itr).first);
            }
         }
      }

      inline const RestoreSinksInfoGroup& getRestoreSinksInfoGroup() const
      {
         return _restoreSinksInfoGroup;
      }

      inline RestoreSinksInfoGroup& getMutableRestoreSinksInfoGroup()
      {
         return _restoreSinksInfoGroup;
      }

      const RestoreSinksInfo* getRestoreSinksInfoForGroup(const uint8 group) const
      {
         const RestoreSinksInfo* imp = NULL;
         for (RestoreSinksInfoGroup::const_iterator itr = _restoreSinksInfoGroup.begin(); (itr != _restoreSinksInfoGroup.end()); ++itr)
         {
            if ((*itr).first == group)
            {
               imp = &((*itr).second);
               break;
            }
         }
         return imp;
      }

      RestoreSinksInfo* getMutableRestoreSinksInfoForGroup(const uint8 group)
      {
         return const_cast< RestoreSinksInfo* >(getRestoreSinksInfoForGroup(group));
      }

      const RestoreSinkInfo* getRestoreSinkInfo(const uint32 sinkId, const uint8 group = SINKGROUP_NONE) const
      {
         const RestoreSinkInfo* imp = NULL;
         bool terminate = false;
         for (RestoreSinksInfoGroup::const_iterator itr = _restoreSinksInfoGroup.begin(); ((itr != _restoreSinksInfoGroup.end()) && (!terminate)); ++itr)
         {
            if ((group == SINKGROUP_NONE) || ((*itr).first == group))
            {
               for (RestoreSinksInfo::const_iterator titr = (*itr).second.begin(); (titr != (*itr).second.end()); ++titr)
               {
                  if ((*titr).getSinkId() == sinkId)
                  {
                     imp = (&(*titr));
                     break;
                  }
               }
               terminate = ((group != SINKGROUP_NONE) || (imp != NULL)) ? true : false;
            }
         }
         return imp;
      }

      RestoreSinkInfo* getMutableRestoreSinkInfo(const uint32 sinkId, const uint8 group = SINKGROUP_NONE)
      {
         return const_cast< RestoreSinkInfo* >(getRestoreSinkInfo(sinkId, group));
      }

      bool setSinkRestoreStatus(const uint32 sinkId, const uint8 restoreStatus, const uint8 group = SINKGROUP_NONE)
      {
         bool retVal = false;
         bool terminate = false;
         for (RestoreSinksInfoGroup::iterator itr = _restoreSinksInfoGroup.begin(); ((itr != _restoreSinksInfoGroup.end()) && (!terminate)); ++itr)
         {
            if ((group == (*itr).first) || (group == SINKGROUP_NONE))
            {
               for (RestoreSinksInfo::iterator titr = (*itr).second.begin(); (titr != (*itr).second.end()); ++titr)
               {
                  if ((*titr).getSinkId() == sinkId)
                  {
                     (*titr).setRestoreStatus(restoreStatus);
                     retVal = true;
                     break;
                  }
               }
               terminate = (group != SINKGROUP_NONE) ? true : false;
            }
         }
         return retVal;
      }

      uint8 getSinkRestoreStatus(const uint32 sinkId, const uint8 group = SINKGROUP_NONE)
      {
         uint8 restoreStatus = CABIN_RESTORE_STATUS_NONE;
         bool terminate = false;
         for (RestoreSinksInfoGroup::const_iterator itr = _restoreSinksInfoGroup.begin(); ((itr != _restoreSinksInfoGroup.end()) && (!terminate)); ++itr)
         {
            if ((group == (*itr).first) || (group == SINKGROUP_NONE))
            {
               for (RestoreSinksInfo::const_iterator titr = (*itr).second.begin(); (titr != (*itr).second.end()); ++titr)
               {
                  if ((*titr).getSinkId() == sinkId)
                  {
                     restoreStatus = (*titr).getRestoreStatus();
                     break;
                  }
               }
               terminate = ((group != SINKGROUP_NONE) || (restoreStatus != CABIN_RESTORE_STATUS_NONE)) ? true : false;
            }
         }
         return restoreStatus;
      }

      bool isAllSystemsReadyForGroup(const uint8 group) const
      {
         bool isValid = true;
         for (RestoreSinksInfoGroup::const_iterator itr = _restoreSinksInfoGroup.begin(); (itr != _restoreSinksInfoGroup.end()); ++itr)
         {
            if ((*itr).first == group)
            {
               for (RestoreSinksInfo::const_iterator tItr = (*itr).second.begin(); (tItr != (*itr).second.end()); ++tItr)
               {
                  if (!(*tItr).isAllSystemsReady())
                  {
                     isValid = false;
                     break;
                  }
               }
               break;
            }
         }
         return isValid;
      }

   private:
      /**
       *  Member Variable Declaration
       */
      uint32 _activityType;
      RestoreSinksInfoGroup _restoreSinksInfoGroup;
};


class RestoreRegionInfo
{
   public:
      /**
       *  Member Functions Declaration
       */
      RestoreRegionInfo()
      {
         clear();
      }

      virtual ~RestoreRegionInfo()
      {
         clear();
      }

      RestoreRegionInfo(const RestoreRegionInfo& obj)
      {
         clear();
         *this = obj;
      }

      RestoreRegionInfo& operator = (const RestoreRegionInfo& obj)
      {
         if (this != (&obj))
         {
            clear();
            _regionId                   = obj._regionId;
            _activitiesRestoreSinksInfo = obj._activitiesRestoreSinksInfo;
         }
         return (*this);
      }

      void clear()
      {
         _regionId = REGION_INVALID;
         _activitiesRestoreSinksInfo.clear();
      }

      inline void setRegionId(const uint32 regionId)
      {
         _regionId = regionId;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setActivitiesRestoreSinksInfo(const ActivitiesRestoreSinksInfo& info)
      {
         _activitiesRestoreSinksInfo = info;
      }

      inline const ActivitiesRestoreSinksInfo& getActivitiesRestoreSinksInfo() const
      {
         return _activitiesRestoreSinksInfo;
      }

      inline ActivitiesRestoreSinksInfo& getMutableActivitiesRestoreSinksInfo()
      {
         return _activitiesRestoreSinksInfo;
      }

      const RestoreSinkInfo* getRestoreSinkInfo(const int16 sinkId, const uint8 group = SINKGROUP_NONE, const uint8 activityType = CABIN_ACTIVITY_NONE) const
      {
         const RestoreSinkInfo* imp = NULL;
         for (ActivitiesRestoreSinksInfo::const_iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            if (((*itr).getActivityType() == activityType) || (CABIN_ACTIVITY_NONE == activityType))
            {
               imp = (*itr).getRestoreSinkInfo(sinkId, group);
               if ((activityType != CABIN_ACTIVITY_NONE) || (imp != NULL))
               {
                  break;
               }
            }
         }
         return imp;
      }

      RestoreSinkInfo* getMutableRestoreSinkInfo(const int16 sinkId, const uint8 group = SINKGROUP_NONE, const uint8 activityType = CABIN_ACTIVITY_NONE)
      {
         RestoreSinkInfo* imp = const_cast< RestoreSinkInfo* >(getRestoreSinkInfo(sinkId, group, activityType));
         return imp;
      }

      uint8 getSinkRestoreStatus(const int16 sinkId, const uint8 group = SINKGROUP_NONE)
      {
         uint8 restoreStatus = CABIN_RESTORE_STATUS_NONE;
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            restoreStatus = (*itr).getSinkRestoreStatus(sinkId, group);
            if (restoreStatus != CABIN_RESTORE_STATUS_NONE)
            {
               break;
            }
         }
         return restoreStatus;
      }

      void setSinkRestoreStatus(const int16 sinkId, const uint8 restoreStatus, const uint8 group = SINKGROUP_NONE)
      {
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            if ((*itr).setSinkRestoreStatus(sinkId, restoreStatus, group))
            {
               break;
            }
         }
      }

      bool setSystemStatus(const uint8 systemId, const bool status)
      {
         bool isModified = false;
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            for (RestoreSinksInfoGroup::iterator titr = (*itr).getMutableRestoreSinksInfoGroup().begin(); (titr != (*itr).getMutableRestoreSinksInfoGroup().end()); ++titr)
            {
               for (RestoreSinksInfo::iterator sinkItr = (*titr).second.begin(); (sinkItr != (*titr).second.end()); ++sinkItr)
               {
                  isModified = (((*sinkItr).setSystemStatus(systemId, status)) || (isModified)) ? true : false;
               }
            }
         }
         return isModified;
      }

      void removeRestoreSinksInfoGroup(const uint8 group, const uint8 activityType = CABIN_ACTIVITY_NONE)
      {
         bool terminate = false;
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); ((itr != _activitiesRestoreSinksInfo.end()) && (!terminate)); ++itr)
         {
            if (((*itr).getActivityType() == activityType) || (activityType == CABIN_ACTIVITY_NONE))
            {
               (*itr).removeFromRestoreSinksInfoGroup(group);
               terminate = (activityType != CABIN_ACTIVITY_NONE) ? true : false;
            }
         }
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end());)
         {
            if ((*itr).getRestoreSinksInfoGroup().empty())
            {
               itr = _activitiesRestoreSinksInfo.erase(itr);
            }
            else
            {
               ++itr;
            }
         }
      }

      void removeRestoreSinkInfo(const int16 sinkId, const uint8 group = SINKGROUP_NONE, const uint8 activityType = CABIN_ACTIVITY_NONE)
      {
         bool terminate = false;
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); ((itr != _activitiesRestoreSinksInfo.end()) && (!terminate)); ++itr)
         {
            if (((*itr).getActivityType() == activityType) || (activityType == CABIN_ACTIVITY_NONE))
            {
               (*itr).removeRestoreSinkInfo(sinkId, group);
               terminate = (activityType != CABIN_ACTIVITY_NONE) ? true : false;
            }
         }
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end());)
         {
            if ((*itr).getRestoreSinksInfoGroup().empty())
            {
               itr = _activitiesRestoreSinksInfo.erase(itr);
            }
            else
            {
               ++itr;
            }
         }
      }

      bool abortRestoreForSink(const uint8 sinkId, const uint8 group = SINKGROUP_NONE, const uint8 activityType = CABIN_ACTIVITY_NONE)
      {
         bool isModified = false;
         uint8 restoreStatus = getSinkRestoreStatus(sinkId) ;
         if ((restoreStatus != CABIN_RESTORE_STATUS_NONE) && (restoreStatus != CABIN_RESTORE_STATUS_RESTORE))
         {
            isModified = true;
            const RestoreSinkInfo* imp = getRestoreSinkInfo(sinkId, SINKGROUP_CABIN, CABIN_ACTIVITY_AV);
            if (NULL != imp)
            {
               removeRestoreSinksInfoGroup(SINKGROUP_CABIN, CABIN_ACTIVITY_AV);
            }
            else
            {
               removeRestoreSinkInfo(sinkId, group, activityType);
            }
         }
         return isModified;
      }

      void addActivity(const uint8 activityType)
      {
         ActivitiesRestoreSinksInfo::const_iterator itr = _activitiesRestoreSinksInfo.begin();
         for (; (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            if ((*itr).getActivityType() == activityType)
            {
               break;
            }
         }
         if (itr == _activitiesRestoreSinksInfo.end())
         {
            ActivityRestoreSinksInfo info;
            info.setActivityType(activityType);
            _activitiesRestoreSinksInfo.push_back(info);
         }
      }

      void removeActivity(const uint8 activityType)
      {
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            if ((*itr).getActivityType() == activityType)
            {
               _activitiesRestoreSinksInfo.erase(itr);
               break;
            }
         }
      }

      void addToActivity(const uint8 activityType, const RestoreSinkInfo& info, const uint8 group)
      {
         addActivity(activityType);
         for (ActivitiesRestoreSinksInfo::iterator itr = _activitiesRestoreSinksInfo.begin(); (itr != _activitiesRestoreSinksInfo.end()); ++itr)
         {
            if ((*itr).getActivityType() == activityType)
            {
               (*itr).addToRestoreSinksInfoGroup(info, group);
               break;
            }
         }
      }

   private:
      /**
       *  Member Variables Declaration
       */
      uint32 _regionId;
      ActivitiesRestoreSinksInfo _activitiesRestoreSinksInfo;
};


} //namespace Core
} //namespace App


#endif /* CABIN_RESTORATIONHANDLER_TYPES_H */
