/**
 * @file         : CabinRestorationHandler.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinRestorationHandler is to handle the cabin activity restoration
                   for Cabin Source
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CABIN_RESTORATIONHANDLER_HEADER
#define CABIN_RESTORATIONHANDLER_HEADER

#include "CabinConnectionHandlerInterface.h"
#include "CabinRestorationHandlerInterface.h"
#include "CabinsActivityInfoDpHandlerInterface.h"
#include <Core/HmiState/HmiStateHandlerInterface.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/SyncDisplay/SyncDisplayHandlerInterface.h>

/**
 *  Forward Declaration
 */
class IHmiInfoServiceServerComponent;

/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class CabinRestorationHandler : public ICabinRestorationHandler
   , public IHmiStateListenerCB
   , public ICMAStatusListenerCB
   , public ICMPStatusListenerCB
   , public ICMGStatusListenerCB
   , public ICMRStatusListenerCB
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
   , public IMIC1ConnectionStatus
#endif
   , public IHmiFirstAppListenerCB
{
   public:
      /**
       *  Member Functions Declaration
       */
      CabinRestorationHandler();
      virtual ~CabinRestorationHandler();
      void onHmiFirstAppRequest();
      void onHmiStateUpdate(const uint8 currentHmiState);
      void onCMAStatusUpdate(const SystemStatusInfo& info);
      void onCMPStatusUpdate(const SystemStatusInfo& info);
      void onCMGStatusUpdate(const SystemStatusInfo& info);
      void onCMRStatusUpdate(const SystemStatusInfo& info);
      void onUserInteraction(const uint8 regionId, const uint32 sinkId);
      void onUserInteraction(const uint8 regionId, const SinkIdsInfo sinkIds);
      uint8 getSinkRestoreStatus(const uint8 sinkId, const uint8 regionId, const uint8 group = SINKGROUP_NONE);
      void finalize();
      void initialize();

      inline void setRegionHandling(::App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setHMIStateHandler(IHmiStateHandler& imp)
      {
         _hmiStateHandler = (&imp);
      }

      inline void setSyncDisplayHandler(ISyncDisplayHandler& imp)
      {
         _syncDisplayHandler = (&imp);
      }

      inline void setCabinConnectionHandler(ICabinConnectionHandler& imp)
      {
         _cabinConnectionHandler = (&imp);
      }

      inline void setCabinVideoMixInfoDpHandler(ICabinVideoMixInfoDpHandler& imp)
      {
         _cabinVideoMixInfoDpHandler = (&imp);
      }

      inline void setCabinsActivityInfoDpHandler(ICabinsActivityInfoDpHandler& imp)
      {
         _cabinsActivityInfoDpHandler = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      inline void currentMIC1ConnectionStatus(const bool /* status */)
      {
         evalAndUpdateWaitStatusForMic1();
      }
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
   private:
      /**
       *  Typedefs Declaration
       */
      typedef void(CabinRestorationHandler::*SinkLoaderFunction)(const CabinActivitiesInfo&, RestoreRegionInfo&, uint8);
      typedef ::std::vector< SinkLoaderFunction > SinkLoaderFunctionsInfo;
      typedef ::std::map< uint8, SinkLoaderFunctionsInfo > ActivitiesSinkLoaderFunctionsInfo;

      typedef bool(CabinRestorationHandler::*ValidatorFunction)(const uint8);
      typedef ::std::vector< ::std::pair < ValidatorFunction, bool > > ValidatorFunctionsInfo;
      typedef ::std::map< uint8, ValidatorFunctionsInfo > ActivitiesValidatorFunctionsInfo;

      /**
       *  Member Functions Declaration
       */
      CabinRestorationHandler(const CabinRestorationHandler& obj);
      CabinRestorationHandler& operator= (const CabinRestorationHandler& obj);
      void initRestoreRegionsInfo();
      void initActivitiesSinkLoaderFunctionsInfo();
      void initActivitiesValidatorFunctionsInfo();
      void sendCabinsRestorationInfo();
      void evalAndUpdateRestoreStatus();
#ifdef VARIANT_S_FTR_ENABLE_MIC_ACTIVATION
      void evalAndUpdateWaitStatusForMic1();
#endif /* VARIANT_S_FTR_ENABLE_MIC_ACTIVATION */
      void updateCabinVideoMixInfoProperty();
      void loadAmpSinkToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType);
      void loadGlassSinkToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType);
      void loadAmpSourceToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType);
      void loadGlassSourceToRestoreRegionInfo(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType);
      void loadAmpSystemToRestoreRegionInfoGlassSink(const CabinActivitiesInfo& activitiesInfo, RestoreRegionInfo& info, uint8 activityType);
      void evalAndUpdateSystemStatus(const SystemStatusInfo& info, bool systemStatus, uint32 regionId = REGION_INVALID);
      std::vector< uint8 > getActivitiesInfo(const uint8 regionId, const CabinActivitiesInfo& info);
      bool isLastSystemStatusPNMDisconnected(const uint8 regionId);
      bool isLastSystemStatusCockpitModeDisconnected(const uint8 regionId);
      bool isActivityValidToRestore(const uint8 regionId, const uint8 activityType);
      bool evalAndUpdateSinkRestoreStatus(const uint8 status, const int16 sinkId = -1, const uint8 group = SINKGROUP_NONE);
      uint8 getActivityTypeForVideoMixIndex(const uint8 videoMixIndex);
      /**
       *  Member Variables Declaration
       */
      IRegionHandling* _regionHandling;
      IHmiStateHandler* _hmiStateHandler;
      RestoreRegionsInfo _restoreRegionsInfo;
      ISyncDisplayHandler* _syncDisplayHandler;
      ICabinConnectionHandler* _cabinConnectionHandler;
      ICabinVideoMixInfoDpHandler* _cabinVideoMixInfoDpHandler;
      ICabinsActivityInfoDpHandler* _cabinsActivityInfoDpHandler;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
      ActivitiesSinkLoaderFunctionsInfo _activitiesSinkLoaderFunctionsInfo;
      ActivitiesValidatorFunctionsInfo _activitiesValidatorFunctionsInfo;
};


} //namespace Core
} //namespace App


#endif /* CABIN_RESTORATIONHANDLER_HEADER */
