/**
 * @file         : CabinConnectionUtility.h
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinConnectionUtility provides helper functions required for
 *                 cabin connections
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CABINCONNECTIONUTILITY_H
#define CABINCONNECTIONUTILITY_H


#include <vector>
#include <asf/core/Types.h>
#include <boost/shared_ptr.hpp>
#include <Core/RegionHandling/RegionHandlingConstants.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>

/**
 *  Classes Forward Declaration
 */

class IConnectionControllerPrj;
class IHmiInfoServiceServerComponent;

namespace App {
namespace Core {


class IRegionHandling;
class VariantHandling;
class ICabinRestorationHandler;
class CabinConnectionHandlerInfo;
class IApplicationSwitchServerComponent;
} //namespace Core
} //namespace App


/**
 *  Classes Declaration
 */


namespace App {
namespace Core {


class CabinConnectionUtility
{
   public:
      /**
       *  Member Functions Declaration
       */
      CabinConnectionUtility();
      virtual ~CabinConnectionUtility();
      SinkIdsInfo getSinkIdsInfo(const RegionsIdInfo& info) const;
      SinkIdsInfo getGlassSinksInfo(const uint32 regionId, const uint8 groupCategory) const;
      int16 getAmpSinkIdForRegionId(const uint32 regionId) const;
      uint32 getRegionIdForSinkId(const int16 sinkId) const;
      bool isAmpSink(const int16 sinkid) const;
      bool isGlassSink(const int16 sinkid) const;
      bool isGlassSystemsValid(const int16 sinkId) const;
      bool isConnectionAvailable(const CabinConnectionHandlerInfo& info) const;
      bool isGlassSinkGroupCategorySame(const int16 sinkId, const uint8 category) const;
      void setSysMuteStatusForMic1(const bool status, const uint32 regionId);
      bool sendSourceRequest(const int16 srcId, const int16 devId, const int16 sinkId);
      void sendSourceDisconnect(const int16 srcId, const int16 devId, const int16 sinkId);
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      void sendMapOutRequestToNavi(const uint32 regionId, const bool status);
      void updateSinksMapOutStatusAndStream(const uint32 regionId, const std::vector< int16 >& info, const bool status);
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */

      inline void setRegionHandling(App::Core::IRegionHandling& imp)
      {
         _regionHandling = (&imp);
      }

      inline void setVariantHandling(VariantHandling& imp)
      {
         _variantHandling = (&imp);
      }

      inline void setCabinRestorationHandler(ICabinRestorationHandler& imp)
      {
         _cabinRestorationHandler = (&imp);
      }

      inline void setHmiInfoServiceServerComponent(IHmiInfoServiceServerComponent& imp)
      {
         _hmiInfoServiceServerComponent = (&imp);
      }

      inline void setConnectionControllerPrj(const ::boost::shared_ptr< IConnectionControllerPrj >& imp)
      {
         _connectionControllerPrj = imp;
      }

      inline void setApplicationSwitchServerComponent(IApplicationSwitchServerComponent& imp)
      {
         _applicationSwitchServerComponent = (&imp);
      }

      inline bool isValidStateToDisconnect(const uint8 state) const
      {
         bool isValid = false;
         switch (state)
         {
            case STATE_CONNECTED:
            case STATE_SUSPENDED:
            {
               isValid = true;
               break;
            }
            default:
               break;
         }
         return isValid;
      }

   private:
      /**
       *  Member Functions Declaration
       */
      uint16 getPresentationTimeForSrc(const uint8 type, const int16 sinkId, const int16 srcId);
      bool isValidSourceForRestorationAbort(const int16 srcId) const;
      void sendLipSyncCommandToPlugin(const int16 srcId, const int16 sinkId);
      /**
       *  Member Variables Declaration
       */
      IRegionHandling* _regionHandling;
      VariantHandling* _variantHandling;
      ICabinRestorationHandler* _cabinRestorationHandler;
      IHmiInfoServiceServerComponent* _hmiInfoServiceServerComponent;
      IApplicationSwitchServerComponent* _applicationSwitchServerComponent;
      ::boost::shared_ptr< IConnectionControllerPrj > _connectionControllerPrj;
};


} //namespace Core
} //namespace App


#endif //MONOGLASSCONTROLLER_H
