/**
 * @file         : CabinConnectionUtility.h
 * @author       : INF4CV - pgi5cob
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinConnectionUtility provides helper functions required for
 *                 cabin connections
 * @copyright    : (C) 2016 Robert Bosch GmbH
 *                 (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "CabinConnectionUtility.h"
#include "CabinSrcInfo.h"
#include "hmi_trace_if.h"
#include "EventDataUtility.h"
#include "ProjectPluginMsgs.h"
#include "CabinConnectionHandlerTypes.h"
#include "CabinConnectionHandlerDefines.h"
#include "CabinRestorationHandlerInterface.h"
#include <PluginConstants/PluginConstants.h>
#include <Core/VariantHandling/VariantHandling.h>
#include <Core/AudioInterface/ConnectionControllerPrj.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/RegionHandling/RegionHandlingConstants.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponentHandlingInterface.h>
#include <Core/ApplicationSwitchServerComponent/ApplicationSwitchServerComponent.h>


using namespace hmibase::apphmi_master::audio;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/CabinConnectionUtility.cpp.trc.h"
#endif


namespace App {
namespace Core {


CabinConnectionUtility::CabinConnectionUtility() : _regionHandling(NULL)
   , _variantHandling(NULL)
   , _cabinRestorationHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
   , _applicationSwitchServerComponent(NULL)
{
   ETG_TRACE_USR4(("CabinConnectionUtility: CTOR"));
}


CabinConnectionUtility::~CabinConnectionUtility()
{
   ETG_TRACE_USR4(("CabinConnectionUtility: DTOR"));
}


bool CabinConnectionUtility::isAmpSink(const int16 sinkid) const
{
   bool isValid = ((NULL != _regionHandling) && (_regionHandling->getSystemCategoryForSinkId(sinkid) == SYSTEM_CATEGORY_CMA)) ? true : false;
   return isValid;
}


bool CabinConnectionUtility::isGlassSink(const int16 sinkid) const
{
   bool isValid = ((NULL != _regionHandling) && (_regionHandling->getSystemCategoryForSinkId(sinkid) == SYSTEM_CATEGORY_CMG)) ? true : false;
   return isValid;
}


bool CabinConnectionUtility::isGlassSystemsValid(const int16 sinkId) const
{
   SystemEnquiryInfo tSystemEnquiryInfo(getRegionIdForSinkId(sinkId), SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM);
   bool isValid = _regionHandling->isAnyStatusSame(tSystemEnquiryInfo, SYSTEM_STATUS_CONNECTED, sinkId);
   return isValid;
}


bool CabinConnectionUtility::isGlassSinkGroupCategorySame(const int16 sinkId, const uint8 category) const
{
   bool isValid = (NULL != _regionHandling) ? true : false;
   isValid      = ((isValid) && (isGlassSink(sinkId))) ? true : false;
   if ((isValid) && (category != SYSTEM_GROUP_CATEGORY_NONE))
   {
      isValid = _regionHandling->isSinkGroupCategorySame(sinkId, category);
   }
   return isValid;
}


bool CabinConnectionUtility::sendSourceRequest(const int16 srcId, const int16 devId, const int16 sinkId)
{
   bool retVal = false;
   if (NULL != _connectionControllerPrj.get())
   {
      retVal = _connectionControllerPrj->sendCabinSourceRequest(srcId, devId, sinkId);
      ETG_TRACE_USR4(("CabinConnectionUtility: sendSourceRequest: SourceId: %d, DeviceId : %d, SinkId : %d, Status: %d", srcId, devId, sinkId, retVal))
      if (retVal)
      {
         sendLipSyncCommandToPlugin(srcId, sinkId);
         if ((_cabinRestorationHandler != NULL) && (isValidSourceForRestorationAbort(srcId) == true))
         {
            _cabinRestorationHandler->onUserInteraction(getRegionIdForSinkId(sinkId), sinkId);
         }
      }
   }
   return retVal;
}


bool CabinConnectionUtility::isValidSourceForRestorationAbort(const int16 srcId) const
{
   bool retVal = true;
   switch (srcId)
   {
      case SRC_MIC1_ANNOUNCEMENT:
      case SRC_MIC2_ANNOUNCEMENT:
      case SRC_MIC3_ANNOUNCEMENT:
      {
         retVal = false;
         break;
      }
      default:
         break;
   }
   return retVal;
}


void CabinConnectionUtility::sendSourceDisconnect(const int16 srcId, const int16 devId, const int16 sinkId)
{
   ETG_TRACE_USR4(("CabinConnectionUtility: sendSourceDisconnect: SourceId: %d, DeviceId : %d, SinkId : %d", srcId, devId, sinkId));
   if (NULL != _connectionControllerPrj.get())
   {
      _connectionControllerPrj->sendCabinSourceDisconnect(srcId, devId, sinkId);
   }
}


void CabinConnectionUtility::sendLipSyncCommandToPlugin(const int16 srcId, const int16 sinkId)
{
   CabinSrcInfo::SrcType newSrcType = CabinSrcInfo::getSrcType(srcId);
   switch (newSrcType)
   {
      case CabinSrcInfo::CABIN_VIDEO_SOURCE:
      case CabinSrcInfo::CABIN_AV_SOURCE:
      case CabinSrcInfo::CABIN_AUDIO_SOURCE:
      {
         uint8 PluginSourceIndex = CabinSrcInfo::getPluginSourceIndex(srcId);
         boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
         if (NULL != pluginData.get())
         {
            pluginData->addEventData(PluginSourceIndex);
            pluginData->addEventData(static_cast<uint8>(sinkId));
            pluginData->addEventData(getPresentationTimeForSrc(static_cast<uint8>(newSrcType), sinkId, srcId));
         }
         POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_MASTER, CTRBLOCK_NAME_PRESENTATION_TIME_OFFSET, pluginData)));
         break;
      }
      default:
         break;
   }
}


uint16 CabinConnectionUtility::getPresentationTimeForSrc(const uint8 type, const int16 sinkId, const int16 srcId)
{
   uint16 presentationTime = PRESENTATIONTIME_DEFAULT;
   if (NULL != _variantHandling)
   {
      switch (type)
      {
         case CabinSrcInfo::CABIN_VIDEO_SOURCE:
         {
            if (true == CabinSrcInfo::isIPCameraSource(srcId))
            {
               presentationTime = _variantHandling->getPresentationTimeIPCamera();
            }
            else
            {
               presentationTime = _variantHandling->getPresentationTimeAnalogueCamera();
            }
            break;
         }
         case CabinSrcInfo::CABIN_AV_SOURCE:
         {
            if (true == isAmpSink(sinkId))
            {
               if (true == CabinSrcInfo::isCamportHDMI(srcId))
               {
                  presentationTime = _variantHandling->getPresentationTimeAmplifier_HDMI();
               }
               else if (true == CabinSrcInfo::isCamportUSB(srcId))
               {
                  presentationTime = _variantHandling->getPresentationTimeAmplifier_USBVideo();
               }
               else if (true == CabinSrcInfo::isCMRVideoSource(srcId))
               {
                  presentationTime = _variantHandling->getPresentationTimeAmp_DRM();
               }
               else
               {}
            }
            else
            {
               if (true == CabinSrcInfo::isCMRVideoSource(srcId))
               {
                  presentationTime = _variantHandling->getPresentationTimeGlass_DRM();
               }
               else
               {
                  presentationTime = _variantHandling->getPresentationTimeAVSource();
               }
            }
            break;
         }
         case CabinSrcInfo::CABIN_AUDIO_SOURCE:
         {
            presentationTime = _variantHandling->getPresentationTimeAmplifier_Audio();
         }
         default:
            break;
      }
   }
   ETG_TRACE_USR4(("CabinConnectionUtility: getPresentationTimeForSrc: SourceType: %d, SinkId: %d, PresentationTime: %d", type, sinkId, presentationTime));
   return presentationTime;
}


#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING

void CabinConnectionUtility::updateSinksMapOutStatusAndStream(const uint32 regionId, const std::vector< int16 >& info, const bool status)
{
   if ((NULL != _hmiInfoServiceServerComponent) && (!info.empty()))
   {
      uint8 streamType = (status) ? AUDIOCONTROL_RTP_STREAM : AUDIOCONTROL_AVB_STREAM;
      for (std::vector< int16 >::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         _hmiInfoServiceServerComponent->addOrUpdateRegionsMapOutStatusInfo(regionId, (*itr), status);
         ETG_TRACE_USR4(("CabinConnectionUtility: updateSinksMapOutStatusAndStream: SinkId: %d, StreamType: %d", (*itr), streamType));
         boost::shared_ptr<EventDataUtility> pluginData(EventDataUtility::newEventDataUtility());
         if (NULL != pluginData.get())
         {
            pluginData->addEventData(static_cast<uint8>(*itr));
            pluginData->addEventData(streamType);
            POST_MSG_NOTRACE((COURIER_MESSAGE_NEW(PluginCommandReqMsg)(PLUGIN_NAME_MASTER, CTRLBLOCK_NAME_MASTER_STREAM_SELECT, pluginData)));
         }
      }
      _hmiInfoServiceServerComponent->sendRegionsMapOutStatusInfo();
   }
}


void CabinConnectionUtility::sendMapOutRequestToNavi(const uint32 regionId, const bool status)
{
   if (NULL != _applicationSwitchServerComponent)
   {
      ETG_TRACE_USR4(("CabinConnectionUtility: sendMapOutRequestToNavi: RegionId: %d, Status: %d", regionId, status));
      _applicationSwitchServerComponent->sendMapOutSignal(regionId, status);
   }
}


#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */


bool CabinConnectionUtility::isConnectionAvailable(const CabinConnectionHandlerInfo& info) const
{
   bool isValid = false;
   if (info.isActiveSinksInfoValid())
   {
      const stSourceInfo& tSourceInfo       = info.getSourceInfo();
      SinkConnectionMap::const_iterator itr = info.getActiveSinksInfo()->find(tSourceInfo.sinkId);
      if (itr != info.getActiveSinksInfo()->end())
      {
         isValid = (tSourceInfo.srcId == itr->second.srcId) ? true : false;
         isValid = ((isValid) && (tSourceInfo.deviceId == itr->second.deviceId)) ? true : false;
         isValid = ((isValid) && ((itr->second.connectionState == STATE_CONNECTED) || \
                                  (itr->second.connectionState == STATE_SUSPENDED))) ? true : false;
      }
   }
   return isValid;
}


SinkIdsInfo CabinConnectionUtility::getSinkIdsInfo(const RegionsIdInfo& info) const
{
   SinkIdsInfo tInfo;
   if (NULL != _regionHandling)
   {
      for (RegionsIdInfo::const_iterator itr = info.begin(); (itr != info.end()); ++itr)
      {
         (void)_regionHandling->fetchSinkIdsInfo(tInfo, SYSTEM_CATEGORY_NONE, (*itr));
      }
   }
   return tInfo;
}


SinkIdsInfo CabinConnectionUtility::getGlassSinksInfo(const uint32 regionId, const uint8 groupCategory) const
{
   SinkIdsInfo tInfo;
   if (NULL != _regionHandling)
   {
      (void)_regionHandling->fetchSinkIdsInfoForGroupCategory(tInfo, groupCategory, SYSTEM_CATEGORY_CMG, regionId);
   }
   return tInfo;
}


int16 CabinConnectionUtility::getAmpSinkIdForRegionId(const uint32 regionId) const
{
   int16 sinkId = SINK_INVALID;
   if (NULL != _regionHandling)
   {
      SinkIdsInfo tSinkIdsInfo;
      (void)_regionHandling->fetchSinkIdsInfo(tSinkIdsInfo, SYSTEM_CATEGORY_CMA, regionId);
      //INFO: Only one Amp sinkId will be available for a region so first available Amp sinkId is copied.
      if (!tSinkIdsInfo.empty())
      {
         sinkId = tSinkIdsInfo[0];
      }
   }
   return sinkId;
}


uint32 CabinConnectionUtility::getRegionIdForSinkId(const int16 sinkId) const
{
   uint32 regionId = REGION_INVALID;
   if (NULL != _regionHandling)
   {
      (void)_regionHandling->fetchRegionIdForSinkId(sinkId, regionId);
   }
   return regionId;
}


} //namespace Core
} //namespace App
