/**
 * @file         : CabinConnectionHandlerTypes.h
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : CabinConnectionHandlerTypes is to handle the CabinConnectionHandler
 *                 feature types implementation
 * @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#ifndef CABINCONNECTIONHANDLER_TYPES_H
#define CABINCONNECTIONHANDLER_TYPES_H


#include "CabinConnectionHandlerDefines.h"
#include "CabinConnectionHandlerConstants.h"
#include <Core/RegionHandling/RegionHandlingConstants.h>


/**
 *  Classes Declaration
 */

namespace App {
namespace Core {


class CabinConnectionHandlerInfo
{
   public:
      /**
       *  Member Function Declaration
       */
      CabinConnectionHandlerInfo()
      {
         clear();
      }

      virtual ~CabinConnectionHandlerInfo()
      {
         clear();
      }

      CabinConnectionHandlerInfo(const CabinConnectionHandlerInfo& obj)
      {
         clear();
         *this = obj;
      }

      CabinConnectionHandlerInfo& operator = (const CabinConnectionHandlerInfo& obj)
      {
         if (this != &obj)
         {
            _srcReqQueue     = obj._srcReqQueue;
            _connReqQueue    = obj._connReqQueue;
            _srcInfo         = obj._srcInfo;
            _activeSinksInfo = obj._activeSinksInfo;
            _modifiedSinks   = obj._modifiedSinks;
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
            _mapOutInfo      = obj._mapOutInfo;
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
         }
         return *this;
      }

      void clear()
      {
         _modifiedSinks.clear();
         _srcReqQueue     = NULL;
         _connReqQueue    = NULL;
         _activeSinksInfo = NULL;
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
         _mapOutInfo      = NULL;
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
      }

      inline const App::Core::SinkIdsInfo& getModifiedSinksInfo() const
      {
         return _modifiedSinks;
      }

      inline App::Core::SinkIdsInfo& getMutableModifiedSinksInfo()
      {
         return _modifiedSinks;
      }

      inline void setSourceInfo(const HB_AUDIO::stSourceInfo& info)
      {
         _srcInfo = info;
      }

      inline const HB_AUDIO::stSourceInfo& getSourceInfo() const
      {
         return _srcInfo;
      }

      inline HB_AUDIO::stSourceInfo& getMutableSourceInfo()
      {
         return _srcInfo;
      }

      inline void setSourceRequestsQueueInfo(RequestsQueueInfo& info)
      {
         _srcReqQueue = &info;
      }

      inline const RequestsQueueInfo* getSourceRequestsQueueInfo() const
      {
         return _srcReqQueue;
      }

      inline void setConnectionRequestsQueueInfo(RequestsQueueInfo& info)
      {
         _connReqQueue = &info;
      }

      inline const RequestsQueueInfo* getConnectionRequestsQueueInfo() const
      {
         return _connReqQueue;
      }

      inline RequestsQueueInfo* getMutableConnectionRequestsQueueInfo()
      {
         return _connReqQueue;
      }

      inline void setActiveSinksInfo(const SinkConnectionMap& info)
      {
         _activeSinksInfo = &info;
      }

      inline const SinkConnectionMap* getActiveSinksInfo() const
      {
         return _activeSinksInfo;
      }

#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      inline void setRegionMapOutInfo(RegionMapOutInfo& info)
      {
         _mapOutInfo = &info;
      }

      inline const RegionMapOutInfo* getRegionMapOutInfo() const
      {
         return _mapOutInfo;
      }

      inline RegionMapOutInfo* getMutableRegionMapOutInfo()
      {
         return _mapOutInfo;
      }
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */

      bool isActiveSinksInfoValid() const
      {
         bool isValid = (NULL != _activeSinksInfo) ? true : false;
         return isValid;
      }

      bool isSourceRequestsQueueInfoValid() const
      {
         bool isValid = (NULL != _srcReqQueue) ? true : false;
         return isValid;
      }

      bool isConnectionRequestsQueueInfoValid() const
      {
         bool isValid = (NULL != _connReqQueue) ? true : false;
         return isValid;
      }

      void removeFromSourceRequestsQueue(const int16 sinkId)
      {
         if (NULL != _srcReqQueue)
         {
            for (RequestsQueueInfo::iterator itr = _srcReqQueue->begin(); (itr != _srcReqQueue->end()); ++itr)
            {
               if ((*itr).sinkId == sinkId)
               {
                  (void)_srcReqQueue->erase(itr);
                  break;
               }
            }
         }
      }

      void removeFromConnectionRequestsQueue(const HB_AUDIO::stSourceInfo& info)
      {
         if (NULL != _connReqQueue)
         {
            for (RequestsQueueInfo::iterator itr = _connReqQueue->begin(); (itr != _connReqQueue->end()); ++itr)
            {
               if (((*itr).srcId == info.srcId) && ((*itr).connectionState == info.connectionState) && ((*itr).sinkId == info.sinkId))
               {
                  (void)_connReqQueue->erase(itr);
                  break;
               }
            }
         }
      }

      void addToSourceRequestsQueue(const HB_AUDIO::stSourceInfo& info)
      {
         if (NULL != _srcReqQueue)
         {
            removeFromSourceRequestsQueue(info.sinkId);
            _srcReqQueue->push_back(info);
         }
      }

      void addToConnectionRequestsQueue(const HB_AUDIO::stSourceInfo& info)
      {
         if (NULL != _connReqQueue)
         {
            _connReqQueue->push_back(info);
         }
      }

      void addToModifiedMapSinks(const uint16& sinkId)
      {
         if (::std::find(_modifiedSinks.begin(), _modifiedSinks.end(), sinkId) == _modifiedSinks.end())
         {
            _modifiedSinks.push_back(sinkId);
         }
      }

      bool isModifiedSinksInfoValid()
      {
         bool retVal = false;
         if (!_modifiedSinks.empty())
         {
            retVal = true;
         }
         return retVal;
      }

   private:
      /**
       *  Member Variable Declaration
       */
      App::Core::SinkIdsInfo _modifiedSinks;
      HB_AUDIO::stSourceInfo _srcInfo;
      RequestsQueueInfo* _srcReqQueue;
      RequestsQueueInfo* _connReqQueue;
      const SinkConnectionMap* _activeSinksInfo;
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      RegionMapOutInfo* _mapOutInfo;
#endif /* VARIANT_S_FTR_ENABLE_MAP_STREAMING */
};


class SinkMapOutInfo
{
   public:
      /**
       *  Member Function Declaration
       */
      SinkMapOutInfo()
      {
         clear();
      }

      virtual ~SinkMapOutInfo()
      {
         clear();
      }

      SinkMapOutInfo(const SinkMapOutInfo& obj)
      {
         clear();
         *this = obj;
      }

      SinkMapOutInfo(const int16 sinkId)
      {
         clear();
         _sinkId = sinkId;
      }

      SinkMapOutInfo& operator = (const SinkMapOutInfo& obj)
      {
         if (this != &obj)
         {
            _sinkId = obj._sinkId;
            _state  = obj._state;
         }
         return *this;
      }

      void clear()
      {
         _sinkId = -1;
         _state  = STATE_UNKNOWN;
      }

      inline void setSinkId(const int16 sinkId)
      {
         _sinkId = sinkId;
      }

      inline int16 getSinkId() const
      {
         return _sinkId;
      }

      inline void setState(const uint8 state)
      {
         _state = state;
      }

      inline uint8 getState() const
      {
         return _state;
      }

   private:
      /**
       *  Member Variables Declaration
       */
      int16 _sinkId;
      uint8 _state;
};


class RegionMapOutInfo
{
   public:
      /**
       *  Member Function Declaration
       */
      RegionMapOutInfo()
      {
         clear();
      }

      virtual ~RegionMapOutInfo()
      {
         clear();
      }

      RegionMapOutInfo(const RegionMapOutInfo& obj)
      {
         clear();
         *this = obj;
      }

      RegionMapOutInfo(const uint32 regionId)
      {
         clear();
         _regionId = regionId;
      }

      RegionMapOutInfo& operator = (const RegionMapOutInfo& obj)
      {
         if (this != &obj)
         {
            _state     = obj._state;
            _regionId  = obj._regionId;
            _sinksInfo = obj._sinksInfo;
         }
         return *this;
      }

      void clear()
      {
         _state    = STATE_UNKNOWN;
         _regionId = REGION_INVALID;
         _sinksInfo.clear();
      }

      inline void setState(const uint8 state)
      {
         _state = state;
      }

      inline uint8 getState() const
      {
         return _state;
      }

      inline void setRegionId(const uint32 regionId)
      {
         _regionId = regionId;
      }

      inline uint32 getRegionId() const
      {
         return _regionId;
      }

      inline void setSinksMapOutInfo(const SinksMapOutInfo& info)
      {
         _sinksInfo = info;
      }

      inline const SinksMapOutInfo& getSinksMapOutInfo() const
      {
         return _sinksInfo;
      }

      inline SinksMapOutInfo& getMutableSinksMapOutInfo()
      {
         return _sinksInfo;
      }

      inline const SinkMapOutInfo* getSinkMapOutInfo(const int16 sinkId) const
      {
         const SinkMapOutInfo* imp = NULL;
         for (SinksMapOutInfo::const_iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end()); ++itr)
         {
            if ((*itr).getSinkId() == sinkId)
            {
               imp = (&(*itr));
               break;
            }
         }
         return imp;
      }

      inline SinkMapOutInfo* getMutableSinkMapOutInfo(const int16 sinkId)
      {
         SinkMapOutInfo* imp = const_cast< SinkMapOutInfo* >(getSinkMapOutInfo(sinkId));
         return imp;
      }

      inline bool isSinksMapOutInfoValid() const
      {
         bool isValid = (!_sinksInfo.empty()) ? true : false;
         return isValid;
      }

      inline bool isAllSinksStateSame(const uint8 state) const
      {
         bool isValid = false;
         if (!_sinksInfo.empty())
         {
            isValid = true;
            for (SinksMapOutInfo::const_iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end()); ++itr)
            {
               if ((*itr).getState() != state)
               {
                  isValid = false;
                  break;
               }
            }
         }
         return isValid;
      }

      inline bool isSinkStateSame(const int16 sinkId, const uint8 state) const
      {
         bool isValid = false;
         for (SinksMapOutInfo::const_iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end()); ++itr)
         {
            if ((*itr).getSinkId() == sinkId)
            {
               isValid = ((*itr).getState() == state) ? true : false;
               break;
            }
         }
         return isValid;
      }

      inline void addToSinksMapOutInfo(const int16 sinkId)
      {
         if (NULL == getSinkMapOutInfo(sinkId))
         {
            _sinksInfo.push_back(SinkMapOutInfo(sinkId));
         }
      }

      inline void removeFromSinksMapOutInfo(const int16 sinkId)
      {
         for (SinksMapOutInfo::iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end()); ++itr)
         {
            if ((*itr).getSinkId() == sinkId)
            {
               (void)_sinksInfo.erase(itr);
               break;
            }
         }
      }

      inline void removeSinkMapOutInfoForState(const uint8 state)
      {
         for (SinksMapOutInfo::iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end());)
         {
            if ((*itr).getState() == state)
            {
               itr = _sinksInfo.erase(itr);
            }
            else
            {
               ++itr;
            }
         }
      }

      inline void updateSinkState(const uint8 state, const int16 sinkId)
      {
         for (SinksMapOutInfo::iterator itr = _sinksInfo.begin(); (itr != _sinksInfo.end()); ++itr)
         {
            if ((*itr).getSinkId() == sinkId)
            {
               (*itr).setState(state);
               if (sinkId != -1)
               {
                  break;
               }
            }
         }
      }

   private:
      /**
       *  Member Variable Declaration
       */
      uint8 _state;
      uint32 _regionId;
      SinksMapOutInfo _sinksInfo;
};


} //namespace Core
} //namespace App


#endif /* CABINCONNECTIONHANDLER_TYPES_H */
