/**
 * @file         : ConnectionControllerPrj.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ConnectionControllerPrj is to handle the audio connections
 * @copyright    : (c) 2018-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */

#include "hall_std_if.h"
#include "AudioInterfaceHelper.h"
#include "AppHmi_MasterMessages.h"
#include "ConnectionControllerPrj.h"
#include <App/DataModel/DataModel.h>
#include <AppHmi_MasterBase/AudioInterface/Connection.h>
#include <AppHmi_MasterBase/AudioInterface/ResourceController.h>
#include <AppHmi_MasterBase/AudioInterface/AudioControllerObjectManager.h>
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
#include <Core/CabinConnectionHandler/CabinSrcInfo.h>
#include <Core/CabinConnectionHandler/CabinConnectionHandlerInterface.h>
#endif


using namespace ::App::Core;
using namespace ::App::datamodel;
using namespace ::hmibase::apphmi_master::audio;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/ConnectionControllerPrj.cpp.trc.h"
#endif


ConnectionControllerPrj::ConnectionControllerPrj() : m_PrivateSpeakerSinkId(COCKPIT_SINK_ID)
   , m_ActiveAppId(APPID_APPHMI_UNKNOWN)
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   , _cabinConnectionHandler(NULL)
#endif
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: CTOR"));
   ReadPersistentApplication();
}


void ConnectionControllerPrj::vRegisterforConnectionInfoUpdate(IConnectionInfoUpdate* client)
{
   ::std::vector< IConnectionInfoUpdate* >::const_iterator itr = find(_connectionInfoUpdateCB.begin(), _connectionInfoUpdateCB.end(), client);
   if (itr == _connectionInfoUpdateCB.end())
   {
      _connectionInfoUpdateCB.push_back(client);
      if (client)
      {
         client->onUpdateConnectionInfo(m_ConnectionMap);
      }
   }
}


void ConnectionControllerPrj::vUnRegisterforConnectionInfoUpdate(IConnectionInfoUpdate* client)
{
   ::std::vector< IConnectionInfoUpdate* >::iterator itr = find(_connectionInfoUpdateCB.begin(), _connectionInfoUpdateCB.end(), client);
   if (itr != _connectionInfoUpdateCB.end())
   {
      _connectionInfoUpdateCB.erase(itr);
   }
}


void ConnectionControllerPrj::sendConnectionInfoUpdate()
{
   for (std::vector<IConnectionInfoUpdate*>::iterator itr = _connectionInfoUpdateCB.begin(); itr != _connectionInfoUpdateCB.end(); ++itr)
   {
      if ((*itr) != NULL)
      {
         (*itr)->onUpdateConnectionInfo(m_ConnectionMap);
      }
   }
}


void ConnectionControllerPrj::sendLastApplicationRequest()
{
   sendApplicationStart(APPID_APPHMI_MASTER);
}


void ConnectionControllerPrj::sendSourceActivationMsgRequest(int nSrcIndex)
{}


void ConnectionControllerPrj::sendApplicationStart(enApplicationId nAppId)
{
   if (ConnectionController::m_bStartupAvailable)
   {
      ::Courier::Message* msg = COURIER_MESSAGE_NEW(SwitchApplicationMsg)((Courier::Int32)(nAppId));
      if (msg != 0)
      {
         msg->Post();
      }
   }
   else
   {
      ETG_TRACE_USR4(("ConnectionControllerPrj: sendApplicationStart: No startup available"));
   }
}


int16 ConnectionControllerPrj::getCurrentActiveAnnouncementSource(const int16 sinkId)
{
   Sink* pSink                             = NULL;
   int16 sourceIndex                       = SRC_INVALID;
   ResourceController* pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>().get();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(sinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (pSink != NULL)
   {
      for (std::map<int, Connection*>::iterator connIter = m_ConnectionMap.begin(); connIter != m_ConnectionMap.end(); ++connIter)
      {
         Connection* connection = connIter->second;
         if (NULL != connection)
         {
            Sink* poSink     = connection->getSink();
            Source* poSource = connection->getSource();
            if ((poSink != NULL) && (poSource != NULL))
            {
               bool iscurrentActiveAnnouncementSource = (pSink->GetID() == poSink->GetID()) ? true : false;
               iscurrentActiveAnnouncementSource      = ((iscurrentActiveAnnouncementSource) && (static_cast<int>(SOURCE_ANNOUNCEMENT) == poSource->GetType())) ? true : false;
               iscurrentActiveAnnouncementSource      = ((iscurrentActiveAnnouncementSource) && ((connection->getState() == STATE_CONNECTED) || (connection->getState() == STATE_CONNECTING))) ? true : false;
               if (iscurrentActiveAnnouncementSource)
               {
                  sourceIndex = poSource->GetIndex();
                  ETG_TRACE_USR4(("ConnectionControllerPrj: getCurrentActiveAnnouncementSource: Sink: %d, Source: %d, SourceIndex: %d", poSink->GetID(), poSource->GetType(), sourceIndex));
                  break;
               }
            }
         }
      }
   }
   return sourceIndex;
}


int16 ConnectionControllerPrj::getActiveSourceIdOnSink(const int16 sinkId)
{
   Sink* pSink                             = NULL;
   int16 sourceIndex                       = SRC_INVALID;
   ResourceController* pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>().get();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(sinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (pSink != NULL)
   {
      for (std::map<int, Connection*>::iterator connIter = m_ConnectionMap.begin(); connIter != m_ConnectionMap.end(); ++connIter)
      {
         Connection* connection = connIter->second;
         if (NULL != connection)
         {
            Sink*    poSink   = connection->getSink();
            Source*  poSource = connection->getSource();
            if ((poSink != NULL) && (poSource != NULL))
            {
               if ((pSink->GetID() == poSink->GetID()) && (static_cast<int>(SOURCE_MIX) != poSource->GetType()) && (connection->getState() == STATE_CONNECTED))
               {
                  sourceIndex = poSource->GetIndex();
                  ETG_TRACE_USR4(("ConnectionControllerPrj: getActiveSourceIdOnSink: Sink: %d, Source: %d, SourceIndex: %d", poSink->GetID(), poSource->GetType(), sourceIndex));
                  break;
               }
            }
         }
      }
   }
   return sourceIndex;
}


int16 ConnectionControllerPrj::getPrivateSpeakerSinkId()
{
   return m_PrivateSpeakerSinkId;
}


int16 ConnectionControllerPrj::getCurrentActiveSource(const int16 sinkId)
{
   Sink* pSink                             = NULL;
   int16 sourceIndex                       = SRC_INVALID;
   ResourceController* pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>().get();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(sinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (pSink != NULL)
   {
      Connection* conn = getActiveConnection(pSink->GetID());
      if ((conn != NULL) && (conn->getSource() != NULL))
      {
         sourceIndex = conn->getSource()->GetIndex();
      }
   }
   return sourceIndex;
}


bool ConnectionControllerPrj::fetchCurrentActiveSource(const int16 sinkId, int& srcIndex, int& srcDeviceId)
{
   Sink* pSink                             = NULL;
   bool isFetched                          = false;
   ResourceController* pResourceController = AudioControllerObjectManager::getInstance().getResourceController<ResourceController>().get();
   if (pResourceController != NULL)
   {
      pResourceController->mapSinkIdx(sinkId);
      pSink = pResourceController->GetSinkOnIdx();
   }
   if (pSink != NULL)
   {
      Connection* conn = getActiveConnection(pSink->GetID());
      if ((conn != NULL) && (conn->getSource() != NULL))
      {
         isFetched   = true;
         srcIndex    = conn->getSource()->GetIndex();
         srcDeviceId = conn->getSource()->GetDeviceId();
      }
   }
   return isFetched;
}


void ConnectionControllerPrj::evalApplicationStart(int grpIndex, Source* /*src*/)
{
   if (m_bGroupRequestActive == true)
   {
      m_bGroupRequestActive = false;
      sendApplicationStart(getApplicationIdForGroup(grpIndex));
   }
}


void ConnectionControllerPrj::actionOnNoSrc(int nGrpIndex)
{
   if (nGrpIndex == static_cast<int>(GROUP_MEDIA))
   {
      //System mute should be sent to Audio
      m_pHmiAudioServiceStubHandler->sendSourceActivatedRequest(static_cast<int32>(SRC_MEDIA_NO_SRC));
      sendApplicationStart(getApplicationIdForGroup(nGrpIndex));
   }
}


bool ConnectionControllerPrj::groupRequest(int nGrpIndex)
{
   bool ret = false;
   if (ConnectionController::groupRequest(nGrpIndex))
   {
      sendApplicationStart(getApplicationIdForGroup(nGrpIndex));
      ret = true;
   }
   return ret;
}


bool ConnectionControllerPrj::sourceRequest(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: sourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", nSrcIndex, nDevId, nSinkId));
   bool retVal = false;
   if (COCKPIT_SINK_ID == nSinkId)
   {
      retVal = ConnectionController::sourceRequest(nSrcIndex, nDevId, nSinkId);
   }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   else
   {
      if (NULL != _cabinConnectionHandler)
      {
         retVal = _cabinConnectionHandler->onSourceRequest(nSrcIndex, nDevId, nSinkId);
      }
   }
#endif
   return retVal;
}


void ConnectionControllerPrj::sourceDisconnect(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: sourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", nSrcIndex, nDevId, nSinkId));
   if (COCKPIT_SINK_ID == nSinkId)
   {
      ConnectionController::sourceDisconnect(nSrcIndex, nDevId, nSinkId);
   }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   else
   {
      if (NULL != _cabinConnectionHandler)
      {
         _cabinConnectionHandler->onSourceDisconnect(nSrcIndex, nDevId, nSinkId);
      }
   }
#endif
}


void ConnectionControllerPrj::onNewActiveConnections()
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: onNewActiveConnections"));
   m_PrivateSpeakerSinkId = COCKPIT_SINK_ID;
   updateActiveSrcOnSinkId();
   sendConnectionInfoUpdate();
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   ::std::map< int16, stSourceInfo > newSinkInfos;
   ::std::map< int16, stSourceInfo > newMixedSourceSinksInfo;
   ::std::map< int16, stSourceInfo > suspendedSinkInfos;
   ::std::map< int16, stSourceInfo > suspendedMixedSourceSinksInfo;
#endif
   for (::std::map<int, Connection*>::iterator connIter = m_ConnectionMap.begin(); (connIter != m_ConnectionMap.end()); ++connIter)
   {
      Connection* connection = connIter->second;
      if (NULL != connection)
      {
         int connectionState = connection->getState();
         if ((STATE_CONNECTED == connectionState) || (STATE_DISCONNECTING == connectionState) || (STATE_SUSPENDED == connectionState))
         {
            Sink* poSink     = connection->getSink();
            Source* poSource = connection->getSource();
            if ((NULL != poSink) && (NULL != poSource))
            {
               int16 srcId     = static_cast<int16>(poSource->GetIndex());
               int16 sinkId    = poSink->GetID();
               int16 sinkIndex = static_cast<int16>(AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetIndexOnSinkID(sinkId));
               ETG_TRACE_USR4(("ConnectionControllerPrj: onNewActiveConnections: SinkId: %d, SourceId: %d", sinkIndex, srcId));
               if ((PRIVATE_SPEAKER_SINK_ID == sinkIndex) && (STATE_CONNECTED == connectionState))
               {
                  m_PrivateSpeakerSinkId = PRIVATE_SPEAKER_SINK_ID;
               }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
               else if ((COCKPIT_SINK_ID != sinkIndex) && (SINK_INVALID != sinkIndex) && (CabinSrcInfo::SRC_TYPE_UNKNOWN != CabinSrcInfo::getSrcType(srcId)))
               {
                  if (CabinSrcInfo::CABIN_MIX_SOURCE == CabinSrcInfo::getSrcType(srcId))
                  {
                     if (connectionState == STATE_CONNECTED)
                     {
                        stSourceInfo srcInfo;
                        srcInfo.sinkId = sinkIndex;
                        srcInfo.srcId = srcId;
                        srcInfo.deviceId = static_cast<uint32>(poSource->GetDeviceId());
                        srcInfo.connectionState = static_cast<uint32>(connectionState);
                        newMixedSourceSinksInfo.insert(std::pair< int16, stSourceInfo >(sinkIndex, srcInfo));
                     }
                     else if (connectionState == STATE_SUSPENDED)
                     {
                        stSourceInfo srcInfo;
                        srcInfo.sinkId = sinkIndex;
                        srcInfo.srcId = srcId;
                        srcInfo.deviceId = static_cast<uint32>(poSource->GetDeviceId());
                        srcInfo.connectionState = static_cast<uint32>(connectionState);
                        suspendedMixedSourceSinksInfo.insert(std::pair< int16, stSourceInfo >(sinkIndex, srcInfo));
                     }
                  }
                  else
                  {
                     //INFO: Announcement source never goes for suspended state, so only main sources will be enough to guard.
                     if (((connectionState == STATE_CONNECTED) || (getActiveSourceIdOnSink(sinkIndex) == SRC_INVALID)))
                     {
                        stSourceInfo srcInfo;
                        srcInfo.sinkId = sinkIndex;
                        srcInfo.srcId = srcId;
                        srcInfo.deviceId = static_cast<uint32>(poSource->GetDeviceId());
                        srcInfo.connectionState = static_cast<uint32>(connectionState);
                        newSinkInfos.insert(std::pair< int16, stSourceInfo >(sinkIndex, srcInfo));
                     }
                     else if (connectionState == STATE_SUSPENDED)
                     {
                        stSourceInfo srcInfo;
                        srcInfo.sinkId = sinkIndex;
                        srcInfo.srcId = srcId;
                        srcInfo.deviceId = static_cast<uint32>(poSource->GetDeviceId());
                        srcInfo.connectionState = static_cast<uint32>(connectionState);
                        suspendedSinkInfos.insert(std::pair< int16, stSourceInfo >(sinkIndex, srcInfo));
                     }
                  }
               }
               else
               {}
#endif
            }
            else
            {
               ETG_TRACE_USR4(("ConnectionControllerPrj: onNewActiveConnections(): Source/Sink NULL pointer for connection ID : %d", connection->getConnectionID()));
            }
         }
      }
   }
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   if (NULL != _cabinConnectionHandler)
   {
      _cabinConnectionHandler->updateCabinConnections(newSinkInfos, suspendedSinkInfos);
      _cabinConnectionHandler->updateCabinConnectionsForMixSource(newMixedSourceSinksInfo, suspendedMixedSourceSinksInfo);
   }
#endif
}


void ConnectionControllerPrj::setVisibleApplication(enApplicationId nAppId)
{
   m_ActiveAppId = nAppId;
   /* if (m_ActiveAppId == APPHMI_TUNER || m_ActiveAppId == APPHMI_MEDIA || m_ActiveAppId == APPHMI_PHONE)
   {
      WritePersistentApplication();
   }
   ContextSwitchController::getContextSwitchController().updateActiveApplication(nAppId); */
}


void ConnectionControllerPrj::updateActiveSrcOnSinkId()
{
   for (std::map<int, Connection*>::iterator connIter = m_ConnectionMap.begin(); connIter != m_ConnectionMap.end(); ++connIter)
   {
      Connection* connection = connIter->second;
      if (NULL != connection)
      {
         if ((STATE_CONNECTED == connection->getState()) || (STATE_SUSPENDED == connection->getState()))
         {
            Sink* poSink = connection->getSink();
            if (NULL != poSink)
            {
               int16 sinkId     = poSink->GetID();
               int16 sinkIndex  = static_cast<int16>(AudioControllerObjectManager::getInstance().getResourceController<ResourceController>()->GetIndexOnSinkID(sinkId));
               Source* poSource = connection->getSource();
               if (NULL != poSource)
               {
                  int16 srcId = static_cast<uint32>(poSource->GetIndex());
                  ETG_TRACE_USR4(("ConnectionControllerPrj: updateCockpitDTR: SinkId: %d, SourceId: %d", sinkIndex, srcId));
                  ::getDM().updateDirectTextureRender(sinkIndex, srcId);
               }
            }
         }
      }
   }
}


#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO


bool ConnectionControllerPrj::sendCabinSourceRequest(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: sendCabinSourceRequest: SourceId: %d, DeviceId: %d, SinkId: %d", nSrcIndex, nDevId, nSinkId));
   bool retVal = ConnectionController::sourceRequest(nSrcIndex, nDevId, nSinkId);
   return retVal;
}


void ConnectionControllerPrj::sendCabinSourceDisconnect(int nSrcIndex, int nDevId, uint16_t nSinkId)
{
   ETG_TRACE_USR4(("ConnectionControllerPrj: sendCabinSourceDisconnect: SourceId: %d, DeviceId: %d, SinkId: %d", nSrcIndex, nDevId, nSinkId));
   ConnectionController::sourceDisconnect(nSrcIndex, nDevId, nSinkId);
}


#endif


void ConnectionControllerPrj::ReadPersistentApplication()
{
   /* #ifdef DP_DATAPOOL_ID
      dp_tclhmiMasterLastModeMaster dp;
      m_ActiveAppId = (enHmiApplication) dp.tGetData();
      ETG_TRACE_USR4(("ReadPersistentApplication m_ActiveAppId %d", m_ActiveAppId));
   #else
      m_ActiveAppId = APPID_APPHMI_PLAYGROUND;
   #endif */
}


void ConnectionControllerPrj::WritePersistentApplication()
{
   /* #ifdef DP_DATAPOOL_ID
      ETG_TRACE_USR4(("WritePersistentApplication m_ActiveAppId %d", m_ActiveAppId));
      dp_tclhmiMasterLastModeMaster dp;
      dp.vSetData(m_ActiveAppId);
   #endif */
}
