/**
 * @file         : ApplicationSwitchServerComponent.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ApplicationSwitchServerComponent is extention of ApplicationSwitchStub
 * @copyright    : (c) 2017-2020 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "ApplicationSwitchServerComponent.h"
#include "hmi_trace_if.h"
#include "AppHmi_MasterMessages.h"
#include <Core/Utility/MasterUtility.h>
#include <Core/AppFlyInListHandler/AppFlyIn.h>
#include <Core/RegionHandling/RegionHandlingInterface.h>
#include <Core/HmiInfoService/HmiInfoServiceServerComponent.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerInterface.h>


using namespace ::bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;
using namespace ::bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService;


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#include "trcGenProj/Header/ApplicationSwitchServerComponent.cpp.trc.h"
#endif


/**
 *  Defines Declaration
 */

#define DP_INF4CV_VARIANTSELECTION_NAVI 1
#define DP_INF4CV_VARIANTSELECTION_NONNAVI 2
#define NAVIGATION_MODE_REGID_DEFAULT 0

/**
 *  Memeber Functions Definition
 */

namespace App {
namespace Core {


ApplicationSwitchServerComponent::ApplicationSwitchServerComponent() : ApplicationSwitchStub("applicationSwitchServerPort")
   , _speechStatus(false)
   , _regionHandling(NULL)
   , _contextSwitchHandler(NULL)
   , _hmiInfoServiceServerComponent(NULL)
   , _appFlyIn(NULL)
   , _hmiInfoServiceProxyClient(HmiInfoServiceProxy::createProxy("hmiinfoservicePort", *this))
   , _navigationModeChangeUpdate_RegId(NAVIGATION_MODE_REGID_DEFAULT)
{
   ETG_TRACE_USR1(("ApplicationSwitchServerComponent: CTOR"));
   if (_hmiInfoServiceProxyClient.get())
   {
      StartupSync::getInstance().registerPropertyRegistrationIF(this, _hmiInfoServiceProxyClient->getPortName());
   }
}


ApplicationSwitchServerComponent::~ApplicationSwitchServerComponent()
{
   ETG_TRACE_USR1(("ApplicationSwitchServerComponent: DTOR"));
   _speechStatus                  = false;
   _regionHandling                = NULL;
   _contextSwitchHandler         = NULL;
   _hmiInfoServiceServerComponent = NULL;
   _appFlyIn                      = NULL;
}


void ApplicationSwitchServerComponent::registerProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchServerComponent::registerProperties entered"));
}


void ApplicationSwitchServerComponent::deregisterProperties(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchServerComponent::deregisterProperties entered"));
}


void ApplicationSwitchServerComponent::onAvailable(const ::boost::shared_ptr< asf::core::Proxy >& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("CabinSourceRestoration::onAvailable entered"));
   if (_hmiInfoServiceProxyClient == proxy)
   {
      _navigationModeChangeUpdate_RegId = _hmiInfoServiceProxyClient->sendNavigationModeChangeTriggerRegister(*this);
   }
}


void ApplicationSwitchServerComponent::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy, const asf::core::ServiceStateChange& /*stateChange*/)
{
   ETG_TRACE_COMP(("ApplicationSwitchServerComponent::onUnavailable entered"));
   if (_hmiInfoServiceProxyClient == proxy)
   {
      _hmiInfoServiceProxyClient->sendNavigationModeChangeTriggerDeregister(_navigationModeChangeUpdate_RegId);
      _navigationModeChangeUpdate_RegId = NAVIGATION_MODE_REGID_DEFAULT;
   }
}


void ApplicationSwitchServerComponent::onSendContextRequest(const ::boost::shared_ptr< SendContextRequest >& request)
{
   if ((NULL != _contextSwitchHandler) && (NULL != _regionHandling))
   {
      ApplicationContextInfo tInfo;
      tInfo.setRegionId(_regionHandling->getRegionId());
      tInfo.setApplicationId(request->getSourceAppId());
      tInfo.setContextId(request->getSourceActivityId());
      _contextSwitchHandler->onStoreContext(tInfo);
   }
}


void ApplicationSwitchServerComponent::onDisplayAppFlyInReqRequest(const ::boost::shared_ptr< DisplayAppFlyInReqRequest >& request)
{
   //INFO: Interface is muted to avoid unauthorized control
   //TODO: Remove it from .fidl -sve2cob
   /* if (sm_activeapplicationID != Enum_APPID_APPHMI_MASTER && sm_activeapplicationID != Enum_APPID_APPHMI_TUNER && sm_activeapplicationID != Enum_APPID_APPHMI_MEDIA)
   {
      AppFlyInReqMsg* subsurfaceMsg = COURIER_MESSAGE_NEW(AppFlyInReqMsg)((request->getStatus()));
      if (subsurfaceMsg != 0)
      {
         subsurfaceMsg->Post();
      }
   } */
}


void ApplicationSwitchServerComponent::onSpi_ForwardEncoderRequest(const ::boost::shared_ptr< Spi_ForwardEncoderRequest >& request)
{
   //INFO: Interface is muted to avoid unauthorized control
   //TODO: Remove it from .fidl -sve2cob
   /* int32 Encode = request->getEncode();
   int32 EncodeSteps = request->getEncodeSteps();
   DataModelSound::getInstance().updateEncoderVolume(Encode, EncodeSteps); */
}


void ApplicationSwitchServerComponent::onSpi_connectionstatusRequest(const ::boost::shared_ptr< Spi_connectionstatusRequest >& request)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onSpi_connectionstatusRequest: Status: %d", request->getStatus()));
   sm_IN_bIsSPIConnected = request->getStatus();
   /* if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->HideCommonButton(APPID_APPHMI_SPI, static_cast< enActivityIDs >(222)); //TODO: Re-Visit -sve2cob
   } */
}


void ApplicationSwitchServerComponent::onRequestContextSwitchRequest(const ::boost::shared_ptr< RequestContextSwitchRequest >& request)
{
   if (NULL != _contextSwitchHandler)
   {
      ContextSwitchRequestInfo tInfo;
      tInfo.setRegionId(request->getRegionId());
      tInfo.setSourceContextId(request->getSourceActivityId());
      tInfo.setTargetContextId(request->getTargetActivityId());
      tInfo.setSourceApplicationId(request->getSourceAppId());
      tInfo.setTargetApplicationId(request->getTargetAppId());
      _contextSwitchHandler->onContextSwitchRequest(tInfo);
   }
}


void ApplicationSwitchServerComponent::onRequestApplcationSwitchRequest(const ::boost::shared_ptr< RequestApplcationSwitchRequest >& request)
{
   //INFO: Interface is muted to avoid unauthorized control
   //TODO: Remove it from .fidl -sve2cob
   /* ETG_TRACE_USR4(("ApplicationSwitchServerComponent onRequestApplcationSwitchRequest"));
   enApplicationId appID = static_cast<enApplicationId>(request->getAppID());
   SwitchApplicationMsg* appSwitchMsg = COURIER_MESSAGE_NEW(SwitchApplicationMsg)(appID);
   if (appSwitchMsg != 0)
   {
      appSwitchMsg->Post();
   } */
}


void ApplicationSwitchServerComponent::onRequestShowHideWaitSceneRequest(const ::boost::shared_ptr< RequestShowHideWaitSceneRequest >& request)
{
   //TODO: Remove this interface once FO confirms the need -sve2cob
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onRequestShowHideWaitSceneRequest: <-: Status: %d", request->getPopupshow()));
   ShowHideWaitSceneMsg* waitSceneMsg = COURIER_MESSAGE_NEW(ShowHideWaitSceneMsg)(request->getPopupshow());
   if (waitSceneMsg != 0)
   {
      waitSceneMsg->Post();
   }
}


void ApplicationSwitchServerComponent::onSpeechSession_activated_deactivatedRequest(const ::boost::shared_ptr< SpeechSession_activated_deactivatedRequest >& request)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onSpeechSession_activated_deactivatedRequest: <-: Status =%d ", request->getStatus()));
   _speechStatus = request->getStatus();
   AppFlyInActivationReqMsg* oMsg =  COURIER_MESSAGE_NEW(AppFlyInActivationReqMsg)(!_speechStatus);
   if (NULL != oMsg)
   {
      oMsg->Post();
   }
}


void ApplicationSwitchServerComponent::onMapout_activated_deactivatedRequest(const ::boost::shared_ptr< Mapout_activated_deactivatedRequest >& request)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onMapout_activated_deactivatedRequest: <-: RegionId: %d : Status : %d", request->getRegionId(), request->getStatus()));
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   MapOutStatusUpdMsg* oMsg = COURIER_MESSAGE_NEW(MapOutStatusUpdMsg)(request->getRegionId(), request->getStatus());
   if (NULL != oMsg)
   {
      oMsg->Post();
   }
#endif
}


void ApplicationSwitchServerComponent::onActivateDeactivateRVCSceneRequest(const ::boost::shared_ptr< ActivateDeactivateRVCSceneRequest >& request)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onActivateDeactivateRVCSceneRequest: <-: Status : %d", request->getPopupshow()));
   ShowHideRVCSceneMsg* RVCSceneMsg = COURIER_MESSAGE_NEW(ShowHideRVCSceneMsg)(request->getPopupshow());
   if (RVCSceneMsg != 0)
   {
      RVCSceneMsg->Post();
   }
}


void ApplicationSwitchServerComponent::onConfigureAppFlyInStatusRequest(const ::boost::shared_ptr< ConfigureAppFlyInStatusRequest >& request)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: onConfigureAppFlyInStatusRequest::"));
   if (NULL != _appFlyIn)
   {
      _appFlyIn->onAppFlyInButtonStateUpdateRequest(request->getStateInfo());
   }
}


void ApplicationSwitchServerComponent::sendSWCKeyEventSignal(const uint32 event)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSWCKeyEventSignal: ->: Event: %d", event));
   sendSig_SWCKeyEventSignal(event);
}


void ApplicationSwitchServerComponent::sendControlPanelButtonPressUpdateSignal(const uint8 buttonId, const uint8 buttonState, const uint32 receiverAppID)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendControlPanelButtonPressUpdateSignal: ->: buttonId: %d buttonState: %d receiverAppID %d", buttonId, buttonState, receiverAppID));
   sendSig_ControlPanelButtonPressSignal(buttonId, buttonState, receiverAppID);
}


void ApplicationSwitchServerComponent::sendSpeechSessionSignal(const bool status)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSpeechSessionSignal: <-: Status: %d", status));
   if ((status) || ((!status) && (status != _speechStatus)))
   {
      if ((status) && (NULL != _hmiInfoServiceServerComponent)) //TODO: is it really required? -sve2cob
      {
         _hmiInfoServiceServerComponent->sendForegroundApplicationUpdate(sm_activeapplicationID);
      }
      ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSpeechSessionSignal: ->: Status: %d", status));
      sendSig_activate_deactivate_SpeechSessionSignal(status);
   }
}


void ApplicationSwitchServerComponent::sendSPIVoiceRecognitionSignal(const uint32 event)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSPIVoiceRecognitionSignal: <-: Event: %d", event));
   switch (event)
   {
      case 1:
         ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSig_activate_spi_voicerecognition_shortSignal: ->: Event: %d", event));
         sendSig_activate_spi_voicerecognition_shortSignal(true);
         break;
      case 2:
         ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSig_activate_spi_voicerecognition_short_releaseSignal: ->: Event: %d", event));
         sendSig_activate_spi_voicerecognition_short_releaseSignal(true);
         break;
      case 3:
         ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSig_activate_spi_voicerecognition_longSignal: ->: Event: %d", event));
         sendSig_activate_spi_voicerecognition_longSignal(true);
         break;
      case 4:
         ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendSig_activate_spi_voicerecognition_long_releaseSignal: ->: Event: %d", event));
         sendSig_activate_spi_voicerecognition_long_releaseSignal(true);
         break;
      default:
         break;
   }
}


void ApplicationSwitchServerComponent::sendRegionChangePropertyUpdate(const uint32 regionId)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendRegionChangePropertyUpdate: ->: RegionId: %d", regionId));
   setActiveRegionId(regionId);
   sendActiveRegionIdUpdate();
}


void ApplicationSwitchServerComponent::sendMapOutSignal(const uint32 regionId, const bool status)
{
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendMapOutSignal: ->: RegionId: %d : Status: %d", regionId, status));
   sendMapout_Activate_DeactivateSignal(regionId, status);
#endif
}


void ApplicationSwitchServerComponent::sendActivateContextSignal(const uint32 regionId, const uint32 appId, const uint32 contextId)
{
   ETG_TRACE_USR4(("ApplicationSwitchServerComponent: sendActivateContextSignal: ->: RegionId: %d : AppId: %d : ContextId: %d", regionId, appId, contextId));
   sendSig_ActivateContextSignal(static_cast< Region >(regionId), appId, static_cast< enActivityIDs >(contextId));
}


void ApplicationSwitchServerComponent::onNavigationModeChangeTriggerError(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& /* proxy*/, const ::boost::shared_ptr<NavigationModeChangeTriggerError >& error)
{
   if ((NULL != error.get()) && (error->getAct() == _navigationModeChangeUpdate_RegId))
   {
      ETG_TRACE_USR4(("ApplicationSwitchServerComponent::onNavigationModeChangeTriggerError()"));
   }
}


void ApplicationSwitchServerComponent::onNavigationModeChangeTriggerSignal(const ::boost::shared_ptr<bosch::cm::ai::hmi::hmiinfoservice::HmiInfoService::HmiInfoServiceProxy >& /*proxy*/, const ::boost::shared_ptr<NavigationModeChangeTriggerSignal >& signal)
{
   if ((NULL != signal.get()) && (signal->getAct() == _navigationModeChangeUpdate_RegId))
   {
      ETG_TRACE_USR4(("ApplicationSwitchServerComponent::onNavigationModeChangeTriggerSignal carModeStatus: %d, coachModeStatus: %d", signal->getCarModeActive(), signal->getCoachModeActive()));
   }
}


} //namespace Core
} //namespace App
