/**
 * @file         : ApplicationSurfaceInfoHandler.cpp
 * @author       : INF4CV - AppHmi_Master Team
 * @addtogroup   : AppHmi_Master
 * @brief        : ApplicationSurfaceInfoHandler is to handle the application
 *                 surface information.
 * @copyright    : (c) 2019-2019 Robert Bosch Car Multimedia GmbH
 *                 The reproduction, distribution and utilization of this file as
 *                 well as the communication of its contents to others without express
 *                 authorization is prohibited. Offenders will be held liable for the
 *                 payment of damages. All rights reserved in the event of the grant
 *                 of a patent, utility model or design.
 */


#include "ApplicationSurfaceInfoHandler.h"
#include "ProjectBaseTypes.h"


/**
 *  Global Variables Declaration
 */

namespace App {
namespace Core {


ApplicationSurfaceInfoHandler* _applicationSurfaceInfoHandler = NULL;
} //namespace Core
} //namespace App


/**
 *  Static Variables Initialization
 */

namespace App {
namespace Core {


#ifndef WIN32 /* Linux / Other OS env except Windows */
pthread_mutex_t ApplicationSurfaceInfoHandler::_safeLock = PTHREAD_MUTEX_INITIALIZER;
#endif
} //namespace Core
} //namespace App


/**
 *  SM "C" Functions Definition
 */

uint32 fnGetApplicationId(const uint32 surfaceId)
{
   uint32 tAppId = 0;
   if (NULL != ::App::Core::_applicationSurfaceInfoHandler)
   {
      tAppId = ::App::Core::_applicationSurfaceInfoHandler->getApplicationId(surfaceId);
   }
   return tAppId;
}


namespace App {
namespace Core {


ApplicationSurfaceInfoHandler::ApplicationSurfaceInfoHandler()
{
   clear();
   initApplicationsSurfaceInfo();
   _applicationSurfaceInfoHandler = this;
}


ApplicationSurfaceInfoHandler::~ApplicationSurfaceInfoHandler()
{
   clear();
   _applicationSurfaceInfoHandler = NULL;
}


ApplicationSurfaceInfoHandler::ApplicationSurfaceInfoHandler(const ApplicationSurfaceInfoHandler& obj)
{
   clear();
   *this                          = obj;
   _applicationSurfaceInfoHandler = this;
}


ApplicationSurfaceInfoHandler& ApplicationSurfaceInfoHandler::operator= (const ApplicationSurfaceInfoHandler& obj)
{
   if (this != (&obj))
   {
      clear();
      _applicationsSurfaceInfo = obj._applicationsSurfaceInfo;
   }
   return (*this);
}


void ApplicationSurfaceInfoHandler::initApplicationsSurfaceInfo()
{
   if (isValidToInitApplicationsSurfaceInfo())
   {
      ApplicationsSurfaceInfo tApplicationsSurfaceInfo;
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_MASTER,         SURFACEID_MAIN_SURFACE_MASTER));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_TUNER,          SURFACEID_MAIN_SURFACE_TUNER));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_MEDIA,          SURFACEID_MAIN_SURFACE_MEDIA));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_NAVIGATION,     SURFACEID_MAIN_SURFACE_NAVIGATION));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_PHONE,          SURFACEID_MAIN_SURFACE_PHONE));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_SYSTEM,         SURFACEID_MAIN_SURFACE_SYSTEM));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_SPI,            SURFACEID_MAIN_SURFACE_SPI));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_SPEECH,         SURFACEID_MAIN_SURFACE_SDS));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_TESTMODE,       SURFACEID_MAIN_SURFACE_TESTMODE));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_CAMERA,         SURFACEID_MAIN_SURFACE_CAMERA));
      tApplicationsSurfaceInfo.insert(::std::make_pair(APPID_APPHMI_SXM,            SURFACEID_MAIN_SURFACE_SXM));
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_lock(&_safeLock);
#endif
      _applicationsSurfaceInfo = tApplicationsSurfaceInfo;
#ifndef WIN32 /* Linux / Other OS env except Windows */
      pthread_mutex_unlock(&_safeLock);
#endif
   }
}


uint32 ApplicationSurfaceInfoHandler::getApplicationId(const uint32 surfaceId) const
{
   uint32 tAppId = 0;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_lock(&_safeLock);
#endif
   for (ApplicationsSurfaceInfo::const_iterator itr = _applicationsSurfaceInfo.begin(); (itr != _applicationsSurfaceInfo.end()); ++itr)
   {
      if (itr->second == surfaceId)
      {
         tAppId = itr->first;
         break;
      }
   }
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_unlock(&_safeLock);
#endif
   return tAppId;
}


bool ApplicationSurfaceInfoHandler::isValidToInitApplicationsSurfaceInfo() const
{
   bool isValid = false;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_lock(&_safeLock);
#endif
   isValid = (_applicationsSurfaceInfo.empty()) ? true : false;
#ifndef WIN32 /* Linux / Other OS env except Windows */
   pthread_mutex_unlock(&_safeLock);
#endif
   return isValid;
}


} //namespace Core
} //namespace App
