/* ***************************************************************************************
* FILE:          AppFlyIn.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  AppFlyIn.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "AppFlyIn.h"
#include "hmi_trace_if.h"
#include "CgiExtensions/ImageLoader.h"
#include "List/Data/ListDataProviderDistributor.h"
#include <Core/RegionHandling/RegionHandlingTypes.h>
#include <App/Core/CabinConnectionHandler/CabinConnectionHandlerInterface.h>
#include <Core/ContextSwitchHandler/ContextSwitchHandlerTypes.h>
#include <App/Core/AudioInterface/ConnectionControllerPrj.h>
#include <AppHmi_MasterBase/AudioInterface/AudioDefines.h>
#include <Core/AudioInterface/AudioInterfaceHelper.h>


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_APPHMI_MASTER_MAIN
#define ETG_I_FILE_PREFIX                 App::Core::AppFlyInListHandler::AppFlyIn::
#include "trcGenProj/Header/AppFlyIn.cpp.trc.h"
#endif

using namespace bosch::cm::ai::hmi::hmimasterservice::ApplicationSwitch;

namespace App {
namespace Core {


AppFlyIn* AppFlyIn::_theModelInstance = NULL;


AppFlyIn::AppFlyIn()
   : _contextSwitchHandler(NULL)
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   , _cabinConnectionHandler(NULL)
#endif
   , _variantHandling(NULL)
   , _regionHandling(NULL)
   , _sceneStatus(SCENE_STATUS_INACTIVE)
{
}


AppFlyIn::~AppFlyIn()
{
   _sceneStatus            = SCENE_STATUS_INACTIVE;
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   _cabinConnectionHandler = NULL;
#endif
   _contextSwitchHandler   = NULL;
   _variantHandling        = NULL;
   _regionHandling         = NULL;
   _appFlyInDataInfo.clear();
   _appFlyInOrderInfo.clear();
   _appContextInfo.clear();
}


AppFlyIn::AppFlyIn(const AppFlyIn& obj)
   : _contextSwitchHandler(NULL)
#ifdef VARIANT_S_FTR_ENABLE_COACH_AUDIO
   , _cabinConnectionHandler(NULL)
#endif
   , _variantHandling(NULL)
   , _regionHandling(NULL)
   , _sceneStatus(SCENE_STATUS_INACTIVE)
{
   *this = obj;
}


AppFlyIn& AppFlyIn::operator = (const AppFlyIn& obj)
{
   if (this != &obj)
   { }
   return *this;
}


AppFlyIn* AppFlyIn::getInstance()
{
   if (_theModelInstance == NULL)
   {
      _theModelInstance = new AppFlyIn();
   }
   return _theModelInstance;
}


void AppFlyIn::deleteInstance()
{
   if (NULL != _theModelInstance)
   {
      delete _theModelInstance;
      _theModelInstance = NULL;
   }
}


void AppFlyIn::initialize()
{
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->addActiveContextListener((*this));
   }
   if (NULL != _regionHandling)
   {
      _regionHandling->addCMAStatusListener(*this);
      _regionHandling->addCMGStatusListener(*this);
   }
   initAppFlyInListDataInfo();
   initAppFlyInListOrderInfo();
   initApplicationContextInfo();
   ListRegistry::s_getInstance().addListImplementation(LISTID_MASTER_APPFLYIN, this);
}


void AppFlyIn::finalize()
{
   if (NULL != _contextSwitchHandler)
   {
      _contextSwitchHandler->removeActiveContextListener((*this));
   }
   if (NULL != _regionHandling)
   {
      _regionHandling->removeCMAStatusListener(*this);
      _regionHandling->removeCMGStatusListener(*this);
   }
   _appFlyInDataInfo.clear();
   _appFlyInOrderInfo.clear();
   _appContextInfo.clear();
   ListRegistry::s_getInstance().removeListImplementation(LISTID_MASTER_APPFLYIN);
}


bool AppFlyIn::onCourierMessage(const AppFlyInActivationReqMsg& oMsg)
{
   if (!oMsg.GetActivationStatus())
   {
      POST_MSG((COURIER_MESSAGE_NEW(AppFlyInReqMsg)(false)));
   }
   else
   {
      ApplicationContextInfo context;
      if ((NULL != _contextSwitchHandler) && (_contextSwitchHandler->fetchActiveContextInfo(context)))
      {
         onActiveContextUpdate(context);
      }
   }
   return true;
}


bool AppFlyIn::onCourierMessage(const SceneStatusInfoMsg& oMsg)
{
   bool msgProcessed = false;
   if (oMsg.GetViewId() == M1_Scenes_AppFlyIn)
   {
      msgProcessed = true;
      if (_sceneStatus != oMsg.GetActiveStatus())
      {
         _sceneStatus = static_cast<enSceneStatus>(oMsg.GetActiveStatus());
      }
   }
   ETG_TRACE_COMP(("AppFlyIn::SceneStatusInfoMsg::sceneStatus::%d", _sceneStatus));
   return msgProcessed;
}


bool AppFlyIn::onCourierMessage(const ButtonListItemUpdMsg& oMsg)
{
   if ((oMsg.GetListId() == LISTID_MASTER_APPFLYIN) && (oMsg.GetReaction() == enRelease) && (NULL != _regionHandling))
   {
      ETG_TRACE_USR1(("AppFlyIn::ButtonListItemUpdMsg::ListId::%d::RegionId::%d::Item::%d", oMsg.GetListId(), _regionHandling->getRegionId(), oMsg.GetHdl()));
      performListButtonPress(oMsg.GetListId(), _regionHandling->getRegionId(), oMsg.GetHdl());
      return true;
   }
   return false;
}


void AppFlyIn::initAppFlyInListDataInfo()
{
   switch (fnGetVariantId())
   {
      case VEHICLE_TYPE_COACH:
      {
         _appFlyInDataInfo = getCoachListInfo();
      }
      break;
      case VEHICLE_TYPE_TRUCK:
      {
         _appFlyInDataInfo = getTruckListInfo();
      }
      break;
      case VEHICLE_TYPE_SMART:
      {
         _appFlyInDataInfo = getSmartListInfo();
      }
      break;
      default:
         break;
   }
}


void AppFlyIn::initAppFlyInListOrderInfo()
{
   switch (fnGetVariantId())
   {
      case VEHICLE_TYPE_COACH:
      {
         _appFlyInOrderInfo = getCoachListOrder();
      }
      break;
      case VEHICLE_TYPE_TRUCK:
      {
         _appFlyInOrderInfo = getTruckListOrder();
      }
      break;
      case VEHICLE_TYPE_SMART:
      {
         _appFlyInOrderInfo = getSmartListOrder();
      }
      break;
      default:
         break;
   }
}


void AppFlyIn::initApplicationContextInfo()
{
   _appContextInfo = getApplicationContextInfo();
}


::std::map < uint32, AppFlyInsInfo > AppFlyIn::getCoachListInfo() const
{
   if (fnGetVariantType() ==  VARIANT_TYPE_CABIN_ONLY)
   {
#include "AppFlyInListInfoCabinOnly.dat"
   }
   else
   {
#include "AppFlyInListInfoCoach.dat"
   }
}


::std::map < uint32, AppFlyInsInfo > AppFlyIn::getSmartListInfo() const
{
#include "AppFlyInListInfoSmart.dat"
}


::std::map < uint32, AppFlyInsInfo > AppFlyIn::getTruckListInfo() const
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
#include "AppFlyInListInfoTruck.dat"
   }
   else
   {
#include "AppFlyInListInfoSmart.dat"
   }
}


::std::map < uint32, AppFlyInsOrderInfo > AppFlyIn::getCoachListOrder() const
{
   switch (fnGetVariantType())
   {
      case VARIANT_TYPE_NAVI:
      {
#include "AppFlyInOrderInfoCoachNavi.dat"
      }
      break;
      case VARIANT_TYPE_CABIN_ONLY:
      {
#include "AppFlyInOrderInfoCabinOnly.dat"
      }
      break;
      default:
#include "AppFlyInOrderInfoCoachNonNavi.dat"
         break;
   }
}


::std::map < uint32, AppFlyInsOrderInfo > AppFlyIn::getSmartListOrder() const
{
#include "AppFlyInOrderInfoSmart.dat"
}


::std::map < uint32, AppFlyInsOrderInfo > AppFlyIn::getTruckListOrder() const
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
#include "AppFlyInOrderInfoTruck.dat"
   }
   else
   {
#include "AppFlyInOrderInfoSmart.dat"
   }
}


::std::map <uint32, ::std::vector <uint32> > AppFlyIn::getApplicationContextInfo() const
{
#include "ApplicationContextInfo.dat"
}


AppFlyInItemInfo* AppFlyIn::getAppFlyInItemInfo(const uint32 listId, const uint32 regionId)
{
   AppFlyInItemInfo* info = NULL;
   ::std::map < uint32, AppFlyInsInfo >::iterator itr = _appFlyInDataInfo.find(listId);
   if (itr != _appFlyInDataInfo.end())
   {
      AppFlyInsInfo::iterator vItr = (itr->second).find(regionId);
      if (vItr != (itr->second).end())
      {
         info = &vItr->second;
      }
   }
   return info;
}


AppFlyInItemOrder* AppFlyIn::getAppFlyInItemOrder(const uint32 listId, const uint32 regionId, const uint32 appId)
{
   AppFlyInItemOrder* info = NULL;
   ::std::map < uint32, AppFlyInsOrderInfo >::iterator itr = _appFlyInOrderInfo.find(listId);
   if (itr != _appFlyInOrderInfo.end())
   {
      AppFlyInsOrderInfo::iterator vItr = (itr->second).find(regionId);
      if (vItr != (itr->second).end())
      {
         for (AppFlyInItemOrderInfo::iterator impItr = (vItr->second).begin(); impItr != (vItr->second).end(); ++impItr)
         {
            if ((*impItr).getAppId() == appId)
            {
               info = &(*impItr);
               break;
            }
         }
      }
   }
   return info;
}


tSharedPtrDataProvider AppFlyIn::getListDataProvider(const ListDateProviderReqMsg& oMsg)
{
   tSharedPtrDataProvider dataProvider;
   if (oMsg.GetListId() == LISTID_MASTER_APPFLYIN)
   {
      ApplicationContextInfo context;
      if ((NULL != _contextSwitchHandler) && (NULL != _regionHandling) && (_contextSwitchHandler->fetchActiveContextInfo(context) == true))
      {
         uint32 regionId                          = _regionHandling->getRegionId();
         const AppFlyInItemInfo* appFlyInItemInfo = getAppFlyInItemInfo(oMsg.GetListId(), regionId);
         const AppFlyInItemOrder* appFlyItemOrder = getAppFlyInItemOrder(oMsg.GetListId(), regionId, context.getApplicationId());
         if ((appFlyItemOrder != NULL) && (appFlyInItemInfo != NULL))
         {
            ListDataProviderBuilder listBuilder(oMsg.GetListId());
            for (::std::vector< uint32 >::const_iterator itr = (appFlyItemOrder->getItemIndexVector()).begin(); itr != (appFlyItemOrder->getItemIndexVector()).end(); ++itr)
            {
               for (AppFlyInItemInfo::const_iterator item = appFlyInItemInfo->begin(); item != appFlyInItemInfo->end(); ++item)
               {
                  uint32 tempItemIndex = (*item).getItemIndex();
                  if ((((*itr) == tempItemIndex) && ((*item).getButtonState() != BUTTON_STATE_INVISIBLE)) && carmodeAppflyinApplicable(tempItemIndex))
                  {
                     ListDataProviderBuilder::ListItem* listItem = &(listBuilder.AddItem((*item).getItemIndex(), 0UL, "BtnAppFlyIn"));
                     ::Courier::DataBindingUpdater<AppFlyInListInfoDataBindingSource>* listItemDB  = listItem->template AddDataBindingUpdater<AppFlyInListInfoDataBindingSource>();
                     if (NULL != listItemDB)
                     {
                        ::Courier::DataItemContainer<AppFlyInListInfoDataBindingSource>* listItemData = &(listItemDB->GetValueContainer());
                        if (NULL != listItemData)
                        {
                           (*(*listItemData)).mAppFlyInEnability   = ((*item).getButtonState() != BUTTON_STATE_DISABLED) ? true : false;
                           Candera::String tBgImage = (*item).getBgImage();
                           if (((*item).getDynamicBgImage() != "") || (((*item).getButtonState() == BUTTON_STATE_DISABLED) && ((*item).getDisabledBgImage() != "")))
                           {
                              tBgImage = ((*item).getDynamicBgImage() != "") ? (*item).getDynamicBgImage() : (*item).getDisabledBgImage();
                           }
                           Candera::String tIconImage = (*item).getIconImage();
                           if (((*item).getDynamicIconImage() != "") || (((*item).getButtonState() == BUTTON_STATE_DISABLED) && ((*item).getDisabledIconImage() != "")))
                           {
                              tIconImage = ((*item).getDynamicIconImage() != "") ? (*item).getDynamicIconImage() : (*item).getDisabledIconImage();
                           }
                           Candera::String tBorderImage = (*item).getBorderImage();
                           if (((*item).getDynamicBorderImage() != "") || (((*item).getButtonState() == BUTTON_STATE_DISABLED) && ((*item).getDisabledBorderImage() != "")))
                           {
                              tBorderImage = ((*item).getDynamicBorderImage() != "") ? (*item).getDynamicBorderImage() : (*item).getDisabledBorderImage();
                           }
                           (*(*listItemData)).mAppFlyInImageBg     = ImageLoader::getAssetBitmapImage(tBgImage.GetCString());
                           (*(*listItemData)).mAppFlyInImageIcon   = ImageLoader::getAssetBitmapImage(tIconImage.GetCString());
                           (*(*listItemData)).mAppFlyInImageBorder = ImageLoader::getAssetBitmapImage(tBorderImage.GetCString());
                           (*listItemData).MarkAllItemsModified();
                        }
                     }
                     break;
                  }
               }
            }
            dataProvider = listBuilder.CreateDataProvider();
         }
      }
   }
   return dataProvider;
}


bool AppFlyIn::carmodeAppflyinApplicable(const uint32 itemIndex)
{
   bool isValid = true;
   if ((NULL != _regionHandling) &&
         (_regionHandling->getCarModeStatusUpdate() && (itemIndex == ITEM_NAVIGATION || itemIndex == ITEM_MAPOUT)) ||
         (_regionHandling->getCoachModeStatusUpdate() && (itemIndex == ITEM_CARMODE_NAVI || itemIndex == ITEM_CARMODE_MAPOUT)))
   {
      isValid = false;
   }
   return isValid;
}


bool AppFlyIn::isValidContextForApplication(const uint32 appId, const uint32 contextId) const
{
   bool isValid = false;
   ::std::map <uint32, ::std::vector <uint32> >::const_iterator itr = _appContextInfo.find(appId);
   if (itr != _appContextInfo.end())
   {
      for (::std::vector <uint32>::const_iterator vItr = (itr->second).begin(); vItr != (itr->second).end(); ++vItr)
      {
         if ((*vItr) == contextId)
         {
            isValid = true;
            break;
         }
      }
   }
   return isValid;
}


void AppFlyIn::performListButtonPress(const uint32 listId, const uint32 regionId, const uint8 index)
{
   switch (index)
   {
      case ITEM_MAPOUT:
      case ITEM_CARMODE_MAPOUT:
      {
         performMapoutButtonPress(regionId);
         break;
      }
      case ITEM_MEDIA_RADIO:
      {
         performTunerMediaButtonPress(listId, regionId, index);
         break;
      }
      default:
         performDefaultContextSwitchRequest(listId, regionId, index);
         break;
   }
}


void AppFlyIn::performMapoutButtonPress(const uint32 regionId)
{
   if (fnGetVariantType() ==  VARIANT_TYPE_NAVI)
   {
      uint32 contextId = enActivityIDs__eActivityID_NONE;
#ifdef VARIANT_S_FTR_ENABLE_MAP_STREAMING
      if (NULL != _cabinConnectionHandler && (fnGetVariantType() ==  VARIANT_TYPE_NAVI))
      {
         _cabinConnectionHandler->onToggleMapOutStatus(regionId);
      }
#endif
      switch (regionId)
      {
         case REGION_CABIN_A:
         {
            contextId = enActivityIDs__eActivityID_CABINA_HOME;
            break;
         }
         case REGION_CABIN_B:
         {
            contextId = enActivityIDs__eActivityID_CABINB_HOME;
            break;
         }
         default:
         {
            contextId = enActivityIDs__eActivityID_HOME;
            break;
         }
      }
      ContextSwitchRequestInfo reqInfo;
      reqInfo.setRegionId(regionId);
      reqInfo.setTargetApplicationId(APPID_APPHMI_MASTER);
      reqInfo.setTargetContextId(contextId);
      if (NULL != _contextSwitchHandler)
      {
         _contextSwitchHandler->onContextSwitchRequest(reqInfo);
      }
   }
   else
   {
      ETG_TRACE_ERR(("AppFlyIn: performMapoutButtonPress: Feature NA in NonNavi Variant"));
   }
}


void AppFlyIn::performTunerMediaButtonPress(const uint32 listId, const uint32 regionId, const uint8 index)
{
   if ((NULL != _connectionControllerPrj.get()) && (NULL != _regionHandling))
   {
      uint32 sinkId    = _regionHandling->getAudioSinkIdForRegion(regionId);
      uint32 appId     = getApplicationIdForSrc(_connectionControllerPrj.get()->getCurrentActiveSource(sinkId));
      ETG_TRACE_USR1(("AppFlyIn::performTunerMediaButtonPress::Item::%d::SinkId::%d::SrcId::%d::AppId::%d::", index, sinkId, _connectionControllerPrj.get()->getCurrentActiveSource(sinkId), appId));
      switch (appId)
      {
         case APPID_APPHMI_MEDIA:
         {
            ContextSwitchRequestInfo reqInfo;
            reqInfo.setRegionId(regionId);
            reqInfo.setSourceApplicationId(0);
            reqInfo.setSourceContextId(0);
            reqInfo.setTargetApplicationId(APPID_APPHMI_MEDIA);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_MEDIA_PLAY);
            if (NULL != _contextSwitchHandler)
            {
               _contextSwitchHandler->onContextSwitchRequest(reqInfo);
            }
            break;
         }
         case APPID_APPHMI_TUNER:
         {
            ContextSwitchRequestInfo reqInfo;
            reqInfo.setRegionId(regionId);
            reqInfo.setTargetApplicationId(APPID_APPHMI_TUNER);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_TUNER_PLAY);
            if (NULL != _contextSwitchHandler)
            {
               _contextSwitchHandler->onContextSwitchRequest(reqInfo);
            }
            break;
         }
         case APPID_APPHMI_SXM:
         {
            ContextSwitchRequestInfo reqInfo;
            reqInfo.setRegionId(regionId);
            reqInfo.setTargetApplicationId(APPID_APPHMI_SXM);
            reqInfo.setTargetContextId(enActivityIDs__eActivityID_SXM_MAIN);
            if (NULL != _contextSwitchHandler)
            {
               _contextSwitchHandler->onContextSwitchRequest(reqInfo);
            }
            break;
         }
         default:
            break;
      }
   }
}


void AppFlyIn::performDefaultContextSwitchRequest(const uint32 listId, const uint32 regionId, const uint8 index)
{
   AppFlyInItemInfo* itrItem = getAppFlyInItemInfo(listId, regionId);
   if (itrItem != NULL)
   {
      for (AppFlyInItemInfo::iterator itrData = itrItem->begin(); itrData != itrItem->end(); ++itrData)
      {
         if ((*itrData).getItemIndex() == index)
         {
            ContextSwitchRequestInfo reqInfo;
            reqInfo.setRegionId(regionId);
            reqInfo.setTargetApplicationId((*itrData).getTargetAppId());
            reqInfo.setTargetContextId((*itrData).getTargetContext());
            if (NULL != _contextSwitchHandler)
            {
               _contextSwitchHandler->onContextSwitchRequest(reqInfo);
            }
         }
      }
   }
}


void AppFlyIn::onAppFlyInButtonStateUpdateRequest(const ::std::vector< AppFlyInStateInfo >& stateInfo)
{
   bool update = false;
   for (::std::vector< AppFlyInStateInfo >::const_iterator itrState = stateInfo.begin(); itrState != stateInfo.end(); ++itrState)
   {
      AppFlyInItemInfo* itrItem = getAppFlyInItemInfo(LISTID_MASTER_APPFLYIN, (*itrState).getRegionId());
      if (itrItem != NULL)
      {
         for (AppFlyInItemInfo::iterator itrData = itrItem->begin(); itrData != itrItem->end(); ++itrData)
         {
            if ((*itrData).getItemIndex() == (*itrState).getAppFlyInIndex())
            {
               if ((*itrData).getButtonState() != (*itrState).getStatus())
               {
                  (*itrData).setButtonState((*itrState).getStatus());
                  update = true;
               }
               ::std::string canderaPathBgImage = ((*itrState).getBgImage().empty()) ? "" : ::std::string("M1#Images#AppFlyIn#").append((*itrState).getBgImage());
               if ((*itrData).getDynamicBgImage() != canderaPathBgImage.c_str())
               {
                  (*itrData).setDynamicBgImage(::Candera::String(canderaPathBgImage.c_str(), canderaPathBgImage.length()));
                  update = true;
               }
               ::std::string canderaPathBorderImage = ((*itrState).getBorderImage().empty()) ? "" : ::std::string("M1#Images#AppFlyIn#").append((*itrState).getBorderImage());
               if ((*itrData).getDynamicBorderImage() != canderaPathBorderImage.c_str())
               {
                  (*itrData).setDynamicBorderImage(::Candera::String(canderaPathBorderImage.c_str(), canderaPathBorderImage.length()));
                  update = true;
               }
               ::std::string canderaPathIconImage = ((*itrState).getIconImage().empty()) ? "" : ::std::string("M1#Images#AppFlyIn#").append((*itrState).getIconImage());
               if ((*itrData).getDynamicIconImage() != canderaPathIconImage.c_str())
               {
                  (*itrData).setDynamicIconImage(::Candera::String(canderaPathIconImage.c_str(), canderaPathIconImage.length()));
                  update = true;
               }
               break;
            }
         }
      }
   }
   if ((_sceneStatus == SCENE_STATUS_ACTIVE) && (update == true))
   {
      ListRegistry::s_getInstance().updateList(LISTID_MASTER_APPFLYIN);
   }
}


void AppFlyIn::onActiveContextUpdate(const ::App::Core::ApplicationContextInfo& info)
{
   ListRegistry::s_getInstance().updateList(LISTID_MASTER_APPFLYIN);
   bool isValid = isValidContextForApplication(info.getApplicationId(), info.getContextId());
   ETG_TRACE_COMP(("AppFlyIn::onActiveContextUpdate::AppId::%d::ContextId::%d", info.getApplicationId(), info.getContextId()));
   if ((!isValid) || ((isValid) && ((NULL != _contextSwitchHandler) && (info.getApplicationId() == _contextSwitchHandler->getActiveApplicationId()))))
   {
      POST_MSG((COURIER_MESSAGE_NEW(AppFlyInReqMsg)(isValid)));
   }
}


void AppFlyIn::onApplicationSwitchComplete(const uint32 appId)
{
   ApplicationContextInfo contextInfo;
   if ((NULL != _contextSwitchHandler) && (_contextSwitchHandler->fetchActiveContextInfo(contextInfo)))
   {
      bool isValid = isValidContextForApplication(appId, contextInfo.getContextId());
      if ((appId == contextInfo.getApplicationId()) && (isValid))
      {
         POST_MSG((COURIER_MESSAGE_NEW(AppFlyInReqMsg)(isValid)));
      }
   }
}


void AppFlyIn::onCMAStatusUpdate(const SystemStatusInfo& info)
{
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      updateMapOutAppFlyInStatus((*itr));
   }
}


void AppFlyIn::onCMGStatusUpdate(const SystemStatusInfo& info)
{
   const RegionsIdInfo& tRegionsIdInfo = info.getRegionsIdInfo();
   for (RegionsIdInfo::const_iterator itr = tRegionsIdInfo.begin(); (itr != tRegionsIdInfo.end()); ++itr)
   {
      updateMapOutAppFlyInStatus((*itr));
   }
}


void AppFlyIn::updateMapOutAppFlyInStatus(const uint32 regionId)
{
   AppFlyInItemInfo* imp = getAppFlyInItemInfo(LISTID_MASTER_APPFLYIN, regionId);
   if ((NULL != imp) && (NULL != _regionHandling))
   {
      bool isValid = (NULL != _regionHandling) ? true : false;
      isValid      = ((isValid) && (_regionHandling->isAllStatusSame(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMA, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED))) ? true : false;
      isValid      = ((isValid) && (_regionHandling->isAnyStatusSame(SystemEnquiryInfo(regionId, SYSTEM_CATEGORY_CMG, STATUS_CATEGORY_SYSTEM), SYSTEM_STATUS_CONNECTED))) ? true : false;
      uint8 state  = (isValid) ? BUTTON_STATE_ENABLED : BUTTON_STATE_INVISIBLE;
      for (AppFlyInItemInfo::iterator itrData = imp->begin(); (itrData != imp->end()); ++itrData)
      {
         uint32 tempItemIndex = (*itrData).getItemIndex();
         if ((tempItemIndex == ITEM_MAPOUT || tempItemIndex == ITEM_CARMODE_MAPOUT) && ((*itrData).getButtonState() != state))
         {
            (*itrData).setButtonState(state);
            if ((_sceneStatus == SCENE_STATUS_ACTIVE) && (_regionHandling->getRegionId() == regionId))
            {
               ListRegistry::s_getInstance().updateList(LISTID_MASTER_APPFLYIN);
            }
            break;
         }
      }
   }
}


} //Core
} //App
