/* ***************************************************************************************
* FILE:          CameraHall.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  CameraHall.cpp is part of HMI-Base reference/demo/test applications
*    COPYRIGHT:  (c) 2018-2019 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "hall_std_if.h"
#include "CameraHall.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::CameraHall::
#include "trcGenProj/Header/CameraHall.cpp.trc.h"
#endif

#define CTRLBLOCK_NAME_CAMERA_CAMERA_LIST "CameraListData"

using namespace ::App::Core;
using namespace ::CourierTunnelService::CourierMessageReceiver;

namespace App {
namespace Core {

DEFINE_CLASS_LOGGER_AND_LEVEL("App/Core", CameraHall, Info);

CameraHall::CameraHall()
   : HallComponentBase("", "App.Core.AppHmi_Camera", "/org/genivi/NodeStateManager/LifeCycleConsumer/AppHmi_Camera"),
     _CameraMain(NULL),
     _cameraSourceChangeHandler(NULL),
     _vehicleClientHandler(NULL),
     _CameraConfig(NULL),
     _applicationSwitchClient(NULL),
     _CameraListHandler(NULL),
     _NanoMsgClientHandler(NULL),
     _CameraDataBindingHandler(NULL),
     _KDSHandler(NULL)
{
   ETG_I_REGISTER_FILE();
   ETG_TRACE_USR4(("CameraHall : Constructor"));
   _applicationSwitchClient = ApplicationSwitchClientComponent::poGetInstance();
   _pluginClientHandler = PluginClientHandler::getInstance();

   _CameraDataBindingHandler = CameraDataBindingHandler::poGetInstance();
   _CameraConfig = CameraConfig::getInstance();
   _cameraSourceChangeHandler = CameraSourceChangeHandler::getInstance();
   _vehicleClientHandler = VehicleClientHandler::getInstance();

   _CameraMain = CameraMain::poGetInstance();
   _KDSHandler = KDSHandler::poGetInstance();

   if (NULL != _pluginClientHandler)
   {
      std::vector<std::string> cameraCtrlBlockNames;
      cameraCtrlBlockNames.push_back(CTRLBLOCK_NAME_CAMERA_CONNECT);
      cameraCtrlBlockNames.push_back(CTRLBLOCK_NAME_CAMERA_CAMERA_LIST);
      cameraCtrlBlockNames.push_back(CTRLBLOCK_NAME_CAMERA_SPLITVIEW_SEL);
      std::map<std::string, std::vector<std::string> > pluginCtrlBlockMap;
      pluginCtrlBlockMap.insert(std::pair<std::string, std::vector<std::string> >(PLUGIN_NAME_CAMERA, cameraCtrlBlockNames));
      _pluginClientHandler->registerPluginSignals(pluginCtrlBlockMap);
   }
   _CameraListHandler = CameraListHandler::getInstance();
   _NanoMsgClientHandler = NanoMsgClientHandler::poGetInstance();

   if (_applicationSwitchClient != NULL)
   {
      _applicationSwitchClient->setApplicationID(APPID_APPHMI_CAMERA);
   }
   _CameraAvdeccClientHandler = CameraAvdeccClientHandler::getInstance();
   (void)NavigationModeDBHandler::getInstance();
}


CameraHall::~CameraHall()
{
   ETG_TRACE_USR4(("CameraHall() : Class Destructor"));

   if (_applicationSwitchClient != NULL)
   {
      delete _applicationSwitchClient;
      _applicationSwitchClient = NULL;
   }

   if (_CameraMain != NULL)
   {
      delete _CameraMain;
      _CameraMain = NULL;
   }

   if (_cameraSourceChangeHandler != NULL)
   {
      delete _cameraSourceChangeHandler;
      _cameraSourceChangeHandler = NULL;
   }

   if (_vehicleClientHandler != NULL)
   {
      delete _vehicleClientHandler;
      _vehicleClientHandler = NULL;
   }

   if (_CameraConfig != NULL)
   {
      delete _CameraConfig;
      _CameraConfig = NULL;
   }
   if (_CameraListHandler != NULL)
   {
      delete _CameraListHandler;
      _CameraListHandler = NULL;
   }
   if (_NanoMsgClientHandler != NULL)
   {
      delete _NanoMsgClientHandler;
      _NanoMsgClientHandler = NULL;
   }
   if (_CameraDataBindingHandler != NULL)
   {
      delete _CameraDataBindingHandler;
      _CameraDataBindingHandler = NULL;
   }
   if (_KDSHandler != NULL)
   {
      delete _KDSHandler;
      _KDSHandler = NULL;
   }
   if (_CameraAvdeccClientHandler != NULL)
   {
      delete _CameraAvdeccClientHandler;
      _CameraAvdeccClientHandler = NULL;
   }
}


bool CameraHall::onCourierMessage(const onHardKeyPress& msg)
{
   int hardKeyCode = msg.GetHKCode();
   int hardkeyState = msg.GetHKState();
   int retVal = false;
   int _speedStatus = 0;
   bool camStatus = CameraListHandler::getInstance()->getCockpitCameraIndex();
   int cabinIndex = CameraListHandler::getInstance()->getCabinListsize();
   uint8 m_u8activeRegion =  CameraMain::poGetInstance()->getRegionValue();
   bool  spmHmiState = CameraSourceChangeHandler::getInstance()->getSpmHourLogicState();
   bool cmaPnmStatus = CameraSourceChangeHandler::getInstance()->getCMApnmStatus();

   ETG_TRACE_USR4(("CameraHall::onHardKeyPress HKCode GetHKState cmaPnmStatus spmHmiState : %d %d %d %d", msg.GetHKCode(), msg.GetHKState(), cmaPnmStatus, spmHmiState));
   ETG_TRACE_USR4(("CameraHall::onHardKeyPress camStatus cabinIndex m_u8activeRegion sm_gRVCStatus sm_gCMPStatus: %d %d %d %d %d", camStatus, cabinIndex, m_u8activeRegion, sm_gRVCStatus, sm_gCMPStatus));
   if ((sm_gRVCStatus == 0) && (sm_gCMPStatus == 1)) // Allow HK action, only when RVC is inactive.
   {
      if (HARDKEYCODE_HK_CAMERA == hardKeyCode)
      {
         if (m_u8activeRegion == REGION_C)
         {
            ETG_TRACE_USR4(("CameraHall::onHardKeyPress key camval = %d", camStatus));
            if (camStatus != true)
            {
               ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(REGION_C, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_SOURCE_SELECTION);
            }
            else
            {
               int selectedIndex = CameraListHandler::getInstance()->getSelectedCockpitCameraIndex();
               (selectedIndex == -1) ? selectedIndex = 0 : selectedIndex = selectedIndex;
               bool speedAndParkStatus = VehicleClientHandler::getInstance()->getSpeedAndParkingStatus();
               vector<int> camTextId;
               camTextId = CameraListHandler::getInstance()->getAvailableCameraTextIdOnSelection();   // GET the text id from the list of available cameras.
               if (camTextId.size() > 0)
               {
                  if (camTextId.at(selectedIndex) == -1)
                  {
                     POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupNotConfigMsg)()));
                     return true;
                  }
                  CameraConfig::getInstance()-> getSpeedStatus(camTextId.at(selectedIndex), _speedStatus); // Check speed restriction is applicable to cockpit only or cabin only or both
                  bool bSpeedAndParkBrakeStatusValidity = VehicleClientHandler::getInstance()->bGetSpeedAndParkBrakeValidityStatus();
                  //Added validity check for speed and parking brake status. Bug ID 1267971
                  if (bSpeedAndParkBrakeStatusValidity == SPEED_AND_PARKBRAKE_STATUS_VALID)
                  {
                     if (speedAndParkStatus == ALLOW_CAMERA)   // no restrictions. Allow all cameras. ie, speed is 0
                     {
                        CameraListHandler::getInstance()->vCockpitListSelection(selectedIndex, HOMESCREEN_TILE);
                     }
                     if (speedAndParkStatus == ALLOW_PARTIAL_OR_BLOCK_CAMERA) 	// restrictions on few cameras. ie, speed is > 0
                     {
                        // None, cp, cb. cpcb
                        if ((_speedStatus == COCKPIT_ONLY) || (_speedStatus == COCKPIT_CABIN))   // ignore speed.
                        {
                           CameraListHandler::getInstance()->vCockpitListSelection(selectedIndex, HOMESCREEN_TILE);
                        }
                        else
                        {
                           POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
                        }
                     }
                  }
                  else
                  {
                     POST_MSG((COURIER_MESSAGE_NEW(ActivatePopupMsg)()));
                  }
               }
            }
         }
         // cmaPnmStatus Condition is added to fix Bug 1290363: Swivel able to open when cabin camera source is activated during PNM OFF
         else if ((m_u8activeRegion == REGION_A) && (cabinIndex > 0) && (spmHmiState == HOURLOGIC_DISABLED) && (cmaPnmStatus == PNM_CMA_DISABLED))
         {
            ETG_TRACE_USR4(("CameraHall::onHardKeyPress Camera Cabin Region Context Switch"));
            ApplicationSwitchClientComponent::poGetInstance()->requestContextSwitchApplication(REGION_A, APPID_APPHMI_MASTER, enActivityIDs__eActivityID_HOME, APPID_APPHMI_CAMERA, enActivityIDs__eActivityID_CAMERA_CABIN_SRCSELECT);
         }
      }
   }
   return retVal;
}


void CameraHall::onExpired(asf::core::Timer& /*timer*/, boost::shared_ptr<asf::core::TimerPayload> /*data*/)
{
   ETG_TRACE_USR4(("CameraHall::onExpired()"));
}


/**
* Description     : This method will be called on system startup for language translation
*
* @return         : uint8
*/
uint8 CameraHall::readLanguageOnStartUp()
{
   return 0;
}


bool CameraHall::onCourierMessage(const SwitchApplicationReq& oMsg)
{
   ETG_TRACE_USR3(("CameraHall->onCourierMessage:SwitchApplicationReq()"));

   if (_applicationSwitchClient != NULL)
   {
      _applicationSwitchClient->requestSwitchApplication(APPID_APPHMI_MASTER);
   }
   else
   {
      ETG_TRACE_USR3(("CameraHall->Null pointer"));
   }
   return true;
}


bool CameraHall::onCourierMessage(const NavigationCarModeInfo& oMsg)
{
   ETG_TRACE_USR4(("CameraHall::onCourierMessage carMode :%d coachMode :%d", oMsg.GetCarMode(),  oMsg.GetCoachMode()));

   (*_carModeVisibilityStatus).mVisibleStatus = oMsg.GetCarMode();
   _carModeVisibilityStatus.MarkItemModified(ItemKey::CarModeVisibility::VisibleStatusItem);
   _carModeVisibilityStatus.SendUpdate(true);
   return true;
}


} // namespace Core
} // namespace App
