/**************************************************************************************
* @file         : CameraConfig.cpp
* @author       : ECG5- Shanmugapriya
* @addtogroup   :
* @brief        : Class contains necessary business logic required to handle Camera config data.
* @copyright    : (c) 2019-2020 Robert Bosch Car Multimedia GmbH
*                 The reproduction, distribution and utilization of this file as
*                 well as the communication of its contents to others without express
*                 authorization is prohibited. Offenders will be held liable for the
*                 payment of damages. All rights reserved in the event of the grant
*                 of a patent, utility model or design.
**************************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#include "hall_std_if.h"
#include "CameraMain/CameraMain.h"

#include "CameraConfig.h"
#include <iostream>
#include <bits/stdc++.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_APPHMI_CAMERA_HALL
#define ETG_I_TRACE_CHANNEL               TR_TTFIS_APPHMI_CAMERA
#define ETG_I_TTFIS_CMD_PREFIX            "APPHMI_Camera_"
#define ETG_I_FILE_PREFIX                 App::Core::CameraConfig::
#include "trcGenProj/Header/CameraConfig.cpp.trc.h"
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
#include "CameraConfig.h"
#include "CameraDefines.h"
#include "utest/stubs/AudioDefines.h"
#include <iostream>
#include <bits/stdc++.h>
#endif

namespace App {
namespace Core {

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
stCamportSrcMapping camportSrcMap[] =
{
   { CMP1_CAM1, CAMPORT1_CAM1 },
   { CMP1_CAM2, CAMPORT1_CAM2 },
   { CMP1_CAM3, CAMPORT1_CAM3 },
   { CMP1_CAM4, CAMPORT1_CAM4 }
};


stCamConfigInfo cameraConfigInfo;

stCamSrcMapping camSrcMap[] =
{
   { "CMP1_CAM1", SRC_CAMERAPORT1_CAM1 },
   { "CMP1_CAM2", SRC_CAMERAPORT1_CAM2 },
   { "CMP1_CAM3", SRC_CAMERAPORT1_CAM3 },
   { "CMP1_CAM4", SRC_CAMERAPORT1_CAM4 },
   { "CMP1_SPLITSCREEN", SRC_CAMERAPORT1_SPLITSCREEN},
   { "CMP1_IPCAM", SRC_CAMERAPORT1_IPCAM },
   { "CMP2_CAM1", SRC_CAMERAPORT2_CAM1},
   { "CMP2_CAM2", SRC_CAMERAPORT2_CAM2},
   { "CMP2_CAM3", SRC_CAMERAPORT2_CAM3},
   { "CMP2_CAM4", SRC_CAMERAPORT2_CAM4},
   { "CMP2_SPLITSCREEN", SRC_CAMERAPORT2_SPLITSCREEN},
   { "CMP2_IPCAM" 	, SRC_CAMERAPORT2_IPCAM},
   { "CMP3_CAM1", SRC_CAMERAPORT3_CAM1},
   { "CMP3_CAM2", SRC_CAMERAPORT3_CAM2},
   { "CMP3_CAM3", SRC_CAMERAPORT3_CAM3},
   { "CMP3_CAM4", SRC_CAMERAPORT3_CAM4},
   { "CMP3_SPLITSCREEN", SRC_CAMERAPORT3_SPLITSCREEN},
   { "CMP3_IPCAM", SRC_CAMERAPORT3_IPCAM},
   { "CMP4_CAM1", SRC_CAMERAPORT4_CAM1},
   { "CMP4_CAM2", SRC_CAMERAPORT4_CAM2},
   { "CMP4_CAM3", SRC_CAMERAPORT4_CAM3},
   { "CMP4_CAM4", SRC_CAMERAPORT4_CAM4},
   {"CMP4_SPLITSCREEN", SRC_CAMERAPORT4_SPLITSCREEN},
};


stCamSrcNameToCANIndex camSrcNameToIndex[] =
{
   { "CMP1_CAM1", CAM1},
   { "CMP1_CAM2", CAM2},
   { "CMP1_CAM3", CAM3},
   { "CMP1_CAM4", CAM4},
   { "CMP2_CAM1", CAM4},
   { "CMP2_CAM2", CAM5},
   { "CMP2_CAM3", CAM6},
   { "CMP2_CAM4", CAM7},
   { "CMP3_CAM1", CAM7},
   { "CMP3_CAM2", CAM8},
   { "CMP3_CAM3", CAM9},
   { "CMP3_CAM4", CAM10},
};


void CameraConfig::setMappingTextID()
{
   //ID, Translation, ActualTextIsShown ,SHOW WHILE DRIVING, HMI BUTTON AVAILABLE IN REGION
   ETG_TRACE_USR4(("CameraTextNameMapping::setMappingTextID()"));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(0, {TextId_0x150B, "Camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(1, {TextId_0x150C, "Camera 1", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(2, {TextId_0x150D, "Camera 2", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(3, {TextId_0x150E, "Camera 3", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(4, {TextId_0x150F, "Camera 4", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(5, {TextId_0x1510, "Camera 5", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(6, {TextId_0x1511, "Camera 6", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(7, {TextId_0x1512, "Camera 7", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(8, {TextId_0x1513, "Video", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(9, {TextId_0x1514, "Video 1", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(10, {TextId_0x1515, "Video 2", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(11, {TextId_0x1516, "Video 3", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(12, {TextId_0x1517, "Video 4", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(13, {TextId_0x1518, "Video 5", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(14, {TextId_0x1519, "Video 6", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(15, {TextId_0x151A, "Video 7", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(16, {TextId_0x151B, "Channel 1", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(17, {TextId_0x151C, "Channel 2", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(18, {TextId_0x151D, "Channel 3", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(19, {TextId_0x151E, "Channel 4", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(20, {TextId_0x151F, "Channel 5", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(21, {TextId_0x1520, "Channel 6", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(22, {TextId_0x1521, "Channel 7", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(23, {TextId_0x1522, "Exterior camera", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(24, {TextId_0x1523, "Exterior camera 1", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(25, {TextId_0x1524, "Exterior camera 2", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(26, {TextId_0x1525, "Exterior camera 3", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(27, {TextId_0x1526, "Exterior camera 4", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(28, {TextId_0x1527, "Exterior camera 5", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(29, {TextId_0x1528, "Exterior camera 6", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(30, {TextId_0x1529, "Exterior camera 7", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(31, {TextId_0x152A, "Exterior camera, front", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(32, {TextId_0x152B, "Exterior camera, rear", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(33, {TextId_0x152C, "Exterior camera, left", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(34, {TextId_0x152D, "Exterior camera, right", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(35, {TextId_0x152E, "Reversing camera", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(36, {TextId_0x152F, "Corner cam. rear right", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(37, {TextId_0x1530, "Corner camera, rear left", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(38, {TextId_0x1531, "Corner cam. front left", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(39, {TextId_0x1532, "Corner cam. front right", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(40, {TextId_0x1533, "Mirror camera, left", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(41, {TextId_0x1534, "Mirror camera, right", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(42, {TextId_0x1535, "Mirror camera, front", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(43, {TextId_0x1536, "Lug. comp. cam. left", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(44, {TextId_0x1537, "Lug. comp. cam. right", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(45, {TextId_0x1538, "Roof camera", NONE, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(46, {TextId_0x1539, "Camera door", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(47, {TextId_0x153A, "Camera door 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(48, {TextId_0x153B, "Camera door 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(49, {TextId_0x153C, "Camera door 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(50, {TextId_0x153D, "Camera door 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(51, {TextId_0x153E, "Camera door 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(52, {TextId_0x153F, "Camera door 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(53, {TextId_0x1540, "Camera door 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(54, {TextId_0x15AF, "Entrance camera", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(55, {TextId_0x15B0, "Entrance camera 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(56, {TextId_0x15B1, "Entrance camera 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(57, {TextId_0x15B2, "Entrance camera 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(58, {TextId_0x15B3, "Entrance camera 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(59, {TextId_0x15B4, "Entrance camera 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(60, {TextId_0x15B5, "Entrance camera 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(61, {TextId_0x15B6, "Entrance camera 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(62, {TextId_0x1544, "Interior camera", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(63, {TextId_0x1545, "Interior camera 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(64, {TextId_0x1546, "Interior camera 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(65, {TextId_0x1547, "Interior camera 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(66, {TextId_0x1548, "Interior camera 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(67, {TextId_0x1549, "Interior camera 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(68, {TextId_0x154A, "Interior camera 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(69, {TextId_0x154B, "Interior camera 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(70, {TextId_0x154C, "Interior camera, front", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(71, {TextId_0x154D, "Interior camera, front 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(72, {TextId_0x154E, "Interior camera, front 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(73, {TextId_0x154F, "Interior camera, front 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(74, {TextId_0x1550, "Interior camera, front 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(75, {TextId_0x1551, "Interior camera, front 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(76, {TextId_0x1552, "Interior camera, front 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(77, {TextId_0x1553, "Interior camera, front 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(78, {TextId_0x1554, "Interior camera, centre", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(79, {TextId_0x1555, "Interior cam. centre 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(80, {TextId_0x1556, "Interior cam. centre 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(81, {TextId_0x1557, "Interior cam. centre 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(82, {TextId_0x1558, "Interior cam. centre 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(83, {TextId_0x1559, "Interior cam. centre 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(84, {TextId_0x155A, "Interior cam. centre 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(85, {TextId_0x155B, "Interior cam. centre 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(86, {TextId_0x155C, "Interior camera, rear", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(87, {TextId_0x155D, "Interior camera, rear 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(88, {TextId_0x155E, "Interior camera, rear 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(89, {TextId_0x155F, "Interior camera, rear 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(90, {TextId_0x1560, "Interior camera, rear 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(91, {TextId_0x1561, "Interior camera, rear 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(92, {TextId_0x1562, "Interior camera, rear 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(93, {TextId_0x1563, "Interior camera, rear 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(94, {TextId_0x1564, "Upper deck camera", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(95, {TextId_0x1565, "Upper deck camera 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(96, {TextId_0x1566, "Upper deck camera 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(97, {TextId_0x1567, "Upper deck camera 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(98, {TextId_0x1568, "Upper deck camera 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(99, {TextId_0x1569, "Upper deck camera 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(100, {TextId_0x156A, "Upper deck camera 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(101, {TextId_0x156B, "Upper deck camera 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(102, {TextId_0x156C, "Upper deck cam. front", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(103, {TextId_0x15BE, "Upper deck cam. front 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(104, {TextId_0x15BF, "Upper deck cam. front 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(105, {TextId_0x15C0, "Upper deck cam. front 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(106, {TextId_0x15C1, "Upper deck cam. front 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(107, {TextId_0x15C2, "Upper deck cam. front 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(108, {TextId_0x15C3, "Upper deck cam. front 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(109, {TextId_0x15C4, "Upper deck cam. front 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(110, {TextId_0x156D, "Upper deck cam. centre", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(111, {TextId_0x15C5, "Upper deck cam. cent. 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(112, {TextId_0x15C6, "Upper deck cam. cent. 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(113, {TextId_0x15C7, "Upper deck cam. cent. 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(114, {TextId_0x15C8, "Upper deck cam. cent. 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(115, {TextId_0x15C9, "Upper deck cam. cent. 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(116, {TextId_0x15CA, "Upper deck cam. cent. 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(117, {TextId_0x15CB, "Upper deck cam. cent. 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(118, {TextId_0x156E, "Upper deck cam. rear", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(119, {TextId_0x15CC, "Upper deck cam. rear 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(120, {TextId_0x15CD, "Upper deck cam. rear 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(121, {TextId_0x15CE, "Upper deck cam. rear 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(122, {TextId_0x15CF, "Upper deck cam. rear 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(123, {TextId_0x15D0, "Upper deck cam. rear 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(124, {TextId_0x15D1, "Upper deck cam. rear 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(125, {TextId_0x15D2, "Upper deck cam. rear 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(126, {TextId_0x156F, "Lower deck camera", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(127, {TextId_0x1570, "Lower deck camera 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(128, {TextId_0x1571, "Lower deck camera 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(129, {TextId_0x1572, "Lower deck camera 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(130, {TextId_0x1573, "Lower deck camera 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(131, {TextId_0x1574, "Lower deck camera 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(132, {TextId_0x1575, "Lower deck camera 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(133, {TextId_0x1576, "Lower deck camera 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(134, {TextId_0x1577, "Lower deck cam. front", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(135, {TextId_0x15D3, "Lower deck cam. front 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(136, {TextId_0x15D4, "Lower deck cam. front 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(137, {TextId_0x15D5, "Lower deck cam. front 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(138, {TextId_0x15D6, "Lower deck cam. front 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(139, {TextId_0x15D7, "Lower deck cam. front 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(140, {TextId_0x15D8, "Lower deck cam. front 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(141, {TextId_0x15D9, "Lower deck cam. front 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(142, {TextId_0x1578, "Lower deck cam. centre", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(143, {TextId_0x15DA, "Lower deck cam. cent. 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(144, {TextId_0x15DB, "Lower deck cam. cent. 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(145, {TextId_0x15DC, "Lower deck cam. cent. 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(146, {TextId_0x15DD, "Lower deck cam. cent. 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(147, {TextId_0x15DE, "Lower deck cam. cent. 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(148, {TextId_0x15DF, "Lower deck cam. cent. 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(149, {TextId_0x15E0, "Lower deck cam. cent. 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(150, {TextId_0x1579, "Lower deck cam. rear", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(151, {TextId_0x15E1, "Lower deck cam. rear 1", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(152, {TextId_0x15E2, "Lower deck cam. rear 2", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(153, {TextId_0x15E3, "Lower deck cam. rear 3", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(154, {TextId_0x15E4, "Lower deck cam. rear 4", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(155, {TextId_0x15E5, "Lower deck cam. rear 5", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(156, {TextId_0x15E6, "Lower deck cam. rear 6", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(157, {TextId_0x15E7, "Lower deck cam. rear 7", COCKPIT_CABIN, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(158, {TextId_0x157A, "Panoramic camera", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(159, {TextId_0x15E8, "Split screen view", NONE, BOTH}));

   cameraNamefromConfig.insert(pair<int, stCameraNames>(160, {TextId_0x15EA, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(161, {TextId_0x15EB, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(162, {TextId_0x15EC, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(163, {TextId_0x15ED, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(164, {TextId_0x15EE, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(165, {TextId_0x15EF, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(166, {TextId_0x15F0, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(167, {TextId_0x15F1, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(168, {TextId_0x15E9, "Quad screen view", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(169, {TextId_0x15F2, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(170, {TextId_0x15F3, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(171, {TextId_0x15F4, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(172, {TextId_0x15F5, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(173, {TextId_0x15F6, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(174, {TextId_0x15F7, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(175, {TextId_0x15F8, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(176, {TextId_0x1E00, "Luggage [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(177, {TextId_0x1E01, "Doors [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(178, {TextId_0x1E02, "Doors 1 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(179, {TextId_0x1E03, "Doors 2 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(180, {TextId_0x1E04, "Doors 3 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(181, {TextId_0x1E05, "Gangway [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(182, {TextId_0x1E06, "Gangway 1 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(183, {TextId_0x1E07, "Gangway 2 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(184, {TextId_0x1E08, "Gangway 3 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(185, {TextId_0x1E09, "Split screen [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(186, {TextId_0x1E0A, "Split screen 1 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(187, {TextId_0x1E0B, "Split screen 2 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(188, {TextId_0x1E0C, "Split screen 3 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(189, {TextId_0x1E0D, "Split screen 4 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(190, {TextId_0x1E0E, "Split screen 5 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(191, {TextId_0x1E0F, "Split screen 6 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(192, {TextId_0x1E10, "Split screen 7 [2:1]", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(193, {TextId_0x1E11, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(194, {TextId_0x1E12, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(195, {TextId_0x1E13, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(196, {TextId_0x1E14, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(197, {TextId_0x1E15, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(198, {TextId_0x1E16, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(199, {TextId_0x1E17, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(200, {TextId_0x1E18, "Exterior 1", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(201, {TextId_0x1E19, "Exterior 2", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(202, {TextId_0x1E1A, "Exterior 3", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(203, {TextId_0x1E1B, "Exterior 4", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(204, {TextId_0x1E1C, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(205, {TextId_0x1E1D, "Interior 1", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(206, {TextId_0x1E1E, "Interior 2", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(207, {TextId_0x1E1F, "Interior 3", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(208, {TextId_0x1E20, "Interior 4", COCKPIT_CABIN, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(209, {TextId_0x1E21, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(210, {TextId_0x1E22, "Exterior for driver 1", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(211, {TextId_0x1E23, "Exterior for driver 2", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(212, {TextId_0x1E24, "Exterior for driver 3", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(213, {TextId_0x1E25, "Luggage left", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(214, {TextId_0x1E26, "Luggage right", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(215, {TextId_0x1E27, "Interior for driver 1", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(216, {TextId_0x1E28, "Interior for driver 2", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(217, {TextId_0x1E29, "Interior for driver 3", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(218, {TextId_0x1E2A, "Interior for driver 4", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(219, {TextId_0x1E2B, "Interior for driver 5", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(220, {TextId_0x1E2C, "Door front", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(221, {TextId_0x1E2D, "Door centre", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(222, {TextId_0x1E2E, "Door back", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(223, {TextId_0x1E2F, "Upper deck", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(224, {TextId_0x1E30, "Lower deck", COCKPIT_ONLY, COCKPIT}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(225, {TextId_0x1E31, "Scenic view 1", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(226, {TextId_0x1E32, "Scenic view 2", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(227, {TextId_0x1E33, "Scenic view 3", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(228, {TextId_0x1E34, "Scenic view 4", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(229, {TextId_0x1E35, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(230, {TextId_0x1E36, "Announcement", CABIN_ONLY, CABIN}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(231, {TextId_0x1E37, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(232, {TextId_0x1E38, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(233, {TextId_0x1E39, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(234, {TextId_0x1E3A, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(235, {TextId_0x1E3B, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(236, {TextId_0x1E3C, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(237, {TextId_0x1E3D, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(238, {TextId_0x1E3E, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(239, {TextId_0x1E3F, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(240, {TextId_0x1E40, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(241, {TextId_0x1E41, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(242, {TextId_0x1E42, "Camera port 2", NONE, 0})); //CID_R2 11673328
   cameraNamefromConfig.insert(pair<int, stCameraNames>(243, {TextId_0x1E43, "Camera port 3", NONE, 0})); //CID_R2 11673328
   cameraNamefromConfig.insert(pair<int, stCameraNames>(244, {TextId_0x1E44, "Camera port 4", NONE, 0})); //CID_R2 11673328
   cameraNamefromConfig.insert(pair<int, stCameraNames>(245, {TextId_0x1E45, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(246, {TextId_0x1E46, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(247, {TextId_0x1E47, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(248, {TextId_0x1E48, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(249, {TextId_0x1E49, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(250, {TextId_0x1E4A, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(251, {TextId_0x1E4B, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(252, {TextId_0x1E4C, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(253, {TextId_0x1E4D, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(254, {TextId_0x1E4E, "Unknown camera", NONE, BOTH}));
   cameraNamefromConfig.insert(pair<int, stCameraNames>(255, {TextId_0x1E4F, "No signal", NONE, BOTH}));

   //printMappingTable();
}


/******************************************************************************
*NAME        : CameraConfig
*DESIGN SECTION: 6.1.5
*DESCRIPTION : constructor , reads the config file at startup
******************************************************************************/
CameraConfig::CameraConfig()
{
   ETG_TRACE_USR4(("CameraConfig::CameraConfig()"));
   setMappingTextID();
}


/******************************************************************************
*NAME        : ~CameraConfig
*DESIGN SECTION: 6.1.5
*DESCRIPTION : destructor
******************************************************************************/
CameraConfig::~CameraConfig()
{
   ETG_TRACE_USR4(("CameraConfig::~CameraConfig()"));
}


/******************************************************************************
*NAME        : getInstance
*DESIGN SECTION: 6.1.5
*DESCRIPTION : returns singleton object of class CameraConfig
******************************************************************************/
CameraConfig* CameraConfig::getInstance()
{
   ETG_TRACE_USR4(("CameraConfig::getInstance()"));
   static CameraConfig cameraConfig;

   return &cameraConfig;
}


/******************************************************************************
*NAME        : getCabinSrcIdForCam
*DESIGN SECTION: 6.1.5
*DESCRIPTION : function to retrieve corresponding srcID from Audiodefines.h

** example
camsource CMP1_IPCAM of std::string  whose sourceId is 417(SRC_CAMERAPORT1_IPCAM)

******************************************************************************/
_Src CameraConfig::getCabinSrcIdForCam(std::string cameraSelSrc)
{
   ETG_TRACE_USR4(("_Src CameraConfig::getCabinSrcIdForCam(cameraSelSrc %s)", cameraSelSrc));

   _Src currSrc = SRC_INVALID;
   for (uint8 u8Index = 0; u8Index < (sizeof(camSrcMap) / sizeof(stCamSrcMapping)); u8Index++)
   {
      if (cameraSelSrc == (camSrcMap[u8Index].cam))
      {
         currSrc = camSrcMap[u8Index].camSrc;
      }
   }
   ETG_TRACE_USR4(("_Src CameraConfig::getCabinSrcIdForCam (currSrc %d)", currSrc));
   return currSrc;
}


/******************************************************************************
*NAME        :getCockpitSrcIdForCam
*DESIGN SECTION: 6.1.5
*DESCRIPTION :returns the src id for cockpit cam based on the input value
******************************************************************************/
inline uint8 CameraConfig::getCockpitSrcIdForCam(uint8 value)
{
   ETG_TRACE_USR4(("CameraConfig::getCockpitSrcIdForCam(%d)", value));

   uint8 curSrc = -1;
   for (uint8 u8Index = 0; u8Index < (sizeof(camportSrcMap) / sizeof(stCamportSrcMapping)); u8Index++)
   {
      if (value == (uint8)(camportSrcMap[u8Index].cam))
      {
         curSrc = camportSrcMap[u8Index].camportSrc;
      }
   }
   ETG_TRACE_USR4(("SrcIdForCam(%d) is (%d)", value, curSrc));
   return curSrc;
}


//based on key or cantextid --> we get translation text id  example TextId_0x150B
void CameraConfig::getCameraValues(int key, uint32& value)
{
   ETG_TRACE_USR4(("CameraConfig::getCameraValues for %d", key));
   map<int, stCameraNames>::iterator itr;

   if ((key >= CAMERA_RANGE_START) && (key < CAMERA_RANGE_MAX_VALID))
   {
      // for (int i = 0; i < n; i++)
      // {
      for (itr = cameraNamefromConfig.begin(); itr != cameraNamefromConfig.end(); ++itr)
      {
         //if (itr->first == keys[i])
         if (itr->first == key)
         {
            //value[i] = itr->second.PropertyName;
            value = itr->second.PropertyName;
            ETG_TRACE_USR3(("getCameraValues(): Value is  [%u]", value));
            break;
         }
      }
      // }
   }
   else
   {
      ETG_TRACE_USR3(("getCameraValues(): Out of range"));
      value = 0xFF;
   }
}


/******************************************************************************
*NAME        :getCameraConfigCount
*DESIGN SECTION: 6.1.5
*DESCRIPTION :Returns the no. of camera configured to each profile
******************************************************************************/
stCamConfigInfo CameraConfig::getCameraConfigStruct()
{
   ETG_TRACE_USR4(("CameraConfig::getCameraConfigCount()"));

   return cameraConfigInfo;
}


//Check speed restriction is applicable to the camera index or not
//int-->ID(key), uint32--> Translation, string-->"name",int-->speedStatus-SHOW WHILE DRIVING, int-->regionStatus-HMI BUTTON AVAILABLE IN REGION
void CameraConfig::getSpeedStatus(uint32 key, int& _speedStatus)
{
   ETG_TRACE_USR4(("CameraConfig::getSpeedStatus for camera Text Id( %d)", key));
   map<int, stCameraNames>::iterator itr;
   for (itr = cameraNamefromConfig.begin(); itr != cameraNamefromConfig.end(); ++itr)
   {
      if (itr->first == key)
      {
         _speedStatus = itr -> second.speedStatus;
         ETG_TRACE_USR4(("CameraConfig::getSpeedStatus() = %d", _speedStatus));
      }
   }
}


void CameraConfig::getRegionStatus(uint32 key, int& _regionStatus)
{
   ETG_TRACE_USR4(("CameraConfig::getRegionStatus for camera Text Id( %d)", key));
   map<int, stCameraNames>::iterator itr;
   for (itr = cameraNamefromConfig.begin(); itr != cameraNamefromConfig.end(); ++itr)
   {
      if (itr->first == key)
      {
         _regionStatus = itr -> second.regionStatus;
         ETG_TRACE_USR4(("CameraConfig::getRegionStatus() = %d", _regionStatus));
      }
   }
}


/******************************************************************************
*NAME        : getCabinSrcNameForCamId
*DESIGN SECTION: 6.1.5
*DESCRIPTION : function to retrieve corresponding camport source from source id
*****************************************************************************/
std::string CameraConfig::getCabinSrcNameForCamId(int activeCamera)
{
   ETG_TRACE_USR4(("std::string CameraConfig::getCabinSrcNameForCamId activeCamera==>(%d)", activeCamera));

   std::string camSrcName = "";
   for (uint8 u8Index = 0; u8Index < (sizeof(camSrcMap) / sizeof(stCamSrcMapping)); u8Index++)
   {
      if (activeCamera == (camSrcMap[u8Index].camSrc))
      {
         camSrcName = camSrcMap[u8Index].cam;
      }
   }
   ETG_TRACE_USR4(("std::string CameraConfig::getCabinSrcNameForCamId camSrcName==>(%s)", camSrcName.c_str()));
   return camSrcName;
}


/******************************************************************************
*NAME        : getCabinSrcNameForCamId
*DESIGN SECTION: 6.1.5
*DESCRIPTION : function to retrieve corresponding srcID from Audiodefines.h
******************************************************************************/
int CameraConfig::getCANIndexFromCamSrcName(std::string camSrcName)
{
   ETG_TRACE_USR4(("CameraConfig::getCANIndexFromCamSrcName(camSrcName==>%s)", camSrcName.c_str()));
   int index = 0;
   for (uint8 u8Index = 0; u8Index < (sizeof(camSrcNameToIndex) / sizeof(stCamSrcNameToCANIndex)); u8Index++)
   {
      if (camSrcName == (camSrcNameToIndex[u8Index].camSrcName))
      {
         index = camSrcNameToIndex[u8Index].cam;
      }
   }
   ETG_TRACE_USR4(("CameraConfig::getCANIndexFromCamSrcName (CAN Index==>%d)", index));
   return index;
}


#endif

//Unit test Compiler switch filter.

#ifdef VARIANT_S_FTR_ENABLE_UNITTEST
stCamportSrcMapping camportSrcMap[] =
{
   { CMP1_CAM1, CAMPORT1_CAM1 },
   { CMP1_CAM2, CAMPORT1_CAM2 },
   { CMP1_CAM3, CAMPORT1_CAM3 },
   { CMP1_CAM4, CAMPORT1_CAM4 }
};


int CameraConfig::getCockpitSrcIdForCamTest(int value)
{
   // ETG_TRACE_USR4(("CameraConfig::getCockpitSrcIdForCamTest(%d)", value));

   int curSrc = -1;
   for (int u8Index = 0; u8Index < (sizeof(camportSrcMap) / sizeof(stCamportSrcMapping)); u8Index++)
   {
      if (value == (int)(camportSrcMap[u8Index].cam))
      {
         curSrc = camportSrcMap[u8Index].camportSrc;
      }
   }
   //ETG_TRACE_USR4(("SrcIdForCamTest(%d) is (%d)", value, curSrc));
   return curSrc;
}


#endif
} // namespace Core
} // namespace App
