/* ***************************************************************************************
* FILE:          IHMIAlert_Daemon_Plugin.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IHMIAlert_Daemon_Plugin.h is part of HMI-Base delivery
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#ifndef IHMIALERT_DAEMON_PLUGIN_H
#define IHMIALERT_DAEMON_PLUGIN_H

#include <map>
#include <string>
#include <stdint.h>

namespace hmibase {

namespace alertdaemon {

/// Identification enumeration for all supported plugins
namespace PluginId {
enum Enum
{
   /// First plugin ID
   First = 0,

   ///
   Renderer = First,

   /// Last plugin ID
   Last = Renderer,

   /// Number of supported plugins
   Count = Last + 1
};


}

/**
 * @brief The IHMIAlert_Daemon_Plugin class defines the base interace every derived alert daemon plugin must comply.
 */
class IHMIAlert_Daemon_Plugin
{
   public:
      /// Plugin creation function type
      typedef IHMIAlert_Daemon_Plugin& Create();

      ///
      virtual ~IHMIAlert_Daemon_Plugin() {}

      /**
       * @brief Returns the identification of a plugin instance.
       *
       * @return The identification of the plugin instance.
       */
      virtual PluginId::Enum GetId() const = 0;

      /**
       * @brief Returns the name of a plugin instance.
       *
       * @return Name of the plugin instance.
       */
      virtual const char* GetName() const = 0;

      /**
       * @brief Inititializes the plugin to be ready for accessing its data & interfaces.
       *
       * @return True, if initialization was successful.
       */
      virtual bool Init() = 0;

      /**
       * @brief Resets the plugin owned data.
       */
      virtual void Reset() = 0;

      /**
       * @brief Generic interface to trigger an action.
       *
       * @param actionId 32bit information to determine action.
       * @param actionData 32bit data of the action
       */
      virtual void Action(uint32_t actionId, uint32_t actionData) = 0;

      /**
       * @brief Call the plugin's diagnosis interface, e.g. for dumping the current state.
       *
       * @param requestId 32bit information to transport details of the requests nature.
       */
      virtual void RequestCurrentStatus(uint32_t requestId) = 0;

      /**
       * @brief Call the plugin's diagnosis interface, e.g. for dumping the current state.
       */
      virtual void Diagnosis() {}

      /**
       * @brief Call the plugin's test interface, e.g. for testing plugin logic.
       */
      virtual void Test() {}
};


/// Plugin factory map type
typedef std::map<std::string, IHMIAlert_Daemon_Plugin::Create*, std::less<std::string> > HMIAlertDaemonPluginFactoryMap;

/// Declaration of global shared plugin factory map
extern HMIAlertDaemonPluginFactoryMap gHMIAlertDaemonPluginFactoryMap;
}


}


#define HMIALERT_PLUGIN_STATIC_HEADER(pluginName)             \
    static hmibase::alertdaemon::PluginId::Enum Id() { return hmibase::alertdaemon::PluginId::pluginName; }   \
    static const char * Name() { return #pluginName; }


#endif
