/* ***************************************************************************************
* FILE:          WidgetTraverser.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  WidgetTraverser is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "WidgetTraverser.h"

#include <Courier/Visualization/ViewScene2D.h>
#include <CanderaAssetLoader/AssetLoaderBase/AssetProvider.h>
#include <CanderaAssetLoader/AssetLoader2D/Scene2DContext.h>
#include <CanderaAssetLoader/AssetLoaderBase/DefaultAssetProvider.h>
#include <CanderaWidget/Widget2D/Widget2D.h>
#include <Widgets/2D/BaseWidget2D.h>
#include <Widgets/2D/ControlTemplate/ControlTemplate.h>
#include <View/CGI/CgiExtensions/SCHostUtils.h>


namespace hmibase {
namespace widget {
namespace utils {

Candera::Scene2DContext* MessageUtils::getSceneContext(Candera::Widget2D* widget)
{
   Candera::Scene2DContext* context = NULL;
   if (widget != NULL)
   {
      if ((widget->GetParentView() != NULL) && (widget->GetParentView()->Is2D()))
      {
         context = widget->GetParentView()->ToViewScene2D()->GetScene2DContext();
      }

#ifdef WIN32
      //in Scene Composer the view is NULL so we use the scene context to iterate through the widgets
      if ((context == NULL) && (widget->GetNode() != NULL))
      {
         Candera::Scene2D* scene = widget->GetNode()->GetScene();
         BaseWidget2D* baseWidget = Candera::Dynamic_Cast<BaseWidget2D*>(widget);
         if ((scene != NULL) && (baseWidget != NULL) && (baseWidget->GetAssetProvider() != NULL))
         {
#if ((COURIER_VERSION_MAJOR > 3) || ((COURIER_VERSION_MAJOR == 3) && (COURIER_VERSION_MINOR >= 2)))
            context = baseWidget->GetAssetProvider()->GetScene2DById(Candera::Internal::AssetProviderFunctions::GetIdByName(baseWidget->GetAssetProvider(), Candera::Scene2DLib, scene->GetName()));
#else
            context = baseWidget->GetAssetProvider()->GetScene2D(scene->GetName());
#endif
            //if a normal scene was not found try getting the currently edited composite from SceneComposer
            if (context == NULL)
            {
               //this is required in order to have extension widgets working while editing the composites in SceneComposer, without it you will not see the button images
               context = hmibase::view::util::SCHostUtils::GetComposite2DContext(*(baseWidget->GetAssetProvider()), scene->GetName());
            }
         }
      }
#endif
   }
   return context;
}


const Candera::CompositeGroup2D* MessageUtils::getCompositeNode(const Candera::Node2D* node)
{
   while (node != NULL)
   {
      const Candera::CompositeGroup2D* compositeNode = Candera::Dynamic_Cast<const Candera::CompositeGroup2D*>(node);
      if (compositeNode != NULL)
      {
         return compositeNode;
      }
      node = node->GetParent();
   }

   return NULL;
}


bool MessageUtils::distribute(Candera::Scene2DContext* sceneContext, const Courier::Message& msg)
{
   if ((sceneContext != NULL) && (sceneContext->GetScene() != NULL))
   {
      for (Candera::SceneContextBase::WidgetIterator it = sceneContext->GetWidgetIterator(); it.IsValid(); ++it)
      {
         if ((*it)->OnMessage(msg))
         {
            return true;
         }
      }

      SceneNodeMessageDistributor<Candera::Node2D, Candera::CompositeGroup2D> traverser(msg, false);
      Candera::Node2D* node = sceneContext->GetScene();
      traverser.Traverse(const_cast<Candera::Node2D&>(*node));
      if (traverser.isConsumed())
      {
         return true;
      }
   }
   return false;
}


bool MessageUtils::distributeToDescendants(Candera::Scene2DContext* sceneContext, const Candera::Node2D& startNode, const Courier::Message& msg)
{
   if (sceneContext != NULL)
   {
      //check widgets added directly to the scene
      for (Candera::SceneContextBase::WidgetIterator it = sceneContext->GetWidgetIterator(); it.IsValid(); ++it)
      {
         if ((*it)->OnMessage(msg))
         {
            return true;
         }
      }

      //if node is inside a composite check also the local widgets
      const Candera::CompositeGroup2D* compositeGroup = getCompositeNode(startNode.GetParent());
      if (compositeGroup != NULL)
      {
         for (Candera::SceneContextBase::WidgetIterator it = compositeGroup->GetWidgetIterator(); it.IsValid(); ++it)
         {
            if ((*it)->OnMessage(msg))
            {
               return true;
            }
         }
      }

      SceneNodeMessageDistributor<Candera::Node2D, Candera::CompositeGroup2D> traverser(msg, true);
      traverser.Traverse(const_cast<Candera::Node2D&>(startNode));
      if (traverser.isConsumed())
      {
         return true;
      }
   }
   return false;
}


bool MessageUtils::distributeToAncestors(Candera::Scene2DContext* sceneContext, const Candera::Node2D& startNode, const Courier::Message& msg)
{
   if (sceneContext != NULL)
   {
      // iterate widgets from the scene context
      for (Candera::SceneContextBase::WidgetIterator it = sceneContext->GetWidgetIterator(); it.IsValid(); ++it)
      {
         if ((*it)->OnMessage(msg))
         {
            return true;
         }
      }

      // check all the ancestor nodes
      for (const Candera::Node2D* node = &startNode; node != NULL; node = node->GetParent())
      {
         //for composite nodes iterate through their widgets
         const Candera::CompositeGroup2D* compositeGroup = Candera::Dynamic_Cast<const Candera::CompositeGroup2D*>(node);
         if (compositeGroup != NULL)
         {
            for (Candera::CompositeGroup2D::WidgetIterator it = compositeGroup->GetWidgetIterator(); it.IsValid(); ++it)
            {
               if ((*it)->OnMessage(msg))
               {
                  return true;
               }
            }
         }

         //this is required because widgets cloned by the list are not added to their composite group, instead they are kept into the control template data of the node
         if (distributeToControlTemplate(*node, msg))
         {
            return true;
         }
      }
   }
   return false;
}


bool MessageUtils::distributeToControlTemplate(const Candera::Node2D& node, const Courier::Message& msg)
{
   ControlTemplateData* controlTemplateData = ControlTemplate::TryGetControlTemplateData(node);
   if (controlTemplateData != NULL)
   {
      WidgetBaseEnumerator widgets(controlTemplateData->EnumerateAssociatedWidgets());
      while (widgets.MoveNext())
      {
         if ((widgets.Current() != NULL) && widgets.Current()->OnMessage(msg))
         {
            return true;
         }
      }
   }
   return false;
}


}
}


}
