/* ***************************************************************************************
* FILE:          SharedTimerManager.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  SharedTimerManager.cpp is part of HMI-Base framework Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */

#include "widget2D_std_if.h"
#include "SharedTimerManager.h"
#include <AppUtils/Timer.h>

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_TIMER
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/SharedTimerManager.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace util {


/******************************************************************************/
SharedTimerManager::SharedTimerManager()
{
}


/******************************************************************************/
SharedTimerManager::~SharedTimerManager()
{
   // stop all timers and delete them
   for (TimersType::iterator it = _timers.begin(); it != _timers.end(); ++it)
   {
      TimerInfo& info = it->second;
      if (info.Timer != NULL)
      {
         info.Timer->stop();
         FEATSTD_DELETE(info.Timer);
         info.Timer = NULL;
      }
   }
}


/******************************************************************************/
bool SharedTimerManager::matches(FeatStd::Int index, const Util::Timer& timer) const
{
   TimersType::const_iterator it = _timers.find(index);
   return (it != _timers.end()) && (it->second.Timer == &timer);
}


/******************************************************************************/
bool SharedTimerManager::acquire(FeatStd::Int index, FeatStd::UInt32 timeout)
{
   Util::Timer* timer = NULL;
   // search the timer
   TimersType::iterator it = _timers.find(index);
   // timer found => reuse it
   if (it != _timers.end())
   {
      TimerInfo& info = it->second;
      ++info.RefCount;
      timer = info.Timer;
   }
   // timer not found => create it
   else
   {
      timer = FEATSTD_NEW(Util::Timer);
      if (timer != NULL)
      {
         char instanceName[4];
         SNPRINTF(instanceName, sizeof(instanceName), "%u", index);
         timer->setName("StepAnimation", instanceName);

         TimerInfo info;
         ++info.RefCount;
         info.Timer = timer;

         _timers[index] = info;
      }
   }

   // if a timeout is specified and timer is not already running => start it
   if ((timer != NULL) && (timeout > 0) && (timer->getStatus() != Util::Timer::Running))
   {
      ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SharedTimerManager:acquire start timer Index=%d, Delay=%u", index, timeout));

      timer->setTimeoutWithRepeat(timeout, timeout);
      timer->start();
   }

   return timer != NULL;
}


/******************************************************************************/
void SharedTimerManager::release(FeatStd::Int index)
{
   //search the timer
   TimersType::iterator it = _timers.find(index);
   if (it != _timers.end())
   {
      // decrease the reference count
      TimerInfo& info = it->second;
      --info.RefCount;
      // if count is 0, it is not used anymore => stop and delete it
      if (info.RefCount == 0)
      {
         if (info.Timer != NULL)
         {
            ETG_TRACE_USR1_DCL((APP_TRACECLASS_ID(), "SharedTimerManager:release stop timer Index=%d", index));
            info.Timer->stop();
            FEATSTD_DELETE(info.Timer);
            info.Timer = NULL;
         }
         _timers.erase(it);
      }
   }
}


}
}


}
