/* ***************************************************************************************
* FILE:          IndexRangeSet.h
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  IndexRangeSet is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#if !defined(IndexRangeSet_h)
#define IndexRangeSet_h


#include <bitset>
#include <set>


namespace hmibase {
namespace widget {
namespace utils {

/*****************************************************************************/
class IndexRangeSet
{
      friend class IndexRangeSetFactory;

   public:
      virtual ~IndexRangeSet() {}
      virtual bool contains(size_t index) const = 0;

   private:
      virtual void clear() = 0;
      virtual void add(size_t index) = 0;
      virtual void addRange(size_t begin, size_t end) = 0;
};


/*****************************************************************************/
class IndexRangeSetFactory
{
   public:
      /** text is something like "0, 2-3, 7, 11-19"
      */
      static bool fillRanges(IndexRangeSet& rangeSet, const char* text);
};


/*****************************************************************************/
template <size_t SIZE = 256>
class BitSetIndexRangeSet : public IndexRangeSet
{
   public:
      BitSetIndexRangeSet()
      {
      }

      virtual bool contains(size_t index) const
      {
         return _bitset.test(index);
      }

   private:
      BitSetIndexRangeSet(const BitSetIndexRangeSet&);
      BitSetIndexRangeSet& operator=(const BitSetIndexRangeSet&);

      virtual void clear()
      {
         _bitset = std::bitset<SIZE>();
      }

      virtual void add(size_t index)
      {
         _bitset.set(index);
      }

      virtual void addRange(size_t begin, size_t end)
      {
         for (size_t index = begin; index <= end; ++index)
         {
            _bitset.set(index);
         }
      }

      std::bitset<SIZE> _bitset;
};


struct Range;

/*****************************************************************************/
class DynamicIndexRangeSet : public IndexRangeSet
{
   public:
      DynamicIndexRangeSet();
      virtual ~DynamicIndexRangeSet();

      virtual bool contains(size_t index) const;

   private:
      DynamicIndexRangeSet(const DynamicIndexRangeSet&);
      DynamicIndexRangeSet& operator=(const DynamicIndexRangeSet&);

      virtual void clear();
      virtual void add(size_t index);
      virtual void addRange(size_t begin, size_t end);

      std::set<size_t>* _set;
      Range* _first;
};


}
}


}
#endif//IndexRangeSet_
