/* ***************************************************************************************
* FILE:          EffectUtils.cpp
* SW-COMPONENT:  HMI-BASE
*  DESCRIPTION:  EffectUtils is part of HMI-Base Widget Library
*    COPYRIGHT:  (c) 2015-2016 Robert Bosch Car Multimedia GmbH
*
* The reproduction, distribution and utilization of this file as well as the
* communication of its contents to others without express authorization is
* prohibited. Offenders will be held liable for the payment of damages.
* All rights reserved in the event of the grant of a patent, utility model or design.
*
*************************************************************************************** */
#include "widget2D_std_if.h"
#include "EffectUtils.h"

#include <CanderaPlatform/Device/Common/Effects/BitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Effects/BlurEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ColorEffect.h>
#include <CanderaPlatform/Device/Common/Effects/GlBitmapBrushColorMaskBlend.h>
#include <CanderaPlatform/Device/Common/Effects/GlMaskEffect.h>
#include <CanderaPlatform/Device/Common/Effects/MirrorEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ShadowEffect.h>
#include <CanderaPlatform/Device/Common/Effects/ShearEffect.h>
#include <CanderaPlatform/Device/Common/Effects/SolidColorBrushBlend.h>
#include <CanderaPlatform/Device/Common/Effects/TextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowBitmapBrushBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlDropShadowTextBrushBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlFlipEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlFlipBitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlLinearGradientBrush.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlRadialGradientBrush.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineEffect.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineBitmapBrushColorBlend.h>
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Effects/GlOutlineTextBrushBlend.h>


#include <Trace/ToString.h>
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_HMI_WIDGET_EFFECTCONTROL
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "trcGenProj/Header/EffectUtils.cpp.trc.h"
#endif


namespace hmibase {
namespace widget {
namespace utils {


/*****************************************************************************/
template<>
Candera::GlOutlineEffect* EffectUtils::getEffect<Candera::GlOutlineEffect>(Candera::Effect2D* combinedEffect)
{
   Candera::GlOutlineTextBrushBlend* textBrushBlend = Candera::Dynamic_Cast<Candera::GlOutlineTextBrushBlend*>(combinedEffect);
   if (textBrushBlend != NULL)
   {
      return &(textBrushBlend->GetOutlineEffect());
   }

   Candera::GlOutlineBitmapBrushColorBlend* bitmapBrushBlend = Candera::Dynamic_Cast<Candera::GlOutlineBitmapBrushColorBlend*>(combinedEffect);
   if (bitmapBrushBlend != NULL)
   {
      return &(bitmapBrushBlend->GetOutlineEffect());
   }

   return NULL;
}


/*****************************************************************************/
template<>
Candera::GlDropShadowEffect* EffectUtils::getEffect<Candera::GlDropShadowEffect>(Candera::Effect2D* combinedEffect)
{
   Candera::GlDropShadowTextBrushBlend* textBrushBlend = Candera::Dynamic_Cast<Candera::GlDropShadowTextBrushBlend*>(combinedEffect);
   if (textBrushBlend != NULL)
   {
      return &(textBrushBlend->GetDropShadowEffect());
   }

   Candera::GlDropShadowBitmapBrushBlend* bitmapBrushBlend = Candera::Dynamic_Cast<Candera::GlDropShadowBitmapBrushBlend*>(combinedEffect);
   if (bitmapBrushBlend != NULL)
   {
      return &(bitmapBrushBlend->GetDropShadowEffect());
   }

   return NULL;
}


/*****************************************************************************/
template<>
Candera::GlLinearGradientBrush* EffectUtils::getEffect<Candera::GlLinearGradientBrush>(Candera::Effect2D* combinedEffect)
{
   if (combinedEffect != NULL)
   {
      return Candera::Dynamic_Cast<Candera::GlLinearGradientBrush*>(combinedEffect->GetBrushEffect2D());
   }
   return NULL;
}


/*****************************************************************************/
template<>
Candera::GlRadialGradientBrush* EffectUtils::getEffect<Candera::GlRadialGradientBrush>(Candera::Effect2D* combinedEffect)
{
   if (combinedEffect != NULL)
   {
      return Candera::Dynamic_Cast<Candera::GlRadialGradientBrush*>(combinedEffect->GetBrushEffect2D());
   }
   return NULL;
}


/*****************************************************************************/
Candera::ColorProperty* EffectUtils::getColorProperty(Candera::Node2D* node, colorid::ENUM id)
{
   if (node != NULL)
   {
      Candera::RenderNode* renderNode = Candera::Dynamic_Cast<Candera::RenderNode*>(node);
      if (renderNode != NULL)
      {
         return getColorProperty(renderNode->GetEffect(0), id);
      }
   }
   return NULL;
}


/*****************************************************************************/
Candera::ColorProperty* EffectUtils::getColorProperty(Candera::Effect2D* effect, colorid::ENUM id)
{
   if (effect != NULL)
   {
      switch (id)
      {
         case colorid::None:
         {
            Candera::BitmapBrushColorBlend* bitmapBrushColorBlend = Candera::Dynamic_Cast<Candera::BitmapBrushColorBlend*>(effect);
            if (bitmapBrushColorBlend != NULL)
            {
               return &(bitmapBrushColorBlend->GetColorEffect().Color());
            }

            Candera::GlFlipBitmapBrushColorBlend* glFlipBitmapBrushColorBlend = Candera::Dynamic_Cast<Candera::GlFlipBitmapBrushColorBlend*>(effect);
            if (glFlipBitmapBrushColorBlend != NULL)
            {
               return &(glFlipBitmapBrushColorBlend->GetColorEffect().Color());
            }

            Candera::GlBitmapBrushColorMaskBlend* glBitmapBrushColorMaskBlend = Candera::Dynamic_Cast<Candera::GlBitmapBrushColorMaskBlend*>(effect);
            if (glBitmapBrushColorMaskBlend != NULL)
            {
               return &(glBitmapBrushColorMaskBlend->GetColorEffect().Color());
            }

            Candera::BrushEffect2D* brush = effect->GetBrushEffect2D();

            Candera::SolidColorBrush* solidColorBrush = Candera::Dynamic_Cast<Candera::SolidColorBrush*>(brush);
            if (solidColorBrush != NULL)
            {
               return &(solidColorBrush->Color());
            }

            Candera::TextBrush* textBrush = Candera::Dynamic_Cast<Candera::TextBrush*>(brush);
            if (textBrush != NULL)
            {
               return &(textBrush->Color());
            }
            break;
         }

         case colorid::Outline:
         {
            Candera::GlOutlineEffect* outlineEffect = EffectUtils::getEffect<Candera::GlOutlineEffect>(effect);
            return (outlineEffect != NULL) ? &(outlineEffect->OutlineColor()) : NULL;
         }

         case colorid::DropShadow:
         {
            Candera::GlDropShadowEffect* dropShadowEffect = EffectUtils::getEffect<Candera::GlDropShadowEffect>(effect);
            return (dropShadowEffect != NULL) ? &(dropShadowEffect->ShadowColor()) : NULL;
         }

         default:
            break;
      }
   }
   return NULL;
}


/*****************************************************************************/
bool EffectUtils::setColor(Candera::Node2D* node, const ColorType& color, colorid::ENUM id)
{
   Candera::ColorProperty* colorProperty = getColorProperty(node, id);
   if ((colorProperty != NULL) && (colorProperty->Get() != color))
   {
      ETG_TRACE_USR4_DCL((APP_TRACECLASS_ID(), "SetColor color=%24s id=%u node=%s",
                          HMIBASE_TO_STRING(color), id,
                          HMIBASE_TO_STRING_SPN2D(node)));

      colorProperty->Set(color);
      return true;
   }
   else
   {
      return false;
   }
}


/*****************************************************************************/
EffectUtils::ColorType EffectUtils::getColor(const Candera::Node2D* node, colorid::ENUM id)
{
   Candera::ColorProperty* colorProperty = getColorProperty(const_cast<Candera::Node2D*>(node), id);
   if (colorProperty != NULL)
   {
      return ColorType(colorProperty->Get());
   }

   return ColorType(1.0f, 1.0f, 1.0f, 1.0f);
}


/*****************************************************************************/
void EffectUtils::copyColor(const Candera::Node2D* source, Candera::Node2D* destination, colorid::ENUM id)
{
   setColor(destination, getColor(source, id), id);
}


}
}


}
